<?php

namespace Arts\Optimizer\Managers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use \Arts\Utilities\Utilities;

/**
 * Class Preloads
 *
 * Responsible for managing the preloading and prefetching of various assets (e.g., CSS, JS, fonts, images) to improve the performance of the website.
 *
 * @package Arts\Optimizer\Managers
 */
class Preloads extends BaseManager {
	/**
	 * @var bool $enqueue_css_assets Whether to enqueue CSS ass! $is_lazy_dep && ets instead of preloading them.
	 */
	protected $enqueue_css_assets = false;

	protected $assets_discovered = false;

	/**
	 * @var array $css_assets CSS assets to be enqueued.
	 */
	protected $css_assets = array();

	/**
	 * @var string $match_pattern Regular expression pattern to match HTML tags with a specific data attribute.
	 *
	 * It matches HTML tags that contain the attribute `data-arts-component-name`.
	 * The attribute value can be enclosed in either single or double quotes.
	 * This variable can be customized using the `arts/optimizer/preloads/match_pattern` WordPress filter.
	 *
	 * @example "Example" `<div data-arts-component-name="MyCustomComponent">...</div>`
	 * @see Preloads::inject_preload_links()
	 */
	private $match_pattern = '/<[^>]*?data-arts-component-name=(["\'])?((?:.(?!\1|>))*.?)\1/i';

	/**
	 * @var string $search_string The string to search for in the output where preload links will be injected.
	 *
	 * Used as a marker in the HTML output to determine where the preload links should be inserted.
	 * By default, it is set to '<meta charset="UTF-8">' to ensure that preload links are injected early in the document.
	 * This variable can be customized using the `arts/optimizer/preloads/replace_string` WordPress filter.
	 *
	 * @example "Example" `<meta charset="UTF-8">`
	 * @see Preloads::inject_preload_links()
	 */
	private $search_string = '<meta charset="UTF-8">';

	/**
	 * @var array $file_types_attributes_map Map of file extensions to their corresponding preload attributes.
	 *
	 * This associative array maps file extensions (e.g., 'css', 'js', 'woff', 'woff2') to their corresponding preload attributes.
	 * Each entry in the array specifies the attributes to be used when generating preload links for that file type.
	 * The attributes include 'as', 'type', and 'crossorigin'.
	 * This variable can be customized using the `arts/optimizer/preloads/attributes_map` WordPress filter.
	 *
	 * @example "Example"
	 * `array(
	 *   'css'   => array(
	 *     'as'          => 'style',
	 *     'type'        => 'text/css',
	 *     'crossorigin' => 'anonymous',
	 *   ),
	 *   'js'    => array(
	 *     'as'          => 'script',
	 *     'type'        => 'application/javascript',
	 *     'crossorigin' => 'anonymous',
	 *   ),
	 *   'woff'  => array(
	 *     'as'          => 'font',
	 *     'type'        => 'font/woff',
	 *     'crossorigin' => 'anonymous',
	 *   ),
	 *   'woff2' => array(
	 *     'as'          => 'font',
	 *     'type'        => 'font/woff2',
	 *     'crossorigin' => 'anonymous',
	 *   ),
	 * )`
	 */
	private $file_types_attributes_map = array(
		'css'   => array(
			'as'          => 'style',
			'type'        => 'text/css',
			'crossorigin' => 'anonymous',
		),
		'js'    => array(
			'as'          => 'script',
			'type'        => 'application/javascript',
			'crossorigin' => 'anonymous',
		),
		'woff'  => array(
			'as'          => 'font',
			'type'        => 'font/woff',
			'crossorigin' => 'anonymous',
		),
		'woff2' => array(
			'as'          => 'font',
			'type'        => 'font/woff2',
			'crossorigin' => 'anonymous',
		),
	);

	/**
	 * @var array $prefetch_map Map of URLs to be prefetched.
	 *
	 * This array contains URLs that should be prefetched to improve performance.
	 * Each entry in the array represents a URL that will be prefetched using the `rel="prefetch"` attribute.
	 * This variable can be customized using the `arts/optimizer/preloads/prefetch_map` WordPress filter.
	 *
	 * @see Preloads::get_prefetch_assets_html_string()
	 */
	private $prefetch_map = array();

	/**
	 * @var array $preload_assets_map Map of assets to be preloaded.
	 *
	 * This array contains URLs of assets that should be preloaded to improve performance.
	 * Each entry in the array represents a URL that will be preloaded using the `rel="preload"` attribute.
	 * This variable can be customized using the `arts/optimizer/preloads/assets_map` WordPress filter.
	 *
	 * @see Preloads::get_preload_assets_html_string()
	 */
	private $preload_assets_map = array();

	/**
	 * @var array $preload_modules_map Map of modules to be preloaded.
	 *
	 * This array contains URLs of JavaScript modules that should be preloaded to improve performance.
	 * Each entry in the array represents a URL that will be preloaded using the `rel="modulepreload"` attribute.
	 * This variable can be customized using the `arts/optimizer/preloads/modules_map` WordPress filter.
	 *
	 * @see Preloads::get_preload_modules_html_string()
	 */
	private $preload_modules_map = array();

	/**
	 * @var array $preload_images_map Map of images to be preloaded.
	 *
	 * This array contains URLs of images that should be preloaded to improve performance.
	 * Each entry in the array represents a URL that will be preloaded using the `rel="preload"` attribute.
	 * This variable can be customized using the `arts/optimizer/preloads/images_map` WordPress filter.
	 *
	 * @see Preloads::get_preload_images_html_string()
	 */
	private $preload_images_map = array();

	/**
	 * @var array $generated_asset_urls Tracks generated asset URLs to prevent duplicates across all preload methods.
	 */
	private $generated_asset_urls = array();

	/**
	 * @var array $inline_dependencies Tracks dependencies discovered through data-arts-component-dependencies attributes.
	 */
	private $inline_dependencies = array();

	/**
	 * @var string $component_url_before The base URL for component scripts.
	 *
	 * This variable holds the base URL that will be prepended to the component names to form the complete URL for preloading component scripts.
	 * It can be customized using the `arts/optimizer/preloads/component_url_before` WordPress filter.
	 *
	 * @example "Example" `https://example.com/js/components/`
	 * @see Preloads::apply_filters()
	 */
	private $component_url_before;

	/**
	 * @var string $component_url_after The suffix for component scripts.
	 *
	 * This variable holds the suffix that will be appended to the component names to form the complete URL for preloading component scripts.
	 * By default, it is set to '.js'.
	 * It can be customized using the `arts/optimizer/preloads/component_url_after` WordPress filter.
	 *
	 * @example "Example" `.js`
	 * @see Preloads::apply_filters()
	 */
	private $component_url_after = '.js';

	/**
	 * @var bool $preload_only_first_image Whether to preload only the first image in the preload images map.
	 *
	 * This variable determines whether only the first image in the preload images map should be preloaded.
	 * By default, it is set to true.
	 * It can be customized using the `arts/optimizer/preloads/only_first_image` WordPress filter.
	 *
	 * @see Preloads::get_preload_images_html_string()
	 */
	private $preload_only_first_image = true;

	/**
	 * @var string $preload_images_fetch_priority The fetch priority for preloading images.
	 *
	 * This variable determines the fetch priority for preloading images.
	 * By default, it is set to 'low'.
	 * It can be customized using the `arts/optimizer/preloads/images_fetch_priority` WordPress filter.
	 *
	 * @see Preloads::get_preload_link()
	 */
	private $preload_images_fetch_priority = 'low';

	/**
	 * Initializes the Preloads manager with the provided managers.
	 *
	 * @param array $managers An array of managers to be used by the Preloads manager.
	 */
	public function init( $managers ) {
		$this->component_url_before = get_template_directory_uri() . '/js/components/';

		$this->add_managers( $managers );
	}

	/**
	 * Discovers assets with caching support.
	 *
	 * @return void
	 */
	public function discover_assets() {
		// Only discover assets once per page load
		if ( $this->assets_discovered ) {
			return;
		}

		// Discover assets from WordPress scripts
		$this->add_from_wp_scripts();

		// IMPORTANT: Function `add_from_wp_styles` is not needed here, as we handle CSS assets separately

		// Apply filters to customize asset maps
		$this->apply_filters_once();

		// Store CSS assets separately if enqueuing is enabled
		if ( $this->enqueue_css_assets ) {
			$this->extract_css_assets();
		}

		$this->assets_discovered = true;
	}

	/**
	 * Gets a configuration value with caching.
	 *
	 * @param string $key The configuration key.
	 * @param mixed  $default The default value.
	 * @return mixed The configuration value.
	 */
	protected function get_config( $key, $default = null ) {
		static $config_cache = array();

		if ( ! isset( $config_cache[ $key ] ) ) {
			$config_cache[ $key ] = apply_filters( $key, $default );
		}

		return $config_cache[ $key ];
	}

	/**
	 * Applies filters only once to avoid redundant filter applications.
	 *
	 * @return void
	 */
	private function apply_filters_once() {
		static $filters_applied = false;

		if ( $filters_applied ) {
			return;
		}

		// Use cached configuration instead of repeated filter calls
		$this->enqueue_css_assets            = $this->get_config( 'arts/optimizer/preloads/enqueue_css_assets', $this->enqueue_css_assets );
		$this->match_pattern                 = $this->get_config( 'arts/optimizer/preloads/match_pattern', $this->match_pattern );
		$this->search_string                 = $this->get_config( 'arts/optimizer/preloads/replace_string', $this->search_string );
		$this->file_types_attributes_map     = $this->get_config( 'arts/optimizer/preloads/attributes_map', $this->file_types_attributes_map );
		$this->prefetch_map                  = $this->get_config( 'arts/optimizer/preloads/prefetch_map', $this->prefetch_map );
		$this->preload_images_map            = $this->get_config( 'arts/optimizer/preloads/images_map', $this->preload_images_map );
		$this->preload_modules_map           = $this->get_config( 'arts/optimizer/preloads/modules_map', $this->preload_modules_map );
		$this->preload_assets_map            = $this->get_config( 'arts/optimizer/preloads/assets_map', $this->preload_assets_map );
		$this->component_url_before          = $this->get_config( 'arts/optimizer/preloads/component_url_before', $this->component_url_before );
		$this->component_url_after           = $this->get_config( 'arts/optimizer/preloads/component_url_after', $this->component_url_after );
		$this->preload_only_first_image      = $this->get_config( 'arts/optimizer/preloads/only_first_image', $this->preload_only_first_image );
		$this->preload_images_fetch_priority = $this->get_config( 'arts/optimizer/preloads/images_fetch_priority', $this->preload_images_fetch_priority );

		$filters_applied = true;
	}

	/**
	 * Extracts CSS assets from the preload assets map.
	 *
	 * @return void
	 */
	private function extract_css_assets() {
		$this->css_assets = array();

		foreach ( $this->preload_assets_map as $id => $url ) {
			$file_extension = Utilities::get_asset_url_file_extension( $url );
			if ( $file_extension === 'css' ) {
				$this->css_assets[ $id ] = $url;
				// Don't remove from preload_assets_map yet, we'll skip during HTML generation
			}
		}
	}

	/**
	 * Add scripts from WordPress's wp_scripts to the appropriate preload, modulepreload, or prefetch maps.
	 *
	 * @return void
	 */
	private function add_from_wp_scripts() {
		$wp_scripts = wp_scripts();

		if ( ! $wp_scripts || empty( $wp_scripts->queue ) ) {
			return;
		}

		foreach ( $wp_scripts->queue as $handle ) {
			$preload_type = $wp_scripts->get_data( $handle, 'preload_type' );

			if ( ! $preload_type ) {
				continue;
			}

			$id  = "{$handle}";
			$src = $wp_scripts->registered[ $handle ]->src;
			$ver = $wp_scripts->registered[ $handle ]->ver;
			$url = $ver ? "{$src}?ver={$ver}" : $src;

			if ( $preload_type === 'preload' ) {
				$this->preload_assets_map[ $id ] = $url;
			} elseif ( $preload_type === 'modulepreload' ) {
				$this->preload_modules_map[ $id ] = $url;
			} elseif ( $preload_type === 'prefetch' ) {
				$this->prefetch_map[ $id ] = $url;
			}
		}
	}

	/**
	 * Injects preload links into the given output with optimized component discovery.
	 *
	 * @param string $output The output to inject preload links into.
	 * @return string The modified output with preload links injected.
	 */
	public function inject_preload_links( $output ) {
		$replace_string = '';
		$components     = array();

		// Reset the URL tracking for this page load
		$this->generated_asset_urls = array();

		// Use DOM-based component discovery instead of regex
		if ( $this->match_pattern ) {
			$components = $this->find_components_with_dom( $output );
		}

		$output = apply_filters( 'arts/optimizer/preloads/output', $output );

		// Make sure assets are discovered (with caching)
		$this->discover_assets();

		// Generate HTML for various asset types
		if ( ! empty( $this->preload_images_map ) ) {
			$replace_string .= $this->get_preload_images_html_string();
		}

		if ( ! empty( $this->preload_modules_map ) ) {
			$replace_string .= $this->get_preload_modules_html_string();
		}

		if ( ! empty( $this->preload_assets_map ) ) {
			$replace_string .= $this->get_preload_assets_html_string();
		}

		if ( ! empty( $this->prefetch_map ) ) {
			$replace_string .= $this->get_prefetch_assets_html_string();
		}

		if ( ! empty( $components ) ) {
			$replace_string .= $this->get_preload_components_html_string( $components );
		}

		if ( empty( $replace_string ) ) {
			return $output;
		}

		$result = str_replace( $this->search_string, $this->search_string . $replace_string, $output );

		// Clean up large variables
		unset( $components );
		unset( $replace_string );
		unset( $output );

		return $result;
	}

	/**
	 * Returns a string containing HTML links for preloading images.
	 *
	 * This method generates an HTML string that contains preload links for images.
	 * The method checks if only the first image should be preloaded or all images.
	 * If only the first image should be preloaded, it generates a preload link for the first image.
	 * If all images should be preloaded, it generates preload links for each image in the preload images map.
	 * The generated links have the `rel` attribute set to "preload".
	 *
	 * @return string The HTML string for preloading images.
	 */
	private function get_preload_images_html_string() {
		$replace_string = '';

		$this->preload_images_map = array_unique( $this->preload_images_map );

		if ( $this->preload_only_first_image ) {
			$only_first_image_map = reset( $this->preload_images_map );

			// Skip if URL is already generated
			if ( ! isset( $this->generated_asset_urls[ $only_first_image_map ] ) ) {
				$replace_string = $this->get_preload_link(
					$only_first_image_map,
					array(
						'id'            => $only_first_image_map,
						'rel'           => 'preload',
						'echo'          => false,
						'fetchpriority' => $this->preload_images_fetch_priority,
					)
				) . $replace_string;

				// Mark this URL as generated
				$this->generated_asset_urls[ $only_first_image_map ] = true;
			}
		} else {
			foreach ( $this->preload_images_map as $val ) {
				// Skip if URL is already generated
				if ( isset( $this->generated_asset_urls[ $val ] ) ) {
					continue;
				}

				$replace_string = $this->get_preload_link(
					$val,
					array(
						'id'            => $val,
						'rel'           => 'preload',
						'echo'          => false,
						'fetchpriority' => $this->preload_images_fetch_priority,
					)
				) . $replace_string;

				// Mark this URL as generated
				$this->generated_asset_urls[ $val ] = true;
			}
		}

		return $replace_string;
	}

	/**
	 * Returns a string containing HTML links for prefetching assets.
	 *
	 * This method retrieves the unique prefetch assets from the `prefetch_map` array
	 * and generates HTML string containing preload links for each asset.
	 * The generated links have the `rel` attribute set to "prefetch".
	 *
	 * @return string The HTML string containing prefetch links.
	 */
	private function get_prefetch_assets_html_string() {
		$replace_string = '';

		$this->prefetch_map = array_unique( $this->prefetch_map );

		foreach ( $this->prefetch_map as $id => $val ) {
			// Skip if URL is already generated
			if ( isset( $this->generated_asset_urls[ $val ] ) ) {
				continue;
			}

			$replace_string = $this->get_preload_link(
				$val,
				array(
					'id'   => is_string( $id ) ? $id : null,
					'rel'  => 'prefetch',
					'echo' => false,
				)
			) . $replace_string;

			// Mark this URL as generated
			$this->generated_asset_urls[ $val ] = true;
		}

		return $replace_string;
	}

	/**
	 * Finds components in HTML using DOM with position analysis for priority ordering.
	 * Also discovers and processes inline dependencies from data-arts-component-dependencies attributes.
	 *
	 * @param string $html The HTML to search for components.
	 * @return array Found component data with position information, ordered by appearance.
	 */
	private function find_components_with_dom( $html ) {
		$components = array();

		// Only create DOM if HTML is not empty
		if ( empty( $html ) ) {
			return $components;
		}

		// Use libxml to suppress warnings during loading
		$previous_value = libxml_use_internal_errors( true );

		$dom = new \DOMDocument();
		$dom->loadHTML( $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );

		$xpath = new \DOMXPath( $dom );
		$nodes = $xpath->query( '//*[@data-arts-component-name]' );

		$components_data = array();

		foreach ( $nodes as $index => $node ) {
			if ( $node instanceof \DOMElement ) {
				$component_name = $node->getAttribute( 'data-arts-component-name' );

				// Calculate position metrics for priority ordering
				$position_data = $this->calculate_component_position( $node, $dom );

				$components_data[] = array(
					'name'              => $component_name,
					'position'          => $position_data['position'],
					'depth'             => $position_data['depth'],
					'viewport_priority' => $this->get_viewport_priority( $node ),
					'dom_order'         => $index, // Original DOM order as fallback
				);

				// NEW: Process inline dependencies from data-arts-component-dependencies attribute
				$dependencies_attr = $node->getAttribute( 'data-arts-component-dependencies' );
				if ( ! empty( $dependencies_attr ) ) {
					$this->process_inline_component_dependencies( $dependencies_attr );
				}
			}
		}

		// Sort components by priority (position from top, then depth, then DOM order)
		usort( $components_data, array( $this, 'compare_component_priority' ) );

		// Extract just the component names in prioritized order
		foreach ( $components_data as $component_data ) {
			$components[] = $component_data['name'];
		}

		// Reverse the array to ensure first visual components get highest priority
		$components = array_reverse( $components );

		// Restore previous error handling state
		libxml_use_internal_errors( $previous_value );

		return $components;
	}

	/**
	 * Calculates component position metrics for priority determination.
	 *
	 * @param \DOMElement  $node The DOM element to analyze.
	 * @param \DOMDocument $dom The document containing the element.
	 * @return array Position data including distance from top and nesting depth.
	 */
	private function calculate_component_position( $node, $dom ) {
		$position = 0;
		$depth    = 0;
		$current  = $node;

		// Calculate nesting depth and estimate vertical position
		while ( $current && $current !== $dom->documentElement ) {
			$depth++;

			// Estimate position based on preceding siblings
			$preceding_siblings = 0;
			$sibling            = $current->previousSibling;
			while ( $sibling ) {
				if ( $sibling->nodeType === XML_ELEMENT_NODE ) {
					$preceding_siblings++;
				}
				$sibling = $sibling->previousSibling;
			}

			$position += $preceding_siblings * 10; // Weight siblings
			$current   = $current->parentNode;
		}

		return array(
			'position' => $position,
			'depth'    => $depth,
		);
	}

	/**
	 * Determines viewport priority based on DOM structure order.
	 *
	 * @param \DOMElement $node The DOM element to analyze.
	 * @return int Priority score (always 0 for DOM order-based priority).
	 */
	private function get_viewport_priority( $node ) {
		// Return constant priority to rely purely on DOM position order
		return 0;
	}

	/**
	 * Compares two components for priority sorting.
	 *
	 * @param array $a First component data.
	 * @param array $b Second component data.
	 * @return int Comparison result for usort.
	 */
	private function compare_component_priority( $a, $b ) {
		// First compare by document position (lower = higher priority)
		if ( $a['position'] !== $b['position'] ) {
			return $a['position'] - $b['position'];
		}

		// Then by depth (shallower = higher priority)
		if ( $a['depth'] !== $b['depth'] ) {
			return $a['depth'] - $b['depth'];
		}

		// Finally by DOM order as fallback
		return $a['dom_order'] - $b['dom_order'];
	}

	/**
	 * Processes inline component dependencies from data-arts-component-dependencies attribute.
	 *
	 * @param string $dependencies_attr The raw dependencies attribute value.
	 * @return void
	 */
	private function process_inline_component_dependencies( $dependencies_attr ) {
		if ( empty( $dependencies_attr ) ) {
			return;
		}

		// Parse the JSON-formatted dependencies
		$dependencies = $this->parse_options_string_object( $dependencies_attr );

		if ( empty( $dependencies ) || ! is_array( $dependencies ) ) {
			return;
		}

		// Get WordPress scripts and styles for dependency lookup
		$wp_scripts = wp_scripts();
		$wp_styles  = wp_styles();

		foreach ( $dependencies as $dependency_handle ) {
			if ( empty( $dependency_handle ) || ! is_string( $dependency_handle ) ) {
				continue;
			}

			// Track this as an inline dependency
			$this->inline_dependencies[ $dependency_handle ] = true;

			// Check if this dependency is a registered script
			if ( $wp_scripts && isset( $wp_scripts->registered[ $dependency_handle ] ) ) {
				$script = $wp_scripts->registered[ $dependency_handle ];
				$src    = $script->src;
				$ver    = $script->ver;
				$url    = $ver ? "{$src}?ver={$ver}" : $src;

				// Add to preload assets map
				$this->preload_assets_map[ $dependency_handle ] = $url;
			}
			// Check if this dependency is a registered style
			elseif ( $wp_styles && isset( $wp_styles->registered[ $dependency_handle ] ) ) {
				$style = $wp_styles->registered[ $dependency_handle ];
				$src   = $style->src;
				$ver   = $style->ver;
				$url   = $ver ? "{$src}?ver={$ver}" : $src;

				// Add to preload assets map
				$this->preload_assets_map[ $dependency_handle ] = $url;
			}
			// If not found in registered assets, try to construct URL using component URL pattern
			else {
				// Check if this looks like a CSS dependency (contains 'css' in the name)
				$is_css_dependency = strpos( strtolower( $dependency_handle ), 'css' ) !== false;

				if ( $is_css_dependency ) {
					// For CSS dependencies, we might need to construct the URL differently
					// This is a fallback mechanism - developers should register their assets properly
					$constructed_url = apply_filters(
						'arts/optimizer/preloads/inline_dependency_url',
						null,
						$dependency_handle,
						'css'
					);

					if ( $constructed_url ) {
						$this->preload_assets_map[ $dependency_handle ] = $constructed_url;
					}
				} else {
					// For JS dependencies, use the component URL pattern as fallback
					$constructed_url = apply_filters(
						'arts/optimizer/preloads/inline_dependency_url',
						$this->component_url_before . $dependency_handle . $this->component_url_after,
						$dependency_handle,
						'js'
					);

					if ( $constructed_url ) {
						$this->preload_assets_map[ $dependency_handle ] = $constructed_url;
					}
				}
			}
		}
	}

	/**
	 * Parses a string object similar to the JavaScript parseOptionsStringObject function.
	 *
	 * @param string $str_obj The string object to parse.
	 * @return array The parsed result array.
	 */
	private function parse_options_string_object( $str_obj ) {
		if ( empty( $str_obj ) ) {
			return array();
		}

		// Decode HTML entities once
		$decoded_str = html_entity_decode( $str_obj, ENT_QUOTES, 'UTF-8' );

		// Try direct JSON decode first (for properly formatted JSON)
		$result = json_decode( $decoded_str, true );
		if ( json_last_error() === JSON_ERROR_NONE ) {
			return $result;
		}

		// Fall back to conversion for JS-style object notation
		$json_str = $this->convert_string_to_json( $decoded_str );
		$result   = json_decode( $json_str, true );

		return is_array( $result ) ? $result : array();
	}

	/**
	 * Converts a string object to proper JSON format, similar to the JavaScript convertStringToJSON function.
	 *
	 * @param string $str_obj The string object to convert.
	 * @return string The converted JSON string.
	 */
	private function convert_string_to_json( $str_obj ) {
		if ( empty( $str_obj ) ) {
			return '';
		}

		// Replace single quotes with double quotes
		$filtered_str = str_replace( "'", '"', $str_obj );

		// Add quotes around unquoted keys (simplified version of the JS regex)
		// This handles the case where object keys are not properly quoted
		$filtered_str = preg_replace_callback(
			'/(?=[^"]*(?:"[^"]*"[^"]*)*$)(\w+:)|(\w+ :)/',
			function ( $matches ) {
				$match = $matches[0];
				return '"' . substr( $match, 0, -1 ) . '":';
			},
			$filtered_str
		);

		return $filtered_str;
	}

	/**
	 * Returns a string containing HTML links for preloading assets.
	 *
	 * This method iterates over the unique values in the `preload_assets_map` array and generates HTML links for preloading each module.
	 * The generated links have the `rel` attribute set to "preload".
	 *
	 * @return string The HTML string containing the preload links.
	 */
	private function get_preload_assets_html_string() {
		$replace_string           = '';
		$this->preload_assets_map = array_unique( $this->preload_assets_map );

		foreach ( $this->preload_assets_map as $id => $val ) {
			// Skip if URL is already generated
			if ( isset( $this->generated_asset_urls[ $val ] ) ) {
				continue;
			}

			$replace_string = $this->get_preload_link(
				$val,
				array(
					'id'   => is_string( $id ) ? $id : null,
					'rel'  => 'preload',
					'echo' => false,
				)
			) . $replace_string;

			// Mark this URL as generated
			$this->generated_asset_urls[ $val ] = true;
		}

		return $replace_string;
	}

	/**
	 * Returns a string containing HTML links for preloading modules.
	 *
	 * This method iterates over the unique values in the `preload_modules_map` array and generates HTML links for preloading each module.
	 * The generated links have the `rel` attribute set to "modulepreload".
	 *
	 * @return string The HTML string containing the preload module links.
	 */
	private function get_preload_modules_html_string() {
		$replace_string = '';

		$this->preload_modules_map = array_unique( $this->preload_modules_map );

		foreach ( $this->preload_modules_map as $val ) {
			// Skip if URL is already generated
			if ( isset( $this->generated_asset_urls[ $val ] ) ) {
				continue;
			}

			$replace_string = $this->get_preload_link(
				$val,
				array(
					'rel'  => 'modulepreload',
					'echo' => false,
				)
			) . $replace_string;

			// Mark this URL as generated
			$this->generated_asset_urls[ $val ] = true;
		}

		return $replace_string;
	}

	/**
	 * Returns a string containing HTML links for preloading JS components and their dependencies.
	 *
	 * @param array $match_result The match result containing the components to preload.
	 * @return string The HTML string containing the components preloads.
	 */
	private function get_preload_components_html_string( $match_result ) {
		$replace_string = '';

		// Get detailed component data including CSS dependencies
		$components_data = $this->get_preload_components_data();
		$components_map  = array_column( $components_data, null, 'id' );

		// Create separate map for component-specific assets to avoid duplicates
		$component_assets_map = array();

		// Process detected components on the page
		foreach ( $match_result as $component_name ) {
			// First check if the component exists in our registered scripts
			if ( isset( $components_map[ $component_name ] ) ) {
				$component_data = $components_map[ $component_name ];

				// Add the component script to component assets
				$component_assets_map[ $component_name ] = $component_data['url'];

				// Now check for component dependencies
				if ( ! empty( $component_data['deps'] ) ) {
					foreach ( $component_data['deps'] as $dep ) {
						$is_lazy_dep = isset( $components_map[ $dep ] ) && isset( $components_map[ $dep ]['lazy'] ) && $components_map[ $dep ]['lazy'];

						// Check if this dependency was discovered through inline component dependencies
						$is_inline_dependency = isset( $components_map[ $dep ] ) && isset( $components_map[ $dep ]['inline_discovered'] ) && $components_map[ $dep ]['inline_discovered'];

						// If dependency is a CSS file we found earlier
						// Include lazy dependencies if they were discovered inline (bypass lazy check for inline dependencies)
						if ( ( ! $is_lazy_dep || $is_inline_dependency ) && isset( $components_map[ $dep ] ) && isset( $components_map[ $dep ]['type'] ) && $components_map[ $dep ]['type'] === 'css' ) {
							// Add CSS dependency to component assets with original ID as key
							$component_assets_map[ $dep ] = $components_map[ $dep ]['url'];
						}
					}
				}
			} else {
				// Fallback for components not registered in wp_scripts
				$preload_url = apply_filters(
					'arts/optimizer/preloads/component_url',
					array(
						'url_before'     => $this->component_url_before,
						'component_name' => $component_name,
						'url_after'      => $this->component_url_after,
					)
				);
				$preload_url = join( '', $preload_url );

				// Add to component assets with component name as key
				$component_assets_map[ $component_name ] = $preload_url;
			}
		}

		// Ensure uniqueness and preserve order
		$component_assets_map = array_unique( $component_assets_map );

		// Now generate the HTML for component-specific assets
		foreach ( $component_assets_map as $id => $url ) {
			// Skip if URL is already generated
			if ( isset( $this->generated_asset_urls[ $url ] ) ) {
				continue;
			}

			// Check if this is a CSS asset that needs special handling
			$is_css = isset( $components_map[ $id ] ) && isset( $components_map[ $id ]['type'] ) && $components_map[ $id ]['type'] === 'css';

			$replace_string = $this->get_preload_link(
				$url,
				array(
					'id'   => $id,
					'rel'  => 'preload',
					'echo' => false,
				// Our modified get_preload_link will convert CSS preloads to stylesheet links if needed
				)
			) . $replace_string;

			// Mark this URL as generated
			$this->generated_asset_urls[ $url ] = true;
		}

		return $replace_string;
	}

	/**
	 * Returns detailed preload data for components including their CSS dependencies.
	 *
	 * @return array Component data with preload information
	 */
	private function get_preload_components_data() {
		$wp_scripts            = wp_scripts();
		$wp_styles             = wp_styles();
		$allowed_preload_types = array(
			'preload',
			'modulepreload',
			'prefetch',
		);

		$components_data = array();

		// First collect all registered scripts with their dependencies
		foreach ( $wp_scripts->registered as $handle => $script ) {
			$src  = $script->src;
			$ver  = $script->ver;
			$deps = $script->deps; // Get script dependencies

			$preload_type = $wp_scripts->get_data( $handle, 'preload_type' );

			// Skip if not a preloadable component
			if ( ! $preload_type || ! in_array( $preload_type, $allowed_preload_types ) ) {
				continue;
			}

			$url = $ver ? "{$src}?ver={$ver}" : $src;

			$components_data[] = array(
				'id'   => $handle,
				'rel'  => $preload_type,
				'url'  => $url,
				'deps' => $deps,
			);

			// Now check for CSS dependencies of this script
			if ( ! empty( $deps ) && $wp_styles ) {
				foreach ( $deps as $dep ) {
					// Check if dependency is a registered style
					if ( isset( $wp_styles->registered[ $dep ] ) ) {
						$style   = $wp_styles->registered[ $dep ];
						$is_lazy = $wp_styles->get_data( $dep, 'dynamic_load_prevent_autoload' );

						$style_src = $style->src;
						$style_ver = $style->ver;
						$style_url = $style_ver ? "{$style_src}?ver={$style_ver}" : $style_src;

						// Check if this dependency was discovered through inline component dependencies
						$is_inline_discovered = isset( $this->inline_dependencies[ $dep ] );

						// Add CSS dependency to preload assets
						$components_data[] = array(
							'id'                => $dep,
							'rel'               => 'preload', // Always preload CSS dependencies
							'url'               => $style_url,
							'type'              => 'css',    // Mark as CSS for later processing
							'lazy'              => $is_lazy, // Track if this is a lazy-loaded style
							'inline_discovered' => $is_inline_discovered, // Track if discovered through inline dependencies
						);
					}
				}
			}
		}

		return $components_data;
	}

	/**
	 * Generates a preload link tag for a given URL.
	 *
	 * @param string $url The URL to preload.
	 * @param array  $args Optional arguments for customizing the preload link.
	 *    - rel: The relationship type of the link element (default: 'preload').
	 *    - id: The post ID for media (images, videos).
	 *    - size: The size of the media (default: null).
	 *    - echo: Whether to echo the link tag or return it as a string (default: false).
	 *    - fetchpriority: The fetch priority of the resource (default: false).
	 * @return string The preload link tag if echo is false, otherwise nothing.
	 */
	private function get_preload_link( $url, $args = array() ) {
		if ( empty( $url ) ) {
			return '';
		}

		$defaults = array(
			'rel'           => 'preload',
			'id'            => null, // Post ID for media (images, videos),
			'size'          => null, // 'full',
			'echo'          => false,
			'fetchpriority' => false,
		);

		$args = wp_parse_args( $args, $defaults );

		$attributes = array(
			'rel'  => $args['rel'],
			'href' => $url,
		);

		$is_media       = isset( $args['id'] ) && $args['id'] !== null;
		$file_extension = Utilities::get_asset_url_file_extension( $attributes['href'] );

		// Flag to track if we converted to stylesheet
		$converted_to_stylesheet = false;

		// NEW: Check if we should convert preload to stylesheet for CSS files
		if ( $this->enqueue_css_assets && $file_extension === 'css' && $args['rel'] === 'preload' ) {
			// Change to stylesheet and remove unnecessary attributes
			$attributes['rel'] = 'stylesheet';

			// Remove attributes that aren't needed for same-origin stylesheets
			unset( $this->file_types_attributes_map['css']['as'] );
			unset( $this->file_types_attributes_map['css']['crossorigin'] );

			$attributes              = array_merge( $attributes, $this->file_types_attributes_map['css'] );
			$converted_to_stylesheet = true;
		} elseif ( $args['rel'] === 'preload' ) {
			switch ( $file_extension ) {
				case 'css':
					$attributes = array_merge( $attributes, $this->file_types_attributes_map['css'] );
					break;
				case 'js':
					$attributes = array_merge( $attributes, $this->file_types_attributes_map['js'] );
					break;
				case 'woff':
					$attributes = array_merge( $attributes, $this->file_types_attributes_map['woff'] );
					break;
				case 'woff2':
					$attributes = array_merge( $attributes, $this->file_types_attributes_map['woff2'] );
					break;
				default:
					if ( $is_media ) {
						$attributes = array_merge( $attributes, self::get_preload_attributes_for_image( $args ) );
					}
					break;
			}

			if ( is_string( $args['fetchpriority'] ) ) {
				$attributes['fetchpriority'] = $args['fetchpriority'];
			}
		}

		if ( array_key_exists( 'href', $attributes ) && ! empty( $attributes['href'] ) ) {
			// Generate ID with proper format based on link type
			if ( $converted_to_stylesheet ) {
				// For stylesheets, use proper WordPress CSS handle format
				if ( $args['id'] ) {
					// Convert component handle to proper CSS handle format
					// e.g., "circlebuttoncss" -> "circle-button-css"
					$css_handle = $args['id'];

					// Remove 'css' suffix if present to avoid duplication
					if ( substr( $css_handle, -3 ) === 'css' ) {
						$css_handle = substr( $css_handle, 0, -3 );
					}

					// Convert camelCase or concatenated words to kebab-case
					$css_handle = self::slugify( $css_handle );

					$attributes['id'] = $css_handle . '-css';
				} else {
					$attributes['id'] = self::slugify( basename( $attributes['href'] ) ) . '-css';
				}
			} else {
				// Regular preload ID generation
				$attributes['id'] = $this->get_preload_id( $args, $attributes['href'] );
			}

			if ( $is_media && $args['id'] ) {
				if ( ! empty( $attributes['imagesrcset'] ) ) {
					unset( $attributes['href'] );
				}
			}

			$attributes = apply_filters( 'arts/optimizer/preloads/link_attributes', $attributes, $args );

			if ( $args['echo'] ) {
				?>
				<link <?php Utilities::print_attributes( $attributes ); ?>>
				<?php
			} else {
				return '<link ' . Utilities::print_attributes( $attributes, false ) . '>';
			}
		}

		return '';
	}

	/**
	 * Returns the preload ID based on the provided arguments and href.
	 *
	 * @param array  $args The arguments for generating the preload ID.
	 * @param string $href The href value used to generate the preload ID.
	 *
	 * @return string The generated preload ID.
	 */
	private function get_preload_id( $args, $href ) {
		$prefix = 'preload-';

		if ( isset( $args['rel'] ) ) {
			switch ( $args['rel'] ) {
				case 'preload':
					$prefix = 'preload-';
					break;
				case 'modulepreload':
					$prefix = 'modulepreload-';
					break;
				case 'prefetch':
					$prefix = 'prefetch-';
					break;
			}
		}

		if ( $args['id'] ) {
			return $prefix . self::slugify( $args['id'] );
		} else {
			return $prefix . self::slugify( basename( $href ) );
		}
	}

	/**
	 * Converts a string to a slug by replacing special characters with hyphens and converting it to lowercase.
	 *
	 * @param string $string The string to be slugified.
	 * @return string The slugified string.
	 */
	private static function slugify( $string ) {
		// Parse the URL to get the path component
		$parsed_url = parse_url( $string );
		$path       = isset( $parsed_url['path'] ) ? $parsed_url['path'] : $string;

		// Remove the .min suffix if present
		$path = str_replace( '.min', '', $path );

		// Replace unwanted characters with hyphens
		$slug = Utilities::get_slug_from_string( $path, '-' );

		// Remove any leading or trailing hyphens
		return trim( $slug, '-' );
	}

	/**
	 * Generates preload attributes for an image.
	 *
	 * @param array $args {
	 *   @type int    $id   The attachment ID.
	 *   @type string $size Optional. The image size. Default 'full'.
	 * }
	 * @return array Associative array of preload attributes.
	 */
	private static function get_preload_attributes_for_image( $args ) {
		$attributes = array(
			'as' => 'image',
		);

		if ( isset( $args['id'] ) ) {
			$size = isset( $args['size'] ) ? $args['size'] : 'full';

			$attrs = wp_get_attachment_image_src( $args['id'], $size );

			if ( $attrs && $attrs[0] ) {
				$attributes['href'] = $attrs[0];

				$srcset = wp_get_attachment_image_srcset( $args['id'], $size );
				$sizes  = wp_get_attachment_image_sizes( $args['id'], $size );

				if ( $srcset ) {
					$attributes['imagesrcset'] = $srcset;
				}

				if ( $sizes ) {
					$attributes['imagesizes'] = $sizes;
				}
			}
		}

		return $attributes;
	}
}
