<?php

namespace Phox\core\upgrade;

if ( ! defined( 'ABSPATH' ) ) {
    exit( 'Direct script access denied.' );
}

/**
 * Abstract class that provides a structure for managing theme version upgrades.
 *
 * Defines properties and methods necessary for checking and updating the theme version,
 * ensuring upgrade processes are executed when required.
 */
abstract class Upgrade_Abstract
{
    protected $version;

    protected $db_theme_version ;

    public function __construct()
    {
        $this->db_theme_version = get_option( 'phox_version', false );
        $this->set_version();

        $this->run();
        $this->update_current_version();

    }

    /**
     * Sets the version for the implementing class or system.
     *
     * This method is intended to define or update the version details,
     * and must be implemented by subclasses to specify their own versioning logic.
     *
     * @return void
     */
    abstract protected function set_version():void;

    /**
     * Executes the primary logic of the implementing class.
     *
     * This method must be implemented by any subclass, allowing them
     * to define their own specific behavior when the method is invoked.
     *
     * @return void
     */
    abstract public function run():void;

    /**
     * Updates the current version of the theme if the stored version is outdated.
     *
     * The method checks whether the current version of the theme exists and is not empty.
     * It also compares the stored version with the new version and updates the version
     * in the database if the new version is more recent.
     *
     * @return void
     */
    private function update_current_version():void
    {
        if ( ! $this->db_theme_version || empty( $this->db_theme_version ) ) {
            return;
        }
        if ( version_compare( $this->db_theme_version, $this->version, '>=' ) ) {
            return;
        }

        update_option( 'phox_version', $this->version );
    }
}