<?php
namespace Phox\core;

use Phox\core\admin\Status;
use function Sodium\compare;

!defined('ABSPATH') && exit;

require_once ABSPATH . 'wp-admin/includes/plugin.php';

class Upgrade {
    /**
     * The Singleton's instance is stored in a static field. This field is an
     * array, because we'll allow our Singleton to have subclasses. Each item in
     * this array will be an instance of a specific Singleton's subclass. You'll
     * see how this works in a moment.
     */
    private static $instances = [];

    /**
     * The theme version as stored in the db.
     *
     * @access private
     * @var mixed $database_theme_version
     */
    private $database_theme_version;

    /**
     * The previous version.
     *
     * @access private
     * @var string $previous_theme_version
     */
    private $previous_theme_version;

    /**
     * The current version.
     *
     * @access private
     * @var string
     */
    private $current_theme_version;

    /**
     * The status of the theme.
     *
     * @access private
     * @var mixed $thmem_status
     */
    private Status $thmem_status;

    public static function getInstance(): Upgrade
    {
        $cls = static::class;
        if (!isset(self::$instances[$cls])) {
            self::$instances[$cls] = new static();
        }

        return self::$instances[$cls];
    }

    public function __construct()
    {
        $upgrade = false;

        if ( is_admin())
        {
            $this->database_theme_version = get_option( 'phox_version', false );
            $this->current_theme_version = WDES_THEME_VERSION;
            $this->thmem_status = Status::instance();

            if(!$this->thmem_status->get_status()['php']){
                $this->stop_plugin();
            }else{
                $this->start_plugin();
            }

            if ( false === $this->database_theme_version  ) {
                $this->fresh_installation();
            }else{
                if(version_compare($this->database_theme_version, $this->current_theme_version, '<')){
                    $versions = [
                        '250' => '2.5.0',
                    ];

                    foreach ($versions as $version => $theme_version) {

                        $classname = '\Phox\core\upgrade\Upgrade_Migration_' . $version;

                        if($this->database_theme_version && version_compare( $this->database_theme_version, $theme_version, '<' ) ){
                            if ( class_exists( $classname ) ) {
                                new $classname();
                            }
                        }

                    }

                    $previous_versions = get_option('phox_previous_versions', false);

                    $previous_versions[] = $this->database_theme_version;

                    update_option('phox_previous_versions', $previous_versions);
                }
            }

        }

    }

    /**
     * Actions to run on a fresh installation.
     */
    public function fresh_installation() {
        update_option( 'phox_version', $this->current_theme_version );
    }

    /**
     * Deactivates the specified plugin if it is currently active.
     *
     * @return void
     */
    public function stop_plugin(): void
    {
        if (is_plugin_active('phox-host/phox-host.php')) {
            deactivate_plugins('phox-host/phox-host.php');
            set_transient('phox_plugin_was_stopped', true, 6 * HOUR_IN_SECONDS);
        }
    }
    
    

    /**
     * Display admin notice for PHP version incompatibility.
     *
     * @return void
     */
    public function display_php_version_notice(): void
    {
        echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($this->admin_notice_message) . '</p></div>';
    }

    /**
     * Reactivate plugin if it was previously stopped.
     *
     * @return void
     */
    public function start_plugin(): void
    {
        if (!is_plugin_active('phox-host/phox-host.php') && get_transient('phox_plugin_was_stopped')) {
            activate_plugins('phox-host/phox-host.php');
            delete_transient('phox_plugin_was_stopped');
        }
    }


}