( function( $ ) {
	'use strict';

	class ButtonActions {
		constructor( className, ignoreClassName = null ) {
			this.className = className;
			this.ignoreClassName = ignoreClassName;
			this.buttons = null;
			this.data = null;
			this.action = null;
			this.id = null;
			this.icon = null;
			this.handleClick();
		}
		handleClick = () => {
			document.addEventListener( 'click', event => {
				const button = event.target;
				if (
					!button.classList.contains( this.className )
					|| ( this.ignoreClassName && button.classList.contains( this.ignoreClassName ) )
				) {
					return;
				}

				if ( MBFP.nonLoggedIn ) {
					alert( MBFP.loginAlert );
					return;
				}

				this.id = button.dataset.id;
				this.icon = button.dataset.icon;
				this.buttons = document.querySelectorAll( `.${ this.className }[data-id="${ this.id }"]` );

				this.setData();
				this.beforeFetching()
					.then( this.fetch )
					.then( this.afterFetching );
			} );
		};
		setData = () => {
			this.data = new FormData();
			this.data.append( 'action', this.action );
			this.data.append( 'id', this.id );
			this.data.append( 'icon', this.icon );
			this.data.append( '_wpnonce', this.nonce );
		};
		beforeFetching = () => new Promise( ( resolve, reject ) => {
			this.buttons.forEach( button => button.classList.add( 'mbfp-loading' ) );
			resolve();
		} );
		fetch = () => fetch( MBFP.ajaxUrl, {
			method: "POST",
			credentials: 'same-origin',
			body: this.data,
		} );
		handleShowCount = ( button, count ) => {
			let spanCount = button.querySelector( '.mbfp-count' );

			if ( null !== spanCount ) {
				spanCount.innerHTML = count;
			}
		};
		handleIcon = ( button, iconColor, added = false ) => {
			let icon 	 = button.querySelector( '.mbfp-icon' );
			let oldIcon	 = button.dataset.icon;
			let iconSize = button.dataset.iconsize;

			if ( null !== icon ) {
				if( added ) {
					let addedIcon = oldIcon + 's';
					icon.outerHTML = '<svg class="mbfp-icon" xmlns="http://www.w3.org/2000/svg" width="' + iconSize + '" height="' + iconSize + '" viewBox="0 0 24 24" style="fill: ' + iconColor + '">' + MBFP.icon[addedIcon] + '</svg>';
				}
				else {
					icon.outerHTML = '<svg class="mbfp-icon" xmlns="http://www.w3.org/2000/svg" width="' + iconSize + '" height="' + iconSize + '" viewBox="0 0 24 24" style="fill: ' + iconColor + '">' + MBFP.icon[oldIcon] + '</svg>';
				}
			}
			button.classList.remove( 'mbfp-loading' );
		};
	}

	class ButtonAdd extends ButtonActions {
		constructor( className, ignoreClassName ) {
			super( className, ignoreClassName );
			this.action = 'mbfp_add';
			this.nonce = MBFP.addNonce;
		}

		afterFetching = response => response.json().then( ( { success, data } ) => {
			this.buttons.forEach( button => button.classList.remove( 'mbfp-loading' ) );

			if ( !success ) {
				alert( data );
				return;
			}

			this.buttons.forEach( button => {
				button.classList.add( 'mbfp-btn--added' );

				let text = button.querySelector( '.mbfp-text-detail' );
				if ( null !== text ) {
					text.innerHTML = button.dataset.added;
				}
				this.handleIcon( button, data.icon_color, true );
				this.handleShowCount( button, data.count );
			} );
		} );
	}

	class ButtonDelete extends ButtonActions {
		constructor( className, ignoreClassName ) {
			super( className, ignoreClassName );
			this.action = 'mbfp_delete';
			this.nonce = MBFP.deleteNonce;
		}

		afterFetching = response => response.json().then( ( res ) => {
			this.buttons.forEach( button => {
				button.classList.remove( 'mbfp-btn--added' );

				let text = button.querySelector( '.mbfp-text-detail' );
				if ( null !== text ) {
					text.innerHTML = button.dataset.add;
				}
				this.handleIcon( button, res.data.icon_color, false );
				this.handleShowCount( button, res.data.count );
			} );
		} );
	}

	class ButtonDeleteFromTable extends ButtonActions {
		constructor( buttons ) {
			super( buttons );
			this.action = 'mbfp_delete';
			this.nonce = MBFP.deleteNonce;
			this.table = document.querySelector( '.mbfp-posts' );
		}
		beforeFetching = () => new Promise( ( resolve, reject ) => {
			const confirmDelete = confirm( MBFP.confirmDelete );
			if ( confirmDelete ) {
				this.table.classList.add( 'mbfp-loading' );
				resolve();
			} else {
				reject();
			}
		} );
		afterFetching = response => response.json().then( ( res ) => {
			this.table.classList.remove( 'mbfp-loading' );
			this.table.querySelector( `tr[data-id="${ this.id }"]` ).remove();

			if ( res.data.empty_notice ) {
				const noPostNotice = document.createElement( 'div' );
				noPostNotice.classList.add( 'mbfp-notice' );
				noPostNotice.textContent = res.data.empty_notice;
				this.table.after( noPostNotice );
				this.table.remove();
			}
		} );
	}

	new ButtonAdd( 'mbfp-btn', 'mbfp-btn--added' );
	new ButtonDelete( 'mbfp-btn--added' );
	new ButtonDeleteFromTable( 'mbfp-table__delete' );
}( jQuery ) );