<?php

namespace WPFEPP\Components;

if ( ! defined( 'WPINC' ) ) {
	die;
}

use WPFEPP\Constants\Email_Placeholders;
use WPFEPP\Constants\Email_Settings;
use WPFEPP\Constants\Plugin_Actions;
use WPFEPP\Constants\Post_Meta_Keys;
use WPFEPP\Forms\Frontend_Form;
use WPGurus\Components\Component;

/**
 * Sends emails to the user and the site administrator on form submissions.
 * @package WPFEPP\Components
 */
class Email_Manager extends Component {
	/**
	 * Email_Manager constructor.
	 */
	function __construct() {
		parent::__construct();

		$this->register_action( Plugin_Actions::POST_SUBMITTED, array( $this, 'handle_form_submission' ), 10, 2 );
		$this->register_action( 'pending_to_publish', array( $this, 'handle_post_status_change' ) );
		$this->register_action( 'pending_to_draft', array( $this, 'handle_post_status_change' ) );
	}

	/**
	 * Sends emails.
	 *
	 * @param $post_data array The post data submitted by the user.
	 * @param $form Frontend_Form The form that handled the user's request.
	 */
	function handle_form_submission( $post_data, $form ) {
		$post_id = $post_data['ID'];
		$author_email = $this->get_post_author_email( $post_id );
		$admin_email = get_bloginfo( 'admin_email' );
		$email_settings = $form->get_form_emails();

		$author_email_sent = false;
		if ( $email_settings[ Email_Settings::SETTING_USER_EMAILS ] ) {
			$author_email_sent = $this->send_email(
				$author_email,
				$email_settings[ Email_Settings::SETTING_USER_EMAIL_SUBJECT ],
				$email_settings[ Email_Settings::SETTING_USER_EMAIL_CONTENT ],
				$post_id,
				$form
			);
		}

		if ( $email_settings[ Email_Settings::SETTING_ADMIN_EMAILS ] && ! ( $author_email == $admin_email && $author_email_sent ) ) {
			$this->send_email(
				$admin_email,
				$email_settings[ Email_Settings::SETTING_ADMIN_EMAIL_SUBJECT ],
				$email_settings[ Email_Settings::SETTING_ADMIN_EMAIL_CONTENT ],
				$post_id,
				$form
			);
		}
	}

	private function send_email( $to, $subject, $content, $post_id, $form ) {
		$email_settings = $form->get_form_emails();
		$from_email = function ( $email ) use ( $email_settings ) {
			$sender_address_setting = $email_settings[ Email_Settings::SETTING_SENDER_ADDRESS ];
			if ( $sender_address_setting && is_email( $sender_address_setting ) ) {
				return $sender_address_setting;
			}

			return $email;
		};
		$from_name = function ( $name ) use ( $email_settings ) {
			$sender_name_setting = $email_settings[ Email_Settings::SETTING_SENDER_NAME ];
			if ( $sender_name_setting ) {
				return $sender_name_setting;
			}

			return $name;
		};
		$content_type = function () use ( $email_settings ) {
			return 'text/' . $email_settings[ Email_Settings::SETTING_EMAIL_FORMAT ];
		};

		add_filter( 'wp_mail_from', $from_email, 9999 );
		add_filter( 'wp_mail_from_name', $from_name, 9999 );
		add_filter( 'wp_mail_content_type', $content_type, 9999 );

		$subject = $this->prepare( $subject, $post_id );
		$content = $this->prepare( $content, $post_id );

		$email_sent = wp_mail( $to, $subject, $content );

		remove_filter( 'wp_mail_from', $from_email, 9999 );
		remove_filter( 'wp_mail_from_name', $from_name, 9999 );
		remove_filter( 'wp_mail_content_type', $content_type, 9999 );

		return $email_sent;
	}

	/**
	 * @param $post \WP_Post
	 */
	public function handle_post_status_change( $post ) {
		if ( $post->post_status === 'publish' ) {
			$this->send_status_change_email(
				$post,
				Email_Settings::SETTING_USER_PENDING_TO_PUBLISH_EMAILS,
				Email_Settings::SETTING_USER_PENDING_TO_PUBLISH_EMAIL_SUBJECT,
				Email_Settings::SETTING_USER_PENDING_TO_PUBLISH_EMAIL_CONTENT
			);
		} else {
			$this->send_status_change_email(
				$post,
				Email_Settings::SETTING_USER_PENDING_TO_DRAFT_EMAILS,
				Email_Settings::SETTING_USER_PENDING_TO_DRAFT_EMAIL_SUBJECT,
				Email_Settings::SETTING_USER_PENDING_TO_DRAFT_EMAIL_CONTENT
			);
		}
	}

	/**
	 * @param $post \WP_Post
	 * @param $enabled_key
	 * @param $subject_key
	 * @param $content_key
	 */
	private function send_status_change_email( $post, $enabled_key, $subject_key, $content_key ) {
		$form_id = get_post_meta( $post->ID, Post_Meta_Keys::FORM_ID, true );
		if ( ! $form_id ) {
			return;
		}

		$form = new Frontend_Form( $form_id );
		if ( ! $form->is_valid() ) {
			return;
		}

		$email_settings = $form->get_form_emails();
		if ( ! $email_settings[ $enabled_key ] ) {
			return;
		}

		$this->send_email(
			$this->get_post_author_email( $post->ID ),
			$email_settings[ $subject_key ],
			$email_settings[ $content_key ],
			$post->ID, $form
		);
	}

	/**
	 * Prepares a string by replacing all the placeholders with real values.
	 *
	 * @param $value string The value that needs to be prepared.
	 * @param $post_id int The user submitted data.
	 *
	 * @return string A string with all the placeholders replaced with real values.
	 */
	private function prepare( $value, $post_id ) {
		$admin_info = get_userdata( 1 );
		$author_id = $this->get_post_author_id( $post_id );

		$place_holders[ Email_Placeholders::PLACEHOLDER_POST_TITLE ] = wp_strip_all_tags( \get_post_field( 'post_title', $post_id ) );
		$place_holders[ Email_Placeholders::PLACEHOLDER_POST_PERMALINK ] = get_post_permalink( $post_id );
		$place_holders[ Email_Placeholders::PLACEHOLDER_AUTHOR_NAME ] = get_the_author_meta( 'display_name', $author_id );
		$place_holders[ Email_Placeholders::PLACEHOLDER_SITE_NAME ] = html_entity_decode( get_bloginfo( 'name' ), ENT_QUOTES );
		$place_holders[ Email_Placeholders::PLACEHOLDER_SITE_URL ] = get_bloginfo( 'url' );
		$place_holders[ Email_Placeholders::PLACEHOLDER_ADMIN_NAME ] = $admin_info->display_name;
		$place_holders[ Email_Placeholders::PLACEHOLDER_EDIT_LINK ] = sprintf( admin_url( 'edit.php?post_type=%s' ), \get_post_field( 'post_type', $post_id ) );

		foreach ( $place_holders as $placeholder => $val ) {
			$value = str_replace( $placeholder, $val, $value );
		}
		return $value;
	}

	private function get_post_author_id( $post_id ) {
		return get_post_field( 'post_author', $post_id );
	}

	private function get_post_author_email( $post_id ) {
		return get_the_author_meta( 'user_email', $this->get_post_author_id( $post_id ) );
	}
}