<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Elementor_Olympus_News_Grid extends \Elementor\Widget_Base {

	public function get_name() {
		return 'olympus_news_grid';
	}

	public function get_title() {
		return esc_html__( 'News Grid', 'elementor-olympus' );
	}

	public function get_icon() {
		return 'crum-el-w-post-portfolio-carousel-with-image';
	}

	public function get_categories() {
		return [ 'elementor-olympus' ];
	}

	protected function register_controls() {

		$post_taxonomy     = [];
		$all_post_taxonomy = es_post_taxonomy();
		if ( $all_post_taxonomy ) {
			foreach ( $all_post_taxonomy as $post_type => $terms ) {
				$post_taxonomy[ $post_type ] = ucwords( str_replace( [ '-', '_' ], ' ', $post_type ) );
			}
		}

		$this->start_controls_section(
			'olympus_news_grid',
			[
				'label' => esc_html__( 'News Grid', 'elementor-olympus' )
			]
		);

		$this->add_control(
			'post_taxonomy',
			[
				'type'      => \Elementor\Controls_Manager::SELECT,
				'label'     => esc_html__( 'Content Type', 'elementor-olympus' ),
				'default'   => 'post',
				'options'   => $post_taxonomy,
				'separator' => 'before'
			]
		);

		if ( $all_post_taxonomy ) {
			foreach ( $all_post_taxonomy as $post_type => $terms ) {
				$this->add_control(
					'post_taxonomy_' . $post_type,
					[
						'type'        => \Elementor\Controls_Manager::SELECT2,
						'label'       => esc_html__( 'Select content category', 'elementor-olympus' ),
						'options'     => $terms,
						'multiple'    => true,
						'label_block' => true,
						'condition'   => [
							'post_taxonomy' => $post_type
						],
						'separator'   => 'before'
					]
				);
			}
		}

		$this->add_control(
			'order_by',
			[
				'type'      => \Elementor\Controls_Manager::SELECT,
				'label'     => esc_html__( 'Order by', 'elementor-olympus' ),
				'default'   => 'ID',
				'options'   => array(
					'ID'            => esc_html__( 'Post ID', 'elementor-olympus' ),
					'author'        => esc_html__( 'Author', 'elementor-olympus' ),
					'title'         => esc_html__( 'Title', 'elementor-olympus' ),
					'name'          => esc_html__( 'Post name (post slug)', 'elementor-olympus' ),
					'type'          => esc_html__( 'Post type (available since Version 4.0)', 'elementor-olympus' ),
					'date'          => esc_html__( 'Date', 'elementor-olympus' ),
					'modified'      => esc_html__( 'Last modified date', 'elementor-olympus' ),
					'rand'          => esc_html__( 'Random order', 'elementor-olympus' ),
					'comment_count' => esc_html__( 'Number of comments', 'elementor-olympus' )
				),
				'separator' => 'before'
			]
		);

		$this->add_control(
			'order_list',
			[
				'type'      => \Elementor\Controls_Manager::SELECT,
				'label'     => esc_html__( 'Order', 'elementor-olympus' ),
				'default'   => 'ASC',
				'options'   => array(
					'ASC'  => esc_html__( 'ASC', 'elementor-olympus' ),
					'DESC' => esc_html__( 'DESC', 'elementor-olympus' ),
				),
				'separator' => 'before'
			]
		);

		$this->add_control(
			'number_post',
			[
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Number of items displayed', 'elementor-olympus' ),
				'description' => esc_html__( 'The number of items you want to show.', 'elementor-olympus' ),
				'default'     => [
					'size' => 9,
					'unit' => '%',
				],
				'range'       => [
					'%' => [
						'min' => 1,
						'max' => 30,
					],
				],
				'separator'   => 'before'
			]
		);

		$this->add_control(
			'number_of_items',
			[
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Items per row', 'elementor-olympus' ),
				'description' => esc_html__( 'Number of items displayed on one row', 'elementor-olympus' ),
				'default'     => [
					'size' => 3,
					'unit' => '%',
				],
				'range'       => [
					'%' => [
						'min' => 1,
						'max' => 6,
					],
				],
				'separator'   => 'before'
			]
		);

		$this->add_control(
			'custom_class',
			[
				'type'        => \Elementor\Controls_Manager::TEXT,
				'label'       => esc_html__( 'Extra class', 'elementor-olympus' ),
				'description' => esc_html__( 'If you wish to style a particular content element differently, please add a class name to this field and refer to it in your custom CSS file.', 'elementor-olympus' ),
				'separator'   => 'before'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title-css',
			[
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'label' => esc_html__( 'Title', 'elementor-olympus' )
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .entry-title',
			]
		);

		$this->add_control(
			'title-color',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => esc_html__( 'Color', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .entry-title' => 'color: {{SCHEME}};'
				],
				'separator' => 'before'
			]
		);

		$this->add_control(
			'title-color-hover',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => esc_html__( 'Color on hover', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .post-block:hover .entry-title' => 'color: {{SCHEME}};'
				],
				'separator' => 'before'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'text-css',
			[
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'label' => esc_html__( 'Text', 'elementor-olympus' )
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'text_typography',
				'selector' => '{{WRAPPER}} .entry-text',
			]
		);

		$this->add_control(
			'text-color',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => esc_html__( 'Color', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .entry-text' => 'color: {{SCHEME}};'
				],
				'separator' => 'before'
			]
		);

		$this->add_control(
			'text-color-hover',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => esc_html__( 'Color on hover', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .post-block:hover .entry-text' => 'color: {{SCHEME}};'
				],
				'separator' => 'before'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'image-box',
			[
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'label' => esc_html__( 'Image', 'elementor-olympus' )
			]
		);

		$this->add_control(
			'image-width',
			[
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'label'      => __( 'Width', 'elementor-olympus' ),
				'size_units' => [ 'px', 'em', '%' ],
				'range'      => [
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
					'em' => [
						'min' => 1,
						'max' => 100,
					],
					'%'  => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .post-thumb img' => 'width: {{SIZE}}{{UNIT}};',
				],
				'separator'  => 'before'
			]
		);

		$this->add_control(
			'image-height',
			[
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'label'      => __( 'Height', 'elementor-olympus' ),
				'size_units' => [ 'px', 'em', '%' ],
				'range'      => [
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
					'em' => [
						'min' => 1,
						'max' => 100,
					],
					'%'  => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .post-thumb img' => 'height: {{SIZE}}{{UNIT}};',
				],
				'separator'  => 'before'
			]
		);

		$this->add_control(
			'image-background-color',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => __( 'Background Color', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .post-thumb' => 'background-color: {{VALUE}};',
				],
				'separator' => 'before'
			]
		);

		$this->add_group_control(
			'border',
			[
				'name'      => 'image-border',
				'label'     => esc_html__( 'Border', 'elementor-olympus' ),
				'selector'  => '{{WRAPPER}} .post-thumb',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'image-border-radius',
			[
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'label'      => __( 'Border Radius', 'elementor-olympus' ),
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .post-thumb, {{WRAPPER}} .post-thumb img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'  => [
					'border_border!' => '',
				],
			]
		);

		$this->add_control(
			'image-padding',
			[
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'label'      => __( 'Padding', 'elementor-olympus' ),
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .post-thumb' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator'  => 'before'
			]
		);

		$this->add_control(
			'image-margin',
			[
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'label'      => __( 'Margin', 'elementor-olympus' ),
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .post-thumb' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator'  => 'before'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style-box',
			[
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'label' => esc_html__( 'Box style', 'elementor-olympus' )
			]
		);

		$this->add_control(
			'style-background-color',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => __( 'Background Color', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .post-block' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'style-background-color-hover',
			[
				'type'      => \Elementor\Controls_Manager::COLOR,
				'label'     => __( 'Background Color on hover', 'elementor-olympus' ),
				'selectors' => [
					'{{WRAPPER}} .post-block:hover' => 'background-color: {{VALUE}};',
				],
				'separator' => 'before'
			]
		);

		$this->add_control(
			'style-align',
			[
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'label'     => esc_html__( 'Text Align', 'elementor-olympus' ),
				'options'   => [
					'left'    => [
						'title' => esc_html__( 'Left', 'elementor-olympus' ),
						'icon'  => 'fa fa-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Centered', 'elementor-olympus' ),
						'icon'  => 'fa fa-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'elementor-olympus' ),
						'icon'  => 'fa fa-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justify', 'elementor-olympus' ),
						'icon'  => 'fa fa-align-justify',
					]
				],
				'selectors' => [
					'{{WRAPPER}} .post-block' => 'text-align: {{VALUE}};',
				],
				'default'   => 'left',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'style-padding',
			[
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'label'      => __( 'Padding', 'elementor-olympus' ),
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .post-block' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator'  => 'before'
			]
		);

		$this->add_control(
			'style-margin',
			[
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'label'      => __( 'Margin', 'elementor-olympus' ),
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .post-block' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator'  => 'before'
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$orderby         = ! empty( $settings['order_by'] ) ? $settings['order_by'] : 'ID';
		$order           = ! empty( $settings['order_list'] ) ? $settings['order_list'] : 'ASC';
		$post_type       = ! empty( $settings['post_taxonomy'] ) ? $settings['post_taxonomy'] : 'fw-news';
		$terms           = ! empty( $settings[ 'post_taxonomy_' . $post_type ] ) ? $settings[ 'post_taxonomy_' . $post_type ] : [];
		$posts_per_page  = ! empty( $settings['number_post']['size'] ) ? $settings['number_post']['size'] : 3;
		$number_of_items = ! empty( $settings['number_of_items']['size'] ) ? $settings['number_of_items']['size'] : 3;

		$taxonomy_objects = get_object_taxonomies( $post_type, 'objects' );
		$taxonomy         = key( $taxonomy_objects );

		$args = array(
			'post_type'        => $post_type,
			'posts_per_page'   => $posts_per_page,
			'orderby'          => $orderby,
			'order'            => $order,
			'suppress_filters' => false
		);

		if ( $terms ) {
			$args['tax_query'] = array(
				'relation' => 'OR',
				array(
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $terms,
				)
			);
		}

		$the_query = new WP_Query( $args );

		$wrap_class = [ 'post-blocks', 'news-grid' ];
		if ( ! empty( $settings['custom_class'] ) ) {
			$wrap_class[] = $settings['custom_class'];
		}

		// news format settings
		$container_width = 1280;
		$gap_paddings    = 20;
		$grid_size       = intval( 12 / $number_of_items );
		$img_width       = intval( $container_width / ( 12 / $grid_size ) ) - $gap_paddings;
		$img_height      = intval( $img_width * 0.75 );
		$default_src     = get_template_directory_uri() . '/images/post-no-image.png';
		$item_class_add  = $grid_size > 5 ? 'big mb60' : 'mb30';


		ob_start();
		$i = 1;
		if ( $the_query->have_posts() ) {

			while ( $the_query->have_posts() ) {
				$the_query->the_post();
				$thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'large');
				?>
				<div class="col-lg-<?php echo esc_attr( $grid_size ); ?> col-md-<?php echo esc_attr( $grid_size ); ?> col-sm-6 col-xs-12">

					<div class="ui-block post-block <?php echo esc_attr( $item_class_add ) ?>">
						<article class="hentry blog-post blog-post ajax-post-full-height">
							<div class="post-thumb">
								<a href="<?php echo esc_url( get_the_permalink() ) ?>">
									<?php if ( has_post_thumbnail() ) {
										the_post_thumbnail( array( $img_width, $img_height ) );
									} else {
										olympus_html_tag( 'img', array(
											'src'    => $default_src,
											'width'  => $img_width,
											'height' => $img_height,
											'alt'    => ''

										) );
									}
									?>
								</a>

							</div>
							<div class="post-content ajax-col-content-equal-height">
								<?php 
								if ( function_exists( 'olympus_post_category_list' ) ) {
									echo olympus_post_category_list( get_the_ID(), ' ', true );
								}
								?>

								<?php the_title( '<a href="' . esc_url( get_permalink() ) . '" rel="bookmark" class="h4 entry-title post-title">', '</a>' ); ?>
								<div class="entry-text">
									<?php 
									if ( function_exists( 'olympus_html_tag' ) && function_exists( 'olympus_generate_short_excerpt' ) ) {
										echo olympus_html_tag( 'p', array(), olympus_generate_short_excerpt( get_the_ID(), 30, false ) );
									}
									?>
								</div>
								<div class="author-date">
									<?php esc_html_e( 'by', 'olympus' ); ?> <?php if ( function_exists( 'olympus_post_author' ) ) { olympus_post_author( false ); } ?>
									- <?php if ( function_exists( 'olympus_posted_time' ) ) { olympus_posted_time(); } ?>
								</div>

							</div>
							<div class="post-additional-info inline-items">
								<?php
								if ( function_exists( 'olympus_get_post_reactions' ) ) {
									echo olympus_get_post_reactions( 'compact' );
								}
								?>
								<div class="comments-shared">
									<?php
									if ( function_exists( 'olympus_comments_count' ) ) {
										olympus_comments_count();
									}
									?>
								</div>
							</div>
						</article>
					</div>
				</div>
				<?php
				if ( 0 === $i % $number_of_items ) {
					echo '<div class="clearfix"></div>';
				}
				$i ++;
			}
		} else {
			echo '<div class="col-xs-12"><h2>' . esc_html__( ' No posts found', 'elementor-olympus' ) . '</h2></div>';
		}
		wp_reset_query();

		$output = ob_get_clean();
		?>
		<div class="<?php echo esc_attr( implode( ' ', $wrap_class ) ) ?>">
			<div class="row post-list">
				<?php echo $output; ?>
			</div>
		</div>
		<?php
	}
}