<?php


/**
 * Class dima_media_support - DIMA media (video) support V 1.0
 */
class dima_media_support {

	/**
	 * Returns the video thumb url from the video URL
	 *
	 * @param $videoUrl
	 *
	 * @return string
	 */
	static function dima_get_thumb_url( $videoUrl ) {

		switch ( self::dima_detect_video_service( $videoUrl ) ) {
			case 'youtube':
				$maxresdefault = dima_helper::dima_https_or_http() . '://img.youtube.com/vi/' . self::dima_get_youtube_id( $videoUrl ) . '/maxresdefault.jpg';
				$sddefault     = dima_helper::dima_https_or_http() . '://img.youtube.com/vi/' . self::dima_get_youtube_id( $videoUrl ) . '/sddefault.jpg';
				$hqdefault     = dima_helper::dima_https_or_http() . '://img.youtube.com/vi/' . self::dima_get_youtube_id( $videoUrl ) . '/hqdefault.jpg';

				if ( ! self::dima_is_404( $maxresdefault ) ) {
					return $maxresdefault;
				} elseif ( ! self::dima_is_404( $sddefault ) ) {
					return $sddefault;
				} elseif ( ! self::dima_is_404( $hqdefault ) ) {
					return $hqdefault;
				}
				break;

			case 'vimeo':
				// @todo e stricat nu mai merge de ceva timp cred
				$url = 'http://vimeo.com/api/oembed.json?url=https://vimeo.com/' . self::dima_get_vimeo_id( $videoUrl );

				$response = wp_remote_get(
					$url,
					array(
						'timeout'    => 10,
						'sslverify'  => false,
						'user-agent' => 'Mozilla/5.0 (Windows NT 6.3; WOW64; rv:35.0) Gecko/20100101 Firefox/35.0',
					)
				);

				if ( ! is_wp_error( $response ) ) {
					$dima_result = json_decode( wp_remote_retrieve_body( $response ) );

					return ( $dima_result->thumbnail_url );
				}
				break;
		}

		return '';
	}

	static function dima_get_metadata( $videoUrl ) {
		switch ( self::dima_detect_video_service( $videoUrl ) ) {
			case 'youtube':
				$url      = 'https://www.youtube.com/oembed?url=https://www.youtube.com/watch?v=' . self::dima_get_youtube_id( $videoUrl ) . '&format=json';
				$response = wp_remote_get(
					$url,
					array(
						'timeout'    => 10,
						'sslverify'  => false,
						'user-agent' => 'Mozilla/5.0 (Windows NT 6.3; WOW64; rv:35.0) Gecko/20100101 Firefox/35.0',
					)
				);
				if ( ! is_wp_error( $response ) ) {
					$dima_result = json_decode( wp_remote_retrieve_body( $response ) );

					return ( $dima_result );
				}

				break;
			case 'vimeo':
				// eg. https://vimeo.com/api/oembed.json?url=https%3A//vimeo.com/203689226
				$url      = 'http://vimeo.com/api/oembed.json?url=https://vimeo.com/' . self::dima_get_vimeo_id( $videoUrl );
				$response = wp_remote_get(
					$url,
					array(
						'timeout'    => 10,
						'sslverify'  => false,
						'user-agent' => 'Mozilla/5.0 (Windows NT 6.3; WOW64; rv:35.0) Gecko/20100101 Firefox/35.0',
					)
				);
				if ( ! is_wp_error( $response ) ) {
					$dima_result = json_decode( wp_remote_retrieve_body( $response ) );

					return ( $dima_result );
				}

				break;
		}
	}

	/*
	 * youtube
	 */
	private static function dima_get_youtube_id( $videoUrl ) {
		$query_string = array();
		parse_str( parse_url( $videoUrl, PHP_URL_QUERY ), $query_string );

		if ( empty( $query_string['v'] ) ) {
			// explode at ? mark
			$yt_short_link_parts_explode1 = explode( '?', $videoUrl );

			// short link: http://youtu.be/AgFeZr5ptV8
			$yt_short_link_parts = explode( '/', $yt_short_link_parts_explode1[0] );
			if ( ! empty( $yt_short_link_parts[3] ) ) {
				return $yt_short_link_parts[3];
			}

			return $yt_short_link_parts[0];
		} else {
			return $query_string['v'];
		}
	}

	/*
	 * Vimeo id
	 */
	private static function dima_get_vimeo_id( $videoUrl ) {
		sscanf( parse_url( $videoUrl, PHP_URL_PATH ), '/%d', $video_id );

		return $video_id;
	}

	/*
	 * Detect the video service from url
	 */
	static function dima_detect_video_service( $videoUrl ) {
		$videoUrl = strtolower( $videoUrl );
		if ( strpos( $videoUrl, 'youtube.com' ) !== false or strpos( $videoUrl, 'youtu.be' ) !== false ) {
			return 'youtube';
		}
		if ( strpos( $videoUrl, 'dailymotion.com' ) !== false ) {
			return 'dailymotion';
		}
		if ( strpos( $videoUrl, 'vimeo.com' ) !== false ) {
			return 'vimeo';
		}

		return false;
	}

	/**
	 * @param $url
	 *
	 * @return bool
	 */
	private static function dima_is_404( $url ) {
		$headers = get_headers( $url );
		if ( ! empty( $headers[0] ) and strpos( $headers[0], '404' ) !== false ) {
			return true;
		}

		return false;
	}
}

/**
 * @param array $params
 *
 * @since 2.2
 * vc_filter: dima_wpb_getimagesize - to override output of this function
 * @return array|bool
 */
function dima_get_image_by_size( $params = array() ) {
	$params = array_merge(
		array(
			'post_id'    => null,
			'attach_id'  => null,
			'thumb_size' => 'thumbnail',
			'class'      => '',
			'alt'        => '',
			'caption'    => '',
		),
		$params
	);

	if ( ! $params['thumb_size'] ) {
		$params['thumb_size'] = 'thumbnail';
	}

	if ( ! $params['attach_id'] && ! $params['post_id'] ) {
		return false;
	}

	$post_id = $params['post_id'];

	$attach_id   = $post_id ? get_post_thumbnail_id( $post_id ) : $params['attach_id'];
	$attach_id   = apply_filters( 'dima_object_id', $attach_id );
	$thumb_size  = $params['thumb_size'];
	$thumb_class = ( isset( $params['class'] ) && '' !== $params['class'] ) ? $params['class'] . ' ' : '';
	$caption     = $params['caption'];
	global $_wp_additional_image_sizes;
	$thumbnail = $attributes = '';

	if ( is_string( $thumb_size ) && ( ( ! empty( $_wp_additional_image_sizes[ $thumb_size ] ) && is_array( $_wp_additional_image_sizes[ $thumb_size ] ) ) || in_array(
		$thumb_size,
		array(
			'thumbnail',
			'thumb',
			'medium',
			'large',
			'full',
		)
	) )
	) {
		$attributes = array( 'class' => $thumb_class . 'attachment-' . $thumb_size );
		$thumbnail  = wp_get_attachment_image( $attach_id, $thumb_size, false, $attributes );
		if ( $caption ) {
			$thumbnail = '<figure class="wp-caption alignnone" >' . $thumbnail . '<figcaption class="wp-caption-text">' . $caption . '</figcaption></figure>';
		}
	} elseif ( $attach_id ) {
		if ( is_string( $thumb_size ) ) {
			preg_match_all( '/\d+/', $thumb_size, $thumb_matches );
			if ( isset( $thumb_matches[0] ) ) {
				$thumb_size = array();
				$count      = count( $thumb_matches[0] );
				if ( $count > 1 ) {
					$thumb_size[] = $thumb_matches[0][0]; // width
					$thumb_size[] = $thumb_matches[0][1]; // height
				} elseif ( 1 === $count ) {
					$thumb_size[] = $thumb_matches[0][0]; // width
					$thumb_size[] = $thumb_matches[0][0]; // height
				} else {
					$thumb_size = false;
				}
			}
		}
		if ( is_array( $thumb_size ) ) {
			// Resize image to custom size
			$p_img = wpdima_resize( $attach_id, null, $thumb_size[0], $thumb_size[1], true );
			if ( ! $params['alt'] ) {
				$alt = trim( strip_tags( get_post_meta( $attach_id, '_wp_attachment_image_alt', true ) ) );
			} else {
				$alt = trim( strip_tags( $params['alt'] ) );
			}

			$attachment = get_post( $attach_id );
			if ( ! empty( $attachment ) ) {
				$title = trim( strip_tags( $attachment->post_title ) );

				if ( empty( $alt ) ) {
					$alt = trim( strip_tags( $attachment->post_excerpt ) ); // If not, Use the Caption
				}
				if ( empty( $alt ) ) {
					$alt = $title;
				} // Finally, use the title
				if ( $p_img ) {

					$attributes  = array(
						'class'  => $thumb_class,
						'src'    => $p_img['url'],
						'width'  => $p_img['width'],
						'height' => $p_img['height'],
						'alt'    => $alt,
						'title'  => $title,
					);
					$_attributes = dima_stringify_attributes( $attributes );
					if ( ! $caption ) {
						$thumbnail = '<img ' . $_attributes . ' />';
					} else {
						$thumbnail = '<figure class="wp-caption alignnone" ><img ' . $_attributes . ' /><figcaption class="wp-caption-text">' . $caption . '</figcaption></figure>';
					}
				}
			}
		}
	}

	$p_img_large = wp_get_attachment_image_src( $attach_id, 'large' );

	return apply_filters(
		'dima_wpdima_getimagesize',
		array(
			'thumbnail'   => $thumbnail,
			'attributes'  => $attributes,
			'p_img_large' => $p_img_large,
		),
		$attach_id,
		$params
	);
}

/*
* Resize images dynamically using wp built in functions
* Victor Teixeira
*
* php 5.2+
*
* Exemplo de uso:
*
* <?php
* $thumb = get_post_thumbnail_id();
* $image = vt_resize( $thumb, '', 140, 110, true );
* ?>
* <img src="<?php echo $image[url]; ?>" width="<?php echo $image[width]; ?>" height="<?php echo $image[height]; ?>" />
*
*/
if ( ! function_exists( 'wpdima_resize' ) ) {
	/**
	 * @param int    $attach_id
	 * @param string $img_url
	 * @param int    $width
	 * @param int    $height
	 * @param bool   $crop
	 *
	 * @since 4.2
	 * @return array
	 */
	function wpdima_resize( $attach_id = null, $img_url = null, $width, $height, $crop = false ) {
		// this is an attachment, so we have the ID
		$image_src = array();
		if ( $attach_id ) {
			$image_src        = wp_get_attachment_image_src( $attach_id, 'full' );
			$actual_file_path = get_attached_file( $attach_id );
			// this is not an attachment, let's use the image url
		} elseif ( $img_url ) {
			$file_path        = parse_url( $img_url );
			$actual_file_path = rtrim( ABSPATH, '/' ) . $file_path['path'];
			$orig_size        = getimagesize( $actual_file_path );
			$image_src[0]     = $img_url;
			$image_src[1]     = $orig_size[0];
			$image_src[2]     = $orig_size[1];
		}
		if ( ! empty( $actual_file_path ) ) {
			$file_info = pathinfo( $actual_file_path );
			$extension = '.' . $file_info['extension'];

			// the image path without the extension
			$no_ext_path = $file_info['dirname'] . '/' . $file_info['filename'];

			$cropped_img_path = $no_ext_path . '-' . $width . 'x' . $height . $extension;

			// checking if the file size is larger than the target size
			// if it is smaller or the same size, stop right here and return
			if ( $image_src[1] > $width || $image_src[2] > $height ) {

				// the file is larger, check if the resized version already exists (for $crop = true but will also work for $crop = false if the sizes match)
				if ( file_exists( $cropped_img_path ) ) {
					$cropped_img_url = str_replace( basename( $image_src[0] ), basename( $cropped_img_path ), $image_src[0] );
					$vt_image        = array(
						'url'    => $cropped_img_url,
						'width'  => $width,
						'height' => $height,
					);

					return $vt_image;
				}

				if ( false == $crop ) {
					// calculate the size proportionaly
					$proportional_size = wp_constrain_dimensions( $image_src[1], $image_src[2], $width, $height );
					$resized_img_path  = $no_ext_path . '-' . $proportional_size[0] . 'x' . $proportional_size[1] . $extension;

					// checking if the file already exists
					if ( file_exists( $resized_img_path ) ) {
						$resized_img_url = str_replace( basename( $image_src[0] ), basename( $resized_img_path ), $image_src[0] );

						$vt_image = array(
							'url'    => $resized_img_url,
							'width'  => $proportional_size[0],
							'height' => $proportional_size[1],
						);

						return $vt_image;
					}
				}

				// no cache files - let's finally resize it
				$img_editor = wp_get_image_editor( $actual_file_path );

				if ( is_wp_error( $img_editor ) || is_wp_error( $img_editor->resize( $width, $height, $crop ) ) ) {
					return array(
						'url'    => '',
						'width'  => '',
						'height' => '',
					);
				}

				$new_img_path = $img_editor->generate_filename();

				if ( is_wp_error( $img_editor->save( $new_img_path ) ) ) {
					return array(
						'url'    => '',
						'width'  => '',
						'height' => '',
					);
				}
				if ( ! is_string( $new_img_path ) ) {
					return array(
						'url'    => '',
						'width'  => '',
						'height' => '',
					);
				}

				$new_img_size = getimagesize( $new_img_path );
				$new_img      = str_replace( basename( $image_src[0] ), basename( $new_img_path ), $image_src[0] );

				// resized output
				$vt_image = array(
					'url'    => $new_img,
					'width'  => $new_img_size[0],
					'height' => $new_img_size[1],
				);

				return $vt_image;
			}

			// default output - without resizing
			$vt_image = array(
				'url'    => $image_src[0],
				'width'  => $image_src[1],
				'height' => $image_src[2],
			);

			return $vt_image;
		}

		return false;
	}
}


/**
 * Convert array of named params to string version
 * All values will be escaped
 *
 * E.g. f(array('name' => 'foo', 'id' => 'bar')) -> 'name="foo" id="bar"'
 *
 * @param $attributes
 *
 * @return string
 */
function dima_stringify_attributes( $attributes ) {
	$atts = array();
	foreach ( $attributes as $name => $value ) {
		$atts[] = $name . '="' . esc_attr( $value ) . '"';
	}

	return implode( ' ', $atts );
}
