<?php
/**
 * The main Dima library object.
 *
 * @package Dima-Library
 * @since 3.6.0
 */

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'Direct script access denied.' );
}

/**
 * The main Dima library object.
 */
class Dima {
	/**
	 * The one, true instance of the object.
	 *
	 * @static
	 * @access public
	 * @var null|object
	 */
	public static $instance = null;

	/**
	 * dima_Remote_Installer.
	 *
	 * @access public
	 * @var object dima_Remote_Installer.
	 */
	public $remote_install;

	/**
	 * The theme version.
	 *
	 * @static
	 * @access public
	 * @var string
	 */
	public static $version = DIMA_VERSION;

	/**
	 * Bundled Plugins.
	 *
	 * @static
	 * @access public
	 * @var array
	 */
	public static $bundled_plugins = array();

	/**
	 * The original option name.
	 * This is the untainted option name, without using any languages.
	 * If you want the property including language, use $option_name instead.
	 *
	 * @static
	 * @access private
	 * @var string
	 */
	private static $original_option_name = DIMA_THEME_OPTIONS;

	/**
	 * The option name including the language suffix.
	 * If you want the option name without language, use $original_option_name.
	 *
	 * @static
	 * @access private
	 * @var string
	 */
	private static $option_name = '';

	/**
	 * The language we're using.
	 * This is used to modify $option_name.
	 * It is the language code prefixed with a '_'
	 *
	 * @static
	 * @access public
	 * @var string
	 */
	public static $lang = '';

	/**
	 * Determine if the language has been applied to the $option_name.
	 *
	 * @static
	 * @access public
	 * @var bool
	 */
	public static $lang_applied = false;

	/**
	 * Dertermine if the current language is set to "all".
	 *
	 * @static
	 * @access private
	 * @var bool
	 */
	private static $language_is_all = false;


	/**
	 * Determine if we're currently upgrading/migration options.
	 *
	 * @static
	 * @access public
	 * @var bool
	 */
	public static $is_updating = false;

	/**
	 * The current page ID.
	 *
	 * @access public
	 * @var bool|int
	 */
	public static $c_page_id = false;

	/**
	 * An instance of the Dima_Multilingual class.
	 *
	 * @access public
	 * @since 1.0.0
	 * @var object Dima_Multilingual
	 */
	public $multilingual;

	/**
	 * Dima_PWA
	 *
	 * @access public
	 * @var object Dima_PWA
	 */
	public $pwa;
	/**
	 * The class constructor
	 */
	private function __construct() {
		// Add a non-persistent cache group.
		wp_cache_add_non_persistent_groups( 'dima' );
		add_action( 'wp', array( $this, 'set_page_id' ) );
		$this->multilingual = new Dima_Multilingual();
		// Multilingual handling.
		self::multilingual_options();
		$this->remote_install = new Dima_Remote_Installer();
		$this->pwa            = new Dima_PWA();
	}

	/**
	 * Gets the theme version.
	 *
	 * @static
	 * @access public
	 * @since 5.0
	 * @return string
	 */
	public static function get_theme_version() {
		return self::$version;
	}

	/**
	 * Access the single instance of this class.
	 *
	 * @return Dima
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Gets the current page ID
	 *
	 * @return The current page ID..
	 */
	public function get_page_id() {
		if ( ! self::$c_page_id ) {
			self::set_page_id();
		}
		return self::$c_page_id;
	}

	/**
	 * Sets the current page ID.
	 *
	 * @uses self::c_page_id
	 */
	public function set_page_id() {
		if ( ! self::$c_page_id ) {
			self::$c_page_id = self::c_page_id();
		}
	}

	/**
	 * Gets the ID of the current page, post,blog,shop
	 *
	 * @return int|mixed|null|void
	 */
	public function c_page_id() {
		if ( get_option( 'show_on_front' ) && get_option( 'page_for_posts' ) && is_home() ) {
			return get_option( 'page_for_posts' );
		}

		$c_page_id = get_queried_object_id();

		// The WooCommerce shop page.
		if ( ! is_admin() && class_exists( 'WooCommerce' ) && is_shop() ) {
			return get_option( 'woocommerce_shop_page_id' );
		}
		// The WooCommerce product_cat taxonomy page.
		if ( ! is_admin() && class_exists( 'WooCommerce' ) && ( ! is_shop() && ( is_tax( 'product_cat' ) || is_tax( 'product_tag' ) ) ) ) {
			return $c_page_id . '-archive'; // So that other POs do not apply to arhives if post ID matches.
		}
		// The homepage.
		if ( 'posts' === get_option( 'show_on_front' ) && is_home() ) {
			return $c_page_id;
		}
		if ( ! is_singular() && is_archive() ) {
			return $c_page_id . '-archive'; // So that other POs do not apply to arhives if post ID matches.
		}
		if ( ! is_singular() ) {
			return false;
		}
		return $c_page_id;
	}


	/**
	 * Get the private $option_name.
	 * If empty returns the original_option_name.
	 *
	 * @static
	 * @access public
	 * @return string
	 */
	public static function get_option_name() {
		if ( empty( self::$option_name ) ) {
			return self::$original_option_name;
		}
		return self::$option_name;
	}

	/**
	 * Get the private $original_option_name.
	 *
	 * @static
	 * @access public
	 * @return string
	 */
	public static function get_original_option_name() {
		return self::$original_option_name;
	}

	/**
	 * Sets the $lang property for this object.
	 * Languages are prefixed with a '_'
	 *
	 * If we're not currently performing a migration
	 * it also checks if the options for the current language are set.
	 * If they are not, then we will copy the options from the main language.
	 *
	 * @static
	 * @access public
	 */
	public static function multilingual_options() {

		// Set the self::$lang.
		$active_language = Dima_Multilingual::get_active_language();
		if ( ! in_array( $active_language, array( '', 'en', 'all' ) ) ) {
			self::$lang = '_' . $active_language;
		}

		// Make sure the options are copied if needed.
		if ( ! in_array( self::$lang, array( '', 'en', 'all' ) ) && ! self::$lang_applied ) {
			// Set the $option_name property.
			self::$option_name = self::get_option_name();
			// Get the options without using a language (defaults).
			$original_options = get_option( self::$original_option_name, array() );
			// Get options with a language.
			$options = get_option( self::$original_option_name . self::$lang, array() );
			// If we're not currently performing a migration and the options are not set
			// then we must copy the default options to the new language.
			if ( ! self::$is_updating && ! empty( $original_options ) && empty( $options ) ) {
				update_option( self::$original_option_name . self::$lang, get_option( self::$original_option_name ) );
			}
			// Modify the option_name to include the language.
			self::$option_name = self::$original_option_name . self::$lang;
			// Set $lang_applied to true. Makes sure we don't do the above more than once.
			self::$lang_applied = true;
		}
	}

	/**
	 * Undocumented function
	 *
	 * @return void
	 */
	public static function get_dima_theme_option_name() {
		return DIMA_THEME_OPTIONS . self::$lang;
	}

	/**
	 * Undocumented function
	 *
	 * @return void
	 */
	public static function get_dima_option_name( $opt, $add_prefix = true ) {
		$dima_option_group = self::get_dima_theme_option_name();
		$_opt_name         = $add_prefix ? "{$dima_option_group}[$opt]" : $opt;
		return $_opt_name;
	}

	/**
	 * Change the private $language_is_all property.
	 *
	 * @static
	 * @access public
	 * @param bool $is_all Whether we're on the "all" language option or not.
	 * @return null|void
	 */
	public static function set_language_is_all( $is_all ) {
		if ( true === $is_all ) {
			self::$language_is_all = true;
			return;
		}
		self::$language_is_all = false;
	}

	/**
	 * Get the private $language_is_all property.
	 *
	 * @static
	 * @access public
	 * @return bool
	 */
	public static function get_language_is_all() {
		return self::$language_is_all;
	}

		/**
		 * Gets the normalized theme version.
		 *
		 * @static
		 * @access public
		 * @since 5.0
		 * @return string
		 */
	public static function get_normalized_theme_version() {
		$theme_version       = self::$version;
		$theme_version_array = explode( '.', $theme_version );

		if ( isset( $theme_version_array[2] ) && '0' === $theme_version_array[2] ) {
			$theme_version = $theme_version_array[0] . '.' . $theme_version_array[1];
		}

		return $theme_version;
	}

	/**
	 * Gets info for premium (and free) plugins from the PixelDima API
	 * in a JSON format and then format them as an array.
	 *
	 * @static
	 * @access public
	 * @since 5.0
	 * @return array Array of bundled plugins.
	 */
	public static function get_bundled_plugins() {
		if ( empty( self::$bundled_plugins ) ) {

			// Check for transient, if none, grab remote HTML file.
			$plugins = get_transient( 'dima_premium_plugins_info' );
			if ( ! $plugins || empty( $plugins ) ) {

				// $root_url = 'http://ziro.local/aws-update/aws-update/';
				$root_url = 'https://api.pixeldima.com/aws-update/';
				$url      = add_query_arg(
					array(
						'action'  => 'get_plugins',
						'theme'   => DIMA_THEME_FOLDER,
						'version' => self::get_normalized_theme_version(),
					),
					$root_url
				);
				// Get remote HTML file.
				$response = wp_remote_get(
					$url,
					array(
						'user-agent' => 'dima-user-agent',
					)
				);

				// Check for error.
				if ( is_wp_error( $response ) ) {
					return array();
				}

				// Parse remote HTML file.
				$data = wp_remote_retrieve_body( $response );

				// Check for error.
				if ( is_wp_error( $data ) ) {
					return array();
				}

				$data = json_decode( $data, true );

				if ( ! is_array( $data ) ) {
					return array();
				}
				$plugins = array();
				foreach ( $data as $plugin ) {

					if ( $plugin['has_package'] ) {
						// Making sure the correct array index is present, latest_version being returned from server plugin.
						$plugin['version']   = $plugin['latest_version'];
						$plugin['Author']    = $plugin['dima_author'];
						$plugin['AuthorURI'] = $plugin['plugin_author_url'];

					}
					$plugins[ $plugin['slug'] ] = $plugin;
				}
				// Store remote data file in transient, expire after 24 hours.
				set_transient( 'dima_premium_plugins_info', $plugins, 24 * HOUR_IN_SECONDS );
			}

			self::$bundled_plugins = $plugins;
		}
		return self::$bundled_plugins;
	}
}
