<?php

/**
 * Google Fonts.
 *
 * TABLE OF CONTENTS:
 *       01. Fonts List
 *       02. List it by Name
 *       03. Font Weight List
 *       04. Character Sets
 *       05. (F) Is Font Italic ( true or false)
 *       05. (F) Get Font Weight
 *       06. (F) Get Font Family Query
 *
 * @package Dima Framework
 * @subpackage Admin customizer
 * @version   1.0.0
 * @since     1.0.0
 * @author    PixelDima <info@pixeldima.com>
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'DIMA_Webfonts' ) ) :
	/**
	 * Undocumented class
	 */
	final class DIMA_Webfonts {

		/**
		 * The array of fonts
		 *
		 * @access private
		 * @var array
		 */
		private $fonts = array();

			/**
			 * An array of all google fonts.
			 *
			 * @static
			 * @access private
			 * @var array
			 */
		private $google_fonts = array();

			/**
			 * The array of subsets
			 *
			 * @access private
			 * @var array
			 */
		private $subsets = array();
		/**
		 * The google link
		 *
		 * @access private
		 * @var string
		 */
		private $remote_link = '';

		/**
		 * The class constructor.
		 *
		 * @access public
		 */
		public function __construct() {
			// Populate the array of google fonts.
			$this->google_fonts = $this->php_get_google_fonts();

			// Go through our fields and populate $this->fonts.
			$this->loop_fields();

			// Calls all the other necessary methods to populate and create the link.
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue' ), 105 );
			add_filter( 'dima_dynamic_css_final', array( $this, 'add_inline_css' ) );
		}

		/**
		 * Init.
		 *
		 * @access protected
		 * @since 3.5.0
		 */
		protected function init() {
			// Go through $this->fonts and populate $this->remote_link and $this->local_link.
			$this->create_remote_link();

			$this->create_local_link();
		}

		/**
		 * Calls all the other necessary methods to populate and create the link.
		 *
		 * @access public
		 */
		public function enqueue() {

			$this->init();

			if ( 'local' === dima_helper::dima_get_option( 'dima_gfonts_load_method' ) ) {
				return;
			}
			// If $this->remote_link is not empty then enqueue it.
			if ( '' !== $this->remote_link && false === $this->get_fonts_inline_styles() ) {
				wp_enqueue_style( 'dima_google_fonts', $this->remote_link, array(), null );
			}
		}

		/**
		 * Return an array of all available Google Fonts.
		 *
		 * @access private
		 * @return array All Google Fonts.
		 */
		private function php_get_google_fonts() {

			if ( null === $this->google_fonts || empty( $this->google_fonts ) ) {

				$fonts        = self::get_google_fonts( 'all' );
				$google_fonts = array();
				if ( is_array( $fonts ) ) {
					foreach ( $fonts as $font ) {
						$google_fonts[ $font->family ] = array(
							'label'    => $font->family,
							'variants' => $font->variants,
							'subsets'  => $font->subsets,
							'category' => $font->category,
						);
					}
				}

				$this->google_fonts = $google_fonts;
			}
			return $this->google_fonts;
		}

		/**
		 * Get the google fonts from the API or in the cache
		 *
		 * @param  integer $amount
		 *
		 * @return String
		 */
		public static function get_google_fonts( $amount = 30 ) {
			$fontFile = DIMA_TEMPLATE_PATH . '/framework/functions/admin/customizer/include/google-font-list.json';

			$content = json_decode( file_get_contents( $fontFile ) );
			if ( $amount == 'all' ) {
				return $content->items;
			} else {
				return array_slice( $content->items, 0, $amount );
			}
		}


		/**
		 * Get all fonts from fontface.me API.
		 *
		 * @since    3.2
		 * @access   public
		 */
		public static function get_fontface_fonts() {
			// name of transient in database.
			$trans_name = 'dima-get-fonts';
			// time in days between updates, set to 2 days.
			$cache_time = 2 * DAY_IN_SECONDS;
			// get cached fonts.
			$content = get_transient( $trans_name );
			// check for transient. If none, then get all fonts from API.
			if ( $content === false ) {
				// get all fonts from API json content.
				$fontface_api  = 'http://fontface.me/font/all';
				$fonts_content = wp_remote_get( $fontface_api, array( 'sslverify' => false ) );
				// check if it is not a valid request.
				if ( is_wp_error( $fonts_content ) ) {
					return;
				} else {
					$content = json_decode( $fonts_content['body'], true );
					set_transient( $trans_name, $content, $cache_time );
				}
			}
			return $content;
		}

		/**
		 * List of standard fonts
		 *
		 * @since 1.0.0
		 */
		public static function dima_standard_fonts() {
			return apply_filters(
				'dima_standard_fonts_array',
				array(
					'Arial, Helvetica, sans-serif',
					'Arial Black, Gadget, sans-serif',
					'Bookman Old Style, serif',
					'Comic Sans MS, cursive',
					'Courier, monospace',
					'Georgia, serif',
					'Garamond, serif',
					'Impact, Charcoal, sans-serif',
					'Lucida Console, Monaco, monospace',
					'Lucida Sans Unicode, Lucida Grande, sans-serif',
					'MS Sans Serif, Geneva, sans-serif',
					'MS Serif, New York, sans-serif',
					'Palatino Linotype, Book Antiqua, Palatino, serif',
					'Tahoma, Geneva, sans-serif',
					'Times New Roman, Times, serif',
					'Trebuchet MS, Helvetica, sans-serif',
					'Verdana, Geneva, sans-serif',
					'Paratina Linotype',
					'Trebuchet MS',
				)
			);
		}

		/**
		 * [Font weight List]
		 */
		public static function dima_get_font_weights() {
			$dima_list_weights = array(
				'100'       => 'Thin (100)',
				'100italic' => 'Thin Italic (100)',
				'200'       => 'Extra Light (200)',
				'200italic' => 'Extra Light Italic (200)',
				'300'       => 'Light (300)',
				'300italic' => 'Light Italic (300)',
				'regular'   => 'Normal (400)',
				'italic'    => 'Normal Italic (400)',
				'500'       => 'Medium (500)',
				'500italic' => 'Medium Italic (500)',
				'600'       => 'Demi bold (600)',
				'600italic' => 'Demi bold Italic (600)',
				'700'       => 'Bold (700)',
				'700italic' => 'Bold Italic (700)',
				'800'       => 'Heavy Bold (800)',
				'800italic' => 'Heavy Bold Italic (800)',
				'900'       => 'Blac (900)k',
				'900italic' => 'Black Italic (900)',
			);

			return $dima_list_weights;
		}

		/**
		 * [Font weight List Selcted]
		 */
		public static function dima_get_font_weight_selected() {
			$dima_list_weights = array(
				'inherit' => esc_html__( 'Inherit', 'dima' ),
				'100'     => 'Thin (100)',
				'200'     => 'Extra Light (200)',
				'300'     => 'Light (300)',
				'400'     => 'Normal (400)',
				'500'     => 'Medium (500)',
				'600'     => 'Demi bold (600)',
				'700'     => 'Bold (700)',
				'800'     => 'Heavy Bold (800)',
				'900'     => 'Blac (900)k',
			);

			return $dima_list_weights;
		}

		/**
		 * [Character Sets]
		 */
		public static function dima_get_font_subsets() {
			$dima_list_font_subsets = array(
				'arabic'              => esc_html__( 'Arabic', 'dima' ),
				'hebrew'              => esc_html__( 'Hebrew', 'dima' ),
				'latin'               => esc_html__( 'Latin', 'dima' ),
				'latin-ext'           => esc_html__( 'Latin Extended ', 'dima' ),
				'greek'               => esc_html__( 'Greek', 'dima' ),
				'greek-ext'           => esc_html__( 'Greek Extended ', 'dima' ),
				'cyrillic'            => esc_html__( 'Cyrillic', 'dima' ),
				'cyrillic-ext'        => esc_html__( 'Cyrillic Extended ', 'dima' ),
				'japanese'            => esc_html__( 'Japanese ', 'dima' ),
				'chinese-simplified'  => esc_html__( 'Chinese Simplified ', 'dima' ),
				'chinese-traditional' => esc_html__( 'Chinese Traditional ', 'dima' ),
				'vietnamese'          => esc_html__( 'vietnamese', 'dima' ),
			);

			return $dima_list_font_subsets;
		}

		/**
		 * [Is Font Italic ]
		 * * @param $font_weight_and_style
		 *
		 * @return bool
		 */
		public static function dima_is_font_italic( $font_weight_and_style ) {
			if ( strpos( $font_weight_and_style, 'italic' ) ) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Get Font Weight
		 *
		 * @param $font_weight_and_style
		 *
		 * @return mixed
		 */
		public static function dima_get_font_weight( $font_weight_and_style ) {

			$font_weight = ( self::dima_is_font_italic( $font_weight_and_style ) ) ? str_replace( 'italic', '', $font_weight_and_style ) : $font_weight_and_style;

			return $font_weight;
		}

		/**
		 * @param string $family
		 * @param string $weight
		 *
		 * @return string
		 */
		public static function dima_get_font_family_and_weight( $family = '', $weight = '' ) {
			$output = '';
			if ( ! empty( $family ) ) {
				$output = "font-family: \"{$family}\",\"Helvetica Neue\",Helvetica,sans-serif;";
			}
			if ( ! empty( $weight ) ) {
				$output .= "font-weight: {$weight};";
			}

			return $output;
		}

		/**
		 * Undocumented function
		 *
		 * @param [type] $font_family
		 * @param [type] $font_weight
		 * @param string $sorce
		 * @param array  $custom_font_args_all
		 * @return void
		 */
		private function generate_google_font( $font_family, $font_weight, $sorce = '' ) {
			$subsets         = dima_helper::dima_get_option( 'dima_body_subsets_list' );
			$this->subsets[] = $subsets;
			if ( $font_family == 'Poppin' ) {
				$font_family = 'Poppins';
			}
			if ( ! isset( $this->fonts[ $font_family ] ) ) {
				$this->fonts[ $font_family ] = array();
			}
				$font_weight_array = explode( ',', $font_weight );
			foreach ( $font_weight_array as $key => $value ) {
				if ( ! in_array( $value, $this->fonts[ $font_family ], true ) ) {
					$this->fonts[ $font_family ][] = $value;
				}
			}
		}

		/**
		 * Goes through all our fields and then populates the $this->fonts property.
		 *
		 * @access private
		 */
		private function loop_fields() {
			$fonts        = array(
				'dima_body_font_list',
				'dima_logo_font_list',
				'dima_btn_font_list',
				'dima_navbar_font_list',
				'dima_heading_font_list',
				'dima_heading_font_list_2',
				'dima_heading_font_list_3',
				'dima_heading_font_list_4',
				'dima_heading_font_list_5',
				'dima_heading_font_list_6',
			);
			$font_weights = array(
				'dima_body_weights_list',
				'dima_logo_weights_list',
				'dima_btn_weights_list',
				'dima_navbar_weights_list',
				'dima_heading_weights_list',
				'dima_heading_weights_list_2',
				'dima_heading_weights_list_3',
				'dima_heading_weights_list_4',
				'dima_heading_weights_list_5',
				'dima_heading_weights_list_6',
			);
			$font_sources = array(
				'dima_body_font_source',
				'dima_logo_font_source',
				'dima_btn_font_source',
				'dima_navbar_font_source',
				'dima_heading_font_source',
				'dima_heading_2_font_source',
				'dima_heading_3_font_source',
				'dima_heading_4_font_source',
				'dima_heading_5_font_source',
				'dima_heading_6_font_source',
			);
			if ( is_rtl() ) {
				$default_font = array(
					'Droid Arabic Naskh',
					'Droid Arabic Naskh',
					'Droid Arabic Naskh',
					'Droid Arabic Naskh',
					'Droid Arabic Kufi',
					'Droid Arabic Kufi',
					'Droid Arabic Kufi',
					'Droid Arabic Kufi',
					'Droid Arabic Kufi',
					'Droid Arabic Kufi',
				);
			} else {
				$default_font = array(
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
					'Poppins',
				);
			}

			$mi = new MultipleIterator();
			$mi->attachIterator( new ArrayIterator( $fonts ) );
			$mi->attachIterator( new ArrayIterator( $font_weights ) );
			$mi->attachIterator( new ArrayIterator( $font_sources ) );
			$mi->attachIterator( new ArrayIterator( $default_font ) );
			global $dima_customizer_data;
			foreach ( $mi as $value ) {
				list($font, $weight, $source, $default ) = $value;
				$source                                  = dima_helper::dima_get_option( $source );

				if ( 'g_fonts' === $source || '' === $source ) {
					$font       = dima_helper::dima_get_option( $font, $default );
					$weight_opt = dima_helper::dima_get_option( $weight );
					$weight     = ( '' !== $weight_opt ) ? $weight_opt : $dima_customizer_data[ $weight ];

					$this->generate_google_font( $font, $weight, $source );
				} elseif ( 'f_fonts' === $source ) {
					$font_ = dima_helper::dima_get_option( $font, $default );
					wp_enqueue_style( 'fontstatic-' . $font, 'https://www.fontstatic.com/f=' . $font_, array(), DIMA_VERSION, 'all' );
				}
			}
		}
		/**
		 * Get the CSS for local fonts.
		 *
		 * @access public
		 * @since 1.0
		 * @return string
		 */
		public function get_local_fonts_css() {
			$css = '';
			// If we don't have any fonts then we can exit.
			if ( empty( $this->fonts ) ) {
				return;
			}

			// Get font-family + subsets.
			foreach ( $this->fonts as $font => $variants ) {
				$family = new Dima_GFonts_Downloader( $font );
				$css   .= $family->get_fontface_css( $variants );
			}
			return $css;
		}

		/**
		 * Adds googlefont styles inline in dynamic-css.
		 *
		 * @access public
		 * @since 3.5.0
		 * @param string $original_styles The dynamic-css styles.
		 * @return string The dynamic-css styles with any additional stylesheets appended.
		 */
		public function add_inline_css( $original_styles ) {
			$font_styles = $this->get_fonts_inline_styles();
			if ( false === $font_styles ) {
				return $original_styles;
			}
			return $font_styles . $original_styles;

		}

		/**
		 * Undocumented function
		 *
		 * @return void
		 */
		public function get_remote_link() {
			$this->create_remote_link();
			return $this->remote_link;
		}

			/**
			 * Creates the google-fonts link.
			 *
			 * @access private
			 */
		private function create_remote_link() {
			// If we don't have any fonts then we can exit.
			// Get font-family + subsets.
			$link_fonts = array();
			foreach ( $this->fonts as $font => $variants ) {

				$variants = implode( ',', $variants );

				$link_font = str_replace( ' ', '+', $font );
				if ( ! empty( $variants ) ) {
					$link_font .= ':' . $variants;
				}
				$link_fonts[] = $link_font;
			}

			if ( ! empty( $this->subsets ) ) {
				$this->subsets = array_unique( $this->subsets );
			}
			if ( ! empty( $link_fonts ) ) {

				$this->remote_link = add_query_arg(
					array(
						'family'  => str_replace( '%2B', '+', rawurlencode( implode( '|', $link_fonts ) ) ),
						'subset'  => rawurlencode( implode( ',', $this->subsets ) ),
						'display' => 'swap',
					),
					'https://fonts.googleapis.com/css'
				);
			}

		}

		/**
		 * Creates the google-fonts link.
		 *
		 * @access private
		 * @since 5.5.2
		 * @return void
		 */
		private function create_local_link() {
			// If we don't have any fonts then we can exit.
			if ( empty( $this->fonts ) ) {
				return;
			}

			// Get font-family + subsets.
			$link_fonts = array();
			foreach ( $this->fonts as $font => $variants ) {
				$variants  = implode( ',', $variants );
				$link_font = str_replace( ' ', '+', $font );
				if ( ! empty( $variants ) ) {
					$link_font .= ':' . $variants;
				}
				$link_fonts[] = $link_font;
			}
			$this->local_link = add_query_arg(
				array(
					'action' => 'get-gfonts',
					'family' => str_replace( '%2B', '+', rawurlencode( implode( '|', $link_fonts ) ) ),
				),
				site_url()
			);
		}

		/**
		 * Get the contents of googlefonts so that they can be added inline.
		 *
		 * @access protected
		 * @since 3.5.0
		 * @return string|false
		 */
		protected function get_fonts_inline_styles() {
			$transient_name = 'dima_googlefonts_contents';
			$active_lang    = Dima_Multilingual::get_active_language();
			if ( '' !== $active_lang || 'all' !== $active_lang ) {
				$transient_name .= '_' . $active_lang;
			}

			$contents = get_transient( $transient_name );
			if ( false === $contents ) {
				// If we're using local, early exit after getting the styles.
				if ( 'local' === dima_helper::dima_get_option( 'dima_gfonts_load_method' ) ) {
					return $this->get_local_fonts_css();
				}
				if ( '' === $this->remote_link ) {
					$this->create_remote_link();
				}

				// If link is empty, early exit.
				if ( '' === $this->remote_link || ! $this->remote_link ) {
					set_transient( $transient_name, 'failed', DAY_IN_SECONDS );
					return false;
				}

				// Get remote HTML file.
				$response = wp_remote_get( $this->remote_link );

				// Check for errors.
				if ( is_wp_error( $response ) ) {
					set_transient( $transient_name, 'failed', DAY_IN_SECONDS );
					return false;
				}

				// Parse remote HTML file.
				$contents = wp_remote_retrieve_body( $response );
				// Check for error.
				if ( is_wp_error( $contents ) || ! $contents ) {
					set_transient( $transient_name, 'failed', DAY_IN_SECONDS );
					return false;
				}

				// Store remote HTML file in transient, expire after 24 hours.
				set_transient( $transient_name, $contents, DAY_IN_SECONDS );
			}
			// Return false if we were unable to get the contents of the googlefonts from remote.
			if ( 'failed' === $contents ) {
				return false;
			}

			// If we got this far then we can safely return the contents.
			return $contents;
		}
	}
endif;
