<?php
/**
 * AMP
 *
 * @package DIMA
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'DIMA_AMP' ) ) {

	class DIMA_AMP {
		const AMP_QUERY          = 'amp';
		private static $instance = null;

		/**
		 * Creates or returns an instance of this class.
		 */
		public static function instance() {
			// If an instance hasn't been created and set to $instance create an instance and set it to $instance.
			if ( null == self::$instance ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * __construct
		 *
		 * Class constructor where we will call our filter and action hooks.
		 */
		public function __construct() {
			if ( ! DIMA_AMP_IS_ACTIVE || ! dima_helper::dima_am_i_true( dima_helper::dima_get_option( 'dima_amp_enable' ) ) ) {
				return false;
			}

			// Disable the AMP Customizer menu, Control styles from the theme options page.
			remove_action( 'admin_menu', 'amp_add_customizer_link' );

			add_action( 'amp_post_template_head', array( $this, 'dima_head' ) );
			add_action( 'pre_amp_render_post', array( $this, 'dima_content_filters' ) );
			add_action( 'amp_post_template_css', array( $this, 'dima_amp_additional_css_styles' ) );

			add_filter( 'amp_content_max_width', array( $this, 'dima_content_width' ) );
			add_filter( 'amp_post_template_file', array( $this, 'dima_templates_path' ), 10, 3 );
			add_filter( 'dima_amp_body_classes', array( $this, 'dima_amp_body_classes' ), 10, 3 );
			require_once 'modules/class-amp-wc.php';
			add_action( 'amp_post_template_data', array( $this, 'dima_amp_component_scripts' ) );
		}

		/**
		 * Undocumented function
		 *
		 * @param [type] $data
		 * @return void
		 */
		function dima_amp_component_scripts( $data ) {

			$data['amp_component_scripts']['amp-form'] = 'https://cdn.ampproject.org/v0/amp-form-0.1.js';
			$data['amp_component_scripts']['amp-anim'] = 'https://cdn.ampproject.org/v0/amp-anim-0.1.js';
			if ( has_nav_menu( 'dima_amp_menu' ) ) {
				$data['amp_component_scripts']['amp-sidebar'] = 'https://cdn.ampproject.org/v0/amp-sidebar-0.1.js';
			}
			if ( DIMA_WC_IS_ACTIVE && dima_helper::dima_is_product() ) {
				$data['amp_component_scripts']['amp-accordion'] = 'https://cdn.ampproject.org/v0/amp-accordion-0.1.js';
				$data['amp_component_scripts']['amp-carousel']  = 'https://cdn.ampproject.org/v0/amp-carousel-0.2.js';
			}
			$related_posts_is_on = dima_helper::dima_am_i_true( dima_helper::dima_get_option( 'dima_amp_related_posts' ) );
			if ( $related_posts_is_on ) {
				$data['amp_component_scripts']['amp-carousel'] = 'https://cdn.ampproject.org/v0/amp-carousel-0.2.js';
			}
			if ( dima_helper::dima_get_option( 'dima_amp_ad_abover' ) != '' || dima_helper::dima_get_option( 'dima_amp_ad_below' ) != '' ) {
				$data['amp_component_scripts']['amp-ad'] = 'https://cdn.ampproject.org/v0/amp-ad-0.1.js';
			}
			// Share Buttons ----------
			$related_shear_icon_is_on = dima_helper::dima_am_i_true( dima_helper::dima_get_option( 'dima_amp_share_buttons' ) );
			if ( $related_shear_icon_is_on ) {
				$data['amp_component_scripts']['amp-social-share'] = 'https://cdn.ampproject.org/v0/amp-social-share-0.1.js';
			}

			return $data;
		}

		public function dima_amp_body_classes( $classess ) {
			if ( dima_helper::dima_is_product() ) {
				$classess[] = 'amp_dima_product';
			}
			$output = esc_attr( trim( implode( ' ', $classess ) ) );

			return $output;
		}

		/**
		 * dima_content_filters
		 *
		 * ads, formats and share buttons to the post content
		 */
		public function dima_content_filters() {
			if ( is_single() && is_singular() ) {
				add_filter( 'the_content', array( $this, 'dima_ads' ) );
			}

			add_filter( 'the_content', array( $this, 'dima_amp_content' ) );
		}

		/**
		 * dima_amp_post_formats
		 */
		public function dima_amp_content( $content ) {

			global $wp_query;
			$postid      = $wp_query->post->ID;
			$post_format = get_post_format( $postid ) != '' ? get_post_format( $postid ) : 'standard';
			$output      = '';

			if ( dima_helper::dima_is_product() ) {
				$output .= DIMA_AMP_WC::get_instance()->dima_amp_get_woo_item_content( $postid );
			} elseif ( $post_format ) {
				ob_start();
				if ( has_post_thumbnail() ) {
					the_post_thumbnail();
				}
				$output = ob_get_clean();
				if ( ! empty( $output ) ) {
					$output = '<div class="amp-featured">' . $output . '</div>';
				}
			}

			if ( ! empty( $output ) ) {
				$content = $output . $content;
			}

			return $content;
		}

		/**
		 * dima_ads
		 */
		public function dima_ads( $content ) {

			if ( dima_helper::dima_get_option( 'dima_amp_ad_abover' ) != '' ) {
				$content = dima_helper::dima_get_option( 'dima_amp_ad_abover' ) . $content;
			}

			if ( dima_helper::dima_get_option( 'dima_amp_ad_below' ) != '' ) {
				$content = $content . dima_helper::dima_get_option( 'dima_amp_ad_below' );
			}

			return $content;
		}

		/**
		 * dima content width
		 */
		public static function dima_content_width() {
			return 840;
		}

		/**
		 * Do not load Merriweather Google fonts on AMP pages
		 */
		public function dima_remove_google_fonts() {
			remove_action( 'amp_post_template_head', 'amp_post_template_add_fonts' );
		}

		/**
		 * Undocumented function
		 *
		 * @return void
		 */
		public function dima_amp_google_fonts() {
			global $dima_webfonts;
			echo '<link href="' . $dima_webfonts->get_remote_link() . '" rel="stylesheet">';
		}

		/**
		 * head
		 */
		public function dima_head() {
			wp_site_icon();
			self::dima_amp_google_fonts();
			self::dima_remove_google_fonts();
		}

		/**
		 * templates path
		 *
		 * Set custom template path
		 */
		public function dima_templates_path( $file, $type, $post ) {

			switch ( $type ) {
				case 'footer':
				case 'meta-comments-link':
				case 'meta-taxonomy':
				case 'html-start':
				case 'featured-image':
				case 'header-bar':
					$file = DIMA_TEMPLATE_PATH . '/amp_templates/' . $type . '.php';
					break;
				case 'single':
					if ( dima_helper::dima_is_product() ) {
						$file = DIMA_TEMPLATE_PATH . '/amp_templates/woo-' . $type . '.php';
					} else {
						$file = DIMA_TEMPLATE_PATH . '/amp_templates/' . $type . '.php';
					}
					break;
				case 'page':
					if ( is_home() ) {
						$type = 'index';
					}
					$file = DIMA_TEMPLATE_PATH . '/amp_templates/' . $type . '.php';
					break;

			}

			return $file;
		}

		/**
		 * dima_share_buttons
		 *
		 * Add the share buttons
		 */
		public static function dima_share_buttons( $content = '' ) {

			$related_shear_icon_is_on = dima_helper::dima_am_i_true( dima_helper::dima_get_option( 'dima_amp_share_buttons' ) );

			if ( $related_shear_icon_is_on ) {

				$share_buttons = '
					<div class="amp-social full-width">
						<amp-social-share type="facebook" width="60" height="44" data-param-app_id=' . dima_helper::dima_get_option( 'dima_amp_facebook_id' ) . '></amp-social-share>
						<amp-social-share type="twitter" width="60" height="44"></amp-social-share>
						<amp-social-share type="gplus" width="60" height="44"></amp-social-share>
						<amp-social-share type="pinterest" width="60" height="44"></amp-social-share>
						<amp-social-share type="linkedin" width="60" height="44"></amp-social-share>
						<amp-social-share type="email" width="60" height="44"></amp-social-share>
					</div>
				';

				$content = $content . $share_buttons;
			}

			return $content;
		}

		private function dima_amp_get_css_style( $file ) {
			$dima_css_style_dir = '/framework/asset/site/css/styles/';
			$demo               = 'noor_main';
			$rtl                = is_rtl() ? '-rtl' : '';

			if ( filter_var( $file, FILTER_VALIDATE_URL ) === false ) {
				$href = DIMA_TEMPLATE_URL . $dima_css_style_dir . $demo . '/amp/' . $file . $rtl . '.min.css?' . DIMA_VERSION;
			} else {
				$href = $file;
			}

			preg_match( '#^(https?:)?//#', $href, $matches );

			if ( empty( $matches[1] ) ) {
				$href = ( is_ssl() ? 'https' : 'http' ) . ':' . $href;
			}
			$response = wp_remote_post( $href );
			if ( is_wp_error( $response ) ) {
				$error_message = $response->get_error_code();
				if ( $error_message == 'http_request_failed' && is_ssl() ) {
					$href = set_url_scheme( $href, 'http' );
				}
			}
			$cache_key = md5( $href );
			$contents  = get_transient( $cache_key );

			if ( false === $contents ) {
				$r = wp_remote_get( $href );
				if ( 200 !== wp_remote_retrieve_response_code( $r ) ) {
					$contents = new WP_Error( wp_remote_retrieve_response_code( $r ) );
				} else {
					$contents = wp_remote_retrieve_body( $r );
				}
				set_transient( $cache_key, $contents, MONTH_IN_SECONDS );
			}

			if ( ! is_wp_error( $contents ) ) {
				return str_replace( '!important', '', $contents );
			}

			return '';
		}

		/**
		 * Undocumented function
		 *
		 * @return void
		 */
		public function dima_amp_additional_css_styles() {
			$amp_log                 = $amp_logo_footer = '';
			$logo                    = dima_helper::dima_get_option( 'dima_amp_logo' );
			$logo_width              = dima_helper::dima_get_option( 'dima_amp_logo_width' );
			$footer_logo             = dima_helper::dima_get_option( 'dima_amp_footer_logo' );
			$logo_retina             = $var_header_logo_width = dima_helper::dima_get_option( 'dima_header_logo_retina' );
			$is_custom_font          = dima_helper::dima_am_i_true( dima_helper::dima_get_option( 'dima_custom_font' ) );
			$var_page_title_bg_color = dima_helper::dima_get_inherit_option( '_dima_meta_page_title_bg_color', 'dima_page_title_bg_color' );
			$var_main_color          = dima_helper::dima_get_inherit_option( 'dima_meta_main_color', 'dima_main_color' );
			if ( empty( $var_main_color ) ) {
				$var_main_color = '#333333';
			}
			$body_bg_color                 = dima_helper::dima_get_option( 'dima_amp_bg_color' );
			$text_color                    = DIMA_Style::dima_light_or_dark( $body_bg_color, false, '#757575' );
			$title_color                   = dima_helper::dima_get_option( 'dima_amp_title_color', '#333333' );
			$header_bg_color               = dima_helper::dima_get_option( 'dima_amp_header_bg_color' );
			$link_color                    = dima_helper::dima_get_option( 'dima_amp_link_color' );
			$footer_bg_color               = dima_helper::dima_get_option( 'dima_amp_footer_bg_color' );
			$footer_color                  = DIMA_Style::dima_light_or_dark( $footer_bg_color, false, '#333333', '#a2a2a2' );
			$footer_link_color             = DIMA_Style::dima_light_or_dark( $footer_bg_color, false, '#333333', '#FFFFFF' );
			$sec_footer_color              = dima_helper::dima_get_option( 'dima_amp_footer_border_color' );
			$var_body_font_weight_selected = dima_helper::dima_get_option( 'dima_body_weight_selected' );
			$heading_font_weight           = dima_helper::dima_get_option( 'dima_heading_weights_list' );
			if ( is_rtl() && ! $is_custom_font ) {
				$var_body_font_family = 'Droid Arabic Naskh';
				$heading_font_family  = 'Droid Arabic Kufi';
			} else {
				$var_body_font_family = dima_helper::dima_get_option( 'dima_body_font_list' );
				$heading_font_family  = dima_helper::dima_get_option( 'dima_heading_font_list' );
			}
			if ( $amp_logo_footer === '' ) {
				$amp_logo_footer = $footer_logo;
			}
			if ( $logo_retina === '' && $logo === '' ) {
				$amp_log = '';
			} elseif ( $logo !== '' ) {
				$amp_log = $logo;
			} elseif ( $logo_retina !== '' ) {
				$amp_log = $logo_retina;
			}

			echo self::dima_amp_get_css_style( 'amp' );
			if ( ( is_home() || is_single() ) && ! dima_helper::dima_is_product() ) {
				echo self::dima_amp_get_css_style( 'amp-blog' );
			}
			if ( has_nav_menu( 'dima_amp_menu' ) ) {
				echo self::dima_amp_get_css_style( 'amp-menu' );
			}
			if ( DIMA_WC_IS_ACTIVE && dima_helper::dima_is_product() ) {
				echo self::dima_amp_get_css_style( 'amp-woo' );
			}
			if ( AMP_Options_Manager::is_stories_experience_enabled() ) {
				// echo self::dima_amp_get_css_style( amp_get_asset_url( '/css/amp-story-card.css' ) );
				?>
				.latest-stories__meta,
				.latest-stories__title{
					display:none;
				}
				<?php
			}
			?>

			.title_container{
			background: <?php echo esc_attr( $var_page_title_bg_color ); ?>;
			}
			.amp-wp-meta,.amp-dima-header div,.wp-caption-text,.amp-wp-tax-category,
			.amp-wp-tax-tag,.amp-wp-comments-link, .amp-wp-footer p, .back-to-top{
			<?php echo DIMA_Webfonts::dima_get_font_family_and_weight( $var_body_font_family, $var_body_font_weight_selected ); ?>;
			}
			html, p,
			body {
			color: <?php echo esc_attr( $text_color ); ?>;
			<?php echo DIMA_Webfonts::dima_get_font_family_and_weight( $var_body_font_family, $var_body_font_weight_selected ); ?>;
			}
			body{
			background: <?php echo esc_attr( $body_bg_color ); ?>;
			}
			.hamburger svg{
			fill: <?php echo DIMA_Style::dima_light_or_dark( $header_bg_color, false, '#333333', '#FFFFFF' ); ?>
			}
			.amp-wp-title,h1,h2,h3,h4,h5,h6{
			<?php echo DIMA_Webfonts::dima_get_font_family_and_weight( $heading_font_family, $heading_font_weight ); ?>;
			color: <?php echo esc_attr( $title_color ); ?>;
			}
			.amp-wp-comments-link a,a, a:visited {
			color: <?php echo esc_attr( $link_color ); ?>;
			}
			.dima-button,
			a.dima-button,
			a.button,.button{
			background-color: <?php echo esc_attr( $var_main_color ); ?>;
			color:#FFFFFF;
			}
			.dima-button:hover,
			a.button:hover,.button:hover{
			background-color: rgba(<?php echo DIMA_Style::dima_get_rgb_color( $var_main_color ); ?>, 0.8 );
			}

			.amp-wp-article{
			color: <?php echo esc_attr( $text_color ); ?>;
			}
			.amp-dima-header div,.amp-wp-title {
			color: <?php echo esc_attr( $title_color ); ?>;
			}
			.amp-dima-header {
			background-color: <?php echo esc_attr( $header_bg_color ); ?>;
			}
			.amp-dima-header a {
			width: <?php echo esc_attr( $logo_width ); ?>px;
			<?php if ( ! empty( $amp_log ) ) { ?>
				background-image: url(<?php echo esc_url( $amp_log ); ?>);
				background-repeat: no-repeat;
				background-size: contain;
				background-position: center center;
				text-indent: -9999px;
			<?php } else { ?>
				color:#FFFFFF;
				text-align: center;
			<?php } ?>
			}
			.amp-wp-comments-link a {
			border-color: <?php echo esc_attr( $link_color ); ?>;
			}
			blockquote {
			background: <?php echo esc_attr( $body_bg_color ); ?>;
			border-color: <?php echo esc_attr( $header_bg_color ); ?>;
			color: <?php echo esc_attr( $text_color ); ?>;;
			}
			.top-footer{
				padding: 1.5em 0;
			}
			.footer-container {
			background-color: <?php echo esc_attr( $footer_bg_color ); ?>;
			color: <?php echo esc_attr( $footer_color ); ?>;
			}
			.footer-logo {
			background-image: url(<?php echo esc_url( $amp_logo_footer ); ?>);
			margin-bottom: 0;
			}
			footer .dima-footer {
			position: relative;
			border-top: 1px solid <?php echo esc_attr( $sec_footer_color ); ?>;
			color: <?php echo esc_attr( $footer_color ); ?>;
			}
			.copyright a,.dima-menu a, .dima-menu a:hover,
			.dima-menu a:active,.scroll-to-top a,.dima-menu a:visited  {
			color: <?php echo esc_attr( $footer_link_color ); ?>;
			}
			.scroll-to-top {
			background: <?php echo esc_attr( $sec_footer_color ); ?>;
			top:0;
			left:0;
			margin-left:0;
			}
			amp-accordion section > h4[aria-expanded="true"] {
			background-color: <?php echo esc_attr( $var_main_color ); ?>;
			color: #fff;
			}
			<?php
		}

		public function get_endpoint() {
			$endpoint_opt = get_option( 'amphtml_endpoint' );
			$endpoint     = ( $endpoint_opt ) ? $endpoint_opt : self::AMP_QUERY;

			return $endpoint;
		}

		public function is_amp() {
			$endpoint = $this->get_endpoint();

			if ( '' == get_option( 'permalink_structure' ) ) {
				parse_str( $_SERVER['QUERY_STRING'], $url );

				return isset( $url[ $endpoint ] );
			}

			$url_parts   = explode( '?', $_SERVER['REQUEST_URI'] );
			$query_parts = explode( '/', $url_parts[0] );

			$is_amp = ( in_array( $endpoint, $query_parts ) );
			do_action( 'amphtml_is_amp', $is_amp );

			return $is_amp;
		}
	}


	// Instantiate the class.
	new DIMA_AMP();

}

function DIMA_AMP() {
	if ( function_exists( 'is_amp_endpoint' ) && is_amp_endpoint() ) {
		return DIMA_AMP::instance();
	}
}

if ( ! function_exists( 'dima_amp_post_classes' ) ) {
	/**
	 * Handy function to generate class attribute for posts
	 *
	 * @param string|array $append One or more classes to add to the class list.
	 */
	function dima_amp_post_classes( $append = '' ) {

		$class = get_post_class( $append );

		if ( ! has_post_thumbnail() ) {
			$class[] = 'no-thumbnail';
		} else {
			$class[] = 'have-thumbnail';
		}

		$class[] = 'clearfx';

		$class = str_replace( 'hentry', '', join( ' ', $class ) );

		echo 'class="' . $class . '"';

		unset( $class );
	}
}

if ( ! function_exists( 'dima_amp_build_post_featured_image' ) ) {
	function dima_amp_build_post_featured_image( $post_id = '', $img_size = 'larg' ) {

		$featured_html = get_the_post_thumbnail( $post_id, $img_size );
		if ( ! $featured_html ) {
			return;
		}

		$featured_id    = get_post_thumbnail_id( $post_id );
		$featured_image = get_post( $featured_id );
		$dom            = AMP_DOM_Utils::get_dom_from_content( $featured_html );
		AMP_Content_Sanitizer::sanitize_document(
			$dom,
			amp_get_content_sanitizers( get_post() ),
			array(
				'content_max_width' => DIMA_AMP::dima_content_width(),
			)
		);
		$sanitized_html = AMP_DOM_Utils::get_content_from_dom( $dom );
		$caption        = $featured_image->post_excerpt;
		?>
		<figure class="amp-wp-article-featured-image wp-caption">
			<?php echo( $sanitized_html ); ?>
			<?php if ( $caption ) : ?>
				<p class="wp-caption-text">
					<?php echo wp_kses_data( $caption ); ?>
				</p>
			<?php endif; ?>
		</figure>
		<?php
	}
}

if ( ! function_exists( 'dima_amp_clear_query' ) ) {
	/**
	 * Used to get current query.
	 *
	 * @param   bool $reset_query
	 *
	 * @since 1.0.0
	 */
	function dima_amp_clear_query( $reset_query = true ) {

		global $dima_amp_theme_core_query;

		$dima_amp_theme_core_query = null;

		// This will remove obscure bugs that occur when the previous wp_query object is not destroyed properly before another is set up.
		if ( $reset_query ) {
			wp_reset_query();
		}
	}
}

if ( ! function_exists( 'dima_amp_get_query' ) ) {
	/**
	 * Used to get current query.
	 *
	 * @since 1.0.0
	 *
	 * @return  WP_Query|null
	 */
	function dima_amp_get_query() {

		global $dima_amp_theme_core_query;

		// Add default query to ThemeName query if its not added or default query is used.
		if ( ! is_a( $dima_amp_theme_core_query, 'WP_Query' ) ) {
			global $wp_query;

			$dima_amp_theme_core_query = &$wp_query;
		}

		return $dima_amp_theme_core_query;
	}
}

if ( ! function_exists( 'dima_amp_set_query' ) ) {
	/**
	 * Used to get current query.
	 *
	 * @param   WP_Query $query
	 *
	 * @since 1.0.0
	 */
	function dima_amp_set_query( &$query ) {

		global $dima_amp_theme_core_query;

		$dima_amp_theme_core_query = $query;
	}
}

if ( ! function_exists( 'dima_amp_template_part' ) ) {
	/**
	 * Load a template part into a template
	 *
	 * @see   get_template_part for more documentation
	 *
	 * @param string $slug The slug name for the generic template.
	 * @param string $name The name of the specialised template.
	 *
	 * @since 1.0.0
	 */
	function dima_amp_template_part( $slug, $name = null ) {

		$templates = array();
		$name      = (string) $name;
		if ( '' !== $name ) {
			$templates[] = "{$slug}-{$name}.php";
		}

		$templates[] = "{$slug}.php";

		dima_amp_locate_template( $templates, true, false );
	}
}

if ( ! function_exists( 'dima_amp_locate_template' ) ) {
	/**
	 * Retrieve the name of the highest priority amp template file that exists.
	 *
	 * @see   locate_template for more doc
	 *
	 * @param string|array $template_names Template file(s) to search for, in order.
	 * @param bool         $load If true the template file will be loaded if it is found.
	 * @param bool         $require_once Whether to require_once or require. Default true. Has no effect if $load is false.
	 *
	 * @since 1.0.0
	 *
	 * @return string The template filename if one is located.
	 */
	function dima_amp_locate_template( $template_names, $load = false, $require_once = true ) {

		$scan_directories = array(
			TEMPLATEPATH . '/amp_templates/',
		);
		$scan_directories = array_unique( array_filter( $scan_directories ) );

		foreach ( $scan_directories as $theme_directory ) {
			if ( $theme_file_path = dima_amp_load_templates( $template_names, $theme_directory, $load, $require_once ) ) {
				return $theme_file_path;
			}
		}

		// fallback: scan into theme-compat folder
		return dima_amp_load_templates( $template_names, '', $load, $require_once );
	}
}

if ( ! function_exists( 'dima_amp_get_template_info' ) ) {
	function dima_amp_get_template_info() {

		return wp_parse_args(
			array(
				'ScreenShot' => 'screenshot.png',
				'MaxWidth'   => 780,
				'view'       => 'general',
			)
		);
	}
}

if ( ! function_exists( 'dima_amp_load_templates' ) ) {
	/**
	 * Require the template file
	 *
	 * @param string|array $templates
	 * @param string       $theme_directory base directory. scan $templates files into this directory
	 * @param bool         $load
	 * @param bool         $require_once
	 *
	 * @see   better_amp_locate_template for parameters documentation
	 *
	 * @since 1.0.0
	 *
	 * @return bool|string
	 */
	function dima_amp_load_templates( $templates, $theme_directory, $load = false, $require_once = true ) {

		foreach ( (array) $templates as $theme_file ) {

			$theme_file      = ltrim( $theme_file, '/' );
			$theme_directory = trailingslashit( $theme_directory );

			if ( file_exists( $theme_directory . $theme_file ) ) {

				if ( $load ) {
					if ( $require_once ) {
						require_once $theme_directory . $theme_file;
					} else {
						require $theme_directory . $theme_file;
					}
				}

				return $theme_directory . $theme_file;
			}
		}

		return false;
	}
}
