<?php
/**
 * Function used by widgets.
 *
 * @package Dima Framework
 * @subpackage Functions
 * @version   1.0.0
 * @since     1.0.0
 * @author    PixelDima <info@pixeldima.com>
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! function_exists( 'dima_get_entry_meta' ) ) :
	/**
	 * Post Meta
	 *
	 * @param string $args array();
	 * @return void
	 */
	function dima_get_entry_meta( $args = '' ) {

		if ( is_array( $args ) ) {
			$show_meta       = $args['meta'];
			$type            = $args['blog_type'];
			$show_comment_nb = $args['blog_remove_post_comment_number'];
			$show_category   = $args['blog_remove_post_category'];
			$show_author     = $args['blog_remove_post_author'];
			$show_date       = $args['blog_remove_post_date'];
		} else {
			$show_meta       = true;
			$show_comment_nb = $show_category = $show_date = $show_author = false;
			$type            = '';
		}

		if ( $show_author ) {
			$author = sprintf( ' <span class="entry-author">' . esc_html__( 'By', 'noor' ) . ' <a href="%1$s" rel="author" class="vcard"><span  class="fn">%2$s</span></a></span>', get_author_posts_url( get_the_author_meta( 'ID' ) ), get_the_author() );
		} else {
			$author = '';
		}

		if ( $show_date ) {
			$date   = '<li class="post-on text-start"><a href="' . get_permalink() . '">
			<time class="entry-date" datetime="' . esc_attr( get_the_date( 'c' ) ) . '"> ' . esc_html( get_the_date() ) . '</time>
			<time class="updated hide" datetime="' . esc_attr( get_the_modified_date( 'c' ) ) . '"> ' . esc_html( get_the_modified_date() ) . '</time>
			</a>' . $author . '</li>';
			$author = '';
		} else {
			$date = '';
		}

		/**
		 * Comments link
		 */
		$comments = '';
		if ( $show_comment_nb ) {
			if ( comments_open() ) {
				$Comment        = esc_html__( 'Comment', 'noor' );
				$Comments       = esc_html__( 'Comments', 'noor' );
				$post_title     = apply_filters( 'dima_meta_comments_title', get_the_title() );
				$comments_link  = apply_filters( 'dima_comments_link', get_comments_link() );
				$nb_of_comments = apply_filters( 'dima_comments_number', get_comments_number() );

				if ( $nb_of_comments == 0 ) {
					$text = esc_html__( '0 Comment', 'noor' );
				} elseif ( $nb_of_comments == 1 ) {
					$text = $nb_of_comments . ' ' . esc_html( $Comment );
				} else {
					$text = $nb_of_comments . ' ' . esc_html( $Comments );
				}

				$comments = sprintf( '<span class="post-comments">' . wp_kses( dima_get_svg_icon( 'ic_forum' ), dima_helper::dima_get_allowed_html_tag() ) . '<a href="%1$s" title="%2$s" class="meta-comments"> %3$s</a></span>', esc_url( $comments_link ), esc_attr( sprintf( esc_html__( 'Leave a comment on: &ldquo;%s&rdquo;', 'noor' ), $post_title ) ), $text );
			} else {
				$comments = '';
			}
		}

		$categories        = get_the_category();
		$separator         = ', ';
		$categories_output = '';
		$categories_list   = '';

		if ( $show_category ) {
			$cat_nb    = sizeof( $categories );
			$cat_class = '';
			if ( $cat_nb > 6 ) {
				$cat_class = ' full text-start float-start';
			}
			foreach ( $categories as $category ) {
				$categories_output .= '<a href="'
									  . get_category_link( $category->term_id )
									  . '" title="'
									  . esc_attr( sprintf( esc_html__( 'View all posts in: &ldquo;%s&rdquo;', 'noor' ), $category->name ) )
									  . '" rel="category tag"> '
									  . $category->name
									  . '</a>'
									  . $separator;
			}
			if ( $categories_output != '' ) {
				$categories_list = sprintf( '<span class="post-view' . esc_attr( $cat_class ) . '">' . wp_kses( dima_get_svg_icon( 'ic_bookmark' ), dima_helper::dima_get_allowed_html_tag() ) . '%1$s</span>', trim( $categories_output, $separator ) );
			}
		}
		/**
		 * Output
		 */
		$second_title = dima_helper::dima_am_i_true( dima_helper::dima_get_inherit_option( '_dima_meta_second_title_on_post', 'dima_second_title_on_post' ) );
		$hr           = '';
		if ( 'standard' === $type || ( is_singular() && ( ! is_page() ) || is_search() ) ) {
			if ( ( $second_title || ! is_single() ) && ! is_search() ) {
				$hr = '<hr class="entry-title-hr">';
			}
		}
		if ( dima_hide_entry_meta( $show_meta ) ) {
			printf( wp_kses( $hr, dima_helper::dima_get_allowed_html_tag() ) . '<div class="post-meta"><ul class="flex justify-between">%1$s%2$s<li class="flex">%3$s%4$s</li></ul></div>', $date, $author, $categories_list, $comments );
		}
	}
endif;


/**
 * Widget
 *
 * @param int  $comment_posts
 * @param int  $avatar_size
 * @param bool $with_avatar
 */
function dima_most_commented( $comment_posts = 5, $avatar_size = 40, $with_avatar = false ) {
	$comments = get_comments( 'status=approve&number=' . esc_attr( $comment_posts ) );
	foreach ( $comments as $comment ) {
		?>
		<li>
			<?php if ( $with_avatar ) { ?>
				<div class="dima-author-avatar" style="width:<?php echo esc_attr( $avatar_size ); ?>px">
					<?php echo get_avatar( $comment, $avatar_size ); ?>
				</div>
			<?php } ?>
			<a href="<?php echo get_permalink( $comment->comment_post_ID ); ?>#comment-<?php echo esc_attr( $comment->comment_ID ); ?>">
				<?php echo strip_tags( $comment->comment_author ); ?>
				: <?php echo wp_html_excerpt( $comment->comment_content, 80 ); ?>... </a>
		</li>
		<?php
	}
}

/**
 * Customizing Category Widget
 *
 * @param $output
 *
 * @return mixed
 */
function dima_cat_count_span_inline( $output ) {
	// For Product categories.
	$output = str_replace( '<span class="count">(', '<span class="count">', $output );
	// For Posts Product categories.
	$output = str_replace( '</a> (', '</a><span class="count">', $output );

	// For both.
	$output = str_replace( ')', '</span></a>', $output );

	return $output;
}

add_filter( 'wp_list_categories', 'dima_cat_count_span_inline' );


/**
 * @param $result
 * @param $tag
 *
 * @return mixed
 */
function dima_custom_email_confirmation_validation_filter( $result, $tag ) {
	$tag = new WPCF7_Shortcode( $tag );

	if ( 'your-email-confirm' == $tag->name ) {
		$your_email         = isset( $_POST['your-email'] ) ? trim( $_POST['your-email'] ) : '';
		$your_email_confirm = isset( $_POST['your-email-confirm'] ) ? trim( $_POST['your-email-confirm'] ) : '';
		if ( $your_email != $your_email_confirm ) {
			$result->invalidate( $tag, 'Are you sure this is the correct address?' );
		}
	}

	return $result;
}

add_filter( 'wpcf7_validate_email*', 'dima_custom_email_confirmation_validation_filter', 20, 2 );

/*
------------------------------------------------------------*/
//
// Remove some markup and styling from WordPress
//
/*------------------------------------------------------------*/

/*
------------------------------*/
// Remove 'style=' from Tag Cloud
/*------------------------------*/
if ( ! function_exists( 'dima_remove_tag_cloud_inline_style' ) ) :
	function dima_remove_tag_cloud_inline_style( $tag_string ) {
		return preg_replace( "/style='font-size:.+pt;'/", '', $tag_string );
	}

	add_filter( 'wp_generate_tag_cloud', 'dima_remove_tag_cloud_inline_style', 10, 1 );
endif;

/*
------------------------------*/
// Remove Recent Comments Style
/*------------------------------*/
if ( ! function_exists( 'dima_remove_recent_comments_style' ) ) :
	function dima_remove_recent_comments_style() {
		global $wp_widget_factory;
		remove_action(
			'wp_head',
			array(
				$wp_widget_factory->widgets['WP_Widget_Recent_Comments'],
				'recent_comments_style',
			)
		);
	}

	add_action( 'widgets_init', 'dima_remove_recent_comments_style' );
endif;

/*
------------------------------*/
// Remove Gallery Style
/*------------------------------*/
if ( ! function_exists( 'dima_remove_gallery_style' ) ) :
	function dima_remove_gallery_style() {
		add_filter( 'use_default_gallery_style', '__return_false' );
	}

	add_action( 'init', 'dima_remove_gallery_style' );
endif;

// Remove Gallery <br> Tags
if ( ! function_exists( 'dima_remove_br_gallery' ) ) :
	function dima_remove_br_gallery( $output ) {
		return preg_replace( ' / < br style = ( . * ? ) > / mi', '', $output );
	}

	add_filter( 'the_content', 'dima_remove_br_gallery', 11, 2 );
endif;

if ( function_exists( 'ADDTOANY_SHARE_SAVE_KIT' ) ) {
	/**
	 * Remove from excerpts where buttons could be redundant or awkward.
	 *
	 * @return void
	 */
	function dima_addtoany_excerpt_remove() {
		remove_action( 'pre_get_posts', 'A2A_SHARE_SAVE_pre_get_posts' );
	}

	add_action( 'init', 'dima_addtoany_excerpt_remove' );
}
