<?php
/**
 * Class DIMA_About_Me_Widget.
 *
 * @package Dima Framework
 * @subpackage Widget
 * @version   1.0.0
 * @since     1.0.0
 * @author    PixelDima <info@pixeldima.com>
 */

/*  Creating the widget  */

/**
 * Product Details Widget.
 *
 * Displays a product's details in a widget.
 *
 * @since 1.9
 * @return void
 */
class DIMA_EDD_Product_Details_Widget extends WP_Widget {

	/** Constructor */
	public function __construct() {

		$widget_ops  = array(
			'description' => sprintf( __( 'Display the details of a specific %s', 'noor-assistant' ), edd_get_label_singular() ),
			'classname'   => 'dima-widget-edd-box',
		);
		$control_ops = array(
			'width'  => 250,
			'height' => 400,
		);

		parent::__construct( false, $name = sprintf( __( '%1$s - %2$s Details', 'noor-assistant' ), strtoupper( DIMA_THEME_NAME ), edd_get_label_singular() ), $widget_ops, $control_ops );
	}

	/** @see WP_Widget::widget */
	public function widget( $args, $instance ) {
		$args['id'] = ( isset( $args['id'] ) ) ? $args['id'] : 'edd_download_details_widget';

		if ( ! empty( $instance['download_id'] ) ) {
			if ( 'current' === ( $instance['download_id'] ) ) {
				$instance['display_type'] = 'current';
				unset( $instance['download_id'] );
			} elseif ( is_numeric( $instance['download_id'] ) ) {
				$instance['display_type'] = 'specific';
			}
		}

		if ( ! isset( $instance['display_type'] ) || ( 'specific' === $instance['display_type'] && ! isset( $instance['download_id'] ) ) || ( 'current' == $instance['display_type'] && ! is_singular( 'download' ) ) ) {
			return;
		}

		// set correct download ID.
		if ( 'current' == $instance['display_type'] && is_singular( 'download' ) ) {
			$download_id = get_the_ID();
		} else {
			$download_id = absint( $instance['download_id'] );
		}

		// Since we can take a typed in value, make sure it's a download we're looking for.
		$download = get_post( $download_id );
		if ( ! is_object( $download ) || 'download' !== $download->post_type ) {
			return;
		}

		// Variables from widget settings.
		$title          = apply_filters( 'widget_title', $instance['title'], $instance, $args['id'] );
		$download_title = $instance['download_title'] ? apply_filters( 'edd_product_details_widget_download_title', '<h5>' . get_the_title( $download_id ) . '</h5>', $download_id ) : '';
		$categories     = $instance['categories'] ? $instance['categories'] : '';
		$tags           = $instance['tags'] ? $instance['tags'] : '';

		// Used by themes. Opens the widget.
		echo $args['before_widget'];
		// Display the widget title.
		if ( $title ) {
			echo $args['before_title'] . $title . $args['after_title'];
		}
		do_action( 'edd_product_details_widget_before_title', $instance, $download_id );
		// download title.
		echo $download_title;

		echo '<div class="dima_edd_product_details_widget dima-shadow has-hover">';
		?>
		<div class="download-price download-aside text-centr">
			<?php if ( edd_has_variable_prices( $download_id ) ) { ?>
				<!-- Get the price range -->
				<div class="purchase-price price-range">
					<?php echo edd_price_range( $download_id ); ?>
				</div>
			<?php } elseif ( function_exists( 'edd_cp_has_custom_pricing' ) && edd_cp_has_custom_pricing( $download_id ) ) { ?>
				<div class="purchase-price name-price">
					<?php _e( 'Name your price:', 'noor-assistant' ); ?>
				</div>
			<?php } elseif ( edd_is_free_download( $download_id ) ) { ?>
				<div class="purchase-price">
					<?php _e( 'Free', 'noor-assistant' ); ?>
				</div>
			<?php } else { ?>
				<!-- Get the single price -->
				<div class="purchase-price">
					<?php edd_price( $download_id ); ?>
				</div>
			<?php } ?>

			<?php
			// Get purchase button settings
			$behavior    = get_post_meta( $download_id, '_edd_button_behavior', true );
			$hide_button = get_post_meta( $download_id, '_dima_download_hide_purchase_link', true );

			// If it's a direct purchase show this text
			if ( edd_is_free_download( $download_id ) ) {
				$button_text = __( 'Download', 'noor-assistant' );
			} elseif ( $behavior == 'direct' ) {
				$button_text = __( 'Buy Now', 'noor-assistant' );
			} else {
				// if it's an add to cart purchase, get the text from EDD options
				$button_text = ! empty( $edd_options['add_to_cart_text'] ) ? $edd_options['add_to_cart_text'] : __( 'Purchase', 'noor-assistant' );
			}

			?>

			<?php
			// Show the button unless set to not show
			if ( dima_helper::dima_am_i_true( $hide_button ) || empty( $hide_button ) ) {
				echo edd_get_purchase_link(
					array(
						'download_id' => $download_id,
						'price'       => false,
						'direct'      => edd_get_download_button_behavior( $download_id ) == 'direct' ? true : false,
						'text'        => $button_text,
					)
				);
			}
			?>

			<?php do_action( 'dima_below_purchase_button' ); ?>
		</div><!-- .download-price -->

		<?php

		do_action( 'edd_product_details_widget_before_purchase_button', $instance, $download_id );

		do_action( 'dima_below_purchase_sidebar', 0, $categories, $tags );

		do_action( 'edd_product_details_widget_before_end', $instance, $download_id );
		echo '</div>';
		// Used by themes. Closes the widget.
		echo $args['after_widget'];
	}

	/** @see WP_Widget::form */
	public function form( $instance ) {
		// Set up some default widget settings.
		$defaults = array(
			'title'           => sprintf( __( '%s Details', 'noor-assistant' ), edd_get_label_singular() ),
			'display_type'    => 'current',
			'download_id'     => false,
			'download_title'  => 'on',
			'purchase_button' => 'on',
			'categories'      => 'on',
			'tags'            => 'on',
		);

		$instance = wp_parse_args( (array) $instance, $defaults );
		?>

		<?php
		if ( 'current' === ( $instance['download_id'] ) ) {
			$instance['display_type'] = 'current';
			$instance['download_id']  = false;
		} elseif ( is_numeric( $instance['download_id'] ) ) {
			$instance['display_type'] = 'specific';
		}

		?>

		<!-- Title -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php _e( 'Title:', 'noor-assistant' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
				   name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
				   value="<?php echo $instance['title']; ?>"/>
		</p>

		<p>
			<?php _e( 'Display Type:', 'noor-assistant' ); ?><br/>
			<input type="radio"
				   onchange="jQuery(this).parent().next('.download-details-selector').hide();" <?php checked( 'current', $instance['display_type'], true ); ?>
				   value="current" name="<?php echo esc_attr( $this->get_field_name( 'display_type' ) ); ?>"
				   id="<?php echo esc_attr( $this->get_field_id( 'display_type' ) ); ?>-current"><label
					for="<?php echo esc_attr( $this->get_field_id( 'display_type' ) ); ?>-current"><?php _e( 'Current', 'noor-assistant' ); ?></label>
			<input type="radio"
				   onchange="jQuery(this).parent().next('.download-details-selector').show();" <?php checked( 'specific', $instance['display_type'], true ); ?>
				   value="specific" name="<?php echo esc_attr( $this->get_field_name( 'display_type' ) ); ?>"
				   id="<?php echo esc_attr( $this->get_field_id( 'display_type' ) ); ?>-specific"><label
					for="<?php echo esc_attr( $this->get_field_id( 'display_type' ) ); ?>-specific"><?php _e( 'Specific', 'noor-assistant' ); ?></label>
		</p>

		<!-- Download -->
		<?php $display = 'current' === $instance['display_type'] ? ' style="display: none;"' : ''; ?>
		<p class="download-details-selector" <?php echo $display; ?>>
			<label for="<?php echo esc_attr( $this->get_field_id( 'download_id' ) ); ?>"><?php printf( __( '%s:', 'noor-assistant' ), edd_get_label_singular() ); ?></label>
			<?php $download_count = wp_count_posts( 'download' ); ?>
			<?php if ( $download_count->publish < 1000 ) : ?>
				<?php
				$args      = array(
					'post_type'      => 'download',
					'posts_per_page' => - 1,
					'post_status'    => 'publish',
				);
				$downloads = get_posts( $args );
				?>
				<select class="widefat" name="<?php echo esc_attr( $this->get_field_name( 'download_id' ) ); ?>"
						id="<?php echo esc_attr( $this->get_field_id( 'download_id' ) ); ?>">
					<?php foreach ( $downloads as $download ) { ?>
						<option <?php selected( absint( $instance['download_id'] ), $download->ID ); ?>
								value="<?php echo esc_attr( $download->ID ); ?>"><?php echo $download->post_title; ?></option>
					<?php } ?>
				</select>
			<?php else : ?>
				<br/>
				<input type="text" value="<?php echo esc_attr( $instance['download_id'] ); ?>"
					   placeholder="<?php printf( __( '%s ID', 'noor-assistant' ), edd_get_label_singular() ); ?>"
					   name="<?php echo esc_attr( $this->get_field_name( 'download_id' ) ); ?>"
					   id="<?php echo esc_attr( $this->get_field_id( 'download_id' ) ); ?>">
			<?php endif; ?>
		</p>

		<!-- Download title -->
		<p>
			<input <?php checked( $instance['download_title'], 'on' ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'download_title' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'download_title' ) ); ?>" type="checkbox"/>
			<label for="<?php echo esc_attr( $this->get_field_id( 'download_title' ) ); ?>"><?php printf( __( 'Show %s Title', 'noor-assistant' ), edd_get_label_singular() ); ?></label>
		</p>

		<!-- Show purchase button -->
		<p>
			<input <?php checked( $instance['purchase_button'], 'on' ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'purchase_button' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'purchase_button' ) ); ?>" type="checkbox"/>
			<label for="<?php echo esc_attr( $this->get_field_id( 'purchase_button' ) ); ?>"><?php _e( 'Show Purchase Button', 'noor-assistant' ); ?></label>
		</p>

		<!-- Show download categories -->
		<p>
			<?php $category_labels = edd_get_taxonomy_labels( 'download_category' ); ?>
			<input <?php checked( $instance['categories'], 'on' ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'categories' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'categories' ) ); ?>" type="checkbox"/>
			<label for="<?php echo esc_attr( $this->get_field_id( 'categories' ) ); ?>"><?php printf( __( 'Show %s', 'noor-assistant' ), $category_labels['name'] ); ?></label>
		</p>

		<!-- Show download tags -->
		<p>
			<?php $tag_labels = edd_get_taxonomy_labels( 'download_tag' ); ?>
			<input <?php checked( $instance['tags'], 'on' ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'tags' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'tags' ) ); ?>" type="checkbox"/>
			<label for="<?php echo esc_attr( $this->get_field_id( 'tags' ) ); ?>"><?php printf( __( 'Show %s', 'noor-assistant' ), $tag_labels['name'] ); ?></label>
		</p>

		<?php do_action( 'edd_product_details_widget_form', $instance ); ?>
		<?php
	}

	/** @see WP_Widget::update */
	public function update( $new_instance, $old_instance ) {
		$instance = $old_instance;

		$instance['title']           = strip_tags( $new_instance['title'] );
		$instance['download_id']     = strip_tags( $new_instance['download_id'] );
		$instance['display_type']    = isset( $new_instance['display_type'] ) ? strip_tags( $new_instance['display_type'] ) : '';
		$instance['download_title']  = isset( $new_instance['download_title'] ) ? $new_instance['download_title'] : '';
		$instance['purchase_button'] = isset( $new_instance['purchase_button'] ) ? $new_instance['purchase_button'] : '';
		$instance['categories']      = isset( $new_instance['categories'] ) ? $new_instance['categories'] : '';
		$instance['tags']            = isset( $new_instance['tags'] ) ? $new_instance['tags'] : '';

		do_action( 'edd_product_details_widget_update', $instance );

		// If the new view is 'current download' then remove the specific download ID
		if ( 'current' === $instance['display_type'] ) {
			unset( $instance['download_id'] );
		}

		return $instance;
	}

}

/**
 * Register and load the widget
 */
function DIMA_EDD_Product_Details_Widget_Register() {
	register_widget( 'DIMA_EDD_Product_Details_Widget' );
}

add_action( 'widgets_init', 'DIMA_EDD_Product_Details_Widget_Register' );
