<?php
/**
 * Class DIMA_Instagram_Widget.
 *
 * @package Dima Framework
 * @subpackage Widget
 * @version   1.0.0
 * @since     1.0.0
 * @author    PixelDima <info@pixeldima.com>
 *
 */

function dima_insta_widget() {
	register_widget( 'DIMA_Instagram_Widget' );
}

add_action( 'widgets_init', 'dima_insta_widget' );

Class DIMA_Instagram_Widget extends WP_Widget {

	function __construct() {
		$widget_ops  = array(
			'description' => esc_html__( 'Display your latest Instagram photos', 'noor' ),
			'classname'   => 'instagram-widget'
		);
		$control_ops = array( 'width' => 250, 'height' => 400 );
		parent::__construct( false, $name = DIMA_THEME_NAME . ' - ' . esc_html__( 'Instagram', 'noor' ), $widget_ops, $control_ops );
	}

	function widget( $args, $instance ) {

		$title    = empty( $instance['title'] ) ? '' : apply_filters( 'widget_title', $instance['title'] );
		$username = empty( $instance['username'] ) ? '' : $instance['username'];
		$limit    = empty( $instance['number'] ) ? 9 : $instance['number'];
		$size     = empty( $instance['size'] ) ? 'large' : $instance['size'];
		$columns  = empty( $instance['columns'] ) ? 3 : $instance['columns'];
		$target   = empty( $instance['target'] ) ? '_self' : $instance['target'];
		$link     = empty( $instance['link'] ) ? '' : $instance['link'];

		echo $args['before_widget'];

		if ( ! empty( $title ) ) {
			echo $args['before_title'] . wp_kses_post( $title ) . $args['after_title'];
		};

		do_action( 'dima_insta_before_widget', $instance );

		if ( '' !== $username ) {

			$media_array = $this->scrape_instagram( $username );

			if ( is_wp_error( $media_array ) ) {

				echo wp_kses_post( $media_array->get_error_message() );

			} else {

				$insta_classes = apply_filters( 'dima_insta_images_class', 'dima-instagram-images dima-instagram-size-' . $size );
				$row_classes   = apply_filters( 'dima_insta_row_class', 'dima-instagram-row' );
				$child_classes = apply_filters( 'dima_insta_image_class', 'dima-instagram-image' );
				$aclass        = apply_filters( 'dima_insta_a_class', '' );
				$imgclass      = apply_filters( 'dima_insta_imag_class', '' );

				$lightboxdata = "";
				if ( $target == "lightbox" ) {
					$lightboxdata = 'data-fancybox=galleryInsta';
					$_target      = '';
				} else {
					$_target = 'target="' . esc_attr( $target ) . '"';
				}

				// filter for images only?
				if ( $images_only = apply_filters( 'dima_insta_images_only', false ) ) {
					$media_array = array_filter( $media_array, array( $this, 'images_only' ) );
				}
				$row_count = 0;

				// slice list down to required limit.
				$media_array = array_slice( $media_array, 0, $limit );
				// filters for custom classes.

				?>
                <div class="<?php echo esc_attr( $insta_classes ); ?>"><?php
				foreach ( $media_array as $item ) {

					if ( $target == "lightbox" ) {
						$item['link'] = $item['original'];
					}

					if ( $row_count == 0 ) {
						echo "<div class='{$row_classes}'>";
					}

					$row_count ++;

					if ( $target == "lightbox" ) {
						$item['link'] = $item['original'];
					}
					echo '<div class="' . esc_attr( $child_classes ) . '">';
					echo '<a ' . esc_attr( $lightboxdata ) . ' data-caption="' . esc_attr( $item['description'] ) . '" href="' . esc_url( $item['link'] ) . '"  ' . esc_attr( $_target ) . ' class="' . esc_attr( $aclass ) . '">';
					echo '<img src="' . esc_url( $item[ $size ] ) . '"  alt="' . esc_attr( $item['description'] ) . '" class="' . esc_attr( $imgclass ) . '"/>';
					echo '</a></div>';

					if ( $row_count % $columns == 0 ) {
						echo '</div>';
						$row_count = 0;
					}
				}
				?></div><?php
			}
		}

		$linkaclass = apply_filters( 'dima_insta_linka_class', 'instagram-button no-rounded button-block fill small button' );

		switch ( substr( $username, 0, 1 ) ) {
			case '#':
				$url = '//instagram.com/explore/tags/' . str_replace( '#', '', $username );
				break;

			default:
				$url = '//instagram.com/' . str_replace( '@', '', $username );
				break;
		}

		if ( '' !== $link ) {
			?><a
            href="<?php echo trailingslashit( esc_url( $url ) ); ?>" rel="me"
            target="<?php echo esc_attr( $target ); ?>"
            class="<?php echo esc_attr( $linkaclass ); ?>"><?php echo wp_kses_post( $link ); ?></a><?php
		}

		do_action( 'dima_insta_after_widget', $instance );

		echo $args['after_widget'];
	}

	function form( $instance ) {
		$instance = wp_parse_args( (array) $instance, array(
			'title'    => __( 'Instagram', 'noor' ),
			'username' => '',
			'size'     => 'large',
			'link'     => __( 'Follow Me!', 'noor' ),
			'number'   => 9,
			'target'   => '_self',
			'columns'  => 3
		) );
		$title    = $instance['title'];
		$username = $instance['username'];
		$number   = absint( $instance['number'] );
		$size     = $instance['size'];
		$columns  = $instance['columns'];
		$target   = $instance['target'];
		$link     = $instance['link'];
		?>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title', 'noor' ); ?>
                : <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
                         name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
                         value="<?php echo esc_attr( $title ); ?>"/></label></p>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>"><?php esc_html_e( '@username or #tag', 'noor' ); ?>
                : <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>"
                         name="<?php echo esc_attr( $this->get_field_name( 'username' ) ); ?>" type="text"
                         value="<?php echo esc_attr( $username ); ?>"/></label></p>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"><?php esc_html_e( 'Number of photos', 'noor' ); ?>
                : <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"
                         name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>" type="text"
                         value="<?php echo esc_attr( $number ); ?>"/></label></p>
        <p><label
                    for="<?php echo esc_attr( $this->get_field_id( 'columns' ) ); ?>"><?php esc_html_e( 'Number of columns', 'noor' ); ?>
                :
                <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'columns' ) ); ?>"
                       name="<?php echo esc_attr( $this->get_field_name( 'columns' ) ); ?>" type="text"
                       value="<?php echo esc_attr( $columns ); ?>"/></label></p>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'size' ) ); ?>"><?php esc_html_e( 'Photo size', 'noor' ); ?>
                :</label>
            <select id="<?php echo esc_attr( $this->get_field_id( 'size' ) ); ?>"
                    name="<?php echo esc_attr( $this->get_field_name( 'size' ) ); ?>" class="widefat">
                <option value="thumbnail" <?php selected( 'thumbnail', $size ); ?>><?php esc_html_e( 'Thumbnail', 'noor' ); ?></option>
                <option value="small" <?php selected( 'small', $size ); ?>><?php esc_html_e( 'Small', 'noor' ); ?></option>
                <option value="large" <?php selected( 'large', $size ); ?>><?php esc_html_e( 'Large', 'noor' ); ?></option>
                <option value="original" <?php selected( 'original', $size ); ?>><?php esc_html_e( 'Original', 'noor' ); ?></option>
            </select>
        </p>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'target' ) ); ?>"><?php esc_html_e( 'Open links in', 'noor' ); ?>
                :</label>
            <select id="<?php echo esc_attr( $this->get_field_id( 'target' ) ); ?>"
                    name="<?php echo esc_attr( $this->get_field_name( 'target' ) ); ?>" class="widefat">
                <option value="lightbox" <?php selected( 'lightbox', $target ) ?>><?php esc_html_e( 'Lightbox', 'noor' ); ?></option>
                <option value="_self" <?php selected( '_self', $target ); ?>><?php esc_html_e( 'Current window (_self)', 'noor' ); ?></option>
                <option value="_blank" <?php selected( '_blank', $target ); ?>><?php esc_html_e( 'New window (_blank)', 'noor' ); ?></option>
            </select>
        </p>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'link' ) ); ?>"><?php esc_html_e( 'Link text', 'noor' ); ?>
                : <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'link' ) ); ?>"
                         name="<?php echo esc_attr( $this->get_field_name( 'link' ) ); ?>" type="text"
                         value="<?php echo esc_attr( $link ); ?>"/></label></p>
		<?php

	}

	function update( $new_instance, $old_instance ) {
		$instance             = $old_instance;
		$instance['title']    = strip_tags( $new_instance['title'] );
		$instance['username'] = trim( strip_tags( $new_instance['username'] ) );
		$instance['number']   = ! absint( $new_instance['number'] ) ? 9 : $new_instance['number'];
		$instance['size']     = ( ( 'thumbnail' === $new_instance['size'] || 'large' === $new_instance['size'] || 'small' === $new_instance['size'] || 'original' === $new_instance['size'] ) ? $new_instance['size'] : 'large' );
		$instance['columns']  = ! absint( $new_instance['columns'] ) ? 3 : $new_instance['columns'];
		$instance['target']   = ( ( '_self' === $new_instance['target'] || '_blank' === $new_instance['target'] || 'lightbox' === $new_instance['target'] ) ? $new_instance['target'] : '_self' );
		$instance['link']     = strip_tags( $new_instance['link'] );
		if ( $instance['columns'] > 6 ) {
			$instance['columns'] = 6;
		}
		if ( $instance['columns'] < 1 ) {
			$instance['columns'] = 1;
		}

		return $instance;
	}

	// based on https://gist.github.com/cosmocatalano/4544576.
	function scrape_instagram( $username ) {

		$username = trim( strtolower( $username ) );

		switch ( substr( $username, 0, 1 ) ) {
			case '#':
				$url              = 'https://instagram.com/explore/tags/' . str_replace( '#', '', $username );
				$transient_prefix = 'h';
				break;

			default:
				$url              = 'https://instagram.com/' . str_replace( '@', '', $username );
				$transient_prefix = 'u';
				break;
		}

		if ( false === ( $instagram = get_transient( 'dima_insta_' . $transient_prefix . '-' . sanitize_title_with_dashes( $username ) ) ) ) {

			$remote = wp_remote_get( $url );

			if ( is_wp_error( $remote ) ) {
				return new WP_Error( 'site_down', esc_html__( 'Unable to communicate with Instagram.', 'noor' ) );
			}

			if ( 200 !== wp_remote_retrieve_response_code( $remote ) ) {
				return new WP_Error( 'invalid_response', esc_html__( 'Instagram did not return a 200.', 'noor' ) );
			}

			$shards      = explode( 'window._sharedData = ', $remote['body'] );
			$insta_json  = explode( ';</script>', $shards[1] );
			$insta_array = json_decode( $insta_json[0], true );

			if ( ! $insta_array ) {
				return new WP_Error( 'bad_json', esc_html__( 'Instagram has returned invalid data.', 'noor' ) );
			}

			if ( isset( $insta_array['entry_data']['ProfilePage'][0]['graphql']['user']['edge_owner_to_timeline_media']['edges'] ) ) {
				$images = $insta_array['entry_data']['ProfilePage'][0]['graphql']['user']['edge_owner_to_timeline_media']['edges'];
			} elseif ( isset( $insta_array['entry_data']['TagPage'][0]['graphql']['hashtag']['edge_hashtag_to_media']['edges'] ) ) {
				$images = $insta_array['entry_data']['TagPage'][0]['graphql']['hashtag']['edge_hashtag_to_media']['edges'];
			} else {
				return new WP_Error( 'bad_json_2', esc_html__( 'Instagram has returned invalid data.', 'noor' ) );
			}

			if ( ! is_array( $images ) ) {
				return new WP_Error( 'bad_array', esc_html__( 'Instagram has returned invalid data.', 'noor' ) );
			}

			$instagram = array();

			foreach ( $images as $image ) {
				if ( true === $image['node']['is_video'] ) {
					$type = 'video';
				} else {
					$type = 'image';
				}

				$caption = __( 'Instagram Image', 'noor' );
				if ( ! empty( $image['node']['edge_media_to_caption']['edges'][0]['node']['text'] ) ) {
					$caption = wp_kses( $image['node']['edge_media_to_caption']['edges'][0]['node']['text'], array() );
				}

				$instagram[] = array(
					'description' => $caption,
					'link'        => trailingslashit( '//instagram.com/p/' . $image['node']['shortcode'] ),
					'time'        => $image['node']['taken_at_timestamp'],
					'comments'    => $image['node']['edge_media_to_comment']['count'],
					'likes'       => $image['node']['edge_liked_by']['count'],
					'thumbnail'   => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][0]['src'] ),
					'small'       => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][2]['src'] ),
					'large'       => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][4]['src'] ),
					'original'    => preg_replace( '/^https?\:/i', '', $image['node']['display_url'] ),
					'type'        => $type,
				);
			}

			// do not set an empty transient - should help catch private or empty accounts.
			if ( ! empty( $instagram ) ) {
				$instagram = base64_encode( serialize( $instagram ) );
				set_transient( 'dima_insta_' . $transient_prefix . '-' . sanitize_title_with_dashes( $username ), $instagram, apply_filters( 'null_instagram_cache_time', HOUR_IN_SECONDS * 2 ) );
			}
		}

		if ( ! empty( $instagram ) ) {
			return unserialize( base64_decode( $instagram ) );
		} else {
			return new WP_Error( 'no_images', esc_html__( 'Instagram did not return any images.', 'noor' ) );
		}
	}

	function images_only( $media_item ) {

		if ( 'image' === $media_item['type'] ) {
			return true;
		}

		return false;
	}
}