<?php
/**
 * Shortcode mapping of NewsPlus shortcodes
 * for KingComposer plugin
 *
 * @package NewsPlus
 * @subpackage NewsPlus_Shortcodes
 * @since 3.0
 * @version 3.1.4
 */

add_action( 'init', 'newsplus_kc_map_shortcodes' );

function newsplus_kc_map_shortcodes() {

	// Load prebuilt templates
	if ( function_exists( 'kc_prebuilt_template' ) ) {
		$xml_path1 = plugin_dir_path( __FILE__ ) . 'default.xml';
		$xml_path2 = plugin_dir_path( __FILE__ ) . 'food.xml';
		kc_prebuilt_template( __( 'Default', 'newsplus' ), $xml_path1 );
		kc_prebuilt_template( __( 'Food', 'newsplus' ), $xml_path2 );
	}

	// Add shortcode map
	if ( function_exists( 'kc_add_map' ) ) {

		// Categories array
		$cat_arr = array();
		$categories = get_categories();
		foreach( $categories as $category ){
		  $cat_arr[ $category->term_id ] = $category->name;
		}

		// Post types
		$post_type_arr = array();
		foreach ( get_post_types( array( 'public' => true ) ) as $post_type ) {
			if ( ! in_array( $post_type, array( 'revision', 'attachment', 'nav_menu_item' ) ) ) {
				$post_type_arr[ $post_type ] = $post_type;
			}
		}

		// Find taxonomies for use in dropdown
		$taxonomies = get_taxonomies( array( 'public' => true ) );
		$tax_arr = array();
		$terms_arr = array();
		foreach ( $taxonomies as $taxonomy ) {
			$tax = get_taxonomy( $taxonomy );

			// Get terms for each taxonomy
			$term_arr = array();
			$terms = get_terms( array(
				'taxonomy' => $taxonomy,
				'hide_empty' => true,
				'number' => apply_filters( 'newsplus_cat_limit', 999 )
			) );
			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ){
				foreach ( $terms as $term ) {
					$term_arr[ $term->slug ] = $term->name;
				}
			}

			// Store taxonomies in array
			if ( ! in_array( $taxonomy, array( 'nav_menu', 'link_category', 'post_format', 'product_type', 'product_shipping_class' ) ) ) {
				$tax_arr[ $taxonomy ] = $tax->labels->name;
				$terms_arr[ $tax->labels->name ] = $term_arr;
			}
		}

		/**
		 * Query parameters
		 * reusable for various post shortcodes
		 */

		// Query params
		$wp_query_params = array(

			array(
				'type' => 'multiple',
				'label' => __( 'Show data from', 'newsplus' ),
				'name' => 'post_type',
				'options' => $post_type_arr,
				'description' => __( 'Select post types. Use Ctrl + Select or Command + select for mutiple selection.', 'newsplus' )
			),

			array(
				'label' => __( 'Number of Posts', 'newsplus' ),
				'name' => 'num',
				'type' => 'number_slider',
				'options' => array(
					'min' => 1,
					'max' => 100,
					'unit' => '',
					'ratio' => '1',
					'show_input' => true
				),
				'value' => '5',
				'description' => __( 'Provide number of posts to show. E.g. 10', 'newsplus' )

			),

			array(
				'type' => 'multiple',
				'label' => __( 'Taxonomy', 'newsplus' ),
				'name' => 'taxonomy',
				'options' => $tax_arr,
				'description' => __( 'Select taxonomy', 'newsplus' )
			),

			array(
				'name'     => 'terms',
				'label'    => 'Terms',
				'type'     => 'multiple',
				'options'  => $terms_arr,
				'description' => __( 'Select terms', 'newsplus' )
			),

			array(
				'type' => 'autocomplete',
				'label' => __( 'Filter by Post, page or Custom post types', 'newsplus' ),
				'name' => 'post__in',
				'value' => '',
				'description' => __( 'Type keywords to search for a post or page. Then click on result to include a post or page.', 'newsplus' ),
				'admin_label' => true,
			),

			array(
				'type' => 'text',
				'label' => __( 'Filter by Author IDs', 'newsplus' ),
				'name' => 'author__in',
				'value' => '',
				'description' => __( 'Provide numeric IDs of Authors, separated by comma. E.g. 2,6', 'newsplus' )
			),

			array(
				'label' => __( 'Offset', 'newsplus' ),
				'name' => 'offset',
				'type' => 'number_slider',
				'options' => array(
					'min' => 0,
					'max' => 100,
					'unit' => '',
					'ratio' => '1',
					'show_input' => true
				),
				'value' => 0,
				'description' => __( 'Provide an offset number. E.g. 2. Offset is used to skip a particular number of posts from loop.', 'newsplus' )
			),

			array(
				'type' => 'dropdown',
				'label' => __( 'Order', 'newsplus' ),
				'name' => 'order',
				'options' => array(
						'DESC' => esc_attr__( 'Descending', 'newsplus' ),
						'ASC' => esc_attr__( 'Ascending', 'newsplus' )
				),
				'value' => 'DESC',
				'description' => __( 'Select posts order.', 'newsplus' )
			),

			array(
				'type' => 'dropdown',
				'label' => __( 'Order by', 'newsplus' ),
				'name' => 'orderby',
				'options' => array(
						'none' => esc_attr__( 'None', 'newsplus' ),
						'ID' => esc_attr__( 'ID', 'newsplus' ),
						'author' => esc_attr__( 'Author', 'newsplus' ),
						'title' => esc_attr__( 'Title', 'newsplus' ),
						'name' => esc_attr__( 'Name', 'newsplus' ),
						'type' => esc_attr__( 'Post Type', 'newsplus' ),
						'date' => esc_attr__( 'Date', 'newsplus' ),
						'modified' => esc_attr__( 'Last Modified', 'newsplus' ),
						'parent' => esc_attr__( 'Parent ID', 'newsplus' ),
						'rand' => esc_attr__( 'Random', 'newsplus' ),
						'comment_count' => esc_attr__( 'Comment Count', 'newsplus' ),
						'menu_order' => esc_attr__( 'Menu Order', 'newsplus' ),
						'post__in' => esc_attr__( 'Post In', 'newsplus' ),
				),
				'value' => 'date',
				'description' => __( 'Select posts orderby criteria.', 'newsplus' )
			),

			array(
				'name'        => 'ignore_sticky_posts',
				'label'       => 'Sticky Posts',
				'type'        => 'checkbox',
				'description' => __( 'Check to ignore sticky posts', 'newsplus' ),
				'options'     => array( 'true' => __( 'Ignore sticky posts', 'newsplus' ) )
			)

		);

		/**
		 * Content parameters
		 * reusable in different post shortcodes
		 */

		$content_params =  array(

			array(
				'label' => __( 'Heading tag', 'newsplus' ),
				'name' => 'htag',
				'type' => 'dropdown',
				'options' => array(
					'h1' => 'h1',
					'h2' => 'h2',
					'h3' => 'h3',
					'h4' => 'h4',
					'h5' => 'h5',
					'h6' => 'h6'
				),
				'value' => 'h2',
				'description' => __( 'Select heading tag for post title.', 'newsplus' )
			),

			array(
				'label' => __( 'Heading size', 'newsplus' ),
				'name' => 'hsize',
				'type' => 'dropdown',
				'options' => array(
					''   => _x( 'Select', 'dropdown label', 'newsplus' ),
					'12' => '12px',
					'14' => '14px',
					'16' => '16px',
					'18' => '18px',
					'20' => '20px',
					'24' => '24px',
					'34' => '34px'
				),
				'value' => '',
				'description' => __( 'Select heading font size for post title.', 'newsplus' )
			),

			array(
				'name' => 'hide_excerpt',
				'label' => __( 'Hide post excerpt', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide post excerpt', 'newsplus' )
				),
				'description' => __( 'Check to hide post excerpt in post module.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'template',
					'hide_when' => 'list-small'
				)
			),

			array(
				'label' => __( 'Excerpt tag', 'newsplus' ),
				'name' => 'ptag',
				'type' => 'dropdown',
				'options' => array(
					'p' => 'p',
					'span' => 'span',
					'h2' => 'h2',
					'h3' => 'h3',
					'h4' => 'h4',
					'h5' => 'h5',
					'h6' => 'h6'
				),
				'value' => 'p',
				'description' => __( 'Select tag for post excerpt.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'hide_excerpt',
					'hide_when' => 'true'
				)
			),

			array(
				'label' => __( 'Excerpt Size', 'newsplus' ),
				'name' => 'psize',
				'type' => 'dropdown',
				'options' => array(
					''   => _x( 'Select', 'dropdown label', 'newsplus' ),
					'12' => '12px',
					'13' => '13px',
					'14' => '14px',
					'16' => '16px',
					'18' => '18px'
				),
				'value' => '',
				'description' => __( 'Select font size for post excerpt.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'hide_excerpt',
					'hide_when' => 'true'
				)
			),

			array(
				'name' => 'use_word_length',
				'label' => __( 'Use word length', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Use word length', 'newsplus' )
				),
				'description' => __( 'Check to enable trimming in word length instead of character length.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'hide_excerpt',
					'hide_when' => 'true'
				)
			),

			array(
				'label' => __( 'Excerpt length', 'newsplus' ),
				'name' => 'excerpt_length',
				'type' => 'number_slider',
				'options' => array(
					'min' => 1,
					'max' => 500,
					'unit' => '',
					'ratio' => '1',
					'show_input' => true
				),
				'value' => '140',
				'description' => __( 'Provide excerpt length in characters or words. If enabled word length, provide length in words. E.g. 20', 'newsplus' ),
				'relation' => array(
					'parent'    => 'hide_excerpt',
					'hide_when' => 'true'
				)
			),

			array(
				'name' => 'readmore',
				'label' => __( 'Show readmore link', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Show readmore link', 'newsplus' )
				),
				//'value' => 'true', // remove this if you do not need a default content
				'description' => __( 'Check to show a readmore link.', 'newsplus' )
			),

			array(
				'label' => __( 'Readmore text', 'newsplus' ),
				'name' => 'readmore_text',
				'type' => 'text',
				'value' => 'Read more',
				'description' => __( 'The text for readmore button.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'readmore',
					'show_when' => 'true'
				)
			),

			array(
				'name' => 'use_short_title',
				'label' => __( 'Use short title', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Enable short titles support', 'newsplus' )
				),
				'description' => __( 'Check to enable short titles. The posts must have "np_short_title" custom field with the value as short title.', 'newsplus' )
			),

			array(
				'name' => 'ext_link',
				'label' => __( 'Use external links', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Enable external links', 'newsplus' )
				),
				'description' => __( 'Check to enable external link on post title and thumbnail. The posts must have custom field "np_custom_link" with value as external link.', 'newsplus' )
			)
		);

		$content_params_sub = array(
			array(
				'label' => __( 'Heading tag for sub tiles', 'newsplus' ),
				'name' => 'htag_sub',
				'type' => 'dropdown',
				'options' => array(
					'h1' => 'h1',
					'h2' => 'h2',
					'h3' => 'h3',
					'h4' => 'h4',
					'h5' => 'h5',
					'h6' => 'h6'
				),
				'value' => 'h2',
				'description' => __( 'Select heading tag for post title in sub tiles.', 'newsplus' )
			),

			array(
				'label' => __( 'Heading size for sub tiles', 'newsplus' ),
				'name' => 'hsize_sub',
				'type' => 'dropdown',
				'options' => array(
					'12' => '12px',
					'14' => '14px',
					'16' => '16px',
					'18' => '18px',
					'20' => '20px',
					'24' => '24px',
					'34' => '34px'
				),
				'value' => '14',
				'description' => __( 'Select heading font size for post title in sub tiles.', 'newsplus' )
			)
		);

		/**
		 * Meta params
		 */

		 $meta_params = array(
			array(
				'name' => 'hide_cats',
				'label' => __( 'Hide category links', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide category links', 'newsplus' )
				),
				'description' => __( 'Check to hide category links in post module.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'template',
					'hide_when' => 'list-small'
				)
			),

			array(
				'name' => 'hide_reviews',
				'label' => __( 'Hide review stars', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide review stars', 'newsplus' )
				),
				'description' => __( 'Check to hide review stars in post module. The review feature requires WP Review plugin.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'template',
					'hide_when' => 'list-small'
				)
			),

			array(
				'name' => 'hide_date',
				'label' => __( 'Hide post date', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide post publish date', 'newsplus' )
				),
				'description' => __( 'Check to hide post publish date in post module.', 'newsplus' )
			),

			array(
				'label' => __( 'Date format', 'newsplus' ),
				'name' => 'date_format',
				'type' => 'text',
				'value' => get_option( 'date_format' ),
				'description' => __( 'Provide a <a target="_blank" href="https://codex.wordpress.org/Formatting_Date_and_Time">date format</a> for post date. Use \'human\' for "xx days ago" format.', 'newsplus' ),
				'relation' => array(
					'parent'    => 'hide_date',
					'hide_when' => 'true'
				)
			),

			array(
				'name' => 'hide_author',
				'label' => __( 'Hide author link', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide author link', 'newsplus' )
				),
				'description' => __( 'Check to hide author link in post module.', 'newsplus' )
			),

			array(
				'name' => 'hide_views',
				'label' => __( 'Hide views number', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide views number', 'newsplus' )
				),
				'description' => __( 'Check to hide views number in post module. The views feature requires Post Views Counter plugin.', 'newsplus' )
			),

			array(
				'name' => 'hide_comments',
				'label' => __( 'Hide comments link', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide comments link', 'newsplus' )
				),
				'description' => __( 'Check to hide comments link in post module.', 'newsplus' )
			),

			array(
				'name' => 'show_avatar',
				'label' => __( 'Show author avatar', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Show author avatar', 'newsplus' )
				),
				'description' => __( 'Check to show author avatar in post meta', 'newsplus' )
			)
		);

		 $meta_params_sub = array(
			array(
				'name' => 'hide_cats_sub',
				'label' => __( 'Hide category links', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide category links', 'newsplus' )
				),
				'value' => 'true',
				'description' => __( 'Check to hide category links in post module.', 'newsplus' )
			),

			array(
				'name' => 'hide_reviews_sub',
				'label' => __( 'Hide review stars', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide review stars', 'newsplus' )
				),
				'value' => 'true',
				'description' => __( 'Check to hide review stars in post module. The review feature requires WP Review plugin.', 'newsplus' )
			),

			array(
				'name' => 'hide_date_sub',
				'label' => __( 'Hide post date', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide post publish date', 'newsplus' )
				),
				'description' => __( 'Check to hide post publish date in post module.', 'newsplus' )
			),

			array(
				'name' => 'hide_author_sub',
				'label' => __( 'Hide author link', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide author link', 'newsplus' )
				),
				'description' => __( 'Check to hide author link in post module.', 'newsplus' )
			),

			array(
				'name' => 'hide_views_sub',
				'label' => __( 'Hide views number', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide views number', 'newsplus' )
				),
				'value' => 'true',
				'description' => __( 'Check to hide views number in post module. The views feature requires Post Views Counter plugin.', 'newsplus' )
			),

			array(
				'name' => 'hide_comments_sub',
				'label' => __( 'Hide comments link', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Hide comments link', 'newsplus' )
				),
				'value' => 'true',
				'description' => __( 'Check to hide comments link in post module.', 'newsplus' )
			),

			array(
				'name' => 'show_avatar_sub',
				'label' => __( 'Show author avatar', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Show author avatar', 'newsplus' )
				),
				'description' => __( 'Check to show author avatar in post meta', 'newsplus' )
			),

			array(
				'name' => 'readmore_sub',
				'label' => __( 'Show readmore link in sub tiles', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Show readmore link', 'newsplus' )
				),
				//'value' => 'true', // remove this if you do not need a default content
				'description' => __( 'Check to show a readmore link.', 'newsplus' )
			)
		);

		/**
		 * Schema params
		 */

		$schema_params = array(
			array(
				'name' => 'enable_schema',
				'label' => __( 'Enable schema support', 'newsplus' ),
				'type' => 'checkbox',
				'options' => array(
					'true' => __( 'Enable schema support in generated content', 'newsplus' )
				),
				'description' => __( 'Check to enable schema markup in generated content.', 'newsplus' )
			),
			array(
				'name' => 'publisher_logo',
				'label' => __( 'Add publisher logo', 'newsplus' ),
				'type' => 'attach_image_url',
				'description' => __( 'Add your site logo for publisher image schema.', 'newsplus' ),
				'value' => get_template_directory_uri() . '/images/logo.png',
				'relation' => array(
					'parent'    => 'enable_schema',
					'show_when' => 'true'
				)
			)
		);

		// Add shortcode to King Composer
		kc_add_map(
			array(
				'insert_posts' => array(
					'name' => __( 'Posts Module', 'newsplus' ),
					'description' => __( 'Display posts in various styles', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-pencil',
					'title' => 'Posts module',
					'is_container' => false,
					'system_only' => false,
					'nested' => false,

					'params' => array(

						// Query tab
						'Query' => 	$wp_query_params,

						// Display tab
						'Display' => array(
							array(
								'name' => 'template',
								'label' => 'Display Style',
								'type' => 'dropdown',
								'options' => array(
									'grid' => __( 'Grid', 'newsplus' ),
									'list-big' => __( 'List (Big thumbnails)', 'newsplus' ),
									'list-small' => __( 'List (Small thumbnails)', 'newsplus' ),
									'card' => __( 'Card', 'newsplus' ),
									'gallery' => __( 'Gallery', 'newsplus' ),
									'overlay' => __( 'Overlay', 'newsplus' )
								),
								'value' => 'grid',
								'description' => __( 'Select a display style for posts', 'newsplus' )
							),

							array(
								'name'     => 'col',
								'label'    => 'Number of columns',
								'type'     => 'dropdown',
								'options' => array(
									'1' => '1',
									'2' => '2',
									'3' => '3',
									'4' => '4'
								),
								'value' => 1,
								'description' => __( 'Select number of columns for Grid or Gallery template.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => array( 'grid', 'gallery', 'card', 'overlay' )
								)
							),

							array(
								'name' => 'overlay_style',
								'label' => 'Overlay Style',
								'type' => 'dropdown',
								'options' => array(
									'default' => __( 'White Cutout', 'newsplus' ),
									'dark-bg' => __( 'Dark Scrim', 'newsplus' )
								),
								'value' => 'default',
								'description' => __( 'Select a display style for overlay content', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => 'overlay'
								)
							),
							array(
								'name'     => 'split',
								'label'    => 'List layout split ratio',
								'type'     => 'dropdown',
								'options' => array(
									'20-80' => '20-80',
									'25-75' => '25-75',
									'33-67' => '33-67',
									'40-60' => '40-60',
									'50-50' => '50-50',
								),
								'value' => '33-67',
								'description' => __( 'Select a layout split ratio (%) for thumbnail + content in list style module.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => array( 'list-big', 'list-small' )
								)
							),

							array(
								'name' => 'enable_masonry',
								'label' => __( 'Enable masonry layout', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable masonry layout', 'newsplus' )
								),
								'description' => __( 'Check to enable masonry layout for grid and gallery template style.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => array( 'grid', 'gallery', 'card', 'overlay' )
								)
							),

							array(
								'name' => 'lightbox',
								'label' => __( 'Enable lightbox', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable lightbox on gallery', 'newsplus' )
								),
								'description' => __( 'Check to enable lightbox on gallery template type.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => 'gallery'
								)
							),

							array(
								'name' => 'show_title',
								'label' => __( 'Show titles', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Show titles on hover', 'newsplus' )
								),
								'description' => __( 'Check to enable titles on hover in gallery thumbnail', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => 'gallery'
								)
							),

							array(
								'label' => __( 'Image width', 'newsplus' ),
								'name' => 'imgwidth',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image width (in px, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'label' => __( 'Image height', 'newsplus' ),
								'name' => 'imgheight',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image height (in px, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'label' => __( 'Image quality', 'newsplus' ),
								'name' => 'imgquality',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '80',
								'description' => __( 'Provide image quality (in %, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'name' => 'imgcrop',
								'label' => __( 'Image Crop', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable hard crop mode', 'newsplus' )
								),
								'description' => __( 'Check to enable hard cropping of thumbnail image.', 'newsplus' )
							),

							array(
								'name' => 'hide_image',
								'label' => 'Hide image',
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Hide image thumbnail', 'newsplus' )
								),
								//'value' => 'true', // remove this if you do not need a default content
								'description' => __( 'Check to hide image thumbnail', 'newsplus' )
							),

							array(
								'name' => 'hide_video',
								'label' => 'Hide video',
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Hide video embed', 'newsplus' )
								),
								//'value' => 'true', // remove this if you do not need a default content
								'description' => __( 'Check to hide video embed in post module. If featured image is available, it will be shown, else video screenshot will be automatically detected and loaded.', 'newsplus' )
							),
							array(
								'name' => 'video_custom_field',
								'label' => 'Video custom field',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide custom field name for videos. Use this option if your posts already have custom field for videos, with value as valid video URL.', 'newsplus' )
							),
							
							array(
								'name' => 'xclass',
								'label' => 'CSS class for post container',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Any extra class which you wish to add to the post module container.', 'newsplus' )
							),

							array(
								'name' => 'ptclass',
								'label' => 'CSS class for post thumbnail',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Any extra class which you wish to add to the post thumbnail container.', 'newsplus' )
							),
							
							array(
								'name' => 'show_excerpt',
								'label' => __( 'Show post excerpt (for list-small template)', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Show post excerpt', 'newsplus' )
								),
								'description' => __( 'Check to show post excerpt in post module.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => 'list-small'
								)
							),


							array(
								'name' => 'show_cats',
								'label' => __( 'Show category links (for list-small template)', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Show category links', 'newsplus' )
								),
								'description' => __( 'Check to show category links in post module.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => 'list-small'
								)
							),


							array(
								'name' => 'show_reviews',
								'label' => __( 'Show review stars (for list-small template)', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Show review stars', 'newsplus' )
								),
								'description' => __( 'Check to show review stars in post module.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'template',
									'show_when' => 'list-small'
								)
							)

						),

						// Content tab
						'Content' => $content_params,

						// Meta tab
						'Meta' => $meta_params,

						// Schema tab
						'Schema' => $schema_params,
						
						// Styling tab
						__( 'Styling', 'newsplus' ) => array(
							array(
								'name' => 'inline_css',
								'label' => __( 'Custom CSS', 'newsplus' ),
								'type' => 'css',
								'options' => array(
									array(
										'screens' => "any",
										
										__( 'General', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'des' => __( 'This font family will apply to the post module container and it\'s child items.', 'newsplus' ) )
										),											
										
										__( 'Heading', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Heading color', 'newsplus' ), 'selector' => '.entry-title,.entry-title > a' ),
											array( 'property' => 'color', 'label' => __( 'Heading hover color', 'newsplus' ), 'selector' => '.entry-title > a:hover' ),
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => '.entry-title', 'des' => __( 'This font family will apply to the main post heading.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => '.entry-title' ),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => '.entry-title' )
										),											
										
										__( 'Excerpt', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => '.post-excerpt', 'des' => __( 'This font family will apply to the post excerpt.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => '.post-excerpt'),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'color', 'label' => __( 'Color', 'newsplus' ), 'selector' => '.post-excerpt')																								
										),																								
								 
										__( 'Meta', 'newsplus' ) => array(
											array( 'property' => 'background-color', 'label' => __( 'Category links background', 'newsplus' ), 'selector' => '.post-categories > li > a, .dark-bg .post-categories > li > a'),
											array( 'property' => 'color', 'label' => __( 'Category links color', 'newsplus' ), 'selector' => '.post-categories > li > a, .dark-bg .post-categories > li > a'),
											array( 'property' => 'background-color', 'label' => __( 'Category links hover background', 'newsplus' ), 'selector' => '.post-categories > li > a:hover, .dark-bg .post-categories > li > a:hover, .post-categories > li > a.cat-toggle.active-link'),
											array( 'property' => 'color', 'label' => __( 'Category links hover color', 'newsplus' ), 'selector' => '.post-categories > li > a:hover,.dark-bg .post-categories > li > a:hover, .post-categories > li > a.cat-toggle.active-link'),													
											array( 'property' => 'color', 'label' => __( 'Meta text/link color', 'newsplus' ), 'selector' => '.meta-row, .meta-row a'),
											array( 'property' => 'color', 'label' => __( 'Meta links hover color', 'newsplus' ), 'selector' => '.meta-row a:hover'),
										),
										
										__( 'Review', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Star background color', 'newsplus' ), 'selector' => '.review-star' ),
											array( 'property' => 'color', 'label' => __( 'Star foreground color', 'newsplus' ), 'selector' => '.review-result', 'des' => __( 'Click on important attribute flag for priority over default CSS.', 'newsplus' ) )
										),
										
										__( 'Custom', 'newsplus' ) => array(
											array( 'property' => 'custom', 'label' => __( 'Custom CSS', 'newsplus' ), 'des' => __( 'Applies to the main post module container.', 'newsplus' ) )
										)												
									)									
								)
							)									
						),
						
						__( 'Social', 'newsplus' ) => array(
							array(
								'type' => 'checkbox',
								'label' =>  __( 'Social sharing', 'newsplus' ),
								'name' => 'sharing',
								'description' =>  __( 'Check to enable social sharing for post modules', 'newsplus' ),
								'options' => array( 'true' => __( 'Enable social sharing for post modules', 'newsplus' ) )
							),
	
							array(
								'type' => 'multiple',
								'label' => __( 'Social sharing buttons', 'newsplus' ),
								'name' => 'share_btns',
								'options' => array(
									'twitter' => __( 'Twitter', 'newsplus' ),
									'facebook' => __( 'Facebook', 'newsplus' ),
									'whatsapp' => __( 'Whatsapp', 'newsplus' ),
									'googleplus' => __( 'Google Plus', 'newsplus' ),
									'linkedin' => __( 'LinkedIn', 'newsplus' ),
									'pinterest' => __( 'Pinterest', 'newsplus' ),
									'vkontakte' => __( 'VK Ontakte', 'newsplus' ),
									'reddit' => __( 'Reddit', 'newsplus' ),
									'email' => __( 'E Mail', 'newsplus' )
								),
								'description' => __( 'Select social share buttons. Use Ctrl + Select or Command + select for mutiple selection.', 'newsplus' ),
								'value' => '',
								'relation' => array(
									'parent' => 'sharing',
									'show_when' => 'true'
								)
							)								
						)
					)
				), // insert_posts

				'posts_slider' => array(

					'name' => __( 'Posts Slider', 'newsplus' ),
					'description' => __( 'Display a slider of posts', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-tv',
					'title' => 'Posts slider',
					'is_container' => false,
					'system_only' => false,
					'nested' => false,

					'params' => array(
						// Query tab
						'Query' => 	$wp_query_params,

						// Display tab
						'Display' => array(

							array(
								'label' => __( 'Sub style for items', 'newsplus' ),
								'name' => 'substyle',
								'type' => 'dropdown',
								'options' => array(
									'grid' => __( 'Grid', 'newsplus' ),
									'card' => __( 'Card', 'newsplus' ),
									'overlay' => __( 'Overlay', 'newsplus' )
								),
								'value' => 'grid',
								'description' => __( 'Choose a substyle for slider items', 'newsplus' )
							),
							
							array(
								'name' => 'overlay_style',
								'label' => 'Overlay Style',
								'type' => 'dropdown',
								'options' => array(
									'default' => __( 'White Cutout', 'newsplus' ),
									'dark-bg' => __( 'Dark Scrim', 'newsplus' )
								),
								'value' => 'default',
								'description' => __( 'Select a display style for overlay content', 'newsplus' ),
								'relation' => array(
									'parent'    => 'substyle',
									'show_when' => 'overlay'
								)
							),

							array(
								'label' => __( 'Number of items', 'newsplus' ),
								'name' => 'items',
								'type' => 'number_slider',
								'options' => array(
									'min' => 1,
									'max' => 5,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '1',
								'description' => __( 'Provide number of items per viewport.', 'newsplus' )
							),

							array(
								'label' => __( 'Slides margin', 'newsplus' ),
								'name' => 'margin',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => 24,
								'description' => __( 'Provide margin between slides (in px, without unit).', 'newsplus' )
							),

							array(
								'label' => __( 'Slides margin (mobile)', 'newsplus' ),
								'name' => 'margin_mobile',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => 16,
								'description' => __( 'Provide margin between slides for mobile (in px, without unit).', 'newsplus' )
							),

							array(
								'name' => 'smoothheight',
								'label' => __( 'Auto height', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable auto height for slides', 'newsplus' )
								),
								'description' => __( 'Check to enable auto height for slides', 'newsplus' )
							),

							array(
								'name' => 'controlnav',
								'label' => __( 'Dots navigation', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Show dots navigation', 'newsplus' )
								),
								'value' => 'true',
								'description' => __( 'Check to enable dots navigation', 'newsplus' )
							),

							array(
								'name' => 'directionnav',
								'label' => __( 'Prev/Next buttons', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Show prev/next buttons', 'newsplus' )
								),
								'value' => 'true',
								'description' => __( 'Check to enable previous next buttons', 'newsplus' )
							),

							array(
								'name' => 'animationloop',
								'label' => __( 'Loop animation', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Loop animation infinitely', 'newsplus' )
								),
								'description' => __( 'Check to loop animation infinitely', 'newsplus' )
							),

							array(
								'name' => 'slideshow',
								'label' => __( 'Autoplay on start', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Autoplay slider on start', 'newsplus' )
								),
								'value' => 'true',
								'description' => __( 'Check to enable autoplay on start', 'newsplus' )
							),

							array(
								'label' => __( 'Animation speed', 'newsplus' ),
								'name' => 'speed',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 9999,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '400',
								'description' => __( 'Provide animation speed for slides (in miliseconds)', 'newsplus' )
							),

							array(
								'label' => __( 'Animation timeout', 'newsplus' ),
								'name' => 'timeout',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 9999,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '4000',
								'description' => __( 'Provide animation timeout for slides (in miliseconds)', 'newsplus' )
							),

							array(
								'name' => 'effect',
								'label' => 'Slider effect',
								'type' => 'dropdown',
								'options' => array(
									'slide' => __( 'Slide', 'newsplus' ),
									'fade' => __( 'Fade', 'newsplus' )
								),
								'value' => 'slide',
								'description' => __( 'Select an effect for slide animation', 'newsplus' )
							),

							array(
								'label' => __( 'Image width', 'newsplus' ),
								'name' => 'imgwidth',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image width (in px, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'label' => __( 'Image height', 'newsplus' ),
								'name' => 'imgheight',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image height (in px, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'label' => __( 'Image quality', 'newsplus' ),
								'name' => 'imgquality',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',

									'show_input' => true
								),
								'value' => '80',
								'description' => __( 'Provide image quality (in %, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'name' => 'imgcrop',
								'label' => __( 'Image Crop', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable hard crop mode', 'newsplus' )
								),
								'description' => __( 'Check to enable hard cropping of thumbnail image.', 'newsplus' )
							),

							array(
								'name' => 'hide_image',
								'label' => 'Hide image',
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Hide image thumbnail', 'newsplus' )
								),
								//'value' => 'true', // remove this if you do not need a default content
								'description' => __( 'Check to hide image thumbnail', 'newsplus' )
							),

							array(
								'name' => 'hide_video',
								'label' => 'Hide video',
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Hide video embed', 'newsplus' )
								),
								'value' => 'true',
								'description' => __( 'Check to hide video embed in post module. If featured image is available, it will be shown, else video screenshot will be automatically detected and loaded.', 'newsplus' )
							),
							array(
								'name' => 'video_custom_field',
								'label' => 'Video custom field',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide custom field name for videos. Use this option if your posts already have custom field for videos, with value as valid video URL.', 'newsplus' )
							),
							array(
								'name' => 'xclass',
								'label' => 'CSS class for post container',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Any extra class which you wish to add to the post module container.', 'newsplus' )
							),

							array(
								'name' => 'ptclass',
								'label' => 'CSS class for post thumbnail',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Any extra class which you wish to add to the post thumbnail container.', 'newsplus' )
							)

						),

						// Content tab
						'Content' => $content_params,

						// Meta tab
						'Meta' => $meta_params,

						// Schema tab
						'Schema' => $schema_params,				
						
						// Styling tab
						__( 'Styling', 'newsplus' ) => array(
							array(
								'name' => 'inline_css',
								'label' => __( 'Custom CSS', 'newsplus' ),
								'type' => 'css',
								'options' => array(
									array(
										'screens' => "any",
										
										__( 'General', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'des' => __( 'This font family will apply to the post module container and it\'s child items.', 'newsplus' ) )
										),											
										
										__( 'Heading', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Heading color', 'newsplus' ), 'selector' => '.entry-title,.entry-title > a' ),
											array( 'property' => 'color', 'label' => __( 'Heading hover color', 'newsplus' ), 'selector' => '.entry-title > a:hover' ),
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => '.entry-title', 'des' => __( 'This font family will apply to the main post heading.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => '.entry-title' ),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => '.entry-title' )
										),											
										
										__( 'Excerpt', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => '.post-excerpt', 'des' => __( 'This font family will apply to the post excerpt.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => '.post-excerpt'),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'color', 'label' => __( 'Color', 'newsplus' ), 'selector' => '.post-excerpt')																								
										),																								
								 
										__( 'Meta', 'newsplus' ) => array(
											array( 'property' => 'background-color', 'label' => __( 'Category links background', 'newsplus' ), 'selector' => '.post-categories > li > a, .dark-bg .post-categories > li > a'),
											array( 'property' => 'color', 'label' => __( 'Category links color', 'newsplus' ), 'selector' => '.post-categories > li > a, .dark-bg .post-categories > li > a'),
											array( 'property' => 'background-color', 'label' => __( 'Category links hover background', 'newsplus' ), 'selector' => '.post-categories > li > a:hover, .dark-bg .post-categories > li > a:hover, .post-categories > li > a.cat-toggle.active-link'),
											array( 'property' => 'color', 'label' => __( 'Category links hover color', 'newsplus' ), 'selector' => '.post-categories > li > a:hover,.dark-bg .post-categories > li > a:hover, .post-categories > li > a.cat-toggle.active-link'),													
											array( 'property' => 'color', 'label' => __( 'Meta text/link color', 'newsplus' ), 'selector' => '.meta-row, .meta-row a'),
											array( 'property' => 'color', 'label' => __( 'Meta links hover color', 'newsplus' ), 'selector' => '.meta-row a:hover'),
										),
										
										__( 'Review', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Star background color', 'newsplus' ), 'selector' => '.review-star' ),
											array( 'property' => 'color', 'label' => __( 'Star foreground color', 'newsplus' ), 'selector' => '.review-result', 'des' => __( 'Click on important attribute flag for priority over default CSS.', 'newsplus' ) )
										),
										
										__( 'Custom', 'newsplus' ) => array(
											array( 'property' => 'custom', 'label' => __( 'Custom CSS', 'newsplus' ), 'des' => __( 'Applies to the main post module container.', 'newsplus' ) )
										)												
									),
								)
							)									
						),
						
						__( 'Social', 'newsplus' ) => array(
							array(
								'type' => 'checkbox',
								'label' =>  __( 'Social sharing', 'newsplus' ),
								'name' => 'sharing',
								'description' =>  __( 'Check to enable social sharing for post modules', 'newsplus' ),
								'options' => array( 'true' => __( 'Enable social sharing for post modules', 'newsplus' ) )
							),
	
							array(
								'type' => 'multiple',
								'label' => __( 'Social sharing buttons', 'newsplus' ),
								'name' => 'share_btns',
								'options' => array(
									'twitter' => __( 'Twitter', 'newsplus' ),
									'facebook' => __( 'Facebook', 'newsplus' ),
									'whatsapp' => __( 'Whatsapp', 'newsplus' ),
									'googleplus' => __( 'Google Plus', 'newsplus' ),
									'linkedin' => __( 'LinkedIn', 'newsplus' ),
									'pinterest' => __( 'Pinterest', 'newsplus' ),
									'vkontakte' => __( 'VK Ontakte', 'newsplus' ),
									'reddit' => __( 'Reddit', 'newsplus' ),
									'email' => __( 'E Mail', 'newsplus' )
								),
								'description' => __( 'Select social share buttons. Use Ctrl + Select or Command + select for mutiple selection.', 'newsplus' ),
								'value' => '',
								'relation' => array(
									'parent' => 'sharing',
									'show_when' => 'true'
								)
							)								
						)
					)
				), // posts_slider

				'newsplus_grid_list' => array(

					'name' => __( 'Posts Tile Grid', 'newsplus' ),
					'description' => __( 'Display post grids as tiles', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-th-large',
					'title' => 'Posts tile grid',
					'is_container' => false,
					'system_only' => false,
					'nested' => false,

					'params' => array(
						// Query tab
						'Query' => 	$wp_query_params,

						// Display tab
						'Display' => array(

							array(
								'label' => __( 'Display style', 'newsplus' ),
								'name' => 'display_style',
								'type' => 'dropdown',
								'options' => array(
									's1' => __( '1 main image + 2x2 Sub', 'newsplus' ),
									's2' => __( '1 main image + 2x3 Sub', 'newsplus' ),
									's3' => __( '4 images per row', 'newsplus' ),
									's6' => __( '3 images per row', 'newsplus' ),
									's4' => __( '2 images per row', 'newsplus' ),
									's5' => __( '1 image per row', 'newsplus' ),
								),
								'value' => 's1',
								'description' => __( 'Select a display style for tile grid', 'newsplus' )
							),

							array(
								'label' => __( 'Viewport width', 'newsplus' ),
								'name' => 'viewport_width',
								'type' => 'number_slider',
								'options' => array(
									'min' => 300,
									'max' => 3000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '1192',
								'description' => __( 'Provide a viewport width for tiles (in px, without unit)', 'newsplus' )
							),

							array(
								'label' => __( 'Images aspect ratio', 'newsplus' ),
								'name' => 'aspect_ratio',
								'type' => 'text',
								'value' => 0.75,
								'description' => __( 'Provide an aspect ratio for image dimension. E.g. 4:3 ratio will be calculated as 3/4 = .75', 'newsplus' )
							),

							array(
								'label' => __( 'Image quality', 'newsplus' ),
								'name' => 'imgquality',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',

									'show_input' => true
								),
								'value' => '80',
								'description' => __( 'Provide image quality (in %, without unit) for the thumbnail image.', 'newsplus' )
							),

							array(
								'name' => 'xclass',
								'label' => 'Extra CSS class',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Any extra class which you wish to add to the post module container.', 'newsplus' )
							)
						),

						// Content tab
						'Content' => array_merge( $content_params, $content_params_sub ),

						// Meta tab
						'Meta' => $meta_params,

						// Meta sub tab
						'Sub Meta' => $meta_params_sub,

						// Schema tab
						'Schema' => $schema_params,
						
						// Styling tab
						__( 'Styling', 'newsplus' ) => array(
							array(
								'name' => 'inline_css',
								'label' => __( 'Custom CSS', 'newsplus' ),
								'type' => 'css',
								'options' => array(
									array(
										'screens' => "any",
										
										__( 'General', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'des' => __( 'This font family will apply to the post module container and it\'s child items.', 'newsplus' ) )
										),											
										
										__( 'Heading', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Heading color', 'newsplus' ), 'selector' => '.entry-title > a' ),
											array( 'property' => 'color', 'label' => __( 'Heading hover color', 'newsplus' ), 'selector' => '.entry-title > a:hover' ),
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => '.entry-title', 'des' => __( 'This font family will apply to the main post heading.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => '.entry-title' ),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => '.entry-title' ),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => '.entry-title' )
										),
										
										__( 'Subheading', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Heading color', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title > a', 'des' => __( 'Settings in this section applies to the headings of sub tiles in Tile template style', 'newsplus' ) ),
											array( 'property' => 'color', 'label' => __( 'Heading hover color', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title > a:hover' ),
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title', 'des' => __( 'This font family will apply to the smaller headings in sub tiles of tile tempate.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' ),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => 'li.grid-1x1 .entry-title' )
										),												
										
										__( 'Excerpt', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'selector' => '.post-excerpt', 'des' => __( 'This font family will apply to the post excerpt.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ), 'selector' => '.post-excerpt' ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ), 'selector' => '.post-excerpt'),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ), 'selector' => '.post-excerpt'),
											array( 'property' => 'color', 'label' => __( 'Color', 'newsplus' ), 'selector' => '.post-excerpt')																								
										),																								
								 
										__( 'Meta', 'newsplus' ) => array(
											array( 'property' => 'background-color', 'label' => __( 'Category links background', 'newsplus' ), 'selector' => '.post-categories > li > a, .dark-bg .post-categories > li > a'),
											array( 'property' => 'color', 'label' => __( 'Category links color', 'newsplus' ), 'selector' => '.post-categories > li > a, .dark-bg .post-categories > li > a'),
											array( 'property' => 'background-color', 'label' => __( 'Category links hover background', 'newsplus' ), 'selector' => '.post-categories > li > a:hover, .dark-bg .post-categories > li > a:hover, .post-categories > li > a.cat-toggle.active-link'),
											array( 'property' => 'color', 'label' => __( 'Category links hover color', 'newsplus' ), 'selector' => '.post-categories > li > a:hover,.dark-bg .post-categories > li > a:hover, .post-categories > li > a.cat-toggle.active-link'),													
											array( 'property' => 'color', 'label' => __( 'Meta text/link color', 'newsplus' ), 'selector' => '.meta-row, .meta-row a'),
											array( 'property' => 'color', 'label' => __( 'Meta links hover color', 'newsplus' ), 'selector' => '.meta-row a:hover'),
										),
										
										__( 'Review', 'newsplus' ) => array(
											array( 'property' => 'color', 'label' => __( 'Star background color', 'newsplus' ), 'selector' => '.review-star' ),
											array( 'property' => 'color', 'label' => __( 'Star foreground color', 'newsplus' ), 'selector' => '.review-result', 'des' => __( 'Click on important attribute flag for priority over default CSS.', 'newsplus' ) )
										),
										
										__( 'Custom', 'newsplus' ) => array(
											array( 'property' => 'custom', 'label' => __( 'Custom CSS', 'newsplus' ), 'des' => __( 'Applies to the main post module container.', 'newsplus' ) )
										)												
									),
								)
							)									
						),
						
						__( 'Social', 'newsplus' ) => array(
							array(
								'type' => 'checkbox',
								'label' =>  __( 'Social sharing', 'newsplus' ),
								'name' => 'sharing',
								'description' =>  __( 'Check to enable social sharing for post modules', 'newsplus' ),
								'options' => array( 'true' => __( 'Enable social sharing for post modules', 'newsplus' ) )
							),
	
							array(
								'type' => 'multiple',
								'label' => __( 'Social sharing buttons', 'newsplus' ),
								'name' => 'share_btns',
								'options' => array(
									'twitter' => __( 'Twitter', 'newsplus' ),
									'facebook' => __( 'Facebook', 'newsplus' ),
									'whatsapp' => __( 'Whatsapp', 'newsplus' ),
									'googleplus' => __( 'Google Plus', 'newsplus' ),
									'linkedin' => __( 'LinkedIn', 'newsplus' ),
									'pinterest' => __( 'Pinterest', 'newsplus' ),
									'vkontakte' => __( 'VK Ontakte', 'newsplus' ),
									'reddit' => __( 'Reddit', 'newsplus' ),
									'email' => __( 'E Mail', 'newsplus' )
								),
								'description' => __( 'Select social share buttons. Use Ctrl + Select or Command + select for mutiple selection.', 'newsplus' ),
								'value' => '',
								'relation' => array(
									'parent' => 'sharing',
									'show_when' => 'true'
								)
							)								
						)
					)
				), // newsplus_grid_list

				'newsplus_news_ticker' => array(

					'name' => __( 'Posts News Ticker', 'newsplus' ),
					'description' => __( 'Display posts as news ticker', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-ellipsis-h',
					'title' => 'Posts News Ticker',
					'is_container' => false,
					'system_only' => false,
					'nested' => false,

					'params' => array(
						// Query tab
						'Query' => 	$wp_query_params,

						// Display tab
						'Display' => array(

							array(
								'label' => __( 'Title length', 'newsplus' ),
								'name' => 'title_length',
								'type' => 'number_slider',
								'options' => array(
									'min' => 1,
									'max' => 999,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '10',
								'description' => __( 'Provide new title length in words. E.g. 10', 'newsplus' )
							),

							array(
								'label' => __( 'Animation duration', 'newsplus' ),
								'name' => 'duration',
								'type' => 'number_slider',
								'options' => array(
									'min' => 1,
									'max' => 100000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '15000',
								'description' => __( 'Provide ticker animation duration in miliseconds', 'newsplus' )
							),

							array(
								'label' => __( 'Ticker label', 'newsplus' ),
								'name' => 'ticker_label',
								'type' => 'text',
								'value' =>  __( 'Breaking news', 'newsplus' ),
								'description' => __( 'Provide a label for news ticker. Leave blank for no label', 'newsplus' )
							),

							array(
								'name' => 'xclass',
								'label' => 'Extra CSS class',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Any extra class which you wish to add to the ticker contaner.', 'newsplus' )
							)
						),
						
						__( 'Styling', 'newsplus' ) => array(
							array(
								'name' => 'inline_css',
								'label' => __( 'Custom CSS', 'newsplus' ),
								'type' => 'css',
								'options' => array(
									array(
										'screens' => "any",
										
										__( 'General', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'des' => __( 'This font family will apply to the entire post ticker.', 'newsplus' ) )
										),											
										
										__( 'Colors', 'newsplus' ) => array(
											array( 'property' => 'background-color', 'label' => __( 'Label background color', 'newsplus' ), 'selector' => '.ticker-label' ),
											array( 'property' => 'color', 'label' => __( 'Label color', 'newsplus' ), 'selector' => '.ticker-label' ),
											array( 'property' => 'background-color', 'label' => __( 'Ticker background', 'newsplus' ), 'selector' => '.np-news-ticker' ),
											array( 'property' => 'color', 'label' => __( 'Posts title color', 'newsplus' ), 'selector' => '.np-news-ticker a' ),
											array( 'property' => 'color', 'label' => __( 'Posts title hover color', 'newsplus' ), 'selector' => '.np-news-ticker a:hover' )
										),
										
										__( 'Custom', 'newsplus' ) => array(
											array( 'property' => 'custom', 'label' => __( 'Custom CSS', 'newsplus' ), 'des' => __( 'Applies to the main ticker container.', 'newsplus' ) )
										)												
									),
								)
							)									
						)
					)
				), // newsplus_news_ticker

 				'recipe_method' => array(
					'name' => __( 'Recipe Method', 'newsplus' ),
					'description' => __( 'Create recipe method step', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-settings',
					'title' => 'Recipe Method',
					'system_only'	=> true,
					'is_container' => true,
					'except_child'	=> 'newsplus_recipe',
					'params' => array(
						// General tab
						'General' => array(

							array(
								'name' => 'method_title',
								'label' => __( 'Recipe Method title', 'newsplus' ),
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide optional method title if you want to group instructions under a heading.', 'newsplus' )
							)
						)
					)
				),

 				'newsplus_recipe' => array(
					'name' => __( 'Recipe Generator', 'newsplus' ),
					'description' => __( 'Generate recipe content with schema support', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-cutlery',
					'title' => 'Recipe Generator',
					'is_container' => true,
					'system_only' => false,
					//'nested' => false,
					'views'        => array(
						'type'     => 'views_sections',
						'sections' => 'recipe_method',
						'display'  => 'vertical'
					),
					'params' => array(

						// Display tab
						'General' => array(

							array(
								'label' => __( 'Recipe Name', 'newsplus' ),
								'name' => 'name_src',
								'type' => 'dropdown',
								'options' => array(
									'post_title'   => __( 'Use Post Title', 'newsplus' ),
									'custom' => __( 'Custom Name', 'newsplus' )
								),
								'value' => 'post_title',
								'description' => __( 'Choose a name source for Recipe', 'newsplus' )
							),

							array(
								'label' => __( 'Custom Recipe Name', 'newsplus' ),
								'name' => 'name_txt',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide a name for the recipe', 'newsplus' ),
								'relation' => array(
									'parent'    => 'name_src',
									'show_when' => 'custom'
								)
							),

							array(
								'label' => __( 'Recipe image', 'newsplus' ),
								'type' => 'dropdown',
								'name' => 'img_src',
								'options' => array(
										'featured' => __( 'Use featured image', 'newsplus' ),
										'media_lib' => __( 'Select from media library', 'newsplus' ),
										'ext' => __( 'Use external url', 'newsplus' )
								),
								'value' => '',
								'description' => __( 'Select image source', 'newsplus' )
							),

							array(
								'name' => 'img_lib',
								'label' => __( 'Add recipe image', 'newsplus' ),
								'type' => 'attach_image_url',
								'description' => __( 'Add a recipe image', 'newsplus' ),
								'relation' => array(
									'parent'    => 'img_src',
									'show_when' => 'media_lib'
								)
							),

							array(
								'name' => 'img_ext',
								'label' => __( 'Add external image URL', 'newsplus' ),
								'type' => 'text',
								'description' => __( 'Add external image URL', 'newsplus' ),
								'relation' => array(
									'parent'    => 'img_src',
									'show_when' => 'ext'
								)
							),							
							
							array(
								'label' => __( 'Image alt text', 'newsplus' ),
								'name' => 'img_alt',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide an alternative text for image', 'newsplus' )
							),
							
							array(
								'label' => __( 'Image caption', 'newsplus' ),
								'name' => 'img_caption',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide a caption for image', 'newsplus' )
							),

							array(
								'label' => __( 'Image width', 'newsplus' ),
								'name' => 'imgwidth',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image width (in px, without unit) for the recipe image.', 'newsplus' ),
									'relation' => array(
									'parent'    => 'img_src',
									'hide_when' => 'ext'
								)
							),


							array(
								'label' => __( 'Image height', 'newsplus' ),
								'name' => 'imgheight',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image height (in px, without unit) for the recipe image.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'img_src',
									'hide_when' => 'ext'
								)
							),

							array(
								'label' => __( 'Image quality', 'newsplus' ),
								'name' => 'imgquality',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '80',
								'description' => __( 'Provide image quality (in %, without unit) for the thumbnail image.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'img_src',
									'hide_when' => 'ext'
								)
							),

							array(
								'name' => 'imgcrop',
								'label' => __( 'Image Crop', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable hard crop mode', 'newsplus' )
								),
								'description' => __( 'Check to enable hard cropping of thumbnail image.', 'newsplus' ),
								'relation' => array(
									'parent'    => 'img_src',
									'hide_when' => 'ext'
								)
							),

							array(
								'label' => __( 'Image align', 'newsplus' ),
								'type' => 'dropdown',
								'name' => 'img_align',
								'options' => array(
										'none' => __( 'None', 'newsplus' ),
										'left' => __( 'Left', 'newsplus' ),
										'right' => __( 'Right', 'newsplus' ),
										'center' => __( 'Center', 'newsplus' )

								),
								'value' => '',
								'description' => __( 'Select image alignment. Image will be aligned with respect to the summary text.', 'newsplus' )
							),

							array(
								'label' => __( 'Recipe Summary', 'newsplus' ),
								'name' => 'summary',
								'type' => 'textarea',
								'value' => base64_encode( 'This is a recipe summary text. It can be a small excerpt about what you are going to cook.' ),
								'description' => __( 'Provide a short summary or description of your recipe', 'newsplus' )
							),

							array(
								'label' => __( 'Method Section Heading', 'newsplus' ),
								'name' => 'method_heading',
								'type' => 'text',
								'value' => __( 'Method', 'newsplus' ),
								'description' => __( 'Provide a heading for method section', 'newsplus' )
							),

							array(
								'name'        => 'enable_numbering',
								'label'       => 'Enable numbering',
								'type'        => 'checkbox',
								'description' => __( 'Check to enable auto numbering on method steps', 'newsplus' ),
								'options'     => array( 'true' => __( 'Enable auto numbering on method steps', 'newsplus' ) ),
								'value'		  => 'true'
							),

							array(
								'label' => __( 'Other notes', 'newsplus' ),
								'name' => 'other_notes',
								'type' => 'editor',
								'value' => base64_encode( 'This is an extra note from author. This can be any tip, suggestion or fact related to the recipe.' ),
								'description' => __( 'Provide extra notes to be shown at the end of recipe', 'newsplus' )
							),

							array(
								'name'        => 'json_ld',
								'label'       => __( 'Enable JSON LD microdata', 'newsplus' ),
								'type'        => 'checkbox',
								'description' => __( 'Enabling this option will add json ld schema data in recipe container.', 'newsplus' ),
								'options'     => array( 'true' => __( 'Check to enable json-ld microdata', 'newsplus' ) ),
								'value'		  => 'true'
							),

						),

						// Meta tab
						'Recipe Meta' => array(

							array(
								'label' => __( 'Recipe Cuisine', 'newsplus' ),
								'type' => 'multiple',
								'name' => 'recipe_cuisine',
								'options' => array(
										'American' => __( 'American', 'newsplus' ),
										'Chinese' => __( 'Chinese', 'newsplus' ),
										'French' => __( 'French', 'newsplus' ),
										'Indian' => __( 'Indian', 'newsplus' ),
										'Italian' => __( 'Italian', 'newsplus' ),
										'Japanese' => __( 'Japanese', 'newsplus' ),
										'Mediterranean' => __( 'Mediterranean', 'newsplus' ),
										'Mexican' => __( 'Mexican', 'newsplus' )
								),
								'value' => '',
								'description' => __( 'Select recipe cuisine from above list or use custom field below', 'newsplus' )
							),

							array(
								'label' => __( 'Other recipe cuisine', 'newsplus' ),
								'name' => 'recipe_cuisine_other',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide comma separated cuisines if not in above list. E.g. Rajasthani, Gujarati', 'newsplus' )
							),

							array(
								'label' => __( 'Recipe Category', 'newsplus' ),
								'type' => 'multiple',
								'name' => 'recipe_category',
								'options' => array(
										'Appetizer' => __( 'Appetizer', 'newsplus' ),
										'Breakfast' => __( 'Breakfast', 'newsplus' ),
										'Dessert' => __( 'Dessert', 'newsplus' ),
										'Drinks' => __( 'Drinks', 'newsplus' ),
										'Main Course' => __( 'Main Course', 'newsplus' ),
										'Salad' => __( 'Salad', 'newsplus' ),
										'Snack' => __( 'Snack', 'newsplus' ),
										'Soup' => __( 'Soup', 'newsplus' )
								),
								'value' => '',
								'description' => __( 'Select recipe categories from above list or use custom field below', 'newsplus' )
							),

							array(
								'label' => __( 'Other recipe category', 'newsplus' ),
								'name' => 'recipe_category_other',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide comma separated categories if not in above list. E.g. Lunch, Starter', 'newsplus' )
							),

							array(
								'label' => __( 'Cooking Method', 'newsplus' ),
								'name' => 'cooking_method',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide a cooking method. E.g. Roasting, Steaming', 'newsplus' )
							),

							array(
								'label' => __( 'Preparation Time (Minutes)', 'newsplus' ),
								'name' => 'prep_time',
								'type' => 'number_slider',
								'options' => array(
									'min' => 1,
									'max' => 1440,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '5',
								'description' => __( 'Provide preparation time (in minutes). E.g. 10', 'newsplus' )

							),

							array(
								'label' => __( 'Cooking Time (Minutes)', 'newsplus' ),
								'name' => 'cook_time',
								'type' => 'number_slider',
								'options' => array(
									'min' => 1,
									'max' => 1440,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '5',
								'description' => __( 'Provide cooking time (in minutes). E.g. 30', 'newsplus' )

							),

							array(
								'label' => __( 'Recipe Author', 'newsplus' ),
								'type' => 'dropdown',
								'name' => 'author_src',
								'options' => array(
										'post_author' => __( 'Use Post Author', 'newsplus' ),
										'custom' => __( 'Custom Author Name', 'newsplus' )
								),
								'value' => '',
								'description' => __( 'Select author name source for recipe', 'newsplus' )
							),

							array(
								'label' => __( 'Custom Author name', 'newsplus' ),
								'name' => 'author_name',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide name of author', 'newsplus' ),
								'relation' => array(
									'parent'    => 'author_src',
									'show_when' => 'custom'
								)
							),

							array(
								'label' => __( 'Author profile URL', 'newsplus' ),
								'name' => 'author_url',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The profile URL of recipe Author. Leave blank to use WordPress user URL.', 'newsplus' )
							)

						),

						// Ingredients tab
						'Ingredients' => array(

							array(
								'label' => __( 'Ingredients Heading', 'newsplus' ),
								'name' => 'ing_heading',
								'type' => 'text',
								'value' => __( 'Ingredients', 'newsplus' ),
								'description' => __( 'Provide a heading for ingredients section', 'newsplus' )
							),

							array(
								'type'			=> 'group',
								'label'			=> __( 'Ingredients', 'newsplus' ),
								'name'			=> 'ingredients',
								'description'	=> __( 'Add list items for each ingredient group', 'newsplus' ),
								'options'		=> array(
													'add_text' => __( 'Add new ingredient group', 'newsplus' )
												),

								'value' 		=> base64_encode( json_encode( array(
													'1' => array(
														'title' => 'For bread',
														'list' => base64_encode( '500 gm wheat floor' )
													),

													'2' => array(
														'title' => 'For cream and toppings',
														'list' => base64_encode( '2 cup milk cream' )
													)
												)
											)
										),

								'params' => array(

										array(
											'type' => 'text',
											'label' => __( 'Ingredients group title', 'newsplus' ),
											'name' => 'title',
											'description' => __( 'Provide a group title for ingredient list. E.g. Ingredients for curry', 'newsplus' ),
											'admin_label' => true,
										),

										array(
											'type' => 'textarea',
											'label' => __( 'Ingredients (one per line)', 'newsplus' ),
											'name' => 'list',
											'description' => __( 'Provide list of ingredients, one per line. E.g. 1 cup milk, skimmed', 'newsplus' ),
											'admin_label' => true
										)
								)
							)
						),

						// Meta tab
						'Nutrition' => array(

							array(
								'label' => __( 'Recipe Yield', 'newsplus' ),
								'name' => 'recipe_yield',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide a recipe yield. E.g. 1 Pizza, or 1 Bowl Rice', 'newsplus' )
							),

							array(
								'label' => __( 'Serving Size', 'newsplus' ),
								'name' => 'serving_size',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide a serving size per container. E.g. 1 Piece(20g), or 100g', 'newsplus' )
							),

							array(
								'label' => __( 'Calorie per serving (cal)', 'newsplus' ),
								'name' => 'calories',
								'type' => 'text',
								'value' => '',
								'description' => __( 'Provide approximate calories (without unit) per serving. E.g. 240', 'newsplus' )
							),

							array(
								'label' => __( 'Suitable for Diet', 'newsplus' ),
								'type' => 'multiple',
								'name' => 'suitable_for_diet',
								'options' => array(
										'Diabetic' => __( 'Diabetic', 'newsplus' ),
										'Gluten Free' => __( 'Gluten Free', 'newsplus' ),
										'Halal' => __( 'Halal', 'newsplus' ),
										'Hindu' => __( 'Hindu', 'newsplus' ),
										'Kosher' => __( 'Kosher', 'newsplus' ),
										'Low Calorie' => __( 'Low Calorie', 'newsplus' ),
										'Low Fat' => __( 'Low Fat', 'newsplus' ),
										'Low Lactose' => __( 'Low Lactose', 'newsplus' ),
										'Low Salt' => __( 'Low Salt', 'newsplus' ),
										'Vegan' => __( 'Vegan', 'newsplus' ),
										'Vegetarian' => __( 'Vegetarian', 'newsplus' ),
								),
								'value' => '',
								'description' => __( 'Select diet for which this recipe is suitable. Remove selection if not applicable.', 'newsplus' )
							),

							array(
								'label' => __( 'Total Fat', 'newsplus' ),
								'name' => 'total_fat',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Total Fat (g), without unit. Standard daily value is 78g', 'newsplus' )
							),

							array(
								'label' => __( 'Saturated Fat', 'newsplus' ),
								'name' => 'saturated_fat',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Saturated Fat (g), without unit. Standard daily value is 20g', 'newsplus' )
							),

							array(
								'label' => __( 'Trans Fat', 'newsplus' ),
								'name' => 'trans_fat',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Trans Fat (g), without unit. ', 'newsplus' )
							),

							array(
								'label' => __( 'Polyunsaturated Fat', 'newsplus' ),
								'name' => 'polyunsat_fat',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Polyunsaturated Fat (g), without unit. ', 'newsplus' )
							),

							array(
								'label' => __( 'Monounsaturated Fat', 'newsplus' ),
								'name' => 'monounsat_fat',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Monounsaturated Fat (g), without unit. ', 'newsplus' )
							),

							array(
								'label' => __( 'Cholesterol', 'newsplus' ),
								'name' => 'cholesterol',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Cholesterol (mg), without unit. Standard daily value is 300mg', 'newsplus' )
							),

							array(
								'label' => __( 'Sodium', 'newsplus' ),
								'name' => 'sodium',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Sodium (mg), without unit. Standard daily value is 2300mg', 'newsplus' )
							),

							array(
								'label' => __( 'Total Carbohydrate', 'newsplus' ),
								'name' => 'carbohydrate',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Total Carbohydrate (g), without unit. Standard daily value is 275g', 'newsplus' )
							),

							array(
								'label' => __( 'Dietary Fiber', 'newsplus' ),
								'name' => 'fiber',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Dietary Fiber (g), without unit. Standard daily value is 28g', 'newsplus' )
							),

							array(
								'label' => __( 'Total Sugars', 'newsplus' ),
								'name' => 'sugar',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Total Sugars (g), without unit. ', 'newsplus' )
							),

							array(
								'label' => __( 'Added Sugars', 'newsplus' ),
								'name' => 'added_sugar',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Added Sugars (g), without unit. Standard daily value is 50g', 'newsplus' )
							),

							array(
								'label' => __( 'Sugar Alcohal', 'newsplus' ),
								'name' => 'sugar_alcohal',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Sugar Alcohal (g), without unit. ', 'newsplus' )
							),

							array(
								'label' => __( 'Protein', 'newsplus' ),
								'name' => 'protein',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Protein (g), without unit. Standard daily value is 50g', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin D (Cholecalciferol)', 'newsplus' ),
								'name' => 'vitamin_d',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin D (Cholecalciferol) (mcg), without unit. Standard daily value is 10mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Calcium', 'newsplus' ),
								'name' => 'calcium',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Calcium (mg), without unit. Standard daily value is 1300mg', 'newsplus' )
							),

							array(
								'label' => __( 'Iron', 'newsplus' ),
								'name' => 'iron',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Iron (mg), without unit. Standard daily value is 18mg', 'newsplus' )
							),

							array(
								'label' => __( 'Potassium', 'newsplus' ),
								'name' => 'potassium',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Potassium (mg), without unit. Standard daily value is 4700mg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin A', 'newsplus' ),
								'name' => 'vitamin_a',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin A (mcg), without unit. Standard daily value is 900mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin C (Ascorbic Acid)', 'newsplus' ),
								'name' => 'vitamin_c',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin C (Ascorbic Acid) (mg), without unit. Standard daily value is 90mg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin E (Tocopherol)', 'newsplus' ),
								'name' => 'vitamin_e',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin E (Tocopherol) (mg), without unit. Standard daily value is 15mg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin K', 'newsplus' ),
								'name' => 'vitamin_k',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin K (mcg), without unit. Standard daily value is 120mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin B1 (Thiamin)', 'newsplus' ),
								'name' => 'vitamin_b1',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin B1 (Thiamin) (mg), without unit. Standard daily value is 1.2mg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin B2 (Riboflavin)', 'newsplus' ),
								'name' => 'vitamin_b2',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin B2 (Riboflavin) (mg), without unit. Standard daily value is 1.3mg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin B3 (Niacin)', 'newsplus' ),
								'name' => 'vitamin_b3',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin B3 (Niacin) (mg), without unit. Standard daily value is 18mg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin B6 (Pyridoxine)', 'newsplus' ),
								'name' => 'vitamin_b6',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin B6 (Pyridoxine) (mg), without unit. Standard daily value is 2mg', 'newsplus' )
							),

							array(
								'label' => __( 'Folate', 'newsplus' ),
								'name' => 'folate',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Folate (mcg), without unit. Standard daily value is 400mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin B12 (Cobalamine)', 'newsplus' ),
								'name' => 'vitamin_b12',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin B12 (Cobalamine) (mcg), without unit. Standard daily value is 2.4mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Biotin', 'newsplus' ),
								'name' => 'biotin',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Biotin (mcg), without unit. Standard daily value is 30mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Vitamin B5 (Pantothenic acid)', 'newsplus' ),
								'name' => 'vitamin_b5',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Vitamin B5 (Pantothenic acid) (mg), without unit. Standard daily value is 5mg', 'newsplus' )
							),

							array(
								'label' => __( 'Phosphorus', 'newsplus' ),
								'name' => 'phosphorus',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Phosphorus (mg), without unit. Standard daily value is 1250mg', 'newsplus' )
							),

							array(
								'label' => __( 'Iodine', 'newsplus' ),
								'name' => 'iodine',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Iodine (mcg), without unit. Standard daily value is 150mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Magnesium', 'newsplus' ),
								'name' => 'magnesium',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Magnesium (mg), without unit. Standard daily value is 420mg', 'newsplus' )
							),

							array(
								'label' => __( 'Zinc', 'newsplus' ),
								'name' => 'zinc',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Zinc (mg), without unit. Standard daily value is 11mg', 'newsplus' )
							),

							array(
								'label' => __( 'Selenium', 'newsplus' ),
								'name' => 'selenium',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Selenium (mcg), without unit. Standard daily value is 55mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Copper', 'newsplus' ),
								'name' => 'copper',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Copper (mcg), without unit. Standard daily value is 900mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Manganese', 'newsplus' ),
								'name' => 'manganese',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Manganese (mg), without unit. Standard daily value is 2.3mg', 'newsplus' )
							),

							array(
								'label' => __( 'Chromium', 'newsplus' ),
								'name' => 'chromium',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Chromium (mcg), without unit. Standard daily value is 35mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Molybdenum', 'newsplus' ),
								'name' => 'molybdenum',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Molybdenum (mcg), without unit. Standard daily value is 45mcg', 'newsplus' )
							),

							array(
								'label' => __( 'Chloride', 'newsplus' ),
								'name' => 'chloride',
								'type' => 'text',
								'value' => '',
								'description' => __( 'The amount of Chloride (mg), without unit. Standard daily value is 2300mg', 'newsplus' )
							)
						),

						// Hide things tab
						'Hide' => array(

							array(
								'name'        => 'hide_name',
								'label'       => 'Recipe name',
								'type'        => 'checkbox',
								'description' => __( 'If checked, recipe name will not be displayed. It will be used for schema only', 'newsplus' ),
								'options'     => array( 'true' => __( 'Hide recipe name', 'newsplus' ) )
							),

							array(
								'name'        => 'hide_date',
								'label'       => 'Publish date',
								'type'        => 'checkbox',
								'description' => __( 'If checked, publsh date will not be displayed. It will be used for schema only', 'newsplus' ),
								'options'     => array( 'true' => __( 'Hide recipe publish date', 'newsplus' ) )
							),

							array(
								'name'        => 'hide_author',
								'label'       => 'Recipe Author',
								'type'        => 'checkbox',
								'description' => __( 'If checked, author name will not be displayed. It will be used for schema only', 'newsplus' ),
								'options'     => array( 'true' => __( 'Hide author name', 'newsplus' ) )
							),

							array(
								'name'        => 'hide_img',
								'label'       => 'Recipe image',
								'type'        => 'checkbox',
								'description' => __( 'If checked, recipe image will not be displayed. It will be used for schema only', 'newsplus' ),
								'options'     => array( 'true' => __( 'Hide recipe image', 'newsplus' ) )
							),

							array(
								'name'        => 'hide_summary',
								'label'       => 'Recipe summary',
								'type'        => 'checkbox',
								'description' => __( 'If checked, recipe summary will not be displayed. It will be used for schema only.', 'newsplus' ),
								'options'     => array( 'true' => __( 'Hide recipe summary', 'newsplus' ) )
							),							

							array(
								'name'        => 'hide_nutrition',
								'label'       => 'Nutrition Facts',
								'type'        => 'checkbox',
								'description' => __( 'If checked, nutrition facts section will not be displayed.', 'newsplus' ),
								'options'     => array( 'true' => __( 'Hide nutrition facts', 'newsplus' ) )
							)

						)

					) // Nutrition tab

				), // newsplus_recipe
				
				'newsplus_image' => array(
					'name' => __( 'Image', 'newsplus' ),
					'description' => __( 'Display an image with custom sizes', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'fa-image',
					'title' => __( 'Image', 'newsplus' ),

					'params' => array(
					
						// Display tab
						'General' => array(
							array(
								'name' => 'src',
								'label' => __( 'Browse Image', 'newsplus' ),
								'type' => 'attach_image_url',
								'description' => __( 'Upload or select image from media library', 'newsplus' )
							),
							
							array(
								'type' => 'text',
								'label' => __( 'Alternative text', 'newsplus' ),
								'name' => 'alt',
								'value' => '',
								'description' => __( 'Provide an alternative text for image', 'newsplus' )
							),
							
							array(
								'type' => 'text',
								'label' => __( 'Image caption', 'newsplus' ),
								'name' => 'caption',
								'value' => '',
								'description' => __( 'Provide a caption for the image', 'newsplus' )
							),
							
							array(
								'type' => 'dropdown',
								'label' => __( 'Image align', 'newsplus' ),
								'name' => 'align',
								'options' => array(
										'none' => esc_attr__( 'None', 'newsplus' ),
										'left' => esc_attr__( 'Left', 'newsplus' ),
										'right' => esc_attr__( 'Right', 'newsplus' ),
										'center' => esc_attr__( 'Center', 'newsplus' )
								),
								'value' => 'none',
								'description' => __( 'Select image alignment', 'newsplus' )
							),
							
							array(
								'label' => __( 'Image width', 'newsplus' ),
								'name' => 'imgwidth',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image width (in px, without unit) for the image.', 'newsplus' )
							),

							array(
								'label' => __( 'Image height', 'newsplus' ),
								'name' => 'imgheight',
								'type' => 'number_slider',
								'options' => array(
									'min' => 10,
									'max' => 2000,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '',
								'description' => __( 'Provide image height (in px, without unit) for the image.', 'newsplus' )
							),

							array(
								'label' => __( 'Image quality', 'newsplus' ),
								'name' => 'imgquality',
								'type' => 'number_slider',
								'options' => array(
									'min' => 0,
									'max' => 100,
									'unit' => '',
									'ratio' => '1',
									'show_input' => true
								),
								'value' => '80',
								'description' => __( 'Select image quality (in %, without unit) for the image.', 'newsplus' )
							),

							array(
								'name' => 'imgcrop',
								'label' => __( 'Image Crop', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Enable hard crop mode', 'newsplus' )
								),
								'description' => __( 'Check to enable hard cropping of image.', 'newsplus' )
							),
							
							array(
								'name' => 'imgcolor',
								'label' => __( 'Colorize image', 'newsplus' ),
								'type' => 'color_picker',
								'admin_label' => true,
								'description' => __( 'Choose a color for colorizing image', 'newsplus' ),
								'value' => ''
							),
							
							array(
								'name'        => 'imggrayscale',
								'label'       => __( 'Image grayscale', 'newsplus' ),
								'type'        => 'checkbox',
								'description' => __( 'Check to convert image into grayscale', 'newsplus' ),
								'options'     => array( 'true' => __( 'Convert image to grayscale', 'newsplus' ) )
							),
							
							array(
								'type' => 'dropdown',
								'label' => __( 'Image link to', 'newsplus' ),
								'name' => 'linkto',
								'options' => array(
										'none' => esc_attr__( 'None', 'newsplus' ),
										'media' => esc_attr__( 'Media file', 'newsplus' ),
										'custom' => esc_attr__( 'Custom link', 'newsplus' )
								),
								'value' => 'none',
								'description' => __( 'Select image alignment', 'newsplus' )
							),							
							
							array(
								'type' => 'text',
								'label' => __( 'Custom link URL', 'newsplus' ),
								'name' => 'custom_url',
								'value' => '',
								'description' => __( 'Provide a custom link URL for image', 'newsplus' ),
								'relation' => array(
									'parent'    => 'linkto',
									'show_when' => 'custom'
								)
							),							
							
							array(
								'type' => 'text',
								'label' => __( 'Link rel attribute', 'newsplus' ),
								'name' => 'linkrel',
								'value' => '',
								'description' => __( 'Provide a rel attribute for image link', 'newsplus' ),
								'relation' => array(
									'parent'    => 'linkto',
									'hide_when' => 'none'
								)
							),
						
							array(
								'type' => 'text',
								'label' => __( 'Image Title', 'newsplus' ),
								'name' => 'imgtitle',
								'value' => '',
								'description' => __( 'Provide a title for image link', 'newsplus' ),
								'relation' => array(
									'parent'    => 'linkto',
									'hide_when' => 'none'
								)
							),
							
							array(
								'name'        => 'target',
								'label'       => __( 'Open in new tab', 'newsplus' ),
								'type'        => 'checkbox',
								'description' => __( 'Check to open link in new tab', 'newsplus' ),
								'options'     => array( 'true' => __( 'Open link in new tab', 'newsplus' ) )
							),
							
							array(
								'type' => 'text',
								'label' => __( 'Extra CSS class', 'newsplus' ),
								'name' => 'xclass',
								'value' => '',
								'description' => __( 'Any extra class you wish to add to image container', 'newsplus' )
							)				
							
						) // General
					) // Params
				), // newsplus_image
				
				'newsplus_title' => array(
					'name' => __( 'NewsPlus Title', 'newsplus' ),
					'description' => __( 'Content title headings in style', 'newsplus' ),
					'category' => 'NewsPlus',
					'icon' => 'sl-pencil',
					'title' => __( 'NewsPlus Title', 'newsplus' ),
					'is_container' => false,
					'system_only' => false,
					'nested' => false,	
					'params' => array(
						__( 'General', 'newsplus' ) => array(
							array(
								'type' => 'text',
								'label' => __( 'Title Text', 'newsplus' ),
								'name' => 'text',
								'description' => __( 'Provide title text', 'newsplus' )
							),
	
							array(
								'type' => 'text',
								'label' => __( 'Link URL', 'newsplus' ),
								'name' => 'link',
								'description' => __( 'Provide a link URL for the text', 'newsplus' )
							),
	
							array(
								'type' => 'text',
								'label' => __( 'Link title attribute', 'newsplus' ),
								'name' => 'link_title',
								'description' => __( 'Provide a title attribute for the link', 'newsplus' )
							),
							
							array(
								'name' => 'target',
								'label' => __( 'Open link in new tab', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Open link in new tab', 'newsplus' )
								),
								'description' => __( 'Check to open link in new tab.', 'newsplus' )
							),							
	
							array(
								'type' => 'text',
								'label' => __( 'Secondary Text', 'newsplus' ),
								'name' => 'sub_text',
								'description' => __( 'Provide a secondary text for the main text', 'newsplus' )
							),
	
							array(
								'type' => 'text',
								'label' => __( 'Secondary Link URL', 'newsplus' ),
								'name' => 'sub_link',
								'description' => __( 'Provide a link URL for the text', 'newsplus' )
							),
	
							array(
								'type' => 'text',
								'label' => __( 'Sub Link title attribute', 'newsplus' ),
								'name' => 'sub_link_title',
								'description' => __( 'Provide a title attribute for the secondary link', 'newsplus' )
							),
							
							array(
								'name' => 'sub_target',
								'label' => __( 'Open secondary link in new tab', 'newsplus' ),
								'type' => 'checkbox',
								'options' => array(
									'true' => __( 'Open secondary link in new tab', 'newsplus' )
								),
								'description' => __( 'Check to open secondary link in new tab.', 'newsplus' )
							),								
						),
						
						__( 'Display', 'newsplus' ) => array(
							array(
								'type' => 'dropdown',
								'label' => __( 'Title Tag', 'newsplus' ),
								'name' => 'htag',
								'options' => array(
									'h1' => 'h1',
									'h2' => 'h2',
									'h3' => 'h3',
									'h4' => 'h4',
									'h5' => 'h5',
									'h6' => 'h6',
									'p' => 'p',
									'span' => 'span',
									'div' => 'div'
								),
								'value' => 'h2',
								'description' => __( 'Select a tag for title container', 'newsplus' )
							),
	
							array(
								'type' => 'dropdown',
								'label' => __( 'Font Size', 'newsplus' ),
								'name' => 'hsize',
								'options' => array(
										'12' => __( '12px', 'newsplus' ),
										'13' => __( '13px', 'newsplus' ),
										'14' => __( '14px', 'newsplus' ),
										'16' => __( '16px', 'newsplus' ),
										'18' => __( '18px', 'newsplus' ),
										'20' => __( '20px', 'newsplus' ),
										'24' => __( '24px', 'newsplus' )
									),
								'value' => '14',
								'description' => __( 'Select a font size for the title', 'newsplus' )
							),
	
							array(
								'type' => 'dropdown',
								'label' => __( 'Bottom Margin', 'newsplus' ),
								'name' => 'margin_btm',
								'options' => array(
									'theme_defined' => __( 'Theme Defined', 'newsplus' ),
									'0' => __( '0px', 'newsplus' ),
									'2' => __( '2px', 'newsplus' ),
									'4' => __( '4px', 'newsplus' ),
									'6' => __( '6px', 'newsplus' ),
									'8' => __( '8px', 'newsplus' ),
									'10' => __( '10px', 'newsplus' ),
									'12' => __( '12px', 'newsplus' ),
									'16' => __( '16px', 'newsplus' ),
									'20' => __( '20px', 'newsplus' ),
									'24' => __( '24px', 'newsplus' ),
									'30' => __( '30px', 'newsplus' ),
									'32' => __( '32px', 'newsplus' ),
									'36' => __( '36px', 'newsplus' ),
									'40' => __( '40px', 'newsplus' ),
									'50' => __( '50px', 'newsplus' ),
									'60' => __( '60px', 'newsplus' ),
									'70' => __( '70px', 'newsplus' )
								),
								'value' => 'theme_defined',
								'description' => __( 'Select a bottom margin for the label', 'newsplus' )
							),
	
							array(
								'type' => 'dropdown',
								'label' => __( 'Title Style', 'newsplus' ),
								'name' => 'style',
								'options' => array(
									'plain' => __( 'Plain', 'newsplus' ),
									'flag' => __( 'Flag', 'newsplus' ),
									'label' => __( 'Label', 'newsplus' ),
									'bordered' => __( 'Bordered', 'newsplus' ),
									'bar' => __( 'Bar', 'newsplus' )
								),
								'value' => 'plain',
								'description' => __( 'Select a display style for title', 'newsplus' )
							),
	
							array(
								'type' => 'dropdown',
								'label' => __( 'Title Background Color', 'newsplus' ),
								'name' => 'color',
								'options' => array(
									'default' => __( 'Default', 'newsplus' ),
									'aqua' => __( 'Aqua', 'newsplus' ),
									'blue' => __( 'Blue', 'newsplus' ),
									'brown' => __( 'Brown', 'newsplus' ),
									'cyan' => __( 'Cyan', 'newsplus' ),
									'dark-blue' => __( 'Dark Blue', 'newsplus' ),
									'deep-orange' => __( 'Deep Orange', 'newsplus' ),
									'green' => __( 'Green', 'newsplus' ),
									'grey' => __( 'Grey', 'newsplus' ),
									'indigo' => __( 'Indigo', 'newsplus' ),
									'orange' => __( 'Orange', 'newsplus' ),
									'pink' => __( 'Pink', 'newsplus' ),
									'red' => __( 'Red', 'newsplus' ),
									'teal' => __( 'Teal', 'newsplus' )
								),
								'value' => 'flag',
								'description' => __( 'Select a background color for title', 'newsplus' )
							),
							
							array(
								'name'        => 'round_corner',
								'label'       => __( 'Enable round corners', 'newsplus' ),
								'type'        => 'checkbox',
								'description' => __( 'Check to enable round corners on the label', 'newsplus' ),
								'options'     => array( 'true' => __( 'Enable round corners on the label', 'newsplus' ) )
							),
	
							array(
								'type' => 'text',
								'label' => __( 'Extra CSS class', 'newsplus' ),
								'name' => 'xclass',
								'description' => __( 'Any extra class you wish to add to the label container', 'newsplus' )
							)
						),
							
						__( 'Styling', 'newsplus' ) => array(
							array(
								'name' => 'inline_css',
								'label' => __( 'Styling', 'newsplus' ),
								'type' => 'css',
								'options' => array(
									array(
										'screens' => "any",
										
										__( 'General', 'newsplus' ) => array(
											array( 'property' => 'font-family', 'label' => __( 'Font family', 'newsplus' ), 'des' => __( 'This font family will apply to the title.', 'newsplus' ) ),
											array( 'property' => 'font-size', 'label' => __( 'Font size', 'newsplus' ) ),
											array( 'property' => 'line-height', 'label' => __( 'Line height', 'newsplus' ) ),
											array( 'property' => 'font-weight', 'label' => __( 'Font weight', 'newsplus' ) ),
											array( 'property' => 'font-style', 'label' => __( 'Font style', 'newsplus' ) ),
											array( 'property' => 'text-align', 'label' => __( 'Text align', 'newsplus' ) ),
											array( 'property' => 'text-shadow', 'label' => __( 'Text shadow', 'newsplus' ) ),
											array( 'property' => 'text-transform', 'label' => __( 'Text transform', 'newsplus' ) ),
											array( 'property' => 'text-decoration', 'label' => __( 'Text decoration', 'newsplus' ) ),
											array( 'property' => 'letter-spacing', 'label' => __( 'Letter spacing', 'newsplus' ) ),	
											array( 'property' => 'word-break', 'label' => __( 'Word break', 'newsplus' ) )												
										),											
									
										__( 'Custom', 'newsplus' ) => array(
											array( 'property' => 'custom', 'label' => __( 'Custom CSS', 'newsplus' ), 'des' => __( 'Applies to the main post module container.', 'newsplus' ) )
										)												
									)
								)
							)									
						)								
					)
				) // newsplus_title
			)
		);
	} // newsplus_kc_map_shortcodes
} // function_exists