<?php
/**
 * Twenty Seventeen functions and definitions
 *
 * @package NewsPlus
 * @since 1.0.0
 * @version 3.1.4
 */

/**
 * Sets up the content width value based on the theme's design and stylesheet.
 */
if ( ! isset( $content_width ) ) {
	$content_width = 660;
}

/**
 * Sets up theme defaults and registers the various WordPress features that
 * NewsPlus supports.
 */
function newsplus_setup() {

	// Makes theme available for translation.
	load_theme_textdomain( 'newsplus', get_template_directory() . '/languages' );

	// Add visual editor stylesheet support
	add_editor_style();

	// Adds RSS feed links to <head> for posts and comments.
	add_theme_support( 'automatic-feed-links' );

	// Add post formats
	add_theme_support( 'post-formats', array( 'gallery', 'video' ) );

	// Add support for html5 search form
	add_theme_support( 'html5', array( 'search-form' ) );

	// Add navigation menus
	register_nav_menus( array(
		'secondary'	=> __( 'Secondary Top Menu', 'newsplus' ),
		'primary'	=> __( 'Primary Menu', 'newsplus' )
	) );

	// Add support for custom background and set default color
	add_theme_support( 'custom-background', array(
		'default-color' => 'ffffff',
		'default-image' => ''
	) );

	// Add suport for post thumbnails
	add_theme_support( 'post-thumbnails' );

	// Declare theme as WooCommerce compatible
	add_theme_support( 'woocommerce' );
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );

}
add_action( 'after_setup_theme', 'newsplus_setup' );


/**
 * Include widgets and theme option files
 */
require_once( 'includes/post-options.php' );
require_once( 'includes/page-options.php' );
require_once( 'includes/theme-admin-options.php' );
require_once( 'includes/breadcrumbs.php' );
require_once( 'includes/theme-customizer.php' );
require_once( 'includes/class-tgm-plugin-activation.php' );
require_once( 'woocommerce/woocommerce-hooks.php' );
require_once( 'includes/page-builder-layouts.php' );


/**
 * Enqueue scripts and styles for front-end.
 */
function newsplus_scripts_styles() {
	global $post, $wp_styles;

	/*
	 * Add JavaScript for threaded comments when required
	 */
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) )
		wp_enqueue_script( 'comment-reply' );

	/*
	 * Add JavaScript plugins required by the theme
	 */
	wp_enqueue_script( 'jquery' );

	// For contact page template
	if( is_page_template( 'templates/page-contact.php' ) ) {
		wp_register_script( 'jq-validate', get_template_directory_uri() . '/js/jquery.validate.min.js', array(), '', true );
		wp_enqueue_script( 'contact-form', get_template_directory_uri() . '/js/contact-form.js', array( 'jq-validate' ), '', true );
	}
	
	// Masonry
	if ( get_option( 'pls_enable_masonry' ) &&
			(
				( 	( is_home() || is_archive() || is_search() )
					&& 'grid' == get_option( 'pls_archive_template' )
				)
				|| is_page_template( 'templates/archive-2col.php' )
				|| is_page_template( 'templates/archive-3col.php' )
				|| is_page_template( 'templates/archive-4col.php' )
				|| is_page_template( 'templates/blog-grid.php' )
				|| ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'insert_posts') )
			)
		) {
			wp_enqueue_script( 'jquery-masonry' );
	}

	// Load social sharing scripts in footer
	if( is_single() && 'true' == get_option( 'pls_ss_sharing' ) ) {
		$protocol = is_ssl() ? 'https' : 'http';
		if( 'true' == get_option( 'pls_ss_tw' ) )
			wp_enqueue_script( 'twitter_share_script', $protocol . '://platform.twitter.com/widgets.js', '', '', true );
		if( 'true' == get_option( 'pls_ss_gp' ) )
			wp_enqueue_script( 'google_plus_script', $protocol . '://apis.google.com/js/plusone.js', '', '', true );
		if( 'true' == get_option( 'pls_ss_pint' ) )
			wp_enqueue_script( 'pinterest_script', $protocol . '://assets.pinterest.com/js/pinit.js', '', '', true );
		if( 'true' == get_option( 'pls_ss_ln' ) )
			wp_enqueue_script( 'linkedin_script', $protocol . '://platform.linkedin.com/in.js', '', '', true );
		if( 'true' == get_option( 'pls_ss_vk' ) )
			wp_enqueue_script( 'vk_script', $protocol . '://vkontakte.ru/js/api/share.js?9' );
		if( 'true' == get_option( 'pls_ss_yandex' ) )
			wp_enqueue_script( 'yandex_script', $protocol . '://yandex.st/share/share.js', '', '', true );
	}

	// Custom JavaScript file used to initialize plugins
	wp_enqueue_script( 'custom', get_template_directory_uri() . '/js/custom.js', array(), '', true );

	// Localize responsivene menu check for custom.js file
	$trans_arr = array(
		'top_bar_sticky'	=> get_option( 'pls_top_bar_sticky', false ),
		'main_bar_sticky'	=> get_option( 'pls_main_bar_sticky', false ),
		'expand_menu_text'	=> __( 'Expand or collapse submenu', 'newsplus' ),
		'header_style'		=> get_option( 'pls_header_style', false ),
		'mobile_sticky'		=> get_option( 'pls_disable_sticky_on_mobile', false ),
		'collapse_lists' 	=> get_option( 'pls_collapse_lists', false )
	);
		
	if ( 'true' !== get_option( 'pls_disable_resp_css' ) ) {
		if ( 'true' !== get_option( 'pls_disable_resp_menu' ) ) {
			$trans_arr[ 'enable_responsive_menu' ] = 'true';
		}
	}
		
	wp_localize_script( 'custom', 'ss_custom', $trans_arr );	

	/*
	 * Google font CSS file
	 *
	 * The use of custom Google fonts can be configured inside Theme Options Panel.
	 */

	if ( 'true' !== get_option( 'pls_disable_custom_font' ) ) {
		$query_args = array(
			'family' => get_option( 'pls_font_family' ),
			'subset' => get_option( 'pls_font_subset' ),
		);
		wp_enqueue_style( 'newsplus-fonts', add_query_arg( $query_args, "https://fonts.googleapis.com/css" ), array(), null );
	}

	/*
	 * Load stylesheets required by the theme
	 */

	// Main stylesheet
	wp_enqueue_style( 'newsplus-style', get_stylesheet_uri() );

	// Internet Explorer specific stylesheet
	wp_enqueue_style( 'newsplus-ie', get_template_directory_uri() . '/css/ie.css', array(), '' );
	$wp_styles->add_data( 'newsplus-ie', 'conditional', 'lt IE 9' );

	// WooCommerce Custom stylesheet
	if ( class_exists( 'woocommerce' )  ) {
		wp_register_style( 'woocommerce-custom', get_template_directory_uri() . '/woocommerce/woocommerce-custom.css', array(), '' );
		wp_enqueue_style( 'woocommerce-custom' );
	}

	// RTL stylesheet
	if ( 'true' == get_option( 'pls_enable_rtl_css' ) || is_rtl() ) {
		wp_register_style( 'rtl', get_template_directory_uri() . '/rtl.css', array(), '' );
		wp_enqueue_style( 'rtl' );
	}

	// Responsive stylesheet
	if ( 'true' != get_option( 'pls_disable_resp_css' ) ) {
		wp_register_style( 'newsplus-responsive', get_template_directory_uri() . '/responsive.css', array(), '' );
		wp_enqueue_style( 'newsplus-responsive' );

		// Load RTL responsive only if RTL version enabled
		if ( 'true' == get_option( 'pls_enable_rtl_css' ) || is_rtl() ) {
			wp_register_style( 'newsplus-rtl-responsive', get_template_directory_uri() . '/rtl-responsive.css', array(), '' );
			wp_enqueue_style( 'newsplus-rtl-responsive' );
		}
	}
	
	// Color schemes
	if ( 'default' != get_option( 'pls_color_scheme', 'default' ) && 'customizer' != get_option( 'pls_color_scheme', 'default' ) ) {
		wp_register_style( 'newsplus-color-scheme', get_template_directory_uri() . '/css/schemes/' . get_option( 'pls_color_scheme', 'default' ) . '.css', array(), '' );
		wp_enqueue_style( 'newsplus-color-scheme' );
	}	

	// User stylesheet
	if ( 'true' != get_option( 'pls_disable_user_css' )  ) {
		wp_register_style( 'newsplus-user', get_template_directory_uri() . '/user.css', array(), '' );
		wp_enqueue_style( 'newsplus-user' );
	}

}
add_action( 'wp_enqueue_scripts', 'newsplus_scripts_styles', '20' );

/**
 * Add layout width and additional CSS in head node
 */
function newsplus_layout_width() {
	
	$width = floatval( get_option( 'pls_layout_width', 1160 ) );
	$font_size = floatval( get_option( 'pls_base_font_size', 13 ) );

	if ( ( $width >= 800 && $width <= 1600 ) || ( $font_size >= 11 && $font_size <= 18 && $font_size != 13 ) ) { ?>

		<style type="text/css">
		<?php if ( $font_size >= 11 && $font_size <= 18 && $font_size != 13 ) { ?>
		body {
			font-size: <?php echo $font_size . 'px'; ?>;
		}
		<?php }
		if ( $width >= 800 && $width <= 1600 ) { ?>
		#page {
			max-width: <?php echo $width . 'px'; ?>;
		}
		.wrap,
		.primary-nav,
		.is-boxed .top-nav,
		.is-boxed .header-slim.site-header {
			max-width: <?php echo floatval( $width - 48 ) . 'px'; ?>;
		}		
		@media only screen and (max-width: <?php echo floatval( $width + 48 ) . 'px'; ?>) {
			.wrap,
			.primary-nav,
			.is-boxed .top-nav,
			.is-boxed .header-slim.site-header,
			.is-stretched .top-nav .wrap {
				max-width: calc(100% - 48px);
			}			
			.is-boxed .sticky-nav,
			.is-boxed .header-slim.sticky-nav,
			.is-boxed #responsive-menu.sticky-nav {
				max-width: calc(97.5% - 48px);
			}		
		}
		<?php }
		if ( get_option( 'pls_user_css', '' ) ) {
			echo stripslashes( strip_tags( get_option( 'pls_user_css', '' ) ) );
		}
		?>
		</style>
	<?php }
}
add_action( 'wp_head', 'newsplus_layout_width' );

/**
 * Enqueue html5 script in head section for old browsers
 */
function html5_js() {
	$protocol = is_ssl() ? 'https' : 'http'; ?>
	<!--[if lt IE 9]>
	<script src="<?php echo $protocol; ?>://html5shim.googlecode.com/svn/trunk/html5.js"></script>
	<![endif]-->
<?php
}
add_action( 'wp_head', 'html5_js' );

/**
 * Allow custom head markup to be inserted by user from Theme Options
 */
function custom_head_code() {
	$head_code = get_option( 'pls_custom_head_code', '' );
	if ( '' != $head_code ) {
		echo stripslashes( $head_code );
	}
}
add_action( 'wp_head', 'custom_head_code');


/**
 * Register theme widget areas
 */
function newsplus_widget_areas() {
	
	register_sidebar( array(
		'name' 			=> __( 'Default Sidebar A', 'newsplus' ),
		'id' 			=> 'default-sidebar',
		'description' 	=> __( 'The main sidebar', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' 	=> "</aside>",
		'before_title' 	=> '<h3 class="sb-title">',
		'after_title' 	=> '</h3>',
		'handler'		=> 'sidebar'
	) );
	
	register_sidebar( array(
		'name' 			=> __( 'Default Sidebar B', 'newsplus' ),
		'id' 			=> 'sidebar-b',
		'description' 	=> __( 'Additional sidebar for two sidebar layouts', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' 	=> "</aside>",
		'before_title' 	=> '<h3 class="sb-title">',
		'after_title' 	=> '</h3>',
		'handler'		=> 'sidebar'
	) );
	
	register_sidebar( array(
		'name' 			=> __( 'Top widget area', 'newsplus' ),
		'id' 			=> 'top-widget-area',
		'description' 	=> __( 'Top widget area', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="twa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="twa-title">',
		'after_title' 	=> '</h3>'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Fixed widget area left', 'newsplus' ),
		'id' 			=> 'fixed-widget-bar-left',
		'description' 	=> __( 'Fixed widget area on left side', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="fwa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="fwa-title">',
		'after_title' 	=> '</h3>'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Fixed widget area right', 'newsplus' ),
		'id' 			=> 'fixed-widget-bar-right',
		'description' 	=> __( 'Fixed widget area on right side', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="fwa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="fwa-title">',
		'after_title' 	=> '</h3>'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Default Header Bar', 'newsplus' ),
		'id' 			=> 'default-headerbar',
		'description' 	=> __( 'Header Bar', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="hwa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="hwa-title">',
		'after_title' 	=> '</h3>',
		'handler'		=> 'headerbar'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Default Header Column 1', 'newsplus' ),
		'id' 			=> 'default-header-col-1',
		'description' 	=> __( 'Header column 1', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="hwa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="hwa-title">',
		'after_title' 	=> '</h3>'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Default Header Column 2', 'newsplus' ),
		'id' 			=> 'default-header-col-2',
		'description' 	=> __( 'Header column 2', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="hwa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="hwa-title">',
		'after_title' 	=> '</h3>'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Default Header Column 3', 'newsplus' ),
		'id' 			=> 'default-header-col-3',
		'description' 	=> __( 'Header column 3', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="hwa-wrap %2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="hwa-title">',
		'after_title' 	=> '</h3>'
	) );

	register_sidebar( array(
		'name' 			=> __( 'Widget area before content', 'newsplus' ),
		'id' 			=> 'widget-area-before-content',
		'description' 	=> __( 'Widget area before content', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="%2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="widget-title">',
		'after_title' 	=> '</h3>'
	) );	

	register_sidebar( array(
		'name' 			=> __( 'Widget area after content', 'newsplus' ),
		'id' 			=> 'widget-area-after-content',
		'description' 	=> __( 'Widget area after content', 'newsplus' ),
		'before_widget' => '<aside id="%1$s" class="%2$s">',
		'after_widget'	=> "</aside>",
		'before_title' 	=> '<h3 class="widget-title">',
		'after_title' 	=> '</h3>'
	) );

	$cols = get_option( 'pls_footer_cols', 4 );
	for( $i = 1; $i <= $cols; $i++ ) {
		register_sidebar( array(
			'name' 			=> sprintf( __( 'Default Secondary Column %s', 'newsplus' ), $i ),
			'id' 			=> 'secondary-column-' . $i,
			'description' 	=> __( 'Secondary Column', 'newsplus' ),
			'before_widget' => '<aside id="%1$s" class="widget %2$s">',
			'after_widget' 	=> "</aside>",
			'before_title' 	=> '<h3 class="sc-title">',
			'after_title' 	=> '</h3>',
		) );
	}

	// Register exclusive widget areas for each page
	$mypages = get_pages();
	foreach ( $mypages as $pp ) {
		$page_opts = get_post_meta( $pp->ID, 'page_options', true );
		$sidebar_a = isset( $page_opts['sidebar_a'] ) ? $page_opts['sidebar_a'] : '';
		$sidebar_h = isset( $page_opts['sidebar_h'] ) ? $page_opts['sidebar_h'] : '';

		if ( 'true' == $sidebar_h ){
			register_sidebar( array(
				'name' 			=> sprintf( __( '%1$s Header Bar', 'newsplus' ), $pp->post_title ),
				'id' 			=> $pp->ID . '-headerbar',
				'description' 	=> 'Header Bar',
				'before_widget' => '<aside id="%1$s" class="hwa-wrap %2$s">',
				'after_widget' 	=> "</aside>",
				'before_title' 	=> '<h3 class="hwa-title">',
				'after_title' 	=> '</h3>',
				'handler'		=> 'headerbar'
			) );
		};
		if ( 'true' == $sidebar_a ){
			register_sidebar( array(
				'name' 			=> sprintf( __( '%1$s Sidebar', 'newsplus' ), $pp->post_title ),
				'id' 			=> $pp->ID . '-sidebar',
				'description' 	=> 'Sidebar',
				'before_widget' => '<aside id="%1$s" class="widget %2$s">',
				'after_widget' 	=> "</aside>",
				'before_title' 	=> '<h3 class="sb-title">',
				'after_title' 	=> '</h3>',
				'handler'		=> 'sidebar'
			) );
		}
	} // foreach
}
add_action( 'widgets_init', 'newsplus_widget_areas' );


/**
 * Add body class to the theme depending upon options and templates
 */
function newsplus_body_class( $classes ) {

	if ( ( 'ac' == get_option( 'pls_sb_pos', 'ca' ) || is_page_template( 'templates/page-sb-left.php' ) ) && ! ( is_page_template( 'templates/page-sb-right.php' ) || is_page_template( 'templates/page-full.php' )  ) ) {
		$classes[] = 'layout-ac';
	}

	// Enable custom font class only if the font CSS is queued to load.
	if ( wp_style_is( 'newsplus-fonts', 'queue' ) )
		$classes[] = 'custom-font-enabled';

	if ( 'stretched' == get_option( 'pls_layout' ) ) {
		$classes[] = 'is-stretched';
	}
	else {
		$classes[] = 'is-boxed';
	}

	if ( 'true' == get_option( 'pls_enable_rtl_css' ) || is_rtl() ) {
		$classes[] = 'rtl rtl-enabled';
	}
	
	if ( 'true' == get_option( 'pls_social_sticky' ) ) {
		$classes[] = 'np-social-sticky';
	}
	
	$classes[] = 'split-' . esc_attr( get_option( 'pls_sb_ratio', '70-30' ) );
	
	// Sidebar layout classes
	if ( is_single() && 'post' == get_post_type() ) {
		global $post;
		$post_opts = get_post_meta( $post->ID, 'post_options', true );
		$sng_layout = ( isset( $post_opts['sng_layout'] ) && 'global' != $post_opts['sng_layout'] ) ? $post_opts['sng_layout'] : get_option( 'pls_sb_pos', 'ca' );
		$post_full_width 	= ( isset( $post_opts['post_full_width'] ) ) ? $post_opts['post_full_width'] : '';
		
		if ( 'no-sb' != get_option( 'pls_sb_pos', 'ca' ) ) {
			if ( ! $post_full_width ) {
				if ( ! ( $sng_layout == 'ac' || $sng_layout == 'ca' ) ) {
					$classes[] = 'two-sidebars';
				}
			}
		}
		
		$classes[] = 'layout-' . esc_attr( $sng_layout );		
	}
	
	else {
		if ( ! ( is_single() || is_post_type_archive( 'product' ) || is_tax( get_object_taxonomies( 'product' ) ) || is_page_template( 'templates/archive-3col.php' ) || is_page_template( 'templates/archive-4col.php' ) || is_page_template( 'templates/page-full.php' ) || is_page_template( 'templates/page-sb-left.php' ) || is_page_template( 'templates/page-sb-right.php' ) ) ) {
			$pls_sb_pos = get_option( 'pls_sb_pos', 'ca' );
			if ( ! ( $pls_sb_pos == 'ac' || $pls_sb_pos == 'ca' || $pls_sb_pos == 'no-sb' ) ) {
				$classes[] = 'two-sidebars';
			}	
			$classes[] = 'layout-' . esc_attr( get_option( 'pls_sb_pos', 'ca' ) );
		}
	}
	
	// Color scheme class
	$classes[] = 'scheme-' . get_option( 'pls_color_scheme', 'customizer' );

	return $classes;
}
add_filter( 'body_class', 'newsplus_body_class' );

/**
 * next/previous navigation for pages and archives
 */
if ( ! function_exists( 'newsplus_content_nav' ) ) :
function newsplus_content_nav( $html_id ) {
	global $wp_query;
	$html_id = esc_attr( $html_id );
	if ( function_exists( 'wp_pagenavi' ) ) wp_pagenavi();
	else {
	if ( $wp_query->max_num_pages > 1 ) : ?>
		<nav id="<?php echo $html_id; ?>" class="navigation">
			<div class="nav-previous alignleft"><?php next_posts_link( __( '<span class="meta-nav">&larr;</span> Older posts', 'newsplus' ) ); ?></div>
			<div class="nav-next alignright"><?php previous_posts_link( __( 'Newer posts <span class="meta-nav">&rarr;</span>', 'newsplus' ) ); ?></div>
		</nav><!-- #<?php echo $html_id; ?> .navigation -->
	<?php endif;
	}
}
endif;

/**
 * Social Sharing feature on single posts
 */
if ( ! function_exists( 'ss_sharing' ) ) :
	function ss_sharing() {
		
		$protocol = is_ssl() ? 'https' : 'http';
		$share_link = get_permalink();
		$share_title = get_the_title();
		$out = '';
		if ( 'true' == get_option( 'pls_ss_fb' ) ) {
			$out .= '<div class="fb-share-button" data-href="' . esc_url( $share_link ) . '" data-layout="button_count"></div>';
		}
		if ( 'true' == get_option( 'pls_ss_tw' ) ) {
			if ( get_option( 'pls_ss_tw_usrname', false ) ) {
				$out .= '<div class="ss-sharing-btn"><a href="' . $protocol . '://twitter.com/share" class="twitter-share-button" data-url="' . esc_url( $share_link ) . '"  data-text="' . esc_attr( $share_title ) . '" data-via="' . get_option( 'pls_ss_tw_usrname' ) . '">Tweet</a></div>';
			}
			else {
				$out .= '<div class="ss-sharing-btn"><a href="' . $protocol . '://twitter.com/share" class="twitter-share-button" data-url="' . esc_url( $share_link ) . '"  data-text="' . esc_attr( $share_title ) . '">Tweet</a></div>';
			}
		}
		if ( 'true' == get_option( 'pls_ss_gp' ) ) {
			$out .= '<div class="ss-sharing-btn"><div class="g-plusone" data-size="medium" data-href="' . esc_url( $share_link ) . '"></div></div>';
		}
		if ( 'true' == get_option( 'pls_ss_pint' ) ) {
			global $post;
			setup_postdata( $post );
			if ( has_post_thumbnail( $post->ID ) ) {
				$src = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), '', '' );
				$image = $src[0];
			}
			else
				$image = '';
			$description = 'Next%20stop%3A%20Pinterest';
			$share_link = get_permalink();
			$out .= '<div class="ss-sharing-btn"><a data-pin-config="beside" href="//pinterest.com/pin/create/button/?url=' . esc_url( $share_link ) . '&amp;media=' . $image . '&amp;description=' . $description . '" data-pin-do="buttonPin" ><img src="//assets.pinterest.com/images/pidgets/pin_it_button.png" alt="PinIt" /></a></div>';
			wp_reset_postdata();
		}
		if ( 'true' == get_option( 'pls_ss_ln' ) ) {
			$out .= '<div class="ss-sharing-btn"><script type="IN/Share" data-url="' . esc_url( $share_link ) . '" data-counter="right"></script></div>';
		}
		if ( 'true' == get_option( 'pls_ss_vk' ) ) {
			global $post;
			setup_postdata( $post );
			if ( has_post_thumbnail( $post->ID ) ) {
				$src = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), '', '' );
				$image = $src[0];
			}
			else
				$image = '';
			$description = strip_tags( get_the_excerpt() );
			$out .= '<div class="ss-sharing-btn">
						<script type="text/javascript">
						document.write(VK.Share.button({
						url: "' . esc_url( $share_link ) . '",
						title: "' . esc_attr( $share_title ) . '",
						description: "' . $description . '",
						image: "' . $image . '",
						noparse: true
						}));
						</script>
                    </div>';
		}
		if ( 'true' == get_option( 'pls_ss_yandex' ) ) {
			$out .= '<div class="ss-sharing-btn"><div class="yashare-auto-init" data-yashareL10n="ru" data-yashareQuickServices="yaru" data-yashareTheme="counter"></div></div>';
		}

		if ( 'true' == get_option( 'pls_ss_reddit' ) ) {
			$out .= "<a href=\"//www.reddit.com/submit\" onclick=\"window.location = '//www.reddit.com/submit?url=' + encodeURIComponent(window.location); return false\"> <img src=\"//www.redditstatic.com/spreddit7.gif\" alt=\"submit to reddit\" border=\"0\" /> </a>";
		}

		echo $out;
	}
endif;

/**
 * Load FaceBook script in footer
 */
function ss_fb_script() {
		
	if ( is_single() && ( 'true' == get_option( 'pls_ss_sharing' ) ) && ( 'true' == get_option( 'pls_ss_fb' ) ) ) { ?>
        <div id="fb-root"></div>
        <script>(function(d, s, id) {
        var js, fjs = d.getElementsByTagName(s)[0];
        if (d.getElementById(id)) return;
        js = d.createElement(s); js.id = id;
        js.src = "//connect.facebook.net/en_US/sdk.js#xfbml=1&version=v2.3";
        fjs.parentNode.insertBefore(js, fjs);
        }(document, 'script', 'facebook-jssdk'));</script>
<?php }
}
add_action( 'wp_footer', 'ss_fb_script' );

/**
 * Add FaceBook Open Graph Meta Tags on single post
 */
function add_facebook_open_graph_tags() {
	
	if ( ( is_single() && get_option( 'pls_ss_sharing', false ) && get_option( 'pls_ss_fb', false ) ) || ( is_single() && get_option( 'pls_show_social_btns', false ) && in_array( 'facebook', get_option( 'pls_social_btns' ) ) ) ) {
		global $post;
		setup_postdata( $post );
		if ( has_post_thumbnail( $post->ID ) ) {
			$src = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), '', '' );
			$image = $src[0];
		}
		else
			$image = '';
		?>
		<meta property="og:title" content="<?php echo strip_tags( get_the_title() ); ?>"/>
		<meta property="og:type" content="article"/>
		<meta property="og:image" content="<?php echo esc_url( $image ); ?>"/>
		<meta property="og:url" content="<?php echo esc_url( get_permalink() ); ?>"/>
		<meta property="og:description" content="<?php echo strip_tags( get_the_excerpt() ); ?>"/>
		<meta property="og:site_name" content="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>"/>
		<?php wp_reset_postdata();
	}
}
add_action( 'wp_head', 'add_facebook_open_graph_tags', 99 );

/**
 * Add FaceBook OG xmlns in html tag
 */
function add_og_xml_ns( $out ) {

	$protocol = is_ssl() ? 'https' : 'http';
	if ( ( is_single() && get_option( 'pls_ss_sharing', false ) && get_option( 'pls_ss_fb', false ) ) || ( is_single() && get_option( 'pls_show_social_btns', false ) && in_array( 'facebook', get_option( 'pls_social_btns' ) ) ) ) {
		return $out . ' xmlns:og="' . $protocol . '://ogp.me/ns#" xmlns:fb="' . $protocol . '://www.facebook.com/2008/fbml"';
	}
	else
		return $out;
}
add_filter( 'language_attributes', 'add_og_xml_ns' );

/**
 * Enable short codes inside Widgets
 */
add_filter( 'widget_text', 'shortcode_unautop');
add_filter( 'widget_text', 'do_shortcode');

/**
 * Allow HTML in category and term descriptions
 */
foreach ( array( 'pre_term_description' ) as $filter ) {
    remove_filter( $filter, 'wp_filter_kses' );
}
foreach ( array( 'term_description' ) as $filter ) {
    remove_filter( $filter, 'wp_kses_data' );
}

/**
 * Add span tag to post count of categories and archives widget
 */
function cats_widget_postcount_filter( $out ) {
	$out = str_replace( ' (', '<span class="count">(', $out );
	$out = str_replace( ')', ')</span>', $out );
	return $out;
}
add_filter( 'wp_list_categories', 'cats_widget_postcount_filter' );

function archives_widget_postcount_filter( $out ) {
	$out = str_replace( '&nbsp;(', '<span class="count">(', $out );
	$out = str_replace( ')', ')</span>', $out );
	return $out;
}
add_filter('get_archives_link', 'archives_widget_postcount_filter');

/**
 * Assign appropriate heading tag for blog name (SEO improvement)
 */
if ( ! function_exists( 'site_header_tag' ) ) :
	function site_header_tag( $tag_type ) {
		if ( is_front_page() ) {
			$open_tag = '<h1 class="site-title">';
			$close_tag = '</h1>';
		}
		elseif ( is_archive() || is_page_template( 'templates/archive-2col.php' ) || is_page_template( 'templates/archive-3col.php' ) || is_page_template( 'templates/archive-4col.php' ) || is_page_template( 'templates/blog-classic.php' ) || is_page_template( 'templates/blog-list.php' ) || is_page_template( 'templates/blog-grid.php' ) ) {
			$open_tag = '<h3 class="site-title">';
			$close_tag = '</h3>';
		}
		else {
			$open_tag = '<h4 class="site-title">';
			$close_tag = '</h4>';
		}
		if ( 'open' == $tag_type )
			echo $open_tag;
		if ( 'close' == $tag_type )
			echo $close_tag;
	}
endif;


/**
 * Generate site name and logo in header area
 */
if ( ! function_exists( 'newsplus_logo' ) ) :
	function newsplus_logo() {
				
		$schema = newsplus_schema( get_option( 'pls_schema' ) );
		
		if ( 'true' !== get_option( 'pls_logo_check' ) ) {
			
			site_header_tag( 'open' );
			printf( '<a%s href="%s" title="%s" rel="home">%s</a>',
				$schema['url'],
				esc_url( home_url( '/' ) ),
				esc_attr( get_bloginfo( 'name' ) ),
				'' != get_option( 'pls_custom_title' ) ? get_option( 'pls_custom_title' ) : get_bloginfo( 'name' )
			);
			site_header_tag( 'close' );
			
			if ( ! get_option( 'pls_hide_desc', false ) ) {
				echo '<span class="site-description">' . get_bloginfo( 'description' ) . '</span>';	
			}	
		}
		
		else {		
			site_header_tag( 'open' );
			printf( '<a%1$s href="%2$s" title="%3$s" rel="home"><img src="%4$s" alt="%3$s" /></a>',
				$schema['url'],
				esc_url( home_url( '/' ) ),
				esc_attr( get_bloginfo( 'name' ) ),				
				'' != get_option( 'pls_logo' ) ? esc_url( get_option( 'pls_logo' ) ) : get_template_directory_uri() . '/images/logo.png'
			);
			site_header_tag( 'close' );
		}
	}
endif;

/**
 * Funtion to shorten any text by characters
 */
if ( ! function_exists( 'newsplus_short' ) ) :
	function newsplus_short( $text, $limit ) {
		$chars_limit = intval( $limit );
		$chars_text = strlen( $text );
		if ( $chars_text > $chars_limit ) {
			$text = strip_tags( $text );
			$text = $text . " ";
			$text = substr( $text, 0, $chars_limit );
			$text = substr( $text, 0, strrpos( $text, ' ' ) );
			return $text . "&hellip;";
		}
		else {
			return $text;
		}
	}
endif;

/**
 * Get post thumbnail caption
 */

if ( ! function_exists( 'newsplus_post_thumbnail_caption' ) ) {
	function newsplus_post_thumbnail_caption() {
	  global $post;

	  $thumbnail_id    = get_post_thumbnail_id( $post->ID );
	  $thumbnail_image = get_posts( array( 'p' => $thumbnail_id, 'post_type' => 'attachment' ) );

	  if ( $thumbnail_image && isset( $thumbnail_image[0] ) && ! empty( $thumbnail_image[0]->post_excerpt ) ) {
		return '<p class="feat-caption">' . $thumbnail_image[0]->post_excerpt . '</p>';
	  }
	}
}

/**
 * Retrieve the archive title based on the queried object.
 *
 * @since 2.0.0
 * @return string Archive title.
 */
function newsplus_get_the_archive_title() {
	if ( is_category() ) {
		$title = single_cat_title( '', false );
	} elseif ( is_tag() ) {
		$title = single_tag_title( '', false );
	} elseif ( is_author() ) {
		$title = '<span class="vcard">' . get_the_author() . '</span>';
	} elseif ( is_year() ) {
		$title = get_the_date( _x( 'Y', 'yearly archives date format' ) );
	} elseif ( is_month() ) {
		$title = get_the_date( _x( 'F Y', 'monthly archives date format' ) );
	} elseif ( is_day() ) {
		$title = get_the_date( _x( 'F j, Y', 'daily archives date format' ) );
	} elseif ( is_tax( 'post_format' ) ) {
		if ( is_tax( 'post_format', 'post-format-aside' ) ) {
			$title = _x( 'Asides', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-gallery' ) ) {
			$title = _x( 'Galleries', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-image' ) ) {
			$title = _x( 'Images', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-video' ) ) {
			$title = _x( 'Videos', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-quote' ) ) {
			$title = _x( 'Quotes', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-link' ) ) {
			$title = _x( 'Links', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-status' ) ) {
			$title = _x( 'Statuses', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-audio' ) ) {
			$title = _x( 'Audio', 'post format archive title' );
		} elseif ( is_tax( 'post_format', 'post-format-chat' ) ) {
			$title = _x( 'Chats', 'post format archive title' );
		}
	} elseif ( is_post_type_archive() ) {
		$title = post_type_archive_title( '', false );
	} elseif ( is_tax() ) {
		$tax = get_taxonomy( get_queried_object()->taxonomy );
		/* translators: 1: Taxonomy singular name, 2: Current taxonomy term */
		$title = sprintf( __( '%1$s: %2$s' ), $tax->labels->singular_name, single_term_title( '', false ) );
	} else {
		$title = __( 'Archives' );
	}

	return $title;
}

add_filter( 'get_the_archive_title', 'newsplus_get_the_archive_title' );

/**
 * News Ticker
 */
 
if ( ! function_exists( 'newsplus_ticker_output' ) ) :
	function newsplus_ticker_output() {        
		return do_shortcode( sprintf( '[newsplus_news_ticker%s%s%s]',
			get_option( 'pls_ticker_cats' ) ? ' ' . 'cats="' . esc_attr( get_option( 'pls_ticker_cats' ) ) . '"' : '',
			get_option( 'pls_ticker_num' ) ? ' ' . 'num="' . esc_attr( get_option( 'pls_ticker_num' ) ) . '"' : '',
			get_option( 'pls_ticker_label' ) ? ' ' . 'ticker_label="' . esc_html( get_option( 'pls_ticker_label' ) ) . '"' : ''
		) );	
	}
endif;

if ( ! function_exists( 'newsplus_schema' ) ) :
	
	function newsplus_schema( $enable_schema, $imgwidth = 200, $imgheight = 200 ) {

		$sc_arr = array(
			'html'	=> '',
			'container' => '',
			'heading' => '',
			'heading_alt' => '',
			'text' => '',
			'img_cont' => '',
			'img' => '',
			'img_size' => '',
			'url' => ''	,
			'nav' => '',
			'bclist' => '',
			'bcelement' => '',
			'item' => '',
			'name' => '',
			'position' => ''
		);
		
		 // Is single post
		if ( is_single() || is_home() || is_archive() || is_category() ) {
			$html_type = "Blog";
		}
		// Is static front page
		else if (is_front_page()) {
			$html_type = "Website";
		}
		// Is a general page
		 else {
			$html_type = 'WebPage';
		}
		
		if ( 'true' == $enable_schema ) {
			$protocol = is_ssl() ? 'https' : 'http';
			$schema = $protocol . '://schema.org/';
			
			$sc_arr['html'] = 'itemscope="itemscope" itemtype="' . $schema . $html_type . '" ';
			$sc_arr['container'] = ' itemscope="" itemtype="' . $schema . 'BlogPosting" itemprop="blogPost"';
			$sc_arr['heading'] = ' itemprop="headline mainEntityOfPage"';
			$sc_arr['heading_alt'] = ' itemprop="headline"';
			$sc_arr['text'] = ' itemprop="text"';
			
			$sc_arr['img_cont'] = ' itemprop="image" itemscope="" itemtype="' . $schema . 'ImageObject"';
			$sc_arr['img'] = ' itemprop="url"';
			
			$sc_arr['img_size'] = sprintf( ' <meta itemprop="width" content="%s"><meta itemprop="height" content="%s">', $imgwidth, $imgheight );
			$sc_arr['url'] = ' itemprop="url"';
			$sc_arr['nav'] = ' itemscope="itemscope" itemtype="' . $schema . 'SiteNavigationElement"';
			
			$sc_arr['bclist'] = ' itemscope="itemscope" itemtype="' . $schema . 'BreadcrumbList"';
			$sc_arr['bcelement'] = ' itemscope="itemscope" itemtype="' . $schema . 'ListItem" itemprop="itemListElement"';
			$sc_arr['item'] = ' itemprop="item"';
			$sc_arr['name'] = ' itemprop="name"';
			$sc_arr['position'] = ' itemprop="position"';

		}
		
		return $sc_arr;
	
	}

endif;

// Post meta for post modules
if ( ! function_exists( 'newsplus_meta' ) ) :
	function newsplus_meta( $args = array() ) {
		global $post;
		$custom_link = get_post_meta( $post->ID, 'np_custom_link', true );
			$defaults = array(
				'template'	=> 'grid',
				'date_format' => get_option( 'date_format' ),
				'enable_schema' => false,
				'hide_cats' => false,
				'hide_reviews' => false,
				'show_cats' => false,
				'show_reviews' => false,
				'hide_date' => false,
				'hide_author' => false,
				'show_avatar' => false,
				'hide_views' => false,
				'hide_comments' => false,
				'readmore' => false,
				'ext_link' => false,
				'readmore_text' => esc_attr__( 'Read more', 'newsplus' ),
				'publisher_logo' => get_template_directory_uri() . '/images/logo.png',
				'sharing'	=> false,
				'share_btns' => ''
			);

			$args = wp_parse_args( $args, $defaults );

			extract( $args );

			$protocol = is_ssl() ? 'https' : 'http';
			$schema = $protocol . '://schema.org/';
			// Date format
			$date = get_the_time( get_option( 'date_format' ) );

			if ( ! empty( $date_format ) ) {
				if ( $date_format == 'human' ) {
					$date = sprintf( _x( '%s ago', 'human time difference. E.g. 10 days ago', 'newsplus' ), human_time_diff( get_the_time( 'U' ), current_time( 'timestamp' ) ) );
				}
				else {
					$date = get_the_time( esc_attr( $date_format ) );
				}
			}

			$post_id = get_the_ID();

			$post_class_obj = get_post_class( $post_id );
			$post_classes = '';

			// Post classes
			if ( isset( $post_class_obj ) && is_array( $post_class_obj ) ) {
				$post_classes = implode( ' ', $post_class_obj );
			}

			// Category and review stars
			$review_meta = '';
			
			// Create category list
			$cat_list = '<ul class="post-categories">';			
			$hasmore = false;
			$i = 0;
			$cats = get_the_category();
			$cat_limit = apply_filters( 'newsplus_cat_list_limit', 3 );
			$cat_count = intval( count( $cats ) - $cat_limit );
			if ( isset( $cats ) ) {
				foreach( $cats as $cat ) {
					if ( $i == $cat_limit ) {
						$hasmore = true;
						$cat_list .= '<li class="submenu-parent"><a class="cat-toggle" href="#">' . sprintf( esc_attr_x( '+ %d more', 'more count for category list', 'newsplus' ), number_format_i18n( $cat_count ) ) . '</a><ul class="cat-sub submenu">';
					}
					$cat_list .= '<li><a href="' . get_category_link( $cat->cat_ID ) . '">' . $cat->cat_name . '</a></li>';
					$i++;
				}
				$cat_list .= $hasmore ? '</ul></li></ul>' : '</ul>';	
			}		
			
			if ( 'list-small' == $template ) {
				$cat_meta = ( 'true' == $show_cats ) ? $cat_list : '';

				/**
				 * Post reviews
				 * Requires https://wordpress.org/plugins/wp-review/
				 */
				$review_meta = ( function_exists( 'wp_review_show_total' ) && 'true' == $show_reviews ) ? wp_review_show_total( $echo = false ) : '';
			} else {
				$cat_meta = ( 'true' != $hide_cats ) ? $cat_list : '';
				if ( function_exists( 'wp_review_show_total' ) && 'true' !== $hide_reviews ) {
					$review_meta = wp_review_show_total( $echo = false );
				}
			}

			// Author and date meta
			$meta_data = '';

			//if ( ! $hide_author || ! $hide_date ) {
				$author = get_the_author();
				if ( $show_avatar ) {
					$meta_data .= sprintf( '<div%s%s class="author-avatar-32%s"><a%s href="%s" title="%s">%s%s</a></div>',
						$enable_schema ? ' itemscope itemtype="' . $schema . 'Person"' : '',
						$enable_schema ? ' itemprop="author"' : '',
						$hide_author && $hide_date ? ' avatar-only' : '',
						$enable_schema ? ' itemprop="name"' : '',
						esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
						sprintf( esc_html__( 'More posts by %s', 'newsplus' ), esc_attr( $author ) ),
						$enable_schema ? '<span itemprop="image">' . get_avatar( get_the_author_meta( 'user_email' ), 32 ) . '</span>' : get_avatar( get_the_author_meta( 'user_email' ), 32 ),
						$enable_schema ? '<span class="hidden" itemprop="name">' . esc_attr( $author ) . '</span>' : ''

					);
				}

				$meta_data .= sprintf( '<ul class="entry-meta%s">',
					$show_avatar ? ' avatar-enabled' : ''
				);

				// Publisher Schema
				if ( $enable_schema ) {
					$meta_data .= '<li class="publisher-schema" itemscope itemtype="' . $schema . 'Organization" itemprop="publisher"><meta itemprop="name" content="' . esc_attr( get_bloginfo( 'name' ) ) . '"/><div itemprop="logo" itemscope itemtype="' . $protocol . '://schema.org/ImageObject"><img itemprop="url" src="' . esc_url( $publisher_logo ) . '" alt="' . esc_attr( get_bloginfo( 'name' ) ) . '"/></div></li>';
				}

				//if ( ! $hide_date ) {
					$modified_date_format = 'human' == $date_format ? get_option( 'date_format' ) : $date_format;
					$meta_data .= sprintf( '<li class="post-time%1$s"><time%2$s class="entry-date" datetime="%3$s">%4$s</time>%5$s</li>',
						$hide_date ? ' hidden' : '',
						$enable_schema ? ' itemprop="datePublished"' : '',
						esc_attr( get_the_date( 'c' ) ),
						esc_html( $date ),
						$enable_schema ? '<meta itemprop="dateModified" content="' . esc_attr( the_modified_date( 'c', '', '', false ) ) . '"/>' : ''
					);
				//}

				//if ( ! $hide_author ) {
					$meta_data .= sprintf( '<li%1$s%2$s class="post-author%3$s"><span class="screen-reader-text">%4$s </span><a href="%5$s">%6$s</a></li>',
						$enable_schema ? ' itemscope itemtype="' . $schema . 'Person"' : '',
						$enable_schema ? ' itemprop="author"' : '',
						$hide_author ? ' hidden' : '',
						esc_html_x( 'Author', 'Used before post author name.', 'newsplus' ),
						esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
						$enable_schema ? '<span itemprop="name">' . esc_attr( $author ) . '</span>' : esc_attr( $author )
					);
				//}

				$meta_data .= '</ul>';
			//}

			// Comment link
			$num_comments = get_comments_number();
			$comment_meta = '';
			if ( comments_open() && ( $num_comments >= 1 ) && ! $hide_comments ) {
				$comment_meta = sprintf( '<a href="%1$s" class="post-comment" title="%2$s">%3$s%4$s</a>',
					esc_url( get_comments_link() ),
					sprintf( __( 'Comment on %s', 'newsplus' ), esc_attr( get_the_title() ) ),
					$enable_schema ? '<meta itemprop="discussionUrl" content="' . esc_url( get_comments_link() ) . '" />' : '',
					$enable_schema ? '<span itemprop="commentCount">' . $num_comments . '</span>' : $num_comments
				);
			}

			/**
			 * Post views
			 * Requires Plugin https://wordpress.org/plugins/post-views-counter/
			 */
			$views_meta = '';
			if ( function_exists( 'pvc_get_post_views' ) && ! $hide_views ) {
				$views_meta = sprintf( '<span class="post-views">%s</span>',
					pvc_get_post_views()
				);
			}

			// Generate rows of content
			$row_1 = '';
			$row_2 = '';
			$row_3 = '';
			$row_4 = '';
			if ( $review_meta != '' || $cat_meta != '' ) {
				$row_1 .= '<aside class="meta-row cat-row">';
				if ( $cat_meta != '' ) {
					$row_1 .= sprintf( '<div%s class="meta-col%s">%s</div>',
						$enable_schema ? ' itemprop="about"' : '',
						$review_meta != '' ? ' col-60' : '',
						$cat_meta
					);
				}

				if ( $review_meta != '' ) {
					$row_1 .= sprintf( '<div class="meta-col%s">%s</div>',
						$cat_meta != '' ? ' col-40 text-right' : '',
						$review_meta
					);
				}
				$row_1 .= '</aside>';
			}

			//if ( $meta_data || $views_meta || $comment_meta ) {
				$row_4 .= sprintf( '<aside class="meta-row row-3%s">',
					( $hide_date && $hide_author && $hide_views && $hide_comments ) ? ' hidden' : ''
				);
				
				if ( '' == $views_meta && '' == $comment_meta ) {
					$row_4 .= sprintf( '<div class="meta-col">%s</div>', $meta_data );
				}

				elseif ( '' == $meta_data ) {
					$row_4 .= sprintf( '<div class="meta-col">%s%s</div>', $views_meta, $comment_meta );
				}

				else {
					$row_4 .= sprintf( '<div class="meta-col col-60">%s</div><div class="meta-col col-40 text-right">%s%s</div>', $meta_data, $views_meta, $comment_meta );
				}
				$row_4 .= '</aside>';
			//}

			if ( $readmore ) {
				if ( $meta_data ) {
					$row_2 = sprintf( '<aside class="meta-row row-2%s"><div class="meta-col">%s</div></aside>',
						( $hide_date && $hide_author && $hide_views && $hide_comments ) ? ' hidden' : '',
						$meta_data
					);
				}

				if ( $readmore || $views_meta || $comment_meta ) {
					$row_3 = sprintf( '<aside class="meta-row row-3"><div class="meta-col col-50"><a class="readmore-link" href="%s">%s</a></div><div class="meta-col col-50 text-right">%s%s</div></aside>',
						$ext_link && $custom_link ? esc_url( $custom_link) : esc_url( get_permalink() ),
						esc_attr( $readmore_text ),
						$views_meta,
						$comment_meta
					);
				}
			}

			else {
				$row_3 = $row_4;
			}

		$meta_arr = array();
		$meta_arr['row_1'] = $row_1;
		$meta_arr['row_2'] = $row_2;
		$meta_arr['row_3'] = $row_3;
		$meta_arr['row_4'] = $row_4;
		return $meta_arr;
	}
endif;

/**
 * Modify title tag to show itemprop="name"
 */

function newsplus_title_tag_with_schema() {

	global $page, $paged;
	
	$sep = '&#8211;';
	$title = wp_title( $sep, false, 'right' );
	
	$title .= get_bloginfo( 'name', 'display' );
	
	$site_description = get_bloginfo( 'description', 'display' );
	if ( $site_description && ( is_home() || is_front_page() ) ) {
		$title .= " $sep $site_description";
	}
	
	if ( ( $paged >= 2 || $page >= 2 ) && ! is_404() ) {
		$title .= " $sep " . sprintf( __( 'Page %s' ), max( $paged, $page ) );
	}
	
	printf( '<title%s>%s</title>' . PHP_EOL,
		get_option( 'pls_schema' ) ?  ' itemprop="name"' : '',
		$title
	);
}

add_action( 'wp_head', 'newsplus_title_tag_with_schema', 1 );


// Adds itemprop="url" attribute to nav menu links
function newsplus_add_attribute( $atts, $item, $args ) {	
	if ( get_option( 'pls_schema' ) ) {
		$atts['itemprop'] = 'url';
	}	
	return $atts;	
}
add_filter( 'nav_menu_link_attributes', 'newsplus_add_attribute', 10, 3 );


/**
 * Posts not found message
 */
 
 if ( ! function_exists( 'newsplus_no_posts' ) ) :
 	function newsplus_no_posts() {
	?>
        <article id="post-0" class="post no-results not-found">
            <header class="page-header">
                <h1 class="entry-title"><?php _e( 'Nothing Found', 'newsplus' ); ?></h1>
            </header>
            <div class="entry-content">
                <p><?php _e( 'It seems we can&rsquo;t find what you&rsquo;re looking for. Perhaps searching can help.', 'newsplus' ); ?></p>
                <?php get_search_form(); ?>
            </div><!-- .entry-content -->
        </article><!-- #post-0 -->
	<?php
    }
endif;

if ( ! function_exists( 'newsplus_author_bio' ) ) :
	function newsplus_author_bio() {

		$protocol = is_ssl() ? 'https' : 'http';
		$schema_url = $protocol . '://schema.org/';
		if ( get_the_author_meta( 'description' ) ) :
			printf( '<div class="author-info"%s%s>',
				get_option( 'pls_schema' ) ? ' itemscope itemtype="' . $schema_url . 'Person"' : '',
				get_option( 'pls_schema' ) ? ' itemprop="author"' : ''
			);
	
				printf( '<div class="author-avatar author-avatar-64"><img%s src="%s" alt="%s"/></div>',
					get_option( 'pls_schema' ) ? ' itemprop="image"' : '',
					get_avatar_url( get_the_author_meta( 'user_email' ), 64 ),
					get_the_author()                            
				);
	
				printf( '<div class="author-description"><h3%s class="author-title">%s</h3>',
					get_option( 'pls_schema' ) ? ' itemprop="name"' : '',
					get_the_author()
				);

					printf( '<p%s>%s</p>',
					get_option( 'pls_schema' ) ? ' itemprop="description"' : '',
					get_the_author_meta( 'description' )
					);
					?>
					<p class="author-link">
						<a class="more-link" href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>" rel="author">
							<?php printf( __( 'View all posts by %s <span class="meta-nav">&rarr;</span>', 'newsplus' ), get_the_author() ); ?>
						</a>
					</p><!-- .author-link	-->
				</div><!-- .author-description -->
			</div><!-- .author-info -->
		<?php endif; // has description
	}
endif;

if ( ! function_exists ( 'newsplus_video_thumb' ) ) :
	function newsplus_video_thumb( $url = '' ) {
		require_once( ABSPATH . 'wp-includes/class-oembed.php' );
		$oembed = new WP_oEmbed;
		
		$provider = $oembed->discover( $url );
		//$provider = 'http://www.youtube.com/oembed';
		$video = $oembed->fetch( $provider, $url, array() );
		if ( $video ) {
			$title = $video->title;
			$thumb = $video->thumbnail_url;
			return array( 'url' => $thumb, 'title' => $title );
		}
	}
endif;

/**
 * Social Sharing feature on single posts
 */
if ( ! function_exists( 'newsplus_social_sharing' ) ) :
	function newsplus_social_sharing( $sharing_buttons ) {
		global $post;

		setup_postdata( $post );

		// Set variables
		$out = '';
		$list = '';
		$share_image = '';
		$protocol = is_ssl() ? 'https' : 'http';

		if ( has_post_thumbnail( $post->ID ) ) {
			$share_image = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );
		}

		$share_content = strip_tags( get_the_excerpt() );
		$btn_count = count( $sharing_buttons );		
		
		if ( in_array( 'whatsapp', $sharing_buttons ) ) {
			if ( ! wp_is_mobile() ) {
				$btn_count--;
			}
		}

		$out .= '<div id="newsplus-social-sharing" class="ss-sharing-container btns-' . $btn_count . '">';

		$out .= '<ul class="np-sharing clearfix">';

		foreach ( $sharing_buttons as $button ) {

			switch( $button ) {

				case 'twitter':
					$list .= sprintf( '<li class="ss-twitter"><a href="%s://twitter.com/home?status=%s" target="_blank" title="%s"><i class="fa fa-twitter"></i><span class="sr-only">twitter</span></a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on twitter', 'newsplus' ) );
				break;

				case 'facebook':
					$list .= sprintf( '<li class="ss-facebook"><a href="%s://www.facebook.com/sharer/sharer.php?u=%s" target="_blank" title="%s"><i class="fa fa-facebook"></i><span class="sr-only">facebook</span></a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on facebook', 'newsplus' ) );
				break;
				
				case 'whatsapp':
					if ( wp_is_mobile() ) {
						$list .= sprintf( '<li class="ss-whatsapp"><a href="whatsapp://send?text=%s" title="%s" data-action="share/whatsapp/share"><i class="fa fa-whatsapp"></i><span class="sr-only">whatsapp</span></a></li>', urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Whatsapp', 'newsplus' ) );
					}
				break;				

				case 'googleplus':
					$list .= sprintf( '<li class="ss-gplus"><a href="%s://plus.google.com/share?url=%s" target="_blank" title="%s"><i class="fa fa-google-plus"></i><span class="sr-only">google+</span></a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on Google+', 'newsplus' ) );
				break;								

				case 'linkedin':
					$list .= sprintf( '<li class="ss-linkedin"><a href="%s://www.linkedin.com/shareArticle?mini=true&amp;url=%s" target="_blank" title="%s"><i class="fa fa-linkedin"></i><span class="sr-only">linkedin</span></a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on LinkedIn', 'newsplus' ) );
				break;

				case 'pinterest':
					$list .= sprintf( '<li class="ss-pint"><a href="%s://pinterest.com/pin/create/button/?url=%s&amp;media=%s" target="_blank" title="%s"><i class="fa fa-pinterest"></i><span class="sr-only">pinterest</span></a></li>',
						esc_attr( $protocol ),
						urlencode( esc_url( get_permalink() ) ),
						esc_url( $share_image ),
						esc_attr__( 'Pin it', 'newsplus' )
					);
				break;

				case 'vkontakte':
					$list .= sprintf( '<li class="ss-vk"><a href="%s://vkontakte.ru/share.php?url=%s" target="_blank" title="%s"><i class="fa fa-vk"></i><span class="sr-only">vkontakte</span></a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share via VK', 'newsplus' ) );
				break;

				case 'email':
					$list .= sprintf( '<li class="ss-mail"><a href="mailto:someone@example.com?Subject=%s" title="%s"><i class="fa fa-envelope"></i><span class="sr-only">email</span></a></li>', urlencode( esc_attr( get_the_title() ) ), esc_attr__( 'Email this', 'newsplus' ) );

				break;

				case 'print':
					$list .= sprintf( '<li class="ss-print"><a href="#" title="%s"><i class="fa fa-print"></i><span class="sr-only">print</span></a></li>', esc_attr__( 'Print', 'newsplus' ) );
				break;

				case 'reddit':
					$list .= sprintf( '<li class="ss-reddit"><a href="//www.reddit.com/submit" onclick="window.location = \'//www.reddit.com/submit?url=\' + encodeURIComponent(window.location); return false" title="%s"><i class="fa fa-reddit-square"></i><span class="sr-only">reddit</span><span class="sr-only">reddit</span></a></li>', esc_attr__( 'Reddit', 'newsplus' ) );
				break;
			} // switch

		} // foreach

		// Support extra meta items via action hook
		ob_start();
		do_action( 'newsplus_sharing_buttons_li' );
		$out .= ob_get_contents();
		ob_end_clean();

		$out .= $list . '</ul></div>';

		return $out;
	}
endif;

if ( ! function_exists( 'newsplus_sidebar_b' ) ) :

	function newsplus_sidebar_b() {
		
		$flag = false;
		
		if ( is_single() ) {
			global $post;
			$post_opts = get_post_meta( $post->ID, 'post_options', true );
			$sng_layout = ( isset( $post_opts['sng_layout'] ) && 'global' != $post_opts['sng_layout'] ) ? $post_opts['sng_layout'] : get_option( 'pls_sb_pos', 'ca' );
			$post_full_width 	= ( isset( $post_opts['post_full_width'] ) ) ? $post_opts['post_full_width'] : '';
			
			if ( $sng_layout == 'ac' || $sng_layout == 'ca' || $post_full_width ) {
				$flag = false;
			}	
			else {
				$flag = 'true';
			}
				
		}
		
		else {
			if ( get_option( 'pls_sb_pos', 'ca' ) == 'ac' || get_option( 'pls_sb_pos', 'ca' ) == 'ca' ) {
				$flag = false;
			}	
			else {
				$flag = 'true';
			}
		}
		
		if ( $flag && ( 'no-sb' != get_option( 'pls_sb_pos', 'ca' ) ) ) {
			if ( is_active_sidebar( 'sidebar-b' ) ) {
				echo '<div id="sidebar-b" clas="widget-area">';
				dynamic_sidebar( 'sidebar-b' );
				echo '</div><!-- /#sidebar-b -->';
			} else {
				echo '<div class="sb_notifier">';
				_e('<p>Sidebar not configured yet. You can place widgets by navigating to WordPress Appearance > Widgets.</p>', 'newsplus');
				echo '</div>';
			}
		}
	}
endif;

/**
 * Limit category links to xx number
 */
function newsplus_cat_links_limit() {
	if ( is_single() ) {
		return 4;
	}
	else {
		return 2;
	}
}

add_filter( 'newsplus_cat_list_limit', 'newsplus_cat_links_limit' );

function newsplus_make_css() {
	if ( get_option( 'pls_disable_custom_font', false ) ) {
		return;
	}
	$css = '';
	$global_font = get_option( 'pls_global_font', '' );
	$heading_font = get_option( 'pls_heading_font', '' );
	$menu_font = get_option( 'pls_menu_font', '' );
	
	// Font CSS
	if ( '' != $global_font ) {
		newsplus_enqueue_fonts( $global_font );		
		$global_style = newsplus_create_font_style( $global_font );
		$css .= 'body,body.custom-font-enabled{' . $global_style . '}';
	}
	
	if ( '' != $heading_font ) {
		newsplus_enqueue_fonts( $heading_font );		
		$heading_style = newsplus_create_font_style( $heading_font );
		$css .= 'h1,h2,h3,h4,h5,h6{' . $heading_style . '}';
	}
	
	if ( '' != $menu_font ) {
		newsplus_enqueue_fonts( $menu_font );		
		$heading_style = newsplus_create_font_style( $menu_font );
		$css .= '.primary-nav{' . $heading_style . '}';
	}		
	
	echo '<style type="text/css" id="newsplus_custom_css">' . strip_tags( $css ) . '</style>';
}

add_action( 'wp_head', 'newsplus_make_css' );

if ( ! function_exists( 'newsplus_enqueue_fonts' ) ) :	
	function newsplus_enqueue_fonts( $fontsData ) {
		$subset = get_option( 'pls_font_subset', '' );
		// Enqueue font
		if ( isset( $fontsData ) ) {
			wp_enqueue_style( 'newsplus_google_fonts_' . newsplus_create_safe_class( $fontsData ), '//fonts.googleapis.com/css?family=' . $fontsData . $subset );
		}
	}
endif;

if ( ! function_exists( 'newsplus_create_font_style' ) ) :	
	function newsplus_create_font_style( $fontsData ) {         
        $inline_style = '';
		if ( isset( $fontsData ) ) {			
			$fontFamily = explode( ':', $fontsData );
			$styles[] = isset( $fontFamily[0] ) ? 'font-family:\'' . $fontFamily[0] . '\'': '';
			 
			if ( isset( $styles ) && is_array( $styles ) ) {
				foreach( $styles as $attribute ){           
					$inline_style .= $attribute.'; ';       
				}
			}
		}         
        return $inline_style;         
    }
endif;

if ( ! function_exists( 'newsplus_create_safe_class' ) ) :	
	function newsplus_create_safe_class( $class ) {
		return preg_replace( '/\W+/', '', strtolower( str_replace( ' ', '_', strip_tags( $class ) ) ) );
	}
endif;

function newsplus_google_font_list() {
	return apply_filters( 'newsplus_google_font_list',
				array(
					__( '-- Select --', 'wppm' ) => '',
					'ABeeZee' => 'ABeeZee:regular,italic',
					'Abel' => 'Abel:regular',
					'Abhaya Libre' => 'Abhaya Libre:regular,500,600,700,800',
					'Abril Fatface' => 'Abril Fatface:regular',
					'Aclonica' => 'Aclonica:regular',
					'Acme' => 'Acme:regular',
					'Actor' => 'Actor:regular',
					'Adamina' => 'Adamina:regular',
					'Advent Pro' => 'Advent Pro:100,200,300,regular,500,600,700',
					'Aguafina Script' => 'Aguafina Script:regular',
					'Akronim' => 'Akronim:regular',
					'Aladin' => 'Aladin:regular',
					'Aldrich' => 'Aldrich:regular',
					'Alef' => 'Alef:regular,700',
					'Alegreya' => 'Alegreya:regular,italic,700,700italic,900,900italic',
					'Alegreya SC' => 'Alegreya SC:regular,italic,700,700italic,900,900italic',
					'Alegreya Sans' => 'Alegreya Sans:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Alegreya Sans SC' => 'Alegreya Sans SC:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Alex Brush' => 'Alex Brush:regular',
					'Alfa Slab One' => 'Alfa Slab One:regular',
					'Alice' => 'Alice:regular',
					'Alike' => 'Alike:regular',
					'Alike Angular' => 'Alike Angular:regular',
					'Allan' => 'Allan:regular,700',
					'Allerta' => 'Allerta:regular',
					'Allerta Stencil' => 'Allerta Stencil:regular',
					'Allura' => 'Allura:regular',
					'Almendra' => 'Almendra:regular,italic,700,700italic',
					'Almendra Display' => 'Almendra Display:regular',
					'Almendra SC' => 'Almendra SC:regular',
					'Amarante' => 'Amarante:regular',
					'Amaranth' => 'Amaranth:regular,italic,700,700italic',
					'Amatic SC' => 'Amatic SC:regular,700',
					'Amatica SC' => 'Amatica SC:regular,700',
					'Amethysta' => 'Amethysta:regular',
					'Amiko' => 'Amiko:regular,600,700',
					'Amiri' => 'Amiri:regular,italic,700,700italic',
					'Amita' => 'Amita:regular,700',
					'Anaheim' => 'Anaheim:regular',
					'Andada' => 'Andada:regular',
					'Andika' => 'Andika:regular',
					'Angkor' => 'Angkor:regular',
					'Annie Use Your Telescope' => 'Annie Use Your Telescope:regular',
					'Anonymous Pro' => 'Anonymous Pro:regular,italic,700,700italic',
					'Antic' => 'Antic:regular',
					'Antic Didone' => 'Antic Didone:regular',
					'Antic Slab' => 'Antic Slab:regular',
					'Anton' => 'Anton:regular',
					'Arapey' => 'Arapey:regular,italic',
					'Arbutus' => 'Arbutus:regular',
					'Arbutus Slab' => 'Arbutus Slab:regular',
					'Architects Daughter' => 'Architects Daughter:regular',
					'Archivo Black' => 'Archivo Black:regular',
					'Archivo Narrow' => 'Archivo Narrow:regular,italic,700,700italic',
					'Aref Ruqaa' => 'Aref Ruqaa:regular,700',
					'Arima Madurai' => 'Arima Madurai:100,200,300,regular,500,700,800,900',
					'Arimo' => 'Arimo:regular,italic,700,700italic',
					'Arizonia' => 'Arizonia:regular',
					'Armata' => 'Armata:regular',
					'Arsenal' => 'Arsenal:regular,italic,700,700italic',
					'Artifika' => 'Artifika:regular',
					'Arvo' => 'Arvo:regular,italic,700,700italic',
					'Arya' => 'Arya:regular,700',
					'Asap' => 'Asap:regular,italic,500,500italic,700,700italic',
					'Asar' => 'Asar:regular',
					'Asset' => 'Asset:regular',
					'Assistant' => 'Assistant:200,300,regular,600,700,800',
					'Astloch' => 'Astloch:regular,700',
					'Asul' => 'Asul:regular,700',
					'Athiti' => 'Athiti:200,300,regular,500,600,700',
					'Atma' => 'Atma:300,regular,500,600,700',
					'Atomic Age' => 'Atomic Age:regular',
					'Aubrey' => 'Aubrey:regular',
					'Audiowide' => 'Audiowide:regular',
					'Autour One' => 'Autour One:regular',
					'Average' => 'Average:regular',
					'Average Sans' => 'Average Sans:regular',
					'Averia Gruesa Libre' => 'Averia Gruesa Libre:regular',
					'Averia Libre' => 'Averia Libre:300,300italic,regular,italic,700,700italic',
					'Averia Sans Libre' => 'Averia Sans Libre:300,300italic,regular,italic,700,700italic',
					'Averia Serif Libre' => 'Averia Serif Libre:300,300italic,regular,italic,700,700italic',
					'Bad Script' => 'Bad Script:regular',
					'Bahiana' => 'Bahiana:regular',
					'Baloo' => 'Baloo:regular',
					'Baloo Bhai' => 'Baloo Bhai:regular',
					'Baloo Bhaina' => 'Baloo Bhaina:regular',
					'Baloo Chettan' => 'Baloo Chettan:regular',
					'Baloo Da' => 'Baloo Da:regular',
					'Baloo Paaji' => 'Baloo Paaji:regular',
					'Baloo Tamma' => 'Baloo Tamma:regular',
					'Baloo Thambi' => 'Baloo Thambi:regular',
					'Balthazar' => 'Balthazar:regular',
					'Bangers' => 'Bangers:regular',
					'Barrio' => 'Barrio:regular',
					'Basic' => 'Basic:regular',
					'Battambang' => 'Battambang:regular,700',
					'Baumans' => 'Baumans:regular',
					'Bayon' => 'Bayon:regular',
					'Belgrano' => 'Belgrano:regular',
					'Bellefair' => 'Bellefair:regular',
					'Belleza' => 'Belleza:regular',
					'BenchNine' => 'BenchNine:300,regular,700',
					'Bentham' => 'Bentham:regular',
					'Berkshire Swash' => 'Berkshire Swash:regular',
					'Bevan' => 'Bevan:regular',
					'Bigelow Rules' => 'Bigelow Rules:regular',
					'Bigshot One' => 'Bigshot One:regular',
					'Bilbo' => 'Bilbo:regular',
					'Bilbo Swash Caps' => 'Bilbo Swash Caps:regular',
					'BioRhyme' => 'BioRhyme:200,300,regular,700,800',
					'BioRhyme Expanded' => 'BioRhyme Expanded:200,300,regular,700,800',
					'Biryani' => 'Biryani:200,300,regular,600,700,800,900',
					'Bitter' => 'Bitter:regular,italic,700',
					'Black Ops One' => 'Black Ops One:regular',
					'Bokor' => 'Bokor:regular',
					'Bonbon' => 'Bonbon:regular',
					'Boogaloo' => 'Boogaloo:regular',
					'Bowlby One' => 'Bowlby One:regular',
					'Bowlby One SC' => 'Bowlby One SC:regular',
					'Brawler' => 'Brawler:regular',
					'Bree Serif' => 'Bree Serif:regular',
					'Bubblegum Sans' => 'Bubblegum Sans:regular',
					'Bubbler One' => 'Bubbler One:regular',
					'Buda' => 'Buda:300',
					'Buenard' => 'Buenard:regular,700',
					'Bungee' => 'Bungee:regular',
					'Bungee Hairline' => 'Bungee Hairline:regular',
					'Bungee Inline' => 'Bungee Inline:regular',
					'Bungee Outline' => 'Bungee Outline:regular',
					'Bungee Shade' => 'Bungee Shade:regular',
					'Butcherman' => 'Butcherman:regular',
					'Butterfly Kids' => 'Butterfly Kids:regular',
					'Cabin' => 'Cabin:regular,italic,500,500italic,600,600italic,700,700italic',
					'Cabin Condensed' => 'Cabin Condensed:regular,500,600,700',
					'Cabin Sketch' => 'Cabin Sketch:regular,700',
					'Caesar Dressing' => 'Caesar Dressing:regular',
					'Cagliostro' => 'Cagliostro:regular',
					'Cairo' => 'Cairo:200,300,regular,600,700,900',
					'Calligraffitti' => 'Calligraffitti:regular',
					'Cambay' => 'Cambay:regular,italic,700,700italic',
					'Cambo' => 'Cambo:regular',
					'Candal' => 'Candal:regular',
					'Cantarell' => 'Cantarell:regular,italic,700,700italic',
					'Cantata One' => 'Cantata One:regular',
					'Cantora One' => 'Cantora One:regular',
					'Capriola' => 'Capriola:regular',
					'Cardo' => 'Cardo:regular,italic,700',
					'Carme' => 'Carme:regular',
					'Carrois Gothic' => 'Carrois Gothic:regular',
					'Carrois Gothic SC' => 'Carrois Gothic SC:regular',
					'Carter One' => 'Carter One:regular',
					'Catamaran' => 'Catamaran:100,200,300,regular,500,600,700,800,900',
					'Caudex' => 'Caudex:regular,italic,700,700italic',
					'Caveat' => 'Caveat:regular,700',
					'Caveat Brush' => 'Caveat Brush:regular',
					'Cedarville Cursive' => 'Cedarville Cursive:regular',
					'Ceviche One' => 'Ceviche One:regular',
					'Changa' => 'Changa:200,300,regular,500,600,700,800',
					'Changa One' => 'Changa One:regular,italic',
					'Chango' => 'Chango:regular',
					'Chathura' => 'Chathura:100,300,regular,700,800',
					'Chau Philomene One' => 'Chau Philomene One:regular,italic',
					'Chela One' => 'Chela One:regular',
					'Chelsea Market' => 'Chelsea Market:regular',
					'Chenla' => 'Chenla:regular',
					'Cherry Cream Soda' => 'Cherry Cream Soda:regular',
					'Cherry Swash' => 'Cherry Swash:regular,700',
					'Chewy' => 'Chewy:regular',
					'Chicle' => 'Chicle:regular',
					'Chivo' => 'Chivo:300,300italic,regular,italic,700,700italic,900,900italic',
					'Chonburi' => 'Chonburi:regular',
					'Cinzel' => 'Cinzel:regular,700,900',
					'Cinzel Decorative' => 'Cinzel Decorative:regular,700,900',
					'Clicker Script' => 'Clicker Script:regular',
					'Coda' => 'Coda:regular,800',
					'Coda Caption' => 'Coda Caption:800',
					'Codystar' => 'Codystar:300,regular',
					'Coiny' => 'Coiny:regular',
					'Combo' => 'Combo:regular',
					'Comfortaa' => 'Comfortaa:300,regular,700',
					'Coming Soon' => 'Coming Soon:regular',
					'Concert One' => 'Concert One:regular',
					'Condiment' => 'Condiment:regular',
					'Content' => 'Content:regular,700',
					'Contrail One' => 'Contrail One:regular',
					'Convergence' => 'Convergence:regular',
					'Cookie' => 'Cookie:regular',
					'Copse' => 'Copse:regular',
					'Corben' => 'Corben:regular,700',
					'Cormorant' => 'Cormorant:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant Garamond' => 'Cormorant Garamond:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant Infant' => 'Cormorant Infant:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant SC' => 'Cormorant SC:300,regular,500,600,700',
					'Cormorant Unicase' => 'Cormorant Unicase:300,regular,500,600,700',
					'Cormorant Upright' => 'Cormorant Upright:300,regular,500,600,700',
					'Courgette' => 'Courgette:regular',
					'Cousine' => 'Cousine:regular,italic,700,700italic',
					'Coustard' => 'Coustard:regular,900',
					'Covered By Your Grace' => 'Covered By Your Grace:regular',
					'Crafty Girls' => 'Crafty Girls:regular',
					'Creepster' => 'Creepster:regular',
					'Crete Round' => 'Crete Round:regular,italic',
					'Crimson Text' => 'Crimson Text:regular,italic,600,600italic,700,700italic',
					'Croissant One' => 'Croissant One:regular',
					'Crushed' => 'Crushed:regular',
					'Cuprum' => 'Cuprum:regular,italic,700,700italic',
					'Cutive' => 'Cutive:regular',
					'Cutive Mono' => 'Cutive Mono:regular',
					'Damion' => 'Damion:regular',
					'Dancing Script' => 'Dancing Script:regular,700',
					'Dangrek' => 'Dangrek:regular',
					'David Libre' => 'David Libre:regular,500,700',
					'Dawning of a New Day' => 'Dawning of a New Day:regular',
					'Days One' => 'Days One:regular',
					'Dekko' => 'Dekko:regular',
					'Delius' => 'Delius:regular',
					'Delius Swash Caps' => 'Delius Swash Caps:regular',
					'Delius Unicase' => 'Delius Unicase:regular,700',
					'Della Respira' => 'Della Respira:regular',
					'Denk One' => 'Denk One:regular',
					'Devonshire' => 'Devonshire:regular',
					'Dhurjati' => 'Dhurjati:regular',
					'Didact Gothic' => 'Didact Gothic:regular',
					'Diplomata' => 'Diplomata:regular',
					'Diplomata SC' => 'Diplomata SC:regular',
					'Domine' => 'Domine:regular,700',
					'Donegal One' => 'Donegal One:regular',
					'Doppio One' => 'Doppio One:regular',
					'Dorsa' => 'Dorsa:regular',
					'Dosis' => 'Dosis:200,300,regular,500,600,700,800',
					'Dr Sugiyama' => 'Dr Sugiyama:regular',
					'Droid Sans' => 'Droid Sans:regular,700',
					'Droid Sans Mono' => 'Droid Sans Mono:regular',
					'Droid Serif' => 'Droid Serif:regular,italic,700,700italic',
					'Duru Sans' => 'Duru Sans:regular',
					'Dynalight' => 'Dynalight:regular',
					'EB Garamond' => 'EB Garamond:regular',
					'Eagle Lake' => 'Eagle Lake:regular',
					'Eater' => 'Eater:regular',
					'Economica' => 'Economica:regular,italic,700,700italic',
					'Eczar' => 'Eczar:regular,500,600,700,800',
					'Ek Mukta' => 'Ek Mukta:200,300,regular,500,600,700,800',
					'El Messiri' => 'El Messiri:regular,500,600,700',
					'Electrolize' => 'Electrolize:regular',
					'Elsie' => 'Elsie:regular,900',
					'Elsie Swash Caps' => 'Elsie Swash Caps:regular,900',
					'Emblema One' => 'Emblema One:regular',
					'Emilys Candy' => 'Emilys Candy:regular',
					'Engagement' => 'Engagement:regular',
					'Englebert' => 'Englebert:regular',
					'Enriqueta' => 'Enriqueta:regular,700',
					'Erica One' => 'Erica One:regular',
					'Esteban' => 'Esteban:regular',
					'Euphoria Script' => 'Euphoria Script:regular',
					'Ewert' => 'Ewert:regular',
					'Exo' => 'Exo:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Exo 2' => 'Exo 2:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Expletus Sans' => 'Expletus Sans:regular,italic,500,500italic,600,600italic,700,700italic',
					'Fanwood Text' => 'Fanwood Text:regular,italic',
					'Farsan' => 'Farsan:regular',
					'Fascinate' => 'Fascinate:regular',
					'Fascinate Inline' => 'Fascinate Inline:regular',
					'Faster One' => 'Faster One:regular',
					'Fasthand' => 'Fasthand:regular',
					'Fauna One' => 'Fauna One:regular',
					'Federant' => 'Federant:regular',
					'Federo' => 'Federo:regular',
					'Felipa' => 'Felipa:regular',
					'Fenix' => 'Fenix:regular',
					'Finger Paint' => 'Finger Paint:regular',
					'Fira Mono' => 'Fira Mono:regular,500,700',
					'Fira Sans' => 'Fira Sans:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fira Sans Condensed' => 'Fira Sans Condensed:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fira Sans Extra Condensed' => 'Fira Sans Extra Condensed:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fjalla One' => 'Fjalla One:regular',
					'Fjord One' => 'Fjord One:regular',
					'Flamenco' => 'Flamenco:300,regular',
					'Flavors' => 'Flavors:regular',
					'Fondamento' => 'Fondamento:regular,italic',
					'Fontdiner Swanky' => 'Fontdiner Swanky:regular',
					'Forum' => 'Forum:regular',
					'Francois One' => 'Francois One:regular',
					'Frank Ruhl Libre' => 'Frank Ruhl Libre:300,regular,500,700,900',
					'Freckle Face' => 'Freckle Face:regular',
					'Fredericka the Great' => 'Fredericka the Great:regular',
					'Fredoka One' => 'Fredoka One:regular',
					'Freehand' => 'Freehand:regular',
					'Fresca' => 'Fresca:regular',
					'Frijole' => 'Frijole:regular',
					'Fruktur' => 'Fruktur:regular',
					'Fugaz One' => 'Fugaz One:regular',
					'GFS Didot' => 'GFS Didot:regular',
					'GFS Neohellenic' => 'GFS Neohellenic:regular,italic,700,700italic',
					'Gabriela' => 'Gabriela:regular',
					'Gafata' => 'Gafata:regular',
					'Galada' => 'Galada:regular',
					'Galdeano' => 'Galdeano:regular',
					'Galindo' => 'Galindo:regular',
					'Gentium Basic' => 'Gentium Basic:regular,italic,700,700italic',
					'Gentium Book Basic' => 'Gentium Book Basic:regular,italic,700,700italic',
					'Geo' => 'Geo:regular,italic',
					'Geostar' => 'Geostar:regular',
					'Geostar Fill' => 'Geostar Fill:regular',
					'Germania One' => 'Germania One:regular',
					'Gidugu' => 'Gidugu:regular',
					'Gilda Display' => 'Gilda Display:regular',
					'Give You Glory' => 'Give You Glory:regular',
					'Glass Antiqua' => 'Glass Antiqua:regular',
					'Glegoo' => 'Glegoo:regular,700',
					'Gloria Hallelujah' => 'Gloria Hallelujah:regular',
					'Goblin One' => 'Goblin One:regular',
					'Gochi Hand' => 'Gochi Hand:regular',
					'Gorditas' => 'Gorditas:regular,700',
					'Goudy Bookletter 1911' => 'Goudy Bookletter 1911:regular',
					'Graduate' => 'Graduate:regular',
					'Grand Hotel' => 'Grand Hotel:regular',
					'Gravitas One' => 'Gravitas One:regular',
					'Great Vibes' => 'Great Vibes:regular',
					'Griffy' => 'Griffy:regular',
					'Gruppo' => 'Gruppo:regular',
					'Gudea' => 'Gudea:regular,italic,700',
					'Gurajada' => 'Gurajada:regular',
					'Habibi' => 'Habibi:regular',
					'Halant' => 'Halant:300,regular,500,600,700',
					'Hammersmith One' => 'Hammersmith One:regular',
					'Hanalei' => 'Hanalei:regular',
					'Hanalei Fill' => 'Hanalei Fill:regular',
					'Handlee' => 'Handlee:regular',
					'Hanuman' => 'Hanuman:regular,700',
					'Happy Monkey' => 'Happy Monkey:regular',
					'Harmattan' => 'Harmattan:regular',
					'Headland One' => 'Headland One:regular',
					'Heebo' => 'Heebo:100,300,regular,500,700,800,900',
					'Henny Penny' => 'Henny Penny:regular',
					'Herr Von Muellerhoff' => 'Herr Von Muellerhoff:regular',
					'Hind' => 'Hind:300,regular,500,600,700',
					'Hind Guntur' => 'Hind Guntur:300,regular,500,600,700',
					'Hind Madurai' => 'Hind Madurai:300,regular,500,600,700',
					'Hind Siliguri' => 'Hind Siliguri:300,regular,500,600,700',
					'Hind Vadodara' => 'Hind Vadodara:300,regular,500,600,700',
					'Holtwood One SC' => 'Holtwood One SC:regular',
					'Homemade Apple' => 'Homemade Apple:regular',
					'Homenaje' => 'Homenaje:regular',
					'IM Fell DW Pica' => 'IM Fell DW Pica:regular,italic',
					'IM Fell DW Pica SC' => 'IM Fell DW Pica SC:regular',
					'IM Fell Double Pica' => 'IM Fell Double Pica:regular,italic',
					'IM Fell Double Pica SC' => 'IM Fell Double Pica SC:regular',
					'IM Fell English' => 'IM Fell English:regular,italic',
					'IM Fell English SC' => 'IM Fell English SC:regular',
					'IM Fell French Canon' => 'IM Fell French Canon:regular,italic',
					'IM Fell French Canon SC' => 'IM Fell French Canon SC:regular',
					'IM Fell Great Primer' => 'IM Fell Great Primer:regular,italic',
					'IM Fell Great Primer SC' => 'IM Fell Great Primer SC:regular',
					'Iceberg' => 'Iceberg:regular',
					'Iceland' => 'Iceland:regular',
					'Imprima' => 'Imprima:regular',
					
					'Inconsolata' => 'Inconsolata:regular,700',
					'Inder' => 'Inder:regular',
					'Indie Flower' => 'Indie Flower:regular',
					'Inika' => 'Inika:regular,700',
					'Inknut Antiqua' => 'Inknut Antiqua:300,regular,500,600,700,800,900',
					'Irish Grover' => 'Irish Grover:regular',
					'Istok Web' => 'Istok Web:regular,italic,700,700italic',
					'Italiana' => 'Italiana:regular',
					'Italianno' => 'Italianno:regular',
					'Itim' => 'Itim:regular',
					'Jacques Francois' => 'Jacques Francois:regular',
					'Jacques Francois Shadow' => 'Jacques Francois Shadow:regular',
					'Jaldi' => 'Jaldi:regular,700',
					'Jim Nightshade' => 'Jim Nightshade:regular',
					'Jockey One' => 'Jockey One:regular',
					'Jolly Lodger' => 'Jolly Lodger:regular',
					'Jomhuria' => 'Jomhuria:regular',
					'Josefin Sans' => 'Josefin Sans:100,100italic,300,300italic,regular,italic,600,600italic,700,700italic',
					'Josefin Slab' => 'Josefin Slab:100,100italic,300,300italic,regular,italic,600,600italic,700,700italic',
					'Joti One' => 'Joti One:regular',
					'Judson' => 'Judson:regular,italic,700',
					'Julee' => 'Julee:regular',
					'Julius Sans One' => 'Julius Sans One:regular',
					'Junge' => 'Junge:regular',
					'Jura' => 'Jura:300,regular,500,600,700',
					'Just Another Hand' => 'Just Another Hand:regular',
					'Just Me Again Down Here' => 'Just Me Again Down Here:regular',
					'Kadwa' => 'Kadwa:regular,700',
					'Kalam' => 'Kalam:300,regular,700',
					'Kameron' => 'Kameron:regular,700',
					'Kanit' => 'Kanit:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Kantumruy' => 'Kantumruy:300,regular,700',
					'Karla' => 'Karla:regular,italic,700,700italic',
					'Karma' => 'Karma:300,regular,500,600,700',
					'Katibeh' => 'Katibeh:regular',
					'Kaushan Script' => 'Kaushan Script:regular',
					'Kavivanar' => 'Kavivanar:regular',
					'Kavoon' => 'Kavoon:regular',
					'Kdam Thmor' => 'Kdam Thmor:regular',
					'Keania One' => 'Keania One:regular',
					'Kelly Slab' => 'Kelly Slab:regular',
					'Kenia' => 'Kenia:regular',
					'Khand' => 'Khand:300,regular,500,600,700',
					'Khmer' => 'Khmer:regular',
					'Khula' => 'Khula:300,regular,600,700,800',
					'Kite One' => 'Kite One:regular',
					'Knewave' => 'Knewave:regular',
					'Kotta One' => 'Kotta One:regular',
					'Koulen' => 'Koulen:regular',
					'Kranky' => 'Kranky:regular',
					'Kreon' => 'Kreon:300,regular,700',
					'Kristi' => 'Kristi:regular',
					'Krona One' => 'Krona One:regular',
					'Kumar One' => 'Kumar One:regular',
					'Kumar One Outline' => 'Kumar One Outline:regular',
					'Kurale' => 'Kurale:regular',
					'La Belle Aurore' => 'La Belle Aurore:regular',
					'Laila' => 'Laila:300,regular,500,600,700',
					'Lakki Reddy' => 'Lakki Reddy:regular',
					'Lalezar' => 'Lalezar:regular',
					'Lancelot' => 'Lancelot:regular',
					'Lateef' => 'Lateef:regular',
					'Lato' => 'Lato:100,100italic,300,300italic,regular,italic,700,700italic,900,900italic',
					'League Script' => 'League Script:regular',
					'Leckerli One' => 'Leckerli One:regular',
					'Ledger' => 'Ledger:regular',
					'Lekton' => 'Lekton:regular,italic,700',
					'Lemon' => 'Lemon:regular',
					'Lemonada' => 'Lemonada:300,regular,600,700',
					'Libre Baskerville' => 'Libre Baskerville:regular,italic,700',
					'Libre Franklin' => 'Libre Franklin:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Life Savers' => 'Life Savers:regular,700',
					'Lilita One' => 'Lilita One:regular',
					'Lily Script One' => 'Lily Script One:regular',
					'Limelight' => 'Limelight:regular',
					'Linden Hill' => 'Linden Hill:regular,italic',
					'Lobster' => 'Lobster:regular',
					'Lobster Two' => 'Lobster Two:regular,italic,700,700italic',
					'Londrina Outline' => 'Londrina Outline:regular',
					'Londrina Shadow' => 'Londrina Shadow:regular',
					'Londrina Sketch' => 'Londrina Sketch:regular',
					'Londrina Solid' => 'Londrina Solid:regular',
					'Lora' => 'Lora:regular,italic,700,700italic',
					'Love Ya Like A Sister' => 'Love Ya Like A Sister:regular',
					'Loved by the King' => 'Loved by the King:regular',
					'Lovers Quarrel' => 'Lovers Quarrel:regular',
					'Luckiest Guy' => 'Luckiest Guy:regular',
					'Lusitana' => 'Lusitana:regular,700',
					'Lustria' => 'Lustria:regular',
					'Macondo' => 'Macondo:regular',
					'Macondo Swash Caps' => 'Macondo Swash Caps:regular',
					'Mada' => 'Mada:300,regular,500,900',
					'Magra' => 'Magra:regular,700',
					'Maiden Orange' => 'Maiden Orange:regular',
					'Maitree' => 'Maitree:200,300,regular,500,600,700',
					'Mako' => 'Mako:regular',
					'Mallanna' => 'Mallanna:regular',
					'Mandali' => 'Mandali:regular',
					'Marcellus' => 'Marcellus:regular',
					'Marcellus SC' => 'Marcellus SC:regular',
					'Marck Script' => 'Marck Script:regular',
					'Margarine' => 'Margarine:regular',
					'Marko One' => 'Marko One:regular',
					'Marmelad' => 'Marmelad:regular',
					'Martel' => 'Martel:200,300,regular,600,700,800,900',
					'Martel Sans' => 'Martel Sans:200,300,regular,600,700,800,900',
					'Marvel' => 'Marvel:regular,italic,700,700italic',
					'Mate' => 'Mate:regular,italic',
					'Mate SC' => 'Mate SC:regular',
					'Maven Pro' => 'Maven Pro:regular,500,700,900',
					'McLaren' => 'McLaren:regular',
					'Meddon' => 'Meddon:regular',
					'MedievalSharp' => 'MedievalSharp:regular',
					'Medula One' => 'Medula One:regular',
					'Meera Inimai' => 'Meera Inimai:regular',
					'Megrim' => 'Megrim:regular',
					'Meie Script' => 'Meie Script:regular',
					'Merienda' => 'Merienda:regular,700',
					'Merienda One' => 'Merienda One:regular',
					'Merriweather' => 'Merriweather:300,300italic,regular,italic,700,700italic,900,900italic',
					'Merriweather Sans' => 'Merriweather Sans:300,300italic,regular,italic,700,700italic,800,800italic',
					'Metal' => 'Metal:regular',
					'Metal Mania' => 'Metal Mania:regular',
					'Metamorphous' => 'Metamorphous:regular',
					'Metrophobic' => 'Metrophobic:regular',
					'Michroma' => 'Michroma:regular',
					'Milonga' => 'Milonga:regular',
					'Miltonian' => 'Miltonian:regular',
					'Miltonian Tattoo' => 'Miltonian Tattoo:regular',
					'Miniver' => 'Miniver:regular',
					'Miriam Libre' => 'Miriam Libre:regular,700',
					'Mirza' => 'Mirza:regular,500,600,700',
					'Miss Fajardose' => 'Miss Fajardose:regular',
					'Mitr' => 'Mitr:200,300,regular,500,600,700',
					'Modak' => 'Modak:regular',
					'Modern Antiqua' => 'Modern Antiqua:regular',
					'Mogra' => 'Mogra:regular',
					'Molengo' => 'Molengo:regular',
					'Molle' => 'Molle:italic',
					'Monda' => 'Monda:regular,700',
					'Monofett' => 'Monofett:regular',
					'Monoton' => 'Monoton:regular',
					'Monsieur La Doulaise' => 'Monsieur La Doulaise:regular',
					'Montaga' => 'Montaga:regular',
					'Montez' => 'Montez:regular',
					'Montserrat' => 'Montserrat:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Montserrat Alternates' => 'Montserrat Alternates:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Montserrat Subrayada' => 'Montserrat Subrayada:regular,700',
					'Moul' => 'Moul:regular',
					'Moulpali' => 'Moulpali:regular',
					'Mountains of Christmas' => 'Mountains of Christmas:regular,700',
					'Mouse Memoirs' => 'Mouse Memoirs:regular',
					'Mr Bedfort' => 'Mr Bedfort:regular',
					'Mr Dafoe' => 'Mr Dafoe:regular',
					'Mr De Haviland' => 'Mr De Haviland:regular',
					'Mrs Saint Delafield' => 'Mrs Saint Delafield:regular',
					'Mrs Sheppards' => 'Mrs Sheppards:regular',
					'Mukta Vaani' => 'Mukta Vaani:200,300,regular,500,600,700,800',
					'Muli' => 'Muli:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Mystery Quest' => 'Mystery Quest:regular',
					'NTR' => 'NTR:regular',
					'Neucha' => 'Neucha:regular',
					'Neuton' => 'Neuton:200,300,regular,italic,700,800',
					'New Rocker' => 'New Rocker:regular',
					'News Cycle' => 'News Cycle:regular,700',
					'Niconne' => 'Niconne:regular',
					'Nixie One' => 'Nixie One:regular',
					'Nobile' => 'Nobile:regular,italic,700,700italic',
					'Nokora' => 'Nokora:regular,700',
					'Norican' => 'Norican:regular',
					'Nosifer' => 'Nosifer:regular',
					'Nothing You Could Do' => 'Nothing You Could Do:regular',
					'Noticia Text' => 'Noticia Text:regular,italic,700,700italic',
					'Noto Sans' => 'Noto Sans:regular,italic,700,700italic',
					'Noto Serif' => 'Noto Serif:regular,italic,700,700italic',
					'Nova Cut' => 'Nova Cut:regular',
					'Nova Flat' => 'Nova Flat:regular',
					'Nova Mono' => 'Nova Mono:regular',
					'Nova Oval' => 'Nova Oval:regular',
					'Nova Round' => 'Nova Round:regular',
					'Nova Script' => 'Nova Script:regular',
					'Nova Slim' => 'Nova Slim:regular',
					'Nova Square' => 'Nova Square:regular',
					'Numans' => 'Numans:regular',
					'Nunito' => 'Nunito:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Nunito Sans' => 'Nunito Sans:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Odor Mean Chey' => 'Odor Mean Chey:regular',
					'Offside' => 'Offside:regular',
					'Old Standard TT' => 'Old Standard TT:regular,italic,700',
					'Oldenburg' => 'Oldenburg:regular',
					'Oleo Script' => 'Oleo Script:regular,700',
					'Oleo Script Swash Caps' => 'Oleo Script Swash Caps:regular,700',
					'Open Sans' => 'Open Sans:300,300italic,regular,italic,600,600italic,700,700italic,800,800italic',
					'Open Sans Condensed' => 'Open Sans Condensed:300,300italic,700',
					'Oranienbaum' => 'Oranienbaum:regular',
					'Orbitron' => 'Orbitron:regular,500,700,900',
					'Oregano' => 'Oregano:regular,italic',
					'Orienta' => 'Orienta:regular',
					'Original Surfer' => 'Original Surfer:regular',
					'Oswald' => 'Oswald:200,300,regular,500,600,700',
					'Over the Rainbow' => 'Over the Rainbow:regular',
					'Overlock' => 'Overlock:regular,italic,700,700italic,900,900italic',
					'Overlock SC' => 'Overlock SC:regular',
					'Overpass' => 'Overpass:100,100italic,200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Overpass Mono' => 'Overpass Mono:300,regular,600,700',
					'Ovo' => 'Ovo:regular',
					'Oxygen' => 'Oxygen:300,regular,700',
					'Oxygen Mono' => 'Oxygen Mono:regular',
					'PT Mono' => 'PT Mono:regular',
					'PT Sans' => 'PT Sans:regular,italic,700,700italic',
					'PT Sans Caption' => 'PT Sans Caption:regular,700',
					'PT Sans Narrow' => 'PT Sans Narrow:regular,700',
					'PT Serif' => 'PT Serif:regular,italic,700,700italic',
					'PT Serif Caption' => 'PT Serif Caption:regular,italic',
					'Pacifico' => 'Pacifico:regular',
					'Padauk' => 'Padauk:regular,700',
					'Palanquin' => 'Palanquin:100,200,300,regular,500,600,700',
					'Palanquin Dark' => 'Palanquin Dark:regular,500,600,700',
					'Pangolin' => 'Pangolin:regular',
					'Paprika' => 'Paprika:regular',
					'Parisienne' => 'Parisienne:regular',
					'Passero One' => 'Passero One:regular',
					'Passion One' => 'Passion One:regular,700,900',
					'Pathway Gothic One' => 'Pathway Gothic One:regular',
					'Patrick Hand' => 'Patrick Hand:regular',
					'Patrick Hand SC' => 'Patrick Hand SC:regular',
					'Pattaya' => 'Pattaya:regular',
					'Patua One' => 'Patua One:regular',
					'Pavanam' => 'Pavanam:regular',
					'Paytone One' => 'Paytone One:regular',
					'Peddana' => 'Peddana:regular',
					'Peralta' => 'Peralta:regular',
					'Permanent Marker' => 'Permanent Marker:regular',
					'Petit Formal Script' => 'Petit Formal Script:regular',
					'Petrona' => 'Petrona:regular',
					'Philosopher' => 'Philosopher:regular,italic,700,700italic',
					'Piedra' => 'Piedra:regular',
					'Pinyon Script' => 'Pinyon Script:regular',
					'Pirata One' => 'Pirata One:regular',
					'Plaster' => 'Plaster:regular',
					'Play' => 'Play:regular,700',
					'Playball' => 'Playball:regular',
					'Playfair Display' => 'Playfair Display:regular,italic,700,700italic,900,900italic',
					'Playfair Display SC' => 'Playfair Display SC:regular,italic,700,700italic,900,900italic',
					'Podkova' => 'Podkova:regular,500,600,700,800',
					'Poiret One' => 'Poiret One:regular',
					'Poller One' => 'Poller One:regular',
					'Poly' => 'Poly:regular,italic',
					'Pompiere' => 'Pompiere:regular',
					'Pontano Sans' => 'Pontano Sans:regular',
					'Poppins' => 'Poppins:300,regular,500,600,700',
					'Port Lligat Sans' => 'Port Lligat Sans:regular',
					'Port Lligat Slab' => 'Port Lligat Slab:regular',
					'Pragati Narrow' => 'Pragati Narrow:regular,700',
					'Prata' => 'Prata:regular',
					'Preahvihear' => 'Preahvihear:regular',
					'Press Start 2P' => 'Press Start 2P:regular',
					'Pridi' => 'Pridi:200,300,regular,500,600,700',
					'Princess Sofia' => 'Princess Sofia:regular',
					'Prociono' => 'Prociono:regular',
					'Prompt' => 'Prompt:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Prosto One' => 'Prosto One:regular',
					'Proza Libre' => 'Proza Libre:regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Puritan' => 'Puritan:regular,italic,700,700italic',
					'Purple Purse' => 'Purple Purse:regular',
					'Quando' => 'Quando:regular',
					'Quantico' => 'Quantico:regular,italic,700,700italic',
					'Quattrocento' => 'Quattrocento:regular,700',
					'Quattrocento Sans' => 'Quattrocento Sans:regular,italic,700,700italic',
					'Questrial' => 'Questrial:regular',
					'Quicksand' => 'Quicksand:300,regular,500,700',
					'Quintessential' => 'Quintessential:regular',
					'Qwigley' => 'Qwigley:regular',
					'Racing Sans One' => 'Racing Sans One:regular',
					'Radley' => 'Radley:regular,italic',
					'Rajdhani' => 'Rajdhani:300,regular,500,600,700',
					'Rakkas' => 'Rakkas:regular',
					'Raleway' => 'Raleway:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Raleway Dots' => 'Raleway Dots:regular',
					'Ramabhadra' => 'Ramabhadra:regular',
					'Ramaraja' => 'Ramaraja:regular',
					'Rambla' => 'Rambla:regular,italic,700,700italic',
					'Rammetto One' => 'Rammetto One:regular',
					'Ranchers' => 'Ranchers:regular',
					'Rancho' => 'Rancho:regular',
					'Ranga' => 'Ranga:regular,700',
					'Rasa' => 'Rasa:300,regular,500,600,700',
					'Rationale' => 'Rationale:regular',
					'Ravi Prakash' => 'Ravi Prakash:regular',
					'Redressed' => 'Redressed:regular',
					'Reem Kufi' => 'Reem Kufi:regular',
					'Reenie Beanie' => 'Reenie Beanie:regular',
					'Revalia' => 'Revalia:regular',
					'Rhodium Libre' => 'Rhodium Libre:regular',
					'Ribeye' => 'Ribeye:regular',
					'Ribeye Marrow' => 'Ribeye Marrow:regular',
					'Righteous' => 'Righteous:regular',
					'Risque' => 'Risque:regular',
					'Roboto' => 'Roboto:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,900,900italic',
					'Roboto Condensed' => 'Roboto Condensed:300,300italic,regular,italic,700,700italic',
					'Roboto Mono' => 'Roboto Mono:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic',
					'Roboto Slab' => 'Roboto Slab:100,300,regular,700',
					'Rochester' => 'Rochester:regular',
					'Rock Salt' => 'Rock Salt:regular',
					'Rokkitt' => 'Rokkitt:100,200,300,regular,500,600,700,800,900',
					'Romanesco' => 'Romanesco:regular',
					'Ropa Sans' => 'Ropa Sans:regular,italic',
					'Rosario' => 'Rosario:regular,italic,700,700italic',
					'Rosarivo' => 'Rosarivo:regular,italic',
					'Rouge Script' => 'Rouge Script:regular',
					'Rozha One' => 'Rozha One:regular',
					'Rubik' => 'Rubik:300,300italic,regular,italic,500,500italic,700,700italic,900,900italic',
					'Rubik Mono One' => 'Rubik Mono One:regular',
					'Ruda' => 'Ruda:regular,700,900',
					'Rufina' => 'Rufina:regular,700',
					'Ruge Boogie' => 'Ruge Boogie:regular',
					'Ruluko' => 'Ruluko:regular',
					'Rum Raisin' => 'Rum Raisin:regular',
					'Ruslan Display' => 'Ruslan Display:regular',
					'Russo One' => 'Russo One:regular',
					'Ruthie' => 'Ruthie:regular',
					'Rye' => 'Rye:regular',
					'Sacramento' => 'Sacramento:regular',
					'Sahitya' => 'Sahitya:regular,700',
					'Sail' => 'Sail:regular',
					'Salsa' => 'Salsa:regular',
					'Sanchez' => 'Sanchez:regular,italic',
					'Sancreek' => 'Sancreek:regular',
					'Sansita' => 'Sansita:regular,italic,700,700italic,800,800italic,900,900italic',
					'Sarala' => 'Sarala:regular,700',
					'Sarina' => 'Sarina:regular',
					'Sarpanch' => 'Sarpanch:regular,500,600,700,800,900',
					'Satisfy' => 'Satisfy:regular',
					'Scada' => 'Scada:regular,italic,700,700italic',
					'Scheherazade' => 'Scheherazade:regular,700',
					'Schoolbell' => 'Schoolbell:regular',
					'Scope One' => 'Scope One:regular',
					'Seaweed Script' => 'Seaweed Script:regular',
					'Secular One' => 'Secular One:regular',
					'Sevillana' => 'Sevillana:regular',
					'Seymour One' => 'Seymour One:regular',
					'Shadows Into Light' => 'Shadows Into Light:regular',
					'Shadows Into Light Two' => 'Shadows Into Light Two:regular',
					'Shanti' => 'Shanti:regular',
					'Share' => 'Share:regular,italic,700,700italic',
					'Share Tech' => 'Share Tech:regular',
					'Share Tech Mono' => 'Share Tech Mono:regular',
					'Shojumaru' => 'Shojumaru:regular',
					'Short Stack' => 'Short Stack:regular',
					'Shrikhand' => 'Shrikhand:regular',
					'Siemreap' => 'Siemreap:regular',
					'Sigmar One' => 'Sigmar One:regular',
					'Signika' => 'Signika:300,regular,600,700',
					'Signika Negative' => 'Signika Negative:300,regular,600,700',
					'Simonetta' => 'Simonetta:regular,italic,900,900italic',
					'Sintony' => 'Sintony:regular,700',
					'Sirin Stencil' => 'Sirin Stencil:regular',
					'Six Caps' => 'Six Caps:regular',
					'Skranji' => 'Skranji:regular,700',
					'Slabo 13px' => 'Slabo 13px:regular',
					'Slabo 27px' => 'Slabo 27px:regular',
					'Slackey' => 'Slackey:regular',
					'Smokum' => 'Smokum:regular',
					'Smythe' => 'Smythe:regular',
					'Sniglet' => 'Sniglet:regular,800',
					'Snippet' => 'Snippet:regular',
					'Snowburst One' => 'Snowburst One:regular',
					'Sofadi One' => 'Sofadi One:regular',
					'Sofia' => 'Sofia:regular',
					'Sonsie One' => 'Sonsie One:regular',
					'Sorts Mill Goudy' => 'Sorts Mill Goudy:regular,italic',
					'Source Code Pro' => 'Source Code Pro:200,300,regular,500,600,700,900',
					'Source Sans Pro' => 'Source Sans Pro:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900,900italic',
					'Source Serif Pro' => 'Source Serif Pro:regular,600,700',
					'Space Mono' => 'Space Mono:regular,italic,700,700italic',
					'Special Elite' => 'Special Elite:regular',
					'Spectral' => 'Spectral:200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Spicy Rice' => 'Spicy Rice:regular',
					'Spinnaker' => 'Spinnaker:regular',
					'Spirax' => 'Spirax:regular',
					'Squada One' => 'Squada One:regular',
					'Sree Krushnadevaraya' => 'Sree Krushnadevaraya:regular',
					'Sriracha' => 'Sriracha:regular',
					'Stalemate' => 'Stalemate:regular',
					'Stalinist One' => 'Stalinist One:regular',
					'Stardos Stencil' => 'Stardos Stencil:regular,700',
					'Stint Ultra Condensed' => 'Stint Ultra Condensed:regular',
					'Stint Ultra Expanded' => 'Stint Ultra Expanded:regular',
					'Stoke' => 'Stoke:300,regular',
					'Strait' => 'Strait:regular',
					'Sue Ellen Francisco' => 'Sue Ellen Francisco:regular',
					'Suez One' => 'Suez One:regular',
					'Sumana' => 'Sumana:regular,700',
					'Sunshiney' => 'Sunshiney:regular',
					'Supermercado One' => 'Supermercado One:regular',
					'Sura' => 'Sura:regular,700',
					'Suranna' => 'Suranna:regular',
					'Suravaram' => 'Suravaram:regular',
					'Suwannaphum' => 'Suwannaphum:regular',
					'Swanky and Moo Moo' => 'Swanky and Moo Moo:regular',
					'Syncopate' => 'Syncopate:regular,700',
					'Tangerine' => 'Tangerine:regular,700',
					'Taprom' => 'Taprom:regular',
					'Tauri' => 'Tauri:regular',
					'Taviraj' => 'Taviraj:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Teko' => 'Teko:300,regular,500,600,700',
					'Telex' => 'Telex:regular',
					'Tenali Ramakrishna' => 'Tenali Ramakrishna:regular',
					'Tenor Sans' => 'Tenor Sans:regular',
					'Text Me One' => 'Text Me One:regular',
					'The Girl Next Door' => 'The Girl Next Door:regular',
					'Tienne' => 'Tienne:regular,700,900',
					'Tillana' => 'Tillana:regular,500,600,700,800',
					'Timmana' => 'Timmana:regular',
					'Tinos' => 'Tinos:regular,italic,700,700italic',
					'Titan One' => 'Titan One:regular',
					'Titillium Web' => 'Titillium Web:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900',
					'Trade Winds' => 'Trade Winds:regular',
					'Trirong' => 'Trirong:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Trocchi' => 'Trocchi:regular',
					'Trochut' => 'Trochut:regular,italic,700',
					'Trykker' => 'Trykker:regular',
					'Tulpen One' => 'Tulpen One:regular',
					'Ubuntu' => 'Ubuntu:300,300italic,regular,italic,500,500italic,700,700italic',
					'Ubuntu Condensed' => 'Ubuntu Condensed:regular',
					'Ubuntu Mono' => 'Ubuntu Mono:regular,italic,700,700italic',
					'Ultra' => 'Ultra:regular',
					'Uncial Antiqua' => 'Uncial Antiqua:regular',
					'Underdog' => 'Underdog:regular',
					'Unica One' => 'Unica One:regular',
					'UnifrakturCook' => 'UnifrakturCook:700',
					'UnifrakturMaguntia' => 'UnifrakturMaguntia:regular',
					'Unkempt' => 'Unkempt:regular,700',
					
					'Unlock' => 'Unlock:regular',
					'Unna' => 'Unna:regular,italic,700,700italic',
					'VT323' => 'VT323:regular',
					'Vampiro One' => 'Vampiro One:regular',
					'Varela' => 'Varela:regular',
					'Varela Round' => 'Varela Round:regular',
					'Vast Shadow' => 'Vast Shadow:regular',
					'Vesper Libre' => 'Vesper Libre:regular,500,700,900',
					'Vibur' => 'Vibur:regular',
					'Vidaloka' => 'Vidaloka:regular',
					'Viga' => 'Viga:regular',
					'Voces' => 'Voces:regular',
					'Volkhov' => 'Volkhov:regular,italic,700,700italic',
					'Vollkorn' => 'Vollkorn:regular,italic,700,700italic',
					'Voltaire' => 'Voltaire:regular',
					'Waiting for the Sunrise' => 'Waiting for the Sunrise:regular',
					'Wallpoet' => 'Wallpoet:regular',
					'Walter Turncoat' => 'Walter Turncoat:regular',
					'Warnes' => 'Warnes:regular',
					'Wellfleet' => 'Wellfleet:regular',
					'Wendy One' => 'Wendy One:regular',
					'Wire One' => 'Wire One:regular',
					'Work Sans' => 'Work Sans:100,200,300,regular,500,600,700,800,900',
					'Yanone Kaffeesatz' => 'Yanone Kaffeesatz:200,300,regular,700',
					'Yantramanav' => 'Yantramanav:100,300,regular,500,700,900',
					'Yatra One' => 'Yatra One:regular',
					'Yellowtail' => 'Yellowtail:regular',
					'Yeseva One' => 'Yeseva One:regular',
					'Yesteryear' => 'Yesteryear:regular',
					'Yrsa' => 'Yrsa:300,regular,500,600,700',
					'Zeyada' => 'Zeyada:regular',
					'Zilla Slab' => 'Zilla Slab:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic'
				)
			);
}

/**
 * Register the required plugins for this theme.
 */
function newsplus_register_required_plugins() {

	$plugins = array(

        array(
            'name'               => 'NewsPlus Shortcodes',
            'slug'               => 'newsplus-shortcodes',
            'source'             => get_template_directory() . '/plugins/newsplus-shortcodes.zip',
            'required'           => true, // If false, the plugin is only 'recommended' instead of required.
            'version'            => '3.1.4', // E.g. 1.0.0. If set, the active plugin must be this version or higher.
            'force_activation'   => false, // If true, plugin is activated upon theme activation and cannot be deactivated until theme switch.
            'force_deactivation' => false, // If true, plugin is deactivated upon theme switch, useful for theme-specific plugins.
            'external_url'       => '', // If set, overrides default API URL and points to an external URL.
        ),

		// Kingcomposer Page Builder Plugin
		array(
            'name'      => 'King Composer',
            'slug'      => 'kingcomposer',
            'required'  => true,
        ),
		
		// Contact form 7
		array(
            'name'      => 'Contact Form 7',
            'slug'      => 'contact-form-7',
            'required'  => false,
        ),
		
		// WP Review
		array(
            'name'      => 'WP Review',
            'slug'      => 'wp-review',
            'required'  => false,
        ),
		
		// Siteorigin Page Builder Plugin
		array(
            'name'      => 'Post Views Counter',
            'slug'      => 'post-views-counter',
            'required'  => false,
        )
    );

	/*
	 * Array of configuration settings. Amend each line as needed.
	 *
	 * TGMPA will start providing localized text strings soon. If you already have translations of our standard
	 * strings available, please help us make TGMPA even better by giving us access to these translations or by
	 * sending in a pull-request with .po file(s) with the translations.
	 *
	 * Only uncomment the strings in the config array if you want to customize the strings.
	 */
	$config = array(
		'id'           => 'newsplus',             // Unique ID for hashing notices for multiple instances of TGMPA.
		'default_path' => '',                      // Default absolute path to bundled plugins.
		'menu'         => 'tgmpa-install-plugins', // Menu slug.
		'has_notices'  => true,                    // Show admin notices or not.
		'dismissable'  => true,                    // If false, a user cannot dismiss the nag message.
		'dismiss_msg'  => '',                      // If 'dismissable' is false, this message will be output at top of nag.
		'is_automatic' => false,                   // Automatically activate plugins after installation or not.
		'message'      => '',                      // Message to output right before the plugins table.
	);

    tgmpa( $plugins, $config );
}

add_action( 'tgmpa_register', 'newsplus_register_required_plugins' );
?>