<?php
if ( !class_exists( 'MayosisPopularSearchWidget' ) ) {

class MayosisPopularSearchWidget {
    // Constructor
    public function __construct() {
        // Initialize plugin
        add_action( 'init', array( $this, 'mayosisPsearchInit' ) );
        add_action('admin_menu', array($this, 'mayosisPsearchAddAdminMenu'), 999999);
        add_action( 'admin_post_mayosisPsearchDeleteKeyword', array( $this, 'mayosisPsearchDeleteKeyword' ) );
        add_action( 'wp_ajax_mayosisPsearchGetKeywords', array( $this, 'mayosisPsearchGetKeywords' ) );
        add_action( 'admin_post_mayosisPsearchUpdateThreshold', array( $this, 'mayosisPsearchUpdateThreshold' ) );

        // Page load
        add_action( 'template_redirect', array( $this, 'mayosisPsearchTemplateRedirect' ) );
    }
    
    // Plugin initialization
    public function mayosisPsearchInit() {
        // Removed load_plugin_textdomain
    }
    
    // Page load
    public function mayosisPsearchTemplateRedirect() {
        if ( is_search() ) {
            // Store search term
            $query = $this->mayosisPsearchStrtolower( trim( get_search_query() ) );
            
            // Sanitize query to remove special characters and HTML/JavaScript code
            $query = $this->mayosisPsearchSanitizeKeyword($query);
            
            // Check if query is empty after sanitization
            if ( empty($query) ) {
                return;
            }

            $options = get_option( 'mayosis_popular_search_widget' );
            if ( !is_array( $options ) ) {
                $options = $this->mayosisPsearchGetDefaultOptions();
            }
            $max = isset($options['max']) ? $options['max'] : $this->mayosisPsearchGetDefaultOptions()['max'];
            
            $data = get_option( 'mayosis_popular_search_widget_data', array() );
            if ( !is_array( $data ) ) {
                $data = array();
            }

            if ( isset( $data[$query] ) ) {
                $data[$query]++;
            } else {
                $data[$query] = 1;
            }

            arsort($data); // Sort by most searched
            if ( count( $data ) > $max ) {
                $data = array_slice( $data, 0, $max, true );
            }
            
            update_option( 'mayosis_popular_search_widget_data', $data );
        }
    }
    
    // Sanitize the keyword to remove special characters and HTML/JavaScript code
    private function mayosisPsearchSanitizeKeyword( $query ) {
        // Strip all HTML tags
        $query = strip_tags($query);
        // Remove all special characters, leaving only alphanumeric characters and spaces
        $query = preg_replace('/[^a-zA-Z0-9\s]/', '', $query);
        // Replace multiple spaces with a single space
        $query = preg_replace('/\s+/', ' ', $query);
        return trim($query);
    }

    // Add admin menu
    public function mayosisPsearchAddAdminMenu() {
        add_submenu_page(
            'mayosis-admin-menu', 
            __('Searched Data', 'mayosis-core'), 
            __('Searched Data', 'mayosis-core'), 
            'manage_options', 
            'popular-searches', 
            array( $this, 'mayosisPsearchAdminPage' )
        );
    }

    // Admin page
    public function mayosisPsearchAdminPage() {
        $options = get_option( 'mayosis_popular_search_widget' );
        $threshold = isset( $options['threshold'] ) ? $options['threshold'] : 5;
        ?>
        <div class="wrap">
            <div class="mayosis_elementor_importer_header">
                <h1><?php _e('Searched Data', 'mayosis-core'); ?></h1>
            </div>
            
            <div class="msx-top-search-plist-fd">
            
            <input type="text" id="mayosisPsearchSearchInput" placeholder="<?php _e('Search...', 'mayosis-core'); ?>">
            
            <form method="post" action="<?php echo admin_url( 'admin-post.php' ); ?>">
                <input type="hidden" name="action" value="mayosisPsearchUpdateThreshold">
                <label for="mayosisPsearchThreshold"><?php _e('Set Popular Search Threshold:', 'mayosis-core'); ?></label>
                <input type="number" id="mayosisPsearchThreshold" name="threshold" value="<?php echo esc_attr( $threshold ); ?>">
                <input type="submit" value="<?php _e('Update', 'mayosis-core'); ?>" class="button-primary">
            </form>
            </div>
            <div id="mayosisPsearchKeywordsTable">
                <p><?php _e('Please wait for Search Data Initiating', 'mayosis-core'); ?></p>
            </div>
        </div>
        <script type="text/javascript">
            var mayosisPsearch = {
                loadingMessage: '<?php _e('Please wait for Search Data Initiating', 'mayosis-core'); ?>'
            };
        </script>
        
        <?php
    }

    // Delete search keyword
    public function mayosisPsearchDeleteKeyword() {
        if ( isset( $_GET['keyword'] ) && current_user_can( 'manage_options' ) ) {
            $keyword = urldecode( $_GET['keyword'] );
            $data = get_option( 'mayosis_popular_search_widget_data', array() );
            if ( isset( $data[$keyword] ) ) {
                unset( $data[$keyword] );
                update_option( 'mayosis_popular_search_widget_data', $data );
            }
        }
        wp_redirect( admin_url( 'admin.php?page=popular-searches' ) );
        exit;
    }

    // Update threshold
    public function mayosisPsearchUpdateThreshold() {
        if ( isset( $_POST['threshold'] ) && current_user_can( 'manage_options' ) ) {
            $options = get_option( 'mayosis_popular_search_widget' );
            $options['threshold'] = intval( $_POST['threshold'] );
            update_option( 'mayosis_popular_search_widget', $options );
        }
        wp_redirect( admin_url( 'admin.php?page=popular-searches' ) );
        exit;
    }

    // Get keywords for AJAX request
    public function mayosisPsearchGetKeywords() {
        if ( !current_user_can( 'manage_options' ) ) {
            wp_die( __('You do not have sufficient permissions to access this page.', 'mayosis-core') );
        }
        
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        $keywords_per_page = 50;
        $offset = ($page - 1) * $keywords_per_page;

        $options = get_option( 'mayosis_popular_search_widget' );
        $threshold = isset( $options['threshold'] ) ? $options['threshold'] : 5;

        $data = get_option( 'mayosis_popular_search_widget_data', array() );
        if ( !is_array( $data ) ) {
            $data = array();
        }

        $filtered_data = array();
        foreach ($data as $keyword => $count) {
            if ( empty($search) || stripos($keyword, $search) !== false ) {
                $filtered_data[$keyword] = $count;
            }
        }

        $total_keywords = count($filtered_data);
        $total_pages = ceil($total_keywords / $keywords_per_page);

        $keywords_html = $this->mayosisPsearchRenderTable(array_slice($filtered_data, $offset, $keywords_per_page, true), $total_pages, $page, $threshold);

        wp_send_json_success(array(
            'keywords_html' => $keywords_html,
            'total_pages' => $total_pages
        ));
    }

    private function mayosisPsearchRenderTable($data, $total_pages, $current_page, $threshold) {
        ob_start();
        ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php _e('Keyword', 'mayosis-core'); ?></th>
                    <th><?php _e('Count', 'mayosis-core'); ?></th>
                    <th><?php _e('Actions', 'mayosis-core'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($data)) : ?>
                    <tr><td colspan="3"><?php _e('No keywords found', 'mayosis-core'); ?></td></tr>
                <?php else : ?>
                    <?php foreach ($data as $keyword => $count) : ?>
                        <tr>
                            <td>
                                <?php echo esc_html($keyword); ?>
                                <?php if ($count >= $threshold) : ?>
                                    <span class="popular-badge"><?php _e('Popular', 'mayosis-core'); ?></span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html($count); ?></td>
                            <td><a href="<?php echo admin_url('admin-post.php?action=mayosisPsearchDeleteKeyword&keyword=' . urlencode($keyword)); ?>"><?php _e('Delete', 'mayosis-core'); ?></a></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        <?php if ($total_pages > 1) : ?>
            <div class="pagination">
                <?php for ($i = 1; $i <= $total_pages; $i++) : ?>
                    <a href="#" class="page-numbers <?php echo $i == $current_page ? 'current' : ''; ?>" data-page="<?php echo $i; ?>"><?php echo $i; ?></a>
                <?php endfor; ?>
            </div>
        <?php endif; ?>
        <?php
        return ob_get_clean();
    }
    
    // Widgets initialization
    public function mayosisPsearchWidgetsInit() {
        $widget_ops = array(
            'classname' => 'widget_mayosis_popular_search', 
            'description' => __('Shows popular searches', 'mayosis-core'),
        );
        wp_register_sidebar_widget( 'mayosispopularsearchwidget', __('Mayosis Popular Searches', 'mayosis-core'), 
            array( $this, 'mayosisPsearchWidget' ), $widget_ops );
        wp_register_widget_control( 'mayosispopularsearchwidget', __('Mayosis Popular Searches', 'mayosis-core'), 
            array( $this, 'mayosisPsearchWidgetControl' ) );
    }
    
    public function mayosisPsearchWidget( $args ) {
        extract( $args );
        $options = get_option( 'mayosis_popular_search_widget', array() );
        $title = isset( $options['title'] ) ? $options['title'] : '';
        $title = apply_filters( 'widget_title', $title );
        if ( empty($title) )
            $title = '&nbsp;';
        echo $before_widget . $before_title . $title . $after_title, "\n";
        $this->mayosisPsearchShowPopularSearches();
        echo $after_widget;
    }
    
    public function mayosisPsearchShowPopularSearches() {
        $options = get_option( 'mayosis_popular_search_widget' );
        if ( !is_array( $options ) ) {
            $options = $this->mayosisPsearchGetDefaultOptions();
        }

        $data = get_option( 'mayosis_popular_search_widget_data', array() );
        if ( !is_array( $data ) ) {
            $data = array();
        }

        $has_popular_keywords = false;

        if ( count( $data ) > 0 ) {
            echo '<ul> <li>';
            foreach ( $data as $search => $count ) {
                if ( $count < $options['threshold'] ) {
                    continue; // Skip keywords searched less than the threshold
                }

                echo '<a href="', get_search_link( $search ), '"';
                if ( isset($options['nofollow']) && $options['nofollow'] ) {
                    echo ' rel="nofollow"';
                }
                echo '>', esc_html( $search ), '</a>';

                $has_popular_keywords = true;
            }
            echo '</li></ul>';
        }

        if ( !$has_popular_keywords ) {
            echo '<p>' . __('No popular keywords', 'mayosis-core') . '</p>';
        }
    }
    
    public function mayosisPsearchWidgetControl() {
        $options = $newoptions = get_option('mayosis_popular_search_widget', array() );
        if ( count( $options ) == 0 ) {
            $options = $this->mayosisPsearchGetDefaultOptions();
            update_option( 'mayosis_popular_search_widget', $options );
        }
        if ( isset( $_POST['mayosis_psw_submit'] ) ) {
            $options['title'] = strip_tags( stripslashes( $_POST['mayosis_psw_title'] ) );
            $options['max'] = (int)( $_POST['mayosis_psw_max'] );
            $options['nofollow'] = isset( $_POST['mayosis_psw_nofollow'] );
            update_option( 'mayosis_popular_search_widget', $options );
        }
        $title = esc_attr( $options['title'] );
        $max = esc_attr( $options['max'] );
        $nofollow = isset($options['nofollow']) ? $options['nofollow'] : false;
    ?>
    <p><label for="mayosis_psw_title"><?php _e('Title:', 'mayosis-core'); ?> <input class="widefat" id="mayosis_psw_title" name="mayosis_psw_title" type="text" value="<?php echo $title; ?>" /></label></p>
    <p><label for="mayosis_psw_max"><?php _e('Max searches:', 'mayosis-core'); ?> <input id="mayosis_psw_max" name="mayosis_psw_max" type="text" size="3" maxlength="5" value="<?php echo $max; ?>" /></label></p>
    <p><label for="mayosis_psw_nofollow"><?php _e('Add <code>rel="nofollow"</code> to links:', 'mayosis-core'); ?> <input id="mayosis_psw_nofollow" name="mayosis_psw_nofollow" type="checkbox" value="yes" <?php checked( $nofollow, true ); ?>" /></label></p>
    <input type="hidden" id="mayosis_psw_submit" name="mayosis_psw_submit" value="1" />
    <?php
    }
    
    // Make string lowercase
    private function mayosisPsearchStrtolower( $str ) {
        if ( function_exists( 'mb_strtolower' ) ) {
            return mb_strtolower( $str );
        } else {
            return strtolower( $str );
        }
    }
    
    private function mayosisPsearchGetDefaultOptions() {
        return array(
            'title' => '',
            'max' => 15,
            'nofollow' => true,
            'threshold' => 5,
        );
    }
}

// Add functions from WP2.8 for previous WP versions
if ( !function_exists( 'esc_html' ) ) {
    function esc_html( $text ) {
        return esc_html( $text );
    }
}

if ( !function_exists( 'esc_attr' ) ) {
    function esc_attr( $text ) {
        return esc_attr( $text );
    }
}

// Add functions from WP3.0 for previous WP versions
if ( !function_exists( 'get_search_link' ) ) {
    function get_search_link( $query = '' ) {
        global $wp_rewrite;
    
        if ( empty($query) )
            $search = get_search_query();
        else
            $search = stripslashes($query);
    
        $permastruct = $wp_rewrite->get_search_permastruct();
    
        if ( empty( $permastruct ) ) {
            $link = home_url('?s=' . urlencode($search) );
        } else {
            $search = urlencode($search);
            $search = str_replace('%2F', '/', $search); // %2F(/) is not valid within a URL, send it unencoded.
            $link = str_replace( '%search%', $search, $permastruct );
            $link = trailingslashit( get_option( 'home' ) ) . user_trailingslashit( $link, 'search' );
        }
    
        return apply_filters( 'search_link', $link, $search );
    }
}

$wp_mayosis_popular_search_widget = new MayosisPopularSearchWidget();

// Show popular searches anywhere in the theme
function mayosis_show_popular_searches() {
    global $wp_mayosis_popular_search_widget;
    $wp_mayosis_popular_search_widget->mayosisPsearchShowPopularSearches();
}

} // END
