<?php
defined( 'ABSPATH' ) || exit;

require_once __DIR__ . '/includes/lb_helper.php';

class LicenseBox_Manager {

	protected $api;
	protected $verify_data;
	protected $update_data;
	protected $plugin_file;

	public function __construct( $plugin_file ) {
		$this->plugin_file = $plugin_file;
		$this->api         = new LicenseBoxExternalAPI();

		$this->verify_data = $this->api->verify_license();
		$this->update_data = get_transient( 'licensebox_next_update_check' );

		if ( false === $this->update_data ) {
			$this->update_data = $this->api->check_update();
			set_transient( 'licensebox_next_update_check', $this->update_data, 12 * HOUR_IN_SECONDS );
		}

		add_action( 'after_plugin_row_' . plugin_basename( $plugin_file ), [ $this, 'show_update_notice' ], 10, 3 );
		add_action( 'after_plugin_row_' . plugin_basename( $plugin_file ), [ $this, 'show_license_notice' ], 10, 3 );
		add_filter( 'plugin_action_links_' . plugin_basename( $plugin_file ), [ $this, 'add_settings_link' ] );
		add_filter( 'the_content', [ $this, 'append_content_footer' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_css' ] );

		// Load the UI file (can use class methods via $GLOBALS if needed)
		require_once __DIR__ . '/licensebox-wp-options.php';
	}

	public function show_update_notice() {
		if (
			! empty( $this->update_data['status'] )
			&& ! empty( $this->verify_data['status'] )
			&& version_compare( $this->api->get_current_version(), ( $this->update_data['version'] ?? '0.0.0' ), '<' )
		) {
			$update_msg = esc_html( (string) ( $this->update_data['message'] ?? '' ) );
			echo <<<HTML
<tr class="active">
	<td colspan="4">
		<div class="update-message notice inline notice-warning notice-alt" style="margin: 5px 20px 10px 20px">
			<p>
				<b>{$update_msg}</b>
				<a href="options-general.php?page=licensebox-test" style="text-decoration: underline;">Update now</a>.
			</p>
		</div>
	</td>
</tr>
HTML;
		}
	}

	public function show_license_notice() {
		if ( empty( $this->verify_data['status'] ) ) {
			echo <<<HTML
<tr class="active">
	<td colspan="4">
		<div class="notice notice-error inline notice-alt" style="margin: 5px 20px 10px 20px">
			<p>
				<b>License is not set yet, Please enter your license code to use the plugin.</b>
				<a href="options-general.php?page=licensebox-test" style="text-decoration: underline;">Enter License Code</a>.
			</p>
		</div>
	</td>
</tr>
HTML;
		}
	}

	public function add_settings_link( $links ) {
		$links[] = '<a href="options-general.php?page=licensebox-test">' . esc_html__( 'Settings', 'mayosis' ) . '</a>';
		return $links;
	}

	public function append_content_footer( $content ) {
		if ( ! is_admin() ) {
			if ( ! empty( $this->verify_data['status'] ) ) {
				$content .= '<footer><b>Sample plugin to test LicenseBox is active, Thank you for purchasing.</b></footer>';
			} else {
				$content .= '<footer><b>You have not activated the Sample plugin yet or your License is invalid, Please enter a valid license in the plugin options page.</b></footer>';
			}
		}
		return $content;
	}

	public function enqueue_admin_css( $hook_suffix ) {
		if ( isset( $_GET['page'] ) && $_GET['page'] === 'licensebox-test' ) {
			wp_enqueue_style(
				'licensebox-admin-style',
				plugin_dir_url( $this->plugin_file ) . 'assets/css/licensebox-admin.css',
				[],
				'1.0.0'
			);
		}
	}

	// Getter for external access (like licensebox-wp-sample-options.php)
	public function get_api() {
		return $this->api;
	}

	public function get_verify_data() {
		return $this->verify_data;
	}

	public function get_update_data() {
		return $this->update_data;
	}
}

// Initialize the LicenseBox Manager (pass __FILE__ from the plugin entry point)
$GLOBALS['licensebox_manager'] = new LicenseBox_Manager( __FILE__ );
