<?php if(count(get_included_files()) == 1) exit("No direct script access allowed");

require_once(ABSPATH . 'wp-admin/includes/file.php');

define("LB_API_DEBUG", false);

define("LB_TEXT_CONNECTION_FAILED", 'Server is unavailable at the moment, please try again.');
define("LB_TEXT_INVALID_RESPONSE", 'Server returned an invalid response, please contact support.');
define("LB_TEXT_VERIFIED_RESPONSE", 'Verified! Thanks for purchasing.');
define("LB_TEXT_PREPARING_MAIN_DOWNLOAD", 'Preparing to download main update...');
define("LB_TEXT_MAIN_UPDATE_SIZE", 'Main Update size:');
define("LB_TEXT_DONT_REFRESH", '(Please do not refresh the page).');
define("LB_TEXT_DOWNLOADING_MAIN", 'Downloading main update...');
define("LB_TEXT_UPDATE_PERIOD_EXPIRED", 'Your update period has ended or your license is invalid, please contact support.');
define("LB_TEXT_UPDATE_PATH_ERROR", 'Folder does not have write permission or the update file path could not be resolved, please contact support.');
define("LB_TEXT_MAIN_UPDATE_DONE", 'Main update files downloaded and extracted.');
define("LB_TEXT_UPDATE_EXTRACTION_ERROR", 'Update zip extraction failed.');
define("LB_TEXT_PREPARING_SQL_DOWNLOAD", 'Preparing to download SQL update...');
define("LB_TEXT_SQL_UPDATE_SIZE", 'SQL Update size:');
define("LB_TEXT_DOWNLOADING_SQL", 'Downloading SQL update...');
define("LB_TEXT_SQL_UPDATE_DONE", 'SQL update files downloaded.');
define("LB_TEXT_UPDATE_WITH_SQL_DONE", 'Application was successfully updated, please import the downloaded SQL file in your database manually.');
define("LB_TEXT_UPDATE_WITHOUT_SQL_DONE", 'Application was successfully updated, there were no SQL updates.');

class LicenseBoxExternalAPI {

    private $product_id;
    private $api_url;
    private $api_key;
    private $api_language;
    private $current_version;
    private $verify_type;
    private $verification_period;
    private $current_path;
    private $root_path;
    private $license_file;
    private $theme_slug;

    public function __construct() { 
        $this->product_id = 'CE7E27A9';
        $this->api_url = 'https://verify.inhaz.com/';
        $this->api_key = '7ADCCEDC500887876E5A';
        $this->api_language = 'english';
        $this->verify_type = 'envato';
        $this->verification_period = 7;
        $this->current_path = realpath(get_theme_root());
        $this->root_path = realpath(get_theme_root());
        $this->license_file = $this->current_path.'/.lic';
        
        // Get parent theme and version from style.css
        $current_theme = wp_get_theme();
        $this->theme_slug = $current_theme->get_template(); // Always get parent theme slug
        $parent_theme = wp_get_theme($this->theme_slug); // Explicitly get parent theme
        $this->current_version = $parent_theme->get('Version'); // Get version from parent
    }

    public function check_local_license_exist() {
        return is_file($this->license_file);
    }

    public function get_current_version() {
        return $this->current_version;
    }

    private function init_wp_fs() {
        global $wp_filesystem;
        if(false === ($credentials = request_filesystem_credentials(''))) {
            return false;
        }
        if(!WP_Filesystem($credentials)) { 
            request_filesystem_credentials('');
            return false;
        }
        return true;
    }

    private function write_wp_fs($file_path, $content) {
        global $wp_filesystem;
        $save_file_to = $file_path;
        if($this->init_wp_fs()) {    
            if($wp_filesystem->put_contents($save_file_to, $content, FS_CHMOD_FILE)) {
                return true;
            } else {
                return false;
            }
        }
    }

    private function read_wp_fs($file_path) {
        global $wp_filesystem;
        if($this->init_wp_fs()) {    
            return $wp_filesystem->get_contents($file_path);
        }
    }

    private function call_api($method, $url, $data = null) {
        $wp_args = array('body' => $data);    
        $wp_args['method'] = $method;

        $this_url = site_url();
        $this_ip = getenv('SERVER_ADDR')?:
            $this->get_ip_from_third_party()?:
            gethostbyname(gethostname());

        $wp_args['headers'] = array(
            'Content-Type' => 'application/json', 
            'LB-API-KEY' => $this->api_key, 
            'LB-URL' => $this_url, 
            'LB-IP' => $this_ip, 
            'LB-LANG' => $this->api_language
        );
        $wp_args['timeout'] = 30;

        $result = wp_remote_request($url, $wp_args);

        if(is_wp_error($result) && !LB_API_DEBUG) {
            $rs = array(
                'status' => FALSE, 
                'message' => LB_TEXT_CONNECTION_FAILED
            );
            return json_encode($rs);
        }
        $http_status = $result['response']['code'];
        if($http_status != 200) {
            if(LB_API_DEBUG) {
                $temp_decode = json_decode($result['body'], true);
                $rs = array(
                    'status' => FALSE, 
                    'message' => ((!empty($temp_decode['error']))?
                        $temp_decode['error']:
                        $temp_decode['message'])
                );
                return json_encode($rs);
            } else {
                $rs = array(
                    'status' => FALSE, 
                    'message' => LB_TEXT_INVALID_RESPONSE
                );
                return json_encode($rs);
            }
        }
        return $result['body'];
    }

    public function check_connection() {
        $get_data = $this->call_api(
            'POST',
            $this->api_url.'api/check_connection_ext'
        );
        $response = json_decode($get_data, true);
        return $response;
    }

    public function get_latest_version() {
        $data_array = array(
            "product_id" => $this->product_id
        );
        $get_data = $this->call_api(
            'POST',
            $this->api_url.'api/latest_version', 
            json_encode($data_array)
        );
        $response = json_decode($get_data, true);
        return $response;
    }

    public function activate_license($license, $client, $create_lic = true) {
        $data_array = array(
            "product_id" => $this->product_id,
            "license_code" => $license,
            "client_name" => $client,
            "verify_type" => $this->verify_type
        );
        $get_data = $this->call_api(
            'POST',
            $this->api_url.'api/activate_license', 
            json_encode($data_array)
        );
        $response = json_decode($get_data, true);
        if(!empty($create_lic)) {
            if($response['status']) {
                $licfile = trim($response['lic_response']);
                $this->write_wp_fs($this->license_file, $licfile);
            } else {
                if(is_writeable($this->license_file)) {
                    unlink($this->license_file);
                }
            }
        }
        return $response;
    }

    public function verify_license($time_based_check = false, $license = false, $client = false) {
        if(!empty($license) && !empty($client)) {
            $data_array = array(
                "product_id" => $this->product_id,
                "license_file" => null,
                "license_code" => $license,
                "client_name" => $client
            );
        } else {
            if(is_file($this->license_file)) {
                $data_array = array(
                    "product_id" => $this->product_id,
                    "license_file" => $this->read_wp_fs($this->license_file),
                    "license_code" => null,
                    "client_name" => null
                );
            } else {
                $data_array = array();
            }
        } 
        $res = array('status' => TRUE, 'message' => LB_TEXT_VERIFIED_RESPONSE);
        if($time_based_check && $this->verification_period > 0) {
            ob_start();
            if(session_status() == PHP_SESSION_NONE) {
                session_start();
            }
            $type = (int) $this->verification_period;
            $today = date('d-m-Y');
            if(empty($_SESSION["61b3e8f750b8a85"])) {
                $_SESSION["61b3e8f750b8a85"] = '00-00-0000';
            }
            if($type == 1) {
                $type_text = '1 day';
            } elseif($type == 3) {
                $type_text = '3 days';
            } elseif($type == 7) {
                $type_text = '1 week';
            } elseif($type == 30) {
                $type_text = '1 month';
            } elseif($type == 90) {
                $type_text = '3 months';
            } elseif($type == 365) {
                $type_text = '1 year';
            } else {
                $type_text = $type.' days';
            }
            if(strtotime($today) >= strtotime($_SESSION["61b3e8f750b8a85"])) {
                $get_data = $this->call_api(
                    'POST',
                    $this->api_url.'api/verify_license', 
                    json_encode($data_array)
                );
                $res = json_decode($get_data, true);
                if($res['status'] == true) {
                    $tomo = date('d-m-Y', strtotime($today. ' + '.$type_text));
                    $_SESSION["61b3e8f750b8a85"] = $tomo;
                }
            }
            session_write_close();
            ob_end_clean();
        } else {
            $get_data = $this->call_api(
                'POST',
                $this->api_url.'api/verify_license', 
                json_encode($data_array)
            );
            $res = json_decode($get_data, true);
        }
        return $res;
    }

    public function deactivate_license($license = false, $client = false) {
        if(!empty($license) && !empty($client)) {
            $data_array = array(
                "product_id" => $this->product_id,
                "license_file" => null,
                "license_code" => $license,
                "client_name" => $client
            );
        } else {
            if(is_file($this->license_file)) {
                $data_array = array(
                    "product_id" => $this->product_id,
                    "license_file" => $this->read_wp_fs($this->license_file),
                    "license_code" => null,
                    "client_name" => null
                );
            } else {
                $data_array = array();
            }
        }
        $get_data = $this->call_api(
            'POST',
            $this->api_url.'api/deactivate_license', 
            json_encode($data_array)
        );
        $response = json_decode($get_data, true);
        if($response['status']) {
            if(is_writeable($this->license_file)) {
                unlink($this->license_file);
            }
        }
        return $response;
    }

     public function check_update() {
        // Always check parent theme version, even if child theme is active
        $parent_theme = wp_get_theme($this->theme_slug);
        $local_version = $parent_theme->get('Version');
        
        // Check with API
        $data_array = array(
            "product_id" => $this->product_id,
            "current_version" => $local_version
        );
        
        $get_data = $this->call_api(
            'POST',
            $this->api_url.'api/check_update', 
            json_encode($data_array)
        );
        
        $response = json_decode($get_data, true);
        
        // Debug output
        error_log("Parent theme version: {$local_version}");
        error_log("API version: {$response['version']}");
        error_log("API status: {$response['status']}");
        
        // Only return update available if version is actually newer
        if($response['status'] && version_compare($local_version, $response['version'], '<')) {
            return $response;
        }
        
        // Otherwise return no update available
        return array(
            'status' => false,
            'message' => 'Theme is up to date',
            'version' => $local_version
        );
    }
    
    public function download_update($update_id, $type, $version, $license = false, $client = false) {
        global $wp_filesystem;
        
        if (!$this->init_wp_fs()) {
            exit("<br>Failed to initialize filesystem access");
        }

        $data_array = $this->prepare_license_data($license, $client);
        
        ob_end_flush();
        ob_implicit_flush(true);
        ob_start();

        $themes_root = get_theme_root();
        $theme_dir = $themes_root . '/' . $this->theme_slug;
        
        // Get current version from parent theme
        $parent_theme = wp_get_theme($this->theme_slug);
        $current_version = $parent_theme->get('Version');

        echo "Current parent theme version: {$current_version}<br>";
        echo "Preparing to update to version: {$version}<br>";
        ob_flush();

        // Download update package
        $zip_path = $this->download_update_package($update_id, $version, $data_array, $themes_root);
        if (!$zip_path) {
            exit("<br>Failed to download update package");
        }

        // Create backup of parent theme
        $backup_path = $this->create_theme_backup($theme_dir, $version);
        if (!$backup_path) {
            unlink($zip_path);
            exit("<br>Failed to create backup");
        }

        // Apply update to parent theme
        $success = $this->extract_and_replace_theme($zip_path, $themes_root, $this->theme_slug);
        unlink($zip_path);
        
        if (!$success) {
            $this->restore_from_backup($backup_path, $theme_dir, $this->theme_slug);
            exit("<br>Update failed. Theme restored from backup.");
        }

        // Clean up backup
        $wp_filesystem->delete($backup_path, true);

        // Force refresh parent theme data
        wp_clean_theme_cache($this->theme_slug);
        $updated_theme = wp_get_theme($this->theme_slug);
        $new_version = $updated_theme->get('Version');

        // Verify version was updated
        if (version_compare($new_version, $version, '==')) {
            echo "<br>Successfully updated parent theme to version {$new_version}<br>";
            
            // Update current version in memory
            $this->current_version = $new_version;
            
            // Clear all caches
            delete_transient('licensebox_next_update_check');
            wp_clean_themes_cache();
            delete_site_transient('update_themes');
            
            // Handle SQL updates if needed
            if ($type) {
                $this->handle_sql_update($update_id, $version, $data_array);
            } else {
                echo LB_TEXT_UPDATE_WITHOUT_SQL_DONE;
            }
        } else {
            // Version mismatch - restore from backup
            $this->restore_from_backup($backup_path, $theme_dir, $this->theme_slug);
            wp_clean_theme_cache($this->theme_slug);
            $restored_version = wp_get_theme($this->theme_slug)->get('Version');
            
            echo "<br>Update failed - version mismatch<br>";
            echo "Expected version: {$version}<br>";
            echo "Actual version after update: {$new_version}<br>";
            echo "Restored to version: {$restored_version}<br>";
            echo "Please check the update package or contact support.<br>";
        }

        ob_flush();
        ob_end_flush();
        return true;
    }
    private function prepare_license_data($license, $client) {
        if (!empty($license) && !empty($client)) {
            return array(
                "license_file" => null,
                "license_code" => $license,
                "client_name" => $client
            );
        } elseif (is_file($this->license_file)) {
            return array(
                "license_file" => $this->read_wp_fs($this->license_file),
                "license_code" => null,
                "client_name" => null
            );
        }
        return array();
    }

    private function download_update_package($update_id, $version, $data_array, $themes_root) {
        echo LB_TEXT_PREPARING_MAIN_DOWNLOAD."<br>";
        ob_flush();

        $source = $this->api_url."api/download_update/main/".$update_id;
        $wp_args = array(
            'body' => json_encode($data_array),
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'LB-API-KEY' => $this->api_key,
                'LB-URL' => site_url(),
                'LB-IP' => $this->get_server_ip(),
                'LB-LANG' => $this->api_language
            ),
            'timeout' => 30
        );

        echo LB_TEXT_DOWNLOADING_MAIN."<br>";
        ob_flush();

        $response = wp_remote_request($source, $wp_args);
        if (is_wp_error($response)) {
            return false;
        }

        $zip_path = $themes_root . '/theme_update_' . $version . '.zip';
        global $wp_filesystem;
        if (!$wp_filesystem->put_contents($zip_path, $response['body'], FS_CHMOD_FILE)) {
            return false;
        }
        return $zip_path;
    }

    private function create_theme_backup($theme_dir, $version) {
        $backup_dir = WP_CONTENT_DIR . '/theme_backups/' . basename($theme_dir) . '_' . $version . '_' . time();
        
        if (!wp_mkdir_p($backup_dir)) {
            return false;
        }

        $dir_iterator = new RecursiveDirectoryIterator($theme_dir, FilesystemIterator::SKIP_DOTS);
        $iterator = new RecursiveIteratorIterator($dir_iterator, RecursiveIteratorIterator::SELF_FIRST);
        
        global $wp_filesystem;
        
        foreach ($iterator as $item) {
            $target = $backup_dir . '/' . $iterator->getSubPathName();
            if ($item->isDir()) {
                wp_mkdir_p($target);
            } else {
                if (!$wp_filesystem->copy($item, $target)) {
                    return false;
                }
            }
        }
        
        return $backup_dir;
    }

    private function extract_and_replace_theme($zip_path, $themes_root, $theme_slug) {
        $zip = new ZipArchive;
        if ($zip->open($zip_path) !== true) {
            return false;
        }

        // Create temporary extraction directory
        $temp_dir = $themes_root . '/temp_' . $theme_slug . '_' . time();
        if (!wp_mkdir_p($temp_dir)) {
            $zip->close();
            return false;
        }

        // Extract to temporary directory
        if (!$zip->extractTo($temp_dir)) {
            $zip->close();
            $this->rrmdir($temp_dir);
            return false;
        }
        $zip->close();

        // Find the extracted theme folder
        $extracted_theme = $this->find_extracted_theme($temp_dir, $theme_slug);
        if (!$extracted_theme) {
            $this->rrmdir($temp_dir);
            return false;
        }

        // Remove old theme
        $old_theme_path = $themes_root . '/' . $theme_slug;
        if (is_dir($old_theme_path)) {
            $this->rrmdir($old_theme_path);
        }

        // Rename extracted theme to correct name
        if (!rename($extracted_theme, $old_theme_path)) {
            $this->rrmdir($temp_dir);
            return false;
        }

        // Clean up
        $this->rrmdir($temp_dir);
        return true;
    }

    private function find_extracted_theme($temp_dir, $theme_slug) {
        $dir_iterator = new RecursiveDirectoryIterator($temp_dir, FilesystemIterator::SKIP_DOTS);
        $iterator = new RecursiveIteratorIterator($dir_iterator, RecursiveIteratorIterator::SELF_FIRST);
        
        foreach ($iterator as $item) {
            if ($item->isDir() && basename($item) === $theme_slug) {
                return $item->getPathname();
            }
        }
        
        // If not found by name, get the first directory
        $dirs = glob($temp_dir . '/*', GLOB_ONLYDIR);
        return count($dirs) === 1 ? $dirs[0] : false;
    }

    private function restore_from_backup($backup_path, $theme_dir, $theme_slug) {
        // First remove current theme files if they exist
        if (is_dir($theme_dir)) {
            $this->rrmdir($theme_dir);
        }

        // Find the actual theme folder in the backup
        $backup_theme = $this->find_extracted_theme($backup_path, $theme_slug);
        if (!$backup_theme) {
            return false;
        }

        // Restore the theme
        return rename($backup_theme, $theme_dir);
    }

    private function handle_sql_update($update_id, $version, $data_array) {
        echo LB_TEXT_PREPARING_SQL_DOWNLOAD."<br>";
        ob_flush();
        
        $source = $this->api_url."api/download_update/sql/".$update_id;
        $wp_args = array(
            'body' => json_encode($data_array),
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'LB-API-KEY' => $this->api_key,
                'LB-URL' => site_url(),
                'LB-IP' => $this->get_server_ip(),
                'LB-LANG' => $this->api_language
            ),
            'timeout' => 30
        );

        echo LB_TEXT_DOWNLOADING_SQL."<br>";
        ob_flush();

        $response = wp_remote_request($source, $wp_args);
        if (is_wp_error($response)) {
            exit(LB_TEXT_CONNECTION_FAILED);
        }

        $sql_path = WP_CONTENT_DIR . '/update_sql_' . $version . '.sql';
        global $wp_filesystem;
        if (!$wp_filesystem->put_contents($sql_path, $response['body'], FS_CHMOD_FILE)) {
            exit(LB_TEXT_UPDATE_PATH_ERROR);
        }

        echo LB_TEXT_SQL_UPDATE_DONE."<br><br>";
        echo LB_TEXT_UPDATE_WITH_SQL_DONE;
        ob_flush();
    }

    private function get_server_ip() {
        return getenv('SERVER_ADDR') ?: 
               $this->get_ip_from_third_party() ?: 
               gethostbyname(gethostname());
    }

    private function get_ip_from_third_party() {
        $response = wp_remote_get('http://ipecho.net/plain', array('timeout' => 30));
        return !is_wp_error($response) ? $response['body'] : false;
    }

    private function rrmdir($dir) {
        if (!is_dir($dir)) return;
        
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        
        global $wp_filesystem;
        
        foreach ($files as $file) {
            if ($file->isDir()) {
                $wp_filesystem->rmdir($file->getRealPath());
            } else {
                $wp_filesystem->delete($file->getRealPath());
            }
        }
        
        $wp_filesystem->rmdir($dir);
    }
}