(function($){
    "use strict";

    // Optimized debounce function
    function debounce(func, wait, immediate) {
        let timeout;
        return function() {
            const context = this, args = arguments;
            const later = function() {
                timeout = null;
                if (!immediate) func.apply(context, args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func.apply(context, args);
        };
    }

    function productSearch(form) {
        const search = form.find('.search');
        const category = form.find('.category');
        const resultBox = form.next('.search-results');
        const query = search.val().trim();

        // Don't clear results if query is short - just don't make AJAX call
        if (query.length < 2) { // Reduced from 3 to 2 for better UX
            if (query.length === 0) {
                resultBox.removeClass('active').addClass('empty').html('');
            }
            return;
        }

        // Show loading indicator immediately
        form.find('.search-wrapper').addClass('wcloading');
        resultBox.addClass('searching').html('<div class="mayosis-ajax-searching">Searching...</div>');

        $.ajax({
            url: opt.ajaxUrl,
            type: 'POST',
            data: {
                action: 'search_product',
                keyword: query,
                category: category.val()
            },
            success: function(data) {
                form.find('.search-wrapper').removeClass('wcloading');
                resultBox.removeClass('searching');
                
                if (data && data.length) {
                    resultBox
                        .removeClass('empty')
                        .addClass('active')
                        .html('<div>' + data + '</div>');
                    $(".mayosis-ajax-s-offcanvas").addClass("offcanvas-full-height");
                } else {
                    resultBox
                        .removeClass('empty')
                        .addClass('active')
                        .html('<div class="mayosis-ajax-not-found">' + opt.noResults + '</div>');
                }
            },
            error: function(xhr, status, error) {
                form.find('.search-wrapper').removeClass('wcloading');
                resultBox.removeClass('searching')
                    .addClass('active')
                    .html('<div class="mayosis-ajax-error">Search failed. Please try again.</div>');
                console.error("Search error:", status, error);
            },
            timeout: 5000 // Set timeout to 5 seconds
        });
    }

    $(document).ready(function() {
        $('form[name="product-search"]').each(function() {
            const form = $(this);
            const search = form.find('.search');
            const category = form.find('.category');
            
            // Use optimized debounce with 300ms delay
            const debouncedSearch = debounce(function() {
                productSearch(form);
            }, 300, false);

            // Handle category changes
            category.on('change', debouncedSearch);
            
            // Handle search input
            search.on('input', function() {
                // Show loading immediately for better UX when we have enough characters
                if ($(this).val().trim().length >= 2) {
                    form.find('.search-wrapper').addClass('wcloading');
                }
                debouncedSearch();
            });

            // Close results when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.product-search').length) {
                    form.next('.search-results').removeClass('active').html('');
                }
            });

            // Keyboard navigation
            search.on('keydown', function(e) {
                if (e.keyCode === 27) { // ESC key
                    form.next('.search-results').removeClass('active').html('');
                }
            });
        });

        // Handle clicks on search results to prevent form submission
        $(document).on('click', '.mayosis-ajax-grid-search a, .mayosis-ajax-list-search a', function(e) {
            e.preventDefault();
            window.location.href = $(this).attr('href');
        });
    });

})(jQuery);