<?php
/**
 * One Click Demo Import Files for Maxreach Theme
 *
 * @package Maxreach
 * @since 1.0.0
 * @version 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get demo import files for One Click Demo Import plugin
 *
 * @since 1.0.0
 * @return array Array of import files or empty array if none available
 */
function keystone_ocdi_import_files() {
	// Get the current theme's stylesheet slug
	$theme_slug = wp_get_theme()->get_stylesheet();
	
	// Build the API URL for the specific theme
	$api_url_base = defined( 'MAXREACH_API_URL_BASE' ) ? MAXREACH_API_URL_BASE : BUCKET_API_URL . '/wp-content/uploads/bucket-manager/';
	$api_url = trailingslashit( $api_url_base ) . $theme_slug . '/demo-data.json';
	
	// Sanitize the URL
	$api_url = esc_url_raw( $api_url );
	
	// Execute the API request with proper error handling
	$response = wp_remote_get(
		$api_url,
		array(
			'headers' => array(
				'User-Agent' => 'MaxreachTheme/' . KEYSTONE_THEME_VERSION . ' (WordPress/' . get_bloginfo( 'version' ) . '; ' . home_url() . ')',
				'Accept' => 'application/json',
				'X-Requested-With' => 'XMLHttpRequest',
			),
			'timeout' => 30,
			'sslverify' => true,
			'httpversion' => '1.1',
		)
	);
	
	// Check for WordPress HTTP API errors
	if ( is_wp_error( $response ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'Maxreach: Demo import failed for theme %s: %s', esc_html( $theme_slug ), esc_html( $response->get_error_message() ) ) );
		}
		return array();
	}
	
	// Check HTTP response code
	$response_code = wp_remote_retrieve_response_code( $response );
	if ( 200 !== $response_code ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'Maxreach: Demo import HTTP %d for theme %s', $response_code, esc_html( $theme_slug ) ) );
		}
		return array();
	}
	
	// Retrieve and validate the response body
	$json_content = wp_remote_retrieve_body( $response );
	if ( empty( $json_content ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'Maxreach: Demo import empty response for theme %s', esc_html( $theme_slug ) ) );
		}
		return array();
	}
	
	// Decode the JSON response with error checking
	$import_files = json_decode( $json_content, true );
	
	// Check for JSON decode errors
	if ( json_last_error() !== JSON_ERROR_NONE ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'Maxreach: Demo import JSON decode error for theme %s: %s', esc_html( $theme_slug ), json_last_error_msg() ) );
		}
		return array();
	}
	
	// Validate the decoded data structure
	if ( ! is_array( $import_files ) || empty( $import_files ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'Maxreach: Demo import invalid data structure for theme %s', esc_html( $theme_slug ) ) );
		}
		return array();
	}
	
	// Sanitize and validate each import file entry
	$sanitized_import_files = array();
	foreach ( $import_files as $import_file ) {
		if ( is_array( $import_file ) && isset( $import_file['import_file_name'] ) ) {
			// Sanitize the import file data
			$sanitized_file = array(
				'import_file_name' => sanitize_text_field( $import_file['import_file_name'] ),
				'import_file_url' => esc_url_raw( isset( $import_file['import_file_url'] ) ? $import_file['import_file_url'] : '' ),
				'import_widget_file_url' => esc_url_raw( isset( $import_file['import_widget_file_url'] ) ? $import_file['import_widget_file_url'] : '' ),
				'import_customizer_file_url' => esc_url_raw( isset( $import_file['import_customizer_file_url'] ) ? $import_file['import_customizer_file_url'] : '' ),
				'import_redux' => isset( $import_file['import_redux'] ) && is_array( $import_file['import_redux'] ) ? $import_file['import_redux'] : array(),
				'preview_url' => esc_url_raw( isset( $import_file['preview_url'] ) ? $import_file['preview_url'] : '' ),
				'import_preview_image_url' => esc_url_raw( isset( $import_file['import_preview_image_url'] ) ? $import_file['import_preview_image_url'] : '' ),
			);
			
			// Only add if we have at least a name and import file URL
			if ( ! empty( $sanitized_file['import_file_name'] ) && ! empty( $sanitized_file['import_file_url'] ) ) {
				$sanitized_import_files[] = $sanitized_file;
			}
		}
	}
	
	// Return sanitized data or empty array
	if ( ! empty( $sanitized_import_files ) ) {
		return $sanitized_import_files;
	} else {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'Maxreach: Demo import no valid files found for theme %s', esc_html( $theme_slug ) ) );
		}
		return array();
	}
}


