<?php

defined( 'ABSPATH' ) || exit;

use WGL_Extensions\Includes\WGL_Elementor_Helper;
use WGL_Extensions\WGL_Framework_Global_Variables;

/**
 * Dynamic Styles
 *
 *
 * @package lunfest\core\class
 * @author WebGeniusLab <webgeniuslab@gmail.com>
 * @since 1.0.0
 */
class WGL_Framework_Dynamic_Styles
{
    protected static $instance;

    private $template_directory_uri;
    private $use_minified;
    private $enqueued_stylesheets = [];
    private $header_page_id;

    public function __construct()
    {
        // do nothing.
    }

    public static function instance()
    {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    public function construct()
    {
        $this->template_directory_uri = get_template_directory_uri();
        $this->use_minified = WGL_Framework::get_option('use_minified') ? '.min' : '';
        $this->header_building_tool = WGL_Framework::get_option('header_building_tool');
        $this->gradient_enabled = WGL_Framework::get_mb_option('use-gradient', 'mb_page_colors_switch', 'custom');

        $this->enqueue_styles_and_scripts();
        $this->add_body_classes();
    }

    public function enqueue_styles_and_scripts()
    {
        add_action( 'wp_enqueue_scripts', [ $this, 'frontend_stylesheets' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'frontend_scripts' ] );

        //* Elementor Compatibility
        add_action( 'wp_enqueue_scripts', [ $this, 'get_elementor_css_theme_builder' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'elementor_column_fix' ] );

        add_action( 'admin_enqueue_scripts', [ $this, 'admin_stylesheets' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_scripts' ] );
    }

    public function get_elementor_css_theme_builder()
    {
        $current_post_id = get_the_ID();
        $css_files = [];

        $locations[] = $this->get_elementor_css_cache_header();
        $locations[] = $this->get_elementor_css_cache_header_sticky();
        $locations[] = $this->get_elementor_css_cache_footer();
        $locations[] = $this->get_elementor_css_cache_side_panel();

        foreach ($locations as $location) {
            //* Don't enqueue current post here (let the preview/frontend components to handle it)
            if ($location && $current_post_id !== $location) {
                $css_file = new \Elementor\Core\Files\CSS\Post($location);
                $css_files[] = $css_file;
            }
        }

        if (!empty($css_files)) {
            \Elementor\Plugin::$instance->frontend->enqueue_styles();
            foreach ($css_files as $css_file) {
                $css_file->enqueue();
            }
        }
    }

    public function get_elementor_css_cache_header()
    {
        if (
            ! apply_filters( 'wgl/header/enable', true )
            || ! class_exists( '\Elementor\Core\Files\CSS\Post' )
        ) {
            // Bailtout.
            return;
        }

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta( 'mb_customize_header_layout' )
            && 'default' !== rwmb_meta( 'mb_header_content_type' )
        ) {
            $this->header_building_tool = 'elementor';
            $this->header_page_id = rwmb_meta( 'mb_customize_header' );
        } else {
            $this->header_page_id = WGL_Framework::get_option( 'header_page_select' );
        }

        if ( 'elementor' === $this->header_building_tool ) {
            return $this->multi_language_support( $this->header_page_id, 'header' );
        }
    }

    public function get_elementor_css_cache_header_sticky()
    {
        if (
            ! apply_filters( 'wgl/header/enable', true )
            || 'elementor' !== $this->header_building_tool
            || ! class_exists( '\Elementor\Core\Files\CSS\Post' )
        ) {
            // Bailtout.
            return;
        }

        $header_sticky_page_id = '';

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta( 'mb_customize_header_layout' )
            && 'default' !== rwmb_meta( 'mb_sticky_header_content_type' )
        ) {
            $header_sticky_page_id = rwmb_meta( 'mb_customize_sticky_header' );
        } elseif ( WGL_Framework::get_option( 'header_sticky' ) ) {
            $header_sticky_page_id = WGL_Framework::get_option( 'header_sticky_page_select' );
        }

        return $this->multi_language_support( $header_sticky_page_id, 'header' );
    }

    public function get_elementor_css_cache_footer()
    {
        $footer = apply_filters( 'wgl/footer/enable', true );
        $footer_switch = $footer[ 'footer_switch' ] ?? '';

        if (
            ! $footer_switch
            || 'elementor' !== WGL_Framework::get_mb_option( 'footer_building_tool', 'mb_footer_switch', 'on' )
            || ! class_exists( '\Elementor\Core\Files\CSS\Post' )
        ) {
            // Bailout.
            return;
        }

        $footer_page_id = WGL_Framework::get_mb_option( 'footer_page_select', 'mb_footer_switch', 'on' );

        return $this->multi_language_support( $footer_page_id, 'footer' );
    }

    public function get_elementor_css_cache_side_panel()
    {
        if (
            !WGL_Framework::get_option('side_panel_enabled')
            || 'elementor' !== WGL_Framework::get_mb_option('side_panel_building_tool', 'mb_customize_side_panel', 'custom')
            || !class_exists('\Elementor\Core\Files\CSS\Post')
        ) {
            // Bailout.
            return;
        }

        $sp_page_id = WGL_Framework::get_mb_option('side_panel_page_select', 'mb_customize_side_panel', 'custom');

        return $this->multi_language_support($sp_page_id, 'side_panel');
    }

    public function multi_language_support($page_id, $page_type)
    {
        if (!$page_id) {
            // Bailout.
            return;
        }

        $page_id = intval($page_id);

        if (class_exists('Polylang') && function_exists('pll_current_language')) {
            $currentLanguage = pll_current_language();
            $translations = PLL()->model->post->get_translations($page_id);

            $polylang_id = $translations[$currentLanguage] ?? '';
            $page_id = $polylang_id ?: $page_id;
        }

        if (class_exists('SitePress')) {
            $wpml_id = wpml_object_id_filter($page_id, $page_type, false, ICL_LANGUAGE_CODE);
            if (
                $wpml_id
                && 'trash' !== get_post_status($wpml_id)
            ) {
                $page_id = $wpml_id;
            }
        }

        return $page_id;
    }

    public function elementor_column_fix()
    {
        $css = '.elementor-container > .elementor-row > .elementor-column > .elementor-element-populated,'
            . '.elementor-container > .elementor-column > .elementor-element-populated {'
                . 'padding-top: 0;'
                . 'padding-bottom: 0;'
            . '}';

        $css .= '.elementor-column-gap-default > .elementor-row > .elementor-column > .elementor-element-populated,'
            . '.elementor-column-gap-default > .elementor-column > .theiaStickySidebar > .elementor-element-populated,'
            . '.elementor-column-gap-default > .elementor-column > .elementor-element-populated {'
                . 'padding-left: 15px;'
                . 'padding-right: 15px;'
            . '}';

        wp_add_inline_style('elementor-frontend', $css);
    }

    public function frontend_stylesheets()
    {
        wp_enqueue_style(
            WGL_Framework_Global_Variables::get_theme_slug() . '-theme-info',
            get_bloginfo('stylesheet_url'),
            [],
            WGL_Framework_Global_Variables::get_theme_version()
        );

        $this->enqueue_css_variables();
        $this->enqueue_additional_styles();
        $this->enqueue_theme_stylesheet( 'main', '/css/' );
        $this->enqueue_pluggable_styles();
        $this->enqueue_theme_stylesheet( 'responsive', '/css/', $this->enqueued_stylesheets );
        $this->enqueue_theme_stylesheet( 'dynamic', '/css/', $this->enqueued_stylesheets );
    }

    public function enqueue_css_variables()
    {
        return wp_add_inline_style(
            WGL_Framework_Global_Variables::get_theme_slug() . '-theme-info',
            $this->retrieve_css_variables_and_extra_styles()
        );
    }

    public function enqueue_additional_styles()
    {
        wp_enqueue_style(
            'font-awesome-5-all',
            $this->template_directory_uri . '/css/font-awesome-5.min.css',
            [],
            WGL_Framework_Global_Variables::get_theme_version()
        );

        wp_enqueue_style(
            WGL_Framework_Global_Variables::get_theme_slug() . '-flaticon',
            $this->template_directory_uri . '/fonts/flaticon/flaticon.css',
            [],
            WGL_Framework_Global_Variables::get_theme_version()
        );
    }

    public function retrieve_css_variables_and_extra_styles()
    {
        $root_vars = $extra_css = '';

        /**
         * Color Variables
         */
        if (
            class_exists('RWMB_Loader')
            && 'custom' === WGL_Framework::get_mb_option('page_colors_switch')
        ) {
            $theme_primary_color = WGL_Framework::get_mb_option('theme-primary-color');
            $theme_secondary_color = WGL_Framework::get_mb_option('theme-secondary-color');
            $theme_tertiary_color = WGL_Framework::get_mb_option('theme-tertiary-color');

            $bg_body = WGL_Framework::get_mb_option( 'body-background-color' );

            $main_font_color = WGL_Framework::get_mb_option( 'theme-content-color' );
            $h_font_color = WGL_Framework::get_mb_option( 'theme-headings-color' );

            $button_color_idle = WGL_Framework::get_mb_option( 'button-color-idle' );
            $button_bg_idle = WGL_Framework::get_mb_option( 'button-bg-idle' );
            $button_color_hover = WGL_Framework::get_mb_option( 'button-color-hover' );
            $button_bg_hover = WGL_Framework::get_mb_option( 'button-bg-hover' );

            $cursor_point_color = WGL_Framework::get_mb_option('cursor_color');

            $scroll_up_arrow_color = WGL_Framework::get_mb_option('scroll_up_arrow_color');
            $scroll_up_arrow_color_bg = WGL_Framework::get_mb_option('scroll_up_arrow_color_bg');

            $this->gradient_enabled && $theme_gradient_from = WGL_Framework::get_mb_option('theme-gradient-from');
            $this->gradient_enabled && $theme_gradient_to = WGL_Framework::get_mb_option('theme-gradient-to');
        } else {
            $theme_primary_color = WGL_Framework_Global_Variables::get_primary_color();
            $theme_secondary_color = WGL_Framework_Global_Variables::get_secondary_color();
            $theme_tertiary_color = WGL_Framework_Global_Variables::get_tertiary_color();

            $main_font_color = WGL_Framework_Global_Variables::get_main_font_color();
            $h_font_color = WGL_Framework_Global_Variables::get_h_font_color();

            $button_color_idle = WGL_Framework_Global_Variables::get_btn_color_idle();
            $button_bg_idle = WGL_Framework_Global_Variables::get_btn_bg_idle();
            $button_color_hover = WGL_Framework_Global_Variables::get_btn_color_hover();
            $button_bg_hover = WGL_Framework_Global_Variables::get_btn_bg_hover();

            $bg_body = WGL_Framework_Global_Variables::get_bg_body_color();

            $cursor_point_color = WGL_Framework_Global_Variables::get_cursor_point_color();

            $scroll_up_arrow_color = WGL_Framework::get_option('scroll_up_arrow_color');
            $scroll_up_arrow_color_bg = WGL_Framework::get_option('scroll_up_arrow_color_bg');

	        $this->gradient_enabled && $theme_gradient = WGL_Framework::get_option('theme-gradient');
        }

	    $root_vars .= '--lunfest-primary-color: ' . esc_attr( $theme_primary_color ?: 'unset' ) . ';';
	    $root_vars .= '--lunfest-secondary-color: ' . esc_attr( $theme_secondary_color ?: 'unset' ) . ';';
	    $root_vars .= '--lunfest-tertiary-color: ' . esc_attr( $theme_tertiary_color ?: 'unset' ) . ';';

	    $root_vars .= '--lunfest-button-color-idle: ' . esc_attr( $button_color_idle ?: 'unset' ) . ';';
	    $root_vars .= '--lunfest-button-bg-idle: ' . esc_attr( $button_bg_idle ?: 'unset' ) . ';';
	    $root_vars .= '--lunfest-button-color-hover: ' . esc_attr( $button_color_hover ?: 'unset' ) . ';';
	    $root_vars .= '--lunfest-button-bg-hover: ' . esc_attr( $button_bg_hover ?: 'unset' ) . ';';

	    $root_vars .= '--lunfest-cursor-point-color: ' . ( $cursor_point_color ? esc_attr($cursor_point_color) : 'unset' ) . ';';

	    $root_vars .= '--lunfest-back-to-top-color: ' . ( $scroll_up_arrow_color ? esc_attr($scroll_up_arrow_color) : 'unset' ) . ';';
	    $root_vars .= '--lunfest-back-to-top-color-bg: ' . ( $scroll_up_arrow_color_bg ? esc_attr($scroll_up_arrow_color_bg) : 'unset' ) . ';';

	    $root_vars .= '--lunfest-body-background: ' . ( $bg_body ? esc_attr($bg_body) : 'unset' ) . ';';
        $root_vars .= '--lunfest-body-rgb-background: ' . esc_attr( WGL_Framework::hex_to_rgb( $bg_body ) ) . ';';

        $root_vars .= '--lunfest-primary-rgb: ' . ( $theme_primary_color ? esc_attr(WGL_Framework::hex_to_rgb($theme_primary_color)) : 'unset' ) . ';';
        $root_vars .= '--lunfest-secondary-rgb: ' . ( $theme_secondary_color ? esc_attr(WGL_Framework::hex_to_rgb($theme_secondary_color)) : 'unset' ) . ';';
        $root_vars .= '--lunfest-tertiary-rgb: ' . ( $theme_tertiary_color ? esc_attr(WGL_Framework::hex_to_rgb($theme_tertiary_color)) : 'unset' ) . ';';
        $root_vars .= '--lunfest-content-rgb: ' . ( $main_font_color ? esc_attr(WGL_Framework::hex_to_rgb($main_font_color)) : 'unset' ) . ';';
        $root_vars .= '--lunfest-header-rgb: ' . ( $h_font_color ? esc_attr(WGL_Framework::hex_to_rgb($h_font_color)) : 'unset' ) . ';';

	    $shop_products_overlay = WGL_Framework::get_option('shop_products_overlay')['rgba'] ?? 'transparent';
	    $root_vars .= '--lunfest-shop-products-overlay: ' . ( !empty($shop_products_overlay) ? esc_attr($shop_products_overlay) : 'transparent' ) . ';';
	    //* ↑ color variables

        /**
         * Headings Variables
         */
        $header_font = WGL_Framework::get_option( 'header-font' );
        $root_vars .= '--lunfest-header-font-family: ' . ( $header_font['font-family'] ? esc_attr($header_font['font-family']) : 'unset' ) . ';';
        $root_vars .= '--lunfest-header-font-weight: ' . ( $header_font['font-weight'] ? esc_attr($header_font['font-weight']) : 'unset' ) . ';';
        $root_vars .= '--lunfest-header-font-color: ' . ( $h_font_color ? esc_attr($h_font_color) : 'unset' ) . ';';

        for ($i = 1; $i <= 6; $i++) {
            ${'header-h' . $i} = WGL_Framework::get_option('header-h' . $i);

            $root_vars .= '--lunfest-h' . $i . '-font-family: ' . (${'header-h' . $i}['font-family'] ? esc_attr(${'header-h' . $i}['font-family']) : 'unset') . ';';
            $root_vars .= '--lunfest-h' . $i . '-font-size: ' . (${'header-h' . $i}['font-size'] ? esc_attr(${'header-h' . $i}['font-size']) : 'unset') . ';';
            $root_vars .= '--lunfest-h' . $i . '-line-height: ' . (${'header-h' . $i}['line-height'] ? esc_attr(${'header-h' . $i}['line-height']) : 'unset') . ';';
            $root_vars .= '--lunfest-h' . $i . '-font-weight: ' . (${'header-h' . $i}['font-weight'] ? esc_attr(${'header-h' . $i}['font-weight']) : 'unset') . ';';
            $root_vars .= '--lunfest-h' . $i . '-text-transform: ' . (${'header-h' . $i}['text-transform'] ? esc_attr(${'header-h' . $i}['text-transform']) : 'unset') . ';';
            $root_vars .= '--lunfest-h' . $i . '-letter-spacing: ' . (${'header-h' . $i}['letter-spacing'] ? esc_attr(${'header-h' . $i}['letter-spacing']) : 'normal') . ';';
        }
        //* ↑ headings variables

        /**
         * Content Variables
         */
        $main_font = WGL_Framework::get_option( 'main-font' );
        $content_font_size = $main_font['font-size'] ?? '';
        $content_line_height = $main_font['line-height'] ?? '';
        $content_line_height = $content_line_height ? round(((int) $content_line_height / (int) $content_font_size), 3) : '';

	    $root_vars .= '--lunfest-content-font-family: ' . ( $main_font['font-family'] ? esc_attr($main_font['font-family']) : 'unset') . ';';
	    $root_vars .= '--lunfest-content-font-size: ' . ( $content_font_size ? esc_attr($content_font_size) : 'unset') . ';';
	    $root_vars .= '--lunfest-content-line-height: ' . ( $content_line_height ? esc_attr($content_line_height) : 'unset') . ';';
	    $root_vars .= '--lunfest-content-font-weight: ' . ( $main_font['font-weight'] ? esc_attr($main_font['font-weight']) : 'unset') . ';';
	    $root_vars .= '--lunfest-content-color: ' . ( $main_font_color ? esc_attr($main_font_color) : 'unset') . ';';
        //* ↑ content variables

        /**
         * Menu Variables
         */
        $menu_font = WGL_Framework::get_option( 'menu-font' );
	    $root_vars .= '--lunfest-menu-font-family: ' . ( $menu_font['font-family'] ? esc_attr($menu_font['font-family']) : 'unset') . ';';
	    $root_vars .= '--lunfest-menu-font-size: ' . ( $menu_font['font-size'] ? esc_attr($menu_font['font-size']) : 'unset') . ';';
	    $root_vars .= '--lunfest-menu-line-height: ' . ( $menu_font['line-height'] ? esc_attr($menu_font['line-height']) : 'unset') . ';';
	    $root_vars .= '--lunfest-menu-font-weight: ' . ( $menu_font['font-weight'] ? esc_attr($menu_font['font-weight']) : 'unset') . ';';
	    $root_vars .= '--lunfest-menu-letter-spacing: ' . ( $menu_font['letter-spacing'] ? esc_attr(floatval($menu_font['letter-spacing'])) . 'em' : 'normal') . ';';
        //* ↑ menu variables

        /**
         * Submenu Variables
         */
        $sub_menu_font = WGL_Framework::get_option('sub-menu-font');
	    $root_vars .= '--lunfest-submenu-font-family: ' . ( $sub_menu_font['font-family'] ? esc_attr($sub_menu_font['font-family']) : 'unset') . ';';
	    $root_vars .= '--lunfest-submenu-font-size: ' . ( $sub_menu_font['font-size'] ? esc_attr($sub_menu_font['font-size']) : 'unset') . ';';
	    $root_vars .= '--lunfest-submenu-line-height: ' . ( $sub_menu_font['line-height'] ? esc_attr($sub_menu_font['line-height']) : 'unset') . ';';
	    $root_vars .= '--lunfest-submenu-font-weight: ' . ( $sub_menu_font['font-weight'] ? esc_attr($sub_menu_font['font-weight']) : 'normal') . ';';
	    $root_vars .= '--lunfest-submenu-letter-spacing: ' . ( $sub_menu_font['letter-spacing'] ? esc_attr(floatval($sub_menu_font['letter-spacing'])) . 'em' : 'normal') . ';';

        $sub_menu_color = WGL_Framework::get_option('sub_menu_color') ?? 'unset';
        $sub_menu_bg = WGL_Framework::get_option('sub_menu_background')['rgba'] ?? 'unset';
        $root_vars .= '--lunfest-submenu-color: ' . ( $sub_menu_color ? esc_attr($sub_menu_color) : 'unset' ) . ';';
        $root_vars .= '--lunfest-submenu-background: ' . ( $sub_menu_bg ? esc_attr($sub_menu_bg) : 'unset' ) . ';';

        $mob_sub_menu_color = WGL_Framework::get_option('mobile_sub_menu_color') ?? 'unset';
        $mob_sub_menu_bg = WGL_Framework::get_option('mobile_sub_menu_background')['rgba'] ?? 'unset';
        $mob_sub_menu_overlay = WGL_Framework::get_option('mobile_sub_menu_overlay')['rgba'] ?? 'unset';
        $root_vars .= '--lunfest-submenu-mobile-color: ' . esc_attr($mob_sub_menu_color) . ';';
        $root_vars .= '--lunfest-submenu-mobile-background: ' . esc_attr($mob_sub_menu_bg) . ';';
        $root_vars .= '--lunfest-submenu-mobile-overlay: ' . esc_attr($mob_sub_menu_overlay) . ';';

        $sub_menu_border = WGL_Framework::get_option('header_sub_menu_bottom_border');
        if ($sub_menu_border) {
            $sub_menu_border_height = WGL_Framework::get_option('header_sub_menu_border_height')['height'] ?? '0';
            $sub_menu_border_color = WGL_Framework::get_option('header_sub_menu_bottom_border_color')['rgba'] ?? 'unset';

            $extra_css .= '.primary-nav ul li ul li:not(:last-child),'
                . '.sitepress_container > .wpml-ls ul ul li:not(:last-child) {'
                    . ($sub_menu_border_height ? 'border-bottom-width: ' . esc_attr($sub_menu_border_height) . 'px;' : '')
                    . ($sub_menu_border_color ? 'border-bottom-color: ' . esc_attr($sub_menu_border_color) . ';' : '')
                    . 'border-bottom-style: solid;'
                . '}';
        }
        //* ↑ submenu variables

        /**
         * Additional Font Variables
         */
        $extra_font = WGL_Framework::get_option('additional-font');
	    $root_vars .= '--lunfest-additional-font-family: ' . ( $extra_font['font-family'] ? esc_attr($extra_font['font-family']) : 'unset' ) . ';';
	    $root_vars .= '--lunfest-additional-font-weight: ' . ( $extra_font['font-weight'] ? esc_attr($extra_font['font-weight']) : 'unset' ) . ';';
        //* ↑ additional font variables

        /**
         * Footer Variables
         */
        if (
            WGL_Framework::get_option('footer_switch')
            && 'widgets' === WGL_Framework::get_option('footer_building_tool')
        ) {
	        $footer_text_color = WGL_Framework::get_option('footer_text_color') ?? 'unset';
	        $footer_heading_color = WGL_Framework::get_option('footer_heading_color') ?? 'unset';
	        $copyright_text_color = WGL_Framework::get_mb_option('copyright_text_color', 'mb_copyright_switch', 'on') ?? 'unset';
            $root_vars .= '--lunfest-footer-content-color: ' . esc_attr($footer_text_color) . ';';
            $root_vars .= '--lunfest-footer-heading-color: ' . esc_attr($footer_heading_color) . ';';
            $root_vars .= '--lunfest-copyright-content-color: ' . esc_attr($copyright_text_color) . ';';
        }
        //* ↑ footer variables

        /**
         * Side Panel Variables
         */
        $sidepanel_title_color = WGL_Framework::get_mb_option('side_panel_title_color', 'mb_customize_side_panel', 'custom') ?? 'unset';
        $root_vars .= '--lunfest-sidepanel-title-color: ' . esc_attr($sidepanel_title_color) . ';';
        //* ↑ side panel variables

        /**
	     * Encoded SVG variables
	     */
        $root_vars .= '--lunfest-button-loading: url(\'data:image/svg+xml; utf8, <svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="489.698px" height="489.698px" viewBox="0 0 489.698 489.698" preserveAspectRatio="none" fill="%23161a1d"><path d="M468.999,227.774c-11.4,0-20.8,8.3-20.8,19.8c-1,74.9-44.2,142.6-110.3,178.9c-99.6,54.7-216,5.6-260.6-61l62.9,13.1    c10.4,2.1,21.8-4.2,23.9-15.6c2.1-10.4-4.2-21.8-15.6-23.9l-123.7-26c-7.2-1.7-26.1,3.5-23.9,22.9l15.6,124.8    c1,10.4,9.4,17.7,19.8,17.7c15.5,0,21.8-11.4,20.8-22.9l-7.3-60.9c101.1,121.3,229.4,104.4,306.8,69.3    c80.1-42.7,131.1-124.8,132.1-215.4C488.799,237.174,480.399,227.774,468.999,227.774z"/><path d="M20.599,261.874c11.4,0,20.8-8.3,20.8-19.8c1-74.9,44.2-142.6,110.3-178.9c99.6-54.7,216-5.6,260.6,61l-62.9-13.1    c-10.4-2.1-21.8,4.2-23.9,15.6c-2.1,10.4,4.2,21.8,15.6,23.9l123.8,26c7.2,1.7,26.1-3.5,23.9-22.9l-15.6-124.8    c-1-10.4-9.4-17.7-19.8-17.7c-15.5,0-21.8,11.4-20.8,22.9l7.2,60.9c-101.1-121.2-229.4-104.4-306.8-69.2    c-80.1,42.6-131.1,124.8-132.2,215.3C0.799,252.574,9.199,261.874,20.599,261.874z"/></svg>\');';

	    $root_vars .= '--lunfest-button-success: url(\'data:image/svg+xml; utf8, <svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="512px" height="512px" viewBox="0 0 24 24" preserveAspectRatio="none" fill="%23161a1d"><path d="m21.73 5.68-13 14a1 1 0 0 1 -.73.32 1 1 0 0 1 -.71-.29l-5-5a1 1 0 0 1 1.42-1.42l4.29 4.27 12.27-13.24a1 1 0 1 1 1.46 1.36z"/></svg>\');';

        $bg_caret = str_replace('#','%23',esc_attr($h_font_color ?? '#000000'));

        $root_vars .= '--lunfest-bg-caret: url(\'data:image/svg+xml; utf8, '.$this->bg_caret($bg_caret).'\');';

        $root_vars .= '--lunfest-bg-caret-white: url(\'data:image/svg+xml; utf8, '.$this->bg_caret('white').'\');';

        $wave_primary_color = str_replace('#','%23',esc_attr($theme_primary_color ?? '#000000'));
        $wave_tertiary_color = str_replace('#','%23',esc_attr($theme_tertiary_color ?? '#000000'));

	    $root_vars .= '--lunfest-wave: url(\'data:image/svg+xml; utf8, '.$this->lunfest_wave($wave_primary_color).'\');';

	    $root_vars .= '--lunfest-wave-tertiarity-color: url(\'data:image/svg+xml; utf8, '.$this->lunfest_wave($wave_tertiary_color).'\');';

        //* ↑ encoded SVG variables
	    /**
	     * Title variables
	     */
	    $root_vars .= '--wgl_price_label: "' . esc_html__( 'Price:', 'lunfest' ) . '";';

	    //* ↑ encoded Title variables

	    /**
         * Elementor Container
         */
        $root_vars .= '--lunfest-elementor-container-width: ' . $this->get_elementor_container_width() . 'px;';
        //* ↑ elementor container

        $css_variables = ':root {' . $root_vars . '}';

        $extra_css .= $this->get_mobile_header_extra_css();
        $extra_css .= $this->get_page_title_responsive_extra_css();
        if (
            class_exists('\Elementor\Plugin')
            && version_compare(ELEMENTOR_VERSION, '3.4', '>')
        ) {
            $extra_css .= $this->init_additional_breakpoints();
        }
        return $css_variables . $this->minify_css($extra_css);
    }

    public function bg_caret($fill = '#000'){

        $output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="512" height="512" x="0" y="0" viewBox="0 0 494.942 494.942" preserveAspectRatio="none" fill="'.esc_attr($fill).'"><g transform="matrix(6.123233995736766e-17,1,-1,6.123233995736766e-17,494.9419860839844,-2.842170943040401e-14)"><path xmlns="http://www.w3.org/2000/svg" d="m35.353 0 424.236 247.471-424.236 247.471z"></path></g></svg>';

        return $this->get_data_from_svg( $output );
    }

    public function lunfest_wave($fill = '#000'){

        $output = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 1200 10" style="enable-background:new 0 0 1200 10;" xml:space="preserve" preserveAspectRatio="none" fill="transparent" stroke="'.esc_attr($fill).'" stroke-width="2px"><path d="M0,2.6l2.2,0.6C4.3,3.8,8.7,5,12.8,5.7c4.2,0.7,8.7,1,12.8,0.4c4.2-0.6,8.3-2.1,12.8-2.8c4.2-0.7,8.3-0.7,12.9-0.9c4.2-0.2,8.3-0.4,12.8-0.1c4.1,0.3,8.3,1.2,12.8,2.2c4.1,1,8.3,2.2,12.5,2.6c4.5,0.4,8.7,0.2,12.8-0.6c4.4-0.7,8.7-1.9,12.9-1.6c4.4,0.3,8.7,2,12.8,2.6c4.3,0.6,8.7,0,12.8-0.4c4.3-0.4,8.7-0.7,12.9-0.6c4.3,0.1,8.7,0.7,12.8,1.2c4.3,0.4,8.7,0.7,12.8,0.7c4.2,0,8.7-0.3,12.8-0.3c4.2,0,8.3,0.3,12.9,0c4.2-0.3,8.3-1.2,12.8-2.1c4.2-0.9,8.3-1.8,12.8-1.5c4.1,0.3,8.3,1.8,12.8,2.5c4.1,0.7,8.3,0.7,12.5,0.2c4.5-0.6,8.7-1.8,12.8-2.6c4.4-0.9,8.7-1.5,12.9-0.7c4.4,0.7,8.7,2.8,12.8,3.1c4.3,0.3,8.7-1.2,12.9-1.8c4.3-0.6,8.7-0.3,12.8,0.3c4.3,0.6,8.7,1.5,12.8,0.9c4.3-0.6,8.7-2.6,12.8-2.8c4.2-0.1,8.7,1.6,12.9,1.8c4.2,0.2,8.3-1.3,12.8-1.2c4.2,0.1,8.3,1.9,12.8,2.2c4.2,0.3,8.3-0.9,12.9-2.1c4.1-1.2,8.3-2.3,12.8-2.6c4.1-0.3,8.3,0.3,12.5,0.9c4.4,0.6,8.7,1.2,12.9,1c4.4-0.1,8.7-1,12.8-1c4.4,0,8.7,0.9,12.9,2c4.3,1.2,8.7,2.6,12.8,3.1c4.3,0.4,8.7-0.2,12.8-0.6c4.3-0.4,8.7-0.7,10.8-0.8l2.1-0.1c8.4-0.4,17-2.4,25-3.5c16.9-2.3,32,2,49,4c4,0,7,0,11,0c5-1,9-3,14-2c12,4,24,3,36,1c27,5,53,1,80,0c5,0,11,1,16,2c3,0,5,0,8,0c7-2,13-4,20-4c5,1,11,2,16,2c11,1,22-1,33,0c3,0,5,1,8,1c4-1,7-3,11-2c10,0,20,1,30,0c12,4,23-1,35-3c3,0,5,0,8,0c4,1,9,1,13,2c7,1,13.1-2.5,20-1c9.1,1.9,18,3,27,5c7,1,14.8,0.3,22-1.5c5.1-1.3,9-2.2,12.5-2.8c4.2-0.7,8.3-0.7,12.9-0.9c4.2-0.2,8.3-0.4,12.8-0.1c4.1,0.3,8.3,1.2,12.8,2.2c4.1,1,8.3,2.2,12.5,2.6c4.4,0.4,8.7,0.2,12.8-0.6c4.4-0.7,8.7-1.9,12.9-1.6c4.4,0.3,8.7,2,12.8,2.6c4.3,0.6,8.7,0,12.8-0.4c4.3-0.4,8.7-0.7,12.9-0.6c4.3,0.1,8.7,0.7,12.8,1.2c4.3,0.4,8.7,0.7,12.8,0.7c4.2,0,8.7-0.3,12.8-0.3c4.2,0,8.3,0.3,12.8,0c4.2-0.3,8.3-1.2,12.9-2.1c4.2-0.9,8.3-1.8,12.8-1.5c4.1,0.3,8.3,1.8,12.8,2.5"/></svg>';
        return $this->get_data_from_svg( $output );
    }

    public function get_data_from_svg( $svg ) {
        return str_replace( [ '<', '>', '#' ], [ '%3C', '%3E', '%23' ], $svg );
    }

    public function init_additional_breakpoints()
    {
        $extra_css = '';
        $breakpoints = array_reverse(\Elementor\Plugin::$instance->breakpoints->get_active_breakpoints());
        $extra_css .= $this->content_alignment_responsive();
        $extra_css .= $this->media_content_responsive();
        $extra_css .= $this->hide_element_responsive(false, '-desktop');
        foreach ( $breakpoints as $breakpoint_name => $breakpoint ) {
            $extra_css .= $this->content_alignment_responsive($breakpoints[$breakpoint_name]->get_value(), '-'. $breakpoint_name);
            $extra_css .= $this->media_content_responsive($breakpoints[$breakpoint_name]->get_value(), '-'. $breakpoint_name);
            $extra_css .= $this->media_alignment_responsive($breakpoints[$breakpoint_name]->get_value(), '-'. $breakpoint_name);
            $extra_css .= $this->hide_element_responsive($breakpoints[$breakpoint_name]->get_value(), '-'. $breakpoint_name);
		}
        return $extra_css;
    }

    public function hide_element_responsive( $value = false, $media = '' )
    {
        $resolution = '-widescreen' === $media ? 'min' : 'max';
        $extra_css = $value ? '@media ('.$resolution.'-width: '.$value.'px) {' : '';
        $extra_css .= '
            .wgl-hidden'.$media.' {
                display: none;
            }';
        $extra_css .= $value ? '}' : '';

        return $extra_css;
    }

    public function content_alignment_responsive( $value = false, $media = '' )
    {
        $resolution = '-widescreen' === $media ? 'min' : 'max';
        $extra_css = $value ? '@media ('.$resolution.'-width: '.$value.'px) {' : '';
        $extra_css .= '
            body .a'.$media.'left {
                text-align: left;
            }
            body .a'.$media.'center {
                text-align: center;
            }
            body .a'.$media.'right {
                text-align: right;
            }
            body .a'.$media.'justify {
                text-align: justify;
            }';
        $extra_css .= $value ? '}' : '';

        return $extra_css;
    }

    public function media_alignment_responsive( $value = false, $media = '' )
    {
        $resolution = '-widescreen' === $media ? 'min' : 'max';
        $extra_css = $value ? '@media ('.$resolution.'-width: '.$value.'px) {' : '';
        $extra_css .= '
            body .a'.$media.'center .wgl-layout-left{
                justify-content: center;
            }
            body .a'.$media.'center .wgl-layout-right{
                justify-content: center;
            }
            body .a'.$media.'left .wgl-layout-left {
                justify-content: flex-start;
            }
            body .a'.$media.'left .wgl-layout-right {
                justify-content: flex-end;
            }

            body .a'.$media.'right .wgl-layout-left{
                justify-content: flex-end;
            }
            body .a'.$media.'right .wgl-layout-right{
                justify-content: flex-start;
            }';
        $extra_css .= $value ? '}' : '';

        return $extra_css;
    }

    public function media_content_responsive( $value = false, $media = '' )
    {
        $resolution = '-widescreen' === $media ? 'min' : 'max';
        $extra_css = $value ? '@media ('.$resolution.'-width: '.$value.'px) {' : '';
        $extra_css .= '
            body .wgl-layout'.$media.'-top {
                flex-direction: column;
            }
            body .wgl-layout'.$media.'-left {
                flex-direction: row;
            }
            body .wgl-layout'.$media.'-right {
                flex-direction: row-reverse;
            }';
        $extra_css .= $value ? '}' : '';

        return $extra_css;
    }

    public function get_elementor_container_width()
    {
        if (
            did_action('elementor/loaded')
            && defined('ELEMENTOR_VERSION')
        ) {
            if (version_compare(ELEMENTOR_VERSION, '3.0', '<')) {
                $container_width = get_option('elementor_container_width') ?: 1140;
            } else {
                $kit_id = (new \Elementor\Core\Kits\Manager())->get_active_id();
                $meta_key = \Elementor\Core\Settings\Page\Manager::META_KEY;
                $kit_settings = get_post_meta($kit_id, $meta_key, true);
                $container_width = $kit_settings['container_width']['size'] ?? 1140;
            }
        }

        return $container_width ?? 1170;
    }

    protected function get_mobile_header_extra_css()
    {
        $extra_css = '';

        $this->get_elementor_css_cache_header();

        if (WGL_Framework::get_option('mobile_header')) {
            $mobile_background = WGL_Framework::get_option('mobile_background')['rgba'] ?? '';
            $mobile_color = WGL_Framework::get_option('mobile_color');

            $extra_css .= '.wgl-theme-header {'
                    . 'background-color: ' . esc_attr($mobile_background) . ' !important;'
                    . 'color: ' . esc_attr($mobile_color) . ' !important;'
                . '}';
        }

        $extra_css .= 'header.wgl-theme-header .wgl-mobile-header {'
                . 'display: block;'
            . '}'
            . '.wgl-site-header,'
            . '.wgl-theme-header .primary-nav {'
                . 'display: none;'
            . '}'
            . '.wgl-theme-header .hamburger-box {'
                . 'display: inline-flex;'
            . '}'
            . 'header.wgl-theme-header .mobile_nav_wrapper .primary-nav {'
                . 'display: block;'
            . '}'
            . '.wgl-theme-header .wgl-sticky-header {'
                . 'display: none;'
            . '}'
            . '.wgl-page-socials {'
                . 'display: none;'
            . '}';

        $mobile_sticky = WGL_Framework::get_option('mobile_sticky');

        if (WGL_Framework::get_option('mobile_over_content')) {
            $extra_css .= 'body .wgl-theme-header {'
                    . 'position: absolute;'
                    . 'z-index: 99;'
                    . 'width: 100%;'
                    . 'left: 0;'
                    . 'top: 0;'
                . '}';

            if ($mobile_sticky) {
                $extra_css .= 'body .wgl-theme-header .wgl-mobile-header {'
                        . 'position: absolute;'
                        . 'left: 0;'
                        . 'width: 100%;'
                    . '}';
            }

        } else {
            $extra_css .= 'body .wgl-theme-header.header_overlap {'
                    . 'position: relative;'
                    . 'z-index: 2;'
                . '}';
        }

        if ( $mobile_sticky ) {
            $extra_css .= 'body .wgl-theme-header,'
                . 'body .wgl-theme-header.header_overlap {'
                .   'position: sticky;'
                .   'top: 0;'
                . '}'
                . '.admin-bar .wgl-theme-header{'
                .   'top: 32px;'
                . '}'
                . 'body.mobile_switch_on{'
                .   'position: static !important;'
                . '}'
                . 'body.admin-bar .sticky_mobile .wgl-menu_outer{'
                .   'top: 0px;'
                .   'height: 100vh;'
                . '}';
        }

        return '@media only screen and (max-width: ' . $this->get_header_mobile_breakpoint() . 'px) {' . $extra_css . '}';
    }

    protected function get_header_mobile_breakpoint()
    {
        $elementor_breakpoint = '';

        if (
            'elementor' === $this->header_building_tool
            && $this->header_page_id
            && did_action('elementor/loaded')
        ) {
            $settings_manager = \Elementor\Core\Settings\Manager::get_settings_managers('page');
            $settings_model = $settings_manager->get_model($this->header_page_id);

            $elementor_breakpoint = $settings_model->get_settings('mobile_breakpoint');
        }

        return $elementor_breakpoint ?: (int) WGL_Framework::get_option('header_mobile_queris');
    }

    protected function get_page_title_responsive_extra_css()
    {
        $responsive_disabled = ! WGL_Framework::get_option( 'page_title_resp_switch' );

        if (
            $this->RWMB_is_active()
            && 'on' === rwmb_meta('mb_page_title_switch')
            && rwmb_meta('mb_page_title_resp_switch')
        ) {
            $responsive_disabled = false;
        }

        if ( $responsive_disabled ) {
            // Bailout.
            return;
        }

        $pt_padding = WGL_Framework::get_mb_option('page_title_resp_padding', 'mb_page_title_resp_switch', true);

        $extra_css = '.page-header {'
            . (!empty($pt_padding['padding-top']) ? 'padding-top: ' . esc_attr((int) $pt_padding['padding-top']) . 'px !important;' : '')
            . (!empty($pt_padding['padding-bottom']) ? 'padding-bottom: ' . esc_attr((int) $pt_padding['padding-bottom']) . 'px !important;' : '')
            . 'min-height: auto !important;'
        . '}';

        $breadcrumbs_switch = WGL_Framework::get_mb_option('page_title_resp_breadcrumbs_switch', 'mb_page_title_resp_switch', true);

        //* Title
        $pt_font = WGL_Framework::get_mb_option('page_title_resp_font', 'mb_page_title_resp_switch', true);
        $pt_color = !empty($pt_font['color']) ? 'color: ' . esc_attr($pt_font['color']) . ' !important;' : '';
        $pt_f_size = !empty($pt_font['font-size']) ? ' font-size: ' . esc_attr((int) $pt_font['font-size']) . 'px !important;' : '';
        $pt_line_height = !empty($pt_font['line-height']) ? ' line-height: ' . esc_attr((int) $pt_font['line-height']) . 'px !important;' : '';
        $pt_additional_style = !(bool) $breadcrumbs_switch ? ' margin-bottom: 0 !important;' : '';
        $title_style = $pt_color . $pt_f_size . $pt_line_height . $pt_additional_style;

        $extra_css .= '.page-header_content .page-header_title {' . $title_style . '}';

        //* Breadcrumbs
        $breadcrumbs_font = WGL_Framework::get_mb_option('page_title_resp_breadcrumbs_font', 'mb_page_title_resp_switch', true);
        $breadcrumbs_color = !empty($breadcrumbs_font['color']) ? 'color: ' . esc_attr($breadcrumbs_font['color']) . ' !important;' : '';
        $breadcrumbs_f_size = !empty($breadcrumbs_font['font-size']) ? 'font-size: ' . esc_attr((int) $breadcrumbs_font['font-size']) . 'px !important;' : '';
        $breadcrumbs_line_height = !empty($breadcrumbs_font['line-height']) ? 'line-height: ' . esc_attr((int) $breadcrumbs_font['line-height']) . 'px !important;' : '';
        $breadcrumbs_display = !(bool) $breadcrumbs_switch ? 'display: none !important;' : '';
        $breadcrumbs_style = $breadcrumbs_color . $breadcrumbs_f_size . $breadcrumbs_line_height . $breadcrumbs_display;

        $extra_css .= '.page-header_content .page-header_breadcrumbs {' . $breadcrumbs_style . '}';

        //* Blog Single Type 3
        if (
            is_single()
            && 'post' === get_post_type()
            && '3' === WGL_Framework::get_mb_option('post_single_type_layout', 'mb_post_layout_conditional', 'custom')
        ) {
            $blog_t3_padding = WGL_Framework::get_option('single_padding_layout_3');
	        $blog_t3_p_top = $blog_t3_padding[ 'padding-top' ] ?? '';
	        $blog_t3_p_bottom = $blog_t3_padding[ 'padding-bottom' ] ?? '';
            $blog_t3_p_top_responsive = $blog_t3_p_top > $blog_t3_p_bottom ? 80 + (int) $blog_t3_p_bottom : (int) $blog_t3_p_top;
            $blog_t3_p_top_responsive = $blog_t3_p_top_responsive > 100 ? 100 : $blog_t3_p_top_responsive;
            $blog_t3_style = 'padding-top: ' . $blog_t3_p_top_responsive . 'px !important;';

            $extra_css .= '.single-post .post_featured_bg > .blog-post {' . esc_attr( $blog_t3_style ) . '}';
        }

        $pt_breakpoint = (int) WGL_Framework::get_mb_option('page_title_resp_resolution', 'mb_page_title_resp_switch', true);

        return '@media (max-width: ' . $pt_breakpoint . 'px) {' . $extra_css . '}';
    }

    /**
     * Enqueue theme stylesheets
     *
     * Function keeps track of already enqueued stylesheets and stores them in `enqueued_stylesheets[]`
     *
     * @param string   $tag      Unprefixed handle.
     * @param string   $file_dir Optional. Path to stylesheet folder, relative to theme root folder.
     * @param string[] $deps     Optional. An array of registered stylesheet handles this stylesheet depends on.
     */
    public function enqueue_theme_stylesheet( String $tag, $file_dir = '/css/pluggable/', $deps = [] )
    {
        $prefixed_tag = WGL_Framework_Global_Variables::get_theme_slug() . '-' . $tag;

        wp_enqueue_style(
            $prefixed_tag,
            $this->template_directory_uri . $file_dir . $tag . $this->use_minified . '.css',
            $deps,
            WGL_Framework_Global_Variables::get_theme_version()
        );

        $this->enqueued_stylesheets[] = $prefixed_tag;
    }

    public function enqueue_pluggable_styles()
    {
        //* Preloader
        WGL_Framework::get_option( 'preloader' ) && $this->enqueue_theme_stylesheet( 'preloader' );

        //* Page 404|Search
        ( is_404() || is_search() ) && $this->enqueue_theme_stylesheet( 'page-404' );

        //* Gutenberg
        WGL_Framework::get_option( 'disable_wp_gutenberg' )
            ? wp_dequeue_style( 'wp-block-library' )
            : $this->enqueue_theme_stylesheet( 'gutenberg' );

        //* Post Single
        if ( is_single() ) {
            $post_type = get_post()->post_type;
            if (
                'post' === $post_type
                || 'portfolio' === $post_type
                || 'event' === $post_type
                || 'location' === $post_type
            ) {
                $this->enqueue_theme_stylesheet( 'blog-post-single' );
            } elseif ( 'team' === $post_type ) {
                $this->enqueue_theme_stylesheet( 'team-post-single' );
            }
        }

        //* WooCommerce Plugin
        class_exists( 'WooCommerce' ) && $this->enqueue_theme_stylesheet( 'woocommerce' );

        //* Side Panel
        if (
            WGL_Framework::get_option( 'side_panel_enabled' )
            || 'side_panel' === ( get_queried_object()->post_type ?? '' )
        ) {
            $this->enqueue_theme_stylesheet( 'side-panel' );
        }

        //* WPML plugin
        class_exists( 'SitePress' ) && $this->enqueue_theme_stylesheet( 'wpml' );
    }

    public function frontend_scripts()
    {
        wp_enqueue_script(
            WGL_Framework_Global_Variables::get_theme_slug() . '-theme-addons',
            $this->template_directory_uri . '/js/theme-addons' . $this->use_minified . '.js',
            ['jquery'],
            WGL_Framework_Global_Variables::get_theme_version(),
            true
        );

        wp_enqueue_script(
            WGL_Framework_Global_Variables::get_theme_slug() . '-theme',
            $this->template_directory_uri . '/js/theme.js',
            ['jquery'],
            WGL_Framework_Global_Variables::get_theme_version(),
            true
        );

        wp_localize_script(
            WGL_Framework_Global_Variables::get_theme_slug() . '-theme',
            'wgl_core',
            ['ajaxurl' => esc_url(admin_url('admin-ajax.php'))]
        );

        if (is_singular() && comments_open() && get_option('thread_comments')) {
            wp_enqueue_script('comment-reply');
        }
    }

    public function admin_stylesheets()
    {
        wp_enqueue_style(
            WGL_Framework_Global_Variables::get_theme_slug() . '-admin',
            $this->template_directory_uri . '/core/admin/css/admin.css',
            [],
            WGL_Framework_Global_Variables::get_theme_version()
        );

        $this->enqueue_additional_styles();

        wp_enqueue_style( 'wp-color-picker' );
    }

    public function admin_scripts()
    {
        wp_enqueue_media();

        wp_enqueue_script('wp-color-picker');
	    wp_localize_script('wp-color-picker', 'wpColorPickerL10n', [
		    'clear' => esc_html__('Clear', 'lunfest'),
		    'clearAriaLabel' => esc_html__('Clear color', 'lunfest'),
		    'defaultString' => esc_html__('Default', 'lunfest'),
		    'defaultAriaLabel' => esc_html__('Select default color', 'lunfest'),
		    'pick' => esc_html__('Select', 'lunfest'),
		    'defaultLabel' => esc_html__('Color value', 'lunfest'),
        ]);

        wp_enqueue_script(
            WGL_Framework_Global_Variables::get_theme_slug() . '-admin',
            $this->template_directory_uri . '/core/admin/js/admin.js',
            [],
            WGL_Framework_Global_Variables::get_theme_version()
        );

        $currentTheme = wp_get_theme();
        $theme_name = false == $currentTheme->parent()
            ? wp_get_theme()->get('Name')
            : wp_get_theme()->parent()->get('Name');
        $theme_name = trim($theme_name);

        $purchase_code = $email = '';
        if (WGL_Framework::wgl_theme_activated()) {
            $theme_details = get_option('wgl_licence_validated');
            $purchase_code = $theme_details['purchase'] ?? '';
            $email = $theme_details['email'] ?? '';
        }

        wp_localize_script(
            WGL_Framework_Global_Variables::get_theme_slug() . '-admin',
            'wgl_verify',
            [
                'ajaxurl' => esc_js(admin_url('admin-ajax.php')),
                'wglUrlActivate' => esc_js(WGL_Theme_Verify::get_instance()->api . 'verification'),
                'wglUrlReset' => esc_js(WGL_Theme_Verify::get_instance()->api . 'reset_activation'),
                'wglUrlDeactivate' => esc_js(WGL_Theme_Verify::get_instance()->api . 'deactivate'),
                'domainUrl' => esc_js(site_url('/')),
                'themeName' => esc_js($theme_name),
                'purchaseCode' => esc_js($purchase_code),
                'email' => esc_js($email),
                'message' => esc_js(esc_html__('Thank you, your license has been validated', 'lunfest')),
                'ajax_nonce' => esc_js(wp_create_nonce('_notice_nonce')),
                'titleCodeRigistered' => esc_js(esc_html__('This purchase code has been registered', 'lunfest')),
                'messageCodeRigistered' => esc_js(esc_html__('Please go to your previous working environment and deactivate the purchase code to use it again (WP dashboard -> WebGeniusLab -> Activate Theme -> click on the button "Deactivate" )', 'lunfest')),
                'messageLostCode' => esc_js(esc_html__('Lost access to your previous site?', 'lunfest'))
            ]
        );
    }

    protected function add_body_classes()
    {
        add_filter( 'body_class', function ( Array $classes ) {
            if ( $this->gradient_enabled ) {
                $classes[] = 'theme-gradient';
            }

            if (
                is_single()
                && 'post' === get_post_type( get_queried_object_id() )
                && '3' === WGL_Framework::get_mb_option( 'post_single_type_layout', 'mb_post_layout_conditional', 'custom' )
            ) {
                $classes[] = WGL_Framework_Global_Variables::get_theme_slug() . '-blog-type-overlay';
            }

            return $classes;
        } );

        add_filter( 'wgl/header/mobile_width', function ($data) {
            return $this->get_header_mobile_breakpoint();
        } );
    }

    public function RWMB_is_active()
    {
        $id = ! is_archive() ? get_queried_object_id() : 0;

        return class_exists( 'RWMB_Loader' ) && 0 !== $id;
    }

    public function minify_css($css = null)
    {
        $css = str_replace(',{', '{', $css);
        $css = str_replace(', ', ',', $css);
        $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
        $css = str_replace(array("\r\n", "\r", "\n", "\t", '  ', '    ', '    '), '', $css);
        $css = trim($css);

        return $css;
    }
}

function wgl_dynamic_styles()
{
    return WGL_Framework_Dynamic_Styles::instance();
}

wgl_dynamic_styles()->construct();
