<?php
/**
 * Custom functions that deal with the integration of WP Job Manager Products.
 * See: https://astoundify.com/downloads/wp-job-manager-products/
 *
 * @package Listingeasy
 */

/**
 * Remove the action that ads the product details after the listing's single content
 */
function gt3_remove_wpjm_products_content_output() {

	if ( class_exists( 'WP_Job_Manager_Products' ) ) {
		//get the instance of WP_Job_Manager_Products that holds the instance of WPJMP_Products
		$wpjmp_instance = WP_Job_Manager_Products::instance();
		remove_action( 'single_job_listing_end', array( $wpjmp_instance->products, 'listing_display_products' ) );
	}
}

// Display products on listing page
add_action( 'single_job_listing_end', 'gt3_remove_wpjm_products_content_output', 1 );

// Display product price on listing archives
function gt3_listing_product_price( $post ) {
	$output = '';

	//get linked products
	$products_IDs = gt3_get_listing_products( $post->ID );
	if ( ! empty( $products_IDs ) ) {
		$first_productID = array_shift( $products_IDs );
		$product = wc_get_product( $first_productID );
		if ( method_exists( $product, 'get_price_html' ) ) {
			if (strlen($product->get_price_html()) > 0) {
				$output .= '<span class="product__price">' . $product->get_price_html() . '</span>';
			}
		}
	}

	echo  $output;
}

/* -------- WIDGETS -------- */

function gt3_register_widget_areas_wpjm_products() {
	register_widget( 'Listing_Sidebar_Products_Widget' );
}

add_action( 'widgets_init', 'gt3_register_widget_areas_wpjm_products' );

class Listing_Sidebar_Products_Widget extends WP_Widget {

	private $defaults = array(
			'title'           => '',
			'subtitle'        => '',
	);

	function __construct() {
		parent::__construct(
				'listing_sidebar_products', // Base ID
				'&#x1F537; ' . esc_html__( 'Listing', 'listingeasy') . ' &raquo; ' . esc_html__('Products', 'listingeasy' ), // Name
				array(
						'description' => esc_html__( 'The products linked to the current listing.', 'listingeasy' ),
				) // Widget Options
		);
	}

	public function widget( $args, $instance ) {

		$placeholders = $this->get_placeholder_strings();
		//only put in the default title if the user hasn't saved anything in the database e.g. $instance is empty (as a whole)
		$title    = apply_filters( 'widget_sidebar_title', ( empty( $instance ) || ! isset( $instance['title'] ) ) ? $placeholders['title'] : $instance['title'], $instance, $this->id_base );
		$subtitle = '';

		if ( empty( $instance['subtitle'] ) ) {
			$subtitle = $placeholders['subtitle'];
		} else {
			$instance['subtitle'];
		}

		$products_IDs = gt3_get_listing_products();

		if ( ! empty( $products_IDs ) ) :

			$query_args = apply_filters( 'woocommerce_related_products_args', array(
					'post_type'            => 'product',
					'ignore_sticky_posts'  => 1,
					'no_found_rows'        => 1,
					'posts_per_page'       => -1,
					'post__in'             => $products_IDs,
			) );

			$products = new WP_Query( $query_args );

			echo  $args['before_widget'];

			?><div class="listing_widget_wrapper"><?php

			if ( $products->have_posts() ) { ?>
				<h3 class="widget-title">
					<?php
					echo  $title;

					if ( ! empty( $subtitle ) ) { ?>
						<span class="widget_subtitle">
					<?php echo  $subtitle; ?>
				</span>
					<?php } ?>
				</h3>

				<div class="listing-products__items">

					<?php
					while ( $products->have_posts() ) : $products->the_post();

						wc_get_template_part( 'content', 'single-product-listing-sidebar' );

					endwhile;

					wp_reset_postdata(); ?>

				</div><!-- .listing-products__items -->

				<?php
			}

			?></div><?php

			echo  $args['after_widget'];

		endif;
	}

	public function form( $instance ) {
		$original_instance = $instance;

		//Defaults
		$instance = wp_parse_args(
				(array) $instance,
				$this->defaults );

		$placeholders = $this->get_placeholder_strings();

		$title = esc_attr( $instance['title'] );
		//if the user is just creating the widget ($original_instance is empty)
		if ( empty( $original_instance ) && empty( $title ) ) {
			$title = $placeholders['title'];
		}

		$subtitle = esc_attr( $instance['subtitle'] );
		//if the user is just creating the widget ($original_instance is empty)
		if ( empty( $original_instance ) && empty( $subtitle ) ) {
			$subtitle = $placeholders['subtitle'];
		} ?>

		<p>
			<label for="<?php echo  $this->get_field_id( 'title' ); ?>"><?php esc_html_e( 'Title:', 'listingeasy' ); ?></label>
			<input class="widefat" id="<?php echo  $this->get_field_id( 'title' ); ?>" name="<?php echo  $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo  $title; ?>" placeholder="<?php echo esc_attr( $placeholders['title'] ); ?>"/>
		</p>

		<p>
			<label for="<?php echo  $this->get_field_id( 'subtitle' ); ?>"><?php esc_html_e( 'Subtitle:', 'listingeasy' ); ?></label>
			<input class="widefat" id="<?php echo  $this->get_field_id( 'subtitle' ); ?>" name="<?php echo  $this->get_field_name( 'subtitle' ); ?>" type="text" value="<?php echo  $subtitle; ?>" />
		</p>

		<p>
			<?php echo  $this->widget_options['description']; ?>
		</p>
	<?php }

	/**
	 * @param array $new_instance
	 * @param array $old_instance
	 *
	 * @return array
	 */
	public function update( $new_instance, $old_instance ) {
		$instance                    = $old_instance;
		$instance['title']           = strip_tags( $new_instance['title'] );
		$instance['subtitle']        = strip_tags( $new_instance['subtitle'] );

		return $instance;
	}

	private function get_placeholder_strings() {
		$placeholders = apply_filters( 'listing_sidebar_products_backend_placeholders', array() );

		$placeholders = wp_parse_args(
				(array) $placeholders,
				array(
						'title'    => esc_html__( 'Book a Reservation', 'listingeasy' ),
						'subtitle' => esc_html__( 'Powered by WooCommerce Plugin', 'listingeasy' )
				) );

		return $placeholders;
	}

} // class Listing_Sidebar_Products_Widget


function gt3_get_listing_products( $post_ID = null ) {

	if ( null === $post_ID ) {
		$post_ID = get_the_ID();
	}

	if ( empty( $post_ID) || is_wp_error( $post_ID ) ) {
		return false;
	}
	$products = get_post_meta( $post_ID, '_products', true );

	if ( ! $products || ! is_array( $products ) ) {
		return false;
	}
	return $products;
}