<?php
/**
 * Yardımcı fonksiyonlar
 * 
 * @package Kanews
 * @developer Kan Themes
 * @version 7.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Typography stillerini oluşturan yardımcı fonksiyon
 * 
 * @param array $typography Typography ayarları
 * @param bool $important !important eklenip eklenmeyeceği
 * @return string CSS stilleri
 */
function kanews_get_typography_styles($typography = [], $important = false) {
    if (empty($typography)) {
        return '';
    }

    $styles = '';
    $suffix = $important ? ' !important' : '';
    
    if (!empty($typography['fontFamily'])) {
        $styles .= 'font-family: ' . esc_attr($typography['fontFamily']) . $suffix . ';';
    }
    if (!empty($typography['fontSize'])) {
        $styles .= 'font-size: ' . esc_attr($typography['fontSize']) . $suffix . ';';
    }
    if (!empty($typography['fontWeight'])) {
        $styles .= 'font-weight: ' . esc_attr($typography['fontWeight']) . $suffix . ';';
    }
    if (!empty($typography['lineHeight'])) {
        $styles .= 'line-height: ' . esc_attr($typography['lineHeight']) . $suffix . ';';
    }
    if (!empty($typography['letterSpacing'])) {
        $styles .= 'letter-spacing: ' . esc_attr($typography['letterSpacing']) . $suffix . ';';
    }
    if (!empty($typography['textTransform'])) {
        $styles .= 'text-transform: ' . esc_attr($typography['textTransform']) . $suffix . ';';
    }
    if (!empty($typography['fontStyle'])) {
        $styles .= 'font-style: ' . esc_attr($typography['fontStyle']) . $suffix . ';';
    }
    if (!empty($typography['textDecoration'])) {
        $styles .= 'text-decoration: ' . esc_attr($typography['textDecoration']) . $suffix . ';';
    }
    if (!empty($typography['textAlign'])) {
        $styles .= 'text-align: ' . esc_attr($typography['textAlign']) . $suffix . ';';
    }
    
    return $styles;
}

/**
 * HTML içeriğindeki inline stilleri toplayıp style tag'i olarak döndürür
 * 
 * @param string $content HTML içeriği
 * @return string Style tag'i ve orijinal içerik
 */
function kanews_collect_inline_styles($content) {
    if (empty($content)) {
        return $content;
    }

    // DOM işlemleri için içeriği yükle
    $dom = new DOMDocument();
    @$dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'), LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    
    // Style tag'i oluştur
    $style_tag = $dom->createElement('style');
    $style_content = '';
    
    // Tüm elementleri döngüye al
    $elements = $dom->getElementsByTagName('*');
    foreach ($elements as $element) {
        if ($element->hasAttribute('style')) {
            $style = $element->getAttribute('style');
            $class_name = 'kanews-inline-style-' . uniqid();
            
            // Mevcut class'ı al ve yeni class'ı ekle
            $current_class = $element->getAttribute('class');
            $new_class = $current_class ? $current_class . ' ' . $class_name : $class_name;
            $element->setAttribute('class', $new_class);
            
            // Style'ı style tag'ine ekle (orijinal inline style korunacak)
            $style_content .= '.' . $class_name . ' { ' . $style . ' } ';
        }
    }
    
    // Style içeriğini ekle
    if (!empty($style_content)) {
        $style_tag->textContent = $style_content;
        
        // İlk div elementini bul
        $first_div = $dom->getElementsByTagName('div')->item(0);
        if ($first_div) {
            // Style tag'ini ilk div'den önce ekle
            $first_div->parentNode->insertBefore($style_tag, $first_div);
        } else {
            // Div bulunamazsa documentElement'in başına ekle
            $dom->documentElement->insertBefore($style_tag, $dom->documentElement->firstChild);
        }
    }
    
    // HTML'i döndür
    return $dom->saveHTML();
}
