<?php
/**
 * Listing items data manager
 *
 * @package   package_name
 * @author    Cherry Team
 * @license   GPL-2.0+
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

if ( ! class_exists( 'Jet_Engine_Listings_Data' ) ) {

	/**
	 * Define Jet_Engine_Listings_Data class
	 */
	class Jet_Engine_Listings_Data {

		/**
		 * Current listing object
		 *
		 * @var object
		 */
		private $current_object = null;

		/**
		 * Current listing document
		 *
		 * @var array
		 */
		private $current_listing = false;

		/**
		 * <ain listing document for current page
		 * @var null
		 */
		private $main_listing = null;

		/**
		 * Default object holder
		 *
		 * @var mixed
		 */
		private $default_object = null;

		/**
		 * Listing settings defaults
		 * @var array
		 */
		private $defaults = false;

		/**
		 * Set current listing from outside
		 *
		 * @param void
		 */
		public function set_listing( $listing_doc = null ) {

			if ( $listing_doc->get_settings( 'is_main' ) ) {
				$this->main_listing = $listing_doc;
			}

			$this->current_listing = $listing_doc;
		}

		/**
		 * Reset current listing object
		 *
		 * @return void
		 */
		public function reset_listing() {
			$this->current_listing = $this->main_listing;
			$this->reset_current_object();
		}

		/**
		 * Returns current listing object
		 *
		 * @return [type] [description]
		 */
		public function get_listing() {
			return $this->current_listing;
		}

		/**
		 * Retuns current object fields array
		 * @return [type] [description]
		 */
		public function get_object_fields( $where = 'elementor' ) {

			$fields = array();

			switch ( $this->get_listing_source() ) {

				case 'posts':

					$fields = array(
						'post_title'   => __( 'Title', 'jet-engine' ),
						'post_date'    => __( 'Date', 'jet-engine' ),
						'post_content' => __( 'Content', 'jet-engine' ),
						'post_excerpt' => __( 'Excerpt', 'jet-engine' ),
					);

					break;

				case 'terms':

					$fields = array(
						'name'        => __( 'Term name', 'jet-engine' ),
						'description' => __( 'Term description', 'jet-engine' ),
						'count'       => __( 'Posts count', 'jet-engine' ),
					);

					break;

				case 'users':

					$fields = array(
						'ID'              => __( 'ID', 'jet-engine' ),
						'user_login'      => __( 'Login', 'jet-engine' ),
						'user_nicename'   => __( 'Nickname', 'jet-engine' ),
						'user_email'      => __( 'E-mail', 'jet-engine' ),
						'user_url'        => __( 'URL', 'jet-engine' ),
						'user_registered' => __( 'Registration Date', 'jet-engine' ),
						'display_name'    => __( 'Display Name', 'jet-engine' ),
					);

					break;
			}

			$result = array();

			if ( 'blocks' === $where ) {

				foreach ( $fields as $value => $label ) {
					$result[] = array(
						'value' => $value,
						'label' => $label,
					);
				}

			} else {
				$result = $fields;
			}

			return $result;

		}

		/**
		 * Get listing default property
		 *
		 * @param  string $prop [description]
		 * @return [type]       [description]
		 */
		public function listing_defaults( $prop = 'listing_source' ) {

			if ( ! empty( $this->defaults ) ) {
				return isset( $this->defaults[ $prop ] ) ? $this->defaults[ $prop ] : false;
			}

			$default = array(
				'listing_source'    => 'posts',
				'listing_post_type' => 'post',
				'listing_tax'       => 'category',
			);

			if ( ! $this->get_listing() ) {

				$default_object = $this->get_default_object();

				if ( ! $default_object ) {
					$this->defaults = $default;
					return isset( $this->defaults[ $prop ] ) ? $this->defaults[ $prop ] : false;
				}

				$listing = apply_filters( 'jet-engine/listing/data/custom-listing', false, $this, $default_object );

				if ( ! $listing ) {

					if ( isset( $default_object->post_type ) ) {
						$this->defaults = array(
							'listing_source'    => 'posts',
							'listing_post_type' => $default_object->post_type,
							'listing_tax'       => 'category',
						);
					} else {
						$this->defaults = array(
							'listing_source'    => 'terms',
							'listing_post_type' => 'post',
							'listing_tax'       => $default_object->taxonomy,
						);
					}

				} else {
					$this->defaults = $listing;
				}

				return isset( $this->defaults[ $prop ] ) ? $this->defaults[ $prop ] : false;
			}

		}

		/**
		 * Returns listing source
		 *
		 * @return string
		 */
		public function get_listing_source() {

			$listing = $this->get_listing();

			if ( ! $listing ) {
				return $this->listing_defaults( 'listing_source' );
			} else {
				return $listing->get_settings( 'listing_source' );
			}
		}

		/**
		 * Returns post type for query
		 *
		 * @return string
		 */
		public function get_listing_post_type() {

			$listing = $this->get_listing();

			if ( ! $listing ) {

				$post_type = get_post_type();

				$blacklisted = array(
					'elementor_library',
					'jet-theme-core',
				);

				if ( $post_type && ! in_array( $post_type, $blacklisted ) ) {
					return $post_type;
				} else {
					return $this->listing_defaults( 'listing_post_type' );
				}

			} else {
				return $listing->get_settings( 'listing_post_type' );
			}
		}

		/**
		 * Returns taxonomy for query
		 *
		 * @return string
		 */
		public function get_listing_tax() {

			$lising = $this->get_listing();

			if ( ! $lising ) {
				return $this->listing_defaults( 'listing_tax' );
			} else {
				return $lising->get_settings( 'listing_tax' );
			}
		}

		/**
		 * Set $current_object property
		 *
		 * @param object $object
		 */
		public function set_current_object( $object = null ) {
			$this->current_object = $object;
		}

		/**
		 * Set $current_object property
		 *
		 * @param object $object
		 */
		public function reset_current_object() {
			$this->current_object = null;
		}

		/**
		 * Returns $current_object property
		 *
		 * @return object
		 */
		public function get_current_object() {

			if ( null === $this->current_object ) {
				$this->current_object = $this->get_default_object();
			}

			return $this->current_object;
		}

		/**
		 * Returns default object
		 *
		 * @return [type] [description]
		 */
		public function get_default_object() {

			if ( null !== $this->default_object ) {
				return $this->default_object;
			}

			$default_object = false;

			if ( is_singular() ) {
				global $post;
				$default_object = $this->default_object = $post;
			} elseif ( is_tax() || is_category() || is_tag() ) {
				$default_object = $this->default_object =  get_queried_object();
			} elseif ( wp_doing_ajax() ) {
				$post_id = isset( $_REQUEST['editor_post_id'] ) ? $_REQUEST['editor_post_id'] : false;

				if ( ! $post_id ) {
					$default_object = $this->default_object = false;
				} else {
					$default_object = $this->default_object = get_post( $post_id );
				}

			} elseif ( is_archive() || is_home() || is_post_type_archive() ) {
				global $post;
				$default_object = $post;
			} else {
				$default_object = $this->default_object = false;
			}

			$this->default_object = apply_filters( 'jet-engine/listings/data/default-object', $default_object, $this );

			return $this->default_object;

		}

		/**
		 * Returns requested property from current object
		 *
		 * @param  [type] $property [description]
		 * @return [type]           [description]
		 */
		public function get_prop( $property = null ) {

			$object = $this->get_current_object();
			$class  = get_class( $object );
			$vars   = get_object_vars( $object );

			if ( 'WP_User' === $class ) {
				$vars = ! empty( $vars['data'] ) ? (array) $vars['data'] : array();
				if ( 'user_nicename' === $property ) {
					$vars['user_nicename'] = get_user_meta( $object->ID, 'nickname', true );
				}
			}

			return isset( $vars[ $property ] ) ? $vars[ $property ] : false;
		}

		/**
		 * Remove tabs and accordions from allowed fields list
		 *
		 * @param  [type] $fields [description]
		 * @return [type]         [description]
		 */
		public function sanitize_meta_fields( $fields ) {
			return array_filter( $fields, function( $field ) {
				if ( ! empty( $field['object_type'] ) && 'field' !== $field['object_type'] ) {
					return false;
				} else {
					return true;
				}
			} );
		}

		/**
		 * Returns option value by combined key
		 *
		 * @param  [type] $key [description]
		 * @return [type]      [description]
		 */
		public function get_option( $key = null ) {

			if ( ! jet_engine()->options_pages || ! $key ) {
				return null;
			}

			$data = explode( '::', $key );

			if ( 2 !== count( $data ) ) {
				return null;
			}

			$page_slug = $data[0];
			$option    = $data[1];

			if ( ! $page_slug || ! $option ) {
				return null;
			}

			$page = isset( jet_engine()->options_pages->registered_pages[ $page_slug ] ) ? jet_engine()->options_pages->registered_pages[ $page_slug ] : false;

			if ( ! $page ) {
				return;
			}

			return $page->get( $option );

		}

		/**
		 * Returns current meta
		 *
		 * @param  [type] $key [description]
		 * @return [type]      [description]
		 */
		public function get_meta( $key ) {

			$object = $this->get_current_object();

			if ( ! $object ) {
				return false;
			}

			$class  = get_class( $object );
			$result = '';

			switch ( $class ) {
				case 'WP_Post':

					if ( jet_engine()->relations->is_relation_key( $key ) ) {
						$single = false;
					} else {
						$single = true;
					}

					return get_post_meta( $object->ID, $key, $single );

				case 'WP_Term':
					return get_term_meta( $object->term_id, $key, true );

				case 'WP_User':
					return get_user_meta( $object->ID, $key, true );

			}

		}

		/**
		 * Get permalink to current post/term
		 *
		 * @return string
		 */
		public function get_current_object_permalink() {

			$object = $this->get_current_object();
			$class  = get_class( $object );
			$result = '';

			switch ( $class ) {
				case 'WP_Post':
					return get_permalink( $object->ID );

				case 'WP_Term':
					return get_term_link( $object->term_id );

				case 'WP_User':
					return apply_filters( 'jet-engine/listings/data/user-permalink', false, $object );
			}

			return null;

		}

		/**
		 * Returns available list sources
		 *
		 * @return [type] [description]
		 */
		public function get_field_sources() {

			$sources = array(
				'object' => __( 'Post/Term/User Data', 'jet-engine' ),
				'meta'   => __( 'Meta Data', 'jet-engine' ),
			);

			if ( jet_engine()->options_pages ) {
				$sources['options_page'] = __( 'Options', 'jet-engine' );
			}

			return apply_filters( 'jet-engine/listings/data/sources', $sources );
		}

	}

}
