/**
 * Extend default number object with format function
 *
 * @param integer n: length of decimal
 * @param integer x: length of whole part
 * @param mixed   s: sections delimiter
 * @param mixed   c: decimal delimiter
 */
Number.prototype.jetFormat = function( n, x, s, c ) {
	var re = '\\d(?=(\\d{' + (x || 3) + '})+' + (n > 0 ? '\\D' : '$') + ')',
		num = this.toFixed(Math.max(0, ~~n));
	return (c ? num.replace('.', c) : num).replace(new RegExp(re, 'g'), '$&' + (s || ''));
};

( function( $ ) {

	"use strict";

	var JetSmartFilterSettings = window.JetSmartFilterSettings || false;
	var xhr = null;

	var JetSmartFilters = {

		currentQuery: null,
		page: false,
		controls: false,
		resetFilters: false,
		resetHierarchy: false,
		currentHierarchy: {
			filterID: false,
			depth: false,
			trail: [],
		},
		hasRedirectData: false,
		savedFiltersQuery: [],

		init: function() {

			var self = JetSmartFilters;

			$( document )
				.on( 'click.JetSmartFilters', '.apply-filters__button[data-apply-type="reload"]', self.applyFilters )
				.on( 'click.JetSmartFilters', 'button[data-apply-type="ajax-reload"]', self.applyAjaxFilters )
				.on( 'click.JetSmartFilters', 'button[data-apply-type="ajax"]', self.applyAjaxFilters )
				.on( 'click.JetSmartFilters', '.jet-active-filter', self.removeFilter )
				.on( 'click.JetSmartFilters', '.jet-remove-all-filters__button', self.removeAllFilters )
				.on( 'change.JetSmartFilters', 'input[data-apply-type="ajax"]', self.applyAjaxFilters )
				.on( 'change.JetSmartFilters', 'select[data-hierarchical="1"]', self.getNextHierarchyLevels )
				.on( 'change.JetSmartFilters', 'select[data-apply-type="ajax"]', self.applyAjaxFilters )
				.on( 'keypress.JetSmartFilters', 'input.jet-search-filter__input', self.applySearchFilterOnEnter )

				.on( 'click.JetSmartFilters', '.jet-filters-pagination__link', self.applyPagination )

				.on( 'jet-filter-add-rating-vars', self.processRating )
				.on( 'jet-filter-add-checkboxes-vars', self.processCheckbox )
				.on( 'jet-filter-add-color-image-vars', self.processCheckbox )
				.on( 'jet-filter-add-check-range-vars', self.processCheckbox )
				.on( 'jet-filter-add-range-vars', self.processRange )
				.on( 'jet-filter-add-date-range-vars', self.processRange )
				.on( 'jet-filter-add-select-vars', self.processSelect )
				.on( 'jet-filter-add-search-vars', self.processSearch )
				.on( 'jet-filter-add-radio-vars', self.processRadio )

				.on( 'jet-filter-remove-checkboxes-vars', self.removeCheckbox )
				.on( 'jet-filter-remove-color-image-vars', self.removeCheckbox )
				.on( 'jet-filter-remove-check-range-vars', self.removeCheckbox )
				.on( 'jet-filter-remove-range-vars', self.removeRange )
				.on( 'jet-filter-remove-date-range-vars', self.removeDateRange )
				.on( 'jet-filter-remove-select-vars', self.removeSelect )
				.on( 'jet-filter-remove-search-vars', self.removeSearch )
				.on( 'jet-filter-remove-radio-vars', self.removeCheckbox )
				.on( 'jet-filter-remove-rating-vars', self.removeCheckbox )

				.on( 'click.JetSmartFilters', 'input.jet-rating-star__input', self.unselectRating)

				.on( 'jet-filter-load', self.applyLoader )
				.on( 'jet-filter-loaded', self.removeLoader )

				.on( 'jet-engine-request-calendar', self.addFiltersToCalendarRequest )

				.on( 'jet-filter-before-loaded', self.showRemoveAllButtonOnAjax );

			$( window ).on( 'elementor/frontend/init', function() {

				self.maybeApplyDataFromStorage();
				self.showRemoveAllButtonOnReload();

				if ( ! self.hasRedirectData ){
					if ( false !== JetSmartFilterSettings && JetSmartFilterSettings.refresh_controls ) {
						$.each( JetSmartFilterSettings.refresh_provider, function( provider, instances ) {
							$.each( instances, function( index, queryID ) {
								setTimeout( function() {
									self.refreshControls( provider, queryID );
								} );
							});
						});
					}
				}

				self.hasRedirectData = false;

			} );

		},

		maybeApplyDataFromStorage: function() {

			var storageData = localStorage.getItem( 'jet_smart_filters_query' );

			if ( storageData ){
				storageData = JSON.parse( storageData );
				JetSmartFilters.currentQuery = storageData.query;
				JetSmartFilters.controls = storageData.controls;
				JetSmartFilters.hasRedirectData = true;

				$( 'div[data-content-provider="' + storageData.provider + '"][data-query-id="' + storageData.queryID + '"]' ).each( function() {

					var $this          = $( this ),
						queryType      = $this.data( 'query-type' ),
						queryVar       = $this.data( 'query-var' ),
						filterType     = $this.data( 'smart-filter' ),
						filterID       = parseInt( $this.data( 'filter-id' ), 10 ),
						key            = '_' + queryType + '_' + queryVar,
						skip           = false;

					$( document ).trigger(
						'jet-filter-load',
						[ $this, JetSmartFilters, storageData.provider, JetSmartFilters.currentQuery, storageData.queryID ]
					);

					if ( storageData.hierarchy && filterID === storageData.hierarchy.filterID ) {

						var filterDepth = parseInt( $this.data( 'depth' ), 10 );

						if ( 0 === filterDepth ) {
							JetSmartFilters.currentHierarchy = storageData.hierarchy;
							JetSmartFilters.getAllHierarchyLevels( $this, storageData.hierarchy.trail );
						}

						if ( filterDepth > JetSmartFilters.currentHierarchy.depth ) {
							skip = true;
						}
					}

					key = JetSmartFilters.addQueryVarSuffix( key, $this );

					if ( ! skip ) {
						$( document ).trigger(
							'jet-filter-add-' + filterType + '-vars',
							[ $this, key, JetSmartFilters, 'object' ]
						);
					}
				} );

				xhr = JetSmartFilters.ajaxRequest(
					false,
					'jet_smart_filters',
					storageData.provider,
					storageData.query,
					storageData.props,
					storageData.queryID
				);

				localStorage.removeItem( 'jet_smart_filters_query' );

			}
		},

		showRemoveAllButtonOnAjax: function( e, $scope, $filters, provider, query, queryID, response ) {
			var removeButton = '.jet-remove-all-filters__button[data-apply-provider="' + provider + '"][data-query-id="' + queryID + '"]';

			$( removeButton ).each( function() {
				var $scope = $( this );

				if ( !$.isEmptyObject( JetSmartFilters.currentQuery ) ) {
					$scope.parent().removeClass( 'hide' );
					$scope.parent().addClass( 'show' );
				} else {
					$scope.parent().removeClass( 'show' );
					$scope.parent().addClass( 'hide' );
				}

			} );
		},

		showRemoveAllButtonOnReload: function(  ) {
			$( '.jet-remove-all-filters__button' ).each( function(){
				var $scope        = $( this ),
					provider      = $scope.data( 'apply-provider' ),
					reloadType    = $scope.data( 'apply-type' ),
					queryID       = $scope.data( 'query-id' );

				if ( 'ajax' !== reloadType ){
					var $searchText = document.location.search,
						hasActiveFilters = $searchText.toLowerCase().indexOf( provider + '/' + queryID );

					if ( 0 < hasActiveFilters ){
						$scope.parent().removeClass( 'hide' );
						$scope.parent().addClass( 'show' );
					}
				}

			} );
		},

		resetCurrentHierarchy: function() {
			JetSmartFilters.currentHierarchy = {
				filterID: false,
				depth: false,
				trail: [],
			};
		},

		addFiltersToCalendarRequest: function( event ) {
			window.JetEngine.currentRequest.query    = JetSmartFilters.getQuery( 'object', 'jet-engine-calendar' );
			window.JetEngine.currentRequest.provider = 'jet-engine-calendar';
		},

		providerSelector: function( providerWrap, queryID ) {

			var delimiter = '';

			if ( providerWrap.inDepth ) {
				delimiter = ' ';
			}

			return providerWrap.idPrefix + queryID + delimiter + providerWrap.selector;

		},

		applyLoader: function ( event, $scope, JetSmartFilters, provider, query, queryID ) {

			var providerWrap = JetSmartFilterSettings.selectors[ provider ],
				$provider    = null;

			if ( ! queryID ) {
				queryID = 'default';
			}

			if ( 'default' === queryID ) {
				$provider = $( providerWrap.selector );
			} else {
				$provider = $( JetSmartFilters.providerSelector( providerWrap, queryID ) );
			}

			$provider.addClass( 'jet-filters-loading' );
			$( 'div[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"], button[data-apply-provider="' + provider + '"][data-query-id="' + queryID + '"]' ).addClass( 'jet-filters-loading' );

			if ( 'jet-engine' === provider && JetSmartFilterSettings.settings[ provider ] ) {
				var useLoadMore = JetSmartFilterSettings.settings[ provider ][ queryID ].use_load_more || false,
					$loadMore   = $( '#' + JetSmartFilterSettings.settings[ provider ][ queryID ].load_more_id );

				if ( useLoadMore && $loadMore.length ) {
					$loadMore.addClass( 'jet-filters-loading' );
				}
			}

		},

		removeLoader: function ( event, $scope, JetSmartFilters, provider, query, queryID ) {

			var providerWrap = JetSmartFilterSettings.selectors[ provider ],
				$provider    = null;

			if ( ! queryID ) {
				queryID = 'default';
			}

			if ( 'default' === queryID ) {
				$provider = $( providerWrap.selector );
			} else {
				$provider = $( JetSmartFilters.providerSelector( providerWrap, queryID ) );
			}

			$provider.removeClass( 'jet-filters-loading' );
			$( 'div[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"], button[data-apply-provider="' + provider + '"][data-query-id="' + queryID + '"]' ).removeClass( 'jet-filters-loading' );

			if ( 'jet-engine' === provider ) {
				var useLoadMore = JetSmartFilterSettings.settings[ provider ][ queryID ].use_load_more || false,
					$loadMore   = $( '#' + JetSmartFilterSettings.settings[ provider ][ queryID ].load_more_id );

				if ( useLoadMore && $loadMore.length ) {
					$loadMore.removeClass( 'jet-filters-loading' );
				}
			}

			if ( false !== $scope ){
				if( $scope.hasClass('jet-filters-pagination__link') && $scope.data('apply-type') ){
					$('html, body').stop().animate({ scrollTop: $provider.offset().top }, 500);
				}
			}

		},

		removeFilter: function() {

			var $filter    = $( this ),
				$filters   = $filter.closest( '.jet-active-filters' ),
				data       = $filter.data( 'filter' ),
				provider   = $filters.data( 'apply-provider' ),
				reloadType = $filters.data( 'apply-type' ),
				queryID = $filters.data( 'query-id' );

			$( document ).trigger(
				'jet-filter-remove-' + data.type + '-vars',
				[$filter, data, JetSmartFilters, provider]
			);

			if ( 'rating' === data.type ) {
				JetSmartFilters.resetRating( provider, queryID );
			}

			if ( 'ajax' !== reloadType ) {
				JetSmartFilters.requestFilter( provider );
			}

		},

		removeAllFilters: function() {

			var $scope        = $( this ),
				provider      = $scope.data( 'apply-provider' ),
				reloadType    = $scope.data( 'apply-type' ),
				queryID       = $scope.data( 'query-id' ),
				$currentQuery = JetSmartFilters.currentQuery;

			if ( !queryID ) {
				queryID = 'default';
			}

			if ( 'reload' === reloadType ) {
				document.location.search = JetSmartFilters.addQueryArg(
					'jet-smart-filters',
					provider + '/' + queryID,
					null,
					'append'
				);
			} else {
				JetSmartFilters.resetFilters = true;

				$( '.jet-filter div[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"]' ).each( function() {

					var $this          = $( this ),
						queryType      = $this.data( 'query-type' ),
						filterId       = $this.data( 'filter-id' ),
						queryVar       = $this.data( 'query-var' ),
						filterType     = $this.data( 'smart-filter' ),
						key            = '_' + queryType + '_' + queryVar;

					key = JetSmartFilters.addQueryVarSuffix( key, $this );

					if ( 'undefined' !== typeof $currentQuery[ key ] ){

						var data = {
							'id' : filterId,
							'type' : filterType,
							'value' : $currentQuery[key],
							'queryVar' : queryVar
						};

						console.log( JetSmartFilters.currentHierarchy );

						$( document ).trigger(
							'jet-filter-remove-' + filterType + '-vars',
							[ $this, data, JetSmartFilters, provider ]
						);

						if ( 'rating' === filterType ) {
							JetSmartFilters.resetRating( provider, queryID );
						}

					}

				} );

				JetSmartFilters.resetFilters = false;
				JetSmartFilters.resetHierarchy = false;
				JetSmartFilters.ajaxFilters( $scope );
			}

		},

		removeRange: function( event, $scope, data, JetSmartFilters, provider ) {

			var filterID  = data.id,
				$filter   = JetSmartFilters.getFilterElement( filterID ),
				$slider   = $filter.find( '.jet-range__slider' ),
				$input    = $filter.find( '.jet-range__input' ),
				$min      = $filter.find( '.jet-range__values-min' ),
				$max      = $filter.find( '.jet-range__values-max' ),
				min       = $slider.data( 'min' ),
				max       = $slider.data( 'max' ),
				applyType = $input.data( 'apply-type' );

			$slider.slider( 'values', [ min, max ] );

			$min.text( min );
			$max.text( max );

			$input.val( min + ':' + max );

			if( JetSmartFilters.resetFilters ){
				return;
			}

			if ( 'ajax' === applyType ) {
				$input.trigger( 'change.JetSmartFilters' );
			} else if ( 'ajax-reload' === applyType ) {
				JetSmartFilters.ajaxFilters( $input );
			}

		},

		removeDateRange: function( event, $scope, data, JetSmartFilters, provider ) {

			var filterID = data.id,
				$filter  = JetSmartFilters.getFilterElement( filterID ),
				$from    = $filter.find( '.jet-date-range__from' ),
				$to      = $filter.find( '.jet-date-range__to' ),
				$input   = $filter.find( '.jet-date-range__input' ),
				$submit  = $filter.find( '.jet-date-range__submit' );

			$from.val( '' );
			$to.val( '' );
			$input.val( '' );

			if( JetSmartFilters.resetFilters ){
				return;
			}

			$submit.trigger( 'click.JetSmartFilters' );
		},

		removeCheckbox: function( event, $scope, data, JetSmartFilters, provider ) {

			var filterID  = data.id,
				$last     = null,
				$filter   = JetSmartFilters.getFilterElement( filterID ),
				applyType = null;

			$filter.find( 'input:checked' ).each( function() {
				var $this = $( this );
				$this.removeAttr( 'checked' );
				$last = $this;
			});

			if( JetSmartFilters.resetFilters ){
				return;
			}

			if ( $last ) {

				applyType = $last.data( 'apply-type' );

				if ( 'ajax' === applyType ) {
					$last.trigger( 'change.JetSmartFilters' );
				} else if ( 'ajax-reload' === applyType ) {
					JetSmartFilters.ajaxFilters( $last );
				}
			}

		},

		removeSelect: function( event, $scope, data, JetSmartFilters, provider ) {

			var filterID  = data.id,
				$select   = JetSmartFilters.getFilterElement( filterID, 'div[data-filter-id="' + filterID + '"] select' ),
				applyType = $select.data( 'apply-type' );

			$select.find( 'option:selected' ).removeAttr( 'selected' );

			if ( JetSmartFilters.resetFilters ) {

				var $selectItem =  $( $select[0] );

				if ( $selectItem.data( 'hierarchical' ) && ! JetSmartFilters.resetHierarchy ) {
					JetSmartFilters.resetHierarchy = true;
					JetSmartFilters.getNextHierarchyLevelsHandler( $selectItem );
				}

				return;
			}

			if ( 'ajax' === applyType ) {
				$( $select[0] ).trigger( 'change.JetSmartFilters' );
			} else if ( 'ajax-reload' === applyType ) {
				JetSmartFilters.ajaxFilters( $select );
			}

		},

		removeSearch: function( event, $scope, data, JetSmartFilters, provider ) {

			var filterID = data.id,
				$filter  = JetSmartFilters.getFilterElement( filterID );

			$filter.find( 'input' ).val( '' );

			if( JetSmartFilters.resetFilters ){
				return;
			}

			$filter.find( '.jet-search-filter__submit' ).trigger( 'click.JetSmartFilters' );

		},

		unselectRating : function (){

			var $this = $(this);

			if ( $this.hasClass('is-checked') ){

				$this.attr('checked', false);
				$this.removeClass('is-checked');

				var applyType = $this.data( 'apply-type' );

				if ( 'ajax' === applyType ) {
					$this.trigger( 'change.JetSmartFilters' );
				}

			} else {
				$this.siblings().removeClass('is-checked');
				$this.addClass('is-checked');
			}

		},

		resetRating: function( provider, queryID ) {

			var $rating = $( 'input.jet-rating-star__input[data-apply-provider="' + provider + '"][data-query-id="' + queryID + '"]' );

			$rating.each( function() {
				if ( $(this).hasClass('is-checked') ){
					$(this).removeClass('is-checked');
				}
			} );

		},

		getFilterElement: function( filterID, selector ) {

			if ( ! selector ) {
				selector = 'div[data-filter-id="' + filterID + '"]';
			}

			var $el = $( selector );

			if ( ! $el.length ) {

				if ( window.elementorProFrontend && window.elementorFrontend ) {

					$.each( window.elementorFrontend.documentsManager.documents, function( index, elementorDocument ) {
						if ( 'popup' === elementorDocument.$element.data( 'elementor-type' ) ) {

							var $popupEl = elementorDocument.$element.find( selector );

							if ( $popupEl.length ) {
								$el = $popupEl;
							}
						}
					});

				}
			}

			return $el;

		},

		addQueryVarSuffix: function( queryVar, $scope ) {

			var queryVarSuffix = $scope.data( 'query-var-suffix' );

			if ( queryVarSuffix ) {
				queryVar = queryVar + '|' + queryVarSuffix;
			}

			return queryVar;
		},

		processSearch: function( event, $scope, queryVar, JetSmartFilters, queryType ) {

			queryVar = JetSmartFilters.addQueryVarSuffix( queryVar, $scope );

			if ( JetSmartFilters.hasRedirectData ){

				if ( JetSmartFilters.currentQuery[queryVar] ){
					$scope.find( 'input[type="search"]' ).val( JetSmartFilters.currentQuery[queryVar] );
				}

				return;
			}

			var val = $scope.find( 'input[type="search"]' ).val();

			if ( ! val ) {
				return;
			}

			if ( 'url' === queryType ) {

				JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
					queryVar,
					val,
					JetSmartFilters.currentQuery,
					'replace'
				);

			} else {
				JetSmartFilters.currentQuery[ queryVar ] = val;
			}

		},

		processRadio: function( event, $scope, queryVar, JetSmartFilters, queryType ) {

			queryVar = JetSmartFilters.addQueryVarSuffix( queryVar, $scope );

			if ( JetSmartFilters.hasRedirectData ){

				if ( JetSmartFilters.currentQuery[queryVar] && 'string' === typeof JetSmartFilters.currentQuery[queryVar] ){
					$scope.find( 'input[value="' + JetSmartFilters.currentQuery[ queryVar ] + '"]' ).attr( 'checked', true );
				}

				return;
			}

			var val = $scope.find( 'input:checked' ).val();

			if ( ! val ) {
				return;
			}

			if ( 'url' === queryType ) {

				JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
					queryVar,
					val,
					JetSmartFilters.currentQuery,
					'replace'
				);

			} else {
				JetSmartFilters.currentQuery[ queryVar ] = val;
			}

		},

		processRating: function( event, $scope, queryVar, JetSmartFilters, queryType ) {

			queryVar = JetSmartFilters.addQueryVarSuffix( queryVar, $scope );

			if ( JetSmartFilters.hasRedirectData ){

				if ( JetSmartFilters.currentQuery[ queryVar ] ){
					$scope.find( 'input[value="' + JetSmartFilters.currentQuery[ queryVar ] + '"]' ).attr( 'checked', true ).addClass('is-checked');
				}

				return;
			}

			var val = $scope.find( 'input:checked' ).addClass('is-checked').val();

			if ( ! val ) {
				return;
			}

			if ( 'url' === queryType ) {

				JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
					queryVar,
					val,
					JetSmartFilters.currentQuery,
					'replace'
				);

			} else {
				JetSmartFilters.currentQuery[ queryVar ] = val;
			}

		},

		processSelect: function( event, $scope, queryVar, JetSmartFilters, queryType ) {

			queryVar = JetSmartFilters.addQueryVarSuffix( queryVar, $scope );

			if ( JetSmartFilters.hasRedirectData ){

				if ( JetSmartFilters.currentQuery[queryVar] && 'string' === typeof JetSmartFilters.currentQuery[queryVar] ){
					$scope.find( 'option[value="' + JetSmartFilters.currentQuery[queryVar] + '"]' ).attr( 'selected', true );
				}

				return;
			}

			var val            = $scope.find( 'option:selected' ).val(),
				isHierarchical = $scope.data( 'hierarchical' );

			if ( ! val ) {
				return;
			}

			if ( 'url' === queryType ) {

				JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
					queryVar,
					val,
					JetSmartFilters.currentQuery,
					'replace'
				);

			} else {

				if ( JetSmartFilters.currentQuery[ queryVar ] && ! isHierarchical ) {

					if ( ! JetSmartFilters.isArray( JetSmartFilters.currentQuery[ queryVar ] ) ) {
						JetSmartFilters.currentQuery[ queryVar ] = [ JetSmartFilters.currentQuery[ queryVar ] ];
					}

					JetSmartFilters.currentQuery[ queryVar ].push( val );

				} else {
					JetSmartFilters.currentQuery[ queryVar ] = val;
				}

			}

		},

		isArray: function(o) {
			return Object.prototype.toString.call(o) === '[object Array]';
		},

		getAllHierarchyLevels: function( $scope, trail ) {

			if ( ! $scope.length || ! trail.length ) {
				return;
			}

			var toDepth  = trail.length - 1,
				filterId = $scope.data( 'filter-id' );

			$.ajax({
				url: JetSmartFilterSettings.ajaxurl,
				type: 'POST',
				dataType: 'json',
				data: {
					action: 'jet_smart_filters_get_hierarchy_level',
					values: trail,
					filter_id: $scope.data( 'filter-id' ),
					args: {
						apply_type: $scope.data( 'apply-type' ),
						content_provider: $scope.data( 'content-provider' ),
						query_id: $scope.data( 'query-id' ),
						layout_options: $scope.data( 'layout-options' ),
					}
				},
			}).done( function( response ) {
				for ( var j = 0; j <= toDepth + 1; j++ ) {
					var $item = $( 'div[data-hierarchy="' + filterId + '-' + j + '"]' );
					if ( $item.length ) {
						$item.replaceWith( response.data[ 'level_' + j ] );
					}
				}
			});

		},

		getNextHierarchyLevelsHandler: function( $el ) {

			var $scope     = $el.closest( 'div[data-hierarchical="1"]' ),
				$container = $el.closest( 'div.jet-filter' ),
				depth      = 0,
				newDepth   = 0,
				maxDepth   = 0,
				values     = [],
				filterId   = 0,
				indexer    = false,
				provider   = $scope.data( 'content-provider' ),
				queryID    = $scope.data( 'query-id' ),
				applyType  = $scope.data( 'apply-type' ),
				$item;

			if ( ! $scope.length ) {
				return;
			}

			depth    = parseInt( $scope.data( 'depth' ), 10 );
			newDepth = depth + 1;
			maxDepth = parseInt( $scope.data( 'max-depth' ), 10 );
			filterId = $scope.data( 'filter-id' );

			JetSmartFilters.currentHierarchy.filterID = filterId;
			JetSmartFilters.currentHierarchy.depth    = depth;

			for ( var i = 0; i <= depth; i++ ) {

				var filterSelector = 'div[data-hierarchy="' + filterId + '-' + i + '"]';

				$item = $( filterSelector );

				values.push( {
					value: $item.find( 'option:selected' ).val(),
					tax: $item.data( 'query-var' ),
					selector: filterSelector,
				} );

			}

			for ( var j = newDepth; j <= maxDepth; j++ ) {
				$item = $( 'div[data-hierarchy="' + filterId + '-' + j + '"]' );
				$item.addClass( 'jet-filters-loading' );
			}

			JetSmartFilters.currentHierarchy.trail = values;

			if ( 'yes' === $container.data( 'show-counter' ) && 'reload' === applyType ) {
				if ( JetSmartFilterSettings.queries[ provider ] ) {
					indexer = {
						query: JetSmartFilters.getQuery( 'object', provider ),
						defaults: JetSmartFilterSettings.queries[ provider ][ queryID ],
						provider: provider + '/' + queryID,
					};
				}
			}

			$.ajax({
				url: JetSmartFilterSettings.ajaxurl,
				type: 'POST',
				dataType: 'json',
				data: {
					action: 'jet_smart_filters_get_hierarchy_level',
					depth: newDepth,
					values: values,
					filter_id: $scope.data( 'filter-id' ),
					indexer: indexer,
					args: {
						apply_type: applyType,
						content_provider: provider,
						query_id: queryID,
						layout_options: $scope.data( 'layout-options' ),
					}
				},
			}).done( function( response ) {

				for ( var j = newDepth; j <= maxDepth; j++ ) {
					$item = $( 'div[data-hierarchy="' + filterId + '-' + j + '"]' );
					$item.replaceWith( response.data[ 'level_' + j ] );
				}

				$( document ).trigger(
					'jet-filter-hierarchy-updated',
					[ $scope, JetSmartFilters, response.data ]
				);

			});

		},

		getNextHierarchyLevels: function() {
			JetSmartFilters.getNextHierarchyLevelsHandler( $( this ) );
		},

		processRange: function( event, $scope, queryVar, JetSmartFilters, queryType ) {

			queryVar = JetSmartFilters.addQueryVarSuffix( queryVar, $scope );

			var val     = $scope.find( 'input[type="hidden"]' ).val(),
				$slider = $scope.find( '.jet-range__slider' ),
				values  = val.split( ':' );

			if ( JetSmartFilters.hasRedirectData ){

				if ( JetSmartFilters.currentQuery[ queryVar ] ) {

					values = JetSmartFilters.currentQuery[ queryVar ].split(':');

					$scope.find( 'input[type="hidden"]' ).val( JetSmartFilters.currentQuery[queryVar] );
					$scope.find( 'input.jet-date-range__from' ).val( values[0] );
					$scope.find( 'input.jet-date-range__to' ).val( values[1] );

					if ( $slider.length ) {
						var $min    = $scope.find( '.jet-range__values-min' ),
							$max    = $scope.find( '.jet-range__values-max' ),
							format  = $scope.data( 'format' );

						if ( ! format ) {
							format = {
								'thousands_sep' : '',
								'decimal_sep' : '',
								'decimal_num' : 0,
							};
						}

						$slider.on( 'jet-smart-filters/init-slider', function( event, sliderInstance ) {

							sliderInstance.slider( "values", this.values );

							$min.html( parseInt(this.values[ 0 ]).jetFormat(
								format.decimal_num,
								3,
								format.thousands_sep,
								format.decimal_sep
							) );

							$max.html( parseInt(this.values[ 1 ]).jetFormat(
								format.decimal_num,
								3,
								format.thousands_sep,
								format.decimal_sep
							) );

						}.bind( { values: values } ) );
					}
				}

				return;
			}

			if ( ! values[0] && ! values[1] ) {
				return;
			}

			// Prevent of adding slider defaults
			if ( $slider.length ) {

				var min = $slider.data( 'min' ),
					max = $slider.data( 'max' );

				if ( values[0] && values[0] == min && values[1] && values[1] == max ) {
					return;
				}

			}

			if ( ! val ) {
				return;
			}

			if ( 'url' === queryType ) {

				JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
					queryVar,
					val,
					JetSmartFilters.currentQuery,
					'replace'
				);

			} else {
				JetSmartFilters.currentQuery[ queryVar ] = val;
			}

		},

		processCheckbox: function( event, $scope, queryVar, JetSmartFilters, queryType ) {

			if ( JetSmartFilters.hasRedirectData ) {

				console.log( JetSmartFilters.currentQuery );
				console.log( queryVar );

				if ( JetSmartFilters.currentQuery[queryVar] && 'object' === typeof JetSmartFilters.currentQuery[queryVar] ){
					JetSmartFilters.currentQuery[queryVar].forEach( function( key ) {
						$scope.find( 'input[value="' + key + '"]' ).attr( 'checked', true );
					});
				}

				return;
			}

			queryVar = JetSmartFilters.addQueryVarSuffix( queryVar, $scope );

			if ( 'url' === queryType ) {
				queryVar = queryVar + '[]';
			}

			$scope.find( 'input:checked' ).each( function() {
				if ( 'url' === queryType ) {

					JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
						queryVar,
						$( this ).val(),
						JetSmartFilters.currentQuery,
						'append'
					);

				} else {

					if ( JetSmartFilters.currentQuery[ queryVar ] ) {
						JetSmartFilters.currentQuery[ queryVar ].push( $( this ).val() );
					} else {
						JetSmartFilters.currentQuery[ queryVar ] = [ $( this ).val() ];
					}
				}
			} );

		},

		applyPagination: function() {

			var $this      = $( this ),
				reloadType = $this.data( 'apply-type' ),
				queryID = $this.data( 'query-id' ),
				provider   = $this.data( 'apply-provider' );

			JetSmartFilters.page     = $this.data( 'page' );
			JetSmartFilters.controls = $this.closest( '.jet-smart-filters-pagination' ).data( 'controls' );

			if ( 'ajax' === reloadType ) {
				JetSmartFilters.ajaxFilters( $this );
			} else {
				JetSmartFilters.requestFilter( provider, queryID );
			}

		},

		applySearchFilterOnEnter: function( e ) {

			if ( 'keypress' === e.type && 13 === e.keyCode){
				var $this    = $( this ),
					provider = $this.data( 'apply-provider' ),
					applyType = $this.data( 'apply-type' ),
					queryID  = $this.data( 'query-id' );

				if ( 'ajax-reload' === applyType ){
					JetSmartFilters.ajaxFilters( $this );
				} else {
					JetSmartFilters.requestFilter( provider, queryID );
				}
			}

		},

		applyAjaxFilters: function() {
			var $this = $( this );

			if ( $this.hasClass( 'jet-remove-all-filters__button' ) ) {
				return;
			}

			JetSmartFilters.ajaxFilters( $( this ) );
		},

		refreshControls: function( provider, queryID ) {

			var query  = JetSmartFilters.getQuery( 'object', provider ),
				props  = null,
				paginationType = 'ajax',
				$pager = $( '.jet-smart-filters-pagination[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"]' );

			if ( xhr ) {
				xhr.abort();
			}

			if ( $pager.length ) {
				JetSmartFilters.controls = $pager.data( 'controls' );
				paginationType = $pager.data( 'apply-type' );
			}

			if ( JetSmartFilterSettings.props && JetSmartFilterSettings.props[ provider ] && JetSmartFilterSettings.props[ provider ][ queryID ] ) {
				props = JetSmartFilterSettings.props[ provider ][ queryID ];
			}

			var action = 'ajax' === paginationType ? 'jet_smart_filters_refresh_controls' : 'jet_smart_filters_refresh_controls_reload';

			xhr = JetSmartFilters.ajaxRequest(
				false,
				action,
				provider,
				query,
				props,
				queryID
			);

		},

		ajaxFilters: function( $scope ) {

			var provider = $scope.data( 'apply-provider' ),
				queryID = $scope.data( 'query-id' ),
				props = null,
				$pager = $( '.jet-smart-filters-pagination[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"]' ),
				query = {},
				applyRedirect = $scope.data( 'redirect' ),
				redirectPath = $scope.data( 'redirect-path' ),
				currentHierarchy = JetSmartFilters.currentHierarchy;

			if ( ! queryID ) {
				queryID = 'default';
			}

			query = JetSmartFilters.getQuery( 'object', provider, queryID );

			if ( 'yes' === applyRedirect ){
				var localStorageData = {};

				localStorageData.provider = provider;
				localStorageData.queryID = queryID;
				localStorageData.query = query;

				if ( $pager.length ) {
					localStorageData.controls = $pager.data( 'controls' );
				}

				if ( JetSmartFilterSettings.props && JetSmartFilterSettings.props[ provider ] && JetSmartFilterSettings.props[ provider ][ queryID ] ) {
					localStorageData.props = JetSmartFilterSettings.props[ provider ][ queryID ];
				}

				localStorageData.JetSmartFilterSettings = JetSmartFilterSettings;

				if ( currentHierarchy.trail.length ) {
					localStorageData.hierarchy = currentHierarchy;
				}

				localStorage.setItem( 'jet_smart_filters_query', JSON.stringify( localStorageData ) );
				document.location = redirectPath;

			} else {
				if ( xhr ) {
					xhr.abort();
				}

				if ( $pager.length ) {
					JetSmartFilters.controls = $pager.data( 'controls' );
				}

				$( document ).trigger(
					'jet-filter-load',
					[ $scope, JetSmartFilters, provider, query, queryID ]
				);

				if ( JetSmartFilterSettings.props && JetSmartFilterSettings.props[ provider ] && JetSmartFilterSettings.props[ provider ][ queryID ] ) {
					props = JetSmartFilterSettings.props[ provider ][ queryID ];
				}

				xhr = JetSmartFilters.ajaxRequest(
					$scope,
					'jet_smart_filters',
					provider,
					query,
					props,
					queryID
				);
			}

		},

		getPreparedFilters: function( provider, queryID, $scope ) {

			// Ensure requested provider filters is exists on the page
			if ( ! window.JetSmartFilterSettings ) {
				return;
			}

			if ( ! window.JetSmartFilterSettings.filters[ provider ] ) {
				return;
			}

			if ( ! window.JetSmartFilterSettings.filters[ provider ][ queryID ] ) {
				return;
			}

			var filters        = JetSmartFilterSettings.filters[ provider ][ queryID ],
				result         = {},
				isHierarchical = false,
				depth          = 0,
				changedFID     = 0;

			if ( $scope ) {
				$scope         = $scope.closest( 'div[data-content-provider="' + provider + '"]' );
				isHierarchical = $scope.data( 'hierarchical' );
				depth          = parseInt( $scope.data( 'depth' ), 10 );
				changedFID     = parseInt( $scope.data( 'filter-id' ), 10 );
			}

			if ( ! filters ) {
				return result;
			}

			$.each( filters, function( filterID, filter ) {

				var hierarchyValues = [],
					$item           = null;

				if ( filter.hierarchical ) {

					if ( JetSmartFilters.hasRedirectData ) {
						hierarchyValues = JetSmartFilters.currentHierarchy.trail;
					} else {
						for ( var i = 0; i <= filter.max_depth; i++ ) {

							$item = $( 'div[data-hierarchy="' + filterID + '-' + i + '"]' );

							if ( ( changedFID != filterID ) || ( isHierarchical && i <= depth ) ) {

								hierarchyValues.push( {
									value: $item.find( 'option:selected' ).val(),
									tax: $item.data( 'query-var' ),
								} );

							}

						}
					}

					filter['values'] = hierarchyValues;

				}

				result[ filterID ] = filter;

			} );

			return result;

		},

		ajaxRequest: function( $scope, action, provider, query, props, queryID ) {

			if ( ! queryID ) {
				queryID = 'default';
			}

			var defaults, settings, filters, controls;

			if ( JetSmartFilterSettings.queries[ provider ] ) {
				defaults = JetSmartFilterSettings.queries[ provider ][ queryID ];
			} else {
				defaults = {};
			}

			if ( JetSmartFilterSettings.settings[ provider ] ) {
				settings = JetSmartFilterSettings.settings[ provider ][ queryID ];
			} else {
				settings = {};
			}

			filters                  = JetSmartFilters.getPreparedFilters( provider, queryID, $scope );
			controls                 = JetSmartFilters.controls;
			JetSmartFilters.controls = false;

			$.ajax({
				url: JetSmartFilterSettings.ajaxurl,
				type: 'POST',
				dataType: 'json',
				data: {
					action: action,
					provider: provider + '/' + queryID,
					query: query,
					defaults: defaults,
					settings: settings,
					filters: filters,
					paged: JetSmartFilters.page,
					props: props,
					controls: controls,
				},
			}).done( function( response ) {
				if ( 'jet_smart_filters' === action ) {

					JetSmartFilters.renderResult( response, provider, queryID );

					$( document ).trigger(
						'jet-filter-before-loaded',
						[ $scope, JetSmartFilters, provider, query, queryID, response ]
					);

					$( document ).trigger(
						'jet-filter-loaded',
						[ $scope, JetSmartFilters, provider, query, queryID ]
					);

				} else {
					JetSmartFilters.renderActiveFilters( response.activeFilters, provider, queryID );
					JetSmartFilters.renderPagination( response.pagination, provider, queryID );
				}

				JetSmartFilters.page = false;
			});

		},

		renderResult: function( result, provider, queryID ) {

			if ( ! queryID ) {
				queryID = 'default';
			}

			var providerWrap = JetSmartFilterSettings.selectors[ provider ],
				$scope       = null;

			if ( 'default' === queryID ) {
				$scope = $( providerWrap.selector );
			} else {
				$scope = $( JetSmartFilters.providerSelector( providerWrap, queryID ) );
			}

			if ( 'insert' === providerWrap.action ) {
				$scope.html( result.content );
			} else {
				$scope.replaceWith( result.content );
			}

			JetSmartFilters.triggerElementorWidgets( $scope, provider );

			$( document ).trigger(
				'jet-filter-content-rendered',
				[ $scope, JetSmartFilters, provider, queryID ]
			);

			JetSmartFilters.renderActiveFilters( result.activeFilters, provider, queryID );
			JetSmartFilters.renderPagination( result.pagination, provider, queryID );

		},

		triggerElementorWidgets : function( $scope, provider ) {

			switch ( provider ) {

				case 'jet-engine':

				window.elementorFrontend.hooks.doAction(
					'frontend/element_ready/jet-listing-grid.default',
					$scope,
					$
				);

				break;

			}

			$scope.find( 'div[data-element_type]' ).each( function() {
				var $this       = $( this ),
					elementType = $this.data( 'element_type' );

				if( 'widget' === elementType ){
					elementType = $this.data( 'widget_type' );
					window.elementorFrontend.hooks.doAction( 'frontend/element_ready/widget', $this, $ );
				}

				window.elementorFrontend.hooks.doAction( 'frontend/element_ready/' + elementType, $this, $ );

			});

		},

		renderActiveFilters: function( html, provider, queryID ) {

			if ( ! queryID ) {
				queryID = 'default';
			}

			var $activeFiltersWrap = $( 'div.jet-active-filters[data-apply-provider="' + provider + '"][data-query-id="' + queryID + '"]' );

			if ( $activeFiltersWrap.length ) {
				$activeFiltersWrap.html( html );
				$activeFiltersWrap.find( '.jet-active-filters__title' ).html(
					$activeFiltersWrap.data( 'filters-label' )
				);
			}

		},

		renderPagination: function( html, provider, queryID ) {

			if ( ! queryID ) {
				queryID = 'default';
			}

			var $paginationWrap = $( 'div.jet-smart-filters-pagination[data-apply-provider="' + provider + '"][data-query-id="' + queryID + '"]' );

			if ( $paginationWrap.length ) {
				$paginationWrap.html( html );
			}

		},

		applyFilters: function() {
			var $this    = $( this ),
				provider = $this.data( 'apply-provider' ),
				hasRedirect = $this.data( 'redirect' ),
				redirectPath = $this.data( 'redirect-path' ),
				queryID  = $this.data( 'query-id' );

			JetSmartFilters.requestFilter( provider, queryID, hasRedirect, redirectPath );
		},

		requestFilter: function( provider, queryID, hasRedirect, redirectPath ) {

			var query = JetSmartFilters.getQuery( 'url', provider, queryID );

			if ( JetSmartFilters.page ) {
				query = JetSmartFilters.addQueryArg( 'jet_paged', JetSmartFilters.page, query, 'append' );
			}

			if ( 'yes' === hasRedirect && '' !== redirectPath ){
				document.location = redirectPath + '?' + query;
			} else {
				if ( JetSmartFilters.page ) {
					query = JetSmartFilters.addQueryArg( 'jet_paged', JetSmartFilters.page, query, 'append' );
				}

				document.location.search = query;
			}

		},

		buildQueryID: function( providerID ) {

			if ( JetSmartFilters.currentHierarchy.filterID ) {
				providerID += ';' + JetSmartFilters.currentHierarchy.filterID + ';';

				for ( var i = 0; i < JetSmartFilters.currentHierarchy.trail.length; i++ ) {
					let level = JetSmartFilters.currentHierarchy.trail[ i ];

					if ( 0 < i ) {
						providerID += '|';
					}

					providerID += level.tax + '=' + level.value;

				}

			}

			return providerID;
		},

		getQuery: function( type, provider, queryID ) {

			var query = null;

			if ( ! queryID ) {
				queryID = 'default';
			}

			JetSmartFilters.currentQuery = null;

			if ( 'url' === type ) {
				JetSmartFilters.currentQuery = JetSmartFilters.addQueryArg(
					'jet-smart-filters',
					JetSmartFilters.buildQueryID( provider + '/' + queryID ),
					null,
					'append'
				);
			} else {
				JetSmartFilters.currentQuery = {};
			}

			$( 'div[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"]' ).each( function() {

				var $this          = $( this ),
					queryType      = $this.data( 'query-type' ),
					queryVar       = $this.data( 'query-var' ),
					filterType     = $this.data( 'smart-filter' ),
					filterID       = parseInt( $this.data( 'filter-id' ), 10 ),
					key            = '_' + queryType + '_' + queryVar,
					skip           = false;

				if ( JetSmartFilters.currentHierarchy.filterID === filterID ) {
					var filterDepth = parseInt( $this.data( 'depth' ), 10 );
					if ( filterDepth > JetSmartFilters.currentHierarchy.depth ) {
						skip = true;
					}
				}

				if ( ! skip ) {
					$( document ).trigger(
						'jet-filter-add-' + filterType + '-vars',
						[ $this, key, JetSmartFilters, type ]
					);
				}

			} );

			if ( window.elementorProFrontend && window.elementorFrontend ) {

				$.each( window.elementorFrontend.documentsManager.documents, function( index, elementorDocument ) {
					if ( 'popup' === elementorDocument.$element.data( 'elementor-type' ) ) {
						elementorDocument.$element.find( 'div[data-content-provider="' + provider + '"][data-query-id="' + queryID + '"]' ).each( function() {
							var $this          = $( this ),
								queryType      = $this.data( 'query-type' ),
								queryVar       = $this.data( 'query-var' ),
								filterType     = $this.data( 'smart-filter' ),
								filterID       = parseInt( $this.data( 'filter-id' ), 10 ),
								key            = '_' + queryType + '_' + queryVar,
								skip           = false;

							if ( JetSmartFilters.currentHierarchy.filterID === filterID ) {
								var filterDepth = parseInt( $this.data( 'depth' ), 10 );
								if ( filterDepth > JetSmartFilters.currentHierarchy.depth ) {
									skip = true;
								}
							}

							if ( ! skip ) {
								$( document ).trigger(
									'jet-filter-add-' + filterType + '-vars',
									[ $this, key, JetSmartFilters, type ]
								);
							}

						} );
					}
				});

			}

			query = JetSmartFilters.currentQuery;

			return query;
		},

		addQueryArg: function( key, value, query, action ) {

			key   = encodeURI( key );
			value = encodeURI( value );

			if ( ! query ) {
				query = '';
			}

			var kvp = query.split( '&' );

			if ( 'append' === action ) {

				kvp[ kvp.length ] = [ key, value ].join( '=' );

			} else {

				var i = kvp.length;
				var x;

				while ( i-- ) {
					x = kvp[ i ].split( '=' );

					if ( x[0] == key ) {
						x[1]     = value;
						kvp[ i ] = x.join( '=' );

						break;
					}
				}

				if ( i < 0 ) {
					kvp[ kvp.length ] = [ key, value ].join( '=' );
				}

			}

			return kvp.join( '&' );
		}
	};

	var JSFEProCompat = {

		archivePostsClass: '.elementor-widget-archive-posts',
		defaultPostsClass: '.elementor-widget-posts',
		postsSettings: {},
		skin: 'archive_classic',

		init: function() {
			$( document ).on( 'jet-filter-content-rendered', function( event, $scope, JetSmartFilters, provider ) {

				if ( 'epro-archive' === provider || 'epro-posts' === provider ) {

					var postsSelector = JSFEProCompat.defaultPostsClass,
						$archive = null,
						widgetName = 'posts',
						hasMasonry = false;

					if ( 'epro-archive' === provider ) {
						postsSelector = JSFEProCompat.archivePostsClass;
						widgetName = 'archive-posts';
					}

					$archive = $( $scope.selector ).parent( postsSelector );

					JSFEProCompat.fitImages( $archive );

					JSFEProCompat.postsSettings = $archive.data( 'settings' );

					if( 'widget' === $archive.data( 'element_type' ) ){
						JSFEProCompat.skin = $archive.data( 'widget_type' );
					} else {
						JSFEProCompat.skin = $archive.data( 'element_type' );
					}

					JSFEProCompat.skin = JSFEProCompat.skin.split( widgetName + '.' );
					JSFEProCompat.skin = JSFEProCompat.skin[1];

					hasMasonry = JSFEProCompat.postsSettings[ JSFEProCompat.skin + '_masonry' ];

					if ( 'yes' === hasMasonry ) {
						setTimeout( JSFEProCompat.initMasonry( $archive ), 0 );
					}

				}

			} );
		},

		initMasonry: function( $archive ) {

			var $container = $archive.find( '.elementor-posts-container' ),
				$posts     = $container.find( '.elementor-post' ),
				settings   = JSFEProCompat.postsSettings,
				colsCount  = 1,
				hasMasonry = true;

			$posts.css({
				marginTop: '',
				transitionDuration: ''
			});

			var currentDeviceMode = window.elementorFrontend.getCurrentDeviceMode();

			switch ( currentDeviceMode ) {
				case 'mobile':
					colsCount = settings[ JSFEProCompat.skin + '_columns_mobile' ];
					break;
				case 'tablet':
					colsCount = settings[ JSFEProCompat.skin + '_columns_tablet' ];
					break;
				default:
					colsCount = settings[ JSFEProCompat.skin + '_columns' ];
			}

			hasMasonry = colsCount >= 2;

			$container.toggleClass( 'elementor-posts-masonry', hasMasonry );

			if ( ! hasMasonry ) {
				$container.height('');
				return;
			}

			var verticalSpaceBetween = settings[ JSFEProCompat.skin + '_row_gap' ]['size'];

			if ( ! verticalSpaceBetween ) {
				verticalSpaceBetween = settings[ JSFEProCompat.skin + '_item_gap' ]['size'];
			}

			var masonry = new elementorModules.utils.Masonry({
				container: $container,
				items: $posts.filter( ':visible' ),
				columnsCount: colsCount,
				verticalSpaceBetween: verticalSpaceBetween
			});

			masonry.run();
		},

		fitImage: function( $post ) {
			var $imageParent = $post.find( '.elementor-post__thumbnail' ),
				$image       = $imageParent.find( 'img' ),
				image        = $image[0];

			if ( ! image ) {
				return;
			}

			var imageParentRatio = $imageParent.outerHeight() / $imageParent.outerWidth(),
				imageRatio       = image.naturalHeight / image.naturalWidth;

			$imageParent.toggleClass( 'elementor-fit-height', imageRatio < imageParentRatio );
		},

		fitImages: function( $scope ) {
			var $element  = $scope,
				itemRatio = getComputedStyle( $element[0], ':after' ).content;

			$element.find( '.elementor-posts-container' ).toggleClass( 'elementor-has-item-ratio', !!itemRatio.match(/\d/) );

			$element.find( '.elementor-post' ).each( function () {
				var $post = $(this),
					$image = $post.find( '.elementor-post__thumbnail img' );

				JSFEProCompat.fitImage($post);

				$image.on( 'load', function () {
					JSFEProCompat.fitImage( $post );
				});
			} );
		},
	};

	JSFEProCompat.init();

	var JetSmartFiltersUI = {

		init: function() {

			var widgets = {
				'jet-smart-filters-range.default' : JetSmartFiltersUI.range,
				'jet-smart-filters-date-range.default' : JetSmartFiltersUI.dateRange
			};

			$.each( widgets, function( widget, callback ) {
				window.elementorFrontend.hooks.addAction( 'frontend/element_ready/' + widget, callback );
			});
		},

		range: function( $scope ) {
			$scope.find( '.jet-range' ).each( function() {
				var $el = $( this );
				JetSmartFiltersUI.rangeHandler( $el );
			} );
		},

		rangeHandler: function( $scope ) {

			var $slider = $scope.find( '.jet-range__slider' ),
				$input  = $scope.find( '.jet-range__input' ),
				$min    = $scope.find( '.jet-range__values-min' ),
				$max    = $scope.find( '.jet-range__values-max' ),
				format  = $slider.data( 'format' ),
				slider;

			if ( ! format ) {
				format = {
					'thousands_sep' : '',
					'decimal_sep' : '',
					'decimal_num' : 0,
				};
			}

			slider = $slider.slider({
				range: true,
				min: $slider.data( 'min' ),
				max: $slider.data( 'max' ),
				step: $slider.data( 'step' ),
				values: $slider.data( 'defaults' ),
				slide: function( event, ui ) {
					$input.val( ui.values[ 0 ] + ':' + ui.values[ 1 ] );

					$min.html( ui.values[ 0 ].jetFormat(
						format.decimal_num,
						3,
						format.thousands_sep,
						format.decimal_sep
					) );

					$max.html( ui.values[ 1 ].jetFormat(
						format.decimal_num,
						3,
						format.thousands_sep,
						format.decimal_sep
					) );

				},
				stop: function( event, ui ) {
					$input.trigger( 'change' );
				},
			});

			$slider.trigger( 'jet-smart-filters/init-slider', [ slider ] );

		},

		dateRange: function( $scope ) {

			var $id = $scope.data('id'),
				$from  = $scope.find( '.jet-date-range__from' ),
				$to    = $scope.find( '.jet-date-range__to' ),
				$input = $scope.find( '.jet-date-range__input' ),
				from,
				$texts = JetSmartFilterSettings.datePickerData,
				to;

			from = $from.datepicker({
				defaultDate: '+1w',
				closeText: $texts.closeText,
				prevText: $texts.prevText,
				nextText: $texts.nextText,
				currentText: $texts.currentText,
				monthNames: $texts.monthNames,
				monthNamesShort: $texts.monthNamesShort,
				dayNames: $texts.dayNames,
				dayNamesShort: $texts.dayNamesShort,
				dayNamesMin: $texts.dayNamesMin,
				weekHeader: $texts.weekHeader,
				firstDay: parseInt( JetSmartFilterSettings.misc.week_start, 10 ),
				beforeShow: function (textbox, instance) {
					var $calendar = instance.dpDiv;
					$calendar.addClass('jet-smart-filters-datepicker-' + $id );
				},
				onClose: function (textbox, instance){
					var $calendar = instance.dpDiv;
					$calendar.removeClass('jet-smart-filters-datepicker-' + $id );
				}
			}).on( 'change', function() {
				to.datepicker( 'option', 'minDate', JetSmartFiltersUI.getDate( this ) );
				$input.val( $from.val() + ':' + $to.val() );
			});

			to = $to.datepicker({
				defaultDate: '+1w',
				closeText: $texts.closeText,
				prevText: $texts.prevText,
				nextText: $texts.nextText,
				currentText: $texts.currentText,
				monthNames: $texts.monthNames,
				monthNamesShort: $texts.monthNamesShort,
				dayNames: $texts.dayNames,
				dayNamesShort: $texts.dayNamesShort,
				dayNamesMin: $texts.dayNamesMin,
				weekHeader: $texts.weekHeader,
				firstDay: parseInt( JetSmartFilterSettings.misc.week_start, 10 ),
				beforeShow: function (textbox, instance) {
					var $calendar = instance.dpDiv;
					$calendar.addClass('jet-smart-filters-datepicker-' + $id );
				},
				onClose: function (textbox, instance){
					var $calendar = instance.dpDiv;
					$calendar.removeClass('jet-smart-filters-datepicker-' + $id );
				}
			}).on( 'change', function() {
				from.datepicker( 'option', 'maxDate', JetSmartFiltersUI.getDate( this ) );
				$input.val( $from.val() + ':' + $to.val() );
			});
		},

		getDate: function( element ) {

			var dateFormat = 'mm/dd/yy',
				date;

			try {
				date = $.datepicker.parseDate( dateFormat, element.value );
			} catch ( error ) {
				date = null;
			}

			return date;
		}

	};

	JetSmartFilters.init();

	$( window ).on( 'elementor/frontend/init', JetSmartFiltersUI.init );

	window.JetSmartFilters = JetSmartFilters;

}( jQuery ) );
