<?php

defined('ABSPATH') || exit;

use iRecco_Theme_Helper as iRecco;

/**
 * iRecco Dynamic Styles
 *
 *
 * @package irecco\core\class
 * @author WebGeniusLab <webgeniuslab@gmail.com>
 * @since 1.0.0
 * @version 1.1.11
 */
class iRecco_dynamic_styles
{
    protected static $instance;

    private $theme_slug;
    private $template_directory_uri;
    private $use_minify;
    private $header_page_id;

    public function __construct()
    {
        $this->theme_slug = $this->get_theme_slug();
        $this->template_directory_uri = get_template_directory_uri();
        $this->use_minify = iRecco::get_option('use_minify') ? '.min' : '';
        $this->header_type = iRecco::get_option('header_type');

        $this->enqueue_styles_and_scripts();
        add_action('wp_enqueue_scripts', [$this, 'add_style']);
    }

    public function get_theme_slug()
    {
        return str_replace('-child', '', wp_get_theme()->get('TextDomain'));
    }

    public function enqueue_styles_and_scripts()
    {
        //* Elementor Compatibility
        add_action('wp_enqueue_scripts', [$this, 'get_elementor_css_theme_builder']);
        add_action('wp_enqueue_scripts', [$this, 'elementor_column_fix']);

        add_action('wp_enqueue_scripts', [$this, 'frontend_stylesheets']);
        add_action('wp_enqueue_scripts', [$this, 'frontend_scripts']);

        add_action('admin_enqueue_scripts', [$this, 'admin_stylesheets']);
        add_action('admin_enqueue_scripts', [$this, 'admin_scripts']);
    }

    public function get_elementor_css_theme_builder()
    {
        $current_post_id = get_the_ID();
        $css_files = [];

        $locations[] = $this->get_elementor_css_cache_header();
        $locations[] = $this->get_elementor_css_cache_header_sticky();
        $locations[] = $this->get_elementor_css_cache_footer();
        $locations[] = $this->get_elementor_css_cache_side_panel();

        foreach ($locations as $location) {
            //* Don't enqueue current post here (let the preview/frontend components to handle it)
            if ($location && $current_post_id !== $location) {
                $css_file = new \Elementor\Core\Files\CSS\Post($location);
                $css_files[] = $css_file;
            }
        }

        if (!empty($css_files)) {
            \Elementor\Plugin::$instance->frontend->enqueue_styles();
            foreach ($css_files as $css_file) {
                $css_file->enqueue();
            }
        }
    }

    public function get_elementor_css_cache_header()
    {
        if (!class_exists('\Elementor\Core\Files\CSS\Post')) {
            // Bailtout.
            return;
        }

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta('mb_customize_header_layout')
            && 'default' !== rwmb_meta('mb_header_content_type')
        ) {
            $this->header_type = 'custom';
            $this->header_page_id = rwmb_meta('mb_customize_header');
        } else {
            $this->header_page_id = iRecco::get_option('header_page_select');
        }

        if ('custom' === $this->header_type) {
            return $this->multi_language_support($this->header_page_id, 'header');
        }
    }

    /**
     * @version 1.1.10
     */
    public function get_elementor_css_cache_header_sticky()
    {
        if (
            ! iRecco::get_mb_option( 'header_sticky', 'mb_customize_header_layout', 'custom' )
            || 'custom' !== $this->header_type
            || ! class_exists( '\Elementor\Core\Files\CSS\Post' )
        ) {
            // Bailtout.
            return;
        }

        $header_sticky_page_id = '';

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta( 'mb_customize_header_layout' )
            && 'default' !== rwmb_meta( 'mb_sticky_header_content_type' )
        ) {
            $header_sticky_page_id = rwmb_meta( 'mb_customize_sticky_header' );
        } elseif ( iRecco::get_option( 'header_sticky' ) ) {
            $header_sticky_page_id = iRecco::get_option( 'header_sticky_page_select' );
        }

        return $this->multi_language_support( $header_sticky_page_id, 'header' );
    }

    public function get_elementor_css_cache_footer()
    {
        if ($this->RWMB_is_active()) {
            $mb_footer_switch = rwmb_meta('mb_footer_switch');
            switch ($mb_footer_switch) {
                case 'on':
                    $footer_switch = true;
                    break;
                case 'off':
                    $footer_switch = false;
                    break;
            }
        } elseif (
            is_404()
            && !(bool) iRecco::get_option('404_show_footer')
        ) {
            //* hide on 404 Page
            $footer_switch = false;
        }

        $footer_switch = $footer_switch ?? iRecco::get_option('footer_switch');

        if (
            !$footer_switch
            || 'pages' !== iRecco::get_mb_option('footer_content_type', 'mb_footer_switch', 'on')
            || !class_exists('\Elementor\Core\Files\CSS\Post')
        ) {
            // Bailout.
            return;
        }

        $footer_page_id = iRecco::get_mb_option('footer_page_select', 'mb_footer_switch', 'on');

        return $this->multi_language_support($footer_page_id, 'footer');
    }

    public function get_elementor_css_cache_side_panel()
    {
        if (
            !iRecco::get_option('side_panel_enable')
            || 'pages' !== iRecco::get_mb_option('side_panel_content_type', 'mb_customize_side_panel', 'custom')
            || !class_exists('\Elementor\Core\Files\CSS\Post')
        ) {
            // Bailout.
            return;
        }

        $sp_page_id = iRecco::get_mb_option('side_panel_page_select', 'mb_customize_side_panel', 'custom');

        return $this->multi_language_support($sp_page_id, 'side_panel');
    }

    public function multi_language_support($page_id, $page_type)
    {
        if (!$page_id) {
            // Bailout.
            return;
        }

        $page_id = intval($page_id);

        if (class_exists('Polylang') && function_exists('pll_current_language')) {
            $currentLanguage = pll_current_language();
            $translations = PLL()->model->post->get_translations($page_id);

            $polylang_id = $translations[$currentLanguage] ?? '';
            $page_id = $polylang_id ?: $page_id;
        }

        if (class_exists('SitePress')) {
            $page_id = wpml_object_id_filter($page_id, $page_type, false, ICL_LANGUAGE_CODE);
        }

        return $page_id;
    }

    public function elementor_column_fix()
    {
        $css = '.elementor-column-gap-default > .elementor-column > .elementor-element-populated {'
                . 'padding-left: 15px;'
                . 'padding-right: 15px;'
            . '}';

        wp_add_inline_style('elementor-frontend', $css);
    }

    /**
     * @since 1.0.0
     * @version 1.1.2
     */
    public function frontend_stylesheets()
    {
        wp_enqueue_style($this->theme_slug . '-theme-info', get_bloginfo('stylesheet_url'));

        $this->enqueue_additional_styles();
        wp_enqueue_style('irecco-main', $this->template_directory_uri . '/css/main'.$this->use_minify.'.css');

	    // Rtl css
	    if (is_rtl()) {
		    wp_enqueue_style('irecco-rtl', $this->template_directory_uri . '/css/rtl' . $this->use_minify . '.css' );
	    }
    }

    /**
     * @since 1.1.2
     */
    public function enqueue_additional_styles()
    {
        wp_enqueue_style('font-awesome', $this->template_directory_uri . '/css/font-awesome.min.css');
        wp_enqueue_style('flaticon', $this->template_directory_uri . '/fonts/flaticon/flaticon.css');
    }

    /**
     * @since 1.0.0
     */
    public function frontend_scripts()
    {
        wp_enqueue_script('irecco-theme-addons', $this->template_directory_uri . '/js/theme-addons'.$this->use_minify.'.js', ['jquery'], false, true);
        wp_enqueue_script('irecco-theme', $this->template_directory_uri . '/js/theme.js', ['jquery'], false, true);

        wp_localize_script('irecco-theme', 'wgl_core', [
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'slickSlider' => esc_url(get_template_directory_uri() . '/js/slick.min.js'),
            'JarallaxPlugin' => esc_url(get_template_directory_uri() . '/js/jarallax-video.min.js'),
            'JarallaxPluginVideo' => esc_url(get_template_directory_uri() . '/js/jarallax.min.js'),
            'like' => esc_html__('Like', 'irecco'),
            'unlike' => esc_html__('Unlike', 'irecco')
        ]);

        if (is_singular() && comments_open() && get_option('thread_comments')) {
            wp_enqueue_script('comment-reply');
        }

        wp_enqueue_script('perfect-scrollbar', get_template_directory_uri() . '/js/perfect-scrollbar.min.js');
    }

    /**
     * @since 1.0.0
     */
    public function admin_stylesheets()
    {
        wp_enqueue_style('irecco-admin', $this->template_directory_uri . '/core/admin/css/admin.css');
        wp_enqueue_style('font-awesome', $this->template_directory_uri . '/css/font-awesome.min.css');
        wp_enqueue_style('wp-color-picker');
    }

    /**
     * @since 1.0.0
     */
    public function admin_scripts()
    {
        wp_enqueue_media();

        wp_enqueue_script('wp-color-picker');
        wp_localize_script('wp-color-picker', 'wpColorPickerL10n', [
            'clear' => esc_html__('Clear', 'irecco'),
            'clearAriaLabel' => esc_html__('Clear color', 'irecco'),
            'defaultString' => esc_html__('Default', 'irecco'),
            'defaultAriaLabel' => esc_html__('Select default color', 'irecco'),
            'pick' => esc_html__('Select', 'irecco'),
            'defaultLabel' => esc_html__('Color value', 'irecco'),
        ]);

        wp_enqueue_script('irecco-admin', $this->template_directory_uri . '/core/admin/js/admin.js');

        if (class_exists('RWMB_Loader')) {
            wp_enqueue_script('irecco-metaboxes', $this->template_directory_uri . '/core/admin/js/metaboxes.js');
        }

        $currentTheme = wp_get_theme();
        $theme_name = $currentTheme->parent() == false
            ? wp_get_theme()->get('Name')
            : wp_get_theme()->parent()->get('Name');
        $theme_name = trim($theme_name);

        $purchase_code = $email = '';
        if (iRecco::wgl_theme_activated()) {
            $theme_details = get_option('wgl_licence_validated');
            $purchase_code = $theme_details['purchase'] ?? '';
            $email = $theme_details['email'] ?? '';
        }

        wp_localize_script('irecco-admin', 'wgl_verify', [
            'ajaxurl' => esc_js(admin_url('admin-ajax.php')),
            'wglUrlActivate' => esc_js(Wgl_Theme_Verify::get_instance()->api . 'verification'),
            'wglUrlDeactivate' => esc_js(Wgl_Theme_Verify::get_instance()->api . 'deactivate'),
            'domainUrl' => esc_js(site_url('/')),
            'themeName' => esc_js($theme_name),
            'purchaseCode' => esc_js($purchase_code),
            'email' => esc_js($email),
            'message' => esc_js(esc_html__('Thank you, your license has been validated', 'irecco')),
            'ajax_nonce' => esc_js(wp_create_nonce('_notice_nonce'))
        ]);
    }


    public function minify_css($css = null)
    {
        if (!$css) { return; }

        // Combine css
        $css = str_replace( ', ', ',', $css );

        // Remove comments
        $css = preg_replace( '!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css );

        // Remove space after colons
        $css = str_replace(': ', ':', $css);

        // Remove whitespace
        $css = str_replace( array("\r\n", "\r", "\n", "\t", '  ', '    ', '    '), '', $css );
        $css = trim( $css );

        return $css;
    }

    /**
     * @version 1.1.11
     */
    public function add_style()
    {
        $css = '';

        /**
         * Color Variables
         */
        if (
            class_exists( 'RWMB_Loader' )
            && 'custom' === iRecco::get_mb_option( 'page_colors_switch' )
        ) {
            $theme_primary_color = esc_attr( iRecco::get_mb_option( 'page_theme_color' ) );
            $theme_secondary_color = esc_attr( iRecco::get_mb_option( 'theme-secondary-color' ) );

            $bg_body = esc_attr( iRecco::get_mb_option( 'body_background_color' ) );

            $scroll_up_bg_color = esc_attr( iRecco::get_mb_option( 'scroll_up_bg_color' ) );
            $scroll_up_arrow_color = esc_attr( iRecco::get_mb_option( 'scroll_up_arrow_color' ) );
        } else {
            $theme_primary_color = esc_attr( iRecco::get_option( 'theme-primary-color' ) );
            $theme_secondary_color = esc_attr( iRecco::get_option( 'theme-secondary-color' ) );

            $bg_body = esc_attr( iRecco::get_option( 'body-background-color' ) );

            $scroll_up_bg_color = esc_attr( iRecco::get_option( 'scroll_up_bg_color' ) );
            $scroll_up_arrow_color = esc_attr( iRecco::get_option( 'scroll_up_arrow_color' ) );
        }
        //* ↑ color variables

        /*-----------------------------------------------------------------------------------*/
        /* Body Add Class
        /*-----------------------------------------------------------------------------------*/
        if (defined( 'ELEMENTOR_VERSION' ) && version_compare( ELEMENTOR_VERSION, '3.0', '>=' )) {
            if(
                empty(get_option( 'elementor_element_wrappers_legacy_mode' ))
                || \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_dom_optimization' )
            ){
                add_filter( 'body_class', function( $classes ) {
                    return array_merge( $classes, array( 'new-elementor' ) );
                } );
            }
        }
        /*-----------------------------------------------------------------------------------*/
        /* End Body Add Class
        /*-----------------------------------------------------------------------------------*/

        /**
         * Headings Variables
         */
        $header_font = iRecco::get_option('header-font');

        $header_font_family = $header_font_weight = $header_font_color = '';
        if (!empty($header_font)) {
            $header_font_family = esc_attr( $header_font['font-family']);
            $header_font_weight = esc_attr( $header_font['font-weight']);
            $header_font_color = esc_attr( $header_font['color']);
        }

        // Add Heading h1,h2,h3,h4,h5,h6 variables
        for ( $i = 1; $i <= 6; $i++) {
            ${'header-h'.$i} = iRecco::get_option('header-h'.$i);
            ${'header-h'.$i.'_family'} = ${'header-h'.$i.'_weight'} = ${'header-h'.$i.'_line_height'} = ${'header-h'.$i.'_size'} = ${'header-h'.$i.'_text_transform'} = '';

            if (!empty(${'header-h'.$i})) {
                ${'header-h'.$i.'_family'} = !empty(${'header-h'.$i}["font-family"]) ? esc_attr( ${'header-h'.$i}["font-family"]) : '';
                ${'header-h'.$i.'_weight'} = !empty(${'header-h'.$i}["font-weight"]) ? esc_attr( ${'header-h'.$i}["font-weight"]) : '';
                ${'header-h'.$i.'_line_height'} = !empty(${'header-h'.$i}["line-height"]) ? esc_attr( ${'header-h'.$i}["line-height"]) : '';
                ${'header-h'.$i.'_size'} = !empty(${'header-h'.$i}["font-size"]) ? esc_attr( ${'header-h'.$i}["font-size"]) : '';
                ${'header-h'.$i.'_text_transform'} = !empty(${'header-h'.$i}["text-transform"]) ? esc_attr( ${'header-h'.$i}["text-transform"]) : '';
            }
        }
        //* ↑ headings variables

        /**
         * Content Variables
         */
        $main_font = iRecco::get_option('main-font');
        $content_font_family = $content_line_height = $content_font_size = $content_font_weight = $content_color = '';
        if (!empty($main_font)) {
            $content_font_family = esc_attr( $main_font['font-family']);
            $content_font_size = esc_attr( $main_font['font-size']);
            $content_font_weight = esc_attr( $main_font['font-weight']);
            $content_color = esc_attr( $main_font['color']);
            $content_line_height = esc_attr( $main_font['line-height']);
            $content_line_height = !empty($content_line_height) ? round(((int)$content_line_height / (int)$content_font_size), 3) : '';
        }
        //* ↑ content variables

        /**
         * Menu Variables
         */
        $menu_font = iRecco::get_option('menu-font');
        $menu_font_family = $menu_font_weight = $menu_font_line_height = $menu_font_size = '';
        if (!empty($menu_font)) {
            $menu_font_family = !empty($menu_font['font-family']) ? esc_attr( $menu_font['font-family']) : '';
            $menu_font_weight = !empty($menu_font['font-weight']) ? esc_attr( $menu_font['font-weight']) : '';
            $menu_font_line_height = !empty($menu_font['line-height']) ? esc_attr( $menu_font['line-height']) : '';
            $menu_font_size = !empty($menu_font['font-size']) ? esc_attr( $menu_font['font-size']) : '';
        }
        //* ↑ menu variables

        /**
         * Submenu Variables
         */
        $sub_menu_font = iRecco::get_option('sub-menu-font');
        $sub_menu_font_family = $sub_menu_font_weight = $sub_menu_font_line_height = $sub_menu_font_size = '';
        if (!empty($sub_menu_font)) {
            $sub_menu_font_family = !empty($sub_menu_font['font-family']) ? esc_attr( $sub_menu_font['font-family']) : '';
            $sub_menu_font_weight = !empty($sub_menu_font['font-weight']) ? esc_attr( $sub_menu_font['font-weight']) : '';
            $sub_menu_font_line_height = !empty($sub_menu_font['line-height']) ? esc_attr( $sub_menu_font['line-height']) : '';
            $sub_menu_font_size = !empty($sub_menu_font['font-size']) ? esc_attr( $sub_menu_font['font-size']) : '';
        }
        //* ↑ submenu variables

        /**
         * Additional Font Variables
         */
        $additional_font = iRecco::get_option('additional-font');
        if (!empty($additional_font)) {
            $additional_font_family = esc_attr($additional_font['font-family']);
            $additional_font_weight = esc_attr((int)$additional_font['font-weight']);
        } else {
            $additional_font_family = $additional_font_weight = '';
        }
        //* ↑ additional font variables

        $menu_color_top = iRecco::get_option('header_top_color');
        if (!empty($menu_color_top['rgba'])) {
            $menu_color_top = !empty($menu_color_top['rgba']) ? esc_attr( $menu_color_top['rgba']) : '';
        }

        $menu_color_middle = iRecco::get_option('header_middle_color');
        if (!empty($menu_color_middle['rgba'])) {
            $menu_color_middle = !empty($menu_color_middle['rgba']) ? esc_attr( $menu_color_middle['rgba']) : '';
        }

        $menu_color_bottom = iRecco::get_option('header_bottom_color');
        if (!empty($menu_color_bottom['rgba'])) {
            $menu_color_bottom = !empty($menu_color_bottom['rgba']) ? esc_attr( $menu_color_bottom['rgba']) : '';
        }

        // Set Queries width to apply mobile style
        $sub_menu_color = iRecco::get_option('sub_menu_color');
        $sub_menu_bg = iRecco::get_option('sub_menu_background');
        $sub_menu_bg = $sub_menu_bg['rgba'];

        $sub_menu_border = iRecco::get_option('header_sub_menu_bottom_border');
        $sub_menu_border_height = iRecco::get_option('header_sub_menu_border_height');
        $sub_menu_border_height = $sub_menu_border_height['height'];
        $sub_menu_border_color = iRecco::get_option('header_sub_menu_bottom_border_color');
        if (!empty($sub_menu_border)) {
            $css .= '.primary-nav ul li ul li:not(:last-child), .sitepress_container > .wpml-ls ul ul li:not(:last-child) {'
                . (!empty($sub_menu_border_height) ? 'border-bottom-width: ' . (int) (esc_attr($sub_menu_border_height)) . 'px;' : '')
                . (!empty($sub_menu_border_color['rgba']) ? 'border-bottom-color: ' . esc_attr($sub_menu_border_color['rgba']) . ';' : '')
                . 'border-bottom-style: solid;'
            . '}';
        }

        $mobile_sub_menu_bg = iRecco::get_option('mobile_sub_menu_background')['rgba'];

        $mobile_sub_menu_overlay = iRecco::get_option('mobile_sub_menu_overlay')['rgba'];

        $mobile_sub_menu_color = iRecco::get_option('mobile_sub_menu_color');

        $rgb_primary_color = iRecco::HexToRGB( $theme_primary_color );
        $rgb_secondary_color = iRecco::HexToRGB( $theme_secondary_color );
        $rgb_scroll_up_bg = iRecco::HexToRGB( $scroll_up_bg_color );

        $footer_text_color = iRecco::get_option('footer_text_color');
        $footer_heading_color = iRecco::get_option('footer_heading_color');

        $copyright_text_color = iRecco::get_mb_option('copyright_text_color','mb_copyright_switch','on');

        /*-----------------------------------------------------------------------------------*/
        /* Side Panel Css
        /*-----------------------------------------------------------------------------------*/
        $side_panel_title = iRecco::get_option('side_panel_title_color');
        $side_panel_title = !empty($side_panel_title['rgba']) ? $side_panel_title['rgba'] : '';

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta('mb_customize_side_panel')
        ) {
            $side_panel_title = rwmb_meta("mb_side_panel_title_color");
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Side Panel CSS
        /*-----------------------------------------------------------------------------------*/

        /*-----------------------------------------------------------------------------------*/
        /* Parse CSS
        /*-----------------------------------------------------------------------------------*/
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH .'/wp-admin/includes/file.php';
            WP_Filesystem();
        }

        $filename_class = get_class($wp_filesystem);
        if ($filename_class === 'WP_Filesystem_FTPext') {
            $wp_filesystem = function_exists('wgl_theme_helper') && method_exists(wgl_theme_helper(), 'get_file_system')
                ? wgl_theme_helper()->get_file_system()
                : $wp_filesystem;
        }

        $files = array('theme_content', 'theme_color', 'footer');
        if (class_exists( 'WooCommerce' )) {
            array_push( $files, 'shop');
        }

        foreach ($files as $key => $file) {
            $file = get_theme_file_path( '/core/admin/css/dynamic/'.$file.'.css' );
            if ($wp_filesystem->exists( $file)) {
                $file = $wp_filesystem->get_contents( $file );
                preg_match_all('/\s*\\$([A-Za-z1-9_\-]+)(\s*:\s*(.*?);)?\s*/', $file, $vars);

                $found = $vars[0];
                $varNames = $vars[1];
                $count = count( $found);

                for ($i = 0; $i < $count; $i++) {
                    $varName = trim($varNames[$i]);
                    $file = preg_replace('/\\$'.$varName.'(\W|\z)/', (isset( ${$varName}) ? ${$varName} : "").'\\1', $file);
                }

                $line = str_replace( $found, '', $file);

                $css .= $line;
            }
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Parse css
        /*-----------------------------------------------------------------------------------*/

        $css .= 'body {'
            . (!empty($bg_body) ? 'background: ' . $bg_body . ';' : '')
        . '}'
        . 'ol.commentlist:after {'
            . (!empty($bg_body) ? 'background:'.$bg_body.';' : '')
        . '}';

        /*-----------------------------------------------------------------------------------*/
        /* Typography render
        /*-----------------------------------------------------------------------------------*/
        for ($i = 1; $i <= 6; $i++) {
            $css .= 'h' . $i . ',h' . $i . ' a, h' . $i . ' span {
                ' . (!empty(${'header-h' . $i . '_family'}) ? 'font-family:' . ${'header-h' . $i . '_family'} . ';' : '') . '
                ' . (!empty(${'header-h' . $i . '_weight'}) ? 'font-weight:' . ${'header-h' . $i . '_weight'} . ';' : '') . '
                ' . (!empty(${'header-h' . $i . '_size'}) ? 'font-size:' . ${'header-h' . $i . '_size'} . ';' : '') . '
                ' . (!empty(${'header-h' . $i . '_line_height'}) ? 'line-height:' . ${'header-h' . $i . '_line_height'} . ';' : '') . '
                ' . (!empty(${'header-h' . $i . '_text_transform'}) ? 'text-transform:' . ${'header-h' . $i . '_text_transform'} . ';' : '') . '
            }';
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Typography render
        /*-----------------------------------------------------------------------------------*/

        /*-----------------------------------------------------------------------------------*/
        /* Mobile Header render
        /*-----------------------------------------------------------------------------------*/
        //* Fetch mobile header height to apply it for mobile styles
        $header_mobile_height = iRecco::get_option('header_mobile_height');
        $header_mobile_height = !empty($header_mobile_height['height']) ? 'calc(100vh - '.esc_attr((int)$header_mobile_height['height']).'px)' : '';

        //* Set Queries width to apply mobile style
        $header_type = iRecco::get_option('header_type');

        $header_page_select = iRecco::get_option('header_page_select');
        $header_page_select_id = 0;

        if (!empty($header_page_select)) {
            $header_page_select_id = intval($header_page_select);

            if (class_exists('SitePress')) {
                $header_page_select_id = wpml_object_id_filter($header_page_select_id, 'header', false, ICL_LANGUAGE_CODE);
            }
        }

        if (
            $this->RWMB_is_active()
            && rwmb_meta('mb_customize_header_layout') == 'custom'
            && rwmb_meta('mb_header_content_type') !== 'default'
        ) {
            $header_type = 'custom';
            $header_page_select_id = (int) rwmb_meta('mb_customize_header');

            if (class_exists('SitePress')) {
                $header_page_select_id = wpml_object_id_filter($header_page_select_id, 'header', false, ICL_LANGUAGE_CODE);
            }
        }

        $header_queries = iRecco::get_option('header_mobile_queris');

        if (
            $header_type === 'custom'
            && !empty($header_page_select)
            && did_action('elementor/loaded')
        ) {
            $header_page_select_id = intval($header_page_select);

            $page_settings_manager = \Elementor\Core\Settings\Manager::get_settings_managers('page');
            $page_settings_model = $page_settings_manager->get_model($header_page_select_id);
            $header_queries = $page_settings_model->get_settings('mobile_breakpoint');
        }

        if (iRecco::get_option('mobile_header')) {
            $mobile_background = iRecco::get_option('mobile_background');
            $mobile_color = iRecco::get_option('mobile_color');

            $css .= '@media only screen and (max-width: ' . (int) $header_queries . 'px) {'
                . '.wgl-theme-header{'
                    . 'background-color: ' . esc_attr($mobile_background['rgba']) . ' !important;'
                    . 'color: ' . esc_attr($mobile_color) . ' !important;'
                . '}'
                . '.hamburger-inner, .hamburger-inner:before, .hamburger-inner:after {'
                    . 'background-color: ' . esc_attr($mobile_color) . ';'
                . '}'
            . '}';
        }

        $css .= '@media only screen and (max-width: '.(int)$header_queries.'px) {'
            . '.wgl-theme-header .wgl-mobile-header {'
                . 'display: block;'
            . '}'
            . '.wgl-site-header,'
            . '.wgl-theme-header .primary-nav {'
                . 'display: none;'
            . '}'
            . '.elementor-editor-active .wgl-theme-header .wgl-site-header,'
            . '.elementor-editor-active .wgl-theme-header .primary-nav {'
                . 'display: block;'
            . '}'
            . '.wgl-theme-header .mobile-hamburger-toggle {'
                . 'display: inline-block;'
            . '}'
            . 'header.wgl-theme-header .mobile_nav_wrapper .primary-nav {'
                . 'display: block;'
            . '}'
            . '.wgl-theme-header .wgl-sticky-header {'
                . 'display: none;'
            . '}'
            . '.wgl-social-share_pages {'
                . 'display: none;'
            . '}'
        . '}';

        $mobile_sticky = iRecco::get_option('mobile_sticky');

        if (iRecco::get_option('mobile_over_content')) {
            $css .= '@media (max-width: ' . (int) $header_queries . 'px) {'
                . '.wgl-theme-header {'
                    . 'position: absolute;'
                    . 'z-index: 99;'
                    . 'width: 100%;'
                    . 'left: 0;'
                    . 'top: 0;'
                . '}'
            . '}';
            if ($mobile_sticky) {
                $css .= '@media (max-width: ' . (int) $header_queries . 'px) {'
                    . 'body .wgl-theme-header .wgl-mobile-header {'
                        . 'position: absolute;'
                        . 'left: 0;'
                        . 'width: 100%;'
                    . '}'
                . '}';
            }
        } else {
            $css .= '@media (max-width: ' . (int) $header_queries . 'px) {'
                . 'body .wgl-theme-header.header_overlap {'
                    . 'position: relative;'
                    . 'z-index: 2;'
                . '}'
            . '}';
        }

        if ($mobile_sticky) {
            $css .= '@media (max-width: ' . (int) $header_queries . 'px) {'
                . 'body .wgl-theme-header,'
                . 'body .wgl-theme-header.header_overlap {'
                    . 'position: sticky;'
                    . 'top: 0;'
                . '}'
            . '}';
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Mobile Header render
        /*-----------------------------------------------------------------------------------*/

        /*-----------------------------------------------------------------------------------*/
        /* Page Title Responsive
        /*-----------------------------------------------------------------------------------*/
        $page_title_resp = iRecco::get_option('page_title_resp_switch');
        $rwmb_is_active = false;

        if (
            $this->RWMB_is_active()
            && 'on' === rwmb_meta('mb_page_title_switch')
            && rwmb_meta('mb_page_title_resp_switch')
        ) {
            $page_title_resp = true;
            $rwmb_is_active = true;
        }

        if ($page_title_resp) {
            $page_title_height = iRecco::get_option('page_title_resp_height')['height'];

            $pt_breakpoint = (int) iRecco::get_mb_option('page_title_resp_resolution', 'mb_page_title_resp_switch', true);
            $pt_padding = iRecco::get_mb_option('page_title_resp_padding', 'mb_page_title_resp_switch', true);

            if ($rwmb_is_active) {
                $page_title_height = rwmb_meta('mb_page_title_resp_height');
            }

            $pt_font = iRecco::get_mb_option('page_title_resp_font', 'mb_page_title_resp_switch', true);

            $breadcrumbs_font = iRecco::get_mb_option('page_title_resp_breadcrumbs_font', 'mb_page_title_resp_switch', true);
            $breadcrumbs_switch = iRecco::get_mb_option('page_title_resp_breadcrumbs_switch', 'mb_page_title_resp_switch', true);

            //* Title styles
            $pt_color = !empty($pt_font['color']) ? 'color: ' . esc_attr($pt_font['color']) . ' !important;' : '';
            $pt_f_size = !empty($pt_font['font-size']) ? ' font-size: ' . esc_attr((int) $pt_font['font-size']) . 'px !important;' : '';
            $pt_line_height = !empty($pt_font['line-height']) ? ' line-height: ' . esc_attr((int) $pt_font['line-height']) . 'px !important;' : '';
            $pt_additional_style = !(bool) $breadcrumbs_switch ? ' margin-bottom: 0 !important;' : '';
            $title_style = $pt_color . $pt_f_size . $pt_line_height . $pt_additional_style;

            //* Breadcrumbs Styles
            $breadcrumbs_color = !empty($breadcrumbs_font['color']) ? 'color: ' . esc_attr($breadcrumbs_font['color']) . ' !important;' : '';
            $breadcrumbs_f_size = !empty($breadcrumbs_font['font-size']) ? 'font-size: ' . esc_attr((int) $breadcrumbs_font['font-size']) . 'px !important;' : '';
            $breadcrumbs_line_height = !empty($breadcrumbs_font['line-height']) ? 'line-height: ' . esc_attr((int) $breadcrumbs_font['line-height']) . 'px !important;' : '';
            $breadcrumbs_display = !(bool) $breadcrumbs_switch ? 'display: none !important;' : '';
            $breadcrumbs_style = $breadcrumbs_color . $breadcrumbs_f_size . $breadcrumbs_line_height . $breadcrumbs_display;

            $css .= '@media only screen and (max-width: ' . $pt_breakpoint . 'px) {'
                . '.page-header {'
                    . (!empty($pt_padding['padding-top']) ? 'padding-top: ' . esc_attr((int) $pt_padding['padding-top']) . 'px !important;' : '')
                    . (!empty($pt_padding['padding-bottom']) ? 'padding-bottom: ' . esc_attr((int) $pt_padding['padding-bottom']) . 'px !important;' : '')
                    . (!empty($page_title_height) ? 'height: ' . esc_attr((int) $page_title_height) . 'px !important;' : '')
                . '}'
                . '.page-header_content .page-header_title {'
                    . $title_style
                . '}'
                . '.page-header_content .page-header_breadcrumbs {'
                    . $breadcrumbs_style
                . '}'
            . '}';
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Page Title Responsive
        /*-----------------------------------------------------------------------------------*/

        /*-----------------------------------------------------------------------------------*/
        /* Portfolio Single Responsive
        /*-----------------------------------------------------------------------------------*/
        $portfolio_resp = iRecco::get_option('portfolio_single_resp');

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta('mb_portfolio_post_conditional')
            && rwmb_meta('mb_portfolio_single_resp') == '1'
        ) {
            $portfolio_resp = '1';
        }

        if ('1' == $portfolio_resp) {
            $pf_queries = iRecco::get_mb_option('portfolio_single_resp_breakpoint', 'mb_portfolio_single_resp', true);

            $pf_padding = iRecco::get_mb_option('portfolio_single_resp_padding', 'mb_portfolio_single_resp', true);

            $css .= '@media only screen and (max-width: ' . esc_attr((int) $pf_queries) . 'px) {
                .wgl-portfolio-single_wrapper.single_type-3 .wgl-portfolio-item_bg .wgl-portfolio-item_title_wrap,
                .wgl-portfolio-single_wrapper.single_type-4 .wgl-portfolio-item_bg .wgl-portfolio-item_title_wrap {'
                    . (!empty($pf_padding['padding-top']) ? 'padding-top:'.esc_attr((int) $pf_padding['padding-top']).'px !important;' : '' )
                    . (!empty($pf_padding['padding-bottom']) ? 'padding-bottom:'. esc_attr((int) $pf_padding['padding-bottom']) . 'px  !important;' : '')
                . '}'
            . '}';
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Portfolio Single Responsive
        /*-----------------------------------------------------------------------------------*/

        /*-----------------------------------------------------------------------------------*/
        /* Elementor Theme css
        /*-----------------------------------------------------------------------------------*/
        if (did_action('elementor/loaded')) {
            if (defined('ELEMENTOR_VERSION')) {
                if (version_compare(ELEMENTOR_VERSION, '3.0', '<')) {
                    $container_width = get_option('elementor_container_width');
                    $container_width = !empty($container_width) ? $container_width : 1140;
                } else {
                    $kit_id = (new \Elementor\Core\Kits\Manager())->get_active_id();
                    $meta_key = \Elementor\Core\Settings\Page\Manager::META_KEY;
                    $kit_settings = get_post_meta($kit_id, $meta_key, true);
                    $container_width = $kit_settings['container_width']['size'] ?? 1140;
                }
            }

            $css .= 'body.elementor-page main .wgl-container.wgl-content-sidebar,
                body.elementor-editor-active main .wgl-container.wgl-content-sidebar,
                body.elementor-editor-preview main .wgl-container.wgl-content-sidebar {
                    max-width: ' . intval($container_width) . 'px;
                    margin-left: auto;
                    margin-right: auto;
                }';

            $css .= 'body.single main .wgl-container {
                    max-width: ' . intval($container_width) . 'px;
                    margin-left: auto;
                    margin-right: auto;
                }';
        }
        /*-----------------------------------------------------------------------------------*/
        /* \End Elementor Theme css
        /*-----------------------------------------------------------------------------------*/

        /*-----------------------------------------------------------------------------------*/
        /* Add Inline css
        /*-----------------------------------------------------------------------------------*/

        $css = $this->minify_css($css);
        wp_add_inline_style('irecco-main', $css);

        /*-----------------------------------------------------------------------------------*/
        /* \End Add Inline css
        /*-----------------------------------------------------------------------------------*/
    }

    /**
     * @since 1.1.2
     */
    public function RWMB_is_active()
    {
        $id = !is_archive() ? get_queried_object_id() : 0;

        return class_exists('RWMB_Loader') && 0 !== $id;
    }

    public static function instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }

        return self::$instance;
    }
}

new iRecco_dynamic_styles();
