<?php

/**
* Load theme framework
*
*/
require_once( get_template_directory() . '/lib/framework/ghostpool-framework.php' );
 
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * Create your own ghostpool_theme_setup() function to override in a child theme.
 *
 */
if ( ! function_exists( 'ghostpool_theme_setup' ) ) {
	function ghostpool_theme_setup() {

		// Localisation
		load_theme_textdomain( 'huber', trailingslashit( WP_LANG_DIR ) . 'themes/' );
		load_theme_textdomain( 'huber', get_stylesheet_directory() . '/languages' );
		load_theme_textdomain( 'huber', get_template_directory() . '/languages' );

		// Background customizer
		add_theme_support( 'custom-background' );

		// This theme styles the visual editor with editor-style.css to match the theme style
		add_editor_style( 'lib/css/editor-style.css' );

		// Add default posts and comments RSS feed links to <head>
		add_theme_support( 'automatic-feed-links' );
		
		// Post formats
		add_theme_support( 'post-formats', array( 'quote', 'video', 'audio', 'gallery', 'link' ) );

		// Title support
		add_theme_support( 'title-tag' );
		
		// Indicate widget sidebars can use selective refresh in the Customizer
		add_theme_support( 'customize-selective-refresh-widgets' );
		
		// WooCommerce support
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-zoom' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );
						
		// BuddyPress legacy support
		add_theme_support( 'buddypress-use-legacy' );
				
		// Add Auto Load Next Post Plugin support
		//add_theme_support( 'auto-load-next-post' );
	
		// Register navigation menus
		register_nav_menus( array(
			'gp-desktop-nav' => esc_html__( 'Desktop Navigation', 'huber' ),
			'gp-mobile-nav' => esc_html__( 'Mobile Navigation', 'huber' ),
			'gp-social-nav' => esc_html__( 'Social Navigation', 'huber' ),
			'gp-profile-nav' => esc_html__( 'Profile Navigation', 'huber' ),
			'gp-top-nav' => esc_html__( 'Top Navigation', 'huber' ),
			'gp-footer-nav' => esc_html__( 'Footer Navigation', 'huber' ),
		) );
 
	}
}
add_action( 'after_setup_theme', 'ghostpool_theme_setup' );

/**
* Load theme functions
*
*/
if ( ! function_exists( 'ghostpool_load_theme_functions' ) ) {
	function ghostpool_load_theme_functions() {

		// Theme setup
		require_once( get_template_directory() . '/lib/framework/theme-setup/init.php' );

		// Sidebars
		require_once( get_template_directory() . '/lib/framework/custom-sidebars/custom-sidebars.php' );
	
		// Category options
		require_once( get_template_directory() . '/lib/inc/category-config.php' );

		// Init variables
		require_once( get_template_directory() . '/lib/inc/init-variables.php' );

		// Load plugin defaults
		require_once( get_template_directory() . '/lib/inc/plugin-defaults.php' );
			
		// Load theme converter
		require_once( get_template_directory() . '/lib/inc/theme-converter.php' );
	
		// Load pagination functions
		require_once( get_template_directory() . '/lib/inc/pagination-functions.php' );
				
		// Load query functions
		require_once( get_template_directory() . '/lib/inc/query-functions.php' );
		
		// Load ajax functions		
		require_once( get_template_directory() . '/lib/inc/ajax.php' );
						
		// Load custom menu walker
		require_once( get_template_directory() . '/lib/menus/custom-menu-walker.php' );

		// Load custom menu fields
		require_once( get_template_directory() . '/lib/menus/menu-item-custom-fields.php' );

		// Load user rating functions
		require_once( get_template_directory() . '/lib/inc/user-rating-functions.php' );

		// Load general rating functions
		require_once( get_template_directory() . '/lib/inc/general-rating-functions.php' );	
				
		// Load page header functions
		require_once( get_template_directory() . '/lib/inc/page-header.php' );
		
		// Load hub header functions
		require_once( get_template_directory() . '/lib/inc/hub-header.php' );
		
		// Load video header functionss
		require_once( get_template_directory() . '/lib/inc/video-header.php' );
			
		// Load page title functions
		require_once( get_template_directory() . '/lib/inc/page-titles.php' );

		// Load loop functions
		require_once( get_template_directory() . '/lib/inc/loop-functions.php' );
		
		// Load follow items functions
		if ( ghostpool_option( 'following_items' ) == 'both' OR ( is_user_logged_in() && ghostpool_option( 'following_items' ) == 'members' ) ) {
			require_once( get_template_directory() . '/lib/inc/follow-items.php' );
		}
						
		// Load vote up/down functions
		require_once( get_template_directory() . '/lib/inc/voting.php' );
		
		// Load login functions
		if ( ! is_user_logged_in() && ghostpool_option( 'popup_box' ) == 'enabled' ) {
			require_once( get_template_directory() . '/lib/inc/login-functions.php' );
		}
								
		// Load post submission functions
		require_once( get_template_directory() . '/lib/inc/post-submission.php' );

		// Load post editing functions
		require_once( get_template_directory() . '/lib/inc/post-edit.php' );
						
		// Load BuddyPress functions
		if ( function_exists( 'bp_is_active' ) ) {
			require_once( get_template_directory() . '/lib/inc/bp-functions.php' );
		}

		// Load bbPress functions
		if ( function_exists( 'is_bbpress' ) ) {
			require_once( get_template_directory() . '/lib/inc/bbpress-functions.php' );
		}
		
		// Load Woocommerce functions
		if ( function_exists( 'is_woocommerce' ) ) {
			require_once( get_template_directory() . '/lib/inc/wc-functions.php' );
		}

		// Disable activation redirect
		remove_action( 'admin_init', 'vc_page_welcome_redirect' );
										
	}
}
add_action( 'after_setup_theme', 'ghostpool_load_theme_functions' );

/**
* Remove Visual Composer activation notice
*
*/
if ( ! function_exists( 'ghostpool_remove_wpb_notice' ) ) {
	function ghostpool_remove_wpb_notice() {		
		if ( function_exists( 'vc_set_as_theme' ) && is_admin() ) {
			setcookie( 'vchideactivationmsg', '1', strtotime( '+3 years' ), '/' );
			setcookie( 'vchideactivationmsg_vc11', ( defined( 'WPB_VC_VERSION' ) ? WPB_VC_VERSION : '1' ), strtotime( '+3 years' ), '/' );
		}
	}
}
add_action( 'vc_before_init', 'ghostpool_remove_wpb_notice', 9 );

/**
* Load Visual Composer element classes
*
*/
if ( function_exists( 'vc_set_as_theme' ) && ! function_exists( 'ghostpool_vc_shortcodes_container' ) ) {
	function ghostpool_vc_shortcodes_container() {
		if ( class_exists( 'WPBakeryShortCodesContainer' ) ) {
			class WPBakeryShortCode_Pricing_Table extends WPBakeryShortCodesContainer {}
			class WPBakeryShortCode_Testimonial_Slider extends WPBakeryShortCodesContainer {}
			class WPBakeryShortCode_Team extends WPBakeryShortCodesContainer {}
		}
		if ( class_exists( 'WPBakeryShortCode' ) ) {
			class WPBakeryShortCode_Pricing_Column extends WPBakeryShortCode {}	
			class WPBakeryShortCode_Testimonial extends WPBakeryShortCode {}
			class WPBakeryShortCode_Team_Member extends WPBakeryShortCode {}	
		}
	}
	add_action( 'init', 'ghostpool_vc_shortcodes_container' );
}
				
/**
 * Registered image sizes
 *
 */
if ( ! function_exists( 'ghostpool_image_sizes' ) ) {
	function ghostpool_image_sizes() {				
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'ghostpool-featured-image', 780, 500, false );
		add_image_size( 'ghostpool-page-header-image', 1400, 596, true );
		add_image_size( 'ghostpool-related-image', 576, 356, true );
		add_image_size( 'ghostpool-col-category-image', 576, 356, true );
		add_image_size( 'ghostpool-one-col-category-image', 780, 500, true );
		add_image_size( 'ghostpool-list-category-image', 250, 135, true );
		add_image_size( 'ghostpool-image-tab', 231, 231, true );
		add_image_size( 'ghostpool-posts-widget-image', 100, 80, true );
		add_image_size( 'ghostpool-featured-box-large-image', 979, 438, true );
		add_image_size( 'ghostpool-featured-box-small-image', 639, 200, true );
		add_image_size( 'ghostpool-menu-image', 206, 127, true );
		add_image_size( 'ghostpool-rating-box-image', 730, 310, true );					
	}
}
add_action( 'after_setup_theme', 'ghostpool_image_sizes' );
		
/**
 * Sets the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 *
 */
function ghostpool_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'ghostpool_content_width', 780 );
}
add_action( 'after_setup_theme', 'ghostpool_content_width', 0 );

		
/**
 * Enqueues scripts and styles.
 *
 */	
if ( ! function_exists( 'ghostpool_scripts' ) ) {

	function ghostpool_scripts() {
	
		global $wp_query;

		wp_enqueue_style( 'ghostpool-style', get_stylesheet_uri() );
		
		wp_enqueue_style( 'fontawesome', get_template_directory_uri() . '/lib/fonts/font-awesome/css/font-awesome.min.css' );
					
		if ( ghostpool_option( 'lightbox' ) != 'disabled' ) {
			wp_enqueue_style( 'featherlight', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.min.css' );
			wp_enqueue_style( 'featherlight-gallery', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.gallery.min.css' );
		}
				
		if ( ghostpool_option( 'custom_stylesheet' ) ) {
			wp_enqueue_style( 'ghostpool-custom-style', get_stylesheet_directory_uri() . '/' . ghostpool_option( 'custom_stylesheet' ) );
		}

		$custom_css = '';

		$custom_css .= 
		
		'#gp-desktop-nav .menu li.gp-megamenu > .sub-menu,#gp-desktop-nav .menu li.gp-tab-content-menu .sub-menu,#gp-desktop-nav .menu li.gp-content-menu .sub-menu {left: -' . ( str_replace( 'px', '', ghostpool_option( 'desktop_logo_dimensions', 'width' ) ) + 30 ) . 'px;} ' .
		
		'#gp-main-header{height:' . ghostpool_option( 'desktop_header_height', 'height' ) . ';}' .
	
		'.gp-active,.gp-theme #bbpress-forums .status-closed a{color: ' . ghostpool_option( 'general_link_color', 'hover' ) . ';}' . 

		'#gp-footer{background: -webkit-linear-gradient(left, ' . ghostpool_option( 'copyright_bg' ) . ' 0%, ' . ghostpool_option( 'copyright_bg' ) . ' 50%, ' . ghostpool_option( 'footer_bg' ) . ' 51%, ' . ghostpool_option( 'footer_bg' ) . ' 100%);background: linear-gradient(to right, ' . ghostpool_option( 'copyright_bg' ) . ' 0%, ' . ghostpool_option( 'copyright_bg' ) . ' 50%, ' . ghostpool_option( 'footer_bg' ) . ' 51%, ' . ghostpool_option( 'footer_bg' ) . ' 100%);}' .
		
		'body:not(.gp-has-hub-header) #gp-page-header-video{padding-bottom: ' . ( str_replace( array( '-', 'px' ), '', ghostpool_option( 'content_wrapper_margin', 'margin-top' ) ) + 30 ) . 'px;}' .

		'#gp-cart-nav .widget_shopping_cart .cart_list li a.remove{color:' . ghostpool_option( 'cart_dropdown_link_color', 'regular' ) . '!important;}' .
		
		'#gp-cart-nav .widget_shopping_cart .cart_list li a.remove:hover{color:' . ghostpool_option( 'cart_dropdown_link_color', 'hover' ) . '!important;}' .
		
		// Media Queries
		'@media only screen and (min-width: 1083px){
			.gp-header-overlay #gp-homepage-slider{margin-top: -' . ghostpool_option( 'desktop_header_height', 'height' ) . ';}
		}' .
		
		'@media only screen and (max-width: 1023px){
			#gp-main-header{height:' . ghostpool_option( 'mobile_header_height', 'height' ) . ';}
		}' .

		'@media only screen and (max-width: 767px){
		
				.gp-theme .gp-entry-title.gp-single-title{font-size:' . ghostpool_option( 'post_title_mobile_typography', 'font-size' ) . '; line-height:' . ghostpool_option( 'post_title_mobile_typography', 'line-height' ) . '; letter-spacing:' . ghostpool_option( 'post_title_mobile_typography', 'letter-spacing' ) . ';} ' .
				
				'.gp-theme .gp-entry-header .gp-subtitle{font-size:' . ghostpool_option( 'post_subtitle_mobile_typography', 'font-size' ) . '; line-height:' . ghostpool_option( 'post_subtitle_mobile_typography', 'line-height' ) . '; letter-spacing:' . ghostpool_option( 'post_subtitle_mobile_typography', 'letter-spacing' ) . ';}' .
				
		'}';
			
		if ( ghostpool_option( 'custom_css' ) ) {
			$custom_css .= ghostpool_option( 'custom_css' );
		}

		wp_add_inline_style( 'ghostpool-style', $custom_css );
				
		wp_enqueue_script( 'modernizr', get_template_directory_uri() . '/lib/scripts/modernizr.js', false, '', true );
				
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) { 
			wp_enqueue_script( 'comment-reply' );
		}
		
		wp_enqueue_script( 'jquery-ui-datepicker' );
				
		wp_enqueue_script( 'imagesloaded' );
		
		if ( ghostpool_option( 'smooth_scrolling' ) == 'gp-smooth-scrolling' ) { 
			wp_enqueue_script( 'nicescroll', get_template_directory_uri() . '/lib/scripts/jquery.nicescroll.min.js', false, '', true );
		}
		
		wp_enqueue_script( 'selectivizr', get_template_directory_uri() . '/lib/scripts/selectivizr.min.js', false, '', true );

		wp_enqueue_script( 'placeholder', get_template_directory_uri() . '/lib/scripts/placeholders.min.js', false, '', true );
									
		if ( ghostpool_option( 'lightbox' ) != 'disabled' ) {
			wp_enqueue_script( 'featherlight', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.min.js', array( 'jquery' ), '', true );
			wp_enqueue_script( 'featherlight-gallery', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.gallery.min.js', array( 'jquery' ), '', true );
		}
			
		if ( ghostpool_option( 'back_to_top' ) != 'gp-no-back-to-top' ) { 
			wp_enqueue_script( 'jquery-totop', get_template_directory_uri() . '/lib/scripts/jquery.ui.totop.min.js', array( 'jquery' ), '', true );
		}	

		wp_enqueue_script( 'jquery-flexslider', get_template_directory_uri() . '/lib/scripts/jquery.flexslider-min.js', array( 'jquery' ), '', true );
		
		wp_enqueue_script( 'isotope', get_template_directory_uri() . '/lib/scripts/isotope.pkgd.min.js', false, '', true );

		wp_enqueue_script( 'lazyload', get_template_directory_uri() . '/lib/scripts/jquery.lazyload.min.js', array( 'jquery' ), '', true );
		
		wp_enqueue_script( 'jquery-infinitescroll', get_template_directory_uri() . '/lib/scripts/jquery.infinitescroll.min.js', array( 'jquery' ), '', true );
		
		wp_enqueue_script( 'ghostpool-voting', get_template_directory_uri() . '/lib/scripts/voting.js', array( 'jquery' ) );
		wp_localize_script( 'ghostpool-voting', 'ghostpool_voting', array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ), 
			'nonce'    => wp_create_nonce( 'ghostpool_voting_action' )
		) );
								
		wp_enqueue_script( 'ghostpool-custom', get_template_directory_uri() . '/lib/scripts/custom.js', array( 'jquery' ), '', true );
		
		if ( is_ssl() ) {
			$url = esc_url( 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );
		} else { 
			$url = esc_url( 'http://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );
		}
					
		wp_localize_script( 'ghostpool-custom', 'ghostpool_script', array(
			'lightbox' 					   => ghostpool_option( 'lightbox' ),
			'url'	   					   => $url,
			'max_num_pages' 			   => $wp_query->max_num_pages,
			'get_template_directory_uri'   => get_template_directory_uri()
		) );
										
	}
}
add_action( 'wp_enqueue_scripts', 'ghostpool_scripts' );

/**
 * Enqueues admin scripts and styles.
 *
 */	
if ( ! function_exists( 'ghostpool_admin_scripts' ) ) {
	function ghostpool_admin_scripts() {
		wp_enqueue_style( 'ghostpool-admin', get_template_directory_uri() . '/lib/framework/css/general-admin.css' );
	}
}
add_action( 'admin_enqueue_scripts', 'ghostpool_admin_scripts' );

/**
 * Adds custom classes to the array of body classes.
 *
 */	
if ( ! function_exists( 'ghostpool_body_classes' ) ) {
	function ghostpool_body_classes( $classes ) {
		$classes[] = 'gp-theme';
		$classes[] = ghostpool_option( 'theme_layout' );
		$classes[] = ghostpool_option( 'smooth_scrolling' );
		$classes[] = ghostpool_option( 'back_to_top' );
		$classes[] = ghostpool_option( 'cart_button' );
		$classes[] = ghostpool_option( 'search_button' );
		$classes[] = ghostpool_option( 'social_nav' );
		$classes[] = ghostpool_option( 'profile_nav' );
		$classes[] = ghostpool_option( 'header_layout' );
		$classes[] = ghostpool_option( 'fixed_header' );
		$classes[] = ghostpool_option( 'top_header' );
		$classes[] = isset( $GLOBALS['ghostpool_page_header'] ) ? $GLOBALS['ghostpool_page_header'] : '';
		$classes[] = isset( $GLOBALS['ghostpool_layout'] ) ? $GLOBALS['ghostpool_layout'] : '';
		if ( is_page_template( 'homepage-template.php' ) ) {
			$classes[] = 'gp-homepage';
		}	
		if ( get_post_meta( get_the_ID(), 'template', true ) ) {
			$classes[] = get_post_meta( get_the_ID(), 'template', true );
		}
		if ( 
		
		( get_post_meta( get_the_ID(), 'template', true ) == 'gp-hub-post' OR get_post_meta( get_the_ID(), 'custom_hub_tab_ids', true ) ) 
		
		OR ( ( ghostpool_option( 'hub_header_associated_content' ) == 'enabled' OR ghostpool_option( 'hub_tab_associated_content' ) == 'enabled' ) && ( get_post_meta( get_the_ID(), 'hub_association_ids', true ) OR get_post_meta( get_the_ID(), 'ghostpool_hub_page_id', true ) ) ) 
		
		OR ( get_post_meta( get_the_ID(), 'template', true ) == 'gp-review-post' && ghostpool_option( 'review_hub_header_content' ) == 'enabled' )
		
		OR ( isset( $_GET['review_tabs'] ) && $_GET['review_tabs'] == 'enabled' ) 
		
		) {
			$classes[] = 'gp-has-hub-header';	
		}	
		if ( defined( 'TSS_VERSION' ) ) {	
			$classes[] = 'gp-sticky-sidebars';	
		}	
		return $classes;
	}
}
add_filter( 'body_class', 'ghostpool_body_classes' );
	
/**
 * Content added to header
 *
 */	
if ( ! function_exists( 'ghostpool_wp_header' ) ) {

	function ghostpool_wp_header() {

		// Title fallback for versions earlier than WordPress 4.1
		if ( ! function_exists( '_wp_render_title_tag' ) && ! function_exists( 'ghostpool_render_title' ) ) {
			function ghostpool_render_title() { ?>
				<title><?php wp_title( '|', true, 'right' ); ?></title>
			<?php }
		}

		// Initial variables - variables loaded only once at the top of the page
		ghostpool_init_variables();

		// Add custom JavaScript code
		if ( ghostpool_option( 'js_code' ) ) {
			if ( strpos( ghostpool_option( 'js_code' ), '<script ' ) !== false ) { 
				echo ghostpool_option( 'js_code' ); 
			} else {
				$js_code = str_replace( array( '<script>', '</script>' ), '', ghostpool_option( 'js_code' ) );
				echo '<script>' . $js_code . '</script>';
			}    
		}
			
	}
	
}
add_action( 'wp_head', 'ghostpool_wp_header' );
	
/**
 * Navigation user meta
 *
 */	
if ( ! function_exists( 'ghostpool_nav_user_meta' ) ) {
	function ghostpool_nav_user_meta( $user_id = NULL ) {
	
		// So this can be used without hooking into user_register
		if ( ! $user_id ) {
			$user_id = get_current_user_id(); 
		}

		// Set the default properties if it has not been set yet
		if ( ! get_user_meta( $user_id, 'managenav-menuscolumnshidden', true) ) {
			$meta_value = array( 'link-target', 'xfn', 'description' );
			update_user_meta( $user_id, 'managenav-menuscolumnshidden', $meta_value );
		}
	
	}	
}
add_action( 'admin_init', 'ghostpool_nav_user_meta' );

/**
 * Insert schema meta data
 *
 */	
if ( ! function_exists( 'ghostpool_meta_data' ) ) {
	function ghostpool_meta_data( $post_id ) {
	
		global $post;

		// Get title
		if ( ghostpool_option( 'custom_title' ) ) { 
			$title = esc_attr( ghostpool_option( 'custom_title' ) );
		} else {
			$title = get_the_title( $post_id );
		}
		
		// Get featured image dimensions
		$image = wp_get_attachment_image_src( get_post_thumbnail_id(), 'ghostpool-featured-image' );

		// Meta data
		return '<meta itemprop="mainEntityOfPage" itemType="https://schema.org/WebPage" itemscope content="' . esc_url( get_permalink( $post_id ) ) . '">
		<meta itemprop="headline" content="' . esc_attr( $title ) . '">			
		<div itemprop="image" itemscope itemtype="https://schema.org/ImageObject">
			<meta itemprop="url" content="' . esc_url( wp_get_attachment_url( get_post_thumbnail_id( $post_id ) ) ) . '">
			<meta itemprop="width" content="' . absint( $image[1] ) . '">	
			<meta itemprop="height" content="' . absint( $image[2] ) . '">		
		</div>
		<meta itemprop="author" content="' . get_the_author_meta( 'display_name', $post->post_author ) . '">			
		<meta itemprop="datePublished" content="' . get_the_time( 'Y-m-d' ) . '">
		<meta itemprop="dateModified" content="' . get_the_modified_date( 'Y-m-d' ) . '">
		<div itemprop="publisher" itemscope itemtype="https://schema.org/Organization">
			<div itemprop="logo" itemscope itemtype="https://schema.org/ImageObject">
				<meta itemprop="url" content="' . esc_url( ghostpool_option( 'desktop_logo', 'url' ) ) . '">
				<meta itemprop="width" content="' . absint( ghostpool_option( 'desktop_logo_dimensions', 'width' ) ) . '">
				<meta itemprop="height" content="' . absint( ghostpool_option( 'desktop_logo_dimensions', 'height' ) ) . '">
			</div>
			<meta itemprop="name" content="' . get_bloginfo( 'name' ) . '">
		</div>';

	}
}

/**
 * Insert breadcrumbs
 *
 */	
if ( ! function_exists( 'ghostpool_breadcrumbs' ) ) {
	function ghostpool_breadcrumbs() {
		if ( function_exists( 'yoast_breadcrumb' ) && ! is_front_page() ) { 
			$breadcrumbs = yoast_breadcrumb( '<div id="gp-breadcrumbs">', '</div>' );
		} else {
			$breadcrumbs = '';
		}
	}
}

/**
 * Registers a widget area.
 *
 * @link https://developer.wordpress.org/reference/functions/register_sidebar/
 *
 */
 
// Recent Comments Widget
require_once( get_template_directory() . '/lib/widgets/recent-comments.php' );

// Blog Posts Widget
require_once( get_template_directory() . '/lib/widgets/blog-posts.php' );

// Category Filters Widget
require_once( get_template_directory() . '/lib/widgets/category-filters.php' );

// Popular Posts Widget
require_once( get_template_directory() . '/lib/widgets/popular-posts.php' );

// Showcase Posts Widget
require_once( get_template_directory() . '/lib/widgets/showcase-posts.php' );

// Affiliate Widget
require_once( get_template_directory() . '/lib/widgets/affiliates.php' );

// Images Widget
require_once( get_template_directory() . '/lib/widgets/images.php' );

// Rating Box
require_once( get_template_directory() . '/lib/widgets/rating-box.php' );

if ( ! function_exists( 'ghostpool_widgets_init' ) ) {
	function ghostpool_widgets_init() {

		register_sidebar( array( 
			'name'          => esc_html__( 'Right Sidebar', 'huber' ),
			'id'            => 'gp-right-sidebar',
			'description'   => esc_html__( 'Displayed on posts, pages and post categories.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array( 
			'name'          => esc_html__( 'Left Sidebar', 'huber' ),
			'id'            => 'gp-left-sidebar',
			'description'   => esc_html__( 'Displayed on posts, pages and post categories.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );
		
		register_sidebar( array( 
			'name'          => esc_html__( 'Hub Sidebar', 'huber' ),
			'id'            => 'gp-hub-sidebar',
			'description'   => esc_html__( 'Displayed on hub pages.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array( 
			'name'          => esc_html__( 'Review Sidebar', 'huber' ),
			'id'            => 'gp-review-sidebar',
			'description'   => esc_html__( 'Displayed on review pages.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );
								
		register_sidebar( array(
			'name'          => esc_html__( 'Footer 1', 'huber' ),
			'id'            => 'gp-footer-1',
			'description'   => esc_html__( 'Displayed as the first column in the footer.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );        

		register_sidebar( array(
			'name'          => esc_html__( 'Footer 2', 'huber' ),
			'id'            => 'gp-footer-2',
			'description'   => esc_html__( 'Displayed as the second column in the footer.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );        
	
		register_sidebar( array(
			'name'          => esc_html__( 'Footer 3', 'huber' ),
			'id'            => 'gp-footer-3',
			'description'   => esc_html__( 'Displayed as the third column in the footer.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );        
	
		register_sidebar( array(
			'name'          => esc_html__( 'Footer 4', 'huber' ),
			'id'            => 'gp-footer-4',
			'description'   => esc_html__( 'Displayed as the fourth column in the footer.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );      

		register_sidebar( array(
			'name'          => esc_html__( 'Footer 5', 'huber' ),
			'id'            => 'gp-footer-5',
			'description'   => esc_html__( 'Displayed as the fifth column in the footer.', 'huber' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widgettitle">',
			'after_title'   => '</h3>',
		) );
			
	}
}
add_action( 'widgets_init', 'ghostpool_widgets_init' );

/**
 * Header advertisement
 *
 */	
function ghostpool_add_header_advertisement() {
	if ( function_exists( 'the_ad_placement' ) ) { the_ad_placement( 'header' ); }
}
add_action( 'ghostpool_begin_content_wrapper', 'ghostpool_add_header_advertisement' );

/**
 * Footer advertisement
 *
 */	
function ghostpool_add_footer_advertisement() {
	if ( function_exists( 'the_ad_placement' ) ) { the_ad_placement( 'footer' ); }
}
add_action( 'ghostpool_end_content_wrapper', 'ghostpool_add_footer_advertisement' );

/**
 * Get hub association ID if present
 *
 */	
if ( ! function_exists( 'get_hub_association_id' ) ) {
	function get_hub_association_id( $post_id = '' ) {
		if ( get_post_meta( $post_id, 'primary_hub', true ) ) {
			return get_post_meta( $post_id, 'primary_hub', true );
		} elseif ( get_post_meta( $post_id, 'ghostpool_hub_page_id', true ) ) {
			return get_post_meta( $post_id, 'ghostpool_hub_page_id', true );
		} elseif ( get_post_meta( $post_id, 'hub_association_ids', true ) ) {
			$associated_hub_ids = get_post_meta( $post_id, 'hub_association_ids', true );
			if ( ! is_array( $associated_hub_ids ) ) {
				$associated_hub_ids = explode( ',', $associated_hub_ids );
			}
			$associated_hub_id = str_replace( '"', '', $associated_hub_ids[0] );
			return $associated_hub_ids[0];
		} elseif ( get_post_meta( $post_id, 'custom_hub_tab_ids', true ) ) {
			$associated_hub_ids = get_post_meta( $post_id, 'custom_hub_tab_ids', true );
			if ( ! is_array( $associated_hub_ids ) ) {
				$associated_hub_ids = explode( ',', $associated_hub_ids );
			}
			$associated_hub_id = str_replace( '"', '', $associated_hub_ids[0] );
			return $associated_hub_ids[0];			
		} else {
			return $post_id;
		}
	}
}
		
/**
 * Get author name
 *
 */	
if ( ! function_exists( 'ghostpool_author_name' ) ) {
	function ghostpool_author_name( $post_id = '', $author_link = true ) {
		global $post;
		if ( get_post_meta( get_the_ID(), 'ghostpool_post_submission_username', true ) ) { 
			return get_post_meta( get_the_ID(), 'ghostpool_post_submission_username', true ); 
		} elseif ( $author_link == true ) {
			return apply_filters( 'ghostpool_author_url', '<a href="' . esc_url( get_author_posts_url( $post->post_author ) ) . '">' . get_the_author_meta( 'display_name', $post->post_author ) . '</a>', $post );
		} else {
			return get_the_author_meta( 'display_name', $post->post_author );
		}
	}
}

/**
 * Change password protect text
 *
 */	
if ( ! function_exists( 'ghostpool_password_form' ) ) {
	function ghostpool_password_form() {
		global $post;
		$label = 'pwbox-' . ( empty( $post->ID ) ? rand() : $post->ID );
		$o = '<form action="' . esc_url( site_url( 'wp-login.php?action=postpass', 'login_post' ) ) . '" method="post">
		<p>' . esc_html__( 'To view this protected post, enter the password below:', 'huber' ) . '</p>
		<label for="' . $label . '"><input name="post_password" id="' . $label . '" type="password" size="20" maxlength="20" /></label> <input type="submit" class="pwsubmit" name="Submit" value="' .  esc_attr__( 'Submit', 'huber' ) . '" />
		</form>
		';
		return $o;
	}
}
add_filter( 'the_password_form', 'ghostpool_password_form' );

/**
 * Redirect empty search to search page
 *
 */	
if ( ! function_exists( 'ghostpool_empty_search' ) ) {
	function ghostpool_empty_search( $query ) {
		global $wp_query;
		if ( isset( $_GET['s'] ) && ( $_GET['s'] == '' ) ) {
			$wp_query->set( 's', ' ' );
			$wp_query->is_search = true;
		}
		return $query;
	}
}
add_action( 'pre_get_posts', 'ghostpool_empty_search' );

/**
 * Add lightbox class to image links
 *
 */
if ( ! function_exists( 'ghostpool_lightbox_image_link' ) ) {
	function ghostpool_lightbox_image_link( $content ) {
		if ( ghostpool_option( 'lightbox' ) != 'disabled' ) {
			if ( ghostpool_option( 'lightbox' ) == 'group_images' ) {
				$lightbox = ' data-lightbox="gallery" ';
			} else {
				$lightbox = ' data-featherlight="image" ';
			}
			$pattern = "/<a(.*?)href=('|\")(.*?).(jpg|jpeg|png|gif|bmp|ico)('|\")(.*?)>/i";
			preg_match_all( $pattern, $content, $matches, PREG_SET_ORDER );
			foreach ( $matches as $val ) {
				$pattern = '<a' . $val[1] . 'href=' . $val[2] . $val[3] . '.' . $val[4] . $val[5] . $val[6] . '>';
				$replacement = '<a' . $val[1] . 'href=' . $val[2] . $val[3] . '.' . $val[4] . $val[5] . $lightbox . $val[6] . '>';
				$content = str_replace( $pattern, $replacement, $content );			
			}
			return $content;
		} else {
			return $content;
		}
	}	
}
add_filter( 'the_content', 'ghostpool_lightbox_image_link' );	
add_filter( 'wp_get_attachment_link', 'ghostpool_lightbox_image_link' );
add_filter( 'bbp_get_reply_content', 'ghostpool_lightbox_image_link' );

/**
 * Upload post submission image
 *
 */
if ( ! function_exists( 'ghostpool_insert_attachment' ) ) {	
	function ghostpool_insert_attachment( $file_handler, $post_id ) {
		if ( $_FILES[$file_handler]['error'] !== UPLOAD_ERR_OK )
			return false;		
		require_once( ABSPATH . 'wp-admin' . '/includes/image.php' );
		require_once( ABSPATH . 'wp-admin' . '/includes/file.php' );
		require_once( ABSPATH . 'wp-admin' . '/includes/media.php' );
    	$overrides = array( 'test_form' => false );
		$attach_id = media_handle_upload( $file_handler, $post_id, array(), $overrides );
		if ( ! is_wp_error( $attach_id ) ) {
			if ( ! add_post_meta( $post_id, '_thumbnail_id', $attach_id, true ) ) {
				update_post_meta( $post_id, '_thumbnail_id', $attach_id );
			}
		}
	}
}

/**
 * Get captcha data
 *
 */
if ( ! function_exists( 'ghostpool_captcha' ) ) {
	function ghostpool_captcha() {	
		if ( function_exists( 'ghostpool_custom_captcha' ) ) {
			$captcha = ghostpool_custom_captcha();
		/*} elseif ( has_filter( 'google_invre_is_valid_request_filter' ) ) {
			$captcha = apply_filters( 'google_invre_is_valid_request_filter', true );
			if ( true === $captcha ) {
				$captcha = array();
				$captcha['reason'] = '';
			} else {
				$captcha['reason'] = esc_html__( 'Incorrect captcha.', 'huber' );
			}*/
		} elseif ( has_filter( 'hctpc_verify' ) ) {
			$captcha = apply_filters( 'hctpc_verify', true );
			if ( true === $captcha ) { 
				$captcha = array();
				$captcha['reason'] = ''; 
			} else { 
				$captcha = array();
				$captcha['reason'] = esc_html__( 'Incorrect captcha.', 'huber' ); 
			}
		} elseif ( has_filter( 'cptch_verify' ) ) {
			$captcha = apply_filters( 'cptch_verify', true );
			if ( true === $captcha ) { 
				$captcha = array();
				$captcha['reason'] = ''; 
			} else { 
				$captcha = array();
				$captcha['reason'] = esc_html__( 'Incorrect captcha.', 'huber' );
			}			
		} else {
			$captcha = '';
		}
		return $captcha;
	}	
}

/**
 * Related hubs
 *
 */
if ( ! function_exists( 'ghostpool_related_hubs' ) ) {	
	function ghostpool_related_hubs( $post_id, $option_1, $option_2 ) {
		if ( $option_1 == 'enabled' ) { // User reviews
			if ( get_post_meta( $post_id, 'ghostpool_hub_page_id', true ) ) {
				$associated_hub_ids = get_post_meta( $post_id, 'ghostpool_hub_page_id', true ); 
			} else {
				$associated_hub_ids = get_post_meta( $post_id, 'hub_association_ids', true );
			}
			if ( $associated_hub_ids ) { 
				if ( ! is_array( $associated_hub_ids ) && ! is_serialized( $associated_hub_ids ) ) {
					$associated_hub_ids = str_replace( '"', '', $associated_hub_ids );
					$associated_hub_ids = explode( ',', $associated_hub_ids );
				}	
				if ( is_array( $associated_hub_ids ) ) { ?>			
					<div id="gp-related-hubs">
						<div id="gp-related-hubs-title"><?php echo $option_2; ?></div>
						<?php foreach( $associated_hub_ids as $associated_hub_id ) { ?>
							<?php if ( ( function_exists( 'kdmfi_get_featured_image_id' ) && kdmfi_has_featured_image( 'ghostpool-featured-image-2', $associated_hub_id ) ) OR ( has_post_thumbnail( $associated_hub_id ) ) ) { ?>
								<a href="<?php echo get_permalink( $associated_hub_id ); ?>" class="gp-related-hub-image">
									<span class="gp-tooltip"><?php echo get_the_title( $associated_hub_id ); ?></span>
									<?php if ( function_exists( 'kdmfi_get_featured_image_id' ) && kdmfi_has_featured_image( 'ghostpool-featured-image-2', $associated_hub_id ) ) {
										kdmfi_the_featured_image( 'ghostpool-featured-image-2', 'thumbnail', $associated_hub_id );
									} elseif ( has_post_thumbnail( $associated_hub_id ) ) {
										echo get_the_post_thumbnail( $associated_hub_id, apply_filters( 'ghostpool_related_hub_image_size', 'thumbnail' ) );
									} ?>
								</a>	
							<?php } ?>	
						<?php } ?>	
												
						<div class="gp-clear"></div>
					</div>	
				<?php } ?>
			<?php }
		}
	}	
}

/**
 * Show pending posts counter in dashboard
 *
 */
if ( ! function_exists( 'ghostpool_pending_user_reviews' ) ) {
	function ghostpool_pending_user_reviews() {
	
		global $menu;

		// Recursive array search
		if ( ! function_exists( 'ghostpool_recursive_search' ) ) {
			function ghostpool_recursive_search( $needle, $haystack ) {
				foreach( $haystack as $key => $value ) {
					$current_key = $key;
					if ( $needle === $value OR ( is_array( $value ) && ghostpool_recursive_search( $needle, $value ) !== false ) ) {
						return $current_key;
					}
				}
				return false;
			}
		}
		
		$post_type = 'post';

		// Count posts
		$cpt_count = wp_count_posts( $post_type );

		if ( $cpt_count->pending ) {
		
			// Locate the key of 
			if ( function_exists( 'ghostpool_recursive_search' ) ) {
				$key = ghostpool_recursive_search( 'edit.php', $menu );
			}
			
			// Not found, just in case 
			if ( ! $key )
				return;

			// Modify menu item
			$menu[$key][0] .= sprintf(
				' <span class="awaiting-mod count-%1$s" ><span class="pending-count">%1$s</span></span>',
				$cpt_count->pending 
			);
		}
	
	}
}		
add_action( 'admin_menu', 'ghostpool_pending_user_reviews' );

/**
 * Secondary featured image
 *
 */
if ( ! function_exists( 'ghostpool_secondary_images' ) ) {
    function ghostpool_secondary_images() {
		if ( function_exists( 'kdmfi_get_featured_image_id' ) ) {
			$args_1 = array(
				'id' => 'ghostpool-featured-image-2',
				'desc' => esc_html__( 'Used in the hub header and lists.', 'huber' ),
				'label_name' => esc_html__( 'Featured Image 2', 'huber' ),
				'label_set' => esc_html__( 'Set featured image 2', 'huber' ),
				'label_remove' => esc_html__( 'Remove featured image 2', 'huber' ),
				'label_use' => esc_html__( 'Set featured image 2', 'huber' ),
				'post_type' => array( 'post', 'page' ),
			);
			$featured_images[] = $args_1;
			return $featured_images;
		}	
    }
}
add_filter( 'kdmfi_featured_images', 'ghostpool_secondary_images' );

/**
 * Rewrite endpoints for hub tab URLs
 *
 */
if ( ! function_exists( 'ghostpool_hub_tab_endpoints' ) ) {
	function ghostpool_hub_tab_endpoints() {
		
		if ( ghostpool_option( 'reviews_tab' ) == 'enabled' ) { 
			add_rewrite_endpoint( ghostpool_option( 'reviews_tab_parameter' ), EP_PERMALINK | EP_PAGES );
		}
		if ( ghostpool_option( 'previews_tab' ) == 'enabled' ) { 
			add_rewrite_endpoint( ghostpool_option( 'previews_tab_parameter' ), EP_PERMALINK | EP_PAGES );
		}
		if ( ghostpool_option( 'news_tab' ) == 'enabled' ) {
			add_rewrite_endpoint( ghostpool_option( 'news_tab_parameter' ), EP_PERMALINK | EP_PAGES );
		}
		if ( ghostpool_option( 'images_tab' ) == 'enabled' ) { 
			add_rewrite_endpoint( ghostpool_option( 'images_tab_parameter' ), EP_PERMALINK | EP_PAGES );
		}
		if ( ghostpool_option( 'videos_tab' ) == 'enabled' ) { 
			add_rewrite_endpoint( ghostpool_option( 'videos_tab_parameter' ), EP_PERMALINK | EP_PAGES );
		}
		if ( ghostpool_option( 'forums_tab' ) == 'enabled' ) { 
			add_rewrite_endpoint( ghostpool_option( 'forums_tab_parameter' ), EP_PERMALINK | EP_PAGES );
		}
		if ( $custom_tabs = ghostpool_option( 'global_custom_tabs' ) ) {
			foreach( $custom_tabs as $custom_tab ) {
				preg_match_all( "/(.*):/", $custom_tab, $matches );
				$endpoint = isset( $matches[1][0] ) ? $matches[1][0] : $custom_tab;
				add_rewrite_endpoint( $endpoint, EP_PERMALINK | EP_PAGES );
			}
		}
		
	}
}
add_action( 'init', 'ghostpool_hub_tab_endpoints' );

/**
 * Advanced background ad support
 *
 */
if ( ! function_exists( 'ghostpool_advanced_ads_background_selector' ) ) {
	function ghostpool_advanced_ads_background_selector(){
		return '#gp-page-wrapper, #gp-featured-box';
	}
}
add_filter( 'advanced-ads-pro-background-selector', 'ghostpool_advanced_ads_background_selector' );

/**
 * TGM Plugin Activation class
 *
 */
if ( version_compare( phpversion(), '5.2.4', '>=' ) ) {
	require_once( get_template_directory() . '/lib/inc/class-tgm-plugin-activation.php' );
}

if ( ! function_exists( 'ghostpool_register_required_plugins' ) ) {
	
	function ghostpool_register_required_plugins() {

		$plugins = array(

			array(
				'name'               => esc_html__( 'Huber Plugin', 'huber' ),
				'slug'               => 'huber-plugin',
				'source'             => get_template_directory() . '/lib/plugins/huber-plugin.zip',
				'required'           => true,
				'version'            => '2.3.13',
				'force_activation'   => false,
				'force_deactivation' => false,
			),

			array(
				'name'               => esc_html__( 'WPBakery Page Builder', 'huber' ),
				'slug'               => 'js_composer',
				'source'             => get_template_directory() . '/lib/plugins/js_composer.zip',
				'required'           => true,
				'version'            => '6.8.0',
				'force_activation'	 => false,
				'force_deactivation' => false,
			),
			
			array(
				'name'   		     => esc_html__( 'Theia Sticky Sidebar', 'huber' ),
				'slug'   		     => 'theia-sticky-sidebar',
				'source'   		     => get_template_directory() . '/lib/plugins/theia-sticky-sidebar.zip',
				'required'   		 => false,
				'version'   		 => '1.8.0',
				'force_activation'	 => false,
				'force_deactivation' => false,
			),
			
			array(
				'name'   		     => esc_html__( 'WordPress Popular Posts', 'huber' ),
				'slug'   		     => 'wordpress-popular-posts',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'Advanced Ads', 'huber' ),
				'slug'   		     => 'advanced-ads',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'BuddyPress', 'huber' ),
				'slug'   		     => 'buddypress',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'bbPress', 'huber' ),
				'slug'   		     => 'bbpress',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'WooCommerce', 'huber' ),
				'slug'   		     => 'woocommerce',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'Google Captcha', 'huber' ),
				'slug'   		     => 'google-captcha',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'Mashshare Share Buttons', 'huber' ),
				'slug'   		     => 'mashsharer',
				'required'   		 => false,
			),
			
			array(
				'name'   		     => esc_html__( 'Multiple Featured Images', 'huber' ),
				'slug'   		     => 'multiple-featured-images',
				'required'   		 => false,
			),
			
			array(
				'name'      		=> esc_html__( 'Yoast SEO', 'huber' ),
				'slug'      		=> 'wordpress-seo',
				'required' 			=> false,
				'is_callable'		=> 'wpseo_init',
			),
			
			array(
				'name'      		=> esc_html__( 'Envato Market', 'huber' ),
				'slug'      		=> 'envato-market',
				'source'			=> 'https://envato.github.io/wp-envato-market/dist/envato-market.zip',
				'required' 			=> false,
			),
																													
		);

		$config = array(
			'id'           => 'huber',
			'default_path' => '',
			'menu'         => 'tgmpa-install-plugins',
			'has_notices'  => true,                 
			'dismissable'  => true,                  
			'dismiss_msg'  => '',
			'is_automatic' => true,
			'message'      => '',
		);
 
		tgmpa( $plugins, $config );

	}
	
}
add_action( 'tgmpa_register', 'ghostpool_register_required_plugins' );