<?php

if ( ! function_exists( 'healthfirst_is_installed' ) ) {
	/**
	 * Function that checks if forward plugin installed
	 *
	 * @param string $plugin - plugin name
	 *
	 * @return bool
	 */
	function healthfirst_is_installed( $plugin ) {

		switch ( $plugin ) {
			case 'framework':
				return class_exists( 'QodeFramework' );
				break;
			case 'core':
				return class_exists( 'HealthFirstCore' );
				break;
			case 'woocommerce':
				return class_exists( 'WooCommerce' );
				break;
			case 'gutenberg-page':
				$current_screen = function_exists( 'get_current_screen' ) ? get_current_screen() : array();

				return method_exists( $current_screen, 'is_block_editor' ) && $current_screen->is_block_editor();
				break;
			case 'gutenberg-editor':
				return class_exists( 'WP_Block_Type' );
				break;
			default:
				return false;
		}
	}
}

if ( ! function_exists( 'healthfirst_include_theme_is_installed' ) ) {
	/**
	 * Function that set case is installed element for framework functionality
	 *
	 * @param bool $installed
	 * @param string $plugin - plugin name
	 *
	 * @return bool
	 */
	function healthfirst_include_theme_is_installed( $installed, $plugin ) {

		if ( 'theme' === $plugin ) {
			return class_exists( 'Healthfirst_Handler' );
		}

		return $installed;
	}

	add_filter( 'qode_framework_filter_is_plugin_installed', 'healthfirst_include_theme_is_installed', 10, 2 );
}

if ( ! function_exists( 'healthfirst_template_part' ) ) {
	/**
	 * Function that echo module template part.
	 *
	 * @param string $module name of the module from inc folder
	 * @param string $template full path of the template to load
	 * @param string $slug
	 * @param array $params array of parameters to pass to template
	 */
	function healthfirst_template_part( $module, $template, $slug = '', $params = array() ) {
		echo healthfirst_get_template_part( $module, $template, $slug, $params ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'healthfirst_get_template_part' ) ) {
	/**
	 * Function that load module template part.
	 *
	 * @param string $module name of the module from inc folder
	 * @param string $template full path of the template to load
	 * @param string $slug
	 * @param array $params array of parameters to pass to template
	 *
	 * @return string - string containing html of template
	 */
	function healthfirst_get_template_part( $module, $template, $slug = '', $params = array() ) {
		//HTML Content from template
		$html          = '';
		$template_path = HEALTHFIRST_INC_ROOT_DIR . '/' . $module;

		$temp = $template_path . '/' . $template;
		if ( is_array( $params ) && count( $params ) ) {
			extract( $params ); // @codingStandardsIgnoreLine
		}

		$template = '';

		if ( ! empty( $temp ) ) {
			if ( ! empty( $slug ) ) {
				$template = "{$temp}-{$slug}.php";

				if ( ! file_exists( $template ) ) {
					$template = $temp . '.php';
				}
			} else {
				$template = $temp . '.php';
			}
		}

		if ( $template ) {
			ob_start();
			include( $template ); // phpcs:ignore WPThemeReview.CoreFunctionality.FileInclude.FileIncludeFound
			$html = ob_get_clean();
		}

		return $html;
	}
}

if ( ! function_exists( 'healthfirst_get_page_id' ) ) {
	/**
	 * Function that returns current page id
	 * Additional conditional is to check if current page is any wp archive page (archive, category, tag, date etc.) and returns -1
	 *
	 * @return int
	 */
	function healthfirst_get_page_id() {
		$page_id = get_queried_object_id();

		if ( healthfirst_is_wp_template() ) {
			$page_id = - 1;
		}

		return apply_filters( 'healthfirst_filter_page_id', $page_id );
	}
}

if ( ! function_exists( 'healthfirst_is_wp_template' ) ) {
	/**
	 * Function that checks if current page default wp page
	 *
	 * @return bool
	 */
	function healthfirst_is_wp_template() {
		return is_archive() || is_search() || is_404() || ( is_front_page() && is_home() );
	}
}

if ( ! function_exists( 'healthfirst_get_ajax_status' ) ) {
	/**
	 * Function that return status from ajax functions
	 *
	 * @param string $status - success or error
	 * @param string $message - ajax message value
	 * @param string|array $data - returned value
	 * @param string $redirect - url address
	 */
	function healthfirst_get_ajax_status( $status, $message, $data = null, $redirect = '' ) {
		$response = array(
			'status'   => esc_attr( $status ),
			'message'  => esc_html( $message ),
			'data'     => $data,
			'redirect' => ! empty( $redirect ) ? esc_url( $redirect ) : '',
		);

		$output = json_encode( $response );

		exit( $output ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'healthfirst_get_button_element' ) ) {
	/**
	 * Function that returns button with provided params
	 *
	 * @param array $params - array of parameters
	 *
	 * @return string - string representing button html
	 */
	function healthfirst_get_button_element( $params, $class ) {
		if ( class_exists( 'HealthFirstCore_Button_Shortcode' ) ) {
			return HealthFirstCore_Button_Shortcode::call_shortcode( $params );
		} else {
			$link   = isset( $params['link'] ) ? $params['link'] : '#';
			$target = isset( $params['target'] ) ? $params['target'] : '_self';
			$text   = isset( $params['text'] ) ? $params['text'] : '';

			if ( ! empty( $class ) && 'qodef-size-textual--small' === $class ) {
				$return_html = '<a itemprop="url" class="qodef-theme-button qodef-button qodef-layout--textual' . ' ' . $class . '"' . 'href="' . esc_url( $link ) . '" target="' . esc_attr( $target ) . '">';
				$return_html .= '<span class="qodef-m-text">' . esc_html( $text ) . '</span>';
				$return_html .= '<span class="qodef-m-zigzag"><svg xmlns="http://www.w3.org/2000/svg" width="496.46" height="5.5" viewBox="0 0 496.46 5.5"><g><g><g><path d="M1.05,4.75l5.31-4,4.65,4,4.34-4,4.34,4,5.44-4,4,4,5.16-4,5.31,4,5.1-4,4.88,4" transform="translate(-0.3 0)"/><path d="M49.57,4.75l5.32-4,4.65,4,4.34-4,4.33,4,5.44-4,4,4,5.16-4,5.3,4,5.1-4,4.89,4" transform="translate(-0.3 0)"/></g><g><path d="M98.05,4.75l5.31-4,4.65,4,4.34-4,4.34,4,5.44-4,4,4,5.16-4,5.31,4,5.1-4,4.88,4" transform="translate(-0.3 0)"/><path d="M146.57,4.75l5.32-4,4.65,4,4.34-4,4.33,4,5.44-4,4,4,5.15-4,5.31,4,5.1-4,4.88,4" transform="translate(-0.3 0)"/></g><g><path d="M195.05,4.75l5.31-4,4.65,4,4.34-4,4.34,4,5.44-4,4,4,5.16-4,5.31,4,5.1-4,4.88,4" transform="translate(-0.3 0)"/><path d="M243.57,4.75l5.32-4,4.65,4,4.34-4,4.33,4,5.44-4,4,4,5.15-4,5.31,4,5.1-4,4.89,4" transform="translate(-0.3 0)"/></g><g><path d="M292.05,4.75l5.31-4,4.65,4,4.34-4,4.34,4,5.44-4,4,4,5.16-4,5.31,4,5.1-4,4.88,4" transform="translate(-0.3 0)"/><path d="M340.57,4.75l5.32-4,4.65,4,4.34-4,4.33,4,5.44-4,4,4,5.15-4,5.31,4,5.1-4,4.89,4" transform="translate(-0.3 0)"/></g><g><path d="M389.05,4.75l5.31-4,4.65,4,4.34-4,4.34,4,5.44-4,4,4,5.16-4,5.31,4,5.1-4,4.88,4" transform="translate(-0.3 0)"/><path d="M437.57,4.75l5.32-4,4.65,4,4.34-4,4.33,4,5.44-4,4,4,5.15-4,5.31,4,5.1-4,4.89,4" transform="translate(-0.3 0)"/></g></g><path d="M486.05,4.75l5.31-4,4.65,4" transform="translate(-0.3 0)"/></g></svg></span>';
				$return_html .= '</a>';
			} else {
				$return_html = '<a itemprop="url" class="qodef-theme-button" href="' . esc_url( $link ) . '" target="' . esc_attr( $target ) . '">' . esc_html( $text ) . '</a>';
			}

			return $return_html;
		}
	}
}

if ( ! function_exists( 'healthfirst_render_button_element' ) ) {
	/**
	 * Function that render button with provided params
	 *
	 * @param array $params - array of parameters
	 */
	function healthfirst_render_button_element( $params, $class_name = '' ) {
		$class = isset( $class_name ) && ! empty( $class_name ) ? esc_attr( $class_name ) : '';
		echo healthfirst_get_button_element( $params, $class ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'healthfirst_class_attribute' ) ) {
	/**
	 * Function that render class attribute
	 *
	 * @param string|array $class
	 */
	function healthfirst_class_attribute( $class ) {
		echo healthfirst_get_class_attribute( $class ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'healthfirst_get_class_attribute' ) ) {
	/**
	 * Function that return class attribute
	 *
	 * @param string|array $class
	 *
	 * @return string|mixed
	 */
	function healthfirst_get_class_attribute( $class ) {
		$value = healthfirst_is_installed( 'framework' ) ? qode_framework_get_class_attribute( $class ) : '';

		return $value;
	}
}

if ( ! function_exists( 'healthfirst_get_post_value_through_levels' ) ) {
	/**
	 * Function that returns meta value if exists
	 *
	 * @param string $name name of option
	 * @param int $post_id id of
	 *
	 * @return string value of option
	 */
	function healthfirst_get_post_value_through_levels( $name, $post_id = null ) {
		return healthfirst_is_installed( 'framework' ) && healthfirst_is_installed( 'core' ) ? healthfirst_core_get_post_value_through_levels( $name, $post_id ) : '';
	}
}

if ( ! function_exists( 'healthfirst_get_space_value' ) ) {
	/**
	 * Function that returns spacing value based on selected option
	 *
	 * @param string $text_value - textual value of spacing
	 *
	 * @return int
	 */
	function healthfirst_get_space_value( $text_value ) {
		return healthfirst_is_installed( 'core' ) ? healthfirst_core_get_space_value( $text_value ) : 0;
	}
}

if ( ! function_exists( 'healthfirst_wp_kses_html' ) ) {
	/**
	 * Function that does escaping of specific html.
	 * It uses wp_kses function with predefined attributes array.
	 *
	 * @param string $type - type of html element
	 * @param string $content - string to escape
	 *
	 * @return string escaped output
	 * @see wp_kses()
	 *
	 */
	function healthfirst_wp_kses_html( $type, $content ) {
		return healthfirst_is_installed( 'framework' ) ? qode_framework_wp_kses_html( $type, $content ) : $content;
	}
}

if ( ! function_exists( 'healthfirst_render_svg_icon' ) ) {
	/**
	 * Function that print svg html icon
	 *
	 * @param string $name - icon name
	 * @param string $class_name - custom html tag class name
	 */
	function healthfirst_render_svg_icon( $name, $class_name = '' ) {
		echo healthfirst_get_svg_icon( $name, $class_name ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'healthfirst_get_svg_icon' ) ) {
	/**
	 * Returns svg html
	 *
	 * @param string $name - icon name
	 * @param string $class_name - custom html tag class name
	 *
	 * @return string|html
	 */
	function healthfirst_get_svg_icon( $name, $class_name = '' ) {
		$html  = '';
		$class = isset( $class_name ) && ! empty( $class_name ) ? 'class="' . esc_attr( $class_name ) . '"' : '';

		switch ( $name ) {
			case 'menu':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64px" height="64px" viewBox="0 0 64 64" enable-background="new 0 0 64 64" xml:space="preserve"><line x1="12" y1="21" x2="52" y2="21"/><line x1="12" y1="33" x2="52" y2="33"/><line x1="12" y1="45" x2="52" y2="45"/></svg>';
				break;
			case 'search':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="20" height="20" viewBox="0 0 20 20"><path d="M18.869 19.162l-5.943-6.484c1.339-1.401 2.075-3.233 2.075-5.178 0-2.003-0.78-3.887-2.197-5.303s-3.3-2.197-5.303-2.197-3.887 0.78-5.303 2.197-2.197 3.3-2.197 5.303 0.78 3.887 2.197 5.303 3.3 2.197 5.303 2.197c1.726 0 3.362-0.579 4.688-1.645l5.943 6.483c0.099 0.108 0.233 0.162 0.369 0.162 0.121 0 0.242-0.043 0.338-0.131 0.204-0.187 0.217-0.503 0.031-0.706zM1 7.5c0-3.584 2.916-6.5 6.5-6.5s6.5 2.916 6.5 6.5-2.916 6.5-6.5 6.5-6.5-2.916-6.5-6.5z"></path></svg>';
				break;
			case 'star':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="32" height="32" viewBox="0 0 32 32"><g><path d="M 20.756,11.768L 15.856,1.84L 10.956,11.768L0,13.36L 7.928,21.088L 6.056,32L 15.856,26.848L 25.656,32L 23.784,21.088L 31.712,13.36 z"></path></g></svg>';
				break;
			case 'menu-arrow-right':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" width="10.331" height="10.229" viewBox="0 0 10.331 10.229"><g transform="translate(5.292 -0.071) rotate(45)"><path d="M0,.5H6.826V7.339" fill="none" stroke="currentColor" stroke-miterlimit="10" stroke-width="0.8"/><line x1="6.522" y2="6.547" transform="translate(0.246 0.617)" fill="none" stroke="currentColor" stroke-linejoin="bevel" stroke-width="0.8" style="isolation: isolate"/></g></svg>';
				break;
			case 'slider-arrow-left':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 26 50.9" xml:space="preserve"><polyline points="25.6,0.4 0.7,25.5 25.6,50.6 "/></svg>';
				break;
			case 'slider-arrow-right':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 26 50.9" xml:space="preserve"><polyline points="0.4,50.6 25.3,25.5 0.4,0.4 "/></svg>';
				break;
			case 'pagination-arrow-left':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="12.757" height="12.625" viewBox="0 0 12.757 12.625"><g transform="translate(6.66 12.271) rotate(-135)"><path d="M0,0H8.419V8.435" fill="none" stroke="currentColor" stroke-miterlimit="10" stroke-width="1"/><line x1="8.043" y2="8.075" transform="translate(0.304 0.144)" fill="none" stroke="currentColor" stroke-linejoin="bevel" stroke-width="1" style="isolation: isolate"/></g></svg>';
				break;
			case 'pagination-arrow-right':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="12.757" height="12.625" viewBox="0 0 12.757 12.625"><g transform="translate(6.097 0.354) rotate(45)"><path d="M0,0H8.419V8.435" transform="translate(0 0)" fill="none" stroke="currentColor" stroke-miterlimit="10" stroke-width="1"/><line x1="8.043" y2="8.075" transform="translate(0.304 0.144)" fill="none" stroke="currentColor" stroke-linejoin="bevel" stroke-width="1" style="isolation: isolate"/></g></svg>';
				break;
			case 'close':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="32" height="32" viewBox="0 0 32 32"><g><path d="M 10.050,23.95c 0.39,0.39, 1.024,0.39, 1.414,0L 17,18.414l 5.536,5.536c 0.39,0.39, 1.024,0.39, 1.414,0 c 0.39-0.39, 0.39-1.024,0-1.414L 18.414,17l 5.536-5.536c 0.39-0.39, 0.39-1.024,0-1.414c-0.39-0.39-1.024-0.39-1.414,0 L 17,15.586L 11.464,10.050c-0.39-0.39-1.024-0.39-1.414,0c-0.39,0.39-0.39,1.024,0,1.414L 15.586,17l-5.536,5.536 C 9.66,22.926, 9.66,23.56, 10.050,23.95z"></path></g></svg>';
				break;
			case 'spinner':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" width="512" height="512" viewBox="0 0 512 512"><path d="M304 48c0 26.51-21.49 48-48 48s-48-21.49-48-48 21.49-48 48-48 48 21.49 48 48zm-48 368c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.49-48-48-48zm208-208c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.49-48-48-48zM96 256c0-26.51-21.49-48-48-48S0 229.49 0 256s21.49 48 48 48 48-21.49 48-48zm12.922 99.078c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48c0-26.509-21.491-48-48-48zm294.156 0c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48c0-26.509-21.49-48-48-48zM108.922 60.922c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.491-48-48-48z"></path></svg>';
				break;
			case 'link':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="32.06999969482422" height="33.58000183105469" viewBox="0 0 32.06999969482422 33.58000183105469"><g><path d="M 7.54,15.77c 1.278,1.278, 3.158,1.726, 4.868,1.216L 2.96,7.54C 2.652,7.232, 2.49,6.786, 2.49,6.254 c0-0.88, 0.46-2.004, 1.070-2.614c 0.8-0.8, 2.97-1.686, 3.98-0.682l 9.446,9.448c 0.138-0.462, 0.208-0.942, 0.208-1.422 c0-1.304-0.506-2.526-1.424-3.446L 9.364,1.134C 7.44-0.79, 3.616-0.068, 1.734,1.814C 0.642,2.906-0.036,4.598-0.036,6.23 c0,1.268, 0.416,2.382, 1.17,3.136L 7.54,15.77zM 24.46,16.23c-1.278-1.278-3.158-1.726-4.868-1.216l 9.448,9.448c 0.308,0.308, 0.47,0.752, 0.47,1.286 c0,0.88-0.46,2.004-1.070,2.614c-0.8,0.8-2.97,1.686-3.98,0.682L 15.014,19.594c-0.138,0.462-0.208,0.942-0.208,1.422 c0,1.304, 0.506,2.526, 1.424,3.446l 6.404,6.404c 1.924,1.924, 5.748,1.202, 7.63-0.68c 1.092-1.092, 1.77-2.784, 1.77-4.416 c0-1.268-0.416-2.382-1.17-3.136L 24.46,16.23zM 9.164,9.162C 8.908,9.416, 8.768,9.756, 8.768,10.116s 0.14,0.698, 0.394,0.952l 11.768,11.77 c 0.526,0.524, 1.38,0.524, 1.906,0c 0.256-0.254, 0.394-0.594, 0.394-0.954s-0.14-0.698-0.394-0.952L 11.068,9.162 C 10.544,8.638, 9.688,8.638, 9.164,9.162z"></path></g></svg>';
				break;
		}

		return apply_filters( 'healthfirst_filter_svg_icon', $html );
	}
}
