<?php
/**
 * Grider
 * Grider - Creating a dynamic grid layout has never been easier.
 * Exclusively on https://1.envato.market/grider-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.0
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\GriderElementor;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

use Exception;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Utils;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Css_Filter;
use Merkulove\GriderElementor\Unity\Plugin as UnityPlugin;
use Merkulove\GriderElementor\Unity\Settings;

/** @noinspection PhpUnused */
/**
 * Grider - Custom Elementor Widget.
 **/
class grider_elementor extends Widget_Base {

    /**
     * Use this to sort widgets.
     * A smaller value means earlier initialization of the widget.
     * Can take negative values.
     * Default widgets and widgets from 3rd party developers have 0 $mdp_order
     **/
    public $mdp_order = 1;

    /**
     * Widget base constructor.
     * Initializing the widget base class.
     *
     * @access public
     * @throws Exception If arguments are missing when initializing a full widget instance.
     * @param array      $data Widget data. Default is an empty array.
     * @param array|null $args Optional. Widget default arguments. Default is null.
     *
     * @return void
     **/
    public function __construct( $data = [], $args = null ) {

        parent::__construct( $data, $args );

        wp_register_style(
    'mdp-grider-elementor-admin',
        UnityPlugin::get_url() . 'src/Merkulove/Unity/assets/css/elementor-admin' . UnityPlugin::get_suffix() . '.css',
            [],
            UnityPlugin::get_version()
        );

        wp_register_style(
    'mdp-grider-elementor',
        UnityPlugin::get_url() . 'css/grider-elementor' . UnityPlugin::get_suffix() . '.css',
            [],
            UnityPlugin::get_version()
        );

	    wp_register_script(
    'mdp-grider-elementor',
        UnityPlugin::get_url() . 'js/grider-elementor' . UnityPlugin::get_suffix() . '.js',
            [ 'elementor-frontend', 'wp-api'],
            UnityPlugin::get_version(),
    true
        );

    }

    /**
     * Return a widget name.
     *
     * @return string
     **/
    public function get_name() {

        return 'mdp-grider-elementor';

    }

    /**
     * Return the widget title that will be displayed as the widget label.
     *
     * @return string
     **/
    public function get_title() {

        return esc_html__( 'Grider', 'grider-elementor' );

    }

    /**
     * Set the widget icon.
     *
     * @return string
     */
    public function get_icon() {

        return 'mdp-grider-elementor-widget-icon';

    }

    /**
     * Set the category of the widget.
     *
     * @return array with category names
     **/
    public function get_categories() {

        return [ 'general' ];

    }

    /**
     * Get widget keywords. Retrieve the list of keywords the widget belongs to.
     *
     * @access public
     *
     * @return array Widget keywords.
     **/
    public function get_keywords() {

        return [ 'Merkulove', 'Grider', 'grid', 'post', 'product', 'woo' ];

    }

    /**
     * Get style dependencies.
     * Retrieve the list of style dependencies the widget requires.
     *
     * @access public
     *
     * @return array Widget styles dependencies.
     **/
    public function get_style_depends() {

        return [ 'mdp-grider-elementor', 'mdp-grider-elementor-admin' ];

    }

	/**
	 * Get script dependencies.
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @access public
     *
	 * @return array Element scripts dependencies.
	 **/
	public function get_script_depends() {

		return [ 'mdp-grider-elementor' ];

    }

	/**
	 * Group controllers for setting styles.
	 *
	 * @param $name_Section - The name of the styles section.
	 * @param string $css_class - Specifies one or more class names for an element.
	 * @param array $condition - Interconnected controllers.
	 * @param array $display - Determines which controllers to display. {
	 *      @type $key 'margin'                 => (boolean) - Use margin.
	 *      @type $key 'padding'                => (boolean) - Use padding.
	 *      @type $key 'color'                  => (boolean) - Use color.
	 *      @type $key 'typography'             => (boolean) - Use typography.
	 *      @type $key 'text-shadow'            => (boolean) - Use text shadow.
	 *      @type $key 'alignment'              => (boolean) - Use alignment.
	 *      @type $key 'alignment-justify'      => (boolean) - Use alignment justify.
	 *      @type $key 'alignment-horizontal'   => (boolean) - Use alignment horizontal.
	 *      @type $key 'alignment-vertical'     => (boolean) - Use alignment vertical.
	 *      @type $key 'background'             => (boolean) - Use background.
	 *      @type $key 'border'                 => (boolean) - Use border.
	 *      @type $key 'border-radius'          => (boolean) - Use border radius.
	 *      @type $key 'box-shadow'             => (boolean) - Use box shadow.
	 *      @type $key 'css-filter'             => (boolean) - Use css filter.
	 * }
	 * @param array $separator
     *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return void
	 */
	public function get_settings_style_group( $name_Section,
        $css_class, $condition = [], $display = [], $separator = [ 'after', 'before', 'before'] ) {

		/** Margin. */
		if ( isset($display['margin']) && $display['margin'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_margin',
				[
					'label'      => esc_html__( 'Margin', 'grider-elementor' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} ' . $css_class => 'margin: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
					],
					'toggle'     => true,
					'condition'  => $condition,
				]
			);
		}

		/** Padding. */
		if ( isset($display['padding']) && $display['padding'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_padding',
				[
					'label'      => esc_html__( 'Padding', 'grider-elementor' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} ' . $css_class => 'padding: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
					],
					'toggle'     => true,
					'separator'  => $separator[0],
					'condition'  => $condition,
				]
			);
		}

		/** Text Color. */
		if ( isset($display['color']) && $display['color'] === true ) {
			$this->add_control(
				'grider_' . $name_Section . '_text_color',
				[
					'label'     => esc_html__( 'Color', 'grider-elementor' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} ' . $css_class => 'color: {{VALUE}}',
					],
					'condition' => $condition,
				]
			);
		}

		/** Typography */
		if ( isset($display['typography']) && $display['typography'] === true ) {
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'      => 'grider_' . $name_Section . '_typography',
					'label'     => esc_html__( 'Typography', 'grider-elementor' ),
					'scheme'    => Typography::TYPOGRAPHY_1,
					'selector'  => '{{WRAPPER}} ' . $css_class,
					'condition' => $condition,
				]
			);
		}

		/** Text Shadow. */
		if ( isset($display['text-shadow']) && $display['text-shadow'] === true ) {
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'      => 'grider_' . $name_Section . '_text_shadow',
					'label'     => esc_html__( 'Text Shadow', 'grider-elementor' ),
					'selector'  => '{{WRAPPER}} ' . $css_class,
					'condition' => $condition,
				]
			);
		}

		/** Alignment horizontal. */
		if ( isset($display['alignment-horizontal']) && $display['alignment-horizontal'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_horizontal_align',
				[
					'label'     => esc_html__( 'Horizontal Position', 'grider-elementor' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'flex-start'    => [
							'title' => esc_html__( 'Left', 'grider-elementor' ),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => esc_html__( 'Center', 'grider-elementor' ),
							'icon'  => 'fa fa-align-center',
						],
						'flex-end'   => [
							'title' => esc_html__( 'Right', 'grider-elementor' ),
							'icon'  => 'fa fa-align-right',
						],
					],
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} ' . $css_class => 'justify-content: {{header_align}};',
						'{{WRAPPER}} .mdp-grider-box-content' => 'width: 65%;',
					],
					'toggle'    => true,
					'condition' => $condition,
				]
			);
		}

		/** Alignment vertical. */
		if ( isset($display['alignment-vertical']) && $display['alignment-vertical'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_vertical_align',
				[
					'label'     => esc_html__( 'Vertical Position', 'grider-elementor' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'flex-start'    => [
							'title' => esc_html__( 'Top', 'grider-elementor' ),
							'icon'  => 'eicon-v-align-top',
						],
						'center'  => [
							'title' => esc_html__( 'Center', 'grider-elementor' ),
							'icon'  => 'eicon-v-align-middle',
						],
						'flex-end'   => [
							'title' => esc_html__( 'Bottom', 'grider-elementor' ),
							'icon'  => 'eicon-v-align-bottom',
						],
					],
					'default'   => 'center',
					'selectors' => [
						'{{WRAPPER}} ' . $css_class => 'align-items: {{value}};',
					],
					'toggle'    => true,
					'condition' => $condition,
				]
			);
		}

		/** Alignment. */
		if ( isset($display['alignment']) && $display['alignment'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_align',
				[
					'label'     => esc_html__( 'Alignment', 'grider-elementor' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => esc_html__( 'Left', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center'  => [
							'title' => esc_html__( 'Center', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-center',
						],
						'right'   => [
							'title' => esc_html__( 'Right', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'default'   => 'center',
					'selectors' => [
						'{{WRAPPER}} ' . $css_class => 'text-align: {{header_align}};',
					],
					'toggle'    => true,
					'condition' => $condition,
				]
			);
		}

		/** Alignment justify. */
		if ( isset($display['alignment-justify']) && $display['alignment-justify'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_align_justify',
				[
					'label'     => esc_html__( 'Alignment', 'grider-elementor' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => esc_html__( 'Left', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center'  => [
							'title' => esc_html__( 'Center', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-center',
						],
						'right'   => [
							'title' => esc_html__( 'Right', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-right',
						],
						'justify' => [
							'title' => esc_html__( 'Justify', 'grider-elementor' ),
							'icon'  => 'eicon-text-align-justify',
						],
					],
					'default'   => 'center',
					'selectors' => [
						'{{WRAPPER}} ' . $css_class => 'text-align: {{header_align}};',
					],
					'toggle'    => true,
					'condition' => $condition,
				]
			);
		}

		/** Background. */
		if ( isset($display['background']) && $display['background'] === true ) {
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'      => 'grider_' . $name_Section . '_background',
					'label'     => esc_html__( 'Background', 'grider-elementor' ),
					'types'     => [ 'classic', 'gradient' ],
					'separator' => $separator[1],
					'selector'  => '{{WRAPPER}} ' . $css_class,
					'condition' => $condition,
				]
			);
		}

		/** Border. */
		if ( isset($display['border']) && $display['border'] === true ) {
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'grider_' . $name_Section . '_border',
					'label'     => esc_html__( 'Border', 'grider-elementor' ),
					'selector'  => '{{WRAPPER}} ' . $css_class,
					'separator' => $separator[2],
					'condition' => $condition,
				]
			);
		}

		/** Border Radius. */
		if ( isset($display['border-radius']) && $display['border-radius'] === true ) {
			$this->add_responsive_control(
				'grider_' . $name_Section . '_border_radius',
				[
					'label'      => esc_html__( 'Border Radius', 'grider-elementor' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'em' ],
					'selectors'  => [
						'{{WRAPPER}} ' . $css_class => 'border-radius: {{top}}{{unit}} {{right}}{{unit}} {{bottom}}{{unit}} {{left}}{{unit}};',
					],
					'toggle'     => true,
					'condition'  => $condition,
				]
			);
		}

		/** Box Shadow. */
		if ( isset($display['box-shadow']) && $display['box-shadow'] === true ) {
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'      => 'grider_' . $name_Section . '_box_shadow',
					'label'     => esc_html__( 'Box Shadow', 'grider-elementor' ),
					'selector'  => '{{WRAPPER}} ' . $css_class,
					'condition' => $condition,
				]
			);
		}

		/** CSS filter. */
		if ( isset($display['css-filter']) && $display['css-filter'] === true ) {
			$this->add_group_control(
				Group_Control_Css_Filter::get_type(),
				[
					'name'     => 'grider_' . $name_Section . '_css_filter',
					'label'    => esc_html__( 'CSS filter', 'grider-elementor' ),
					'selector' => '{{WRAPPER}} ' . $css_class,
					'condition' => $condition,
				]
			);
		}

	}

	/**
	 * Get the ID of the very first posts category.
	 */
	public function get_post_category_default() {
		$wp_cat = get_categories('&hide_empty=0');

		$options = [];

		/** Add available post categories. */
		foreach( $wp_cat as $cat_item ){
			$options[] =  $cat_item->cat_ID;
		}

		return min($options);
	}

	/**
	 * We write the existing categories of posts into an array.
	 */
	public function get_post_category() {

		$wp_cat = get_categories('&hide_empty=0');
		$options = array();

		/** Add available post categories. */
		foreach( $wp_cat as $cat_item ){
			$options[$cat_item->cat_ID] =  $cat_item->name;
		}

		return $options;
	}

	/**
	 * Get sources
	 */
	public function get_source() {

		$source = [
			'custom'  => esc_html__( 'Custom', 'grider-elementor' ),
			'posts' => esc_html__( 'Posts', 'grider-elementor' ),
			'gallery' => esc_html__( 'Gallery', 'grider-elementor' ),
		];

		if( is_plugin_active('woocommerce/woocommerce.php') ){
			$source += [ 'woocommerce' => esc_html__( 'WooCommerce', 'grider-elementor' ) ];
        }

	    return $source;
	}

    /**
     *  Get the ID of the very first product's category.
     */
	public function get_products_category_default() {

		$wp_cat = get_categories( array(
			'taxonomy'     => 'product_cat',
			'orderby'      => 'name',
			'pad_counts'   => false,
			'hierarchical' => 1,
			'hide_empty'   => false
		) );

		if( empty($wp_cat) ){
		    return array();
        }

		$options = [];

		/** Add available product categories. */
		foreach( $wp_cat as $cat_item ){
			$options[] =  $cat_item->cat_ID;
		}

		return min($options);

	}
	
	/**
	 * We write the existing product categories into an array.
	 */
	public function get_products_category() {

		$wp_cat = get_categories( array(
			'taxonomy'     => 'product_cat',
			'orderby'      => 'name',
			'pad_counts'   => false,
			'hierarchical' => 1,
			'hide_empty'   => false
		) );

		if( empty($wp_cat) ){
			return array();
		}

		/** Add available product categories. */
        $options = [];
		foreach( $wp_cat as $cat_item ){
			$options[$cat_item->cat_ID] =  $cat_item->name;
		}

		return $options;
	}


	/**
     * Add the widget controls.
     *
     * @access protected
     * @return void with category names
     **/
    protected function register_controls() {

        /** Content Tab. */
        $this->tab_content();

        /** Style Tab. */
        $this->tab_style();

    }

    /**
     * Add widget controls on Content tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_content() {

        /** General */
        $this->general_section_content();

        /** Navigation */
	    $this->navigation_section_content();

	    /** Layout */
	    $this->layout_section_content();

	    /** Content */
	    $this->content_section_content();

    }

    /**
     * Add widget controls on Style tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_style() {

        /** Grid item */
        $this->grid_item_section_styles();

        /** Title */
	    $this->title_section_styles();

	    /** Description */
	    $this->description_section_styles();

	    /** Image */
	    $this->image_section_styles();

	    /** Price */
	    $this->price_section_styles();

	    /** Button */
	    $this->button_section_styles();

    }

	/**
	 *  TAB: Content.
	 */

    /**
     * Add widget controls: Content -> Example Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function general_section_content() {

        $this->start_controls_section( 'general_content', [
            'label' => esc_html__( 'General', 'grider-elementor' ),
            'tab'   => Controls_Manager::TAB_CONTENT
        ] );

            /** Source */
            $this->add_control(
                'general_source',
                [
                    'label' => esc_html__( 'Source', 'grider-elementor' ),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'custom',
                    'options' => $this->get_source(),
                ]
            );

            /** Post category. */
            $this->add_control(
                'general_posts_category',
                [
                    'label' => esc_html__( 'Category', 'grider-elementor' ),
                    'type' => Controls_Manager::SELECT,
                    'default' => $this->get_post_category_default(),
                    'options' => $this->get_post_category(),
                    'condition' => [ 'general_source' => 'posts' ]
                ]
            );

            /** Products category. */
            $this->add_control(
                'general_products_category',
                [
                    'label' => esc_html__( 'Category', 'grider-elementor' ),
                    'type' => Controls_Manager::SELECT,
                    'default' => $this->get_products_category_default(),
                    'options' => $this->get_products_category(),
                    'condition' => [ 'general_source' => 'woocommerce' ]
                ]
            );

            /** Link type. */
            $this->add_control(
                'general_link_type',
                [
                    'label' => esc_html__( 'Link type', 'grider-elementor' ),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'none',
                    'options' => [
                        'none'  => esc_html__( 'None', 'grider-elementor' ),
                        'box' => esc_html__( 'Box', 'grider-elementor' ),
                        'title' => esc_html__( 'Title', 'grider-elementor' ),
                        'button' => esc_html__( 'Button', 'grider-elementor' ),
                    ],
                    'condition' => [ 'general_source!' => 'gallery' ]
                ]
            );

            /** Button Text. */
            $this->add_control(
                'general_button_text',
                [
                    'label' => esc_html__( 'Button Text', 'grider-elementor' ),
                    'type' => Controls_Manager::TEXT,
                    'placeholder' => esc_html__( 'Button Text', 'grider-elementor' ),
                    'default' => esc_html__( 'More', 'grider-elementor' ),
                    'label_block' => true,
                    'condition' => [
                        'general_source' => ['posts', 'woocommerce'],
                        'general_link_type' => 'button'
                    ]
                ]
            );

        $this->end_controls_section();

    }

	/**
	 * Navigation section style
	 */
	public function navigation_section_content() {

		$this->start_controls_section( 'navigation_content', [
			'label' => esc_html__( 'Grid', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_CONTENT
		] );

            /** Dynamic content */
            $this->add_control(
                'navigation_dynamic_content',
                [
                    'label'           => esc_html__( 'Amount items in grid', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px' ],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'step' => 1,

                        ],
                    ],
                    'default' => [
                        'unit' => 'px',
                        'size' => 10,
                    ],
                    'conditions' => [
	                    'relation' => 'or',
	                    'terms' => [
		                    [
			                    'name' => 'general_source',
			                    'operator' => '==',
			                    'value' => 'posts'
		                    ],
		                    [
			                    'name' => 'general_source',
			                    'operator' => '==',
			                    'value' => 'woocommerce'
		                    ]
	                    ]
                    ]
                ]
            );

            /** Gutter */
            $this->add_responsive_control(
                'navigation_gutter',
                [
                    'label'           => esc_html__( 'Gutter', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px' ],
                    'range'           => [
                        'px' => [
                            'min'  => 0,
                            'step' => 1,

                        ],
	                    '%' => [
							'min' => 0,
		                    'step' => 1
	                    ]
                    ],
                    'devices' => [ 'desktop', 'tablet', 'mobile' ],
                    'default' => [
	                    'unit' => 'px',
	                    'size' => 10,
                    ],
                    'selectors'  => [
	                    '{{WRAPPER}} .mdp-grider-container'  => 'gap: {{size}}{{unit}}',
                    ],
                ]
            );

            /** Max columns */
            $this->add_responsive_control(
                'navigation_max_columns',
                [
                    'label'           => esc_html__( 'Max columns', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px' ],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'max' => 10,
                            'step' => 1,
                        ],
                    ],
                    'devices' => [ 'desktop', 'tablet', 'mobile' ],
                    'default' => [
                        'size' => '3',
                        'unit' => 'px',
                    ],
                    'tablet_default' => [
	                    'size' => '2',
	                    'unit' => 'px',
                    ],
                    'mobile_default' => [
	                    'size' => '1',
	                    'unit' => 'px',
                    ],
                    'selectors'  => [
	                    '{{WRAPPER}} .mdp-grider-container.mdp-grider-grid'  => 'grid-template-columns: repeat( {{size}}, 1fr )',
	                    '{{WRAPPER}} .mdp-grider-container.mdp-grider-masonry .mdp-grider-item'  => 'width: calc( 100% / {{size}} - ( {{size}} - 1 ) / {{size}} * ( {{navigation_gutter.size}}{{navigation_gutter.unit}} - 0.1{{navigation_gutter.unit}} ) )'
                    ],
                ]
            );

            /** Masonry  */
            $this->add_control(
                'navigation_masonry',
                [
                    'label' => esc_html__( 'Masonry', 'grider-elementor' ),
                    'type' => Controls_Manager::SWITCHER,
                    'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
                    'label_off' => esc_html__( 'No', 'grider-elementor' ),
                    'return_value' => 'yes',
                    'default' => '',
                ]
            );

            /** Animate */
            $this->add_control(
                'navigation_animate',
                [
                    'label' => esc_html__( 'Animate', 'grider-elementor' ),
                    'type' => Controls_Manager::SWITCHER,
                    'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
                    'label_off' => esc_html__( 'No', 'grider-elementor' ),
                    'return_value' => 'yes',
                    'default' => '',
	                'separator' => 'before'
                ]
            );

            /** Transition. */
            $this->add_control(
                'navigation_transition',
                [
                    'label' => esc_html__( 'Grid animation', 'grider-elementor' ),
                    'type' => Controls_Manager::SELECT,
                    'options' => [
                        'none'        =>  esc_html__('None', 'grider-elementor' ),
                        'rise'        =>  esc_html__('Rise', 'grider-elementor' ),
                        'curl'        =>  esc_html__('Curl', 'grider-elementor' ),
                        'drop-in'     =>  esc_html__('Drop in', 'grider-elementor' ),
                        'slide-up'    =>  esc_html__('Slide Up', 'grider-elementor' ),
                        'slide-right' =>  esc_html__('Slide Right', 'grider-elementor' ),
                        'slide-left'  =>  esc_html__('Slide Left', 'grider-elementor' ),
                        'slide-down'  =>  esc_html__('Slide Down', 'grider-elementor' ),
                        'ripple'      =>  esc_html__('Ripple', 'grider-elementor' ),
                        'bounce'      =>  esc_html__('Bounce', 'grider-elementor' ),
                        'flip-x'      =>  esc_html__('Flip X', 'grider-elementor' ),
                        'flip-y'      =>  esc_html__('Flip Y', 'grider-elementor' ),
                        'jello'       =>  esc_html__('Jello', 'grider-elementor' ),
                        'zoom'        =>  esc_html__('Zoom', 'grider-elementor' ),
                        'fade'        =>  esc_html__('Fade', 'grider-elementor' ),
                    ],
                    'default' => 'zoom',
                    'condition' => [ 'navigation_animate' => 'yes' ],
                    'selectors'  => [
	                    '{{WRAPPER}} .mdp-grider-item'  => 'animation-name: grider-{{value}}'
                    ],
                ]
            );

            $this->add_control(
                'navigation_transition_duration',
                [
                    'label'           => esc_html__( 'Transition duration (milliseconds)', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px' ],
                    'range'           => [
                        'px' => [
                            'min'  => 0,
                            'max'  => 5000,
                            'step' => 200,
                        ],
                    ],
                    'default' => [
                        'unit' => 'px',
                        'size' => 400,
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .mdp-grider-item'  => 'animation-duration: {{size}}ms',
                    ],
                    'condition' => [ 'navigation_animate' => 'yes', 'navigation_transition!' => 'none' ]
                ]
            );

		$this->add_control(
			'navigation_animate_delay',
			[
				'label' => esc_html__( 'Increasing delay', 'grider-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
				'label_off' => esc_html__( 'No', 'grider-elementor' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => [ 'navigation_animate' => 'yes' ],
			]
		);

		$this->end_controls_section();

    }

	/**
	 * Layout section style
	 */
	public function layout_section_content() {

		$this->start_controls_section( 'layout_content', [
			'label' => esc_html__( 'Layout', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_CONTENT,
			'condition' => [ 'general_source!' => 'gallery' ]
		] );

            $layout = new Repeater();

            /** Item. */
            $layout->add_control(
                'layout_item',
                [
                    'label'   => esc_html__( 'Item', 'grider-elementor' ),
                    'type'    => Controls_Manager::SELECT,
                    'default' => 'title',
                    'options' => [
                        'image'       => esc_html__( 'Image', 'grider-elementor' ),
                        'title'       => esc_html__( 'Title', 'grider-elementor' ),
                        'description' => esc_html__( 'Description', 'grider-elementor' ),
                        'price'       => esc_html__( 'Price', 'grider-elementor' ),
                        'button'      => esc_html__( 'Button', 'grider-elementor' ),
                    ],
                ]
            );

            /** Priority. */
            $this->add_control(
                'layout_priority',
                [
                    'label'       => esc_html__( 'Priority', 'grider-elementor' ),
                    'type'        => Controls_Manager::REPEATER,
                    'fields'      => $layout->get_controls(),
                    'default'     => [
                        [ 'layout_item' => 'image' ],
                        [ 'layout_item' => 'title' ],
                        [ 'layout_item' => 'description' ],
                    ],
                    'title_field' => 'Layout item  - {{{layout_item}}}'
                ]
            );

		$this->end_controls_section();

    }

	/**
	 * Content section style
	 */
	public function content_section_content() {

		$this->start_controls_section( 'content_content', [
			'label' => esc_html__( 'Content', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_CONTENT,
			'condition' => [ 'general_source' => ['custom', 'gallery'] ]
		] );

		$grider = new Repeater();

		/** Title. */
		$grider->add_control(
			'grider_content_title',
			[
				'label' => esc_html__( 'Title', 'grider-elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic'               => ['active' => true],
				'placeholder' => esc_html__( 'Title', 'grider-elementor' ),
				'label_block' => true
			]
		);

		/** Image */
		$grider->add_control(
			'grider_content_image',
			[
				'label' => esc_html__( 'Choose Image', 'grider-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		/** Description. */
		$grider->add_control(
			'grider_content_description',
			[
				'label' => esc_html__( 'Description', 'grider-elementor' ),
				'type' => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Description', 'grider-elementor' ),
			]
		);

		/** Button Text. */
		$grider->add_control(
			'grider_content_button_text',
			[
				'label' => esc_html__( 'Button Text', 'grider-elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic'               => ['active' => true],
				'placeholder' => esc_html__( 'Button Text', 'grider-elementor' ),
				'label_block' => true
			]
		);

		/** Link. */
		$grider->add_control(
			'grider_content_link',
			[
				'label' => esc_html__( 'Link', 'grider-elementor' ),
				'type' => Controls_Manager::URL,
				'dynamic' => ['active' => true],
				'placeholder' => esc_html__( 'https://1.envato.market/cc-merkulove', 'grider-elementor' ),
				'show_external' => true,
				'default' => [
					'url' => '',
					'is_external' => false,
					'nofollow' => false,
				],
			]
		);

		$this->add_control(
			'grider_items',
			[
				'label'       => esc_html__( 'Grid Items', 'grider-elementor' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $grider->get_controls(),
				'default'     => [
					[
						'grider_content_title' => esc_html__( 'Title 1', 'grider-elementor' ),
						'grider_content_description' => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.', 'grider-elementor' ),
						'grider_content_button_text' => esc_html__( 'More', 'grider-elementor' ),
					],
					[
						'grider_content_title' => esc_html__( 'Title 2', 'grider-elementor' ),
						'grider_content_description' => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.', 'grider-elementor' ),
						'grider_content_button_text' => esc_html__( 'More', 'grider-elementor' ),
					],
					[
						'grider_content_title' => esc_html__( 'Title 3', 'grider-elementor' ),
						'grider_content_description' => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.', 'grider-elementor' ),
						'grider_content_button_text' => esc_html__( 'More', 'grider-elementor' ),
					]
				],
				'title_field' => 'Item - {{{ grider_content_title }}}',
				'condition' => [ 'general_source' => 'custom' ]
			]
		);

		/** Gallery */
		$this->add_control(
			'grider_gallery',
			[
				'label' => esc_html__( 'Add Images', 'grider-elementor' ),
				'type' => Controls_Manager::GALLERY,
				'default' => [],
				'condition' => [ 'general_source' => 'gallery' ]
			]
		);

		$this->end_controls_section();

    }

	/**
	 *  TAB: Style.
	 */

    /**
     * Add widget controls: Style -> Section Style Example.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function grid_item_section_styles() {

        $this->start_controls_section( 'section_grid_item_style', [
            'label' => esc_html__( 'Grid item', 'grider-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE,
        ] );

			/** Custom width */
		    $this->add_control(
			    'grid_item_style_custom_width',
			    [
				    'label' => esc_html__( 'Static width', 'grider-elementor' ),
				    'type' => Controls_Manager::SWITCHER,
				    'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
				    'label_off' => esc_html__( 'No', 'grider-elementor' ),
				    'return_value' => 'yes',
				    'default' => '',
                    'condition' => [ 'navigation_masonry!' => 'yes' ]
			    ]
		    );

	    $this->add_control(
		    'grid_wrap',
		    [
			    'label' => esc_html__( 'Wrap grid items', 'grider-elementor' ),
			    'type' => Controls_Manager::SWITCHER,
			    'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
			    'label_off' => esc_html__( 'No', 'grider-elementor' ),
			    'return_value' => 'yes',
			    'default' => '',
			    'condition' => [ 'grid_item_style_custom_width' => 'yes', 'navigation_masonry!' => 'yes' ]
		    ]
	    );

            /** Width. */
            $this->add_responsive_control(
                'grid_item_style_width',
                [
                    'label'           => esc_html__( 'Width', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px', '%', 'vw' ],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'max'  => 500,
                            'step' => 1,
                        ],
                    ],
                    'devices' => [ 'desktop', 'tablet', 'mobile' ],
                    'default' => [
						'size' => '33.3',
						'unit' => '%'
                    ],
                    'selectors'  => [
	                    '{{WRAPPER}} .mdp-grider-container'  => 'grid-template-columns: repeat( {{navigation_max_columns.size}}, {{size}}{{unit}} )',
	                    '{{WRAPPER}} .mdp-grider-item'  => 'min-width: {{size}}{{unit}}'
                    ],
                    'condition' => [ 'grid_item_style_custom_width' => 'yes', 'navigation_masonry!' => 'yes' ]
                ]
            );

	    /** Justify grid */
	    $this->add_responsive_control(
		    'justify_content',
		    [
			    'label'     => esc_html__( 'Justify items', 'grider-elementor' ),
			    'type'      => Controls_Manager::SELECT,
			    'options'   => [
				    'stretch'           =>  esc_html__('Stretch', 'grider-elementor' ),
				    'left'              =>  esc_html__('Left', 'grider-elementor' ),
				    'center'            =>  esc_html__('Center', 'grider-elementor' ),
				    'right'             =>  esc_html__('Right', 'grider-elementor' ),
				    'space-around'      =>  esc_html__('Space Around', 'grider-elementor' ),
				    'space-between'     =>  esc_html__('Space Between', 'grider-elementor' ),
				    'space-evenly'      =>  esc_html__('Space Evenly', 'grider-elementor' ),
			    ],
			    'default'   => 'stretch',
			    'selectors' => [
				    '{{WRAPPER}} .mdp-grider-container' => 'justify-content: {{value}}',
			    ],
			    'toggle'    => true,
			    'condition' => [ 'grid_item_style_custom_width' => 'yes', 'navigation_masonry!' => 'yes' ],
		    ]
	    );

	    $this->add_control(
		    'grid_ignore_columns',
		    [
			    'label' => esc_html__( 'Ignore max columns number', 'grider-elementor' ),
			    'type' => Controls_Manager::SWITCHER,
			    'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
			    'label_off' => esc_html__( 'No', 'grider-elementor' ),
			    'return_value' => 'yes',
			    'default' => '',
			    'condition' => [ 'grid_item_style_custom_width' => 'yes', 'navigation_masonry!' => 'yes' ],
			    'separator' => 'after'
		    ]
	    );

	    $this->add_control(
		    'grid_item_style_custom_height',
		    [
			    'label' => esc_html__( 'Static height', 'grider-elementor' ),
			    'type' => Controls_Manager::SWITCHER,
			    'label_on' => esc_html__( 'Yes', 'grider-elementor' ),
			    'label_off' => esc_html__( 'No', 'grider-elementor' ),
			    'return_value' => 'yes',
			    'default' => '',
			    'condition' => [ 'navigation_masonry!' => 'yes' ]
		    ]
	    );

	    $this->add_responsive_control(
		    'grid_item_style_height',
		    [
			    'label'           => esc_html__( 'Min-Height', 'grider-elementor' ),
			    'type'            => Controls_Manager::SLIDER,
			    'size_units'      => [ 'px', '%', 'vh' ],
			    'range'           => [
				    'px' => [
					    'min'  => 1,
					    'max'  => 500,
					    'step' => 1,
				    ],
			    ],
			    'devices' => [ 'desktop', 'tablet', 'mobile' ],
			    'default' => [
				    'size' => '250',
				    'unit' => 'px'
			    ],
			    'selectors'  => [
				    '{{WRAPPER}} .mdp-grider-item'  => 'min-height: {{size}}{{unit}}'
			    ],
			    'condition' => [ 'grid_item_style_custom_height' => 'yes', 'navigation_masonry!' => 'yes' ],
		    ]
	    );

	    $this->add_responsive_control(
		    'align_content',
		    [
			    'label'     => esc_html__( 'Align content', 'grider-elementor' ),
			    'type'      => Controls_Manager::SELECT,
			    'options'   => [
				    'start'             =>  esc_html__('Top', 'grider-elementor' ),
				    'center'            =>  esc_html__('Center', 'grider-elementor' ),
				    'end'               =>  esc_html__('Bottom', 'grider-elementor' ),
				    'space-around'      =>  esc_html__('Space Around', 'grider-elementor' ),
				    'space-between'     =>  esc_html__('Space Between', 'grider-elementor' ),
				    'space-evenly'      =>  esc_html__('Space Evenly', 'grider-elementor' ),
			    ],
			    'default'   => 'start',
			    'selectors' => [
				    '{{WRAPPER}} .mdp-grider-item' => 'align-content: {{value}}',
			    ],
			    'toggle'    => true,
			    'condition' => [ 'grid_item_style_custom_height' => 'yes', 'navigation_masonry!' => 'yes' ],
		    ]
	    );

            $this->get_settings_style_group(
                'grid_item_style',
                '.mdp-grider-item',
                [ 'general_source!' => 'gallery' ],
                [
                    'padding' => true,
                ]
            );

            $this->get_settings_style_group(
                'grid_item_style',
                '.mdp-grider-item .mdp-grider-title, '.
                '{{WRAPPER}} .mdp-grider-item .mdp-grider-description, '.
                '{{WRAPPER}} .mdp-grider-item .mdp-grider-price, '.
                '{{WRAPPER}} .mdp-grider-item .mdp-grider-box-button,'.
                '{{WRAPPER}} .mdp-grider-item .mdp-grider-image-box,'.
                '{{WRAPPER}} .mdp-grider-item .mdp-grider-title-link',
                [ 'general_source!' => 'gallery' ],
                [
                    'color' => true,
                    'typography' => true,
                    'text-shadow' => true,
                    'alignment' => true,
                ]
            );

            $this->get_settings_style_group(
                'grid_item_style',
                '.mdp-grider-item',
                [ 'general_source!' => 'gallery' ],
                [
                    'border' => true,
                    'border-radius' => true,
                ]
            );

            $this->get_settings_style_group(
                'grid_item_style_gallery',
                '.mdp-grider-item',
                [ 'general_source' => 'gallery' ],
                [
                    'padding' => true,
                    'border' => true,
                    'border-radius' => true,
                ]
            );

            $this->add_control(
                'grid_item_header',
                [
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
            );

	    $this->start_controls_tabs( 'tabs_grider_button' );

            /** Content tab. */
            $this->start_controls_tab(
                'tab_normal',
                [
                    'label' => esc_html__('NORMAL', 'grider-elementor')
                ]
            );

                $this->get_settings_style_group(
                    'grid_item_style_normal',
                    '.mdp-grider-item',
                    [],
                    [
                        'background' => true,
                        'box-shadow' => true,
                    ],
                    [ 'none', 'none', 'none']
                );

            $this->end_controls_tab();

            /** Content tab. */
            $this->start_controls_tab(
                'tab_hover',
                [
                    'label' => esc_html__('HOVER', 'grider-elementor')
                ]
            );

                $this->get_settings_style_group(
                    'grid_item_style_hover',
                    '.mdp-grider-item:hover',
                    [],
                    [
                        'background' => true,
                        'box-shadow' => true,
                    ],
	                [ 'none', 'none', 'none']
                );

            $this->end_controls_tab();

	    $this->end_controls_tabs();

        $this->end_controls_section();

    }

	/**
	 * Title section styles
	 */
	public function title_section_styles() {

		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_STYLE,
			'condition' => [ 'general_source!' => 'gallery' ]
		] );

            /** Width. */
            $this->add_responsive_control(
                'title_style_width',
                [
                    'label'           => esc_html__( 'Width', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px', '%'],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'step' => 1,
                        ],
                    ],
                    'default' => [
                        'unit' => '%',
                        'size' => 100,
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .mdp-grider-item>.mdp-grider-title' => 'width: {{size}}{{unit}}',
                        '{{WRAPPER}} .mdp-grider-item .mdp-grider-title-link' => 'width: {{size}}{{unit}}',
                    ],
                ]
            );

            $this->get_settings_style_group(
                'title_style',
                '.mdp-grider-item .mdp-grider-title',
                [],
                [
                    'margin' => true,
                    'padding' => true,
                    'color' => true,
                    'typography' => true,
                    'text-shadow' => true,
                    'alignment' => true,
                    'background' => true,
                    'border' => true,
                    'border-radius' => true,
                    'box-shadow' => true,
                ]
            );

		$this->end_controls_section();

    }

	/**
	 * Description section styles
	 */
	public function description_section_styles() {

		$this->start_controls_section( 'section_description_style', [
			'label' => esc_html__( 'Description', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_STYLE,
			'condition' => [ 'general_source!' => 'gallery' ]
		] );

            /** Width. */
            $this->add_responsive_control(
                'description_style_width',
                [
                    'label'           => esc_html__( 'Width', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px', '%'],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'step' => 1,
                        ],
                    ],
                    'default' => [
                        'unit' => '%',
                        'size' => 100,
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .mdp-grider-item .mdp-grider-description' => 'width: {{size}}{{unit}}',
                    ],
                ]
            );

            $this->get_settings_style_group(
                'description_style',
                '.mdp-grider-item .mdp-grider-description',
                [],
                [
                    'margin' => true,
                    'padding' => true,
                    'color' => true,
                    'typography' => true,
                    'text-shadow' => true,
                    'alignment-justify' => true,
                    'background' => true,
                    'border' => true,
                    'border-radius' => true,
                    'box-shadow' => true,
                ]
            );

		$this->end_controls_section();

    }

	/**
	 * Image section style
	 */
	public function image_section_styles() {

		$this->start_controls_section( 'section_image_style', [
			'label' => esc_html__( 'Image', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_STYLE
		] );

            /** Width. */
            $this->add_responsive_control(
                'image_style_width',
                [
                    'label'           => esc_html__( 'Width', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px', '%'],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'step' => 1,
                        ],
                    ],
                    'default' => [
                        'unit' => '%',
                        'size' => 100,
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .mdp-grider-image-box' => 'max-width: {{size}}{{unit}}',
                    ],
                    'condition' => [ 'general_source!' => 'gallery' ]
                ]
            );

            $this->get_settings_style_group(
                'image_style',
                '.mdp-grider-image',
	            [],
                [
                    'margin' => true,
                    'padding' => true,
                ]
            );

            $this->get_settings_style_group(
                'image_style',
                '.mdp-grider-item .mdp-grider-image-box',
	            [ 'general_source!' => 'gallery' ],
                [
	                'alignment' => true,
                ]
            );

            $this->get_settings_style_group(
                'image_style',
                '.mdp-grider-image',
                [],
                [
	                'background' => true,
	                'border' => true,
	                'border-radius' => true,
	                'box-shadow' => true,
	                'css-filter' => true
                ]
            );

		$this->end_controls_section();

    }

	/**
	 * Price section style
	 */
	public function price_section_styles() {

		$this->start_controls_section( 'section_price_style', [
			'label' => esc_html__( 'Price', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_STYLE,
			'condition' => [ 'general_source' => 'woocommerce' ]
		] );

            /** Width. */
            $this->add_responsive_control(
                'price_style_width',
                [
                    'label'           => esc_html__( 'Width', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px', '%'],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'step' => 1,
                        ],
                    ],
                    'default' => [
                        'unit' => '%',
                        'size' => 100,
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .mdp-grider-item .mdp-grider-price' => 'width: {{size}}{{unit}}',
                    ],
                ]
            );

            $this->get_settings_style_group(
                'price_discount_style',
                '.mdp-grider-item .mdp-grider-price ins, '.
                '{{WRAPPER}} .mdp-grider-item .mdp-grider-price ins .woocommerce-Price-amount.amount, '.
                '{{WRAPPER}} .woocommerce-Price-amount.amount',
                [],
                [
                    'margin' => true,
                    'padding' => true,
                    'color' => true,
                    'typography' => true,
                    'text-shadow' => true,
                ]
            );

            $this->get_settings_style_group(
                'price_discount_style',
                '.mdp-grider-item .mdp-grider-price',
                [],
                [
	                'alignment' => true
                ]
            );

            $this->get_settings_style_group(
                'price_discount_style',
                '.mdp-grider-price ins, {{WRAPPER}} .mdp-grider-price ins .woocommerce-Price-amount.amount, {{WRAPPER}} .woocommerce-Price-amount.amount',
                [],
                [
                    'background' => true,
                    'border' => true,
                    'border-radius' => true,
                    'box-shadow' => true,
                ]
            );

            /** Discount header. */
            $this->add_control(
                'price_discount_style_header',
                [
                    'label' => esc_html__( 'Discount', 'grider-elementor' ),
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
            );

            $this->get_settings_style_group(
                'price_style',
                '.mdp-grider-price del .woocommerce-Price-amount.amount, {{WRAPPER}} .mdp-grider-price del',
                [],
                [
                    'margin' => true,
                    'padding' => true,
                    'color' => true,
                    'typography' => true,
                    'text-shadow' => true,
                    'background' => true,
                    'border' => true,
                    'border-radius' => true,
                    'box-shadow' => true,
                ]
            );

		$this->end_controls_section();

    }

	/**
	 * Button section style
	 */
	public function button_section_styles() {

		$this->start_controls_section( 'section_button_style', [
			'label' => esc_html__( 'Button', 'grider-elementor' ),
			'tab'   => Controls_Manager::TAB_STYLE,
			'condition' => [ 'general_source!' => 'gallery' ]
		] );

            /** Width. */
            $this->add_responsive_control(
                'button_style_width',
                [
                    'label'           => esc_html__( 'Width', 'grider-elementor' ),
                    'type'            => Controls_Manager::SLIDER,
                    'size_units'      => [ 'px', '%'],
                    'range'           => [
                        'px' => [
                            'min'  => 1,
                            'step' => 1,
                        ],
                    ],
                    'default' => [
                        'unit' => '%',
                        'size' => 100,
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .mdp-grider-box-button' => 'width: {{size}}{{unit}};',
                    ],
                ]
            );


		$this->get_settings_style_group(
			'button_style',
			'.mdp-grider-box-button',
			[],
			[
				'margin' => true,
			]
		);

            $this->get_settings_style_group(
                'button_style',
                '.mpd-grider-button',
                [],
                [
	                'padding' => true,
                    'typography' => true,
                    'text-shadow' => true,
                ]
            );

            $this->get_settings_style_group(
                'button_style',
                '.mdp-grider-item .mdp-grider-box-button',
                [],
                [
	                'alignment' => true,
                ]
            );

            $this->get_settings_style_group(
                'button_style',
                '.mpd-grider-button',
                [],
                [
                    'border-radius' => true,
                ]
            );

            $this->add_control(
                'button_style_header',
                [
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
            );

		$this->start_controls_tabs( 'tabs_grider_button_panel' );

            /** Content tab. */
            $this->start_controls_tab(
                'tab_normal_button',
                [
                    'label' => esc_html__('NORMAL', 'grider-elementor')
                ]
            );

                $this->get_settings_style_group(
                    'button_style_normal',
                    '.mpd-grider-button',
                    [],
                    [
                        'color' => true,
                        'background' => true,
                        'border' => true,
                        'box-shadow' => true,
                    ]
                );

            $this->end_controls_tab();

            /** Content tab. */
            $this->start_controls_tab(
                'tab_hover_button',
                [
                    'label' => esc_html__('HOVER', 'grider-elementor')
                ]
            );

                $this->get_settings_style_group(
                    'button_style_hover',
                    '.mpd-grider-button:hover',
                    [],
                    [
                        'color' => true,
                        'background' => true,
                        'border' => true,
                        'box-shadow' => true,
                    ]
                );

            $this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

    }

	/**
	 * We return an array with the names of the layouts.
	 *
	 * @param $settings - we get an array with all the data.
	 *
	 * @return array
	 */
	private function get_LayoutSortable( $settings ) {

		/** We get all the data on the lists of layouts. */
		$layout = $settings['layout_priority'];

		/** An array in which the names of the layouts and their width will be stored. */
		$res = array();

		if( !is_null($layout) ){
			/** We write the names of the layouts and their width into an array. */
			foreach ($layout as $key => $val){ $res[] = [ 'name' => $val['layout_item'] ]; }
        }

		return $res;

	}

	/**
	 * Displays image for custom slide.
	 */
	public function get_custom_image( $settings ) {
        echo sprintf(
    '<div class="mdp-grider-image-box"><img src="%s" class="mdp-grider-image" alt=""></div>',
            esc_url($settings['grider_content_image']['url'])
        );
	}

	/**
	 * Displays title for custom slide.
	 *
	 * @param $settings - slide settings.
	 * @param $options - global settings.
	 */
	public function get_custom_title( $settings,  $options ) {

		$header_html = '';
	    if ( !empty( $settings['grider_content_title'] ) ) {

		    $header_html = sprintf(
			    '<h3 class="mdp-grider-title">%s</h3>',
			    esc_html( $settings['grider_content_title'] )
		    );

		    if( 'title' ===  $options['general_link_type'] && !empty($settings['grider_content_link']['url']) ){

			    $url = $settings['grider_content_link']['url'];
			    $target = $settings['grider_content_link']['is_external'] ? 'target="_blank"' : '';
			    $nofollow = $settings['grider_content_link']['nofollow'] ? 'rel="nofollow"' : '';

                $header_html = wp_sprintf(
	                '<a href="%1$s" class="mdp-grider-title-link" title="%4$s" %2$s %3$s>'.
	                '<h3 class="mdp-grider-title">%4$s</h3></a>',
                    esc_url( $url ),
                    $target,
                    $nofollow,
	                esc_html( $settings['grider_content_title'] )
                );

		    }

        }

		echo wp_kses_post( $header_html );
	}

	/**
	 *  Displays description for custom slide.
	 *
	 * @param $settings - slide settings.
	 */
	public function get_custom_description( $settings ) {
	    if( !empty($settings['grider_content_description']) ){
		    echo sprintf(
			    '<div class="mdp-grider-description">%s</div>',
			    wp_kses_post( $settings['grider_content_description'] )
		    );
        }
	}

	/**
	 * Displays button for custom slide.
	 *
	 * @param $settings - slide settings.
	 * @param $options - global settings.
	 */
	public function get_custom_button( $settings, $options ) {

		if( empty( $settings['grider_content_link']['url'] ) ){ return; }

        $button_html = sprintf(
            '<p class="mdp-grider-box-button"><span class="mpd-grider-button">%s</span></p>',
            esc_html( $settings['grider_content_button_text'] )
        );

	    if( 'button' === $options['general_link_type'] ) {

            $url = $settings['grider_content_link']['url'];
		    $target = $settings['grider_content_link']['is_external'] ? 'target="_blank"' : '';
		    $nofollow = $settings['grider_content_link']['nofollow'] ? 'rel="nofollow"' : '';

            $button_html = wp_sprintf(
        '<p class="mdp-grider-box-button"><a href="%1$s" class="mpd-grider-button" title="%1s4" %2$s %3$s><span>%4$s</span></a></p>',
	            esc_url( $url ),
	            $target,
	            $nofollow,
	            esc_html( $settings['grider_content_button_text'] )
            );

        }

	    echo wp_kses_post( $button_html );

	}

	/**
	 * Display custom slider.
	 *
	 * @param $settings       - Settings from repeater
	 * @param $LayoutSortable - An array of slides layouts.
	 * @param $options        - Global settings.
	 * @param $count          - Cycle counter.
	 */
	public function get_custom_grider_box( $settings, $LayoutSortable,  $options, $count ) {

	    echo sprintf(
	            '<div class="mdp-grider-item item%s">',
                esc_attr($count)
        );

        $this->render_box_link( $settings );

            /** We list the layouts in the given order. */
            foreach ( $LayoutSortable as $val ) {

                switch ( $val[ 'name' ] ) {
                    case 'image':
                        $this->get_custom_image( $settings );
                        break;

                    case 'title':
                        $this->get_custom_title( $settings, $options );
                        break;

                    case 'description':
                        $this->get_custom_description( $settings );
                        break;

                    case 'button':
                       $this->get_custom_button( $settings, $options );
                        break;

                    default:
                        echo '';
                        break;
                }

            }

        echo '</div>';

    }

	/**
	 * Custom: We transfer data to create a slider.
	 *
	 * @param $settings - Global settings.
	 * @param $LayoutSortable - An array of slides layouts.
	 */
	public function get_custom_info( $settings, $LayoutSortable ) {

		/** Count the number. */
		$quantity_custom = count( $settings['grider_items'] ) - 1;

		/** We display imagebox item in a cycle. */
		for ( $i = 0; $i <= $quantity_custom; $i++) {
			$this->get_custom_grider_box( $settings['grider_items'][$i], $LayoutSortable, $settings, $i+1 );
		}

	}

	/**
	 * Display post slider.
	 *
	 * @param $settings       - Post data.
	 * @param $LayoutSortable - An array of slides layouts.
	 * @param $options        - Global settings.
	 * @param $count          - Cycle counter.
	 */
	public function get_posts_grider_box( $settings, $LayoutSortable, $options, $count ) {

	    $typeAnimation = 'none';
		if( $options['navigation_animate'] === 'yes' ){ $typeAnimation = $options['navigation_transition']; }

		echo sprintf(
            '<div class="mdp-grider-item item%s">',
			esc_attr( $count )
        );

		$this->render_box_link( $settings );

            /** We list the layouts in the given order. */
            foreach ($LayoutSortable as $val) {

                switch ( $val['name'] ) {
                    case 'image':
                        $this->get_posts_image( $settings, $options );
                        break;

                    case 'title':
                        $this->get_posts_title( $settings, $options );
                        break;

                    case 'description':
                        $this->get_posts_description( $settings );
                        break;

                    case 'price':
                        if( $options['general_source'] === 'woocommerce' ){
                            $this->get_posts_price( $settings, $options );
                        }
                        break;

                    case 'button':
                        $this->get_posts_button( $settings, $options );
                        break;

                    default:
                        echo '';
                        break;
                }

            }

        echo '</div>';

    }

	/**
	 * Displays price of the product.
	 *
	 * @param $settings - Post data.
	 * @param $options - Settings for displaying product prices.
	 */
	public function get_posts_price( $settings, $options ){
		$product = wc_get_product($settings->ID );

		if( 'woocommerce' === $options['general_source'] ){
			echo sprintf( '<p class="mdp-grider-price">%s</p>', wp_kses_post( $product->get_price_html() ) );
        }
	}

	/**
	 * Displays button for post slide.
	 *
	 * @param $data - post data.
	 * @param $options - global settings.
	 */
	public function get_posts_button( $data, $options ) {
		if( 'button' === $options['general_link_type'] ){
		    echo sprintf(
                '<p class="mdp-grider-box-button"><a href="%s" class="mpd-grider-button">'.
                '<span>%s</span></a></p>',
			    esc_url( $data->guid ),
			    esc_html( $options['general_button_text'] )
            );
        }
	}

	/**
	 * Displays image for post slide.
	 *
	 * @param $data - Post data.
	 */
	public function get_posts_image( $data ) {

		$image_url = get_the_post_thumbnail_url( $data->ID, 'full');

		if( empty($image_url) ){ $image_url =  Utils::get_placeholder_image_src(); }

		echo sprintf(
            '<div class="mdp-grider-image-box"><img src="%s" class="mdp-grider-image" alt=""></div>',
			esc_url($image_url)
        );
	}

	/**
	 * Displays title for post slide.
	 *
	 * @param $data - post data.
	 * @param $options - global settings.
	 */
	public function get_posts_title( $data, $options ){

		/** Makes the title a link. */
        $header_html = sprintf(
        '<h3 class="mdp-grider-title">%s</h3>',
	            esc_html( $data->post_title )
        );

		if( 'title' === $options['general_link_type'] ){
			$header_html = sprintf(
             '<a href="%s" class="mdp-grider-title-link">'.
                    '<h3 class="mdp-grider-title">%s</h3></a>',
				    esc_url( $data->guid ),
				    esc_html( $data->post_title )
            );
        }

		echo wp_kses_post( $header_html );
	}

	/**
	 * Displays description for post slide.
	 *
	 * @param $data - post data.
	 */
	public function get_posts_description( $data ) {
	    if( !empty($data->post_excerpt) ){
		    echo sprintf(
			    '<div class="mdp-grider-description"><p>%s</p></div>',
			    esc_html( $data->post_excerpt )
		    );
        }
	}

	/**
	 * Posts: We transfer data to create a slider.
	 *
	 * @param $settings - Global settings.
	 * @param $LayoutSortable - An array of slides layouts.
	 */
	public function get_posts_info( $settings, $LayoutSortable ) {

		/** We get an array with post data. */
		$posts = get_posts( [ 'category' => $settings['general_posts_category'] ] );
        $amount = $settings[ 'navigation_dynamic_content' ][ 'size' ];

		/** We count the number of imagebox item in the category. */
		$quantity_posts = count( $posts ) - 1;

		if( $quantity_posts >= 0 ) {

			/** We display imagebox item in a cycle. */
			for ( $i = 0; $i <= $quantity_posts; $i ++ ) {

				$this->get_posts_grider_box( $posts[ $i ], $LayoutSortable, $settings, $i );

				$amount = $amount - 1;
                if( $amount < 1 ) { break; }

			}

		}else{
			echo esc_html__('There are no posts in this category.', 'grider-elementor');
		}

	}

	/**
	 * Woocommerce: We transfer data to create a slider.
	 *
	 * @param $settings - Global settings.
	 * @param $LayoutSortable - An array of slides layouts.
	 */
	public function get_woocommerce_info( $settings, $LayoutSortable ) {

		$ck_key = Settings::get_instance()->options['ck_key'];
		$cs_key = Settings::get_instance()->options['cs_key'];

		$getCat = get_term_by( 'id', $settings['general_products_category'], 'product_cat' );

		/** We get an array with post data. */
		$product = get_posts(
			[ 'product_cat' => $getCat->slug,
              'post_type'   => 'product',
              'numberposts' => -1
            ]
        );

		/** We count the number of imagebox item in the category. */
		$quantity_product = count( $product ) - 1;

		if( !empty($ck_key) && !empty($cs_key) ){

		    $woocommerceURL = get_site_url( null, '', 'https' ) .
		                      '/wp-json/wc/v3/products?category=' .
		                      $settings['general_products_category'] .
		                      '&_embed&consumer_key=' . esc_attr($ck_key) . '&consumer_secret=' . esc_attr($cs_key);

		    $productFile = @file_get_contents( $woocommerceURL );

			$productJSON = json_decode( $productFile, true );
			$amount = $settings[ 'navigation_dynamic_content' ][ 'size' ];

			if( !empty($productJSON) ){
				if( $quantity_product >= 0 ){
					/** We display imagebox item in a cycle. */
					for ( $i = 0; $i <= $quantity_product; $i++) {

						$this->get_posts_grider_box( $product[$i], $LayoutSortable, $settings, $i+1 );

						$amount = $amount - 1;
						if( $amount < 1 ) { break; }

					}
				}else{
					echo esc_html__('There are no products in this category.', 'grider-elementor');
				}
			}else{
				echo esc_html__('Woocommerce source  does not support protocol http.', 'grider-elementor');
            }

        }else{
			echo esc_html__('Specify the secret keys woocommerce in the plugin settings.', 'grider-elementor');
        }

	}

	/**
     *  Gallery image output.
     *
	 * @param $settings
	 */
	public function get_gallery_info( $settings ) {
	    $count = 1;

		$typeAnimation = 'none';
		if( $settings['navigation_animate'] === 'yes' ){ $typeAnimation = $settings['navigation_transition']; }

		foreach ( $settings['grider_gallery'] as $image ) {
			echo sprintf(
			    '<div class="mdp-grider-item item%s">'.
                '<img src="%s" class="mdp-grider-image" alt=""></div>',
				esc_attr($count),
				esc_url($image['url'])
            );
			$count++;
		}
	}

	/**
     * Render box-type link for the single grid item
     *
	 * @param $settings
	 *
	 * @return void
	 */
    private function render_box_link( $settings ) {

        // Only for box type of links
        if ( 'box' !== $this->get_settings_for_display( 'general_link_type' ) ) { return; }

        if ( is_object( $settings ) ) {

            // WordPress Posts
	        $url = get_permalink( $settings->ID );
	        $target = '';
	        $nofollow = '';

        } else {

            // Custom Grid Content
	        $url = ! empty( $settings['grider_content_link']['url'] ) ? $settings['grider_content_link']['url'] : '';
	        $target = $settings['grider_content_link']['is_external'] ? 'target="_blank"' : '';
	        $nofollow = $settings['grider_content_link']['nofollow'] ? 'rel="nofollow"' : '';

        }

        // Add link box
	    if ( ! empty ( $url ) ) {

		    echo wp_sprintf(
			    '<a href="%1$s" class="mdp-grider-box-link" %2$s %3$s></a>',
			    $url,
			    $target,
			    $nofollow
		    );

	    }

    }

    /**
     * Render Frontend Output. Generate the final HTML on the frontend.
     *
     * @access protected
     *
     * @return void
     **/
    protected function render() {

	    /** Get all the values from the admin panel. */
	    $settings = $this->get_settings_for_display();

	    /** We return an array with the names of the layouts and their width. */
	    $LayoutSortable  = $this->get_LayoutSortable( $settings );

	    /**
	     * Prepare variables before using.
	     */
	    $max_columns        = !empty( $settings['navigation_max_columns']['size'] ) ?
		                                $settings['navigation_max_columns']['size'] : 3;
	    $max_columns_tablet = !empty( $settings['navigation_max_columns_tablet']['size'] ) ?
		                                $settings['navigation_max_columns_tablet']['size'] : 2;
	    $max_columns_mobile = !empty( $settings['navigation_max_columns_mobile']['size'] ) ?
		                                $settings['navigation_max_columns_mobile']['size'] : 1;

        echo sprintf(
            '<div id="mdp-grider-container-%s" class="mdp-grider-container mdp-grider-%s mdp-grider-container-%s%s%s%s" data-cols-desktop="%s" data-cols-tablet="%s" data-cols-mobile="%s">',
	        esc_attr( $this->get_id() ),
	        esc_attr( $settings[ 'navigation_masonry' ] === 'yes' ? 'masonry' : 'grid' ),
	        esc_attr( $this->get_id() ),
	        esc_attr( $settings[ 'navigation_animate' ] === 'yes' && $settings[ 'navigation_animate_delay' ] === 'yes' ? ' mdp-grider-delay' : '' ),
	        esc_attr( $settings[ 'grid_wrap' ] === 'yes' ? ' mdp-grider-wrap' : '' ),
	        esc_attr( $settings[ 'grid_ignore_columns' ] === 'yes' ? ' mdp-grider-ignore-cols' : '' ),
            esc_attr( $max_columns ),
            esc_attr( $max_columns_tablet ),
            esc_attr( $max_columns_mobile )
        );

            /** We display content depending on the selected source. */
            switch ($settings['general_source']){

	            case 'custom':
		            $this->get_custom_info( $settings, $LayoutSortable );
		            break;

	            case 'posts':
		            $this->get_posts_info( $settings, $LayoutSortable );
		            break;

	            case 'woocommerce':
		            $this->get_woocommerce_info( $settings, $LayoutSortable);
		            break;

	            case 'gallery':
		            $this->get_gallery_info( $settings );
		            break;

	            default:
		            esc_html_e( 'The source of information was not found.', 'grider-elementor' );
            }

        echo '</div>';

    }

	/**
	 * JS render
	 */
	protected function content_template() {

		$ck_key = Settings::get_instance()->options['ck_key'];
		$cs_key = Settings::get_instance()->options['cs_key'];
		$image_url =  Utils::get_placeholder_image_src();
		$plugin_dir = plugin_dir_url(__FILE__);
		$localhost_url = explode("/wp-content/", $plugin_dir);
		?>
        <#
        /** We return an array with the names of the layouts. */
        function layoutSortable() {

        let res = [];

        if( settings.layout_priority ){
        _.each( settings.layout_priority, function( item, index ) {
        res.push( item.layout_item );
        });
        }

        return res;

        }

        const protocol = window.location.protocol;
        const hostName = window.location.hostname;

        /* We return the data of posts and goods. */
        function get_result( urlJSON, callback ) {

        var ret_val;

        jQuery.ajax({
        dataType: "json",
        url: urlJSON,
        async: false,
        success:function( response ){
        ret_val = response;
        }
        });

        return ret_val;
        };

        let addon_id = view.model.id;

        if( settings.general_source === 'custom'){

        /** Displays title. */
        function customTitle( itemContent ){
            if( itemContent.grider_content_title !== '' ){
                #>
                <h3 class="mdp-grider-title">
                    {{{itemContent.grider_content_title}}}
                </h3>
                <#
            }
        }

        /** Displays description. */
        function customDescription( itemContent ){
        #>
        <div class="mdp-grider-description">
            {{{itemContent.grider_content_description}}}
        </div>
        <#
        }

        /** Displays image. */
        function customImage( itemContent ){
        #>
        <div class="mdp-grider-image-box">
            <img src="{{{itemContent}}}" class="mdp-grider-image" alt=""/>
        </div>
        <#
        }

        /** Displays button. */
        function customButton( itemContent ){
        if( itemContent.grider_content_link.url !== '' ){
        #>
        <p class="mdp-grider-box-button">
            <a href="{{{itemContent.grider_content_link.url}}}" class="mpd-grider-button">
                <span>{{{itemContent.grider_content_button_text}}}</span>
            </a>
        </p>
        <#
        }
        }

        /** Display custom content. */
        function customContainer( itemContent, layoutSortable, count ){
        #>
        <div class="mdp-grider-item item-<# {{count}} #>">
            <#
            _.each( layoutSortable, function( item, index ) {
            switch ( item ) {
            case 'image':
            customImage( itemContent.grider_content_image.url );
            break;

            case 'title':
            customTitle( itemContent );
            break;

            case 'description':
            customDescription( itemContent );
            break;

            case 'button':
            customButton( itemContent );
            break;

            default:
            break;
            }
            });
            #>
        </div>
        <#
        }

        function customInfo( layoutSortable ){

        /** Count the number. */
        let quantityCustom = settings.grider_items.length - 1;

        for(let i = 0; i <= quantityCustom; i++){
        customContainer( settings.grider_items[i], layoutSortable, i );
        }

        }

        } // if custom end

        /* Start posts source. */
        let postsContent = '';
        let local = ( hostName === 'localhost' ) ? '<?php echo esc_attr($localhost_url[0]); ?>' : protocol + '//' + hostName;

        if( settings.general_source === 'posts'){

            let urlJSONPosts = local + '/wp-json/wp/v2/posts?categories=' + settings.general_posts_category + '&_embed';

            JSONPosts = get_result(urlJSONPosts);

            let itemsLength = JSONPosts !== undefined ? JSONPosts.length : 0;

            if( itemsLength !== 0 ){
            JSONPosts.forEach( function( content ){

            if( content !== undefined ){

                postsContent += '<div class="mdp-grider-item item' + content.id + '">';

                _.each( layoutSortable(), function( item, index ) {
                switch ( item ) {
                case 'image':
                let imgData = content._embedded["wp:featuredmedia"] !== undefined ?
                        content._embedded["wp:featuredmedia"][0].source_url : '<?php echo esc_url($image_url); ?>';
                    postsContent += '<div class="mdp-grider-image-box">';
                        postsContent += '<img src="' + imgData + '" class="mdp-grider-image" alt="">';
                    postsContent += '</div>';
                    break;

                case 'title':
                    postsContent += '<h3 class="mdp-grider-title">' + content.title.rendered + '</h3>';
                    break;

                case 'description':
                    postsContent += '<div id="company" class="mdp-grider-description">' + content.excerpt.rendered + '</div>';
                    break;

                case 'button':
                    if( settings.general_link_type === 'button' ){
                        postsContent += '<p class="mdp-grider-box-button">';
                            postsContent += '<a href="' + content.guid.rendered + '" class="mpd-grider-button">';
                            postsContent += '<span>' + settings.general_button_text + '</span>';
                            postsContent += '</a>';
                        postsContent += '</p>';
                    }
                    break;
                case 'price':
                    postsContent += '';

                default:
                    console.debug('Posts layout none');
                }
                });

                postsContent += '</div>';

            }

            });

            }

        }

        let productsContent = '';
        if( '<?php echo esc_attr($ck_key); ?>' !== '' && '<?php echo esc_attr($cs_key); ?>' !== '' ){
        if( settings.general_source === 'woocommerce' ){
        let urlJSONProducts = local + '/wp-json/wc/v3/products?category=' + settings.general_products_category +
        '&_embed&consumer_key=<?php echo esc_attr($ck_key); ?>&consumer_secret=<?php echo esc_attr($cs_key); ?>';

        let products = get_result(urlJSONProducts);

        if( products !== undefined ){
        products.forEach( function( content ){

        if( content !== undefined ){

        productsContent += '<div class="mdp-grider-item item' + content.id + '">';

            _.each( layoutSortable(), function( item, index ) {
        switch ( item ) {
            case 'image':
                let imgData = content.images[0] !== undefined ?
                        content.images[0].src : '<?php echo esc_url($image_url); ?>';
                productsContent += '<div class="mdp-grider-image-box">';
                    productsContent += '<img src="' + imgData + '" class="mdp-grider-image" alt="">';
                    productsContent += '</div>';
                break;

            case 'title':
                productsContent += '<h3 class="mdp-grider-title">' + content.name + '</h3>';
                break;

            case 'description':
                productsContent += '<div id="company" class="mdp-grider-description">' +
                    content.short_description + '</div>';
                break;

            case 'price':
                productsContent += '<p class="mdp-grider-price">' +
                    content.price_html + '</p>';
                break;

            case 'button':
                if( settings.general_link_type === 'button' ){
                productsContent += '<p class="mdp-grider-box-button">';
                    productsContent += '<a href="' + content.permalink + '" class="mpd-grider-button">';
                        productsContent += '<span>' + settings.general_button_text + '</span>';
                        productsContent += '</a>';
                    productsContent += '</p>';
                }
                break;

            default:
                console.debug('Products layout none');
            }
            });

            productsContent += '</div>';

        }

        });
        }else{
        productsContent = 'Woocommerce source  does not support protocol http.';
        }

        }

        }else{
        productsContent = 'Specify the secret keys woocommerce in the plugin settings.';
        }

        /* End posts source. */

        function galleryInfo(){
        _.each( settings.grider_gallery, function( item, index ) {
        #>
        <div class="mdp-grider-item item{{{item.id}}}">
            <img src="{{{item.url}}}" class="mdp-grider-image" alt="">
        </div>
        <#
        });
        }

        /* Grid container classes */
        function gridContainerClasses() {

            cssClass = `mdp-grider-container mdp-grider-container-${ addon_id }`;
            cssClass += ` mdp-grider-${ settings.navigation_masonry === 'yes' ? 'masonry' : 'grid' }`;
            cssClass += `${ settings.navigation_animate === 'yes' && settings.navigation_animate_delay === 'yes' ? ' mdp-grider-delay' : '' }`;
            cssClass += `${ settings.grid_wrap === 'yes' ? ' mdp-grider-wrap' : '' }`;
            cssClass += `${ settings.grid_ignore_columns === 'yes' ? ' mdp-grider-ignore-cols' : '' }`;

            return cssClass;

        }
        #>

        <div id="mdp-grider-container-{{{addon_id}}}"
             class="{{{gridContainerClasses()}}}"
             data-cols-desktop="{{{settings.navigation_max_columns.size}}}"
             data-cols-tablet="{{{settings.navigation_max_columns_tablet.size}}}"
             data-cols-mobile="{{{settings.navigation_max_columns_mobile.size}}}">
            <#
            switch( settings.general_source ){

            case 'custom':
                customInfo( layoutSortable() );
                break;

            case 'gallery':
                galleryInfo();
                break;

            case 'posts':
                #>
                {{{postsContent}}}
                <#
                break;

            case 'woocommerce':
                #>
                {{{productsContent}}}
                <#
                break;

            default:
                console.debug('The source of information was not found.');
            }

            #>
        </div>
		<?php
	}

	/**
     * Return link for documentation
     * Used to add stuff after widget
     *
     * @access public
     *
     * @return string
     **/
    public function get_custom_help_url() {
        return 'https://docs.merkulov.design/tag/grider';
    }

}
