/**
 * Grider
 * Grider - Creating a dynamic grid layout has never been easier.
 * Exclusively on https://1.envato.market/grider-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.0
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

/**
 * Grider Helper
 */
class mdpGriderElementor {

    /**
     * Main constructor
     * @param $scope
     */
    constructor( $scope ) {

        new mdpGriderGrid( $scope );
        new mdpGriderMasonry( $scope );

    }

    /**
     * Get max cols for current breakpoint
     * @param $grid
     * @returns {number}
     */
    static getMaxCols( $grid ) {

        const elementorDeviceMode = document.body.dataset.elementorDeviceMode;
        return parseInt( $grid.dataset[ `cols${ this.capitalize( elementorDeviceMode ) }` ] );

    }

    /**
     * Capitalize first letter of the string
     * @param string
     * @returns {string}
     */
    static capitalize( string ) {

        return `${ string.charAt( 0 ).toUpperCase() }${ string.substr( 1 ).toLowerCase() }`;

    }

}

/**
 * Grider main class
 */
class mdpGriderGrid {

    $grids;

    /**
     * Constructor
     * @param $scope - Elementor widget wrapper
     */
    constructor( $scope ) {

        this.$grids = $scope.querySelectorAll( `.mdp-grider-container.mdp-grider-wrap` );
        this.runGrids( this.$grids );

    }

    /**
     * Rund grid for each grider container
     * @param $grids
     */
    runGrids( $grids ) {

        $grids.forEach( $grid => {

            // Initial run
            ! document.body.classList.contains( 'elementor-editor-active' ) ?
                mdpGriderGrid.applyColumnsCount( $grid ) :
                setInterval( () => { mdpGriderGrid.applyColumnsCount( $grid ) }, 10 );

            // Run on resize
            window.addEventListener( 'resize', function () {

                mdpGriderGrid.applyColumnsCount( $grid );

            } );

        } );

    }

    /**
     * Get width of each grid item and store in array
     * @param $items - array of grid items
     * @returns {*[]}
     */
    static getItemsWidth( $items ) {

        let itemsWidth = [];

        $items.forEach( $item => itemsWidth.push( parseInt( window.getComputedStyle( $item ).minWidth ) ) );

        return itemsWidth;

    }

    /**
     * Get count of items that can be placed in grid
     * @param $grid - grid wrapper element
     * @returns {number}
     */
    static getItemsCount( $grid ) {

        const $gridItems = $grid.querySelectorAll( `.mdp-grider-item` );
        const gridWidth = $grid.offsetWidth;
        const gridGap = parseInt( window.getComputedStyle( $grid ).gap );
        const gridItemsWidth = this.getItemsWidth( $gridItems );

        let itemsWidth = 0;
        let itemsCount = 0;

        for ( let i = 0; i < gridItemsWidth.length; i++ ) {

            itemsWidth += gridItemsWidth[ i ];
            itemsCount = i;
            if ( itemsWidth + i * gridGap >= gridWidth ) {

                itemsCount = i !== 0 ? i-1 : 0;
                break;

            }

        }

        // Ignore columns count
        return $grid.classList.contains( 'mdp-grider-ignore-cols' ) ? itemsCount + 1 : mdpGriderElementor.getMaxCols( $grid );

    }

    /**
     * Apply new columns count
     * @param $grid
     */
    static applyColumnsCount( $grid ) {

        const gridMinWidth = window.getComputedStyle( $grid.querySelector( `.mdp-grider-item` ) ).minWidth;

        $grid.style.gridTemplateColumns = `repeat( ${ this.getItemsCount( $grid ) }, ${ gridMinWidth } )`;

    }

}

/**
 * Grider Masonry Grid
 */
class mdpGriderMasonry {

    $grids;

    /**
     * Constructor
     * @param $scope
     */
    constructor( $scope ) {

        this.$grids = $scope.querySelectorAll( `.mdp-grider-container.mdp-grider-masonry` );
        this.runGrids( this.$grids );

    }

    /**
     * Runs grid for each grider container
     * @param $grids
     */
    runGrids( $grids ) {

        $grids.forEach( $grid => {

            // Initial run
            ! document.body.classList.contains( 'elementor-editor-active' ) ?
                mdpGriderMasonry.applyMasonryHeight( $grid ) :
                setInterval( () => { mdpGriderMasonry.applyMasonryHeight( $grid ) }, 10 );

            // Run on resize
            window.addEventListener( 'resize', function () {

                mdpGriderMasonry.applyMasonryHeight( $grid );

            } );

        } );

    }

    /**
     * Get minimal masonry column height
     * @param $grid
     * @returns {number}
     */
    static getMasonryHeight( $grid ) {

        const $gridItems = $grid.querySelectorAll( `.mdp-grider-item` );

        let itemsHeight = 0;
        for ( let i = 0; i < $gridItems.length / mdpGriderElementor.getMaxCols( $grid ); i++ ) {

            itemsHeight += $gridItems[ i ].offsetHeight;

        }

        return itemsHeight;

    }

    /**
     * Crop container related to the longest masonry row
     * @param $grid
     * @returns {number}
     */
    static getContainerHeight ( $grid ) {

        const $items = $grid.querySelectorAll( `.mdp-grider-item` );

        let top = 0;
        let bottom = 0;

        $items.forEach( $item => {

            // Get upper top grid positions
            if ( top === 0 || $item.getBoundingClientRect().top < top ) {

                top = $item.getBoundingClientRect().top;

            }

            // Get lower bottom grid position
            if ( bottom === 0 || $item.getBoundingClientRect().bottom > bottom ) {

                bottom = $item.getBoundingClientRect().bottom;

            }

        } );

        return Math.round( bottom - top );

    }

    /**
     * Fit container for elements stack length
     * @param $grid
     * @returns {boolean}
     */
    static extendContainerHeight( $grid ) {

        const $items = $grid.querySelectorAll( `.mdp-grider-item` );
        if ( $items.length < 1 ) { return false }

        return $grid.getBoundingClientRect().right < $items[ $items.length - 1 ].getBoundingClientRect().left;

    }

    /**
     * Apply CSS
     * @param $grid
     */
    static  applyMasonryHeight( $grid ) {

        // Skip all calculation for 1 column grid
        if ( mdpGriderElementor.getMaxCols( $grid ) === 1 ) {

            $grid.removeAttribute( 'style' );
            return;

        }

        $grid.style.height = `1px`;
        $grid.style.minHeight = `${ mdpGriderMasonry.getMasonryHeight( $grid ) }px`;

        if ( this.extendContainerHeight( $grid ) ) {

            for ( let i = 1; i < 1000; i++ ) {

                $grid.style.minHeight = `${ mdpGriderMasonry.getContainerHeight( $grid ) + i }px`;
                if ( ! this.extendContainerHeight( $grid ) ) { break }

            }

        }

        $grid.style.minHeight = `${ mdpGriderMasonry.getContainerHeight( $grid ) + 1 }px`;

    }

}

/**
 * Init grid
 */
jQuery( window ).on( 'elementor/frontend/init', () => {

    elementorFrontend.hooks.addAction( 'frontend/element_ready/global', ( $scopes ) => {

        let $scope = $scopes[ 0 ];

        if ( 'widget' !== $scope.getAttribute( 'data-element_type' ) ) { return; }
        if ( 'mdp-grider-elementor.default' !== $scope.getAttribute( 'data-widget_type' ) ) { return; }

        new mdpGriderElementor( $scope );

    } );

} );