<?php
/**
 * findgo functions and definitions
 *
 * Set up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme you can override certain functions (those wrapped
 * in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before
 * the parent theme's file, so the child theme functions would be used.
 *
 * @link https://codex.wordpress.org/Theme_Development
 * @link https://codex.wordpress.org/Child_Themes
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters,
 * {@link https://codex.wordpress.org/Plugin_API}
 *
 * @package WordPress
 * @subpackage Findgo
 * @since Findgo 1.3.61
 */

define( 'FINDGO_THEME_VERSION', '1.3.61' );
define( 'FINDGO_DEMO_MODE', false );

if ( !defined('APUSFINDGO_PREFIX') ) {
	define( 'APUSFINDGO_PREFIX', 'apus_findgo_' );
}

if ( ! isset( $content_width ) ) {
	$content_width = 660;
}

if ( ! function_exists( 'findgo_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * @since Findgo 1.0
 */
function findgo_setup() {

	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on findgo, use a find and replace
	 * to change 'findgo' to the name of your theme in all the template files
	 */
	load_theme_textdomain( 'findgo', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * See: https://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 825, 510, true );

	add_image_size( 'findgo-card-image', 375, 230, true );
	add_image_size( 'findgo-card-image-grid2', 440, 280, true );
	add_image_size( 'findgo-list-image', 220, 155, true );
	add_image_size( 'findgo-single-post', 750, 380, true );
	add_image_size( 'findgo-image-list', 250, 165, array( 'left', 'top' ) );
	add_image_size( 'findgo-image-gallery1', 480, 480, true );
	add_image_size( 'findgo-image-gallery2', 1920, 600, true );
	add_image_size( 'findgo-image-gallery3', 700, 370, true );
	add_image_size( 'findgo-image-gallery4', 820, 440, true );
	add_image_size( 'findgo-attachment', 180, 115, true );
	
	// This theme uses wp_nav_menu() in two locations.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu', 'findgo' ),
		'topmenu'  => esc_html__( 'Top Menu', 'findgo' ),
		'suggestions_search'  => esc_html__( 'Suggestions Search', 'findgo' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form', 'comment-form', 'comment-list', 'gallery', 'caption'
	) );

	add_theme_support( 'job-manager-templates' );
	add_theme_support( "woocommerce",array('gallery_thumbnail_image_width' => 110) );
	
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
	/*
	 * Enable support for Post Formats.
	 *
	 * See: https://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'template-posts', array(
		'aside', 'image', 'video', 'quote', 'link', 'gallery', 'status', 'audio', 'chat'
	) );

	$color_scheme  = findgo_get_color_scheme();
	$default_color = trim( $color_scheme[0], '#' );

	// Setup the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'findgo_custom_background_args', array(
		'default-color'      => $default_color,
		'default-attachment' => 'fixed',
	) ) );

	// Add support for Block Styles.
	add_theme_support( 'wp-block-styles' );

	add_theme_support( 'responsive-embeds' );
	
	// Add support for full and wide align images.
	add_theme_support( 'align-wide' );

	// Add support for editor styles.
	add_theme_support( 'editor-styles' );

	// Enqueue editor styles.
	add_editor_style( array( 'css/style-editor.css', findgo_get_fonts_url() ) );

	findgo_get_load_plugins();
}
endif; // findgo_setup
add_action( 'after_setup_theme', 'findgo_setup' );

/**
 * Load Google Front
 */
function findgo_get_fonts_url() {
    $fonts_url = '';

    /* Translators: If there are characters in your language that are not
    * supported by Montserrat, translate this to 'off'. Do not translate
    * into your own language.
    */
    $rubik = _x( 'on', 'Rubik font: on or off', 'findgo' );
    $barlow    = _x( 'on', 'Barlow font: on or off', 'findgo' );
 
    if ( 'off' !== $rubik ) {
        $font_families = array();
 
        if ( 'off' !== $rubik ) {
            $font_families[] = 'Rubik:300,400,500,700,900';
        }

        if ( 'off' !== $barlow ) {
            $font_families[] = 'Barlow:300,400,500,600,700';
        }

 		$font_google_code = findgo_get_config('font_google_code');
 		if (!empty($font_google_code) ) {
 			$font_families[] = $font_google_code;
 		}
        $query_args = array(
            'family' => ( implode( '|', $font_families ) ),
            'subset' => urlencode( 'latin,latin-ext' ),
        );
 		
 		$protocol = is_ssl() ? 'https:' : 'http:';
        $fonts_url = add_query_arg( $query_args, $protocol .'//fonts.googleapis.com/css' );
    }
 
    return esc_url_raw( $fonts_url );
}

function findgo_fonts_url() {  
	$protocol = is_ssl() ? 'https:' : 'http:';
	wp_enqueue_style( 'findgo-theme-fonts', findgo_get_fonts_url(), array(), null );
}
add_action('wp_enqueue_scripts', 'findgo_fonts_url');

/**
 * Enqueue scripts and styles.
 *
 * @since Findgo 1.0
 */
function findgo_enqueue_styles() {
	// load animate version 3.5.0
	wp_enqueue_style( 'animate', get_template_directory_uri() . '/css/animate.css', array(), '3.5.0' );
	
	//load font awesome
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/css/font-awesome.css', array(), '4.5.0' );

	wp_enqueue_style( 'font-flaticon', get_template_directory_uri() . '/css/flaticon.css', array(), '1.0.0' );

	// load bootstrap style
	if ( is_rtl() ) {
		wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/css/bootstrap-rtl.css', array(), '3.2.0' );
	} else {
		wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/css/bootstrap.css', array(), '3.2.0' );
	}

	wp_enqueue_style( 'timepicker', get_template_directory_uri() . '/css/jquery.timepicker.css' );
	wp_enqueue_style( 'magnific', get_template_directory_uri() . '/css/magnific-popup.css', array(), '1.1.0' );
	wp_enqueue_style( 'perfect-scrollbar', get_template_directory_uri() . '/css/perfect-scrollbar.css', array(), '2.3.2' );
	wp_enqueue_style( 'slick', get_template_directory_uri() . '/css/slick.css', array(), '2.3.2' );

	wp_enqueue_style( 'findgo-template', get_template_directory_uri() . '/css/template.css', array(), '3.2' );
	$footer_style = findgo_print_style_footer();
	if ( !empty($footer_style) ) {
		wp_add_inline_style( 'findgo-template', $footer_style );
	}
	
	$custom_style = findgo_custom_styles();
	if ( !empty($custom_style) ) {
		wp_add_inline_style( 'findgo-template', $custom_style );
	}
	wp_enqueue_style( 'findgo-style', get_template_directory_uri() . '/style.css', array(), '3.2' );
}
add_action( 'wp_enqueue_scripts', 'findgo_enqueue_styles', 100 );

function findgo_enqueue_scripts() {
	
	wp_enqueue_script( 'jquery-unveil', get_template_directory_uri() . '/js/jquery.unveil.js', array('jquery'), '20141010', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	wp_enqueue_script( 'bootstrap', get_template_directory_uri() . '/js/bootstrap.min.js', array( 'jquery' ), '20150330', true );
	wp_enqueue_script( 'slick', get_template_directory_uri() . '/js/slick.min.js', array( 'jquery' ), '1.8.0', true );
	
	
	wp_enqueue_script( 'timepicker', get_template_directory_uri() . '/js/jquery.timepicker.min.js' );
	wp_enqueue_script( 'magnific', get_template_directory_uri() . '/js/jquery.magnific-popup.min.js', array( 'jquery' ), '1.1.0', true );
	wp_enqueue_script( 'perfect-scrollbar', get_template_directory_uri() . '/js/perfect-scrollbar.jquery.min.js', array( 'jquery' ), '20150315', true );
	
	wp_register_script( 'findgo-functions', get_template_directory_uri() . '/js/functions.js', array( 'jquery' ), '20150330', true );

	wp_localize_script( 'findgo-functions', 'findgo_opts', array(
		'ajaxurl' => admin_url( 'admin-ajax.php' ),
		'time_format' => str_replace( '\\', '\\\\', get_option( 'time_format' ) ),
		'closed_text' => esc_html__( 'Closed', 'findgo' ),
		'next' => esc_html__('Next', 'findgo'),
		'previous' => esc_html__('Previous', 'findgo'),
	));
	wp_enqueue_script( 'findgo-functions' );
	wp_add_inline_script( 'findgo-script', "(function(html){html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);" );
}
add_action( 'wp_enqueue_scripts', 'findgo_enqueue_scripts', 1 );

/**
 * Display descriptions in main navigation.
 *
 * @since Findgo 1.0
 *
 * @param string  $item_output The menu item output.
 * @param WP_Post $item        Menu item object.
 * @param int     $depth       Depth of the menu.
 * @param array   $args        wp_nav_menu() arguments.
 * @return string Menu item with possible description.
 */
function findgo_nav_description( $item_output, $item, $depth, $args ) {
	if ( 'primary' == $args->theme_location && $item->description ) {
		$item_output = str_replace( $args->link_after . '</a>', '<div class="menu-item-description">' . $item->description . '</div>' . $args->link_after . '</a>', $item_output );
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'findgo_nav_description', 10, 4 );

/**
 * Add a `screen-reader-text` class to the search form's submit button.
 *
 * @since Findgo 1.0
 *
 * @param string $html Search form HTML.
 * @return string Modified search form HTML.
 */
function findgo_search_form_modify( $html ) {
	return str_replace( 'class="search-submit"', 'class="search-submit screen-reader-text"', $html );
}
add_filter( 'get_search_form', 'findgo_search_form_modify' );

/**
 * Function get opt_name
 *
 */
function findgo_get_opt_name() {
	return 'findgo_theme_options';
}
add_filter( 'apus_framework_get_opt_name', 'findgo_get_opt_name' );
/**
 * Function register demo mode
 *
 */
function findgo_register_demo_mode() {
	if ( defined('FINDGO_DEMO_MODE') && FINDGO_DEMO_MODE ) {
		return true;
	}
	return false;
}
add_filter( 'apus_framework_register_demo_mode', 'findgo_register_demo_mode' );
/**
 * Function get demo preset
 *
 */
function findgo_get_demo_preset() {
	$preset = '';
    if ( defined('FINDGO_DEMO_MODE') && FINDGO_DEMO_MODE ) {
    	if ( is_admin() ) {
    		$preset = get_option( 'apus_framework_preset_default' );
    	} elseif ( isset($_COOKIE['apus_preset']) && $_COOKIE['apus_preset'] ) {
    		$presets = get_option( 'apus_framework_presets' );
            if ( is_array($presets) && isset($presets[$_COOKIE['apus_preset']]) ) {
                $preset = $_COOKIE['apus_preset'];
            }
    	} else {
            $preset = get_option( 'apus_framework_preset_default' );
        }
    }
    return $preset;
}
if ( defined('FINDGO_DEMO_MODE') && FINDGO_DEMO_MODE ) {
	function findgo_set_demo_preset() {
	    
        if ( isset($_GET['_preset']) && $_GET['_preset'] ) {
            $presets = get_option( 'apus_framework_presets' );
            if ( is_array($presets) && isset($presets[$_GET['_preset']]) ) {
                $preset = $_GET['_preset'];
                setcookie( 'apus_preset', trim($preset) , time() + 1*24*60*60,'/' );
            	$_COOKIE['apus_preset'] = trim($preset);
            }
        }
	    
	}
	add_action( 'init', 'findgo_set_demo_preset' );
}

function findgo_set_exporter_first_settings_option_keys($option_keys) {
	return array_merge($option_keys, array(
		'job_manager_enable_categories',
		'job_manager_enable_default_category_multiselect',
		'job_manager_submit_job_form_page_id',
		'job_manager_job_dashboard_page_id',
		'job_manager_jobs_page_id',
		'job_manager_bookmark_page_id',
	));
}
add_filter( 'apus_exporter_first_settings_option_keys', 'findgo_set_exporter_first_settings_option_keys' );

function findgo_get_config($name, $default = '') {
	global $apus_options;
    if ( isset($apus_options[$name]) ) {
        return $apus_options[$name];
    }
    return $default;
}

function findgo_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar Default', 'findgo' ),
		'id'            => 'sidebar-default',
		'description'   => esc_html__( 'Add widgets here to appear in your Sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Header Search Sidebar', 'findgo' ),
		'id'            => 'search-sidebar',
		'description'   => esc_html__( 'Add widgets here to appear in your Header.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Blog sidebar', 'findgo' ),
		'id'            => 'blog-sidebar',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Listing archive sidebar', 'findgo' ),
		'id'            => 'listing-archive-sidebar',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Single listing sidebar before', 'findgo' ),
		'id'            => 'listing-sidebar-before',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Single listing sidebar after', 'findgo' ),
		'id'            => 'listing-sidebar-after',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'View Profile Sidebar', 'findgo' ),
		'id'            => 'view-profile-sidebar',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'My Profile Sidebar', 'findgo' ),
		'id'            => 'my-profile-sidebar',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'findgo' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );
}
add_action( 'widgets_init', 'findgo_widgets_init' );

function findgo_get_load_plugins() {

	$plugins[] =(array(
		'name'                     => esc_html__( 'Apus Framework For Themes', 'findgo' ),
        'slug'                     => 'apus-framework',
        'required'                 => true ,
        'source'				   => get_template_directory() . '/inc/plugins/apus-framework.zip'
	));

	$plugins[] =(array(
		'name'                     => esc_html__( 'WPBakery Visual Composer', 'findgo' ),
	    'slug'                     => 'js_composer',
	    'required'                 => true,
	    'source'				   => get_template_directory() . '/inc/plugins/js_composer.zip'
	));
	
	$plugins[] = array(
		'name'                     => esc_html__( 'Revolution Slider', 'findgo' ),
        'slug'                     => 'revslider',
        'required'                 => true ,
        'source'				   => get_template_directory() . '/inc/plugins/revslider.zip'
	);

	$plugins[] =(array(
		'name'                     => esc_html__( 'Cmb2', 'findgo' ),
	    'slug'                     => 'cmb2',
	    'required'                 => true,
	));

	$plugins[] =(array(
		'name'                     => esc_html__( 'MailChimp for WordPress', 'findgo' ),
	    'slug'                     => 'mailchimp-for-wp',
	    'required'                 =>  true
	));

	$plugins[] =(array(
		'name'                     => esc_html__( 'Contact Form 7', 'findgo' ),
	    'slug'                     => 'contact-form-7',
	    'required'                 => true,
	));

	// listing manager plugins
	$plugins[] =(array(
		'name'                     => esc_html__( 'WP Job Manager', 'findgo' ),
	    'slug'                     => 'wp-job-manager',
	    'required'                 => true,
	));

	$plugins[] =(array(
		'name'                     => esc_html__( 'Apus Findgo', 'findgo' ),
        'slug'                     => 'apus-findgo',
        'required'                 => true ,
        'source'				   => get_template_directory() . '/inc/plugins/apus-findgo.zip'
	));

	// woocommerce plugins
	$plugins[] =(array(
		'name'                     => esc_html__( 'WooCommerce', 'findgo' ),
	    'slug'                     => 'woocommerce',
	    'required'                 => true,
	));

	// login/register plugins
	$plugins[] = array(
		'name'                     => esc_html__( 'WordPress Social Login', 'findgo' ),
	    'slug'                     => 'wordpress-social-login',
	    'required'                 => false,
	);

	tgmpa( $plugins );
}

require get_template_directory() . '/inc/plugins/class-tgm-plugin-activation.php';
require get_template_directory() . '/inc/functions-helper.php';
require get_template_directory() . '/inc/functions-frontend.php';

/**
 * Implement the Custom Header feature.
 *
 */
require get_template_directory() . '/inc/custom-header.php';
require get_template_directory() . '/inc/classes/megamenu.php';
require get_template_directory() . '/inc/classes/mobilemenu.php';
require get_template_directory() . '/inc/classes/userinfo.php';
/**
 * Custom template tags for this theme.
 *
 */
require get_template_directory() . '/inc/template-tags.php';


if ( defined( 'APUS_FRAMEWORK_REDUX_ACTIVED' ) ) {
	require get_template_directory() . '/inc/vendors/redux-framework/redux-config.php';
	define( 'FINDGO_REDUX_FRAMEWORK_ACTIVED', true );
}
if ( findgo_is_cmb2_activated() ) {
	require get_template_directory() . '/inc/vendors/cmb2/footer.php';
	require get_template_directory() . '/inc/vendors/cmb2/page.php';
	define( 'FINDGO_CMB2_ACTIVED', true );
}
if( findgo_is_woocommerce_activated() ) {
	require get_template_directory() . '/inc/vendors/woocommerce/functions.php';
	require get_template_directory() . '/inc/vendors/woocommerce/functions-redux-configs.php';
	define( 'FINDGO_WOOCOMMERCE_ACTIVED', true );
}
if ( findgo_is_wp_job_manager_activated() ) {
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-redux-configs.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-submit-form.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-attachments.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-bookmark.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-products.php';
	
	require get_template_directory() . '/inc/vendors/wp-job-manager/vc-functions.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/vc.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-helper.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-hook.php';
	require get_template_directory() . '/inc/vendors/wp-job-manager/functions-review.php';
	define( 'FINDGO_WP_JOB_MANAGER_ACTIVED', true );
}
if ( findgo_is_vc_activated() ) {
	require get_template_directory() . '/inc/vendors/visualcomposer/functions.php';
	if ( defined('WPB_VC_VERSION') && version_compare( WPB_VC_VERSION, '6.0', '>=' ) ) {
		require get_template_directory() . '/inc/vendors/visualcomposer/vc-map-posts2.php';
	} else {
		require get_template_directory() . '/inc/vendors/visualcomposer/vc-map-posts.php';
	}
	require get_template_directory() . '/inc/vendors/visualcomposer/vc-map-theme.php';
	define( 'FINDGO_VISUALCOMPOSER_ACTIVED', true );
}
if ( findgo_is_apus_framework_activated() ) {
	function findgo_init_widgets() {
		require get_template_directory() . '/inc/widgets/custom_menu.php';
		require get_template_directory() . '/inc/widgets/recent_post.php';
		require get_template_directory() . '/inc/widgets/search.php';
		require get_template_directory() . '/inc/widgets/single_image.php';
		require get_template_directory() . '/inc/widgets/listing_search.php';
		require get_template_directory() . '/inc/widgets/user_short_profile.php';
		require get_template_directory() . '/inc/widgets/my_short_profile.php';
		require get_template_directory() . '/inc/widgets/listings.php';
	}
	add_action( 'widgets_init', 'findgo_init_widgets' );
	define( 'FINDGO_FRAMEWORK_ACTIVED', true );
}

if( findgo_is_wc_paid_listings_activated() ) {
	require get_template_directory() . '/inc/vendors/wc-paid-listings/functions.php';
	require get_template_directory() . '/inc/vendors/wc-paid-listings/vc.php';
	define( 'FINDGO_WC_PAID_LISTINGS_ACTIVED', true );
}

function findgo_register_widgets($widgets) {
	$widgets[] = 'instagram';
	return $widgets;
}
add_filter('apus_framework_register_widgets', 'findgo_register_widgets');
/**
 * Customizer additions.
 *
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Custom Styles
 *
 */
require get_template_directory() . '/inc/custom-styles.php';


function findgo_progress_import_steps($steps) {
	return array(
		'first_settings' => array( 'default' => esc_html__('Install First Settings', 'findgo'), 'installing' => esc_html__('Installing First Settings ...', 'findgo'), 'installed' => esc_html__('Installed First Settings', 'findgo') ),
		'content' => array( 'default' => esc_html__('Install Demo Content', 'findgo'), 'installing' => esc_html__('Installing Demo Content ...', 'findgo'), 'installed' => esc_html__('Installed Demo Content', 'findgo') ),
		'widgets' => array( 'default' => esc_html__('Install Widgets', 'findgo'), 'installing' => esc_html__('Installing Widgets ...', 'findgo'), 'installed' => esc_html__('Installed Widgets', 'findgo') ),
		'settings' => array( 'default' => esc_html__('Install Settings', 'findgo'), 'installing' => esc_html__('Installing Settings ...', 'findgo'), 'installed' => esc_html__('Installed Settings', 'findgo') ),
	);
}
add_filter('apus_frammework_progress_import_steps', 'findgo_progress_import_steps' );