<?php

namespace Elementor;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Plugin;

defined('ABSPATH') || exit;

Plugin::instance()->widgets_manager->register(new Widget_Categories_List());

class Widget_Categories_List extends Widget_Base
{
    public function get_name()
    {
        return 'felan-categories-list';
    }

    public function get_title()
    {
        return esc_html__('Categories List', 'felan-framework');
    }

    public function get_icon()
    {
        return 'felan-badge eicon-archive';
    }

    public function get_keywords()
    {
        return ['list', 'category', 'categories'];
    }

    public function get_style_depends()
    {
        return [FELAN_PLUGIN_PREFIX . 'categories-list'];
    }

    protected function register_controls()
    {
        $this->register_layout_section();
        $this->register_job_categories_section();
        $this->register_company_categories_section();
        $this->register_freelancer_categories_section();
        $this->register_service_categories_section();
        $this->register_project_categories_section();
    }

    private function register_layout_section()
    {
        $this->start_controls_section('layout_section', [
            'label' => esc_html__('Layout', 'felan-framework'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control(
            'layout',
            [
                'label' => esc_html__('Layout', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'default' => '01',
                'options' => [
                    '01' => esc_html__('01', 'felan-framework'),
                ],
                'prefix_class' => 'felan-layout-',
            ]
        );

        $this->add_control('cate', [
            'label' => esc_html__('Type Categories', 'felan-framework'),
            'type' => Controls_Manager::SELECT,
            'options' =>  array(
                'jobs-categories' => esc_html__('Jobs Categories', 'felan-framework'),
                'company-categories' => esc_html__('Company Categories', 'felan-framework'),
                'freelancer_categories' => esc_html__('Freelancer Categories', 'felan-framework'),
                'service-categories' => esc_html__('Service Categories', 'felan-framework'),
                'project-categories' => esc_html__('Project Categories', 'felan-framework'),
            ),
            'default' => 'jobs-categories',
        ]);

        $this->add_control(
            'category_depth',
            [
                'label' => esc_html__('Category Depth', 'felan-framework'),
                'type' => Controls_Manager::NUMBER,
                'default' => 1,
                'min' => 0,
                'max' => 5,
                'description' => esc_html__('0 = Parent categories only, 1 = Parent + Level 1, 2 = Parent + Level 1 + Level 2, etc.', 'felan-framework'),
            ]
        );

        $this->add_control(
            'max_categories',
            [
                'label' => esc_html__('Max Categories', 'felan-framework'),
                'type' => Controls_Manager::NUMBER,
                'default' => 6,
                'min' => 1,
                'max' => 20,
                'description' => esc_html__('Maximum total number of categories to display (all levels combined)', 'felan-framework'),
            ]
        );

        $this->add_control(
            'subcategory_orderby',
            [
                'label' => esc_html__('Subcategory order by', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'name' => esc_html__('Name', 'felan-framework'),
                    'id' => esc_html__('ID', 'felan-framework'),
                    'slug' => esc_html__('Slug', 'felan-framework'),
                    'count' => esc_html__('Post count', 'felan-framework'),
                ],
                'default' => 'name',
            ]
        );

        $this->add_control(
            'subcategory_order',
            [
                'label' => esc_html__('Order', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'ASC' => esc_html__('Ascending', 'felan-framework'),
                    'DESC' => esc_html__('Descending', 'felan-framework'),
                ],
                'default' => 'ASC',
            ]
        );

        $this->end_controls_section();
    }

    private function register_job_categories_section()
    {
        $this->start_controls_section('categories_job_section', [
            'label' => esc_html__('Jobs Categories', 'felan-framework'),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [
                'cate' => 'jobs-categories',
            ],
        ]);

        $repeater = new Repeater();

        $taxonomy_terms = get_categories(
            array(
                'taxonomy' => 'jobs-categories',
                'orderby' => 'name',
                'order' => 'ASC',
                'hide_empty' => 0,
                'parent' => 0,
            )
        );

        $categories = $category_slug = [];
        foreach ($taxonomy_terms as $category) {
            $categories[$category->slug] = $category->name;
        }

        $repeater->add_control(
            'jobs_selected_icon',
            [
                'label' => esc_html__('Icon', 'felan-framework'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'jobs_category',
            [
                'label' => esc_html__('Categories', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => $categories,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'jobs_categories_list',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'text' => esc_html__('Category #1', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #2', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #3', 'felan-framework'),
                    ],
                ],
            ]
        );

        $this->end_controls_section();
    }

    private function register_company_categories_section()
    {
        $this->start_controls_section('categories_company_section', [
            'label' => esc_html__('Company Categories', 'felan-framework'),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [
                'cate' => 'company-categories',
            ],
        ]);

        $repeater = new Repeater();

        $taxonomy_terms = get_categories(
            array(
                'taxonomy' => 'company-categories',
                'orderby' => 'name',
                'order' => 'ASC',
                'hide_empty' => 0,
                'parent' => 0,
            )
        );

        $categories = $category_slug = [];
        foreach ($taxonomy_terms as $category) {
            $categories[$category->slug] = $category->name;
        }

        $repeater->add_control(
            'company_selected_icon',
            [
                'label' => esc_html__('Icon', 'felan-framework'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'company_category',
            [
                'label' => esc_html__('Categories', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => $categories,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'company_categories_list',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'text' => esc_html__('Category #1', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #2', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #3', 'felan-framework'),
                    ],
                ],
            ]
        );

        $this->end_controls_section();
    }

    private function register_freelancer_categories_section()
    {
        $this->start_controls_section('categories_freelancer_section', [
            'label' => esc_html__('Freelancer Categories', 'felan-framework'),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [
                'cate' => 'freelancer_categories',
            ],
        ]);

        $repeater = new Repeater();

        $taxonomy_terms = get_categories(
            array(
                'taxonomy' => 'freelancer_categories',
                'orderby' => 'name',
                'order' => 'ASC',
                'hide_empty' => 0,
                'parent' => 0,
            )
        );

        $categories = $category_slug = [];
        foreach ($taxonomy_terms as $category) {
            $categories[$category->slug] = $category->name;
        }

        $repeater->add_control(
            'freelancer_selected_icon',
            [
                'label' => esc_html__('Icon', 'felan-framework'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'freelancer_category',
            [
                'label' => esc_html__('Categories', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => $categories,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'freelancer_categories_list',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'text' => esc_html__('Category #1', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #2', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #3', 'felan-framework'),
                    ],
                ],
            ]
        );

        $this->end_controls_section();
    }

    private function register_service_categories_section()
    {
        $this->start_controls_section('categories_service_section', [
            'label' => esc_html__('Service Categories', 'felan-framework'),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [
                'cate' => 'service-categories',
            ],
        ]);

        $repeater = new Repeater();

        $taxonomy_terms = get_categories(
            array(
                'taxonomy' => 'service-categories',
                'orderby' => 'name',
                'order' => 'ASC',
                'hide_empty' => 0,
                'parent' => 0,
            )
        );

        $categories = $category_slug = [];
        foreach ($taxonomy_terms as $category) {
            $categories[$category->slug] = $category->name;
        }

        $repeater->add_control(
            'service_selected_icon',
            [
                'label' => esc_html__('Icon', 'felan-framework'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'service_category',
            [
                'label' => esc_html__('Categories', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => $categories,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'service_categories_list',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'text' => esc_html__('Category #1', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #2', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #3', 'felan-framework'),
                    ],
                ],
            ]
        );

        $this->end_controls_section();
    }

    private function register_project_categories_section()
    {
        $this->start_controls_section('categories_project_section', [
            'label' => esc_html__('Project Categories', 'felan-framework'),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [
                'cate' => 'project-categories',
            ],
        ]);

        $repeater = new Repeater();

        $taxonomy_terms = get_categories(
            array(
                'taxonomy' => 'project-categories',
                'orderby' => 'name',
                'order' => 'ASC',
                'hide_empty' => 0,
                'parent' => 0,
            )
        );

        $categories = $category_slug = [];
        foreach ($taxonomy_terms as $category) {
            $categories[$category->slug] = $category->name;
        }

        $repeater->add_control(
            'project_selected_icon',
            [
                'label' => esc_html__('Icon', 'felan-framework'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'project_category',
            [
                'label' => esc_html__('Categories', 'felan-framework'),
                'type' => Controls_Manager::SELECT,
                'options' => $categories,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'project_categories_list',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'text' => esc_html__('Category #1', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #2', 'felan-framework'),
                    ],
                    [
                        'text' => esc_html__('Category #3', 'felan-framework'),
                    ],
                ],
            ]
        );

        $this->end_controls_section();
    }


    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $this->add_render_attribute('wrapper', 'class', array('felan-categories-list'));

        $taxonomy = $settings['cate'];
        if ($taxonomy == 'company-categories') {
            $post_type = 'company';
        } elseif ($taxonomy == 'freelancer_categories') {
            $post_type = 'freelancer';
        } elseif ($taxonomy == 'service-categories') {
            $post_type = 'service';
        } elseif ($taxonomy == 'project-categories') {
            $post_type = 'project';
        } else {
            $post_type = 'jobs';
        }

        $category_depth = !empty($settings['category_depth']) ? $settings['category_depth'] : 1;
        $max_categories = !empty($settings['max_categories']) ? $settings['max_categories'] : 6;
        $subcategory_orderby = !empty($settings['subcategory_orderby']) ? $settings['subcategory_orderby'] : 'name';
        $subcategory_order = !empty($settings['subcategory_order']) ? $settings['subcategory_order'] : 'ASC';

        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => 0,
            'parent' => 0,
        );

        $categories = get_categories($args); ?>

        <div <?php echo $this->get_render_attribute_string('wrapper') ?>>
            <?php foreach ($settings[$post_type . '_categories_list'] as $category_item) {
                $migrated = isset($category_item['__fa4_migrated']['selected_icon']);
                $is_new = !isset($category_item['icon']) && Icons_Manager::is_migration_allowed();
                $item_id = $category_item['_id'];
                $item_key = 'item_' . $item_id;
                $category_slug = $category_item[$post_type . '_category'];
                $cate = get_term_by('slug', $category_slug, $taxonomy);

                $this->add_render_attribute($item_key, 'class', array(
                    'list-cate-item',
                    'elementor-repeater-item-' . $item_id,
                ));
                if ($cate) {
                    $cate_id = $cate->term_id; ?>
                    <?php if (!empty($categories)) : ?>
                        <div <?php echo $this->get_render_attribute_string($item_key); ?>>
                            <ul class="list-all-categories">
                                <?php
                                foreach ($categories as $category) {
                                    if ($category->term_id === $cate_id) {
                                        echo '<li>';
                                        echo '<a href="' . esc_url(get_category_link($category->term_id)) . '">';
                                        if ($is_new || $migrated) {
                                            Icons_Manager::render_icon($category_item[$post_type . '_selected_icon'], ['aria-hidden' => 'true']);
                                        } elseif (!empty($category_item['icon'])) {
                                            echo '<i ' . $this->get_render_attribute_string('i') . '></i>';
                                        }
                                        echo esc_html($category->name);
                                        echo '</a>';
                                        // Hiển thị tất cả categories dạng phẳng
                                        $this->display_all_categories_flat(
                                            $category->term_id,
                                            $taxonomy,
                                            $max_categories,
                                            $subcategory_orderby,
                                            $subcategory_order,
                                            $category_depth
                                        );
                                        echo '</li>';
                                    }
                                }
                                ?>
                            </ul>
                        </div>
                    <?php endif; ?>
            <?php
                }
            } ?>
        </div>
    <?php }

    private function display_all_categories_flat($parent_id, $taxonomy, $max_limit = 6, $orderby = 'name', $order = 'ASC', $max_depth = 1)
    {
        $all_categories = array();

        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => 0,
            'parent' => $parent_id,
            'orderby' => $orderby,
            'order' => $order,
        );

        $subcategories = get_categories($args);

        $all_categories = array_merge($all_categories, $subcategories);

        if ($max_depth > 0) {
            foreach ($subcategories as $subcategory) {
                $sub_args = array(
                    'taxonomy' => $taxonomy,
                    'hide_empty' => 0,
                    'parent' => $subcategory->term_id,
                    'orderby' => $orderby,
                    'order' => $order,
                );

                $sub_subcategories = get_categories($sub_args);
                $all_categories = array_merge($all_categories, $sub_subcategories);
            }
        }

        if (!empty($all_categories)) {
            usort($all_categories, function($a, $b) use ($orderby, $order) {
                $value_a = '';
                $value_b = '';

                switch ($orderby) {
                    case 'name':
                        $value_a = $a->name;
                        $value_b = $b->name;
                        break;
                    case 'id':
                        $value_a = $a->term_id;
                        $value_b = $b->term_id;
                        break;
                    case 'slug':
                        $value_a = $a->slug;
                        $value_b = $b->slug;
                        break;
                    case 'count':
                        $value_a = $a->count;
                        $value_b = $b->count;
                        break;
                    default:
                        $value_a = $a->name;
                        $value_b = $b->name;
                }

                if ($order === 'ASC') {
                    return strcasecmp($value_a, $value_b);
                } else {
                    return strcasecmp($value_b, $value_a);
                }
            });
        }

        $all_categories = array_slice($all_categories, 0, $max_limit);

        if (!empty($all_categories)) {
            echo '<ul class="sub-all-categories">';
            foreach ($all_categories as $category) {
                echo '<li>';
                echo '<a href="' . esc_url(get_category_link($category->term_id)) . '">' . esc_html($category->name) . '</a>';
                echo '</li>';
            }
            echo '</ul>';
        }
    }

}
