<?php

/**
 * Class TX_Importer_Service.
 *
 */
class TX_Importer_Service {
	/**
	 * Validate file.
	 *
	 * @param $file
	 *
	 * @throws TX_Error
	 *
	 * @return bool
	 */
	private static function _file_validate( $file ) {
		if ( ! file_exists( $file ) ) {
			throw TX_Error::create( 'File not found!', 5, 'Please check the existence of file <code>' . $file . '</code>' );
		}

		if ( ! is_readable( $file ) ) {
			throw TX_Error::create( 'File is not readable!', 6, 'Please check the permission of file <code>' . $file . '</code>' );
		}

		return true;
	}

	/**
	 * Get file contents.
	 *
	 * @param $file
	 *
	 * @return string
	 * @throws TX_Error
	 */
	private static function _file_get_contents( $file ) {
		self::_file_validate( $file );

		$contents = file_get_contents( $file );
		if ( ! $contents ) {
			throw TX_Error::create( 'Get file content failed!', 6, 'Please check file <code>' . $file . '</code>' );
		}

		return $contents;
	}


	/**
	 * Import theme options.
	 *
	 * @param $options_file
	 *
	 * @return bool
	 * @throws TX_Error
	 */
	public static function customizer( $options_file ) {
		$content = self::_file_get_contents( $options_file );

		$options = maybe_unserialize( $content );
		if ( ! $options || is_string( $options ) ) {
			throw TX_Error::create( 'Decode file failed!', 7, 'Please check file <code>' . $options_file . '</code>' );
		}

		$theme = get_option( 'stylesheet' );
		// Get current theme mods
		$theme_mods = get_option( "theme_mods_{$theme}" );

		// Mergers new options and clean
		if ( $theme_mods ) {
			// Update to a new option so we can restore after that
			update_option( "theme_mods_backup_$theme", $theme_mods );

			$theme_mods = array_merge( $theme_mods, $options );
		} else {
			$theme_mods = $options;
		}
		$theme_mods = tx_clean_theme_mods( $theme_mods );

		// Update theme mods
		update_option( "theme_mods_$theme", $theme_mods );

		return true;
	}

	/**
	 * Analyze content for current demo
	 *
	 * @param $xml
	 */
	public static function analyze_content( $xml ) {
		$tx_wp_import = TX_WP_Import_Service::instance( true );
		$tx_wp_import->analyze_content( $xml );
	}

	/**
	 * Import widgets
	 *
	 * @param $widget_file
	 *
	 * @return bool
	 * @throws TX_Error
	 */
	public static function widget( $widget_file ) {
		$json_data = self::_file_get_contents( $widget_file );
		$json_data = json_decode( $json_data, true );

		if ( empty( $json_data ) || ! is_array( $json_data ) ) {
			throw TX_Error::create( 'Import data could not be read.', 6, 'Please check file <code>' . $widget_file . '</code>' );
		}

		//$tx_widget_importer = new TX_Widget_Importer_Service();
		//$tx_widget_importer->import( $json_data );
		TX_Widget_Importer_Service::import($json_data);

		return true;
	}

	/**
	 * Import Slider Revolution.
	 *
	 * @param array $revsliders
	 *
	 * @return bool
	 * @throws TX_Error
	 */
	public static function revslider( $revsliders ) {
		$THEME_PATH     = get_template_directory();
		$dir_revsliders = apply_filters( 'tx_wizard_importer_directory_revsliders', $THEME_PATH . '/inc/data/revsliders/' );

		if ( ! is_array( $revsliders ) || ! count( $revsliders ) ) {
			return false;
		}

		foreach ( $revsliders as $file ) {
			$f    = $dir_revsliders . $file;
			$file = TX_File_Helper::download_file( $f );

			if ( is_wp_error( $file ) ) {
				throw TX_Error::create( 'Download slider data failed.', 8, '' );
			}

			if ( ! is_file( $file ) ) {
				continue;
			}

			$_FILES['import_file']['tmp_name'] = $file;
			if ( class_exists( 'RevSlider' ) ) {
				$slider   = new RevSlider();
				$response = $slider->importSliderFromPost( true, true );
				if ( ! $response['success'] ) {
					throw TX_Error::create( 'Import slider failed.', 0, '' );
				}
			}
		}

		return true;
	}

	/**
	 * Import Redux Options.
	 *
	 * @param file $import_data
	 *
	 * @return bool
	 */

	public static function redux( $import_data ) {
		// Redux plugin is not active!
		if ( ! class_exists( 'ReduxFramework' ) ) {
			$error_message = esc_html__( 'The Redux plugin is not activated, so the Redux import was skipped!', 'edumodo' );

			TX_Dashboard::add_notification( array(
				'content' => $error_message,
				'type'    => 'error',
			) );

			return;
		}

		$redux_options_raw_data = self::_file_get_contents( $import_data ); 

		$redux_options_data = json_decode( $redux_options_raw_data, true );

		$redux_framework = ReduxFrameworkInstances::get_instance( apply_filters( 'tx_wizard_redux_option', 'edumodo_options' ) );

		if ( isset( $redux_framework->args['opt_name'] ) ) {
			// Import Redux settings.
			$redux_framework->set_options( $redux_options_data );

			return true;
		} else {
			$error_message = sprintf( esc_html__( 'The Redux option name: %s, was not found in this WP site, so it was not imported!', 'edumodo' ), $redux_item['option_name'] );
			// Write error to log file.
			TX_Dashboard::add_notification( array(
				'content' => $error_message,
				'type'    => 'error',
			) );
		}
	}

	/**
	 * Update site settings.
	 *
	 * @param $settings_file
	 */
	public static function settings( $settings_file ) {
		$text     = self::_file_get_contents( $settings_file );
		$settings = maybe_unserialize( $text );

		if ( ! is_array( $settings ) ) {
			return;
		}

		/**
		 * Update basic settings.
		 */
		$basic_settings = TX_Importer_Service::get_key_basic_settings();
		foreach ( $basic_settings as $basic_setting ) {
			if ( isset( $settings[ $basic_setting ] ) ) {
				update_option( $basic_setting, $settings[ $basic_setting ] );
			}
		}

		/**
		 * Mapping settings page slug to page id.
		 */
		$settings_key = TX_Importer_Service::get_key_page_id_settings();
		foreach ( $settings_key as $key ) {
			$page_slug = isset( $settings[ $key ] ) ? $settings[ $key ] : false;

			$page_id = tx_get_page_id_by_path( $page_slug );
			if ( $page_id ) {
				update_option( $key, $page_id );
			}
		}
	}

	/**
	 * Reset demo data.
	 *
	 * @return bool
	 */
	public static function reset_data_demo() {
		$tx_wp_import = TX_WP_Import_Service::instance();
		$tx_wp_import->clean_demo_content();

		self::remove_all_rev_sliders();

		/**
		 * Remove all widgets.
		 */
		TX_Widget_Importer_Service::remove_all();

		/**
		 * Reset show on front page.
		 */
		update_option( 'show_on_front', 'posts' );

		/**
		 * Backup theme mods.
		 */
		$theme             = get_option( 'stylesheet' );
		$theme_mods_backup = get_option( "theme_mods_backup_{$theme}" );
		tx_clean_theme_mods( $theme_mods_backup );
		update_option( "theme_mods_$theme", $theme_mods_backup );

		TX_Importer::update_key_demo_installed( false );

		return true;
	}

	/**
	 * Remove all revolution sliders.
	 */
	private static function remove_all_rev_sliders() {
		if ( ! class_exists( 'RevSlider' ) ) {
			return;
		}

		$slider     = new RevSlider();
		$arrSliders = $slider->getArrSliders();

		foreach ( $arrSliders as $slider ) {
			if ( method_exists( $slider, 'deleteSlider' ) ) {
				$slider->deleteSlider();
			}
		}
	}

	/**
	 * Get key page id settings.
	 *
	 * @return array
	 */
	public static function get_key_page_id_settings() {
		$list = array(
			'woocommerce_shop_page_id',
			'woocommerce_cart_page_id',
			'woocommerce_checkout_page_id',
			'woocommerce_terms_page_id',
			'page_for_posts',
			'page_on_front',
		);

		return apply_filters( 'tx_importer_page_id_settings', $list );
	}

	/**
	 * Get key basic settings.
	 *
	 * @return array
	 */
	public static function get_key_basic_settings() {
		$list = array(
			'show_on_front',
		);

		return apply_filters( 'tx_importer_basic_settings', $list );
	}

}
