<?php

/**
 * Class TX_Wizard_Admin.
 *
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'TX_Wizard_Admin' ) ) {
	class TX_Wizard_Admin extends TX_Singleton {
		/**
		 * Go to theme dashboard.
		 */
		public static function go_to_theme_dashboard() {
			$link_page = admin_url( '?tx-wizard-redirect-to-dashboard' );

			tx_wizard_redirect( $link_page );
		}

		/**
		 * TX_Wizard_Admin constructor.
		 */
		protected function __construct() {
			spl_autoload_register( array( $this, 'autoload' ) );

			$this->init();
			$this->init_hooks();
		}

		/**
		 * Fake page to redirect to dashboard.
		 */
		public function redirect_to_dashboard() {
			$request = isset( $_REQUEST['tx-wizard-redirect-to-dashboard'] );

			if ( ! $request ) {
				return;
			}

			$this->redirect_user();
		}

		/**
		 * Handle redirect the user.
		 */
		private function redirect_user() {
			$url = TX_Dashboard::get_link_main_dashboard();

			if ( TX_Dashboard::check_first_install() ) {
				TX_Admin_Settings::set( 'first_install', false );

				$url = TX_Dashboard::get_link_page_by_slug( 'getting-started' );
			}

			tx_wizard_redirect( $url );
		}

		/**
		 * Init.
		 */
		private function init() {
			$this->run();
		}

		/**
		 * Autoload classes.
		 *
		 * @param $class
		 */
		public function autoload( $class ) {
			$class = strtolower( $class );

			$file_name = 'class-' . str_replace( '_', '-', $class ) . '.php';

			if ( strpos( $class, 'service' ) !== false ) {
				$file_name = 'services/' . $file_name;
			}

			$file = TX_WIZARD_ADMIN_PATH . DIRECTORY_SEPARATOR . $file_name;
			if ( is_readable( $file ) ) {
				require_once $file;
			}
		}

		/**
		 * Notice permission uploads.
		 */
		public function notice_permission_uploads() {
			$dir = WP_CONTENT_DIR;

			$writable = wp_is_writable( $dir );
			if ( $writable ) {
				return;
			}

			TX_Notification::add_notification(
				array(
					'id'          => 'permission_uploads',
					'type'        => 'error',
					'content'     => __( "<h3>Important!</h3>Your server doesn't not have a permission to write in <strong>WP Uploads</strong> folder ($dir).
									The theme may not work properly with the issue. Please check this <a href='https://goo.gl/guirO5' target='_blank'>guide</a> to fix it.", 'edumodo' ),
					'dismissible' => false,
					'global'      => true,
				)
			);
		}

		/**
		 * Run admin core.
		 */
		private function run() {
			TX_Modal::instance();
			TX_Dashboard::instance();
			TX_Importer_Mapping::instance();
			TX_Welcome_Panel::instance();
			TX_Exporter::instance();
		}

		/**
		 * Init hooks.
		 */
		private function init_hooks() {
			add_action( 'admin_init', array( $this, 'redirect_to_dashboard' ) );
			add_action( 'admin_menu', array( $this, 'remove_unnecessary_menus' ), 999 );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ), 5 );
			add_action( 'admin_head', array( $this, 'admin_styles' ) );
			add_filter( 'tx_wizard_installer_hidden_menu', array( $this, 'hidden_menu_installer' ) );
			add_action( 'tx_wizard_dashboard_init', array( $this, 'notice_permission_uploads' ) );
			add_filter( 'tx_wizard_admin_memory_limit', array( $this, 'memory_limit' ) );
		}

		/**
		 * Raise memory limit.
		 *
		 * @param $current
		 *
		 * @return string
		 */
		public function memory_limit( $current ) {
			$current_limit_int = wp_convert_hr_to_bytes( $current );

			if ( $current_limit_int > 268435456 ) {
				return $current;
			}

			return '256M';
		}


		/**
		 * Hidden menu installer.
		 *
		 * @return bool
		 */
		public function hidden_menu_installer() {
			return true;
		}

		/**
		 * Add custom style inline in admin.
		 */
		public function admin_styles() {
			global $_wp_admin_css_colors;

			$colors = array(
				'#222',
				'#333',
				'#0073aa',
				'#00a0d2',
			);

			$pack = get_user_meta( get_current_user_id(), 'admin_color', true );
			if ( is_array( $_wp_admin_css_colors ) ) {
				foreach ( $_wp_admin_css_colors as $key => $package ) {
					if ( $pack == $key ) {
						$package = (array) $package;
						$colors  = $package['colors'];
					}
				}

			}

			TX_Template_Helper::template( 'admin-styles.php', array( 'colors' => $colors, 'key' => $pack ), true );
		}

		/**
		 * Remove unnecessary menus.
		 */
		public function remove_unnecessary_menus() {
			global $submenu;
			unset( $submenu['themes.php'][15] );
			unset( $submenu['themes.php'][20] );
		}


		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			wp_register_script( 'tx-wizard-admin', TX_WIZARD_ADMIN_URI . '/assets/js/core.js', array( 'jquery' ), TX_WIZARD_VERSION );

			wp_register_script( 'tx-wizard-clipboard', TX_WIZARD_ADMIN_URI . '/assets/js/clipboard.min.js', array(), '1.6.0' );

			wp_register_style( 'tx-font-awesome', 'https://use.fontawesome.com/e8cbfd9eca.css', array(), '4.7.0' );

			wp_enqueue_style( 'tx-admin', TX_WIZARD_ADMIN_URI . '/assets/css/admin.css', array(), TX_WIZARD_VERSION );

			if ( is_rtl() ) {
				wp_enqueue_style( 'tx-rtl-admin', TX_WIZARD_ADMIN_URI . '/assets/css/rtl.css', array(), TX_WIZARD_VERSION );
			}

			$this->localize_script();
		}

		/**
		 * Localize script.
		 */
		private function localize_script() {
			wp_localize_script( 'tx-wizard-admin', 'tx_wizard_settings', array(
				'active'  => 'yes',
				'is_free' => 'no',
			) );
		}
	}

	/**
	 * Thim Core Admin init.
	 */
	function tx_wizard_admin_init() {
		TX_Wizard_Admin::instance();
	}

	add_action( 'after_setup_theme', 'tx_wizard_admin_init', 99999 );
}

/**
 * Include functions.
 */
include_once TX_WIZARD_ADMIN_PATH . '/functions.php';
