<?php

/**
 * Class TX_Dashboard.
 *
 */
class TX_Dashboard extends TX_Singleton {
	/**
	 * Do not edit value.
	 *
	 * @var string
	 */
	private static $main_slug = 'dashboard';

	/**
	 * @var string
	 */
	public static $prefix_slug = 'tx-';

	/**
	 * List sub pages.
	 * @var array
	 */
	private static $sub_pages = array();

	/**
	 * @var null
	 */
	private static $current_key_page = null;

	/**
	 * Check first install.
	 */
	public static function check_first_install() {
		return TX_Admin_Settings::get( 'first_install', true );
	}

	/**
	 * Get link page by slug.
	 *
	 * @param $slug
	 *
	 * @return string
	 */
	public static function get_link_page_by_slug( $slug ) {
		return admin_url( 'admin.php?page=' . self::$prefix_slug . $slug );
	}

	/**
	 * Get link main dashboard.
	 *
	 * @param array $args [key => value] => &key=value
	 *
	 * @return string
	 */
	public static function get_link_main_dashboard( $args = null ) {
		$url = self::get_link_page_by_slug( self::$main_slug );

		if ( is_array( $args ) ) {
			$url = add_query_arg( $args, $url );
		}

		return $url;
	}

	/**
	 * Get key (slug) current page.
	 */
	public static function get_current_page_key() {
		if ( is_null( self::$current_key_page ) ) {
			$current_page = isset( $_GET['page'] ) ? $_GET['page'] : '';
			$prefix_slug  = TX_Dashboard::$prefix_slug;

			$pages = self::get_sub_pages();
			foreach ( $pages as $key => $page ) {
				if ( $prefix_slug . $key === $current_page ) {
					self::$current_key_page = $key;

					return self::$current_key_page;
				}
			}

			self::$current_key_page = false;
		}

		return self::$current_key_page;
	}

	/**
	 * Check current request is for a page of Thim Core Dashboard interface.
	 *
	 * @return bool True if inside Thim Core Dashboard interface, false otherwise.
	 */
	public static function is_dashboard() {
		$current_page = self::get_current_page_key();

		return (bool) ( $current_page );
	}

	/**
	 * Set list sub pages.
	 */
	private static function set_sub_pages() {
		self::$sub_pages = apply_filters( 'tx_dashboard_sub_pages', array() );
	}

	/**
	 * Get list sub pages.
	 *
	 * @return array
	 */
	public static function get_sub_pages() {
		if ( empty( self::$sub_pages ) ) {
			self::set_sub_pages();
		}

		return self::$sub_pages;
	}

	/**
	 * Add notifications.
	 *
	 * @param array $args
	 */
	public static function add_notification( $args = array() ) {
		$current_page = self::get_current_page_key();

		$default = array(
			'content' => '',
			'type'    => 'success',
			'page'    => $current_page,
		);
		$args    = wp_parse_args( $args, $default );

		$page = $args['page'];
		if ( $page !== $current_page ) {
			return;
		}

		$type    = $args['type'];
		$content = $args['content'];
		add_action( 'tx_dashboard_notifications', function () use ( $type, $content ) {

			?>
            <div class="txw-notice txw-<?php echo esc_attr( $type ); ?>">
                <div class="content"><?php echo $content; ?></div>
            </div>
			<?php
		} );
	}

	/**
	 * Get page template.
	 *
	 * @param $template
	 * @param array $args
	 *
	 * @return bool
	 */
	public static function get_template( $template, $args = array() ) {
		return TX_Template_Helper::template( 'dashboard/' . $template, $args, true );
	}

	/**
	 * TX_Dashboard constructor.
	 */
	protected function __construct() {
		$this->init();
		$this->init_hooks();
	}

	/**
	 * Init.
	 */
	private function init() {

	    $show_update = apply_filters('tx_wizard_envato_update', false);

		TX_Notification::instance();
		TX_Main_Dashboard::instance();
		TX_Getting_Started::instance();
		TX_Importer::instance();
		TX_Plugins_Manager::instance();
		if($show_update) {
			TX_Updates::instance();
        }
	}

	/**
	 * Get page template.
	 *
	 * @param        string $template
	 * @param  mixed $args
	 *
	 * @return bool
	 */
	private function get_page_template( $template, $args = array() ) {
		return TX_Template_Helper::template( 'dashboard/' . $template, $args, true );
	}

	/**
	 * Init hooks.
	 */
	private function init_hooks() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		add_action( 'admin_menu', array( $this, 'add_sub_menu_pages' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'tx_dashboard_notifications', array( $this, 'add_notification_requirements' ) );

		add_filter( 'admin_body_class', array( $this, 'admin_body_class' ) );
		add_action( 'admin_bar_menu', array( $this, 'add_menu_admin_bar' ), 50 );
		add_filter( 'admin_footer_text', array( $this, 'admin_footer_text' ) );
		add_filter( 'update_footer', array( $this, 'admin_footer_version' ), 999 );
		add_action( 'tx_wizard_dashboard_init', array( $this, 'handle_notification_error' ) );
	}

	/**
	 * Handle notification error.
	 */
	public function handle_notification_error() {
		$code = ! empty( $_REQUEST['tx-wizard-error'] ) ? $_REQUEST['tx-wizard-error'] : false;
		if ( ! $code ) {
			return;
		}

		$messages = array(
			'something_went_wrong' => __( 'Something went wrong! Please try again later.', 'edumodo' )
		);
		$messages = apply_filters( 'tx_wizard_list_error_messages', $messages );

		if ( ! isset( $messages[ $code ] ) ) {
			return;
		}

		TX_Notification::add_notification( array(
			'id'          => 'tx_wizard_dashboard_error',
			'type'        => 'error',
			'content'     => $messages[ $code ],
			'dismissible' => false,
			'global'      => false,
		) );
	}

	/**
	 * Add notification requirements.
	 */
	public function add_notification_requirements() {
		$version_require = '2.3';

		if ( version_compare( phpversion(), $version_require, '>=' ) ) {
			return;
		}

		?>
        <div class="txw-notice txw-error">
            <div class="content">
				<?php printf( __( '<strong>Important:</strong> We found out your system is using PHP version %1$s. Please consider upgrading to version %2$s or higher.', 'edumodo' ), phpversion(), $version_require ); ?>
            </div>
        </div>
		<?php
		exit();
	}

	/**
	 * Filter admin footer version (on the right).
	 *
	 * @param $msg
	 *
	 * @return string
	 */
	public function admin_footer_version( $msg ) {
		if ( ! self::is_dashboard() ) {
			return $msg;
		}

		return sprintf( 'Wizard Version %s', TX_WIZARD_VERSION );
	}

	/**
	 * Filter admin footer text.
	 *
	 * @param $html
	 *
	 * @return string
	 */
	public function admin_footer_text( $html ) {
		if ( ! self::is_dashboard() ) {
			return $html;
		}

		$text = sprintf( __( 'Thank you for creating with <a href="%s" target="_blank">ThemeXpert</a>.', 'edumodo' ), __( 'https://themexpert.com' ) );

		return '<span id="footer-thankyou">' . $text . '</span>';
	}

	/**
	 * Add admin bar menu.
	 *
	 * @param $wp_admin_bar WP_Admin_Bar
	 */
	public function add_menu_admin_bar( $wp_admin_bar ) {
		if ( ! current_user_can( 'edit_theme_options' ) ) {
			return;
		};

		if ( is_admin() ) {
			return;
		}

		$menu_title = __( 'Edumodo', 'edumodo' );

		$args = array(
			'id'    => 'tx_wizard',
			'title' => $menu_title,
			'href'  => self::get_link_main_dashboard()
		);
		$wp_admin_bar->add_node( $args );

		$pages = self::get_sub_pages();
		foreach ( $pages as $key => $page ) {
			$args = array(
				'id'     => self::$prefix_slug . $key,
				'title'  => $page['title'],
				'href'   => self::get_link_page_by_slug( $key ),
				'parent' => 'tx_wizard'
			);
			$wp_admin_bar->add_node( $args );
		}
	}

	/**
	 * Enqueue scripts.
	 */
	public function enqueue_scripts() {
		if ( ! self::is_dashboard() ) {
			return;
		}

		do_action( 'tx_wizard_dashboard_enqueue_scripts' );

		wp_enqueue_script( 'tx-wizard-admin' );
		wp_enqueue_script( 'tx-modal-v2' );
		wp_enqueue_script( 'tx-dashboard', TX_WIZARD_ADMIN_URI . '/assets/js/dashboard.js', array( 'jquery-ui-sortable' ), TX_WIZARD_VERSION );
		wp_enqueue_style( 'tx-dashboard', TX_WIZARD_ADMIN_URI . '/assets/css/dashboard.css', array(), TX_WIZARD_VERSION );

		$this->localize_script();
	}

	/**
	 * Localize script.
	 */
	private function localize_script() {
		wp_localize_script( 'tx-dashboard', 'tx_dashboard', array(
			'admin_ajax' => admin_url( 'admin-ajax.php?action=tx_dashboard_order_boxes' )
		) );
	}

	/**
	 * Add class to body class in admin.
	 *
	 * @param $body_classes
	 *
	 * @return string
	 */
	public function admin_body_class( $body_classes ) {
		if ( ! self::is_dashboard() ) {
			return $body_classes;
		}

		$current_page_key = self::get_current_page_key();
		$prefix           = self::$prefix_slug;
		$current_page     = $prefix . $current_page_key;
		$main_page        = $prefix . self::$main_slug;

		$body_classes .= ' ' . $main_page . ' ' . $current_page . '-wrapper';

		return $body_classes;
	}

	/**
	 * Add menu page (Main page).
	 */
	public function add_menu_page() {
		$title      = __( 'Edumodo', 'edumodo' );
		$menu_title = apply_filters( 'tx_wizard_theme_dashboard_menu_title', $title );

		add_menu_page(
			$title,
			$menu_title,
			'manage_options',
			self::$prefix_slug . self::$main_slug,
			array(
				$this,
				'master_template'
			),
			TX_WIZARD_ADMIN_URI . '/assets/images/fav.png',
			2
		);
	}

	/**
	 * Add sub menu pages.
	 */
	public function add_sub_menu_pages() {
		$sub_pages = $this->get_sub_pages();
		$prefix    = TX_Dashboard::$prefix_slug;

		foreach ( $sub_pages as $key => $page ) {
			$default = array(
				'title'    => '',
				'template' => '',
			);
			$page    = wp_parse_args( $page, $default );

			$slug  = $prefix . $key;
			$title = $page['title'];

			add_submenu_page( self::$prefix_slug . self::$main_slug, $title, $title, 'manage_options', $slug, array( $this, 'master_template' ) );
		}
	}

	/**
	 * Master template.
	 */
	public function master_template() {
		do_action( 'tx_wizard_dashboard_init' );

		$this->get_page_template( 'master.php' );
	}
}
