<?php

/**
 * Class TX_Getting_Started
 */
class TX_Getting_Started extends TX_Admin_Sub_Page {
	/**
	 * @var string
	 */
	public $key_page = 'getting-started';

	/**
	 * Get steps.
	 *
	 * @return array
	 */
	public static function get_steps() {
		$steps = array();

		$steps[] = array(
			'key'   => 'welcome',
			'title' => __( 'Welcome', 'edumodo' ),
		);

		$steps[] = array(
			'key'   => 'quick-setup',
			'title' => __( 'Setup', 'edumodo' ),
		);


		$plugins = TX_Plugins_Manager::get_required_plugins_inactive();
		if ( count( $plugins ) > 0 ) {
			$steps[] = array(
				'key'   => 'install-plugins',
				'title' => __( 'Plugins', 'edumodo' ),
			);
		}

		$steps[] = array(
			'key'   => 'import-demo',
			'title' => __( 'Import', 'edumodo' ),
		);

		$steps[] = array(
			'key'   => 'support',
			'title' => __( 'Support', 'edumodo' ),
		);

		$steps[] = array(
			'key'   => 'finish',
			'title' => __( 'Ready', 'edumodo' ),
		);

		return $steps;
	}

	/**
	 * Get link redirect to step.
	 *
	 * @param $step
	 *
	 * @return string
	 */
	public static function get_link_redirect_step( $step ) {
		$self = self::instance();
		$base = TX_Dashboard::get_link_page_by_slug( $self->key_page );

		return add_query_arg( array( 'redirect' => $step ), $base );
	}

	/**
	 * TX_Getting_Started constructor.
	 */
	protected function __construct() {
		parent::__construct();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks() {
		add_action( 'admin_init', array( $this, 'redirect_to_tep' ) );
		add_action( 'txw_after_dashboard_wrapper', array( $this, 'add_modals_importer' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'tx_getting_started_main_content', array( $this, 'render_step_templates' ) );
		add_action( 'wp_ajax_tx-get-started', array( $this, 'handle_ajax' ) );
		add_filter( 'tx_dashboard_sub_pages', array( $this, 'add_sub_page' ) );
		add_action( 'admin_init', array( $this, 'notice_visited' ) );
	}

	/**
	 * Notice customer has already visited.
	 */
	public function notice_visited() {
		if ( ! $this->is_myself() ) {
			return;
		}

		if ( ! self::already_visited() ) {
			return;
		}

		TX_Notification::add_notification(
			array(
				'id'          => 'gs-visited',
				'type'        => 'warning',
				'content'     => __( 'Oops! It seems you already setup your site.', 'edumodo' ),
				'dismissible' => true,
				'global'      => false,
			)
		);
	}

	/**
	 * Redirect to step.
	 */
	public function redirect_to_tep() {
		if ( ! $this->is_myself() ) {
			return;
		}

		$step_redirect = ! empty( $_GET['redirect'] ) ? $_GET['redirect'] : false;
		if ( ! $step_redirect ) {
			return;
		}

		$steps = self::get_steps();
		foreach ( $steps as $index => $step ) {
			if ( $step['key'] == $step_redirect ) {
				$url = $this->get_link_myself() . "#step-$index";

				tx_wizard_redirect( $url );
			}
		}
	}

	/**
	 * Check already visited.
	 *
	 * @return bool
	 */
	public static function already_visited() {
		$option = TX_Admin_Settings::get( 'getting_started_visited', false );

		return (bool) $option;
	}

	/**
	 * Add modals importer.
	 */
	public function add_modals_importer() {
		if ( ! $this->is_myself() ) {
			return;
		}

		TX_Dashboard::get_template( 'partials/importer-modal.php' );
		TX_Dashboard::get_template( 'partials/importer-uninstall-modal.php' );
	}

	/**
	 * Add sub page.
	 *
	 * @param $sub_pages
	 *
	 * @return mixed
	 */
	public function add_sub_page( $sub_pages ) {
		$sub_pages['getting-started'] = array(
			'title' => __( 'Getting Started', 'edumodo' ),
		);

		return $sub_pages;
	}

	/**
	 * Enqueue scripts.
	 */
	public function enqueue_scripts() {
		if ( ! $this->is_myself() ) {
			return;
		}

		wp_enqueue_script( 'tx-plugins', TX_WIZARD_ADMIN_URI . '/assets/js/plugins/plugins.js', array( 'jquery' ) );
		wp_enqueue_script( 'tx-importer', TX_WIZARD_ADMIN_URI . '/assets/js/importer.js', array( 'wp-util', 'jquery', 'backbone', 'underscore' ) );
		wp_enqueue_script( 'tx-getting-started', TX_WIZARD_ADMIN_URI . '/assets/js/getting-started.js', array( 'tx-plugins', 'tx-importer' ) );

		$this->localize_script();
	}

	/**
	 * Localize script.
	 */
	private function localize_script() {
		wp_localize_script( 'tx-getting-started', 'tx_gs', array(
			'url_ajax' => admin_url( 'admin-ajax.php?action=tx-get-started&step=' ),
			'steps'    => self::get_steps()
		) );

		$tx_plugins_manager = TX_Plugins_Manager::instance();
		$tx_plugins_manager->localize_script();

		$tx_importer = TX_Importer::instance();
		$tx_importer->localize_script();
	}

	/**
	 * Handle ajax.
	 */
	public function handle_ajax() {
		$step = ! empty( $_REQUEST['step'] ) ? $_REQUEST['step'] : false;

		switch ( $step ) {
			case 'quick-setup':
				$this->handle_quick_setup();
				break;

			case 'finish':
				$this->handle_finish();
				break;

			default:
				break;
		}

		wp_die();
	}

	/**
	 * Handle finish getting started.
	 */
	private function handle_finish() {
		TX_Admin_Settings::set( 'getting_started_visited', true );
		wp_send_json_success();
	}

	/**
	 * Handle quick setup.
	 */
	private function handle_quick_setup() {
		$blog_name = isset( $_POST['blogname'] ) ? esc_html( $_POST['blogname'] ) : false;
		if ( $blog_name !== false ) {
			update_option( 'blogname', $blog_name );
		}

		$blog_description = isset( $_POST['blogdescription'] ) ? esc_html( $_POST['blogdescription'] ) : false;
		if ( $blog_description !== false ) {
			update_option( 'blogdescription', $blog_description );
		}

		wp_send_json_success( __( 'Saving successful!', 'edumodo' ) );
	}

	/**
	 * Render step templates.
	 */
	public function render_step_templates() {
		$steps = self::get_steps();

		foreach ( $steps as $index => $step ) {
			$key = strtolower( $step['key'] );
			$this->render_step_template( $key );
		}
	}

	/**
	 * Get step template by slug.
	 *
	 * @param string $slug
	 *
	 * @return bool
	 */
	public function render_step_template( $slug ) {
		$html = TX_Template_Helper::template( 'dashboard/gs-steps/' . $slug . '.php' );

		return TX_Template_Helper::template( 'dashboard/gs-steps/master.php', array( 'slug' => $slug, 'html' => $html ), true );
	}
}
