import apiFetch from '@wordpress/api-fetch';
import {
	Button,
	__experimentalInputControl as InputControl,
	SelectControl,
} from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import React from 'react';
import { Controller, FormProvider, useForm } from 'react-hook-form';
import { useMutation } from 'react-query';
import { z } from 'zod';
import { localized } from '../../../utils/global';
import { COUNTRY_CODES } from '../constant/country-code';

const billingFormSchema = z.object({
	business_name: z.string().optional(),
	tax_id: z.string().optional(),
	address_street: z.string().optional(),
	address_apt: z.string().optional(),
	address_city: z.string().optional(),
	address_zip: z.string().optional(),
	address_country_code: z.string().optional(),
	address_state: z.string().optional(),
});

type BillingFormSchema = z.infer<typeof billingFormSchema>;
export const Billing = () => {
	const form = useForm<BillingFormSchema>({
		defaultValues: Object.entries(localized.fs?.billing?.details).reduce(
			(acc: Partial<BillingFormSchema>, [k, v]) => {
				if (k in billingFormSchema.shape) {
					acc[k as keyof BillingFormSchema] = v as string | undefined;
				}
				return acc;
			},
			{},
		),
	});

	const billingMutation = useMutation((data: BillingFormSchema) => {
		const formData = new FormData();
		formData.append('action', localized.fs?.billing?.action);
		formData.append('security', localized.fs?.billing?.security);
		formData.append('module_id', localized.fs?.module_id);
		for (let key in data) {
			formData.append(
				`billing[${key}]`,
				data[key as keyof BillingFormSchema] ?? '',
			);
		}
		return new Promise((resolve, reject) => {
			apiFetch({
				url: localized.fs?.ajax_url,
				method: 'POST',
				body: formData,
			})
				.then((res: any) => {
					if (!res.success) {
						reject(new Error(res.error));
					} else {
						resolve(res);
					}
				})
				.catch((e) => {
					reject(e);
				});
		});
	}, {});
	const onSubmit = (data: BillingFormSchema) => {
		billingMutation.mutate(data);
	};
	return (
		<div>
			<h2>{__('Billing Details', 'colormag')}</h2>
			<div>
				<FormProvider {...form}>
					<form onSubmit={form.handleSubmit(onSubmit)}>
						<div className="flex gap-4 w-full">
							<Controller
								name={'business_name'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('Business Name', 'colormag')}
										onChange={field.onChange}
										value={field.value}
									/>
								)}
							/>
							<Controller
								name={'tax_id'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('Tax/ Vat ID:', 'colormag')}
										onChange={field.onChange}
										value={field.value}
									/>
								)}
							/>
						</div>
						<div className="flex gap-4 w-full mt-6">
							<Controller
								name={'address_street'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('Address Line 1:', 'colormag')}
										onChange={field.onChange}
										value={field.value}
									/>
								)}
							/>
							<Controller
								name={'address_apt'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('Address Line 2:', 'colormag')}
										onChange={field.onChange}
										value={field.value}
									/>
								)}
							/>
						</div>
						<div className="flex gap-4 w-full mt-6">
							<Controller
								name={'address_city'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('City/Town:', 'colormag')}
										onChange={field.onChange}
										value={field.value}
									/>
								)}
							/>
							<Controller
								name={'address_zip'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('Zip/Postal Code:', 'colormag')}
										onChange={field.onChange}
										value={field.value}
									/>
								)}
							/>
						</div>
						<div className="flex gap-4 w-full mt-6 cm-country">
							<Controller
								name={'address_country_code'}
								control={form.control}
								render={({ field }) => (
									<SelectControl
										__nextHasNoMarginBottom
										className="w-full"
										label={__('Country:', 'colormag')}
										onChange={field.onChange}
										value={field.value}
										options={[
											{
												disabled: true,
												label: 'Select an Option',
												value: '',
											},
											...(COUNTRY_CODES
												? Object.entries(COUNTRY_CODES).map(
														([code, name]: [string, string]) => {
															return {
																label: name,
																value: code,
															};
														},
													)
												: []),
										]}
									/>
								)}
							/>
							<Controller
								name={'address_state'}
								control={form.control}
								render={({ field }) => (
									<InputControl
										className="w-full"
										label={__('State/Province:', 'colormag')}
										value={field.value}
										onChange={field.onChange}
									/>
								)}
							/>
						</div>
						<Button
							type="submit"
							className="mt-4 bg-[#2563EB] text-[#FFFFFF] rounded-es-sm py-2 px-4 hover:text-[#FFFFFF]"
						>
							{__('Update', 'colormag')}
						</Button>
					</form>
				</FormProvider>
			</div>
		</div>
	);
};
