<?php
/**
* Authenticate users via social networks.
*
* Ref: http://miled.github.io/wordpress-social-login/developer-api-authentication.html
**
* Side note: I don't usually over-comment codes, but this is the main script and I had to since
*            many users with diffrent "skill levels" may want to understand how this piece of code works.
**
* To sum things up, here is how it works (bit hard to explain, so bare with me):
*
* Let assume a user come to page at our website and he click on of the providers icons in order connect.
*
*  - If &action=pgssl_authenticate is found in the current url, then PGSSL will display a loading screen,
*  - That loading screen will refresh it self adding &redirect_to_provider=ture to the url, which will trigger the next step,
*  - Next, PGSSL will instantiate Hybridauth main class, build the required provider config then initiate the auth protocol /hybridauth/?hauth.start=PROVIDER_ID,
*  - Hybridauth will redirect the user to the selected provider site to ask for his consent (authorisation to access his profile),
*  - If the user gives his authorisation for your application, the provider will redirect the user back to Hybridauth entry point /hybridauth/?hauth.done=PROVIDER_ID,
*  - Hybridauth will redirect the user to the given callback url.
*  - In that callback url, PGSSL will display a second loading screen This loading screen will generate and submit a form with a hidden input &action= pgssl_authenticated to the current url which will trigger the second part of the auth process,
*  - PGSSL will grab the user profile from the provider, attempt to identify him and create a new WordPress user if he doesn't exist. In this step, and when enabled, PGSSL will also import the user contacts and map his profile data to Buddypress xporfiles tables,
*  - Finally, PGSSL will authenticate the user within WordPress (give him a sweet cookie) and redirect him back to Redirect URL
**
* Functions execution order is the following:
*
*     do_action('init')
*     .       pgssl_process_login()
*     .       .       pgssl_process_login_begin()
*     .       .       .       pgssl_render_redirect_to_provider_loading_screen()
*     .       .       .       Hybridauth\Hybridauth::authenticate()
*     .       .       .       pgssl_render_return_from_provider_loading_screen()
*     .       .
*     .       .       pgssl_process_login_end()
*     .       .       .       pgssl_process_login_get_user_data()
*     .       .       .       .       pgssl_process_login_request_user_social_profile()
*     .       .       .       .       .       Hybridauth\Hybridauth::getUserProfile()
*     .       .       .       .
*     .       .       .       .       pgssl_process_login_complete_registration()
*     .       .       .
*     .       .       .       pgssl_process_login_create_wp_user()
*     .       .       .
*     .       .       .       pgssl_process_login_update_pgssl_user_data()
*     .       .       .       .       pgssl_store_hybridauth_user_profile()
*     .       .       .       .       pgssl_buddypress_xprofile_mapping()
*     .       .       .       .       pgssl_store_hybridauth_user_contacts()
*     .       .       .
*     .       .       .       pgssl_process_login_authenticate_wp_user()
*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
* Entry point to the authentication process
*
* This function runs after WordPress has finished loading but before any headers are sent.
* This function will analyse the current URL parameters and start the login process whenever an
* action is found: $_REQUEST['action'] eq pgssl_*
*
* Example of valid origin url:
*    wp-login.php
*       ?action=pgssl_authenticate                        // current step
*       &mode=login                                                  // auth mode
*       &provider=Twitter                                            // selected provider
*       &redirect_to=http%3A%2F%2Fexample.com%2Fwordpress%2F%3Fp%3D1 // where the user come from
*
* Ref: http://codex.wordpress.org/Plugin_API/Action_Reference/init
*/
function pgssl_process_login() {
	// > check for actions
	$action = isset( $_REQUEST['action'] ) ? $_REQUEST['action'] : null;

	if( ! in_array( $action, array( "pgssl_authenticate", "pgssl_profile_completion", "pgssl_account_linking", "pgssl_authenticated" ) ) ) {
		return false;
	}

	require_once PGS_SOCIAL_LOGIN_ABS_PATH . 'hybridauth/library/src/autoload.php';

	// authentication mode
	$auth_mode = pgssl_process_login_get_auth_mode();

	// start loggin the auth process, if debug mode is enabled
	pgssl_watchdog_init();

	// halt, if mode login and user already logged in
	if( 'login' == $auth_mode && is_user_logged_in() ) {
		$current_user = wp_get_current_user();

		return pgssl_process_login_render_notice_page( sprintf( __( "You are already logged in as %s. Do you want to <a href='%s'>log out</a>?", 'pgs-core' ), $current_user->display_name, wp_logout_url( home_url() ) ) );
	}

	// halt, if mode link and user not logged in
	if( 'link' == $auth_mode && ! is_user_logged_in() ) {
		return pgssl_process_login_render_notice_page( sprintf( __( "You have to be logged in to be able to link your existing account. Do you want to <a href='%s'>login</a>?", 'pgs-core' ), wp_login_url( home_url() ) ) );
	}

	// halt, if mode test and not admin
	if( 'test' == $auth_mode && ! current_user_can('manage_options') ) {
		return pgssl_process_login_render_notice_page( __( 'You do not have sufficient permissions to access this page.', 'pgs-core' ) );
	}

	// Bouncer :: Allow authentication?
	if( pgssl_social_authentication_enabled() == 2 ) {
		return pgssl_process_login_render_notice_page( __( "Authentication through social networks is currently disabled.", 'pgs-core' ) );
	}

	add_action( 'pgssl_clear_user_php_session', 'pgssl_process_login_clear_user_php_session' );

	// HOOKABLE:
	do_action( "pgssl_process_login_start" );

	// if action=pgssl_authenticate
	// > start the first part of authentication (redirect the user to the selected provider)
	if( $action == "pgssl_authenticate" ) {
		return pgssl_process_login_begin();
	}

	// if action=pgssl_authenticated or action=pgssl_profile_completion
	// > finish the authentication process (create new user if doesn't exist in database, then log him in within wordpress)
	pgssl_process_login_end();
}

add_action( 'init', 'pgssl_process_login' );

/**
* Start the first part of authentication
*
* Steps:
*     1. Display a loading screen while hybridauth is redirecting the user to the selected provider
*     2. Build the hybridauth config for the selected provider (keys, scope, etc)
*     3. Instantiate the class Hybridauth\Hybridauth and redirect the user to provider to ask for authorisation for this website
*     4. Display a loading screen after user come back from provider as we redirect the user back to Widget::Redirect URL
*/
function pgssl_process_login_begin() {
	// HOOKABLE:
	do_action( "pgssl_process_login_begin_start" );

	$config     = null;
	$hybridauth = null;
	$provider   = null;
	$adapter    = null;

	// check if php session are working as expected
	if( ! pgssl_process_login_check_php_session() ) {
		return pgssl_process_login_render_notice_page( sprintf( __( 'The session identifier is missing.', 'pgs-core' ), home_url() ) );
	}

	// HOOKABLE: selected provider name
	$provider = pgssl_process_login_get_selected_provider();

	if( ! $provider ) {
		return pgssl_process_login_render_notice_page( __( 'Bouncer says this makes no sense.', 'pgs-core' ) );
	}

	/* 1. Display a loading screen while hybridauth is redirecting the user to the selected provider */

	// the loading screen should refresh it self with a new arg in url: &redirect_to_provider=true
	if( ! isset( $_REQUEST["redirect_to_provider"] ) ) {
		do_action( 'pgssl_clear_user_php_session' );

		return pgssl_render_redirect_to_provider_loading_screen( $provider );
	}

	/*  2. Build the hybridauth config for the selected provider (keys, scope, etc) */

	// provider enabled?
	if( ! get_option( 'pgssl_settings_' . $provider . '_enabled' ) ) {
		return pgssl_process_login_render_notice_page( __( "Unknown or disabled provider.", 'pgs-core' ) );
	}

	$config = pgssl_process_login_build_provider_config( $provider );

	/* 3. Instantiate the class Hybridauth and redirect the user to provider to ask for authorisation for this website */

	// HOOKABLE:
	do_action( "pgssl_hook_process_login_before_hybridauth_authenticate", $provider, $config );

	try
	{
		// create an instance oh hybridauth with the generated config
		$hybridauth = new Hybridauth\Hybridauth( $config );

		// start the authentication process via hybridauth
		// > if not already connected hybridauth::authenticate() will redirect the user to the provider
		// > where he will be asked for his consent (most providers ask for consent only once).
		// > after that, the provider will redirect the user back to this same page (and this same line).
		// > if the user is successfully connected to provider, then this time hybridauth::authenticate()
		// > will just return the provider adapter
		pgssl_set_provider_config_in_session_storage( $provider, $config );

		$adapter = $hybridauth->authenticate( $provider );
	}

	// if hybridauth fails to authenticate the user, then we display an error message
	catch( Exception $e ) {
		return pgssl_process_login_render_error_page( $e, $config, $provider );
	}

	// HOOKABLE:
	do_action( "pgssl_hook_process_login_after_hybridauth_authenticate", $provider, $config, $hybridauth, $adapter );

	/* 4. Display a loading screen after user come back from provider as we redirect the user back to Widget::Redirect URL */

	// get Widget::Authentication display
	$pgssl_authentication_display = pgssl_authentication_display();

	// authentication mode
	$auth_mode = pgssl_process_login_get_auth_mode();

	$redirect_to = isset( $_REQUEST[ 'redirect_to' ] ) ? $_REQUEST[ 'redirect_to' ] : home_url();

	// build the authenticateD, which will make pgssl_process_login() fire the next step pgssl_process_login_end()
	$authenticated_url = site_url( 'wp-login.php', 'login_post' ) . ( strpos( site_url( 'wp-login.php', 'login_post' ), '?' ) ? '&' : '?' ) . "action=pgssl_authenticated&provider=" . $provider . '&mode=' . $auth_mode;

	// display a loading screen
	return pgssl_render_return_from_provider_loading_screen( $provider, $authenticated_url, $redirect_to, $pgssl_authentication_display );
}

/**
* Finish the authentication process
*
* Steps:
*     1. Get the user profile from provider
*     2. Create new wordpress user if he didn't exist in database
*     3. Store his Hybridauth profile, contacts and BP mapping
*     4. Authenticate the user within wordpress
*/
function pgssl_process_login_end() {
	// HOOKABLE:
	do_action( "pgssl_process_login_end_start" );

	// HOOKABLE: set a custom Redirect URL
	$redirect_to = pgssl_process_login_get_redirect_to();

	// HOOKABLE: selected provider name
	$provider = pgssl_process_login_get_selected_provider();

	// authentication mode
	$auth_mode = pgssl_process_login_get_auth_mode();

	$is_new_user             = false; // is it a new or returning user
	$user_id                 = ''   ; // wp user id
	$adapter                 = ''   ; // hybriauth adapter for the selected provider
	$hybridauth_user_profile = ''   ; // hybriauth user profile
	$requested_user_login    = ''   ; // username typed by users in Profile Completion
	$requested_user_email    = ''   ; // email typed by users in Profile Completion

	// provider is enabled?
	if( ! get_option( 'pgssl_settings_' . $provider . '_enabled' ) ) {
		return pgssl_process_login_render_notice_page( __( "Unknown or disabled provider.", 'pgs-core' ) );
	}

	if( 'test' == $auth_mode ) {
		$redirect_to = admin_url( 'admin.php?page=pgssl_settings&pgssl_page=auth-paly&provider=' . $provider );

		return wp_safe_redirect( $redirect_to );
	}

	if( 'link' == $auth_mode ) {
		// a social account cant be associated with more than one wordpress account.

		$hybridauth_user_profile = pgssl_process_login_request_user_social_profile( $provider );

        $adapter = pgssl_process_login_get_provider_adapter( $provider );

		$user_id = (int) pgssl_get_stored_hybridauth_user_id_by_provider_and_provider_uid( $provider, $hybridauth_user_profile->identifier );

		if( $user_id && $user_id != get_current_user_id() ) {
			return pgssl_process_login_render_notice_page( sprintf( __( "Your <b>%s ID</b> is already linked to another account on this website.", 'pgs-core'), $provider ) );
		}

		$user_id = get_current_user_id();

		// doesn't hurt to double check
		if( ! $user_id ) {
			return pgssl_process_login_render_notice_page( __( "Sorry, we couldn't link your account.", 'pgs-core' ) );
		}
	} elseif( 'login' != $auth_mode ) {
		return pgssl_process_login_render_notice_page( __( 'Bouncer says no.', 'pgs-core' ) );
	}

	if( 'login' == $auth_mode ) {
		// returns user data after he authenticate via hybridauth
		list
		(
			$user_id                ,
			$adapter                ,
			$hybridauth_user_profile,
			$requested_user_login   ,
			$requested_user_email   ,
			$wordpress_user_id
		)
		= pgssl_process_login_get_user_data( $provider, $redirect_to );

		// if no associated user were found in pgssl_usersprofiles, create new WordPress user
		if( ! $wordpress_user_id ) {
			$user_id = pgssl_process_login_create_wp_user( $provider, $hybridauth_user_profile, $requested_user_login, $requested_user_email );

			$is_new_user = true;
			$redirect_to = apply_filters('pgssl_redirect_after_registration', $redirect_to);
		}else{
			$user_id = $wordpress_user_id;
			$is_new_user = false;
		}
	}

	// if user is found in pgssl_usersprofiles but the associated WP user account no longer exist
	// > this should never happen! but just in case: we delete the user pgssl_usersprofiles/pgssl_userscontacts entries and we reset the process
	$wp_user = get_userdata( $user_id );

	if( ! $wp_user ) {
		pgssl_delete_stored_hybridauth_user_data( $user_id );

		return pgssl_process_login_render_notice_page( sprintf( __( "Sorry, we couldn't connect you. <a href=\"%s\">Please try again</a>.", 'pgs-core' ), site_url( 'wp-login.php', 'login_post' ) ) );
	}

	// store user hybridauth profile (pgssl_usersprofiles), contacts (pgssl_userscontacts) and buddypress mapping
	pgssl_process_login_update_pgssl_user_data( $is_new_user, $user_id, $provider, $adapter, $hybridauth_user_profile, $wp_user );

	// finally create a wordpress session for the user
	pgssl_process_login_authenticate_wp_user( $user_id, $provider, $redirect_to, $adapter, $hybridauth_user_profile, $wp_user );
}

/**
* Returns user data after he authenticate via hybridauth
*
* Steps:
*    1. Grab the user profile from hybridauth
*    2. Run Bouncer::Filters if enabled (domains, emails, profiles urls)
*    3. Check if user exist in database by looking for the couple (Provider name, Provider user ID) or verified email
*    4. Deletegate detection of user id to custom functions / hooks
*    5. If Bouncer::Profile Completion is enabled and user didn't exist, we require the user to complete the registration (user name & email)
*/
function pgssl_process_login_get_user_data( $provider, $redirect_to ) {
	// HOOKABLE:
	do_action( "pgssl_process_login_get_user_data_start", $provider, $redirect_to );

	$user_id                  = null;
	$config                   = null;
	$hybridauth               = null;
	$adapter                  = null;
	$hybridauth_user_profile  = null;
	$requested_user_login     = '';
	$requested_user_email     = '';
	$wordpress_user_id        = 0;

	/* 1. Grab the user profile from social network */

	if( ! ( isset( $_SESSION['pgssl::userprofile'] ) && $_SESSION['pgssl::userprofile'] && $hybridauth_user_profile = json_decode( $_SESSION['pgssl::userprofile'] ) ) ) {
		$hybridauth_user_profile = pgssl_process_login_request_user_social_profile( $provider );

		$_SESSION['pgssl::userprofile'] = json_encode( $hybridauth_user_profile );
	}

	$adapter = pgssl_process_login_get_provider_adapter( $provider );

	$hybridauth_user_email          = sanitize_email( $hybridauth_user_profile->email );
	$hybridauth_user_email_verified = sanitize_email( $hybridauth_user_profile->emailVerified );

	/* 2. Run Bouncer::Filters if enabled (domains, emails, profiles urls) */

	// Bouncer::Filters by emails domains name
	// Bouncer::Filters by e-mails addresses
	// Bouncer::Filters by profile urls

	/* 3. Check if user exist in database by looking for the couple (Provider name, Provider user ID) or verified email */

	// check if user already exist in pgssl_usersprofiles
	$user_id = (int) pgssl_get_stored_hybridauth_user_id_by_provider_and_provider_uid( $provider, $hybridauth_user_profile->identifier );

	// if not found in pgssl_usersprofiles, then check his verified email
	if( ! $user_id && ! empty( $hybridauth_user_email_verified ) ) {
		// check if the verified email exist in wp_users
		$user_id = (int) pgssl_wp_email_exists( $hybridauth_user_email_verified );

		// check if the verified email exist in pgssl_usersprofiles
		if( ! $user_id ) {
			$user_id = (int) pgssl_get_stored_hybridauth_user_id_by_email_verified( $hybridauth_user_email_verified );
		}

		// if the user exists in Wordpress
		if( $user_id ) {
			$wordpress_user_id = $user_id;
		}
	}

	/* 4 Deletegate detection of user id to custom filters hooks */

	// HOOKABLE:
	$user_id = apply_filters( 'pgssl_hook_process_login_alter_user_id', $user_id, $provider, $hybridauth_user_profile );

	/* 5. If Bouncer::Profile Completion is enabled and user didn't exist, we require the user to complete the registration (user name & email) */
	if( ! $user_id ) {
		// Accept new registrations?
		if( ! get_option( 'users_can_register' ) ) {
			return pgssl_process_login_render_notice_page( __( "Registration is now closed.", 'pgs-core' ) );
		}

		// Accounts linking/mapping
		// > > not implemented yet!
		if( pgssl_accounts_linking_enabled() == 1 ) {
			do
			{
				list
				(
					$shall_pass,
					$user_id,
					$requested_user_login,
					$requested_user_email
				)
				= pgssl_process_login_new_users_gateway( $provider, $redirect_to, $hybridauth_user_profile );
			}
			while( ! $shall_pass );
			$wordpress_user_id = $user_id;
		}

		// Bouncer::Profile Completion
		elseif( ( pgssl_social_profile_completion_require_email() == 1 && empty( $hybridauth_user_email ) )
			|| pgssl_social_profile_completion_change_username() == 1 ) {
			do
			{
				list
				(
					$shall_pass,
					$user_id,
					$requested_user_login,
					$requested_user_email
				)
				= pgssl_process_login_new_users_gateway( $provider, $redirect_to, $hybridauth_user_profile );
			}
			while( ! $shall_pass );
		}

	}else{
		$wordpress_user_id = $user_id;
	}

	/* 6. returns user data */

	return array(
		$user_id,
		$adapter,
		$hybridauth_user_profile,
		$requested_user_login,
		$requested_user_email,
		$wordpress_user_id
	);
}

/**
* Create a new wordpress user
*
* Ref: http://codex.wordpress.org/Function_Reference/wp_insert_user
*/
function pgssl_process_login_create_wp_user( $provider, $hybridauth_user_profile, $requested_user_login, $requested_user_email ) {
	// HOOKABLE:
	do_action( "pgssl_process_login_create_wp_user_start", $provider, $hybridauth_user_profile, $requested_user_login, $requested_user_email );

	$user_login = '';
	$user_email = '';

	// if coming from "complete registration form"
	if( $requested_user_login ) {
		$user_login = $requested_user_login;
	}

	if( $requested_user_email ) {
		$user_email = $requested_user_email;
	}

	if( ! $user_login ) {
		// attempt to generate user_login from hybridauth user profile display name
		$user_login = $hybridauth_user_profile->displayName;

		// sanitize user login
		$user_login = sanitize_user( $user_login, true );

		// remove spaces and dots
		$user_login = trim( str_replace( array( ' ', '.' ), '_', $user_login ) );
		$user_login = trim( str_replace( '__', '_', $user_login ) );

		// if user profile display name is not provided
		if( empty( $user_login ) ) {
			// may be that $user_email is empty then we got wp error login can't be empty, so check it now
			if ( $user_email ) {
				$user_login = sanitize_user( current( explode( '@', $user_email ) ), true );
			} else {
				$user_login = sanitize_user( current( explode( '@', $hybridauth_user_profile->email ) ), true );
			}
		}
	}

	// user name should be unique
	if( username_exists( $user_login ) ) {
		$i = 1;
		$user_login_tmp = $user_login;

		do
		{
			$user_login_tmp = $user_login . "_" . ($i++);
		}
		while( username_exists ($user_login_tmp));

		$user_login = $user_login_tmp;
	}

	if( ! $user_email ) {
		$user_email = $hybridauth_user_profile->email;

		// generate an email if none
		if( ! isset ( $user_email ) OR ! is_email( $user_email ) ) {
			$user_email = strtolower( $provider . "_user_" . $user_login ) . '@example.com';
		}

		// email should be unique
		if( pgssl_wp_email_exists ( $user_email ) ) {
			do
			{
				$user_email = md5( uniqid( wp_rand( 10000, 99000 ) ) ) . '@example.com';
			}
			while( pgssl_wp_email_exists( $user_email ) );
		}
	}

	$display_name = $hybridauth_user_profile->displayName;

	if( empty( $display_name ) ) {
		$display_name = $hybridauth_user_profile->firstName;
	}

	if( empty( $display_name ) ) {
		$display_name = strtolower( $provider ) . "_user";
	}

	$userdata = array(
		'user_login'    => $user_login,
		'user_email'    => $user_email,

		'display_name'  => $display_name,

		'first_name'    => $hybridauth_user_profile->firstName,
		'last_name'     => $hybridauth_user_profile->lastName,
		'user_url'      => $hybridauth_user_profile->profileURL,
		'description'   => $hybridauth_user_profile->description,

		'user_pass'     => wp_generate_password(18)
	);

	// Bouncer::Membership level
	$userdata['role'] = get_option('default_role');

	// Bouncer::User Moderation

	// HOOKABLE: change the user data
	$userdata = apply_filters( 'pgssl_hook_process_login_alter_wp_insert_user_data', $userdata, $provider, $hybridauth_user_profile );

	// HOOKABLE: This action runs just before creating a new wordpress user.
	do_action( 'pgssl_hook_process_login_before_wp_insert_user', $userdata, $provider, $hybridauth_user_profile );

	// DEPRECIATED: as of 2.2.3
	// do_action( 'pgssl_hook_process_login_before_insert_user', $userdata, $provider, $hybridauth_user_profile );

	// HOOKABLE: This action runs just before creating a new wordpress user, it delegate user insert to a custom function.
	$user_id = apply_filters( 'pgssl_hook_process_login_delegate_wp_insert_user', $userdata, $provider, $hybridauth_user_profile );

	// echo '<pre>';
	// print_r($userdata);
	// echo '</pre>';
	// exit;

	// Create a new WordPress user
	if( ! $user_id || ! is_integer( $user_id ) ) {
		$user_id = wp_insert_user( $userdata );

		//On success
		if ( ! is_wp_error( $user_id ) ) {

			$user = get_userdata( $user_id );

			/* translators: %s: user login */
			$message  = sprintf( __( 'Username: %s', 'pgs-core' ), $user->user_login ) . "\r\n";

			/* translators: %s: user password */
			$message  .= sprintf( __( 'Password: %s', 'pgs-core' ), $userdata['user_pass'] ) . "\r\n\r\n";

			$message .= wp_login_url() . "\r\n";

			// The blogname option is escaped with esc_html on the way into the database in sanitize_option
			// we want to reverse this for the plain text arena of emails.
			$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );

			wp_mail(
				$user->user_email,                                                                            // User Email
				wp_specialchars_decode( sprintf( __( '[%s] Your username and password info', 'pgs-core' ), $blogname ) ), // Subject
				$message,                                                                                     // Message
				''                                                                                            // Headers
			);
		}
	}

	// do not continue without user_id
	if( ! $user_id || ! is_integer( $user_id ) ) {
		if( is_wp_error( $user_id ) ) {
			return pgssl_process_login_render_notice_page( __( "An error occurred while creating a new user: ", 'pgs-core' ) . $user_id->get_error_message() );
		}

		return pgssl_process_login_render_notice_page( __( "An error occurred while creating a new user!", 'pgs-core' ) );
	}

	// wp_insert_user may fail on first and last name meta, expliciting setting to correct.
	update_user_meta($user_id, 'first_name', apply_filters( 'pre_user_first_name',$userdata['first_name']));
	update_user_meta($user_id, 'last_name', apply_filters( 'pre_user_last_name', $userdata['last_name']));

	// Send notifications
	if( pgssl_users_social_login_notification() == 1 ) {
		pgssl_admin_notification( $user_id, $provider );
	}

	// HOOKABLE: This action runs just after a wordpress user has been created
	// > Note: At this point, the user has been added to wordpress database, but NOT CONNECTED.
	do_action( 'pgssl_hook_process_login_after_wp_insert_user', $user_id, $provider, $hybridauth_user_profile );

	// DEPRECIATED: as of 2.2.3
	// do_action( 'pgssl_hook_process_login_after_create_wp_user', $user_id, $provider, $hybridauth_user_profile );

	// returns the user created user id
	return $user_id;
}

/**
* Store user data
*
* Steps:
*     1. Store Hybridauth user profile
*     2. Import user contacts
*     3. Launch BuddyPress Profile mapping
*/
function pgssl_process_login_update_pgssl_user_data( $is_new_user, $user_id, $provider, $adapter, $hybridauth_user_profile, $wp_user ) {
	// HOOKABLE:
	do_action( "pgssl_process_login_update_pgssl_user_data_start", $is_new_user, $user_id, $provider, $adapter, $hybridauth_user_profile, $wp_user );

	// store user hybridauth user profile in table pgssl_usersprofiles
	// > only sotre the user profile if it has changed since last login.
	pgssl_store_hybridauth_user_profile( $user_id, $provider, $hybridauth_user_profile );

	// map hybridauth user profile to buddypress xprofile table, if enabled
	// > Profile mapping will only work with new users..
	if( $is_new_user ) {
		pgssl_buddypress_xprofile_mapping( $user_id, $provider, $hybridauth_user_profile );
	}

	// import user contacts into pgssl_usersprofiles, if enabled
	// > only import the contacts list once per user per provider.
	pgssl_store_hybridauth_user_contacts( $user_id, $provider, $adapter );
}

/**
* Authenticate a user within wordpress
*
* Ref: http://codex.wordpress.org/Function_Reference/wp_set_auth_cookie
* Ref: http://codex.wordpress.org/Function_Reference/wp_safe_redirect
*/
function pgssl_process_login_authenticate_wp_user( $user_id, $provider, $redirect_to, $adapter, $hybridauth_user_profile, $wp_user ) {
	// HOOKABLE:
	do_action( "pgssl_process_login_authenticate_wp_user_start", $user_id, $provider, $redirect_to, $adapter, $hybridauth_user_profile, $wp_user );

	// update some fields in usermeta for the current user
	update_user_meta( $user_id, 'pgssl_current_provider', $provider );

	if(  $hybridauth_user_profile->photoURL ) {
		update_user_meta( $user_id, 'pgssl_current_user_image', $hybridauth_user_profile->photoURL );
	}

	// Bouncer::User Moderation

	// current user role
	$role = current( $wp_user->roles );

	// HOOKABLE: This action runs just before logging the user in (before creating a WP cookie)
	do_action( "pgssl_hook_process_login_before_wp_set_auth_cookie", $user_id, $provider, $hybridauth_user_profile );

	// DEPRECIATED: as of 2.2.3
	// do_action( 'pgssl_hook_process_login_before_set_auth_cookie', $user_id, $provider, $hybridauth_user_profile );

	// Set WP auth cookie
	wp_set_auth_cookie( $user_id, true );

	// let keep it std
	do_action( 'wp_login', $wp_user->user_login, $wp_user );

	// HOOKABLE: This action runs just before redirecting the user back to $redirect_to
	// > Note: If you have enabled User Moderation, then the user is NOT NECESSARILY CONNECTED
	// > within wordpress at this point (in case the user $role == 'pending').
	// > To be sure the user is connected, use pgssl_hook_process_login_before_wp_set_auth_cookie instead.
	do_action( "pgssl_hook_process_login_before_wp_safe_redirect", $user_id, $provider, $hybridauth_user_profile, $redirect_to );

	// DEPRECIATED: as of 2.2.3
	// do_action( 'pgssl_hook_process_login_before_set_auth_cookie', $user_id, $provider, $hybridauth_user_profile );

	do_action( 'pgssl_clear_user_php_session' );

	// Display debugging instead of redirecting the user
	// > this will give a complete report : database queries and fired hooks
	// pgssl_display_dev_mode_debugging_area(); die(); // ! keep this line commented unless you know what you are doing :)

	// That's it. We done.
	wp_safe_redirect( $redirect_to );

	// for good measures
	die();
}

/**
*  Build required hybridauth configuration for the given provider
*/
function pgssl_process_login_build_provider_config( $provider ) {
	require_once PGS_SOCIAL_LOGIN_ABS_PATH . 'hybridauth/library/src/autoload.php';

	$config = array();
	$config["current_page"] = Hybridauth\HttpClient\Util::getCurrentUrl(true);
	$config["callback"] = PGS_SOCIAL_LOGIN_HYBRIDAUTH_ENDPOINT_URL . 'callbacks/' . strtolower( $provider ) . '.php';
	$config["providers"] = array();
	$config["providers"][$provider] = array();
	$config["providers"][$provider]["enabled"] = true;
	$config["providers"][$provider]["keys"] = array( 'id' => null, 'key' => null, 'secret' => null );

	// provider application id ?
	if( get_option( 'pgssl_settings_' . $provider . '_app_id' ) ) {
		$config["providers"][$provider]["keys"]["id"] = get_option( 'pgssl_settings_' . $provider . '_app_id' );
	}

	// provider application key ?
	if( get_option( 'pgssl_settings_' . $provider . '_app_key' ) ) {
		$config["providers"][$provider]["keys"]["key"] = get_option( 'pgssl_settings_' . $provider . '_app_key' );
	}

	// provider application secret ?
	if( get_option( 'pgssl_settings_' . $provider . '_app_secret' ) ) {
		$config["providers"][$provider]["keys"]["secret"] = get_option( 'pgssl_settings_' . $provider . '_app_secret' );
	}

	// set custom config for facebook
	if( strtolower( $provider ) == "facebook" ) {
		$config["providers"][$provider]["display"] = "popup";
		$config["providers"][$provider]["trustForwarded"] = true;

		// switch to fb::display 'page' if auth in page
		if( pgssl_authentication_display() == 2 ) {
			$config["providers"][$provider]["display"] = "page";
		}

		$config["providers"][$provider]["scope"] = "email, public_profile";
	}

	// set custom config for google
	if( strtolower( $provider ) == "google" ) {
		$config["providers"][$provider]["scope"] = "https://www.googleapis.com/auth/userinfo.profile https://www.googleapis.com/auth/userinfo.email";
	}

	$provider_scope = isset( $config["providers"][$provider]["scope"] ) ? $config["providers"][$provider]["scope"] : '' ;

	// HOOKABLE: allow to overwrite scopes
	$config["providers"][$provider]["scope"] = apply_filters( 'pgssl_hook_alter_provider_scope', $provider_scope, $provider );

	// HOOKABLE: allow to overwrite hybridauth config for the selected provider
	$config["providers"][$provider] = apply_filters( 'pgssl_hook_alter_provider_config', $config["providers"][$provider], $provider );

	return $config;
}

/**
*  Grab the user profile from social network
*/
function pgssl_process_login_request_user_social_profile( $provider ) {
	$adapter                 = null;
	$config                  = null;
	$hybridauth_user_profile = null;

	try
	{
		// get idp adapter
		$adapter = pgssl_process_login_get_provider_adapter( $provider );

		$config = pgssl_get_provider_config_from_session_storage( $provider );

		// if user authenticated successfully with social network
		if( $adapter->isConnected() ) {
			// grab user profile via hybridauth api
			$hybridauth_user_profile = $adapter->getUserProfile();
		}

		// if user not connected to provider (ie: session lost, url forged)
		else
		{
			return pgssl_process_login_render_notice_page( sprintf( __( "Sorry, we couldn't connect you with <b>%s</b>. <a href=\"%s\">Please try again</a>.", 'pgs-core' ), $provider, site_url( 'wp-login.php', 'login_post' ) ) );
		}
	}

	// if things didn't go as expected, we dispay the appropriate error message
	catch( Exception $e ) {
		return pgssl_process_login_render_error_page( $e, $config, $provider, $adapter );
	}

	return $hybridauth_user_profile;
}

/**
* Returns hybriauth idp adapter.
*/
function pgssl_process_login_get_provider_adapter( $provider ) {
	require_once PGS_SOCIAL_LOGIN_ABS_PATH . 'hybridauth/library/src/autoload.php';

	$config = pgssl_get_provider_config_from_session_storage( $provider );

	$hybridauth = new Hybridauth\Hybridauth( $config );

	return $hybridauth->getAdapter( $provider );
}

/**
* Returns redirect_to (callback url)
*
* By default, once a user  authenticate, he will be automatically redirected to the page where he come from (referer).
* If PGSSL wasn't able to identify the referer url (or if the user come wp-login.php), then they will be redirected to
* Widget::Redirect URL instead.
*
* When Widget::Force redirection is set to Yes, users will be always redirected to Widget::Redirect URL.
*
* Note: Widget::Redirect URL can be customised using the filter 'pgssl_hook_process_login_alter_redirect_to'
*/
function pgssl_process_login_get_redirect_to() {
	// force redirection?
	$pgssl_redirect_url = pgssl_redirect_url();

	if( pgssl_force_redirect_url() == 1 ) {
		$redirect_to = apply_filters( 'pgssl_process_login_alter_redirect_url', $pgssl_redirect_url );

		return $redirect_to;
	}

	// get a valid $redirect_to
	if( isset( $_REQUEST[ 'redirect_to' ] ) && $_REQUEST[ 'redirect_to' ] != '' ) {
		$redirect_to = $_REQUEST[ 'redirect_to' ];

		// Redirect to https if user wants ssl
		if( isset( $secure_cookie ) && $secure_cookie && false !== strpos( $redirect_to, 'wp-admin') ) {
			$redirect_to = preg_replace( '|^http://|', 'https://', $redirect_to );
		}

		// we don't go there..
		if( strpos( $redirect_to, 'wp-admin') ) {
			$redirect_to = $pgssl_redirect_url;
		}

		// nor there..
		if( strpos( $redirect_to, 'wp-login.php') ) {
			$redirect_to = $pgssl_redirect_url;
		}
	}

	if( empty( $redirect_to ) ) {
		$redirect_to = $pgssl_redirect_url;
	}

	if( empty( $redirect_to ) ) {
		$redirect_to = home_url();
	}

	$redirect_to = apply_filters( 'pgssl_hook_process_login_alter_redirect_to', $redirect_to );

	return $redirect_to;
}

/**
* Display an error message in case user authentication fails
*/
function pgssl_process_login_render_error_page( $e, $config = null, $provider = null, $adapter = null ) {
	// HOOKABLE:
	do_action( "pgssl_process_login_render_error_page", $e, $config, $provider, $adapter );

	$assets_base_url  = PGS_SOCIAL_LOGIN_PLUGIN_URL . 'assets/img/';

	$message  = "";
	$notes    = "";
	$apierror = substr( $e->getMessage(), 0, 256 );

	if( is_object( $adapter ) ) {
		$adapter->disconnect();
	}

	return pgssl_render_error_page( $message, $notes, $provider, $apierror, $e );
}

/**
* Display an notice message
*/
function pgssl_process_login_render_notice_page( $message ) {
	// HOOKABLE:
	do_action( "pgssl_process_login_render_notice_page", $message );

	return pgssl_render_notice_page( $message );
}

/**
* Returns the selected provider from _REQUEST, default to null
*/
function pgssl_process_login_get_selected_provider() {
	$provider = isset( $_REQUEST["provider"] ) ? sanitize_text_field( $_REQUEST["provider"] ) : null;

	return apply_filters( 'pgssl_hook_process_login_alter_provider', $provider ) ;
}

/**
* Returns the selected auth mode from _REQUEST, default to login
*/
function pgssl_process_login_get_auth_mode() {
	$auth_mode = isset( $_REQUEST["mode"] ) ? sanitize_text_field( $_REQUEST["mode"] ) : 'login';

	return apply_filters( 'pgssl_hook_process_login_alter_auth_mode', $auth_mode ) ;
}

/**
* Clear the stored data n php session
*/
function pgssl_process_login_clear_user_php_session() {
	$_SESSION["HYBRIDAUTH::STORAGE"] = array(); // used by hybridauth library. to clear as soon as the auth process ends.
	$_SESSION["HA::STORE"]           = array(); // used by hybridauth library. to clear as soon as the auth process ends.
	$_SESSION["HA::CONFIG"]          = array(); // used by hybridauth library. to clear as soon as the auth process ends.
	$_SESSION["pgssl::userprofile"]    = array(); // used to temporarily store the user profile so we don't make unnecessary calls to social apis.
}

/**
* Returns IDP actual name
*/
function pgssl_get_provider_name_by_id( $provider_id) {
        $providers_config = pgssl_get_providers();

        foreach( $providers_config as $provider_settings ) {
			if ( $provider_settings['provider_id'] == $provider_id ) {
				return $provider_name = $provider_settings['provider_name'];
			}
        }

        return $provider_id;
}

/**
* Check Php session
*/
function pgssl_process_login_check_php_session() {
	if( isset( $_SESSION["pgssl::plugin"] ) && $_SESSION["pgssl::plugin"] ) {
		return true;
	}
}

/**
 * Returns redirect url for when a new account was created
 */
function pgssl_new_register_redirect_url($redirect_to) {
	return $redirect_to;
}
add_filter("pgssl_redirect_after_registration", "pgssl_new_register_redirect_url", 10, 1);
