<?php
/**
 * WP QUADS plugin functions
 *
 * @package BoomBox_Theme
 * @since 1.0.0
 * @version 2.0.4
 */

// Prevent direct script access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

if ( ! boombox_plugin_management_service()->is_plugin_active( 'quick-adsense-reloaded/quick-adsense-reloaded.php' ) || ! function_exists( 'quads_register_ad' ) ) {
	return;
}

if( ! class_exists( 'Boombox_WP_Quads' ) ) {

	class Boombox_WP_Quads {

		/**
		 * Holds class single instance
		 * @var null
		 */
		private static $_instance = null;

		/**
		 * Get instance
		 * @return Boombox_WP_Quads|null
		 */
		public static function get_instance() {

			if (null == static::$_instance) {
				static::$_instance = new Boombox_WP_Quads();
			}

			return static::$_instance;

		}

		/**
		 * Boombox_WP_Quads constructor.
		 */
		private function __construct() {
			$this->hooks();
			$this->register_ads();
		}

		/**
		 * A dummy magic method to prevent Boombox_WP_Quads from being cloned.
		 *
		 */
		public function __clone() {
			throw new Exception('Cloning ' . __CLASS__ . ' is forbidden');
		}

		/**
		 * Setup Hooks
		 */
		public function hooks() {
			add_action( 'admin_bar_menu', array( $this, 'simulate_the_content_filter' ), 999 );
			add_action( 'boombox/single/sortables/before_navigation', array( $this, 'render_ad_before_single_navigation' ) );
			add_action( 'boombox/single/sortables/after_comments', array( $this, 'render_ad_after_single_comments' ) );
			add_filter( 'boombox/admin/page/meta-boxes/structure', array( $this, 'edit_page_metaboxes_structure' ), 10, 2 );
			add_filter( 'boombox/customizer/fields/archive_main_posts', array( $this, 'add_inject_settings_to_customizer' ), 10, 3 );
			add_filter( 'boombox/customizer/fields/home_main_posts', array( $this, 'add_inject_settings_to_customizer' ), 10, 3 );
			add_filter( 'boombox/customizer_default_values', array( $this, 'edit_customizer_default_values' ), 10, 1 );

			if( boombox_plugin_management_service()->is_plugin_active( 'wp-quads-pro/wp-quads-pro.php' ) ) {
				add_action( 'pre_option_quads_wp_quads_pro_license_active', array( $this, 'simulate_license' ), 10, 1 );
			}
		}

		/**
		 * Simulate theme in build license
		 * @param mixed $value Current value
		 *
		 * @return stdClass
		 */
		public function simulate_license( $value ) {

			$quads_options = get_option( 'quads_settings' );
			$has_pro_key = isset( $quads_options[ 'quads_wp_quads_pro_license_key' ] ) && $quads_options[ 'quads_wp_quads_pro_license_key' ];
			if( ! $has_pro_key ) {
				$value = new stdClass();
				$value->success = true;
				$value->license = 'valid';
				$value->expires = 'lifetime';
			}

			return $value;
		}

		/**
		 * Simulate 'the_content' filter run for pages that does not have it
		 */
		public function simulate_the_content_filter() {

			global $wp_the_query, $the_content;
			if( $wp_the_query->is_singular() ) {
				$the_content = true;
			}

		}

		/**
		 * Get ads locations
		 * @return array
		 */
		private function get_locations() {

			/***** Header */
			$header = array(
				array(
					'location'    => 'boombox-inside-header',
					'description' => esc_html( 'Inside Header' ),
				),
			);

			/***** Index template */
			$home = array(
				array(
					'location'    => 'boombox-index-before-header',
					'description' => esc_html( 'Home: Before Header' ),
				),
				array(
					'location'    => 'boombox-index-before-featured-area',
					'description' => esc_html( 'Home: Before Featured Area' ),
				),
				array(
					'location'    => 'boombox-index-after-featured-area',
					'description' => esc_html( 'Home: After Featured Area' ),
				),
				array(
					'location'    => 'boombox-index-before-content',
					'description' => esc_html( 'Home: Before Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-index-after-content',
					'description' => esc_html( 'Home: After Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-index-before-footer',
					'description' => esc_html( 'Home: Before Footer' ),
				),
			);

			/***** Archive template */
			$archive = array(
				array(
					'location'    => 'boombox-archive-before-header',
					'description' => esc_html( 'Archive: Before Header' ),
				),
				array(
					'location'    => 'boombox-archive-before-featured-area',
					'description' => esc_html( 'Archive: Before Featured Area' ),
				),
				array(
					'location'    => 'boombox-archive-after-featured-area',
					'description' => esc_html( 'Archive: After Featured Area' ),
				),
				array(
					'location'    => 'boombox-archive-before-content',
					'description' => esc_html( 'Archive: Before Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-archive-after-content',
					'description' => esc_html( 'Archive: After Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-archive-before-footer',
					'description' => esc_html( 'Archive: Before Footer' ),
				),
			);

			/***** Page template */
			$page = array(
				array(
					'location'    => 'boombox-page-before-header',
					'description' => esc_html( 'Page: Before Header' ),
				),
				array(
					'location'    => 'boombox-page-before-featured-area',
					'description' => esc_html( 'Page: Before Featured Area' ),
				),
				array(
					'location'    => 'boombox-page-after-featured-area',
					'description' => esc_html( 'Page: After Featured Area' ),
				),
				array(
					'location'    => 'boombox-page-before-content',
					'description' => esc_html( 'Page: Before Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-page-after-content',
					'description' => esc_html( 'Page: After Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-page-before-footer',
					'description' => esc_html( 'Page: Before Footer' ),
				),
			);

			/***** Single template */
			$single = array(
				array(
					'location'    => 'boombox-single-before-header',
					'description' => esc_html( 'Single: Before Header' ),
				),
				array(
					'location'    => 'boombox-single-before-content',
					'description' => esc_html( 'Single: Before Content Theme Area' ),
				),
				array(
					'location'    => 'boombox-single-after-next-prev-buttons',
					'description' => esc_html( 'Single: After "Next/Prev" Buttons' ),
				),
				array(
					'location'    => 'boombox-single-before-navigation',
					'description' => esc_html( 'Single: Before Navigation Area' ),
				),
				array(
					'location'    => 'boombox-single-after-comments-section',
					'description' => esc_html( 'Single: After Comments Section' ),
				),
				array(
					'location'    => 'boombox-single-after-also-like-section',
					'description' => esc_html( 'Single: After "Also Like" Section' ),
				),
				array(
					'location'    => 'boombox-single-after-more-from-section',
					'description' => esc_html( 'Single: After "More From" Section' ),
				),
				array(
					'location'    => 'boombox-single-after-dont-miss-section',
					'description' => esc_html( 'Single: After "Don\'t miss" Section' ),
				),
				array(
					'location'    => 'boombox-single-before-footer',
					'description' => esc_html( 'Single: Before Footer' ),
				),
			);

			/***** Listings template */
			$listing = array(
				array(
					'location'    => 'boombox-listing-type-grid-instead-post',
					'description' => esc_html( 'Instead of "grid" or "three column" listing post' ),
				),
				array(
					'location'    => 'boombox-listing-type-non-grid-instead-post',
					'description' => esc_html( 'Instead of none grid listing post' ),
				),
			);

			$locations = array_merge( $header, $home, $archive, $page, $single, $listing );

			return apply_filters( 'boombox/custom_ad_locations', $locations );

		}

		/**
		 * Register ads custom areas
		 */
		private function register_ads() {
			$ad_locations = $this->get_locations();
			foreach( (array)$ad_locations as $ad_location ) {
				quads_register_ad( $ad_location );
			}
		}

		/**
		 * Get advertisement for location
		 * @param string $location Current location
		 * @param array $args Advertisement arguments
		 *
		 * @return string
		 */
		public function get_adv( $location, $args = array() ) {

			$html = '';

			$args = wp_parse_args( $args, array(
				'class' => array(),
				'tag'   => 'aside',
				'tmp_query' => false,
				'cur_query' => false,
				'before' => '',
				'after'  => '',
			) );

			$hide_ads = false;
			if( is_singular() ){
				$config = boombox_get_post_meta( get_the_ID(), '_quads_config_visibility' );
				if( isset( $config['NoAds'] ) && $config['NoAds'] ){
					$hide_ads = true;
				}
			}
			$hide_ads = apply_filters( 'boombox/advertisement/hide_ads', $hide_ads, $location );

			global $wp_query;
			if( $args['tmp_query'] ) {
				$wp_query = $args['tmp_query'];
			}

			if( function_exists( 'quads_ad' ) && $location && ! $hide_ads ) {

				$query_condition = $args['cur_query'] ? $args['cur_query']->have_posts() : have_posts();

				if( $query_condition ) {

					$adv = quads_ad( array( 'location' => $location, 'echo' => false ) );

					if( $adv ) {

						if ( is_array( $args[ 'class' ] ) ) {
							$args[ 'class' ] = trim( implode( ' ', $args[ 'class' ] ) );
						}
						$args[ 'class' ] = 'bb-advertisement ' . $args[ 'class' ];

						$html .= $args[ 'before' ];
							$html .= '<' . $args[ 'tag' ] . ' class="' . esc_attr( $args[ 'class' ] ) . '">';
								$html .= '<div class="inner">' . $adv . '</div>';
							$html .= '</' . $args[ 'tag' ] . '>';
						$html .= $args[ 'after' ];

					}

				}

			}

			if( $args['cur_query'] ) {
				$wp_query = $args['cur_query'];
			}

			return $html;

		}

		/**
		 * Edit page metaboxes structure and add additional fields
		 *
		 * @param array                 $structure  Current structure
		 * @param Boombox_Page_Meta_Box $instance   Current instance
		 * @return array
		 */
		public function edit_page_metaboxes_structure( $structure, $instance ) {

			$structure['tab_listing']['fields'] = array_merge( $structure['tab_listing']['fields'], array(
				// "Injects" heading
				'boombox_listing_injects_heading' => array(
					'name'     => 'boombox_listing_injects_heading',
					'type'     => 'custom',
					'priority' => 60,
					'sub_priority' => 0,
					'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Injects', 'boombox' ) ),
					'active_callback' => array(
						array(
							'field_id' => 'boombox_listing_type',
							'value'    => 'none',
							'compare'  => '!=',
						)
					),
				),
				// Page Ad Field
				'boombox_page_ad' => array(
					'id'                => 'boombox_page_ad',
					'label'		        => esc_html__( 'Ad', 'boombox' ),
					'type'		        => 'select',
					'name'		        => 'boombox_page_ad',
					'choices' 	        => Boombox_Choices_Helper::get_instance()->get_injects(),
					'priority'          => 60,
					'sub_priority'      => 20,
					'default'	        => 'none',
					'active_callback'    => array(
						array(
							'field_id' => 'boombox_listing_type',
							'value'    => 'none',
							'compare'  => '!='
						)
					),
					'callback'          => 'sanitize_text_field'
				),
				// Inject Ad As Post
				'boombox_inject_ad_instead_post' => array(
					'label'		        => esc_html__( 'Inject Ad As Post', 'boombox' ),
					'type'		        => 'number',
					'name'		        => 'boombox_inject_ad_instead_post',
					'priority'          => 60,
					'sub_priority'      => 21,
					'default'	        => 1,
					'callback'          => array( $this, 'sanitize_page_metaboxes_ad_instead_post_value' ),
					'value_callback'    => array( $this, 'render_page_metaboxes_ad_instead_post_value' ),
					'attributes'        => array(
						'min' => 1
					),
					'active_callback'    => array(
						array(
							'field_id' => 'boombox_listing_type',
							'value'    => 'none',
							'compare'  => '!='
						),
						array(
							'field_id' => 'boombox_page_ad',
							'value'    => 'none',
							'compare'  => '!='
						)
					),
				),
			) );

			return $structure;
		}

		/**
		 * Sanitize page metaboxes instead post value on rendering
		 *
		 * @param int $value Current value
		 * @return int
		 */
		public function render_page_metaboxes_ad_instead_post_value( $value ) {
			global $post;

			return max( 1, min( $value, boombox_get_post_meta( $post->ID, 'boombox_posts_per_page' ) ) );

		}

		/**
		 * Sanitize page metaboxes instead post value on saving
		 *
		 * @param mixed $value Current value
		 * @return int
		 */
		public function sanitize_page_metaboxes_ad_instead_post_value( $value ) {
			return min( max( 1, absint( $value ) ), absint( $_POST[ AIOM_Base::KEY ]['boombox_posts_per_page'] ) );
		}

		/**
		 * Render ad before single navigation section
		 */
		public function render_ad_before_single_navigation() {
			if( is_single() ) {
				boombox_the_advertisement( 'boombox-single-before-navigation', array( 'class' => 'large bb-before-nav-area' ) );
			}
		}

		/**
		 * Render ad after single comments sections
		 */
		public function render_ad_after_single_comments() {
			boombox_the_advertisement( 'boombox-single-after-comments-section', array( 'class' => 'large bb-after-comments-sec' ) );
		}

		/**
		 * Add extra fields to theme customizer
		 *
		 * @param array  $fields   Current fields
		 * @param string $section  Section ID
		 * @param array  $defaults Default values
		 *
		 * @return mixed
		 */
		public function add_inject_settings_to_customizer ( $fields, $section, $defaults ) {
			$priority = false;
			$section_prefix = '';
			if ( $section == boombox_customizer_get_archive_main_posts_section_id() ) {
				$priority = 100;
				$section_prefix = 'archive_main_posts_';
			} else if ( $section == boombox_customizer_get_home_main_posts_section_id() ) {
				$priority = 100;
				$section_prefix = 'home_main_posts_';
			}

			if ( $priority ) {
				$fields = array_merge( $fields, array(
					array(
						'settings' => $section_prefix . 'inject_ad',
						'label'    => esc_html__( 'Ad', 'boombox' ),
						'section'  => $section,
						'type'     => 'select',
						'priority' => $priority,
						'default'  => $defaults[ $section_prefix . 'inject_ad' ],
						'multiple' => 1,
						'choices'  => Boombox_Choices_Helper::get_instance()->get_injects(),
					),
					array(
						'settings' => $section_prefix . 'injected_ad_position',
						'label'    => esc_html__( 'Inject Ad As Post', 'boombox' ),
						'section'  => $section,
						'type'     => 'bb-number',
						'priority' => $priority,
						'default'  => $defaults[ $section_prefix . 'injected_ad_position' ],
						'choices'  => array(
							'min'  => 1,
							'step' => 1,
						),
						'active_callback'    => array(
							array(
								'setting'  => $section_prefix . 'inject_ad',
								'value'    => 'none',
								'operator' => '!=',
							),
						),
					)
				) );
			}

			return $fields;
		}

		/**
		 * Setup default values for customizer extra fields
		 *
		 * @param $values
		 *
		 * @return mixed
		 */
		public function edit_customizer_default_values ( $values ) {
			$section_prefixes = array( 'archive_main_posts_', 'home_main_posts_' );
			foreach ( $section_prefixes as $prefix ) {
				$values[ $prefix . 'inject_ad' ] = 'none';
				$values[ $prefix . 'injected_ad_position' ] = 1;
			}

			return $values;
		}

	}

	Boombox_WP_Quads::get_instance();

}