<?php
/**
 * Boombox generic field for metaboxes
 *
 * @package BoomBox_Theme
 * @since   2.0.0
 * @verion  2.0.0
 */

// Prevent direct script access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

/**
 * Class AIOM_Base_Field
 */
abstract class AIOM_Base_Field {

	/**
	 * Holds field name
	 * @var string
	 */
	private $name;

	/**
	 * Get field name
	 * @return string
	 */
	protected function get_name() {
		return sprintf( '%s[%s]', AIOM_Base::KEY, $this->name );
	}

	/**
	 * Holds field ID
	 * @var string
	 */
	private $id;

	/**
	 * Get field ID
	 * @return string
	 */
	protected function get_id() {
		return $this->id;
	}

	/**
	 * Holds field HTML classes
	 * @var string
	 */
	private $class;

	/**
	 * Get field HTML classes
	 * @return string
	 */
	protected function get_class() {
		return $this->class;
	}

	/**
	 * Holds field meta key ( database key )
	 * @var string
	 */
	private $meta_key;

	/**
	 * Get field meta key ( database key )
	 * @return string
	 */
	protected function get_meta_key() {
		return $this->meta_key;
	}

	/**
	 * Holds field label
	 * @var string
	 */
	private $label;

	/**
	 * Get field label
	 * @return string
	 */
	protected function get_label() {
		return $this->label;
	}

	/**
	 * Holds field description
	 * @var string
	 */
	private $description;

	/**
	 * Get field description
	 * @return string
	 */
	protected function get_description() {
		return $this->description;
	}

	/**
	 * Holds form field field attributes
	 * @var string
	 */
	private $attributes;

	/**
	 * Get form field attributes
	 * @return string
	 */
	protected function get_attributes() {
		return implode( ' ', array_values( $this->attributes ) );
	}

	/**
	 * Check if has attribute
	 * @param string $attribute Attribute name
	 *
	 * @return bool
	 */
	protected function has_attribute( $attribute ) {
		return array_key_exists( $attribute, $this->attributes );
	}

	/**
	 * Holds field text
	 * @var string
	 */
	private $text;

	/**
	 * Get field text
	 * @return bool|string
	 */
	protected function get_text() {
		return $this->text;
	}

	/**
	 * Holds field value rendering callback
	 * @var string|null
	 */
	private $value_callback;

	/**
	 * Get field value rendering callback
	 * @return null|string
	 */
	protected function get_value_callback() {
		return $this->value_callback;
	}

	/**
	 * Holds field default value
	 * @var mixed
	 */
	private $default;

	/**
	 * Get field default value
	 * @return mixed
	 */
	protected function get_default() {
		return $this->default;
	}

	/**
	 * Holds field actual value
	 * @var mixed
	 */
	private $value;

	/**
	 * Get field actual value
	 * @return mixed
	 */
	protected function get_value() {
		return $this->value;
	}

	/**
	 * Holds field wrapper classes
	 * @var string
	 */
	private $wrapper_class;

	/**
	 * Get field wrapper classes
	 * @return string
	 */
	protected function get_wrapper_class() {
		return $this->wrapper_class;
	}

	/**
	 * Holds field wrapper attributes
	 * @var string
	 */
	private $wrapper_attributes;

	/**
	 * Get field wrapper attributes
	 * @return string
	 */
	protected function get_wrapper_attributes() {
		return $this->wrapper_attributes;
	}

	/**
	 * Holds field active callback generated element
	 * @var string
	 */
	private $active_callback;

	/**
	 * Get field active callback generated element
	 * @return string
	 */
	protected function get_active_callback() {
		return $this->active_callback;
	}

	/**
	 * AIOM_Base_Field constructor.
	 * @param array $args {
	 *      Field arguments
	 *
	 * @type string         $name                   Field name attribute. Required
	 * @type string         $id                     Field id attribute: Optional. Will use field name if omitted
	 * @type string|array   $class                  Field CSS classes
	 * @type string         $meta_key               Field meta key to use in database. Will use field name if omitted
	 * @type string         $label                  Field label
	 * @type string         $description            Field description
	 * @type array          $attributes             Field HTML attributes
	 * @type callable       $value_callback         Function to use for field value rendering
	 * @type mixed          $default                Field default value
	 * @type string         $value                  Field actual value
	 * @type string|array   $wrapper_class          Field wrapper CSS classes
	 * @type array          $wrapper_attributes     Field wrapper HTML attributes
	 *
	 * }
	 * @param string|bool|null  $tab_id     Field tab ID
	 * @param array             $data       Meta data
	 * @param array             $structure  Fields structure
	 */
	public function __construct( $args, $tab_id, $data, $structure ){

		/***** Field Name */
		$name = isset( $args['name'] ) ? $args['name'] : false;
		if( ! $name ) {
			return;
		}
		$this->name = $name;

		/***** Field ID */
		$id = isset( $args['id'] ) && $args['id'] ? $args['id'] : $name;
		$this->id = $id;

		/***** Field class */
		$class = isset( $args['class'] ) && ! empty( $args['class'] ) ? $args['class'] : array();
		if( is_array( $class ) ) {
			$class = implode( ' ', $class );
		}
		$this->class = $class;

		/***** Field class */
		$meta_key = isset( $args['meta_key'] ) && $args['meta_key'] ? $args['meta_key'] : $name;
		$this->meta_key = $meta_key;

		/***** Field label */
		$label = isset( $args['label'] ) && $args['label'] ? $args['label'] : false;
		$this->label = $label;

		/***** Field description */
		$description = isset( $args['description'] ) && $args['description'] ? $args['description'] : false;
		$this->description = $description;

		/***** Field attributes */
		$attributes = isset( $args['attributes'] ) && is_array( $args['attributes'] ) ? $args['attributes'] : array();
		foreach( $attributes as $f_attr => $f_val ) {
			$attributes[ $f_attr ] = sprintf( '%s="%s"', $f_attr, $f_val );
		}
		$this->attributes = $attributes;

		/***** Field text */
		$text = isset( $args['text'] ) && $args['text'] ? $args['text'] : false;
		$this->text = $text;

		/***** Field value callback */
		$value_callback = isset( $args['value_callback'] ) && is_callable( $args['value_callback'] ) ? $args['value_callback'] : NULL;
		$this->value_callback = $value_callback;

		/***** Field default value */
		$default = isset( $args['default'] ) && $args['default'] ? $args['default'] : NULL;
		$this->default = $default;

		/***** Field value */
		$value = isset( $data[ $this->meta_key ] ) ? $data[ $this->meta_key ] : $this->default;
		$value = maybe_unserialize( $value );
		if( $this->value_callback ) {
			$value = call_user_func( $this->value_callback, $value );
		}
		$this->value = $value;

		/***** Field wrapper classes */
		$wrapper_class = isset( $args['wrapper_class'] ) && ! empty( $args['wrapper_class'] ) ? $args['wrapper_class'] : array();
		if( is_array( $wrapper_class ) ) {
			$wrapper_class = implode( ' ', $wrapper_class );
		}
		$this->wrapper_class = $wrapper_class;

		/***** Field wrapper attributes */
		$wrapper_attributes = isset( $args['wrapper_attributes'] ) && is_array( $args['wrapper_attributes'] ) ?
			$args['wrapper_attributes'] : array();
		foreach( $wrapper_attributes as $w_attr => $w_val ) {
			$wrapper_attributes[ $w_attr ] = sprintf( '%s="%s"', $w_attr, $w_val );
		}
		$wrapper_attributes = implode( ' ', array_values( $wrapper_attributes ) );
		$this->wrapper_attributes = $wrapper_attributes;

		/***** Field dependency */
		$this->active_callback = '';
		if( $this->has_active_callback( $args ) ) {
			$active_callback_data = $this->generate_active_callback_data(
				$args,
				$structure[ $tab_id ][ 'fields' ],
				$data );

			$this->active_callback = $active_callback_data->element;

			if( ! $active_callback_data->visible ) {
				$this->wrapper_class .= ' bb-hidden';
			}
		}

		/**
		 * Enqueue assets if any
		 */
		$this->enqueue();

	}

	/**
	 * Enqueue required assets
	 */
	public function enqueue() {}

	/**
	 * Get supported comparison operators
	 * @return array
	 */
	private function get_comparison_operators() {
		return array(
			'===',
			'==',
			'=',
			'!==',
			'!=',
			'>=',
			'<=',
			'>',
			'<',
			'IN',
			'NOT IN'
		);
	}

	/**
	 * Check field for active callback
	 * @param array $args Field arguments
	 *
	 * @return bool
	 */
	private function has_active_callback( $args ) {
		return ( isset( $args['active_callback'] ) && ! empty( $args['active_callback'] ) );
	}

	/**
	 * Generate active callback data for current field
	 *
	 * @param array $args   Field arguments
	 * @param array $fields Current fields structure
	 * @param array $data   Current data
	 *
	 * @return stdClass
	 */
	private function generate_active_callback_data( $args, $fields, $data ) {
		$callback = $args['active_callback'];


		$relation = 'AND';
		if( isset( $callback['relation'] ) && in_array( strtoupper( $callback['relation'] ), array( 'AND', 'OR' ) ) ) {
			$relation = strtoupper( $callback['relation'] );
			unset( $callback['relation'] );
		}

		$superiors = array();
		$is_visible = ( $relation == 'AND' );
		$callback_data = array(
			'relation' => $relation,
			'fields'   => array(),
		);

		foreach( $callback as $requirement ) {
			$r = wp_parse_args( $requirement, array(
				'field_id' => false,
				'compare'  => '==',
				'value'    => ''
			) );

			if( $r['field_id'] && in_array( $r['compare'], $this->get_comparison_operators() ) ) {

				$field_id = $r['field_id'];

				/***** check visibility */
				if( isset( $fields[ $field_id ][ 'meta_key' ] ) ) {
					$parent_meta_key = $fields[ $field_id ][ 'meta_key' ];
				} else {
					$parent_meta_key = $fields[ $field_id ][ 'name' ];
				}
				$parent_value = isset( $data[ $parent_meta_key ] ) ? $data[ $parent_meta_key ] : $fields[ $field_id ][ 'default' ];

				switch ( $r['compare'] ) {
					case '===':
						$show = ( $parent_value === $r['value'] ) ? true : false;
						break;
					case '==':
					case '=':
						$show = ( $parent_value == $r['value'] ) ? true : false;
						break;
					case '!==':
						$show = ( $parent_value !== $r['value'] ) ? true : false;
						break;
					case '!=':
						$show = ( $parent_value != $r['value'] ) ? true : false;
						break;
					case '>=':
						$show = ( $r['value'] >= $parent_value ) ? true : false;
						break;
					case '<=':
						$show = ( $r['value'] <= $parent_value ) ? true : false;
						break;
					case '>':
						$show = ( $r['value'] > $parent_value ) ? true : false;
						break;
					case '<':
						$show = ( $r['value'] < $parent_value ) ? true : false;
						break;
					case 'IN':
						$r['value'] = ( array )$r['value'];
						$show = in_array( $parent_value, $r['value'] ) ? true : false;
						break;
					case 'NOT IN':
						$r['value'] = ( array )$r['value'];
						$show = in_array( $parent_value, $r['value'] ) ? false : true;
						break;
					default:
						$show = ( $parent_value == $r['value'] ) ? true : false;
				}

				if( $relation == 'AND' ) {
					$is_visible = $is_visible && $show;
				} else {
					$is_visible = $is_visible || $show;
				}

				/***** Configure field */
				$r[ 'jq_selector' ] = '#' . $field_id;
				if( in_array( $fields[ $field_id ][ 'type' ], array( 'radio', 'radio-image', 'checkbox' ) ) ) {
					$r[ 'jq_selector' ] .= ':checked';
				}

				/***** setup callback data */
				$callback_data[ 'fields' ][] = $r;
				$superiors[] = 'bb-superior-' . $field_id;

			}
		}

		$return = new stdClass();
		$return->visible = $is_visible;
		$return->element = '';

		if( ! empty( $callback_data[ 'fields' ] ) ) {
			$superiors = implode( ' ', $superiors );
			$callback_data = json_encode( $callback_data );
			$return->element = sprintf( '<span class="bb-hidden %s">%s</span>', $superiors, $callback_data );
		}

		return $return;
	}

	/**
	 * Render field
	 *
	 * @return mixed
	 */
	abstract public function render();

}