<?php
/**
 * WP QUADS plugin functions
 *
 * @package BoomBox_Theme
 */

// Prevent direct script access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

if ( ! boombox_plugin_management_service()->is_plugin_active( 'quick-adsense-reloaded/quick-adsense-reloaded.php' ) || ! function_exists( 'quads_register_ad' ) ) {
	return;
}

if( ! class_exists( 'Boombox_WP_Quads' ) ) {

	class Boombox_WP_Quads {

		/**
		 * Holds class single instance
		 * @var null
		 */
		private static $_instance = null;

		/**
		 * Get instance
		 * @return Boombox_WP_Quads|null
		 */
		public static function get_instance() {

			if (null == static::$_instance) {
				static::$_instance = new Boombox_WP_Quads();
			}

			return static::$_instance;

		}

		/**
		 * Holds ads custom location data
		 * @var array
		 */
		private $ads;

		/**
		 * Boombox_WP_Quads constructor.
		 */
		private function __construct() {
			$this->hooks();
			$this->set_properties();
			$this->register_ads();
		}

		/**
		 * A dummy magic method to prevent Boombox_WP_Quads from being cloned.
		 *
		 */
		public function __clone() {
			throw new Exception('Cloning ' . __CLASS__ . ' is forbidden');
		}

		/**
		 * Setup Hooks
		 */
		public function hooks() {
			add_action( 'admin_bar_menu', array( $this, 'simulate_the_content_filter' ), 999 );
			add_action( 'boombox/single/sortables/before_navigation', array( $this, 'render_ad_before_single_navigation' ) );
			add_action( 'boombox/single/sortables/after_comments', array( $this, 'render_ad_after_single_comments' ) );
			add_filter( 'boombox/admin/page/meta-boxes/structure', array( $this, 'edit_page_metaboxes_structure' ), 10, 2 );
			add_filter( 'boombox/customizer/fields/archive_main_posts', array( $this, 'add_inject_settings_to_customizer' ), 10, 3 );
			add_filter( 'boombox/customizer/fields/home_main_posts', array( $this, 'add_inject_settings_to_customizer' ), 10, 3 );
			add_filter( 'boombox/customizer_default_values', array( $this, 'edit_customizer_default_values' ), 10, 1 );

			if( boombox_plugin_management_service()->is_plugin_active( 'wp-quads-pro/wp-quads-pro.php' ) ) {
				add_action( 'pre_option_quads_wp_quads_pro_license_active', array( $this, 'simulate_license' ), 10, 3 );
			}
		}

		/**
		 * Simulate theme in build license
		 * @param mixed $value Current value
		 * @param string $option Option name
		 * @param mixed $default Option default value
		 *
		 * @return stdClass
		 */
		public function simulate_license( $value, $option, $default ) {

			$quads_options = get_option( 'quads_settings' );
			$has_pro_key = isset( $quads_options[ 'quads_wp_quads_pro_license_key' ] ) && $quads_options[ 'quads_wp_quads_pro_license_key' ];
			if( ! $has_pro_key ) {
				$value = new stdClass();
				$value->success = true;
				$value->license = 'valid';
				$value->expires = 'lifetime';
			}

			return $value;
		}

		/**
		 * Simulate 'the_content' filter run for pages that does not have it
		 */
		public function simulate_the_content_filter() {

			global $wp_the_query, $the_content;
			if( $wp_the_query->is_singular() ) {
				$the_content = true;
			}

		}

		/**
		 * Setup ads custom locations
		 */
		private function set_properties() {

			$this->ads = array(
				array(
					'location' => 'boombox-archive-before-header',
					'description' => esc_html('Archive: Before Header')
				),
				array(
					'location' => 'boombox-archive-before-content',
					'description' => esc_html('Archive: Before content theme area')
				),
				array(
					'location' => 'boombox-archive-after-content',
					'description' => esc_html('Archive: After content theme area')
				),
				array(
					'location' => 'boombox-page-before-header',
					'description' => esc_html('Page: Before Header')
				),
				array(
					'location' => 'boombox-page-before-content',
					'description' => esc_html('Page: Before content theme area')
				),
				array(
					'location' => 'boombox-page-after-content',
					'description' => esc_html('Page: After content theme area')
				),
				array(
					'location' => 'boombox-single-before-header',
					'description' => esc_html('Single: Before Header')
				),
				array(
					'location' => 'boombox-single-before-content',
					'description' => esc_html('Single: Before content theme area')
				),
				array(
					'location' => 'boombox-single-after-next-prev-buttons',
					'description' => esc_html('Single: After "Next/Prev" buttons')
				),
				array(
					'location' => 'boombox-single-before-navigation',
					'description' => esc_html('Single: Before navigation area')
				),
				array(
					'location' => 'boombox-single-after-also-like-section',
					'description' => esc_html('Single: After "Also Like" section')
				),
				array(
					'location' => 'boombox-single-after-more-from-section',
					'description' => esc_html('Single: After "More From" section')
				),
				array(
					'location' => 'boombox-single-after-comments-section',
					'description' => esc_html('Single: After Comments section')
				),
				array(
					'location' => 'boombox-single-after-dont-miss-section',
					'description' => esc_html('Single: After "Don\'t miss" section')
				),
				array(
					'location' => 'boombox-listing-type-grid-instead-post',
					'description' => esc_html('Instead of "grid" or "three column" listing post')
				),
				array(
					'location' => 'boombox-listing-type-non-grid-instead-post',
					'description' => esc_html('Instead of none grid listing post')
				)
			);

		}

		/**
		 * Register ads custom areas
		 */
		private function register_ads() {
			foreach( (array)$this->ads as $ad_location ) {
				quads_register_ad( $ad_location );
			}
		}

		/**
		 * Edit page metaboxes structure and add additional fields
		 *
		 * @param array                 $structure  Current structure
		 * @param Boombox_Page_Meta_Box $instance   Current instance
		 * @return array
		 */
		public function edit_page_metaboxes_structure( $structure, $instance ) {

			$structure['tab_listing']['fields'] = array_merge( $structure['tab_listing']['fields'], array(
				// "Injects" heading
				'boombox_listing_injects_heading' => array(
					'name'     => 'boombox_listing_injects_heading',
					'type'     => 'custom',
					'priority' => 60,
					'sub_priority' => 0,
					'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Injects', 'boombox' ) ),
					'active_callback' => array(
						array(
							'field_id' => 'boombox_listing_type',
							'value'    => 'none',
							'compare'  => '!=',
						)
					),
				),
				// Page Ad Field
				'boombox_page_ad' => array(
					'id'                => 'boombox_page_ad',
					'label'		        => esc_html__( 'Ad', 'boombox' ),
					'type'		        => 'select',
					'name'		        => 'boombox_page_ad',
					'choices' 	        => Boombox_Choices_Helper::get_instance()->get_injects(),
					'priority'          => 60,
					'sub_priority'      => 20,
					'default'	        => 'none',
					'active_callback'    => array(
						array(
							'field_id' => 'boombox_listing_type',
							'value'    => 'none',
							'compare'  => '!='
						)
					),
					'callback'          => 'sanitize_text_field'
				),
				// Inject Ad After Post
				'boombox_inject_ad_instead_post' => array(
					'label'		        => esc_html__( 'Inject Ad After Post', 'boombox' ),
					'type'		        => 'number',
					'name'		        => 'boombox_inject_ad_instead_post',
					'priority'          => 60,
					'sub_priority'      => 21,
					'default'	        => 1,
					'callback'          => array( $this, 'sanitize_page_metaboxes_ad_instead_post_value' ),
					'value_callback'    => array( $this, 'render_page_metaboxes_ad_instead_post_value' ),
					'attributes'        => array(
						'min' => 1
					),
					'active_callback'    => array(
						array(
							'field_id' => 'boombox_listing_type',
							'value'    => 'none',
							'compare'  => '!='
						),
						array(
							'field_id' => 'boombox_page_ad',
							'value'    => 'none',
							'compare'  => '!='
						)
					),
				),
			) );

			return $structure;
		}

		/**
		 * Sanitize page metaboxes instead post value on rendering
		 *
		 * @param int $value Current value
		 * @return int
		 */
		public function render_page_metaboxes_ad_instead_post_value( $value ) {
			global $post;

			return max( 1, min( $value, boombox_get_post_meta( $post->ID, 'boombox_posts_per_page' ) ) );

		}

		/**
		 * Sanitize page metaboxes instead post value on saving
		 *
		 * @param mixed $value Current value
		 * @return int
		 */
		public function sanitize_page_metaboxes_ad_instead_post_value( $value ) {
			return min( max( 1, absint( $value ) ), absint( $_POST[ AIOM_Base::KEY ]['boombox_posts_per_page'] ) );
		}

		/**
		 * Render ad before single navigation section
		 */
		public function render_ad_before_single_navigation() {
			if( is_single() ) {
				boombox_the_advertisement( 'boombox-single-before-navigation', array( 'class' => 'large' ) );
			}
		}

		/**
		 * Render ad after single comments sections
		 */
		public function render_ad_after_single_comments() {
			if( is_single() ) {
				boombox_the_advertisement( 'boombox-single-after-comments-section', array( 'class' => 'large' ) );
			}
		}

		/**
		 * Add extra fields to theme customizer
		 *
		 * @param array  $fields   Current fields
		 * @param string $section  Section ID
		 * @param array  $defaults Default values
		 *
		 * @return mixed
		 */
		public function add_inject_settings_to_customizer ( $fields, $section, $defaults ) {
			$priority = false;
			$section_prefix = '';
			if ( $section == boombox_customizer_get_archive_main_posts_section_id() ) {
				$priority = 100;
				$section_prefix = 'archive_main_posts_';
			} else if ( $section == boombox_customizer_get_home_main_posts_section_id() ) {
				$priority = 90;
				$section_prefix = 'home_main_posts_';
			}

			if ( $priority ) {
				$fields = array_merge( $fields, array(
					array(
						'settings' => $section_prefix . 'inject_ad',
						'label'    => esc_html__( 'Ad', 'boombox' ),
						'section'  => $section,
						'type'     => 'select',
						'priority' => $priority,
						'default'  => $defaults[ $section_prefix . 'inject_ad' ],
						'multiple' => 1,
						'choices'  => Boombox_Choices_Helper::get_instance()->get_injects(),
					),
					array(
						'settings' => $section_prefix . 'injected_ad_position',
						'label'    => esc_html__( 'Inject Ad After Post', 'boombox' ),
						'section'  => $section,
						'type'     => 'bb-number',
						'priority' => $priority,
						'default'  => $defaults[ $section_prefix . 'injected_ad_position' ],
						'choices'  => array(
							'min'  => 1,
							'step' => 1,
						),
						'active_callback'    => array(
							array(
								'setting'  => $section_prefix . 'inject_ad',
								'value'    => 'none',
								'operator' => '!=',
							),
						),
					)
				) );
			}

			return $fields;
		}

		/**
		 * Setup default values for customizer extra fields
		 *
		 * @param $values
		 *
		 * @return mixed
		 */
		public function edit_customizer_default_values ( $values ) {
			$section_prefixes = array( 'archive_main_posts_', 'home_main_posts_' );
			foreach ( $section_prefixes as $prefix ) {
				$values[ $prefix . 'inject_ad' ] = 'none';
				$values[ $prefix . 'injected_ad_position' ] = 1;
			}

			return $values;
		}

		public function asd( $license ) {
			$license = new stdClass();
			$license->success = true;
			$license->license = '';
			$license->license = 'valid';
			$license->expires = 'lifetime';

			return $license;
		}

	}

	Boombox_WP_Quads::get_instance();

}