<?php
/**
 * Boombox contributor role permissions and settings
 *
 * @package BoomBox_Theme
 */

// Prevent direct script access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

/**
 * Setup permissions for user roles
 */
function  boombox_edit_user_role_permissions(){

	$current_user = wp_get_current_user();

	if( current_user_can( 'contributor' ) && in_array( 'contributor', $current_user->roles ) ) {

		if( is_admin() ){

			/**
			 * Removes from post and pages
			 */
			remove_post_type_support( 'post', 'comments' );
			remove_post_type_support( 'post', 'author' );

			/**
			 * Hooks
			 */
			add_action( 'admin_menu', 'boombox_remove_admin_menus' );
			add_action( 'wp_before_admin_bar_render', 'boombox_admin_bar_render' );
			add_action( 'admin_init', 'boombox_admin_menu_redirect');

			/**
			 * Removes from admin menu
			 */
			function boombox_remove_admin_menus() {
				remove_menu_page( 'edit-comments.php' );
				remove_menu_page( 'tools.php' );
				remove_menu_page( 'index.php' );
				remove_menu_page( 'about.php' );
			}

			/**
			 * Removes from admin bar
			 */
			function boombox_admin_bar_render() {
				global $wp_admin_bar;
				$wp_admin_bar->remove_menu('comments');
			}

			/**
			 * Redirect any user trying to access dashboard, comments or tools pages
			 */
			function boombox_admin_menu_redirect() {
				global $pagenow;
				$deprecated_pages = array(
					'index.php',
					'edit-comments.php',
					'tools.php',
					'about.php'
				);
				if ( in_array( $pagenow, $deprecated_pages ) ) {
					wp_redirect(admin_url( 'profile.php' )); exit;
				}
			}
		}
	}
}
add_action( 'init', 'boombox_edit_user_role_permissions', 100 );

/**
 * Update single post 'total_views' meta
 *
 * @param int $scale    Selected view scale
 * @param int $post_id  Post ID
 */
function boombox_update_post_total_view( $scale, $post_id ) {
	if( absint( $scale ) > 0 ) {
		$total = intval( boombox_get_post_meta( $post_id, 'total_views' ) );
		$total += $scale;

		update_post_meta( $post_id, 'total_views', $total );
	}
}
add_action( 'boombox/view_total_updated', 'boombox_update_post_total_view', 10, 2 );

/**
 * Update post author 'total_views' meta
 * @param $scale
 * @param $post_id
 */
function boombox_update_author_total_view( $scale, $post_id ) {

	$author_id = get_post_field( 'post_author', (int)$post_id );
	if( !$author_id ) {
		return;
	}

	$total_posts_view_count = get_user_meta( $author_id, 'total_posts_view_count', true );
	if( ! $total_posts_view_count ) {
		$total_posts_view_count = $scale;
	} else {
		$total_posts_view_count += $scale;
	}
	update_user_meta( $author_id, 'total_posts_view_count', $total_posts_view_count );
}
add_action( 'boombox/view_total_updated', 'boombox_update_author_total_view', 10, 2 );

/**
 * Creating predefined rate jobs
 *
 * @param array $jobs
 * @return array
 * @since 1.0.0
 * @version 2.0.0
 */
function boombox_create_predefined_jobs( array $jobs ) {
	$set = boombox_get_theme_options_set( array(
		'extras_post_ranking_system_trending_conditions',
		'extras_post_ranking_system_trending_enable',
		'extras_post_ranking_system_hot_enable',
		'extras_post_ranking_system_popular_enable',
		'extras_post_ranking_system_trending_posts_count',
		'extras_post_ranking_system_trending_minimal_score',
		'extras_post_ranking_system_hot_posts_count',
		'extras_post_ranking_system_hot_minimal_score',
		'extras_post_ranking_system_popular_posts_count',
		'extras_post_ranking_system_popular_minimal_score'
	) );

	$is_module_active = boombox_module_management_service()->is_module_active( 'prs' );
	if ( $is_module_active && Boombox_Rate_Criteria::get_criteria_by_name( $set['extras_post_ranking_system_trending_conditions'] ) ) {
		/***** Trending */
		if ( $set['extras_post_ranking_system_trending_enable'] ) {
			$job = boombox_get_rate_job(
				$set['extras_post_ranking_system_trending_conditions'],
				'post',
				'day',
				absint( $set['extras_post_ranking_system_trending_posts_count'] ),
				max( absint( $set['extras_post_ranking_system_trending_minimal_score'] ), 1 )
			);
			Boombox_Rate_Cron::register_job( $job );

			$jobs[ 'trending' ] = $job;
		}
		/***** Hot */
		if ( $set['extras_post_ranking_system_hot_enable'] ) {
			$job = boombox_get_rate_job(
				$set['extras_post_ranking_system_trending_conditions'],
				'post',
				'week',
				absint( $set['extras_post_ranking_system_hot_posts_count'] ),
				max( absint( $set['extras_post_ranking_system_hot_minimal_score'] ), 1 )
			);
			Boombox_Rate_Cron::register_job( $job );

			$jobs[ 'hot' ] = $job;
		}
		/***** Popular */
		if ( $set['extras_post_ranking_system_popular_enable'] ) {
			$job = boombox_get_rate_job(
				$set['extras_post_ranking_system_trending_conditions'],
				'post',
				'month',
				absint( $set['extras_post_ranking_system_popular_posts_count'] ),
				max( absint( $set['extras_post_ranking_system_popular_minimal_score'] ), 1 )
			);
			Boombox_Rate_Cron::register_job( $job );

			$jobs[ 'popular' ] = $job;
		}
	}

	return $jobs;
}
add_filter( 'boombox_rate_jobs', 'boombox_create_predefined_jobs', 10, 1 );

/**
 * Get assets suffix
 * @return string
 */
function boombox_get_assets_suffix() {
	return ( ! defined( 'SCRIPT_DEBUG' ) || true !== SCRIPT_DEBUG ) ? '.min' : '';
}

/**
 * Check whether a reactions are enabled.
 * @since 1.0.0
 * @version 2.0.0
 */
function boombox_reactions_is_enabled() {
	$is_module_active = boombox_module_management_service()->is_module_active( 'prs' );

	return ( $is_module_active && (bool)boombox_get_theme_option( 'extras_post_reaction_system_enable' ) );
}

/**
 * Get "paged" value for pages
 *
 * @return int
 */
function boombox_get_paged() {
	global $paged;
	if( is_front_page() ){
		$paged = absint( get_query_var( 'page' ) ) ? absint( get_query_var( 'page' ) ) : 1;
	}else{
		if( get_query_var( 'paged' ) ) {
			$paged = absint( get_query_var( 'paged' ) );
		} elseif( get_query_var( 'page' ) ) {
			$paged = absint( get_query_var( 'page' ) );
		} else {
			$paged = 1;
		}
	}
	return $paged;
}

/**
 * Get query for pages by condition
 *
 * @param $conditions
 * @param $time_range
 * @param $tax_conditions
 * @param $params
 * @return null|WP_Query
 */
function boombox_get_posts_query( $conditions, $time_range, $tax_conditions, $params = array() ) {

	$categories = isset( $tax_conditions['category'] ) ? $tax_conditions['category'] : array();
	$tags = isset( $tax_conditions['tag'] ) ? $tax_conditions['tag'] : array();
	$reactions = array();
	if( boombox_reactions_is_enabled() && taxonomy_exists( 'reaction' ) && isset( $tax_conditions['reaction'] ) ) {
		$reactions = $tax_conditions['reaction'];
	}

	$params = wp_parse_args( $params, array(
		'posts_per_page'        => - 1,
		'post_type'             => array( 'post' ),
		'paged'                 => 1,
		'posts_count'           => -1,
		'is_grid'               => false,
		'is_page_query'         => true,
		'excluded_categories'   => array(),
		'is_live'               => false,
		'fake_meta_key'         => null,
		'ignore_sticky_posts'   => true,
		'excluded_posts'        => array(),
		'offset'                => null
	) );

	global $wpdb, $post;

	$query           		= null;
	$page_ad         		= false;
	$page_newsletter 		= false;
	$page_products 			= false;
	$instead         		= 0;
	$instead_newsletter     = 0;
	$page_product_position  = 0;
	$page_product_count 	= 0;

	if( is_page() && $params['is_page_query'] ){
		$page_ad = boombox_get_post_meta( $post->ID, 'boombox_page_ad' );
		$instead = boombox_get_post_meta( $post->ID, 'boombox_inject_ad_instead_post' );

		$page_newsletter = boombox_get_post_meta( $post->ID, 'boombox_page_newsletter' );
		$instead_newsletter = boombox_get_post_meta( $post->ID, 'boombox_inject_newsletter_instead_post' );

		$page_product_position = boombox_get_post_meta( $post->ID, 'boombox_page_injected_products_position' );
		$page_product_count = boombox_get_post_meta( $post->ID, 'boombox_page_injected_products_count' );
		$page_products = boombox_get_post_meta( $post->ID, 'boombox_page_products_inject' );
	}

	$params['post_type'] = boombox_get_post_types_args( $params['post_type'] );
	$is_module_active = boombox_module_management_service()->is_module_active( 'prs' );
	if( $is_module_active && Boombox_Rate_Criteria::get_criteria_by_name( $conditions ) ) {
		$args = array(
			'ignore_sticky_posts' => $params['ignore_sticky_posts']
		);

		$categories_args = boombox_categories_args( $categories );
		if ( $categories_args ) {
			$args['tax_query'][] = $categories_args;
		}
		$tags_args = boombox_tags_args( $tags );
		if ( $tags_args ) {
			$args['tax_query'][] = $tags_args;
		}
		$reaction_args = boombox_reactions_args( $reactions );
		if ( $reaction_args ) {
			$args['tax_query'][] = $reaction_args;
		}
		if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
			$args['tax_query']['relation'] = 'AND';
		}

		if( -1 != $params['posts_per_page'] ){
			$args['posts_per_page'] = $params['posts_per_page'];
		}else{
			$args['nopaging'] = true;
		}

		if( $params['paged'] ){
			$args['paged'] = $params['paged'];
		}

		if( $params['offset'] ) {
			$args['offset'] = $params['offset'];
		}

		if( ! empty( $params['excluded_categories'] ) ){
			$args['category__not_in'] = $params['excluded_categories'];
		}

		if( ! empty( $params['excluded_posts'] ) ){
			$args['post__not_in'] = $params['excluded_posts'];
		}

		$is_adv_enabled = boombox_is_adv_enabled( $page_ad );
		$is_newsletter_enabled = boombox_is_newsletter_enabled( $page_newsletter );
		$is_product_enabled = boombox_is_product_enabled( $page_products );
		if( ( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ) && $params['is_page_query'] ){
			Boombox_Loop_Helper::init( array(
				'is_adv_enabled' 		=> $is_adv_enabled,
				'instead_adv' 			=> $instead,
				'is_newsletter_enabled' => $is_newsletter_enabled,
				'instead_newsletter' 	=> $instead_newsletter,
				'is_product_enabled' 	=> $is_product_enabled,
				'page_product_position' => $page_product_position,
				'page_product_count' 	=> $page_product_count,
				'skip' 					=> $params['is_grid'],
				'posts_per_page' 		=> $params['posts_per_page'],
				'paged' 				=> $params['paged']
			) );
		}

		if( 'all' == $time_range ){
			$params['is_live'] = true;
		}

		$job = boombox_get_rate_job( $conditions, $params['post_type'], $time_range, $params['posts_count'], 0, $params['is_live']);
		$args = apply_filters('boombox/query_args/' . $conditions, $args);
		$rate_query = new Boombox_Rate_Query( $args, $job, $params['fake_meta_key'] );
		$query = $rate_query->get_wp_query();

	} else {
		switch ( $conditions ) {

			case 'recent':
				$args = array(
					'post_status'         => 'publish',
					'post_type'           => $params['post_type'],
					'orderby'             => 'date',
					'order'               => 'DESC',
					'posts_per_page'      => $params['posts_per_page'],
					'offset'              => $params['offset'],
					'ignore_sticky_posts' => $params['ignore_sticky_posts']
				);

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( $params['paged'] ){
					$args['paged'] = $params['paged'];
				}

				if( ! empty( $params['excluded_categories'] ) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$is_adv_enabled = boombox_is_adv_enabled( $page_ad );
				$is_newsletter_enabled = boombox_is_newsletter_enabled( $page_newsletter );
				$is_product_enabled = boombox_is_product_enabled( $page_products );
				if( ( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ) && $params['is_page_query'] ){
					Boombox_Loop_Helper::init( array(
						'is_adv_enabled'        => $is_adv_enabled,
						'instead_adv'           => $instead,
						'is_newsletter_enabled' => $is_newsletter_enabled,
						'instead_newsletter'    => $instead_newsletter,
						'is_product_enabled'    => $is_product_enabled,
						'page_product_position' => $page_product_position,
						'page_product_count'    => $page_product_count,
						'skip'                  => $params[ 'is_grid' ],
						'posts_per_page'        => $params[ 'posts_per_page' ],
						'paged'                 => $params[ 'paged' ],
					) );
				}

				$args = apply_filters('boombox/query_args/recent', $args);
				$query = new WP_Query( $args );

				break;

			case 'most_shared':
				// get a most shared posts ids
				$args = array(
					'post_status'         => 'publish',
					'post_type'           => $params['post_type'],
					'posts_per_page'      => -1,
					'ignore_sticky_posts' => $params['ignore_sticky_posts'],
					'fields'              => 'ids',
					'orderby'             => 'meta_value',
					'order'               => 'DESC',
					'meta_query'          => array(
						array(
							'key'     => boombox_get_shares_meta_key(),
							'value'   => 0,
							'compare' => '>'
						),
					),
				);

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( ! empty( $params['excluded_categories'] ) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$most_shared_ids = array();
				$main_query = new WP_Query( $args );
				if( $main_query->have_posts() ){
					$most_shared_ids = $main_query->posts;
				}

				// get a fake posts ids ( for trending pages )
				if( null != $params['fake_meta_key'] ){
					$args = array(
						'post_status'         => 'publish',
						'post_type'           => $params['post_type'],
						'posts_per_page'      => -1,
						'ignore_sticky_posts' => $params['ignore_sticky_posts'],
						'fields'              => 'ids',
						'meta_query'          => array(
							array(
								'key'     => $params['fake_meta_key'],
								'value'   => 0,
								'compare' => '>'
							),
						),
					);

					$fake_posts_query = new WP_Query( $args );
					if( $fake_posts_query->have_posts() ){
						$most_shared_ids = array_merge( $most_shared_ids, $fake_posts_query->posts );
					}
				}

				if( empty( $most_shared_ids ) ){
					// Passing an empty array to post__in will return all posts.
					// to prevent this we set into array fake post id
					$most_shared_ids = array( 0 );
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$most_shared_ids = array_diff( $most_shared_ids, $params['excluded_posts'] );
				}

				$args = array(
					'post_status'         => 'publish',
					'post_type'           => $params['post_type'],
					'posts_per_page'      => $params['posts_per_page'],
					'offset'              => $params['offset'],
					'meta_key'            => boombox_get_shares_meta_key(),
					'orderby'             => 'meta_value_num',
					'order'               => 'DESC',
					'ignore_sticky_posts' => $params['ignore_sticky_posts'],
					'post__in'            => $most_shared_ids
				);

				if( $params['paged'] ){
					$args['paged'] = $params['paged'];
				}

				$is_adv_enabled = boombox_is_adv_enabled( $page_ad );
				$is_newsletter_enabled = boombox_is_newsletter_enabled( $page_newsletter );
				$is_product_enabled = boombox_is_product_enabled( $page_products );
				if( ( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ) && $params['is_page_query'] ){
					Boombox_Loop_Helper::init( array(
						'is_adv_enabled' 		=> $is_adv_enabled,
						'instead_adv' 			=> $instead,
						'is_newsletter_enabled' => $is_newsletter_enabled,
						'instead_newsletter' 	=> $instead_newsletter,
						'is_product_enabled' 	=> $is_product_enabled,
						'page_product_position' => $page_product_position,
						'page_product_count' 	=> $page_product_count,
						'skip' 					=> $params['is_grid'],
						'posts_per_page' 		=> $params['posts_per_page'],
						'paged' 				=> $params['paged']
					) );
				}

				$args = apply_filters('boombox/query_args/most_shared', $args);
				$query = new WP_Query( $args );

				break;

			case 'featured':
				$args = array(
					'post_status'         => 'publish',
					'post_type'           => $params['post_type'],
					'posts_per_page'      => $params['posts_per_page'],
					'offset'              => $params['offset'],
					'orderby'             => 'date',
					'order'               => 'DESC',
					'meta_query'          => array(
						array(
							'key'     => 'boombox_is_featured',
							'compare' => '=',
							'value'   => 1
						)
					),
					'ignore_sticky_posts' => $params['ignore_sticky_posts'],
				);

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( $params['paged'] ){
					$args['paged'] = $params['paged'];
				}

				if( ! empty( $params['excluded_categories'] ) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$is_adv_enabled = boombox_is_adv_enabled( $page_ad );
				$is_newsletter_enabled = boombox_is_newsletter_enabled( $page_newsletter );
				$is_product_enabled = boombox_is_product_enabled( $page_products );
				if( ( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ) && $params['is_page_query'] ){
					Boombox_Loop_Helper::init( array(
						'is_adv_enabled' 		=> $is_adv_enabled,
						'instead_adv' 			=> $instead,
						'is_newsletter_enabled' => $is_newsletter_enabled,
						'instead_newsletter' 	=> $instead_newsletter,
						'is_product_enabled' 	=> $is_product_enabled,
						'page_product_position' => $page_product_position,
						'page_product_count' 	=> $page_product_count,
						'skip' 					=> $params['is_grid'],
						'posts_per_page' 		=> $params['posts_per_page'],
						'paged' 				=> $params['paged']
					) );
				}

				$args = apply_filters('boombox/query_args/featured', $args);
				$query = new WP_Query( $args );

				break;

			case 'featured_frontpage':
				$args = array(
					'post_status'         => 'publish',
					'post_type'           => $params['post_type'],
					'posts_per_page'      => $params['posts_per_page'],
					'offset'              => $params['offset'],
					'orderby'             => array( 'meta_value_num' => 'DESC', 'date' => 'DESC' ),
					'meta_query'          => array(
						'relation'	=> 'OR',
						array(
							'key'     => 'boombox_is_featured',
							'compare' => '>',
							'value'   => 0
						),
						array(
							'key'     => 'boombox_is_featured_frontpage',
							'compare' => '>',
							'value'   => 0
						),
					),
					'ignore_sticky_posts' => $params['ignore_sticky_posts'],
				);

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( $params['paged'] ){
					$args['paged'] = $params['paged'];
				}

				if( !empty( $params['excluded_categories'] ) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$is_adv_enabled = boombox_is_adv_enabled( $page_ad );
				$is_newsletter_enabled = boombox_is_newsletter_enabled( $page_newsletter );
				$is_product_enabled = boombox_is_product_enabled( $page_products );
				if( ( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ) && $params['is_page_query'] ){
					Boombox_Loop_Helper::init( array(
						'is_adv_enabled' 		=> $is_adv_enabled,
						'instead_adv' 			=> $instead,
						'is_newsletter_enabled' => $is_newsletter_enabled,
						'instead_newsletter' 	=> $instead_newsletter,
						'is_product_enabled' 	=> $is_product_enabled,
						'page_product_position' => $page_product_position,
						'page_product_count' 	=> $page_product_count,
						'skip' 					=> $params['is_grid'],
						'posts_per_page' 		=> $params['posts_per_page'],
						'paged' 				=> $params['paged']
					) );
				}

				$args = apply_filters('boombox/query_args/featured_frontpage', $args);
				$query = new WP_Query( $args );

				break;

			case 'related':
				if( $post ) {

					$related_ids = array();

					// Exclude current post.
					$params['excluded_posts'][] = $post->ID;

					// get posts id's filtered by current post tags
					$related_args = array(
						'posts_per_page'        => $params['posts_per_page'],
						'post_type'             => $params['post_type'],
						'post_status'           => 'publish',
						'ignore_sticky_posts'   => $params['ignore_sticky_posts'],
						'post__not_in'          => array_unique( $params['excluded_posts'] ),
						'fields'                => 'ids'
					);

					if( !empty( $params['excluded_categories'] ) ){
						$related_args['category__not_in'] = $params['excluded_categories'];
					}

					$categories_args = boombox_categories_args( $categories );
					if ( $categories_args ) {
						$related_args['tax_query'][] = $categories_args;
					}
					$tags_args = boombox_tags_args( $tags );
					if ( $tags_args ) {
						$related_args['tax_query'][] = $tags_args;
					}
					$reaction_args = boombox_reactions_args( $reactions );
					if ( $reaction_args ) {
						$related_args['tax_query'][] = $reaction_args;
					}
					if ( isset( $related_args['tax_query'] ) && ( count( $related_args['tax_query'] ) > 1 ) ) {
						$related_args['tax_query']['relation'] = 'AND';
					}

					$related_query = new WP_Query( $related_args );
					if( $related_query->have_posts() ){
						$related_ids = $related_query->posts;
					}

					// if related posts smaller than necessary, add ids from recent posts
					if( $related_query->found_posts < $params['posts_per_page'] && ! apply_filters( 'boombox/query_args/related/disable_fill_with_recent_posts', false ) ){
						$exclude_ids = $related_ids;
						$exclude_ids[] = (int) $post->ID;
						$add_count = $params['posts_per_page'] - $related_query->found_posts;
						$recent_args = array(
							'posts_per_page'        => $add_count,
							'post_type'             => $params['post_type'],
							'post_status'           => 'publish',
							'ignore_sticky_posts'   => $params['ignore_sticky_posts'],
							'fields'                => 'ids'
						);
						if( 0 != $add_count ){
							$recent_args['post__not_in'] = $exclude_ids;
						}
						if( !empty( $params['excluded_categories'] ) ){
							$related_args['category__not_in'] = $params['excluded_categories'];
						}

						$add_query = new WP_Query( $recent_args );
						if( $add_query->have_posts() ){
							$related_ids = array_merge( $related_ids, $add_query->posts );
						}
					}

					if( ! empty( $params['excluded_posts'] ) ){
						$related_ids = array_diff( $related_ids, $params['excluded_posts'] );
					}

					// get related posts by ids
					$args = array(
						'post_type'           => $params['post_type'],
						'post_status'         => 'publish',
						'post__in'            => $related_ids,
						'orderby'             => 'post__in',
						'posts_per_page'      => $params['posts_per_page'],
						'offset'              => $params['offset'],
						'ignore_sticky_posts' => $params['ignore_sticky_posts'],
					);
					$args = apply_filters('boombox/query_args/related', $args);
					$query = new WP_Query( $args );
				}

				break;

			case 'more_from':
				$args = array(
					'post_type'           => $params['post_type'],
					'post_status'         => 'publish',
					'posts_per_page'      => $params['posts_per_page'],
					'offset'              => $params['offset'],
					'ignore_sticky_posts' => $params['ignore_sticky_posts'],
				);

				if ( is_single() ) {
					$params['excluded_posts'][] = $post->ID;
				}

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( !empty( $params['excluded_categories'] ) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$args = apply_filters('boombox/query_args/more_from', $args);
				$query = new WP_Query( $args );

				break;

			case 'dont_miss':
				$args = array(
					'post_type'           => $params['post_type'],
					'post_status'         => 'publish',
					'posts_per_page'      => $params['posts_per_page'],
					'offset'              => $params['offset'],
					'ignore_sticky_posts' => $params['ignore_sticky_posts']
				);

				if ( is_single() ) {
					$params['excluded_posts'][] = $post->ID;
				}

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( !empty( $params['excluded_categories']) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$args = apply_filters('boombox/query_args/dont_miss', $args);
				$query = new WP_Query( $args );

				break;

			case 'random':
				$args = array(
					'post_status'         => 'publish',
					'post_type'           => $params['post_type'],
					'orderby'             => 'rand',
					'order'               => 'DESC',
					'posts_per_page'      => $params['posts_per_page'],
					'ignore_sticky_posts' => $params['ignore_sticky_posts']
				);

				$categories_args = boombox_categories_args( $categories );
				if ( $categories_args ) {
					$args['tax_query'][] = $categories_args;
				}
				$tags_args = boombox_tags_args( $tags );
				if ( $tags_args ) {
					$args['tax_query'][] = $tags_args;
				}
				$reaction_args = boombox_reactions_args( $reactions );
				if ( $reaction_args ) {
					$args['tax_query'][] = $reaction_args;
				}
				if ( isset( $args['tax_query'] ) && ( count( $args['tax_query'] ) > 1 ) ) {
					$args['tax_query']['relation'] = 'AND';
				}

				if( $params['paged'] ){
					$args['paged'] = $params['paged'];
				}

				if( ! empty( $params['excluded_categories'] ) ){
					$args['category__not_in'] = $params['excluded_categories'];
				}

				if( ! empty( $params['excluded_posts'] ) ){
					$args['post__not_in'] = $params['excluded_posts'];
				}

				$is_adv_enabled = boombox_is_adv_enabled( $page_ad );
				$is_newsletter_enabled = boombox_is_newsletter_enabled( $page_newsletter );
				$is_product_enabled = boombox_is_product_enabled( $page_products );
				if( ( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ) && $params['is_page_query'] ){
					Boombox_Loop_Helper::init( array(
						'is_adv_enabled' 		=> $is_adv_enabled,
						'instead_adv' 			=> $instead,
						'is_newsletter_enabled' => $is_newsletter_enabled,
						'instead_newsletter' 	=> $instead_newsletter,
						'is_product_enabled' 	=> $is_product_enabled,
						'page_product_position' => $page_product_position,
						'page_product_count' 	=> $page_product_count,
						'skip' 					=> $params['is_grid'],
						'posts_per_page' 		=> $params['posts_per_page'],
						'paged' 				=> $params['paged']
					) );
				}

				$args = apply_filters('boombox/query_args/random', $args);

				$query = new WP_Query( $args );

				break;
		}
	}

	return $query;
}

/**
 * Creates rate job
 *
 * @param $criteria_name
 * @param $post_type
 * @param $time_range
 * @param $posts_count
 * @param $min_count
 * @param $is_live
 *
 * @return Boombox_Rate_Job
 */
function boombox_get_rate_job( $criteria_name, $post_type, $time_range, $posts_count, $min_count = 0, $is_live = false ){
	$job_name = md5(uniqid(rand(), true));
	$job = new Boombox_Rate_Job( $job_name, $post_type, 'publish', $criteria_name, $time_range, $posts_count, $min_count, $is_live );

	return $job;
}

/**
 * Check against sidebar enabled feature
 * @return bool
 *
 * @since   2.0.0
 * @version 2.0.0
 */
function boombox_is_primary_sidebar_enabled( $sidebar_type = '' ) {

	if ( boombox_is_fragment_cache_enabled() ) {
		?>
		<!-- mfunc <?php echo W3TC_DYNAMIC_SECURITY; ?>
			$enabled = wp_is_mobile() ? boombox_get_theme_option( 'mobile_global_enable_sidebar' ) : true;
		-->
		<?php $enabled = wp_is_mobile() ? boombox_get_theme_option( 'mobile_global_enable_sidebar' ) : true; ?>
		<!-- /mfunc <?php echo W3TC_DYNAMIC_SECURITY; ?> -->
		<?php
	} else if ( boombox_is_page_cache_enabled() ) {
		$enabled = true;
	} else {
		$enabled = wp_is_mobile() ? boombox_get_theme_option( 'mobile_global_enable_sidebar' ) : true;
	}

	return ( $enabled && ( $sidebar_type != 'no-sidebar' ) ) ;
}

/**
 * Check against secondary sidebar enabled feature
 * @param $option
 *
 * @return bool
 *
 * @since 2.0.0
 * @version 2.0.0
 */
function boombox_is_secondary_sidebar_enabled( $option ) {
	$secondary_sidebar_enable_choices = array(
		'2-sidebars-1_4-1_4',
		'2-sidebars-small-big'
	);

	return in_array( $option, $secondary_sidebar_enable_choices );
}

/**
 * Get related posts query
 *
 * @param $conditions
 * @param $posts_per_page
 *
 * @return WP_Query
 */
function boombox_get_related_posts_items( $conditions, $posts_per_page ){
	global $post;

	$time_range = '';
	$categories = array();
	$tags       = array();

	$rel_tags = get_the_terms( $post->ID, 'post_tag' );
	if ( !empty( $rel_tags ) ) {
		$tags = wp_list_pluck( $rel_tags, 'slug' );
	}

	$query = boombox_get_posts_query( $conditions, $time_range, array( 'category' => $categories, 'tag' => $tags ), array(
		'posts_per_page' => $posts_per_page
	));

	return $query;
}

/**
 * Get "More From" Section posts query
 *
 * @param $conditions
 * @param $post_first_category
 * @param $posts_per_page
 *
 * @return WP_Query
 */
function boombox_get_more_from_posts_items( $conditions, $post_first_category, $posts_per_page ){

	$time_range = '';
	$categories = array();
	$tags       = array();

	if( $post_first_category ){
		$categories = array(
			$post_first_category->slug
		);
	}

	$query = boombox_get_posts_query( $conditions, $time_range, array( 'category' => $categories, 'tag' => $tags ), array(
		'posts_per_page' => $posts_per_page
	));

	return $query;
}

/**
 * Get "Don't Miss" Section posts query
 *
 * @param $conditions
 * @param $posts_per_page
 *
 * @return WP_Query
 */
function boombox_get_dont_miss_posts_items($conditions, $posts_per_page){

	$query = boombox_get_posts_query( $conditions, 'all', array(), array(
		'posts_per_page' => $posts_per_page
	));

	return $query;
}

/**
 * Get Trending Posts
 *
 * @param $type
 * @param $posts_per_page
 * @param array $params
 * @return bool|WP_Query
 */
function boombox_get_trending_posts( $type, $posts_per_page, $params = array() ){

	$params = wp_parse_args( $params, array(
		'paged' 				=> 1,
		'is_grid' 				=> false,
		'page_ad' 				=> 'none',
		'instead_ad' 			=> 1,
		'page_newsletter' 		=> 'none',
		'instead_newsletter' 	=> 1,
		'page_product'			=> 'none',
		'page_product_position' => 1,
		'page_product_count'    => 1,
		'is_widget' 			=> false,
		'ignore_sticky_posts'	=> true
	) );

	static $boombox_trending_query;

	if( $params['is_widget'] ){
		unset( $boombox_trending_query[ $type ] );
	}

	if( !isset( $boombox_trending_query[ $type ] ) ){
		$post_type     = 'post';
		$query         = null;
		$fake_meta_key = null;
		$criteria_name = boombox_get_theme_option( 'extras_post_ranking_system_trending_conditions' );

		if ( 'trending' == $type ) {
			$time_range  = 'day';
			$posts_count = boombox_get_theme_option( 'extras_post_ranking_system_trending_posts_count' );
			$fake_meta_key = 'boombox_keep_trending';
		} elseif ( 'hot' == $type ) {
			$time_range  = 'week';
			$posts_count = boombox_get_theme_option( 'extras_post_ranking_system_hot_posts_count' );
			$fake_meta_key = 'boombox_keep_hot';
		} elseif ( 'popular' == $type ) {
			$time_range  = 'month';
			$posts_count = boombox_get_theme_option( 'extras_post_ranking_system_popular_posts_count' );
			$fake_meta_key = 'boombox_keep_popular';
		}

		$is_module_active = boombox_module_management_service()->is_module_active( 'prs' );
		if ( $is_module_active && Boombox_Rate_Criteria::get_criteria_by_name( $criteria_name ) && $job =
			Boombox_Rate_Job::get_job_by_name( $type ) ) {
			$args = array(
				'nopaging'            => false,
				'ignore_sticky_posts' => $params['ignore_sticky_posts']
			);

			if( -1 != $posts_per_page ){
				$args['posts_per_page'] = $posts_per_page;
			}else{
				$args['nopaging'] = true;
			}

			if( $params['paged'] ){
				$args['paged'] = $params['paged'];
			}

			$is_adv_enabled = boombox_is_adv_enabled( $params['page_ad'] );
			$is_newsletter_enabled = boombox_is_newsletter_enabled( $params['page_newsletter'] );
			$is_product_enabled = boombox_is_product_enabled( $params['page_product'] );

			if( $is_adv_enabled || $is_newsletter_enabled || $is_product_enabled ){
				Boombox_Loop_Helper::init( array(
					'is_adv_enabled' 		=> $is_adv_enabled,
					'instead_adv' 			=> $params['instead_ad'],
					'is_newsletter_enabled' => $is_newsletter_enabled,
					'instead_newsletter' 	=> $params['instead_newsletter'],
					'is_product_enabled' 	=> $is_product_enabled,
					'page_product_position' => $params['page_product_position'],
					'page_product_count' 	=> $params['page_product_count'],
					'skip' 					=> $params['is_grid'],
					'posts_per_page' 		=> $posts_per_page,
					'paged' 				=> $params['paged']
				) );
			}

			$rate_query = new Boombox_Rate_Query( $args, $job, $fake_meta_key );

			$query = $rate_query->get_wp_query();

		} elseif ( 'most_shared' == $criteria_name || 'recent' === $criteria_name || 'featured' === $criteria_name ) {
			$categories = array();
			$tags       = array();
			$is_page_query = true;
			$excluded_categories = array();
			$is_live = false;
			$query = boombox_get_posts_query( $criteria_name, $time_range, array( 'category' => $categories, 'tag' => $tags ), array(
				'posts_per_page' => $posts_per_page,
				'post_type' => $post_type,
				'paged' => $params['paged'],
				'posts_count' => $posts_count,
				'is_grid' => $params['is_grid'],
				'is_page_query' => $is_page_query,
				'excluded_categories' => $excluded_categories,
				'is_live' => $is_live,
				'fake_meta_key' => $fake_meta_key
			) );
		}

		if ( $query ){
			if( 'trending' == $type){
				$boombox_trending_query[ 'trending' ] = $query;
			}elseif( 'hot' == $type){
				$boombox_trending_query[ 'hot' ] = $query;
			}elseif( 'popular' == $type){
				$boombox_trending_query[ 'popular' ] = $query;
			}
			return $boombox_trending_query[ $type ];
		}

	}else{
		return $boombox_trending_query[ $type ];
	}

	return false;
}

/**
 * Get trending page id by type
 *
 * @param $type 'trending' |'hot' |'popular'
 *
 * @return int|mixed
 */
function boombox_get_trending_page_id( $type ){
	$customize_setting_slug = "extras_post_ranking_system_{$type}_page";
	$trending_page_id = boombox_get_theme_option( $customize_setting_slug );

	return $trending_page_id;
}

/**
 * Return true if is trending page
 *
 * @param $type 'trending' |'hot' |'popular'
 * @param int $post_id
 *
 * @return bool
 */
function boombox_is_trending_page( $type, $post_id = 0 ){
	$is_trending = false;

	if( 0 == $post_id ){
		global $post;
		if( is_object( $post ) ){
			$post_id = $post->ID;
		}
	}

	$post = get_post( $post_id );

	if( $post ){
		$trending_page_id = boombox_get_trending_page_id( $type );
		$is_trending = ( $trending_page_id == $post_id );
	}

	return $is_trending;
}

/**
 * Check, if post is trending
 *
 * @param string $type trending,hot,popular
 * @param int $post_id Post ID
 *
 * @return bool
 */
function boombox_is_post_trending( $type, $post_id ) {
	$time_range = false;
	$posts_count = false;
	$set = boombox_get_theme_options_set( array(
		'extras_post_ranking_system_trending_conditions',
		'extras_post_ranking_system_trending_enable',
		'extras_post_ranking_system_hot_enable',
		'extras_post_ranking_system_popular_enable',
		'extras_post_ranking_system_trending_posts_count',
		'extras_post_ranking_system_hot_posts_count',
		'extras_post_ranking_system_popular_posts_count',
	) );

	if ( ( 'trending' === $type && ! $set[ 'extras_post_ranking_system_trending_enable' ] ) ||
		( 'hot' === $type && ! $set[ 'extras_post_ranking_system_hot_enable' ] ) ||
		( 'popular' === $type && ! $set[ 'extras_post_ranking_system_popular_enable' ] ) ) {
		return false;
	}

	$is_module_active = boombox_module_management_service()->is_module_active( 'prs' );
	if ( $is_module_active && Boombox_Rate_Criteria::get_criteria_by_name( $set[ 'extras_post_ranking_system_trending_conditions' ] ) ) {
		$keep_trending = ! ! ( boombox_get_post_meta( $post_id, 'boombox_keep_' . $type ) );

		return $keep_trending || Boombox_Rate_Cron::is_post_rated( Boombox_Rate_Job::get_job_by_name( $type ), $post_id );
	}

	if ( 'trending' === $type && $set[ 'extras_post_ranking_system_trending_enable' ] ) {
		$time_range = 'day';
		$posts_count = $set[ 'extras_post_ranking_system_trending_posts_count' ];
	} else if ( 'hot' === $type && $set[ 'extras_post_ranking_system_hot_enable' ] ) {
		$time_range = 'week';
		$posts_count = $set[ 'extras_post_ranking_system_hot_posts_count' ];
	} else if ( 'popular' === $type && $set[ 'extras_post_ranking_system_popular_enable' ] ) {
		$time_range = 'month';
		$posts_count = $set[ 'extras_post_ranking_system_popular_posts_count' ];
	}

	$is_trending = false;
	if ( $time_range && $posts_count && in_array( $set[ 'extras_post_ranking_system_trending_conditions' ], array( 'most_shared', 'recent' ) ) ) {
		$query = boombox_get_trending_posts( $type, $posts_count );
		if ( is_object( $query ) && $query->have_posts() ) {
			$trending_ids = wp_list_pluck( $query->posts, 'ID' );
			$is_trending = in_array( $post_id, $trending_ids );
		}
	}

	return $is_trending;
}

/**
 * Return true, if is trending, hot or popular page
 *
 * @param int $post_id Page ID to check
 *
 * @return bool
 */
function boombox_is_trending( $post_id = 0 ){

	$post_id = absint( $post_id );
	if( ! $post_id ) {
		$post_id = get_the_ID();
	}
	$cache_key = 'is_trending_' . $post_id;

	$is_trending = boombox_cache_get( $cache_key );
	if( ! $is_trending ) {

		$is_trending = false;

		if (
			$post_id
			&& (
				boombox_is_trending_page( 'trending', $post_id )
				|| boombox_is_trending_page( 'hot', $post_id )
				|| boombox_is_trending_page( 'popular', $post_id )
			)
		) {
			$is_trending = true;
		}

		boombox_cache_set( $cache_key, $is_trending );

	}

	return $is_trending;
}

/**
 * Get Trending Navigation Items
 *
 * @return array
 */
function boombox_get_trending_navigation_items(){
	$set = boombox_get_theme_options_set( array(
		'extras_badges_trending_icon',
		'extras_post_ranking_system_trending_disable',
		'extras_post_ranking_system_hot_disable',
		'extras_post_ranking_system_popular_disable',
	) );

	$trending_pages     = array(
		'trending' => array(
			'page' => 'trending',
			'icon' => $set['extras_badges_trending_icon'],
			'disabled' => $set['extras_post_ranking_system_trending_disable'],
		),
		'hot' => array(
			'page' => 'hot',
			'icon' => 'hot',
			'disabled' => $set['extras_post_ranking_system_hot_disable'],
		),
		'popular' => array(
			'page' => 'popular',
			'icon' => 'popular',
			'disabled' => $set['extras_post_ranking_system_popular_disable'],
		)
	);
	$trending_pages_nav = array();
	foreach ( $trending_pages as $trending_page_key => $tr_page_options ) {
		$page_id = boombox_get_trending_page_id ( $tr_page_options['page'] );
		if ( $page_id && ! $tr_page_options['disabled'] ) {
			$trending_page = get_post( $page_id );
			if ( $trending_page ) {
				$trending_pages_nav[ $trending_page_key ][ 'id' ] = $page_id;
				$trending_pages_nav[ $trending_page_key ][ 'key' ] = $tr_page_options['page'];
				$trending_pages_nav[ $trending_page_key ][ 'href' ] = get_permalink( $trending_page->ID );
				$trending_pages_nav[ $trending_page_key ][ 'name' ] = esc_html( get_the_title( $trending_page ) );
				$trending_pages_nav[ $trending_page_key ][ 'icon' ] = $tr_page_options[ 'icon' ];
			}

		}
	}

	return $trending_pages_nav;
}

/**
 * Get post reaction settings
 *
 * @param int $post_id
 *
 * @return array
 */
function boombox_get_post_reaction_settings( $post_id ){

	$reaction_total = array();
	$boombox_all_reactions = array();
	$reaction_restrictions = array();
	$reactions_login_require = (bool) boombox_get_theme_option( 'extras_post_reaction_system_login_require' );
	$reaction_item_class     = 'js-reaction-item';
	$authentication_url      = '#';
	$authentication_class    = '';

	if( boombox_module_management_service()->is_module_active( 'prs' ) ) {
		$reaction_total        = Boombox_Reaction_Helper::get_reaction_total( $post_id );
		$boombox_all_reactions = function_exists( 'boombox_get_all_reactions' ) ? boombox_get_all_reactions() : false;
		$reaction_restrictions = Boombox_Reaction_Helper::get_post_reaction_restrictions( $post_id );
	}
	if ( $reactions_login_require == true && ! is_user_logged_in() ) {
		$authentication_class = 'js-authentication';
		$authentication_url   = '#sign-in';
		$reaction_item_class  = '';
	}

	return array(
		'reaction_total'          => $reaction_total,
		'all_reactions'           => $boombox_all_reactions,
		'reaction_restrictions'   => $reaction_restrictions,
		'reactions_login_require' => $reactions_login_require,
		'reaction_item_class'     => $reaction_item_class,
		'authentication_url'      => $authentication_url,
		'authentication_class'    => $authentication_class
	);
}

/**
 * Get Time Range args for query argument
 *
 * @param $time_range
 *
 * @return array
 */
function boombox_get_time_range_args( $time_range ) {
	$args = array();

	if ( $time_range == 'all' || $time_range == '' ){
		return $args;
	}

	$args['date_query'] = array(
		array(
			'after' => sprintf( esc_html__( "1 %s ago", 'boombox' ), $time_range )
		)
	);

	return $args;
}

/**
 * Get categories args for query argument
 *
 * @param $categories
 *
 * @return array
 */
function boombox_categories_args( $categories ) {
	$args = array();
	if ( empty( $categories ) ) {
		return $args;
	}
	if ( ! is_array( $categories ) || 0 == count( $categories ) || '' == $categories[0] ) {
		return $args;
	}

	$args = array(
		'taxonomy'          => 'category',
		'field'             => 'slug',
		'terms'             => $categories,
		'operator'          => 'IN',
		'include_children'  => false
	);

	return apply_filters( 'boombox/categories-args', $args );
}

/**
 * Get tags args for query argument
 *
 * @param $tags
 *
 * @return array
 */
function boombox_tags_args( $tags ) {
	$args = array();
	if ( empty( $tags ) ) {
		return $args;
	}
	if( is_string( $tags ) ) {
		$tags = explode( ',', preg_replace('/\s+/', '', sanitize_text_field( $tags ) ) );
	}
	if ( 0 == count( $tags ) || '' == $tags[0] ) {
		return $args;
	}

	$args = array(
		'taxonomy' => 'post_tag',
		'field'    => 'slug',
		'terms'    => $tags,
		'operator' => 'IN',
		'include_children'  => false
	);

	return apply_filters( 'boombox/tags-args', $args );
}

/**
 * Get categories args for query argument
 *
 * @param $reactions
 *
 * @return array
 */
function boombox_reactions_args( $reactions ) {
	$args = array();
	if ( empty( $reactions ) ) {
		return $args;
	}
	if ( ! is_array( $reactions ) || 0 == count( $reactions ) || '' == $reactions[0] ) {
		return $args;
	}

	$args = array(
		'taxonomy'          => 'reaction',
		'field'             => 'slug',
		'terms'             => $reactions,
		'operator'          => 'IN',
		'include_children'  => false
	);

	return apply_filters( 'boombox/reactions-args', $args );
}

/**
 * Get post types args for query
 *
 * @param $post_types
 *
 * @return array
 */
function boombox_get_post_types_args ( $post_types ){

	if( ! is_string( $post_types ) && ! is_array( $post_types ) ) {
		$post_types  = array( 'post' );
	}

	if( is_string( $post_types ) ){
		$post_types = explode( ',', preg_replace('/\s+/', '', $post_types ) );
	}

	if( empty( $post_types ) ) {
		$post_types  = array( 'post' );
	}

	return $post_types;
}

/**
 * Get post first category
 *
 * @param $post
 * @return bool
 */
function boombox_get_post_first_category( $post ) {
	if( ! $post ){
		global $post;
	}
	$post_id = $post->ID;

	$post_categories = get_the_category( $post_id );

	if( ! empty( $post_categories ) ){
		$post_first_category = $post_categories[0];

		return $post_first_category;
	}

	return false;
}

/**
 * Add additional classes to badges warpper element
 *
 * @param string $classes  Current classes
 * @param string $taxonomy Term Taxonomy
 * @param int    $term_id  Term ID
 * @return string
 */
function boombox_add_additional_badge_classes( $classes, $taxonomy, $term_id ) {
	if( in_array( $taxonomy, array( 'reaction', 'category', 'post_tag' ) ) ) {
		$classes .= sprintf( ' %1$s-%2$d', $taxonomy, $term_id);
	}
	return $classes;
}
add_filter( 'boombox_badge_wrapper_advanced_classes', 'boombox_add_additional_badge_classes', 10, 3 );

/**
 * Edit comment reply URL
 *
 * @param string $link Current URL
 * @param array $args Args @see get_comment_reply_link()
 * @return string
 */
function boombox_edit_comment_reply_url( $link, $args ) {
	if ( get_option( 'comment_registration' ) && ! is_user_logged_in() ) {
		$link = sprintf( '<a rel="nofollow" class="comment-reply-login js-authentication" href="#sign-in">%s</a>',
			$args['login_text']
		);
	}

	return $link;
}
add_filter( 'comment_reply_link', 'boombox_edit_comment_reply_url', 10, 2 );

/**
 * Show post badge list
 *
 * @param array $args {
 *      Optional. Override default arguments.
 *
 * @type int        $post_id                    Post ID
 * @type bool       $badges                     Show badges
 * @type int|string $badges_count               Badges count
 * @type string     $badges_before              HTML before badges
 * @type string     $badges_after               HTML after badges
 * @type bool       $post_type_badges           Show post type badges
 * @type int|string $post_type_badges_count     Post type badges count
 * @type string     $post_type_badges_before    HTML before Post type badges
 * @type string     $post_type_badges_after     HTML after Post type badges
 * @type bool       $is_amp                     Is currently AMP version rendered
 * }
 * @return array|null
 */
function boombox_get_post_badge_list( $args = '' ) {

	$args = wp_parse_args( $args, array(
		'post_id' 					=> 0,
		'badges' 					=> true,
		'badges_count' 				=> 2,
		'badges_before'				=> '<div class="bb-badge-list">',
		'badges_after'				=> '</div>',
		'post_type_badges'			=> true,
		'post_type_badges_count' 	=> 1,
		'post_type_badges_before'	=> '<div class="bb-badge-list boombox-format">',
		'post_type_badges_after'	=> '</div>',
		'is_amp'                    => boombox_is_amp()
	) );

	$return = array(
		'badges'            => '',
		'post_type_badges'  => ''
	);

	// if all badges are disabled
	$hide_badges_option = boombox_hide_badges_option();
	$show_post_type_badges = boombox_get_theme_option( 'extras_badges_post_type_badges' );
	if ( $hide_badges_option['hide_trending_badges'] && $hide_badges_option['hide_category_badges'] &&
		$hide_badges_option['hide_reactions_badges'] && ! $show_post_type_badges ) {
		return $return;
	}

	if( 0 == $args['post_id'] ){
		global $post;
		if( is_object( $post ) ){
			$args['post_id'] = $post->ID;
		}
	}

	$post = get_post( $args['post_id'] );

	if ( 'post' !== get_post_type() || ! $post ) {
		return $return;
	}

	$args['post_id'] = $post->ID;
	$hash = md5( json_encode( $args ) );

	/***** return data from cache if we already have it */
	if( $cached = boombox_cache_get( $hash ) ) {
		return $cached;
	}

	$badges         			= array();
	$post_type_badges  			= array();
	$badges_counter 			= 0;
	$post_type_badges_counter 	= 0;

	// trending badge
	if ( ! $hide_badges_option['hide_trending_badges'] && $args['badges'] ) {
		$trending_types = array(
			'trending' => esc_html__( 'Trending', 'boombox' ),
			'hot' => esc_html__( 'Hot', 'boombox' ),
			'popular' => esc_html__( 'Popular', 'boombox' )
		);

		foreach ( $trending_types as $trending_type => $trending_name ) {
			$show_badge  = (bool) boombox_get_theme_option( "extras_post_ranking_system_{$trending_type}_badge" );
			$is_trending = boombox_is_post_trending( $trending_type, $args['post_id'] );
			if ( $show_badge && $is_trending ) {
				$trending_page_id                   = boombox_get_trending_page_id( $trending_type );
				$trending_icon_name                 = boombox_get_trending_icon_name( 'icon', $trending_type );
				$badges[ $badges_counter ]['name']  = ucfirst( $trending_name );
				$badges[ $badges_counter ]['icon']  = ! empty( $trending_icon_name ) ? '<i class="icon icon-' . $trending_icon_name . '"></i>' : '';
				$badges[ $badges_counter ]['link']  = get_permalink( $trending_page_id );
				$badges[ $badges_counter ]['class'] = esc_attr( 'trending' );
				$badges[ $badges_counter ]['taxonomy'] 	= 'trending';
				$badges[ $badges_counter ]['term_id'] 	= '';
				if( $args['is_amp'] ) {
					$badges[ $badges_counter ]['amp'] = array(
						'icon_type' => 'icon',
						'icon'      => ! empty( $trending_icon_name ) ? $trending_icon_name : ''
					);
				}
				++ $badges_counter;
				break;
			}
		}
	}

	$post_categories = $post_tags = array();
	if( ( ! $hide_badges_option['hide_category_badges'] && $args['badges'] ) || ( $show_post_type_badges && $args['post_type_badges'] ) ) {
		$post_categories = get_the_category( $args['post_id'] );
		$post_tags = wp_get_post_tags( $args['post_id'] );
	}

	if ( ! empty( $post_categories ) ) {

		$post_categories_slugs = wp_list_pluck( $post_categories, 'slug' );

		if ( ! $hide_badges_option['hide_category_badges'] && $args['badges'] ) {
			// categories badges
			$categories_with_badges = boombox_get_categories_with_badges( 'extras_badges_show_for_categories' );

			foreach ( $categories_with_badges as $key => $categories_with_badge ) {
				if ( in_array( $key, $post_categories_slugs ) ) {
					$badges[ $badges_counter ] = $categories_with_badge;
					++ $badges_counter;
					if( 2 == $key + 1 ){
						break;
					}
				}
			}

		}

		if( $show_post_type_badges && $args['post_type_badges'] ) {
			// post type badges categories badges
			$post_type_badges_categories_with_badges = boombox_get_categories_with_badges( 'extras_badges_categories_for_post_type_badges' );

			foreach ( $post_type_badges_categories_with_badges as $key => $post_type_badges_categories_with_badge ) {
				if ( in_array( $key, $post_categories_slugs ) ) {
					$post_type_badges[ $post_type_badges_counter ] = $post_type_badges_categories_with_badge;
					++ $post_type_badges_counter;
					break;
				}
			}
		}
	}

	if ( ! empty( $post_tags ) ) {

		$post_tags_slugs = wp_list_pluck( $post_tags, 'slug' );

		if ( ! $hide_badges_option['hide_category_badges'] && $args['badges'] ) {
			// post tag badges
			$post_tags_with_badges = boombox_get_post_tags_with_badges( 'extras_badges_show_for_post_tags' );

			foreach ( $post_tags_with_badges as $key => $post_tags_with_badge ) {
				if ( in_array( $key, $post_tags_slugs ) ) {
					$badges[ $badges_counter ] = $post_tags_with_badge;
					++ $badges_counter;
					break;
				}
			}
		}

		if( $show_post_type_badges && $args['post_type_badges'] ) {
			// post type badges post tag badges
			$post_type_badges_post_tags_with_badges = boombox_get_post_tags_with_badges( 'extras_badges_post_tags_for_post_type_badges' );

			foreach ( $post_type_badges_post_tags_with_badges as $key => $post_type_badges_post_tags_with_badge ) {
				if ( in_array( $key, $post_tags_slugs ) ) {
					$post_type_badges[ $post_type_badges_counter ] = $post_type_badges_post_tags_with_badge;
					++ $post_type_badges_counter;
					break;
				}
			}
		}

	}

	// reactions badges
	if( ! $hide_badges_option['hide_reactions_badges'] && ! is_tax( 'reaction' ) && $args['badges'] ){
		$reactions = boombox_get_post_reactions( $args['post_id'] );
		if( is_array( $reactions ) && count( $reactions ) > 0 ) {

			foreach ( $reactions as $key => $reaction ) {
				$reaction_icon_url = boombox_get_reaction_icon_url( $reaction->term_id );
				$badges[ $badges_counter ]['name']  	= $reaction->name;
				$badges[ $badges_counter ]['icon']  	= ! empty( $reaction_icon_url ) ? ' <img src="' . esc_url( $reaction_icon_url ) . '" alt="' . $reaction->name . '">' : '';
				$badges[ $badges_counter ]['link']  	= get_term_link( $reaction->term_id );
				$badges[ $badges_counter ]['class'] 	= $reaction->taxonomy;
				$badges[ $badges_counter ]['taxonomy'] 	= $reaction->taxonomy;
				$badges[ $badges_counter ]['term_id'] 	= $reaction->term_id;
				if( $args['is_amp'] ) {
					$badges[ $badges_counter ]['amp'] = array(
						'icon_type' => 'image',
						'icon'      => ! empty( $reaction_icon_url ) ? esc_url( $reaction_icon_url ) : ''
					);
				}
				++$badges_counter;
				if( 2 == $key + 1 ){
					break;
				}
			}
		}
	}

	$badges_return = '';
	$post_type_badges_return = '';

	if( ! empty( $badges ) ) {
		// for "You may also like", "More From" and "Don't miss" sections on post single page
		$args['badges_count'] = ( is_single() && 2 == $args['badges_count'] ) ? 1 : $args['badges_count'];
		$badges = array_slice( $badges, 0, $args['badges_count'] );

		if( $args['is_amp'] ) {
			$badges_return = $badges;
		} else {
			$badges_return .= $args['badges_before'];
			foreach ( $badges as $badge_key => $badge ) {
				$badge_class = apply_filters( 'boombox_badge_wrapper_advanced_classes', esc_attr( $badge['class'] ), $badge['taxonomy'], $badge['term_id'] );
				$badge_url = esc_url( $badge['link'] );
				$badge_title = esc_html( $badge['name'] );
				$badge_icon = wp_kses_post( $badge['icon'] );

				$badges_return .= sprintf(
					'<a class="bb-badge badge %1$s" href="%2$s" title="%3$s"><span class="circle">%4$s</span><span class="text">%3$s</span></a>',
					$badge_class,
					$badge_url,
					$badge_title,
					$badge_icon
				);
			}
			$badges_return .= $args['badges_after'];
		}
	}

	if( ! empty( $post_type_badges ) ) {
		$post_type_badges = array_slice( $post_type_badges, 0, $args['post_type_badges_count'] );

		if( $args['is_amp'] ) {
			$post_type_badges_return = $post_type_badges;
		} else {
			$post_type_badges_return .= $args['post_type_badges_before'];
			foreach ( $post_type_badges as $badge_key => $post_type_badge ) {
				$badge_class = sprintf( 'category format-%d', $post_type_badge['term_id'] );
				$badge_title = esc_html( $post_type_badge['name'] );
				$badge_icon = wp_kses_post( $post_type_badge['icon'] );

				$post_type_badges_return .= sprintf(
					'<span class="bb-badge badge %1$s" title="%2$s"><span class="circle">%3$s</span><span class="text">%2$s</span></span>',

					$badge_class,
					$badge_title,
					$badge_icon
				);
			}
			$post_type_badges_return .= $args['post_type_badges_after'];
		}
	}

	$return['badges'] = $badges_return;
	$return['post_type_badges'] = $post_type_badges_return;

	boombox_cache_set( $hash, $return );

	return $return;

}

/**
 * Hide badges option
 *
 * @return mixed
 */
function boombox_hide_badges_option(){
	$cache_key = 'hide_badges';
	$hide_badges = boombox_cache_get( $cache_key );

	if( ! $hide_badges ){
		$set = boombox_get_theme_options_set( array(
			'extras_badges_trending',
			'extras_badges_category',
			'extras_badges_reactions'
		) );
		$hide_badges = array(
			'hide_trending_badges'  => ! $set['extras_badges_trending'],
			'hide_category_badges'  => ! $set['extras_badges_category'],
			'hide_reactions_badges' => ! $set['extras_badges_reactions']
		);

		boombox_cache_set( $cache_key, $hide_badges );
	}

	return apply_filters( 'boombox/hide-badges', $hide_badges );
}

/**
 * Get categories with badges
 *
 * @return array
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_categories_with_badges( $theme_option ) {
	$categories_with_badges = array();
	$categories = boombox_get_theme_option( $theme_option );
	foreach ( $categories as $category ) {
		$category = get_term_by( 'slug', $category, 'category' );
		if ( $category ) {
			$term_icon = boombox_get_term_icon_html( $category->term_id, $category->name, $category->taxonomy );

			if ( $term_icon ) {
				$categories_with_badges[ $category->slug ] = array(
					'term_id'  => $category->term_id,
					'taxonomy' => 'category',
					'name'     => $category->name,
					'link'     => esc_url( get_term_link( $category->term_id ) ),
					'icon'     => $term_icon,
					'class'    => 'category',
				);
			}
		}
	}

	return $categories_with_badges;
}

/**
 * Get post_tags with badges
 *
 * @return array
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_post_tags_with_badges( $theme_option ) {
	$post_tags_with_badges = array();
	$post_tags = boombox_get_theme_option( $theme_option );
	foreach ( (array)$post_tags as $post_tag ) {
		$post_tag = get_term_by( 'slug', $post_tag, 'post_tag' );
		if ( $post_tag ) {
			$term_icon = boombox_get_term_icon_html( $post_tag->term_id, $post_tag->name, $post_tag->taxonomy );
			if ( $term_icon ) {
				$post_tags_with_badges[ $post_tag->slug ] = array(
					'term_id'  => $post_tag->term_id,
					'taxonomy' => 'post_tag',
					'name'     => $post_tag->name,
					'link'     => esc_url( get_term_link( $post_tag->term_id ) ),
					'icon'     => $term_icon,
					'class'    => 'post_tag',
				);
			}
		}
	}

	return $post_tags_with_badges;
}

/**
 * @param $key 'trending' || 'hot' || 'popular'
 *
 * @return mixed|string
 */
function get_trending_icon_by_key( $key ) {
	switch ( $key ) {
		case 'trending':
			$icon = boombox_get_theme_option('extras_badges_trending_icon');
			break;
		case 'hot':
			$icon = 'hot';
			break;
		case 'popular':
			$icon = 'popular';
			break;
		default:
			$icon = '';
	}

	return $icon;
}

/**
 * @param $get_by 'post_id' or 'icon'
 * @param $key 'trending_TYPE_post_id' or 'icon_key' ( 'trending', 'hot', 'popular' )
 *
 * @return mixed|string
 */
function boombox_get_trending_icon_name( $get_by, $key ){
	$icon_key = '';

	if( $get_by == 'post_id' ) {
		if( boombox_is_trending_page( 'trending', $key ) ){
			$icon_key = 'trending';
		} elseif( boombox_is_trending_page( 'hot', $key ) ){
			$icon_key = 'hot';
		} elseif( boombox_is_trending_page( 'popular', $key ) ){
			$icon_key = 'popular';
		}
	} else if( $get_by == 'icon' ) {
		$icon_key = $key;
	}

	return $icon_key ? get_trending_icon_by_key( $icon_key ) : '';
}

/**
 * Get Term icon name
 *
 * @param $term_id
 * @return string
 */
function boombox_get_term_icon_name( $term_id, $taxonomy ){
	switch( $taxonomy ) {
		case 'category':
			$meta_key = 'cat_icon_name';
			break;
		case 'post_tag':
			$meta_key = 'post_tag_icon_name';
			break;
		default:
			$meta_key = false;
	}
	if( $meta_key ) {
		$cat_icon_name = sanitize_text_field( boombox_get_term_meta( $term_id, $meta_key ) );
		if( $cat_icon_name ){
			return $cat_icon_name;
		}
	}
	return '';
}

function boombox_get_term_custom_image_icon( $term_id ) {
	$term_image_icon_url = false;
	$term_image_icon_id = boombox_get_term_meta( $term_id, 'term_image_icon_id' );

	if( $term_image_icon_id ) {
		$term_image_icon_url = wp_get_attachment_thumb_url( $term_image_icon_id );
	}

	return $term_image_icon_url;
}

function boombox_get_term_icon_html( $id, $name, $tax ) {
	$html = false;

	if( $term_image_icon_url = boombox_get_term_custom_image_icon( $id ) ) {
		$html = sprintf( '<img src="%1$s" alt="%2$s" />', $term_image_icon_url, $name );
	} elseif( $icon_name = boombox_get_term_icon_name( $id, $tax ) ) {
		$html = sprintf('<i class="icon icon-%1$s"></i>', $icon_name );
	}

	return $html;
}

/**
 * Get single post reactions
 *
 * @param int $post_id Post ID
 * @return array|null
 */
function boombox_get_post_reactions( $post_id ){
	$reactions = NULL;
	$is_module_active = boombox_module_management_service()->is_module_active( 'prs' );

	if( $is_module_active && function_exists( 'boombox_get_reaction_taxonomy_name' ) ) {
		$reactions = array();
		$reactions_ids = Boombox_Reaction_Helper::get_post_reactions( $post_id );
		if( ! empty( $reactions_ids ) ){
			$taxonomy = boombox_get_reaction_taxonomy_name();
			foreach( $reactions_ids as $reaction_id ){
				$reaction = get_term_by( 'term_id', $reaction_id, $taxonomy );
				if( $reaction ){
					$reactions[] = $reaction;
				}
			}
		}
	}

	return $reactions;
}



/**
 * Get list type classes
 *
 * @param string $list_type             Current listing type
 * @param array  $add_grid_class        Additional classes for grid type layouts
 * @param array $additional_classes     Additional classes for all listings types
 * @return array|string
 */
function boombox_get_list_type_classes( $list_type, $add_grid_class = array(), $additional_classes = array() ) {
	$classes = array();

	switch ( $list_type ) {
		case 'grid':
		case 'grid-2-1':
			$classes = array_merge( array( 'post-grid' ), (array)$add_grid_class );
			break;

		case 'four-column':
			$classes = array( 'post-grid', 'col-3' );
			break;

		case 'list':
			$classes = array( 'post-list', 'list', 'big-item' );
			break;

		case 'list2':
			$classes = array( 'post-list', 'list' );
			break;

		case 'classic':
		case 'classic2':
		case 'stream':
			$classes = array( 'post-list', 'standard' );
			break;
		case 'masonry-boxed':
		case 'masonry-stretched':
			$classes = array( 'masonry-grid' );
			break;
	}


	$classes = apply_filters( 'boombox/list-type-classes', array_merge( $classes, $additional_classes ), $list_type );

	if ( $classes ) {
		array_unshift( $classes, 'hfeed' );
		$classes = esc_attr( implode( ' ', $classes ) );
	}

	return $classes;
}

/**
 * Render listing container classes by listing type
 *
 * @param string $list_type           Listing type
 * @param string $additional_position Additional position
 *
 * @return string
 */
function boombox_get_container_classes_by_type( $list_type, $additional_position = '' ) {
	$classes = '';
	switch ( $list_type ) {
		case 'stream':
			$classes = 'bb-stream';
			break;
		case 'masonry-stretched':
			$classes = 'bb-stretched-full';
			break;
	}

	return $classes;
}

/**
 * Get Term Custom Thumbnail URL
 *
 * @return string|null
 */
function boombox_get_term_thumbnail_url(){
	$src = NULL;
	$queried_object = get_queried_object();
	if( $queried_object ){
		switch( $queried_object->taxonomy ) {
			case 'category':
				$meta_key = 'cat_thumbnail_id';
				break;
			case 'post_tag':
				$meta_key = 'post_tag_thumbnail_id';
				break;
			default:
				$meta_key = '';
		}
		$thumbnail_id = absint( boombox_get_term_meta( $queried_object->term_id, $meta_key ) );
		if ( $thumbnail_id ) {
			$image = wp_get_attachment_image_src( $thumbnail_id, 'image1600' );
			if( $image ){
				$src = $image[0];
			}
		}
	}
	return $src;
}

/**
 * Generate compose button html
 *
 * @param array  $classes    CSS classes
 * @param string $label      Button label
 * @param bool   $icon       Whether to use the icon
 * @param bool   $static_url Constant URL for the button
 *
 * @return string
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_create_post_button( $classes = array(), $label = '', $icon = false, $static_url = false ) {

	if ( $static_url ) {
		$args = array(
			'url'     => apply_filters( 'boombox/create_post_button/static_url', $static_url ),
			'classes' => $classes,
		);
	} else {
		if ( is_user_logged_in() ) {
			$ssl = is_ssl() ? 'https' : 'http';
			$url = admin_url( 'post-new.php', $ssl );
		} else {
			$url = esc_url( '#sign-in' );
			$classes[] = 'js-authentication';
		}

		$args = apply_filters( 'boombox/create-post-button-args', array(
			'url'     => $url,
			'classes' => $classes,
		) );

		$args[ 'url' ] = isset( $args[ 'url' ] ) ? $args[ 'url' ] : $url;
		$args[ 'classes' ] = isset( $args[ 'classes' ] ) ? (array)$args[ 'classes' ] : array();
	}

	if ( $icon ) {
		array_unshift( $args[ 'classes' ], 'icon' );
	}

	$button = sprintf( '<a class="%1$s" href="%2$s">%3$s %4$s</a>',
		esc_attr( implode( ' ', $args[ 'classes' ] ) ),
		esc_url( $args[ 'url' ] ),
		$icon ? '<i class="icon-plus_bb"></i>' : '',
		$label ? $label : esc_html__( 'Create a post', 'boombox' )
	);

	return $button;
}

/**
 * Return Log Out Button
 *
 * @return string
 */
function boombox_get_logout_button() {
	if ( is_user_logged_in() ) {
		$url     = wp_logout_url( home_url() );
		$title   = esc_attr__( 'Log Out', 'boombox' );
		$classes = esc_attr( implode( ' ', array( 'user', 'icon-sign-out' ) ) );

		return sprintf( '<a class="%1$s" href="%2$s" title="%3$s"></a>',
			$classes,
			$url,
			$title
		);
	}

	return '';
}

/**
 * Return point classes
 *
 * @param $post_id
 *
 * @return array
 */
function boombox_get_point_classes( $post_id ){
	$classes = array(
		'up' => '',
		'down' => ''
	);

	if( boombox_module_management_service()->is_module_active( 'prs' ) ) {
		if ( Boombox_Point_Count_Helper::pointed_up( $post_id ) ) {
			$classes[ 'up' ] = 'active';
		} else if ( Boombox_Point_Count_Helper::pointed_down( $post_id ) ) {
			$classes[ 'down' ] = 'active';
		}
	}

	return $classes;
}

/**
 * Return post point count
 *
 * @param $post_id
 *
 * @return int
 */
function boombox_get_post_point_count( $post_id ){
	$points_count = 0;
	if( boombox_module_management_service()->is_module_active( 'prs' ) ) {
		$points_count = Boombox_Point_Count_Helper::get_post_points( $post_id );
	}
	return $points_count;
}

/**
 * Return views count
 *
 * @param $post_id
 *
 * @return int
 */
function boombox_get_views_count( $post_id ){
	$views_count = 0;
	if( boombox_module_management_service()->is_module_active( 'prs' ) ) {
		$views_count = Boombox_View_Count_Helper::get_post_views( $post_id );
	}

	return $views_count;
}

/**
 * Show advertisement
 *
 * @param string $location   Ad location
 * @param array  $args  Adv arguments
 */
function boombox_the_advertisement( $location, $args = array() ) {
	if( ! boombox_plugin_management_service()->is_plugin_active('quick-adsense-reloaded/quick-adsense-reloaded.php') ) {
		return;
	}

	$args = wp_parse_args( $args, array(
		'class' => array(),
		'tag'   => 'div',
		'tmp_query' => false,
		'cur_query' => false,
		'before' => '',
		'after'  => '',
	) );

	$hide_ads = false;
	if( is_singular() ){
		$config = boombox_get_post_meta( get_the_ID(), '_quads_config_visibility' );
		if( isset( $config['NoAds'] ) && $config['NoAds'] ){
			$hide_ads = true;
		}
	}
	$hide_ads = apply_filters( 'boombox/single_post/hide_ads', $hide_ads, $location );

	global $wp_query;
	if( $args['tmp_query'] ) {
		$wp_query = $args['tmp_query'];
	}

	if( function_exists('quads_ad') && $location && ! $hide_ads && quads_has_ad( $location ) ) {

		$adv = quads_ad( array( 'location' => $location, 'echo' => false ) );
		if( ! $adv ) {
			$adv = sprintf(
				'<div class="massage">%s</div>',
				esc_html__( 'There are no ads set to this area or maximum limit of ads on a single page has been reached', 'boombox' )
			);
		}

		if( is_array( $args['class'] ) ) {
			$args['class'] = trim( implode(' ', $args['class'] ) );
		}
		$args['class'] = 'advertisement ' . $args['class'];

		echo $args['before']; ?>
		<<?php echo $args['tag']; ?> class="<?php echo esc_attr( $args['class'] ); ?>">
			<div class="inner"><?php echo $adv; ?></div>
		</<?php echo $args['tag']; ?>>
		<?php
		echo $args['after'];

	}

	if( $args['cur_query'] ) {
		$wp_query = $args['cur_query'];
	}
}

/**
 * Return advertisement settings
 *
 * @param string $listing_type Current listing type
 * @return array
 */
function boombox_get_adv_settings( $listing_type ){
	if( in_array( $listing_type, array( 'grid', 'four-column' ) ) ) {
		$size = '';
		$location = 'boombox-listing-type-grid-instead-post';
	} else {
		$size = 'large';
		$location = 'boombox-listing-type-non-grid-instead-post';
	}
	return array(
		'size' => $size,
		'location' => $location
	);
}

/**
 * Check if advertisement inject is enabled
 *
 * @param string $status Current status
 * @return bool
 */
function boombox_is_adv_enabled( $status ){
	return (
		boombox_plugin_management_service()->is_plugin_active( 'quick-adsense-reloaded/quick-adsense-reloaded.php' )
		&& 'inject_into_posts_list' == $status
	);
}

/**
 * Check if newsletter inject is enabled
 *
 * @param string $status Current status
 * @return bool
 */
function boombox_is_newsletter_enabled( $status ){
	return (
		boombox_plugin_management_service()->is_plugin_active( 'mailchimp-for-wp/mailchimp-for-wp.php' )
		&& 'inject_into_posts_list' == $status
	);
}

/**
 * Check if product inject is enabled
 *
 * @param string $status Current status
 * @return bool
 */
function boombox_is_product_enabled( $status ) {
	return (
		boombox_plugin_management_service()->is_plugin_active( 'woocommerce/woocommerce.php' )
		&& 'inject_into_posts_list' == $status
	);
}

/**
 * Check whether post has thumbnail
 *
 * @param int|WP_Post $post Optional post
 * @return bool
 */
function boombox_has_post_thumbnail( $post = null ) {
	return (bool)apply_filters( 'boombox/post/has-thumbnail', has_post_thumbnail( $post ), $post );
}

/**
 * Get next attachment post for attachment template
 *
 * @return array|bool|mixed|null|string|WP_Post
 */
function boombox_get_next_attachment_post() {
	global $wpdb;
	$post = get_post();

	$sql = $wpdb->prepare( 'SELECT
              p.`ID`
            FROM
              `' . $wpdb->posts . '` AS p
              LEFT JOIN `' . $wpdb->posts . '` AS `p1`
                ON `p1`.`ID` = p.`post_parent`
            WHERE p.`post_type` = %s
              AND p.post_date > %s
              AND ( p1.`post_status` IS NULL OR p1.`post_status` = %s', 'attachment', $post->post_date, 'publish' );

	if ( is_user_logged_in() ) {
		$user_id = get_current_user_id();

		$post_type_object = get_post_type_object( $post->post_type );
		if ( empty( $post_type_object ) ) {
			$post_type_cap    = $post->post_type;
			$read_private_cap = 'read_private_' . $post_type_cap . 's';
		} else {
			$read_private_cap = $post_type_object->cap->read_private_posts;
		}

		/*
		 * Results should include private posts belonging to the current user, or private posts where the
		 * current user has the 'read_private_posts' cap.
		 */
		$private_states = get_post_stati( array( 'private' => true ) );
		foreach ( (array) $private_states as $state ) {
			if ( current_user_can( $read_private_cap ) ) {
				$sql .= $wpdb->prepare( " OR p1.post_status = %s", $state );
			} else {
				$sql .= $wpdb->prepare( " OR (p.post_author = %d AND p1.post_status = %s)", $user_id, $state );
			}
		}
	}

	$sql .= ") ORDER BY p.post_date ASC LIMIT 1";

	$query_key = 'adjacent_post_' . md5( $sql );
	$result = wp_cache_get( $query_key, 'counts' );
	if ( false !== $result ) {
		if ( $result )
			$result = get_post( $result );
		return $result;
	}

	$result = $wpdb->get_var( $sql );
	if ( null === $result ) {
		$result = '';
	}

	wp_cache_set( $query_key, $result, 'counts' );

	if ( $result ) {
		$result = get_post( $result );

	}

	return $result;

}

/**
 * Get previous attachment post for attachment template
 *
 * @return array|bool|mixed|null|string|WP_Post
 */
function boombox_get_previous_attachment_post() {
	global $wpdb;
	$post = get_post();

	$sql = $wpdb->prepare( 'SELECT
              p.`ID`
            FROM
              `' . $wpdb->posts . '` AS p
              LEFT JOIN `' . $wpdb->posts . '` AS `p1`
                ON `p1`.`ID` = p.`post_parent`
            WHERE p.`post_type` = %s
              AND p.post_date < %s
              AND ( p1.`post_status` IS NULL OR p1.`post_status` = %s', 'attachment', $post->post_date, 'publish' );

	if ( is_user_logged_in() ) {
		$user_id = get_current_user_id();

		$post_type_object = get_post_type_object( $post->post_type );
		if ( empty( $post_type_object ) ) {
			$post_type_cap    = $post->post_type;
			$read_private_cap = 'read_private_' . $post_type_cap . 's';
		} else {
			$read_private_cap = $post_type_object->cap->read_private_posts;
		}

		/*
		 * Results should include private posts belonging to the current user, or private posts where the
		 * current user has the 'read_private_posts' cap.
		 */
		$private_states = get_post_stati( array( 'private' => true ) );
		foreach ( (array) $private_states as $state ) {
			if ( current_user_can( $read_private_cap ) ) {
				$sql .= $wpdb->prepare( " OR p1.post_status = %s", $state );
			} else {
				$sql .= $wpdb->prepare( " OR (p.post_author = %d AND p1.post_status = %s)", $user_id, $state );
			}
		}
	}

	$sql .= ") ORDER BY p.post_date DESC LIMIT 1";

	$query_key = 'adjacent_post_' . md5( $sql );
	$result = wp_cache_get( $query_key, 'counts' );
	if ( false !== $result ) {
		if ( $result )
			$result = get_post( $result );
		return $result;
	}

	$result = $wpdb->get_var( $sql );
	if ( null === $result ) {
		$result = '';
	}

	wp_cache_set( $query_key, $result, 'counts' );

	if ( $result ) {
		$result = get_post( $result );
	}

	return $result;
}

if ( ! function_exists( 'boombox_post_date' ) ) {

	/**
	 * Render HTML with date information for current post.
	 *
	 * @param array $args
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_date( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'echo'    => true,
			'display' => true,
		) );

		$html = '';

		$wrapper_classes = array( 'posted-on' );
		if ( ! $args[ 'display' ] ) {
			$wrapper_classes[] = 'mf-hide';
		}

		$microdata = array( 'entry-date/updated' => '', 'entry-date/published' => '' );
		if ( is_single() && get_the_ID() == get_queried_object_id() ) {
			$microdata[ 'entry-date/updated' ] = 'itemprop="dateModified"';
			$microdata[ 'entry-date/published' ] = 'itemprop="datePublished"';
		}
		$time_string = sprintf( '<time class="entry-date published" %2$s datetime="%1$s">%3$s</time><time class="entry-date updated mf-hide" %5$s datetime="%4$s">%6$s</time>',

			esc_attr( get_the_date( 'c' ) ),            /* 1 */
			$microdata[ 'entry-date/published' ],         /* 2 */
			apply_filters( 'boombox_post_date', human_time_diff( get_the_time( 'U' ), current_time( 'timestamp' ) ) . " " . esc_html__( 'ago', 'boombox' ) ), /* 3 */

			esc_attr( get_the_modified_date( 'c' ) ),   /* 4 */
			$microdata[ 'entry-date/updated' ],           /* 5 */
			apply_filters( 'boombox_post_date', human_time_diff( get_the_modified_time( 'U' ), current_time( 'timestamp' ) ) . " " . esc_html__( 'ago', 'boombox' ) ) /* 6 */

		);

		$html .= sprintf( '<span class="%1$s">%2$s</span>',
			implode( ' ', $wrapper_classes ),
			$time_string
		);

		if ( $args[ 'echo' ] ) {
			echo $html;
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'boombox_post_author_meta' ) ) {

	/**
	 * Create a meta box for the post author with author and date data
	 *
	 * @param array $args
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_author_meta( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'before'      => '<div class="post-author-meta">',
			'after'       => '</div>',
			'author'      => true,
			'author_args' => array(),
			'date'        => true,
			'date_args'   => array(),
			'echo'        => true,
		) );

		$args = apply_filters( 'boombox/post-author-meta-args', $args );

		$html = '';
		$html .= boombox_post_author( array_merge( (array)$args[ 'author_args' ], array( 'echo' => false, 'display' => $args[ 'author' ] ) ) );
		$html .= boombox_post_date( array_merge( (array)$args[ 'date_args' ], array( 'echo' => false, 'display' => $args[ 'date' ] ) ) );

		$args[ 'before' ] = ( $args[ 'author' ] || $args[ 'date' ] ) ? $args[ 'before' ] : '';
		$args[ 'after' ] = ( $args[ 'author' ] || $args[ 'date' ] ) ? $args[ 'after' ] : '';

		$html = $args[ 'before' ] . apply_filters( 'boombox/post-author-meta-html', $html ) . $args[ 'after' ];

		if ( $args[ 'echo' ] ) {
			echo $html;
		} else {
			return $html;
		}

	}

}

if ( ! function_exists( 'boombox_post_author' ) ) {
	/**
	 * Get / Render  HTML with author information for current post.
	 *
	 * @param array $args
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_author( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'display'        => true,
			'with_avatar'    => false,
			'post_author_id' => false,
			'with_desc'      => false,
			'microdata'      => false,
			'microformat'    => true,
			'echo'           => true,
		) );

		$html = '';

		$author_avatar_size = apply_filters( 'boombox_author_avatar_size', 74 );
		$author_id = $args[ 'post_author_id' ] ? $args[ 'post_author_id' ] : get_the_author_meta( 'ID' );
		$author_name = wp_kses_post( get_the_author_meta( 'display_name', $author_id ) );
		$author_url = esc_url( get_author_posts_url( $author_id ) );
		$author_avatar = get_avatar( $author_id, $author_avatar_size, '', $author_name, array( 'extra_attr' => ( $args[ 'microdata' ] ? 'itemprop="image"' : '' ) ) );

		$author_html = '';
		if ( $args[ 'with_avatar' ] ) {
			$author_html .= sprintf( '
					<div class="avatar circle-frame">
						<a href="%1$s">%2$s</a>
					</div>
				',
				$author_url,
				$author_avatar
			);
		}

		$author_info = sprintf( '
				<span class="byline">%1$s <a class="%2$s" %3$s href="%4$s"><span %5$s>%6$s</span></a></span>
			',
			apply_filters( 'boombox/author/posted-by', esc_html__( 'by', 'boombox' ), 'author_meta' ),    /* 1 */
			$args[ 'microformat' ] ? 'url fn n' : '',                                                    /* 2 */
			$args[ 'microdata' ] ? 'itemprop="url"' : '',                                                    /* 3 */
			$author_url,                                                                                /* 4 */
			$args[ 'microdata' ] ? 'itemprop="name"' : '',                                                /* 5 */
			$author_name                                                                                /* 6 */

		);

		if ( $args[ 'with_desc' ] ) {
			$author_info .= sprintf( '<div class="about"><p class="text note">%1$s</p></div>',
				wp_kses_post( get_the_author_meta( 'description' ) )
			);
		}

		$author_html .= sprintf( '<div class="author-info">%1$s</div>', $author_info );

		$html_classes = array( 'author' );
		if ( $args[ 'microformat' ] ) {
			$html_classes[] = 'vcard';
		}
		if ( ! $args[ 'display' ] ) {
			$html_classes[] = 'mf-hide';
		}
		$html = sprintf(
			'<div class="%1$s" %2$s>%3$s</div>',
			implode( ' ', $html_classes ),
			$args[ 'microdata' ] ? 'itemprop="author" itemscope itemtype="http://schema.org/Person"' : '',
			$author_html
		);

		if ( $args[ 'echo' ] ) {
			echo $html;
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'boombox_post_author_expanded_info' ) ) {

	/**
	 * Render author expanded information block
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_author_expanded_info( $args = array() ) {
		if ( is_singular() || is_author() ) {

			$args = wp_parse_args( $args, array( 'echo' => true ) );

			$author_avatar_size = 186;
			$author_id = get_the_author_meta( 'ID' );
			$author_name = wp_kses_post( get_the_author_meta( 'display_name', $author_id ) );
			$author_url = esc_url( get_author_posts_url( $author_id ) );
			$author_avatar = get_avatar( $author_id, $author_avatar_size, '', $author_name, array( 'extra_attr' => 'itemprop="image"', 'type' => 'full' ) );
			$bio = apply_filters( 'author_bio', wp_kses_post( get_the_author_meta( 'description' ) ), $author_id );

			$html = sprintf(
				'<article class="bb-author-vcard author-vcard mb-md %7$s">
					<div class="author vcard" itemscope="" itemtype="http://schema.org/Person">
						<header>
							<div class="avatar auth-avatar circle-frame">
								<a href="%1$s">%2$s</a>
							</div>
							<div class="header-info">
								<h3 class="auth-name">
									<span class="auth-byline">%4$s</span>
									<a class="auth-url url fn n" itemprop="url" href="%1$s">
										<span itemprop="name">%3$s</span>
									</a>
									%8$s
								</h3>
							</div>
						</header>

						<div class="author-info">
							<div class="auth-desc text note" itemprop="description">%6$s</div>
							%5$s
						</div>
					</div>
				</article>',

				$author_url,                                                                                    /* 1 */
				$author_avatar,                                                                                    /* 2 */
				$author_name,                                                                                    /* 3 */
				apply_filters( 'boombox/author/posted-by', esc_html__( 'Posted by', 'boombox' ), 'expanded' ),    /* 4 */
				apply_filters( 'author_extended_data', '', $author_id ),                                        /* 5 */
				$bio,                                                                                            /* 6 */
				$bio ? '' : 'no-author-info',                                                                    /* 7 */
				apply_filters( 'boombox/author/name_row', '', $author_id )                                       /* 8 */
			);

			if ( $args[ 'echo' ] ) {
				echo $html;
			} else {
				return $html;
			}
		}
	}
}

if ( ! function_exists( 'boombox_single_post_link_pages' ) ) {

	/**
	 * Prints HTML for single post link pages navigation
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_single_post_link_pages( $args = array() ) {

		global $page, $numpages, $multipage, $more;

		$defaults = apply_filters( 'boombox/single/link-pages-args', array(
			'before'             => '<p>' . __( 'Pages:' ),
			'after'              => '</p>',
			'link_before'        => '',
			'link_after'         => '',
			'link_class'         => '',
			'paging'             => '',
			'reverse'            => 0,
			'next'               => 1,
			'next_class'         => array( 'next-page' ),
			'prev'               => 1,
			'prev_class'         => array( 'prev-page' ),
			'in_same_term'       => false,
			'excluded_terms'     => '',
			'taxonomy'           => 'category',
			'url_on_end'         => false,
			'prev_text_on_end'   => '',
			'next_text_on_end'   => '',
			'previous_page_link' => __( 'Previous page', 'boombox' ),        // paginated prev page
			'next_page_link'     => __( 'Next page', 'boombox' ),            // paginated next page
			'previous_post_link' => __( 'Previous post', 'boombox' ),        // prev page
			'next_post_link'     => __( 'Next post', 'boombox' ),            // next page
			'go_to_prev_next'    => 1,
			'pagelink'           => '%',
			'link_wrap_before'   => '',
			'link_wrap_after'    => '',
			'echo'               => 1,
		) );
		$r = wp_parse_args( $args, $defaults );
		$r[ 'link_class' ] = $r[ 'link_class' ] ? sprintf( 'class="%s"', ( is_array( $r[ 'link_class' ] ) ? implode( ' ', $r[ 'link_class' ] ) : $r[ 'link_class' ] ) ) : '';
		$r[ 'next_class' ] = is_array( $r[ 'next_class' ] ) ? $r[ 'next_class' ] : explode( ' ', preg_replace( '/\s\s+/', ' ', $r[ 'next_class' ] ) );
		$r[ 'prev_class' ] = is_array( $r[ 'prev_class' ] ) ? $r[ 'prev_class' ] : explode( ' ', preg_replace( '/\s\s+/', ' ', $r[ 'prev_class' ] ) );

		$has_page_pagination = false;
		$is_amp = boombox_is_amp();
		$prev_output = $next_output = '';
		$render_prev_output = $render_next_output = true;

		$next_post_function = 'get_next_post';
		$previous_post_function = 'get_previous_post';
		if ( is_attachment() ) {
			$next_post_function = 'boombox_get_next_attachment_post';
			$previous_post_function = 'boombox_get_previous_attachment_post';
		}

		if ( $multipage && $more ) {

			// previous page
			if ( $r[ 'prev' ] ) {

				$prev = $page - 1;
				// paginated single post
				if ( $prev > 0 ) {

					$link = str_replace( 'href="', sprintf( '%s href="', $r[ 'link_class' ] ), _wp_link_page( $prev ) ) . $r[ 'link_before' ] . $r[ 'previous_page_link' ] . $r[ 'link_after' ] . '</a>';
					$prev_output = apply_filters( 'wp_link_pages_link', $link, $prev );
					$has_page_pagination = true;

					// not paginated post ( go to next/prev post )
				} else if ( $r[ 'go_to_prev_next' ] && $boombox_post = ( $r[ 'reverse' ] ? $next_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) : $previous_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) ) ) {

					$permalink = $is_amp ? amp_get_permalink( $boombox_post->ID ) : get_permalink( $boombox_post->ID );

					$prev_output = sprintf( '<a href="%1$s" %2$s rel="prev">', esc_url( $permalink ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $r[ 'previous_post_link' ] . $r[ 'link_after' ] . '</a>';

					// no single post pagination and no next/prev post
				} else {

					if ( $r[ 'url_on_end' ] ) {
						$text = $r[ 'prev_text_on_end' ] ? $r[ 'prev_text_on_end' ] : $r[ 'previous_post_link' ];
						$url = $r[ 'url_on_end' ];
						$type = 'url_on_end';
					} else {
						$text = $r[ 'previous_post_link' ];
						$url = '#';
						$type = 'empty';
						$r[ 'prev_class' ][] = $r[ 'go_to_prev_next' ] ? 'bb-disabled' : 'p-hidden';
						if ( $is_amp ) {
							$render_prev_output = false;
						}
					}

					$url = apply_filters( 'boombox/prev_post_empty_url', $url, $type, $is_amp );
					$prev_output = sprintf( '<a href="%1$s" %2$s rel="prev">', esc_url( $url ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $text . $r[ 'link_after' ] . '</a>';

				}

			}

			// next page
			if ( $r[ 'next' ] ) {
				$next = $page + 1;
				// paginated single post
				if ( $next <= $numpages ) {

					$link = str_replace( 'href="', sprintf( '%s href="', $r[ 'link_class' ] ), _wp_link_page( $next ) ) . $r[ 'link_before' ] . $r[ 'next_page_link' ] . $r[ 'link_after' ] . '</a>';
					$next_output = apply_filters( 'wp_link_pages_link', $link, $next );
					$has_page_pagination = true;

					// not paginated post ( go to next/prev post )
				} else if ( $r[ 'go_to_prev_next' ] && $boombox_post = ( $r[ 'reverse' ] ? $previous_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) : $next_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) ) ) {

					$permalink = $is_amp ? amp_get_permalink( $boombox_post->ID ) : get_permalink( $boombox_post->ID );

					$next_output = sprintf( '<a href="%1$s" %2$s rel="next">', esc_url( $permalink ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $r[ 'next_post_link' ] . $r[ 'link_after' ] . '</a>';

					// no single post pagination and no next/prev post
				} else {
					if ( $r[ 'url_on_end' ] ) {
						$text = $r[ 'next_text_on_end' ] ? $r[ 'next_text_on_end' ] : $r[ 'next_post_link' ];
						$url = $r[ 'url_on_end' ];
						$type = 'url_on_end';
					} else {
						$text = $r[ 'next_post_link' ];
						$url = '#';
						$type = 'empty';
						$r[ 'next_class' ][] = $r[ 'go_to_prev_next' ] ? 'bb-disabled' : 'p-hidden';
						if ( $is_amp ) {
							$render_next_output = false;
						}
					}

					$url = apply_filters( 'boombox/next_post_empty_url', $url, $type, $is_amp );
					$next_output = sprintf( '<a href="%1$s" %2$s rel="next">', esc_url( $url ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $text . $r[ 'link_after' ] . '</a>';

				}
			}

		}

		if ( ! $prev_output && $r[ 'prev' ] ) {
			if ( $r[ 'go_to_prev_next' ] && $boombox_post = ( $r[ 'reverse' ] ? $next_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) : $previous_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) ) ) {
				$permalink = $is_amp ? amp_get_permalink( $boombox_post->ID ) : get_permalink( $boombox_post->ID );
				$prev_output = sprintf( '<a href="%1$s" %2$s rel="prev">', esc_url( $permalink ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $r[ 'previous_post_link' ] . $r[ 'link_after' ] . '</a>';
			} else {

				if ( $r[ 'url_on_end' ] ) {
					$text = $r[ 'prev_text_on_end' ] ? $r[ 'prev_text_on_end' ] : $r[ 'previous_post_link' ];
					$url = $r[ 'url_on_end' ];
					$type = 'url_on_end';
				} else {
					$text = $r[ 'previous_post_link' ];
					$url = '#';
					$type = 'empty';
					$r[ 'prev_class' ][] = $r[ 'go_to_prev_next' ] ? 'bb-disabled' : 'p-hidden';
					if ( $is_amp ) {
						$render_prev_output = false;
					}
				}

				$url = apply_filters( 'boombox/prev_post_empty_url', $url, $type, $is_amp );
				$prev_output = sprintf( '<a href="%1$s" %2$s rel="prev">', esc_url( $url ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $text . $r[ 'link_after' ] . '</a>';
			}
		}

		if ( ! $next_output && $r[ 'next' ] ) {

			if ( $r[ 'go_to_prev_next' ] && $boombox_post = ( $r[ 'reverse' ] ? $previous_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) : $next_post_function( $r[ 'in_same_term' ], $r[ 'excluded_terms' ], $r[ 'taxonomy' ] ) ) ) {
				$permalink = $is_amp ? amp_get_permalink( $boombox_post->ID ) : get_permalink( $boombox_post->ID );
				$next_output = sprintf( '<a href="%1$s" %2$s rel="next">', esc_url( $permalink ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $r[ 'next_post_link' ] . $r[ 'link_after' ] . '</a>';
			} else {
				if ( $r[ 'url_on_end' ] ) {
					$text = $r[ 'next_text_on_end' ] ? $r[ 'next_text_on_end' ] : $r[ 'next_post_link' ];
					$url = $r[ 'url_on_end' ];
					$type = 'url_on_end';
				} else {
					$text = $r[ 'next_post_link' ];
					$url = '#';
					$type = 'empty';
					$r[ 'next_class' ][] = $r[ 'go_to_prev_next' ] ? 'bb-disabled' : 'p-hidden';
					if ( $is_amp ) {
						$render_next_output = false;
					}
				}

				$url = apply_filters( 'boombox/next_post_empty_url', $url, $type, $is_amp );
				$next_output = sprintf( '<a href="%1$s" %2$s rel="next">', esc_url( $url ), $r[ 'link_class' ] ) . $r[ 'link_before' ] . $text . $r[ 'link_after' ] . '</a>';
			}
		}

		$r[ 'paging' ] = ( $numpages > 1 ) ? $r[ 'paging' ] : '';

		if ( $render_prev_output ) {
			$prev_output = sprintf( $r[ 'link_wrap_before' ], implode( ' ', $r[ 'prev_class' ] ) ) . $prev_output . $r[ 'link_wrap_after' ];
		} else {
			$prev_output = '';
		}
		if ( $render_next_output ) {
			$next_output = sprintf( $r[ 'link_wrap_before' ], implode( ' ', $r[ 'next_class' ] ) ) . $next_output . $r[ 'link_wrap_after' ];
		} else {
			$next_output = '';
		}

		if ( $has_page_pagination ) {
			$r[ 'before' ] = strtr( $r[ 'before' ], array( 'next-prev-pagination' => 'next-prev-pagination boombox-keep' ) );
		} else if ( apply_filters( 'boombox/single/link_pages/hide_container', ! $r[ 'go_to_prev_next' ] ) ) {
			$r[ 'before' ] = strtr( $r[ 'before' ], array( 'next-prev-pagination' => 'next-prev-pagination hidden' ) );
		}

		$output = '';
		if ( $render_prev_output || $render_next_output ) {
			$output = $r[ 'before' ] . $prev_output . $r[ 'paging' ] . $next_output . $r[ 'after' ];
		}

		if ( $r[ 'echo' ] ) {
			echo $output;
		} else {
			return $output;
		}

	}

}

if ( ! function_exists( 'boombox_show_thumbnail' ) ) {

	/**
	 * Check if thumbnail should be shown for multipage post
	 * @return bool
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_show_thumbnail() {
		global $page, $multipage;

		return $multipage ? ( $page == 1 ) : true;
	}

}

/**
 * Get HTML subtitle for current post.
 *
 * @return string
 * @since   2.0.0
 * @version 2.0.0
 */
function boombox_get_post_subtitle() {
	global $post;

	$html = '';
	if ( get_post_type() == 'post' ) {

		$excerpt_length = apply_filters( 'excerpt_length', 25 );
		$excerpt_more = apply_filters( 'excerpt_more', ' ' . '[&hellip;]' );

		if ( is_single() ) {
			$boombox_subtitle = $post->post_excerpt;
		} else {
			$boombox_subtitle = get_the_excerpt( $post );
			$boombox_subtitle = wp_trim_words( $boombox_subtitle, $excerpt_length, $excerpt_more );
		}

		$boombox_subtitle = apply_filters( 'boombox/the_post_subtitle', $boombox_subtitle );

		if ( $boombox_subtitle ) {

			if ( boombox_is_amp() ) {
				$html = sprintf( '<h2 class="post-summary m-b-sm-1">%s</h2>', $boombox_subtitle );
			} else {

				$defaults = array(
					'before'    => '',
					'after'     => '',
					'classes'   => 'entry-summary entry-sub-title',
					'microdata' => is_single() ? 'itemprop="description"' : '',
				);

				$args = wp_parse_args( apply_filters( 'boombox/post/subtitle-args', array() ), $defaults );

				$html = sprintf(
					'%1$s<p class="%2$s" %3$s>%4$s</p>%5$s',
					$args[ 'before' ],                        /* 1 */
					$args[ 'classes' ],                        /* 2 */
					$args[ 'microdata' ],                        /* 3 */
					wp_kses_post( $boombox_subtitle ),        /* 4 */
					$args[ 'after' ]                            /* 5 */
				);
			}
		}
	}

	return $html;
}

if ( ! function_exists( 'boombox_post_share_count' ) ) {
	/**
	 *
	 * Get | Render share count HTML for current post.
	 *
	 * @param bool   $html     As HTML
	 * @param bool   $echo     Render or return
	 * @param string $location Location
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_share_count( $html = true, $echo = true, $location = '' ) {

		$count = 0;
		$share_limit = 0;
		if ( function_exists( 'essb_core' ) ) {

			$share_limit = 0;
			if ( 'share-box' !== $location ) {
				$post_id = get_the_ID();
				$count = boombox_post_shares_count( (int)boombox_get_post_meta( $post_id, 'essb_c_total' ), $post_id );
			}

		} else if ( boombox_plugin_management_service()->is_plugin_active( 'mashsharer/mashshare.php' ) && 'post' === get_post_type() ) {
			$post_id = get_the_ID();
			$count = roundshares( boombox_get_post_meta( $post_id, 'mashsb_shares' ) + getFakecount() );
			global $mashsb_options;
			$share_limit = isset( $mashsb_options[ 'hide_sharecount' ] ) ? $mashsb_options[ 'hide_sharecount' ] : 0;
		}

		$share_limit = apply_filters( 'boombox/post-share-count-threshold', $share_limit );
		if ( $count ) {
			if ( ( $share_limit && $count < $share_limit ) || ! $count ) {
				$return = '';
			} else {
				$count = boombox_numerical_word( $count );
				$return = $html ? sprintf( '<span class="post-share-count"><i class="icon icon-share"></i>%s</span>', $count ) : $count;
			}
		} else {
			$return = '';
		}

		if ( $echo ) {
			echo $return;
		} else {
			return $return;
		}
	}
}

if ( ! function_exists( 'boombox_post_view_vote_count' ) ) {
	/**
	 * Prints HTML with views and votes count for provided post
	 *
	 * @param int   $post_id Post ID
	 * @param array $args    Arguments
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_view_vote_count( $post_id, $args = array() ) {

		$args = wp_parse_args( $args, array(
			'show_views' => true,
			'show_votes' => true,
		) );

		$views_html = $args[ 'show_views' ] ? boombox_get_post_view_count_html( $post_id ) : '';
		$votes_html = $args[ 'show_votes' ] ? boombox_get_post_vote_count_html( $post_id ) : '';
		if ( ( $args[ 'show_views' ] && $views_html ) || ( $args[ 'show_votes' ] && $votes_html ) ) {
			printf( '<span class="post-meta-wrapper">%1$s%2$s</span>',
				$views_html,
				$votes_html
			);
		}

	}

}

if ( ! function_exists( 'boombox_get_post_view_count_html' ) ) {

	/**
	 * Get views count HTML for provided post
	 *
	 * @param int $post_id Post ID
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_get_post_view_count_html( $post_id ) {
		$view_count = boombox_get_views_count( $post_id );

		$html = '';
		if ( ! boombox_is_view_count_tresholded( $view_count ) ) {
			$html = sprintf( '<span class="post-view-count"><i class="icon icon-eye"></i>%1$s</span>',
				boombox_numerical_word( $view_count )
			);
		}

		return $html;
	}

}

if ( ! function_exists( 'boombox_get_post_vote_count_html' ) ) {

	/**
	 * Get votes count HTML for provided post
	 *
	 * @param int $post_id Post ID
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_get_post_vote_count_html( $post_id ) {
		return sprintf( '<span class="post-vote-count"><i class="icon icon-vote"></i>%1$s</span>',
			boombox_get_post_point_count( $post_id )
		);
	}

}

if ( ! function_exists( 'boombox_post_share_buttons' ) ) {
	/**
	 * Render share buttons
	 *
	 * @see     Mashshare Plugin
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_share_buttons() {
		if ( function_exists( 'essb_core' ) ) {
			if ( ! is_single() ) {
				ob_start();
				Boombox_Essb::get_instance()->render();
				$content = ob_get_clean();

				echo strtr( $content, array( 'href="#comments"' => sprintf( 'href="%s"', ( get_permalink() . '#comments' ) ) ) );
			} else {
				Boombox_Essb::get_instance()->render();
			}
		} else if ( boombox_plugin_management_service()->is_plugin_active( 'mashsharer/mashshare.php' ) ) {
			echo do_shortcode( '[mashshare shares="false"]' );
		}
	}
}

if ( ! function_exists( 'boombox_post_share_mobile_buttons' ) ) {

	/**
	 * Render mobile share buttons
	 *
	 * @param bool $show_comments Include comments count
	 * @param bool $show_share    Inlude share count
	 * @param bool $show_points   Include points count
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_share_mobile_buttons( $show_comments, $show_share, $show_points ) {

		$boombox_post_share_box_elements = array();
		if ( $show_comments ) {
			$comments_count = get_comments_number();
			if ( $comments_count ) {
				$boombox_post_share_box_elements[] = sprintf( '<span class="mobile-comments-count">%1$d</span> %2$s', $comments_count, _n( 'comment', 'comments', $comments_count, 'boombox' ) );
			}
		}
		if ( $show_share ) {
			$share_count = boombox_post_share_count( false, false );
			if ( $share_count ) {
				$boombox_post_share_box_elements[] = sprintf( '<span class="mobile-shares-count">%s</span> %2$s', $share_count, _n( 'share', 'shares', $share_count, 'boombox' ) );
			}
		}
		if ( $show_points ) {
			$points_count = boombox_get_post_point_count( get_the_ID() );
			if ( $points_count ) {
				$boombox_post_share_box_elements[] = sprintf( '<span class="mobile-votes-count">%1$d</span> %2$s', $points_count, _n( 'point', 'points', $points_count, 'boombox' ) );
			}
		}

		if ( ! empty( $boombox_post_share_box_elements ) ) {
			echo sprintf( '<div class="mobile-info">%s</div>', implode( ', ', $boombox_post_share_box_elements ) );
		}

	}

}

if ( ! function_exists( 'boombox_mailchimp_form' ) ) {
	/**
	 * Newsletter Form HTML
	 *
	 * @see     Mashshare Plugin
	 *
	 * @param array $args Parsing arguments
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_mailchimp_form( $args = array() ) {
		$html = '';

		$args = wp_parse_args( $args, array(
			'tag'     => 'div',
			'class'   => '',
			'echo'    => true,
		) );

		if ( boombox_plugin_management_service()->is_plugin_active( 'mailchimp-for-wp/mailchimp-for-wp.php' ) ) {
			static $form_id;

			if ( ! $form_id ) {
				$forms = mc4wp_get_forms( array( 'numberposts' => 1 ) );
				if ( (bool)$forms ) {
					$form = array_pop( $forms );
					$form_id = $form->ID;
				}
			}

			if( $form_id ) {

				$before = '<' . $args['tag'] . ' class="newsletter-box mb-md ' . wp_kses_post( $args[ 'class' ] ) . '">';
				$after = '</' . $args['tag'] . '>';
				$html .= sprintf(
					'%1$s
						<div class="widget widget_mc4wp_form_widget horizontal">
							<h2 class="widget-title">%2$s</h2>
							%3$s
						</div>
					%4$s',
					$before,
					apply_filters( 'boombox_mailchimp_form_title', esc_html__( 'Get The Newsletter', 'boombox' ) ),
					do_shortcode( '[mc4wp_form id="' . $form_id . '"]' ),
					$after
				);

			}
		}

		if ( $args[ 'echo' ] ) {
			echo $html;
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'boombox_post_points' ) ) {

	/**
	 * Render post points HTML
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_points() {
		global $post;

		$point_classes = boombox_get_point_classes( $post->ID );
		$points_login_require = boombox_get_theme_option( 'extras_post_ranking_system_points_login_require' );
		$container_class = 'js-post-point';
		$authentication_url = '';
		$authentication_tag = 'button';
		if ( $points_login_require && ! is_user_logged_in() ) {
			$point_classes[ 'up' ] .= ' js-authentication';
			$point_classes[ 'down' ] .= ' js-authentication';
			$authentication_url = esc_url( '#sign-in' );
			$container_class = '';
			$authentication_tag = 'a';
		}
		$authentication_href = ! empty( $authentication_url ) ? 'href="' . $authentication_url . '"' : '';
		$points_count = boombox_get_post_point_count( $post->ID ); ?>

	<div class="post-rating <?php echo esc_attr( $container_class ); ?>" data-post_id="<?php echo $post->ID; ?>">
		<div class="inner">
			<<?php echo $authentication_tag; ?> <?php echo $authentication_href; ?> class="up
			point-btn <?php echo esc_attr( $point_classes[ 'up' ] ); ?>" data-action="up">
			<i class="icon-arrow-up"></i>
		</<?php echo $authentication_tag; ?>>

		<<?php echo $authentication_tag; ?> <?php echo $authentication_href; ?> class="down
		point-btn <?php echo esc_attr( $point_classes[ 'down' ] ); ?>" data-action="down">
		<i class="icon-arrow-down"></i>
		</<?php echo $authentication_tag; ?>>

		<span class="count">
					<i class="icon spinner-pulse"></i>
					<span class="text" label="<?php esc_html_e( 'points', 'boombox' ); ?>">
						<?php echo $points_count; ?>
					</span>
				</span>
		</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'boombox_categories_list' ) ) {

	/**
	 * Render post categories list
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_categories_list() {
		$categories_list = get_the_category_list( '<span class="mf-hide">, </span>' );
		if ( ! empty( $categories_list ) ) {

			$defaults = array(
				'before'    => '',
				'after'     => '',
				'classes'   => 'bb-cat-links',
				'microdata' => is_single() ? 'itemprop="keywords"' : '',
			);

			$args = wp_parse_args( apply_filters( 'boombox/post/categories-list', $defaults ), $defaults );

			printf(
				'%1$s<div class="%2$s" %3$s>%4$s</div>%5$s',
				$args[ 'before' ],                                        /* 1 */
				$args[ 'classes' ],                                        /* 2 */
				$args[ 'microdata' ],                                        /* 3 */
				$categories_list,                                        /* 4 */
				$args[ 'after' ]                                            /* 5 */
			);
		}
	}

}

if ( ! function_exists( 'boombox_tags_list' ) ) {

	/**
	 * Render post tags list
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_tags_list() {
		$tags_list = get_the_tag_list( '', '<span class="mf-hide">, </span>' );
		if ( ! empty( $tags_list ) ) {

			$defaults = array(
				'before'    => '',
				'after'     => '',
				'classes'   => 'bb-tags',
				'microdata' => is_single() ? 'itemprop="keywords"' : '',
			);

			$args = wp_parse_args( apply_filters( 'boombox/post/tags-list', $defaults ), $defaults );

			printf(
				'%1$s<div class="%2$s" %3$s>%4$s</div>%5$s',
				$args[ 'before' ],                                        /* 1 */
				$args[ 'classes' ],                                        /* 2 */
				$args[ 'microdata' ],                                        /* 3 */
				$tags_list,                                                /* 4 */
				$args[ 'after' ]                                            /* 5 */
			);
		}
	}

}

if ( ! function_exists( 'boombox_post_comments' ) ) {
	/**
	 * Render post comments HTML
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_post_comments() {
		if ( ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
			$comments_count = (int)get_comments_number( get_the_ID() );

			if ( $comments_count ) {
				printf( '<div class="post-comments"><a href="%1$s"><i class="icon-boombox-comment"></i><span>%2$d</span></a></div>',
					get_comments_link(),
					$comments_count );
			}
		}
	}
}

if ( ! function_exists( 'boombox_show_post_views' ) ) {
	/**
	 * Render post views HTML
	 */
	function boombox_show_post_views() {
		if ( is_singular() ) {
			global $post;
			$views_count = boombox_get_views_count( $post->ID );

			if ( ! boombox_is_view_count_tresholded( $views_count ) ) {
				printf( '<div class="views"><i class="icon icon-eye"></i><span class="count">%1$s</span>%2$s</div>',
					boombox_numerical_word( $views_count ),
					_n( 'view', 'views', $views_count, 'boombox' ) );
			}
		}
	}
}

if ( ! function_exists( 'boombox_remove_editor_article_classes' ) ) {

	/**
	 * Modify editor classes
	 *
	 * @param array $classes Current classes
	 * @param       $class
	 * @param int   $post_id Post ID
	 *
	 * @return array
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_remove_editor_article_classes( $classes, $class, $post_id ) {

		$index = array_search( 'hentry', $classes );
		if ( false !== $index ) {
			unset( $classes[ $index ] );
		}

		return $classes;
	}

}

if ( ! function_exists( 'boombox_render_affiliate_content' ) ) {

	/**
	 * Render affiliate post pricing content
	 *
	 * @since   1.0.0
	 * @version 2.0.0
	 */
	function boombox_render_affiliate_content() {
		$boombox_post_regular_price = boombox_get_post_meta( get_the_ID(), 'boombox_post_regular_price' );
		$boombox_post_discount_price = boombox_get_post_meta( get_the_ID(), 'boombox_post_discount_price' );
		$boombox_post_affiliate_link = boombox_get_post_meta( get_the_ID(), 'boombox_post_affiliate_link' );

		if ( $boombox_post_regular_price || $boombox_post_discount_price || $boombox_post_affiliate_link ) {

			$html = '';
			$current_price = $old_price = false;
			if ( $boombox_post_regular_price && $boombox_post_discount_price ) {
				$current_price = $boombox_post_discount_price;
				$old_price = $boombox_post_regular_price;
			} else if ( $boombox_post_regular_price ) {
				$current_price = $boombox_post_regular_price;
			} else if ( $boombox_post_discount_price ) {
				$current_price = $boombox_post_discount_price;
			}

			if ( $old_price ) {
				$html .= sprintf( '<div class="old-price">%1$s</div>', $old_price );
			}
			if ( $current_price ) {
				$html .= sprintf( '<div class="current-price">%1$s</div>', $current_price );
			}
			if ( $html ) {
				$html = sprintf( '<div class="bb-price-block">%1$s</div>', $html );
			}

			if ( $boombox_post_affiliate_link ) {
				$html .= sprintf( '<a class="item-url" href="%1$s" target="_blank" rel="nofollow">%2$s</a>', $boombox_post_affiliate_link, apply_filters( 'boombox_affiliate_button_label', esc_html__( 'Check It Out', 'boombox' ) ) );
			}

			if ( $html ) {
				printf( '<div class="affiliate-content">%s</div>', $html );
			}
		}
	}

}
add_action( 'boombox_affiliate_content', 'boombox_render_affiliate_content' );

/**
 * Check for HTMl containing gif or mp4 video
 *
 * @param string $html HTML to check
 *
 * @return bool
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_html_contains_gif_image( $html ) {

	preg_match( '/src="(?P<image>.*\.(gif|mp4))"/i', $html, $has_gif );

	return (bool)$has_gif;

}

/**
 * Check for HTMl containing embed or mp4 video
 *
 * @param string $html HTML to check
 *
 * @return bool
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_html_contains_embed_or_static_video( $html ) {
	preg_match( '/class=("|"([^"]*)\s)(boombox-featured-embed|boombox-featured-video)("|\s([^"]*)")/', $html, $has_video );

	return (bool)$has_video;
}

/**
 * Wrap embed within responsive container
 *
 * @param string $html    Embed HTMl
 * @param string $classes Additional CSS classes for wrapper
 *
 * @return string
 * @since   1.0.0
 * @verison 2.0.0
 */
function boombox_wrap_embed_within_responsive_container( $html, $classes = '' ) {
	return sprintf( '<div class="boombox-responsive-embed %1$s">%2$s</div>', $classes, $html );
}

/**
 * Get "NSFW" category id
 *
 * @return int
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_nsfw_category_id() {
	$term_id = 0;
	$term = get_term_by( 'name', apply_filters( 'boombox/nsfw_category', 'nsfw' ), 'category' );

	if ( $term ) {
		$term_id = $term->term_id;
	}

	return $term_id;
}

/**
 * Edit post thumbnail HTML
 *
 * @param string       $html              Current HTML
 * @param int          $post_id           Post ID
 * @param int          $post_thumbnail_id Post thumbnail ID
 * @param string|array $size              Thumbnail size
 * @param string|array $attr              Thumbnail attributes
 *
 * @return string
 */
function boombox_post_thumbnail_fallback( $html, $post_id, $post_thumbnail_id, $size, $attr ) {
	if ( in_array( $size, array( 'boombox_image545', 'boombox_image768', 'boombox_image768x450' ) ) ) {
		$post_thumbnail = wp_get_attachment_image_src( $post_thumbnail_id, 'full' );
		if ( 'gif' == strtolower( substr( $post_thumbnail[ 0 ], -3 ) ) ) {
			$html = get_the_post_thumbnail( $post_id, 'full', $attr );
		}
	}

	if ( boombox_is_nsfw_post( $post_id ) ) {
		$html .= boombox_get_nsfw_message();
	}

	return $html;
}
add_filter( 'post_thumbnail_html', 'boombox_post_thumbnail_fallback', 20, 5 );

/**
 * Get NSFW message
 *
 * @return string
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_nsfw_message() {

	$args = array(
		'icon'    => '<i class="icon icon-skull"></i>',
		'title'   => sprintf( '<h3>%s</h3>', esc_html__( 'Not Safe For Work', 'boombox' ) ),
		'content' => sprintf( '<p>%s</p>', esc_html__( 'Click to view this post.', 'boombox' ) ),
	);

	$args = wp_parse_args( (array)apply_filters( 'boombox_nsfw_message_args', $args ), $args );

	$html = sprintf( '<div class="nsfw-post"><div class="nsfw-content">%s</div></div>', ( $args[ 'icon' ] . $args[ 'title' ] . $args[ 'content' ] ) );

	return $html;
}

/**
 * Get post thumbnail caption
 * @return string
 *
 * @since 1.0.0
 * @version 2.0.0
 */
function boombox_get_post_thumbnail_caption( $echo = true ) {
	$html = '';
	$caption = get_the_post_thumbnail_caption();
	if ( $caption ) {

		$args = array(
			'before'  => '',
			'after'   => '',
			'classes' => 'thumbnail-caption',
		);

		$args = wp_parse_args( apply_filters( 'boombox/post/thumbnail/caption', $args ), $args );

		$html = sprintf(
			'%1$s<div class="%2$s">%3$s</div>%4$s',
			$args[ 'before' ],
			$args[ 'classes' ],
			get_the_post_thumbnail_caption(),
			$args[ 'after' ]
		);
	}

	return $html;
}

/**
 * Get embed data from URL
 *
 * @param string $url URL to check
 *
 * @return array
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_embed_video_data_from_url( $url ) {

	$embed_data = array();

	while ( true ) {
		/***** "Youtube" Iframe */
		preg_match( boombox_get_regex( 'youtube' ), $url, $youtube_matches );
		if ( isset( $youtube_matches[ 1 ] ) && $youtube_matches[ 1 ] ) {
			$embed_data = array(
				'type'     => 'youtube',
				'video_id' => $youtube_matches[ 1 ],
			);
			break;
		}

		/***** "Vimeo" Iframe */
		preg_match( boombox_get_regex( 'vimeo' ), $url, $vimeo_matches );
		if ( isset( $vimeo_matches[ 5 ] ) && $vimeo_matches[ 5 ] ) {
			$embed_data = array(
				'type'     => 'vimeo',
				'video_id' => $vimeo_matches[ 5 ],
			);
			break;
		}

		/***** "Dailymotion" Iframe */
		preg_match( boombox_get_regex( 'dailymotion' ), $url, $dailymotion_matches );
		if ( isset( $dailymotion_matches[ 1 ] ) && $dailymotion_matches[ 1 ] ) {
			$embed_data = array(
				'type'     => 'dailymotion',
				'video_id' => $dailymotion_matches[ 1 ],
			);
			break;
		}

		/***** "Vine" Iframe */
		preg_match( boombox_get_regex( 'vine' ), $url, $vine_matches );
		if ( isset( $vine_matches[ 1 ] ) && $vine_matches[ 1 ] ) {
			$embed_data = array(
				'type'     => 'vine',
				'video_id' => $vine_matches[ 1 ],
			);
			break;
		}

		/***** "Ok" Iframe */
		preg_match( boombox_get_regex( 'ok' ), $url, $ok_matches );
		if ( isset( $ok_matches[ 2 ] ) && $ok_matches[ 2 ] ) {
			$embed_data = array(
				'type'     => 'ok',
				'video_id' => $ok_matches[ 2 ],
			);
			break;
		}

		/***** "Facebook" Iframe */
		preg_match( boombox_get_regex( 'facebook' ), $url, $fb_matches );
		if ( isset( $fb_matches[ 1 ] ) && $fb_matches[ 1 ] ) {
			$embed_data = array(
				'type'      => 'facebook',
				'video_url' => $fb_matches[ 1 ],
			);
			break;
		}

		/***** "Vidme" Iframe */
		preg_match( boombox_get_regex( 'vidme' ), $url, $vidme_matches );
		if ( isset( $vidme_matches[ 1 ] ) && $vidme_matches[ 1 ] ) {
			$embed_data = array(
				'type'     => 'vidme',
				'video_id' => $vidme_matches[ 1 ],
			);
			break;
		}

		/***** "VK" Iframe */
		preg_match( boombox_get_regex( 'vk' ), $url, $vk_matches );
		if ( isset( $vk_matches[ 2 ] ) && $vk_matches[ 2 ] ) {
			parse_str( $vk_matches[ 2 ], $vk_matches );
			if ( isset( $vk_matches[ 'id' ], $vk_matches[ 'oid' ], $vk_matches[ 'hash' ] ) ) {
				$embed_data = array(
					'type' => 'vk',
					'id'   => $vk_matches[ 'id' ],
					'oid'  => $vk_matches[ 'oid' ],
					'hash' => $vk_matches[ 'hash' ],
				);
			}
			break;
		}

		/***** "Twitch" Iframe */
		preg_match( boombox_get_regex( 'twitch' ), $url, $twitch_matches );
		if ( isset( $twitch_matches[ 2 ] ) && $twitch_matches[ 2 ] ) {
			$embed_data = array(
				'type'        => 'twitch',
				'stream_type' => isset( $twitch_matches[ 1 ] ) && $twitch_matches[ 1 ] ? 'video' : 'channel',
				'video_id'    => $twitch_matches[ 2 ],
			);
			break;
		}

		/***** "Coub" Iframe */
		preg_match( boombox_get_regex( 'coub' ), $url, $coub_matches );
		if ( isset( $coub_matches[ 3 ] ) && $coub_matches[ 3 ] ) {
			$embed_data = array(
				'type'     => 'coub',
				'video_id' => $coub_matches[ 3 ],
			);
			break;
		}

		/***** "Twitter" Iframe */
		preg_match( boombox_get_regex( 'twitter' ), $url, $twitter_matches );
		if ( isset( $twitter_matches[ 1 ] ) && isset( $twitter_matches[ 2 ] ) && $twitter_matches[ 1 ] && $twitter_matches[ 2 ] ) {
			$embed_data = array(
				'type'      => 'twitter',
				'video_url' => $url,
			);
			break;
		}

		break;
	}

	return $embed_data;
}

/**
 * Get embed iframe HTML
 *
 * @param       $embed_type       string  Embed type youtube, vimeo, dailymotion
 * @param array $params           array   Embed additional attributes
 *
 * @return string
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_embed_html( $embed_type, $params = array() ) {

	$html = '';
	$embed_src = false;
	$iframe_atts = array();

	switch ( $embed_type ) {

		case 'youtube':
			$embed_atts = apply_filters( 'boombox/embed/youtube/src_atts',
				array( 'autoplay' => 0 ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( 'https://www.youtube.com/embed/%1$s?%2$s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/youtube/iframe_atts',
				array(
					'type'            => 'text/html',
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				), $params[ 'video_id' ]
			);

			break;
		case 'vimeo':

			$embed_atts = apply_filters( 'boombox/embed/vimeo/src_atts',
				array( 'autopause' => 1, 'badge' => 0, 'byline' => 0, 'loop' => 0, 'title' => 0, 'autoplay' => 0 ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( '//player.vimeo.com/video/%1$s?%2$s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/vimeo/iframe_atts',
				array(
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'video_id' ]
			);

			break;
		case 'dailymotion':

			$embed_atts = apply_filters( 'boombox/embed/dailymotion/src_atts',
				array( 'autoplay' => 0 ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( '//www.dailymotion.com/embed/video/%1$s?%2$s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/dailymotion/iframe_atts',
				array(
					'type'            => 'text/html',
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'video_id' ]
			);

			break;
		case 'vine':

			$embed_atts = apply_filters( 'boombox/embed/vine/src_atts',
				array( 'audio' => 1 ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( 'https://vine.co/v/%s/embed/simple?%s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/vine/iframe_atts',
				array(
					'width'             => '100%',
					'height'            => 376,
					'src'               => $embed_src,
					'frameborder'       => 0,
					'allowfullscreen'   => true,
					'allowTransparency' => 'true',
				),
				$params[ 'video_id' ]
			);

			break;
		case 'ok':

			$embed_atts = apply_filters( 'boombox/embed/ok/src_atts',
				array( 'autoplay' => 0 ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( 'https://ok.ru/videoembed/%s?%s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/ok/iframe_atts',
				array(
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'video_id' ]
			);

			break;
		case 'vidme':

			$embed_atts = apply_filters( 'boombox/embed/vidme/src_atts',
				array( 'stats' => 1 ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( 'https://vid.me/e/%s?%s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/vidme/iframe_atts',
				array(
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'video_id' ]
			);

			break;
		case 'vk':
			$embed_atts = apply_filters( 'boombox/embed/vk/src_atts',
				array( 'id' => $params[ 'id' ], 'oid' => $params[ 'oid' ], 'hash' => $params[ 'hash' ] ),
				$params[ 'id' ]
			);
			$embed_src = sprintf( 'https://vk.com/video_ext.php?%s', build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/vidme/iframe_atts',
				array(
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'id' ]
			);

			break;
		case 'facebook':
		case 'twitter':

			global $wp_embed;
			$html = $wp_embed->shortcode( array(), $params[ 'video_url' ] );

			break;
		case 'twitch':

			$embed_atts = array( 'autoplay' => 'false' );
			if ( $params[ 'stream_type' ] == 'video' ) {
				$embed_atts[ 'video' ] = 'v' . $params[ 'video_id' ];
			} else if ( $params[ 'stream_type' ] == 'channel' ) {
				$embed_atts[ 'channel' ] = $params[ 'video_id' ];
			}

			$embed_atts = apply_filters( 'boombox/embed/twitch/src_atts', $embed_atts, $params[ 'video_id' ] );
			$embed_src = sprintf( 'https://player.twitch.tv/?%s', build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/twitch/iframe_atts',
				array(
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'video_id' ]
			);

			break;
		case 'coub':

			$embed_atts = apply_filters( 'boombox/embed/coub/src_atts',
				array( 'muted' => 'false', 'autostart' => 'false', 'originalSize' => 'false', 'startWithHD' => 'false' ),
				$params[ 'video_id' ]
			);
			$embed_src = sprintf( '//coub.com/embed/%s?%s', $params[ 'video_id' ], build_query( $embed_atts ) );

			$iframe_atts = apply_filters( 'boombox/embed/coub/iframe_atts',
				array(
					'width'           => '100%',
					'height'          => 376,
					'src'             => $embed_src,
					'frameborder'     => 0,
					'allowfullscreen' => true,
				),
				$params[ 'video_id' ]
			);

			break;
	}

	if ( $embed_src ) {
		foreach ( $iframe_atts as $attr => $value ) {
			$iframe_atts[ $attr ] = is_bool( $value ) ? $attr : sprintf( '%s="%s"', $attr, $value );
		}

		$html = sprintf( '<iframe %s></iframe>', implode( ' ', $iframe_atts ) );
	}

	return $html;
}

/**
 * Get HTML for HTML video
 *
 * @param      $video_url        string          HTML video source
 * @param null $image_src        string|null     Image src to use if javascript is disabled or is old browser
 *
 * @return string
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_html_video( $video_url, $image_src = NULL ) {

	$set = boombox_get_theme_options_set( array(
		'extras_video_control_enable_mp4_video_loop',
		'extras_video_control_mp4_video_sound',
		'extras_video_control_mp4_video_player_controls'
	) );

	$image = '';
	if ( $image_src ) {
		$image = sprintf( '<img src="%s" title="%s">',
			esc_url( $image_src ),
			esc_html__( 'To view this video please enable JavaScript, and consider upgrading to a web browser that supports HTML5 video', 'boombox' )
		);
	}

	$video_atts = array();

	/***** Video loop */
	if ( $set['extras_video_control_enable_mp4_video_loop'] ) {
		$video_atts[] = 'loop';
	}

	/***** Video mute */
	if ( $set['extras_video_control_mp4_video_sound'] == 'muted' ) {
		$video_atts[] = 'muted';
	}

	/***** Video Controls */
	$sound_icons = '';
	if ( $set['extras_video_control_mp4_video_player_controls'] == 'mute' ) {
		$sound_icons = sprintf( '
            <span class="bb-btn-circle btn-sm btn-volume">
                <i class="volume-off icon icon-volume-mute %1$s"></i>
                <i class="volume-on icon icon-volume-up %2$s"></i>
            </span>',
			$set['extras_video_control_mp4_video_sound'] == 'muted' ? '' : 'hidden',
			$set['extras_video_control_mp4_video_sound'] == 'muted' ? 'hidden' : ''
		);
	} else if ( $set['extras_video_control_mp4_video_player_controls'] == 'full_controls' ) {
		$video_atts[] = 'controls';
	}

	$html = sprintf(
		'<video width="100%%" height="auto" %1$s >
            <source src="%2$s" type="video/mp4">%3$s
		</video>
		<span class="bb-btn-circle btn-lg btn-play"><i class="icon icon-video2"></i></span>%4$s
		<span class="badge-duration hidden"></span>
        ',
		implode( ' ', $video_atts ),
		esc_url( $video_url ),
		$image,
		$sound_icons
	);

	return $html;
}

/**
 * Check if html mp4 video allowed to render within template
 *
 * @param $template string  Current template
 *
 * @return bool
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_is_video_mp4_allowed( $template ) {
	return ( ( 'single' == $template ) || ( 'listing' == $template && boombox_get_theme_option( 'extras_video_control_enable_mp4_video_on_post_listings' ) ) );
}

/**
 * Check if html embed video allowed to render within template
 *
 * @param $template string  Current template
 *
 * @return bool
 *
 * @since 1.0.0
 * @version 2.0.0
 */
function boombox_is_video_embed_allowed( $template ) {
	return ( ( 'single' == $template ) || ( 'listing' == $template && boombox_get_theme_option( 'extras_video_control_enable_embed_video_on_post_listings' ) ) );
}

/**
 * Get post featured video HTML
 *
 * @param int $post_id              int     Post ID
 * @param     $featured_image_size  string  Image Size
 * @param     $params               array   Additional attributes
 *
 * @return string
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_post_featured_video( $post_id = 0, $featured_image_size = 'full', $params = array() ) {

	$featured_video_html = '';
	if ( 0 === $post_id ) {
		global $post;
		if ( is_object( $post ) ) {
			$post_id = $post->ID;
		}
	}

	if ( ! $post_id ) {
		return $featured_video_html;
	}

	$featured_video_url = boombox_get_post_meta( $post_id, 'boombox_video_url' );

	if ( ! empty( $featured_video_url ) ) {

		$template = ( isset( $params[ 'template' ] ) && $params[ 'template' ] ) ? $params[ 'template' ] : '';
		$featured_image = wp_get_attachment_image_src( get_post_thumbnail_id( $post_id ), $featured_image_size );
		$featured_image_src = isset( $featured_image[ 0 ] ) ? $featured_image[ 0 ] : '';
		$featured_image_style = $featured_image_src ? 'style="background-image:url(' . esc_url( $featured_image_src ) . ')"' : '';
		$featured_image_class = ! $featured_image_src ? esc_attr( 'no-thumbnail' ) : '';
		$featured_video = array();

		while ( true ) {

			/***** Embed video */
			if ( boombox_is_video_embed_allowed( $template ) ) {
				$embed_data = boombox_get_embed_video_data_from_url( $featured_video_url );
				if ( ! empty( $embed_data ) ) {
					switch ( $embed_data[ 'type' ] ) {
						case 'facebook':
						case 'twitter':
							$featured_video = array(
								'before' => '<div class="clearfix boombox-featured-embed">',
								'after'  => '</div>',
								'html'   => boombox_get_embed_html( $embed_data[ 'type' ], array( 'video_url' => $embed_data[ 'video_url' ] ) ),
							);
							break;
						case 'vk':
							$featured_video = array(
								'before' => '<div class="clearfix boombox-featured-embed">',
								'after'  => '</div>',
								'html'   => boombox_get_embed_html( $embed_data[ 'type' ], array( 'id' => $embed_data[ 'id' ], 'oid' => $embed_data[ 'oid' ], 'hash' => $embed_data[ 'hash' ] ) ),
							);
							break;
						case 'twitch':
							$featured_video = array(
								'before' => '<div class="clearfix boombox-featured-embed">',
								'after'  => '</div>',
								'html'   => boombox_get_embed_html( $embed_data[ 'type' ], array( 'video_id' => $embed_data[ 'video_id' ], 'stream_type' => $embed_data[ 'stream_type' ] ) ),
							);
							break;
						default:
							$featured_video = array(
								'before' => sprintf( '<div class="video-wrapper boombox-featured-embed %s">', $featured_image_class ),
								'after'  => '</div>',
								'html'   => boombox_get_embed_html( $embed_data[ 'type' ], array( 'video_id' => $embed_data[ 'video_id' ] ) ),
							);
					}
					break;
				}
			}

			/***** HTML video */
			if ( boombox_is_video_mp4_allowed( $template ) && strrpos( $featured_video_url, '.mp4' ) !== false ) {
				$featured_video = array(
					'before' => sprintf( '<div class="video-wrapper boombox-featured-video %s" %s>', $featured_image_class, $featured_image_style ),
					'after'  => '</div>',
					'html'   => boombox_get_html_video( $featured_video_url, $featured_image_src ),
				);
				break;
			}

			break;
		}

		$featured_video_html = ! empty( $featured_video ) ? ( $featured_video[ 'before' ] . $featured_video[ 'html' ] . $featured_video[ 'after' ] ) : '';
		$featured_video_html .= boombox_is_nsfw_post( $post_id ) ? boombox_get_nsfw_message() : '';
	}

	return $featured_video_html;
}

/**
 * Get meta key for share calculation
 *
 * @return string
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_shares_meta_key() {
	return apply_filters( 'boombox/shares_meta_key', 'mashsb_shares' );
}

/**
 * Get comment form options
 *
 * @return array
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_comment_form_args() {

	$commenter = wp_get_current_commenter();
	$req = get_option( 'require_name_email' );
	$aria_req = ( $req ? " aria-required='true'" : '' );

	$args = array(
		'title_reply_before' => '<h2 id="reply-title" class="comment-reply-title">',
		'title_reply'        => '',
		'title_reply_to'     => '',
		'title_reply_after'  => '</h2>',
		'class_submit'       => 'submit-button',
		'label_submit'       => esc_html__( 'Post', 'boombox' ),
		'fields'             => array(
			'author' =>
				'<div class="row"><div class="col-lg-6 col-md-6"><div class="input-field">' .
				'<input id="author" name="author" type="text" placeholder="' . esc_html__( 'Name *', 'boombox' ) . '" ' . $aria_req . ' ' .
				' value="' . esc_attr( $commenter[ 'comment_author' ] ) . '">' .
				'</div></div>',
			'email'  =>
				'<div class="col-lg-6 col-md-6"><div class="input-field">' .
				'<input id="email" name="email" type="text" placeholder="' . esc_html__( 'Email *', 'boombox' ) . '" ' . $aria_req . ' ' .
				'value="' . esc_attr( $commenter[ 'comment_author_email' ] ) . '">' .
				'</div></div></div>',
		),
		'must_log_in'        => '<p class="must-log-in">' . sprintf( __( 'You must be <a href="%s" class="js-authentication">logged in</a> to post a comment.', 'boombox' ), '#sign-in' ) . '</p>',
		'comment_field'      =>
			'<div class="comment-form-comment"><div class="input-field">' .
			'<textarea id="comment" name="comment" placeholder="' . esc_html__( 'Write a comment *', 'boombox' ) . '" aria-required="true"></textarea>' .
			'</div></div>',
	);

	return apply_filters( 'boombox/comment_form_args', $args );
}

/**
 * Get post thumbnail wrapper link
 *
 * @param string $url    URL for wrapper
 * @param string $target URL target attribute
 * @param string $rel    URL rel attribute
 * @param string $class  CSS classes for wrapper anchor
 *
 * @return array
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_post_thumbnail_wrapper_link( $url, $target, $rel, $class = "" ) {
	$class = $class ? sprintf( 'class="%s"', $class ) : '';

	return array(
		'before' => sprintf( '<a href="%1$s" title="%2$s" %3$s %4$s>', $url, esc_attr( the_title_attribute( array( 'echo' => false ) ) ), $target, $class ),
		'after'  => '</a>',
	);
}

/**
 * Get post thumbnail wrapper before / after
 *
 * @param string $html    Current HTML to wrap
 * @param bool   $is_nsfw Is NSFW post
 * @param string $url     URL for wrapper
 * @param string $target  URL target attribute
 * @param string $rel     URL rel attribute
 * @param string $class   CSS classes for wrapper anchor
 *
 * @return array
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_post_thumbnail_wrapper( $html, $is_nsfw, $url, $target, $rel, $class = '' ) {

	$return = array( 'before' => '', 'after' => '' );

	while ( true ) {

		if ( boombox_html_contains_gif_image( $html ) ) {
			break;
		}

		if ( boombox_html_contains_embed_or_static_video( $html ) ) {
			break;
		}

		$return = array_merge( $return, boombox_get_post_thumbnail_wrapper_link( $url, $target, $rel, $class ) );
		break;

	}

	return $return;
}

/**
 * Get user meta description
 *
 * @param int $user_id User ID
 *
 * @return string
 */
function boombox_get_user_meta_description( $user_id ) {
	$desciption = wp_kses_post( get_the_author_meta( 'description', $user_id ) );
	$desciption = apply_filters( 'author_bio', $desciption, $user_id );

	return strip_tags( $desciption );
}

/**
 * Get view count threshold
 * @return int
 *
 * @since   2.0.0
 * @version 2.0.0
 */
function boombox_get_view_count_threshold() {
	return absint( apply_filters( 'boombox/post-views-count-threshold', 0 ) );
}

/**
 * Conditional check for view count treshold
 *
 * @param int $view_count View count
 *
 * @return bool
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_is_view_count_tresholded( $view_count ) {
	return ( $view_count < boombox_get_view_count_threshold() );
}

/**
 * Check if is AMP version
 *
 * @return bool
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_is_amp() {
	$is_amp_endpoint = false;

	if ( function_exists( 'boombox_amp' ) ) {
		$is_amp_endpoint = boombox_amp()->is_amp();
	}

	return $is_amp_endpoint;
}

/**
 * Get default thumbnail URL for post
 *
 * @param string $size                 Optional. Image size to use.
 * @param        $post                 int|WP_Post  $post Optional. Post ID or WP_Post object.  Default is global
 *                                     `$post`.
 *
 * @return string
 *
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_get_post_dafault_thumbnail_url( $size = 'full', $post = NULL ) {
	$url = apply_filters( 'boombox/post-default-thumbnail', '', $size, get_post( $post ) );

	return esc_url( $url );
}

/**
 * Render hidden SEO title
 * @since   1.0.0
 * @version 2.0.0
 */
function boombox_render_hidden_seo_title() {
	$title = wp_title( ' - ', false, 'right' );
	if ( ! $title ) {
		$title = wp_get_document_title();
	}

	if ( $title ) {
		printf( '<h1 class="mf-hide site-title">%s</h1>', $title );
	}
}

/**
 * Get single post sortable sections
 * @return array
 * @since 2.0.0
 * @version 2.0.0
 */
function boombox_get_single_sortable_sections() {
	$sections = boombox_get_theme_option( 'single_post_general_sections' );

	return apply_filters( 'boombox/single/sortable_sections', $sections );
}

/**
 * Check against fragment cache enabled feature
 * @return bool
 *
 * @since   2.0.0
 * @version 2.0.0
 */
function boombox_is_fragment_cache_enabled() {
	return (bool)apply_filters( 'boombox/frgcache.enabled', false );
}

/**
 * Check against fragment cache enabled feature
 * @return bool
 *
 * @since   2.0.0
 * @version 2.0.0
 */
function boombox_is_page_cache_enabled() {
	return (bool)apply_filters( 'boombox/pgcache.enabled', false );
}

/**
 * Get body classes by sidebar type
 *
 * @param string $type        Chosen sidebar type
 * @param string $orientation Sidebar orientation
 * @param bool   $reverse     Sidebar reverse
 *
 * @return string
 */
function boombox_get_body_classes_by_sidebar_type( $type, $orientation, $reverse = false ) {
	$has_sidebar = true;

	switch ( $type ) {
		case '1-sidebar-1_3':
			$classes = 'one-sidebar sidebar-1_3';
			break;
		case '1-sidebar-1_4':
			$classes = 'one-sidebar sidebar-1_4';
			break;
		case '2-sidebars-1_4-1_4':
			$classes = 'two-sidebar sidebar-1_4';
			break;
		case '2-sidebars-small-big':
			$classes = 'two-sidebar sidebar-1_3-1_4';
			break;
		case 'no-sidebar':
			$classes = 'no-sidebar';
			$has_sidebar = false;
			break;
		default:
			$classes = 'no-sidebar';
			$has_sidebar = false;
	}

	if ( $has_sidebar ) {
		if ( $reverse ) {
			$orientation = ( $orientation == 'right' ) ? 'left' : 'right';
		}
		$classes .= sprintf( ' %s-sidebar', $orientation );
	}

	return $classes;

}