<?php
/**
 * Register a post meta box using a class.
 *
 * @package BoomBox_Theme
 * @since   1.0.0
 * @version 2.0.0
 */

// Prevent direct script access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

if ( ! class_exists( 'Boombox_Post_Meta_Box' ) ) {

	if( ! class_exists( 'AIOM_Base' ) ) {
		require_once ( 'all-in-one-meta' . DIRECTORY_SEPARATOR . 'class-aiom-base.php' );
	}

	/**
	 * Class Boombox_Post_Meta_Box
	 */
	final class Boombox_Post_Meta_Box extends AIOM_Base {

		/**
		 * Holds wp object type: taxonomy or post
		 * @var string
		 */
		protected $object = 'post';

		/**
		 * Hold object type: post, page or any other registered post type
		 * @var string
		 */
	    protected $object_type = 'post';

		/**
		 * Holds class single instance
		 * @var null
		 */
		private static $_instance = null;

		/**
		 * Get instance
		 * @return Boombox_Post_Meta_Box|null
		 */
		public static function get_instance() {

			if (null == static::$_instance) {
				static::$_instance = new self();
			}

			return static::$_instance;

		}

		/**
		 * Boombox_Post_Meta_Box constructor.
		 */
		private function __construct() {
			$this->hooks();
		}

		/**
		 * A dummy magic method to prevent Boombox_Post_Meta_Box from being cloned.
		 *
		 */
		public function __clone() {
			throw new Exception('Cloning ' . __CLASS__ . ' is forbidden');
		}

		/**
		 * Setup Hooks
		 */
		public function hooks() {
			add_action( 'add_meta_boxes', array( $this, 'add_metabox' ), 1 );
			add_action( 'save_post', array( $this, 'save_metabox' ), 10, 2 );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		}

		/**
         * Setup structure
		 * @return array
		 */
		public function get_structure() {
			$structure = array(
				'tab_global' => array(
					'title' 	=> esc_html__( 'Global', 'boombox' ),
					'active'	=> true,
					'icon'      => false,
					'fields' => array(
						// Hide Featured Image
						'boombox_hide_featured_image' => array(
							'name'      => 'boombox_hide_featured_image',
							'type'		=> 'select',
							'label'		=> esc_html__( 'Hide Featured Image', 'boombox' ),
							'priority'  => 20,
							'choices' 	=> Boombox_Choices_Helper::get_instance()->get_post_featured_image_appearance(),
							'default'	=> 'customizer',
							'callback'  => 'sanitize_text_field'
						),
						// Layout
						'boombox_layout' => array(
							'name'		=> 'boombox_layout',
							'type'		=> 'select',
							'label'		=> esc_html__( 'Layout', 'boombox' ),
							'priority'  => 30,
							'choices' 	=> array(
								'inherit' => __( 'Inherit', 'boombox' ),
								'custom'  => __( 'Custom', 'boombox' )
							),
							'default'	=> '1-sidebar-1_3',
							'callback'  => 'sanitize_text_field'
						),
						// Sidebar Type
						'boombox_sidebar_type' => array(
							'name'		=> 'boombox_sidebar_type',
							'type'		=> 'radio-image',
							'label'		=> esc_html__( 'Sidebar Type', 'boombox' ),
							'priority'  => 40,
							'choices' 	=> Boombox_Choices_Helper::get_instance()->get_sidebar_types(),
							'default'	=> '1-sidebar-1_3',
							'active_callback'=> array(
								array(
									'field_id' => 'boombox_layout',
									'value' => 'custom',
									'compare' => '='
								)
							),
							'callback'  => 'sanitize_text_field'
						),
						// Sidebar Orientation
						'boombox_sidebar_orientation' => array(
							'name'		=> 'boombox_sidebar_orientation',
							'type'		=> 'radio',
							'label'		=> esc_html__( 'Sidebar Orientation', 'boombox' ),
							'priority'  => 50,
							'choices' 	=> array(
								'right' => __( 'Right', 'boombox' ),
								'left'  => __( 'Left', 'boombox' )
							),
							'config'      => array( 'axis' => 'vertical' ),
							'default'	=> 'right',
							'active_callback'=> array(
								array(
									'field_id' => 'boombox_layout',
									'value' => 'custom',
									'compare' => 'IN'
								)
							),
							'callback'  => 'sanitize_text_field'
						),
						// Video URL
						'boombox_video_url' => array(
							'name'		    => 'boombox_video_url',
							'type'		    => 'text',
							'label'		    => esc_html__( 'Video URL', 'boombox' ),
							'priority'      => 60,
							'description'   => esc_html__( 'Supported Videos: MP4, Youtube, Vimeo, Dailymotion, Vine, Odnoklassniki, Facebook, Vidme, Vkontakte, Twitch, Coub, Twitter', 'boombox' ),
							'default'	    => '',
							'callback'      => array( $this, 'sanitize_video_url' ),
							'class'         => 'regular-text',
							'id'            => 'boombox_video_url',
						)
						// other fields go here
					)
				),
				'tab_affiliate' => array(
					'title' => esc_html__( 'Affiliate', 'boombox' ),
					'active'	=> false,
					'icon'      => false,
					'fields' => array(
						// Regular Price
						'boombox_post_regular_price' => array(
							'name'      => 'boombox_post_regular_price',
							'type'		=> 'text',
							'label'		=> esc_html__( 'Regular Price', 'boombox' ),
							'priority'  => 20,
							'default'	=> '',
							'callback'  => 'sanitize_text_field',
							'class'     => 'regular-text',
						),
						// Discount Price
						'boombox_post_discount_price' => array(
							'name'		=> 'boombox_post_discount_price',
							'type'		=> 'text',
							'label'		=> esc_html__( 'Discount Price', 'boombox' ),
							'priority'  => 30,
							'default'	=> '',
							'callback'  => 'sanitize_text_field',
							'class'     => 'regular-text',
						),
						// Affiliate Link
						'boombox_post_affiliate_link' => array(
							'name'		=> 'boombox_post_affiliate_link',
							'type'		=> 'text',
							'label'		=> esc_html__( 'Affiliate Link', 'boombox' ),
							'priority'  => 40,
							'default'	=> '',
							'callback'  => 'sanitize_text_field',
							'class'     => 'regular-text',
						),
						// Post Link
						'boombox_post_affiliate_link_use_as_post_link' => array(
							'name'		=> 'boombox_post_affiliate_link_use_as_post_link',
							'type'		=> 'checkbox',
							'label'		=> '',
							'priority'  => 50,
							'text'		=> esc_html__( 'Use as post link', 'boombox' ),
							'default'	=> '0',
							'callback'  => array( $this, 'sanitize_checkbox' )
						),
						// other fields go here
					)
				),
				// other tabs go here
			);

			return apply_filters( 'boombox/admin/post/meta-boxes/structure', $structure, $this );
		}

		/**
		 * Add meta box.
		 */
		public function add_metabox() {
			/**
			 * Add Advanced Fields to Post screen
			 */
			if( apply_filters( 'boombox/admin/post/meta-boxes/show-screen-advanced', true ) ) {
				add_meta_box(
					'boombox-post-metabox',
					__('Boombox Post Advanced Fields', 'boombox'),
					array($this, 'render_metabox'),
					$this->object_type,
					'side',
					'high'
				);
			}

			if( apply_filters( 'boombox/admin/post/meta-boxes/show-screen-main', true ) ) {
				add_meta_box(
					'boombox-post-metabox-2',
					__('Boombox Post Advanced Fields', 'boombox'),
					array($this, 'render_metabox_2'),
					$this->object_type,
					'normal',
					'high'
				);
			}
		}

		/**
		 * Enqueue Scripts and Styles
		 */
		public function admin_enqueue_scripts() {
			$screen = get_current_screen();
			
			if ( $screen && ( $this->object_type === $screen->id ) ) {
				wp_enqueue_style( 'boombox-admin-meta-style', BOOMBOX_ADMIN_URL . 'metaboxes/assets/css/boombox-metabox-style.css', array(), boombox_get_assets_version() );
				wp_enqueue_script( 'boombox-admin-meta-script', BOOMBOX_ADMIN_URL . 'metaboxes/assets/js/boombox-metabox-script.js', array( 'jquery' ), boombox_get_assets_version(), true );
			}
		}

		/**
		 * Render the advanced fields meta box.
		 *
		 * @param $post
		 */
		public function render_metabox( $post ) {

			// Add nonce for security and authentication.
			wp_nonce_field( 'boombox_advanced_fields_nonce_action', 'boombox_nonce' );

			// Use get_post_meta to retrieve an existing value from the database.
			$boombox_is_featured = boombox_get_post_meta( $post->ID, 'boombox_is_featured' );
			$boombox_is_featured = empty( $boombox_is_featured ) ? false : true;

			$boombox_is_featured_frontpage = boombox_get_post_meta( $post->ID, 'boombox_is_featured_frontpage' );
			$boombox_is_featured_frontpage = empty( $boombox_is_featured_frontpage ) ? false : true;

			$boombox_keep_trending = boombox_get_post_meta( $post->ID, 'boombox_keep_trending' );
			$boombox_keep_trending = empty( $boombox_keep_trending ) ? false : true;

			$boombox_keep_hot = boombox_get_post_meta( $post->ID, 'boombox_keep_hot' );
			$boombox_keep_hot = empty( $boombox_keep_hot ) ? false : true;

			$boombox_keep_popular = boombox_get_post_meta( $post->ID, 'boombox_keep_popular' );
			$boombox_keep_popular = empty( $boombox_keep_popular ) ? false : true;

			// Display the form, using the current value.
			?>
			<div class="boombox-advanced-fields">

				<?php // Featured Field ?>
				<div class="boombox-form-row">
					<input type="hidden" name="boombox_is_featured" value="0" />
					<input type="checkbox" id="boombox_is_featured" name="boombox_is_featured" value="1" <?php checked( $boombox_is_featured, true, true ); ?> />
					<label for="boombox_is_featured"><?php esc_html_e( 'Featured', 'boombox' ); ?></label>
				</div>

				<?php // Featured On Homepage Field ?>
				<div class="boombox-form-row">
					<input type="hidden" name="boombox_is_featured_frontpage" value="0" />
					<input type="checkbox" id="boombox_is_featured_frontpage" name="boombox_is_featured_frontpage" value="10" <?php checked( $boombox_is_featured_frontpage, true, true ); ?> />
					<label for="boombox_is_featured_frontpage"><?php esc_html_e( 'Featured On Front Page', 'boombox' ); ?></label>
				</div>

				<?php // Keep Trending ?>
				<div class="boombox-form-row">
					<input type="hidden" name="boombox_keep_trending" value="0" />
					<input type="checkbox" id="boombox_keep_trending" name="boombox_keep_trending" value="1" <?php checked( $boombox_keep_trending, true, true ); ?> />
					<label for="boombox_keep_trending"><?php esc_html_e( 'Keep Trending', 'boombox' ); ?></label>
				</div>

				<?php // Keep Hot ?>
				<div class="boombox-form-row">
					<input type="hidden" name="boombox_keep_hot" value="0" />
					<input type="checkbox" id="boombox_keep_hot" name="boombox_keep_hot" value="1" <?php checked( $boombox_keep_hot, true, true ); ?> />
					<label for="boombox_keep_hot"><?php esc_html_e( 'Keep Hot', 'boombox' ); ?></label>
				</div>

				<?php // Keep Popular ?>
				<div class="boombox-form-row">
					<input type="hidden" name="boombox_keep_popular" value="0" />
					<input type="checkbox" id="boombox_keep_popular" name="boombox_keep_popular" value="1" <?php checked( $boombox_keep_popular, true, true ); ?> />
					<label for="boombox_keep_popular"><?php esc_html_e( 'Keep Popular', 'boombox' ); ?></label>
				</div>

			</div>
		<?php
		}

		/**
		 * Render the advanced fields meta box.
		 *
		 * @param $post
		 */
		function render_metabox_2( $post ){

			$this->data = boombox_get_post_meta( $post->ID );

			$this->render();

		}

		/**
		 * Handles saving the meta box.
		 *
		 * @param int $post_id Post ID.
		 * @param WP_Post $post Post object.
		 *
		 * @return null
		 */
		public function save_metabox( $post_id, $post ) {
		    
		    if( $post->post_type != $this->object_type ) {
		        return;
            }

			parent::save_data( $post_id );

			/* OK, it's safe for us to save the data now. */
			if ( isset( $_POST['boombox_is_featured'] ) ) {
				$boombox_is_featured = $_POST['boombox_is_featured'] ? 1 : 0;
				update_post_meta( $post_id, 'boombox_is_featured', (int) $boombox_is_featured );
			}

			if ( isset( $_POST['boombox_is_featured_frontpage'] ) ) {
				$boombox_is_featured_frontpage = $_POST['boombox_is_featured_frontpage'] ? $_POST['boombox_is_featured_frontpage'] : 0;
				update_post_meta( $post_id, 'boombox_is_featured_frontpage', (int) $boombox_is_featured_frontpage );
			}

			if ( isset( $_POST['boombox_keep_trending'] ) ) {
				$boombox_keep_trending = $_POST['boombox_keep_trending'] ? PHP_INT_MAX : false;
				update_post_meta( $post_id, 'boombox_keep_trending', $boombox_keep_trending );
			}

			if ( isset( $_POST['boombox_keep_hot'] ) ) {
				$boombox_keep_hot = $_POST['boombox_keep_hot'] ? PHP_INT_MAX : false;
				update_post_meta( $post_id, 'boombox_keep_hot', $boombox_keep_hot );
			}

			if ( isset( $_POST['boombox_keep_popular'] ) ) {
				$boombox_keep_popular = $_POST['boombox_keep_popular'] ? PHP_INT_MAX : false;
				update_post_meta( $post_id, 'boombox_keep_popular', $boombox_keep_popular );
			}

			do_action( 'boombox/admin/' . $this->object_type . '/meta-boxes/save', $post_id, $post );

		}

	}

	Boombox_Post_Meta_Box::get_instance();
}