<?php
/**
 * Register a page meta box using a class.
 *
 * @package BoomBox_Theme
 * @since   1.0.0
 * @version 2.0.0
 */

// Prevent direct script access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

if ( ! class_exists( 'Boombox_Page_Meta_Box' ) ) {

	if ( ! class_exists( 'AIOM_Base' ) ) {
		require_once( 'all-in-one-meta' . DIRECTORY_SEPARATOR . 'class-aiom-base.php' );
	}

	/**
	 * Class Boombox_Page_Meta_Box
	 */
	final class Boombox_Page_Meta_Box extends AIOM_Base {

		/**
		 * Holds page version
		 */
		const PAGE_VERSION = '2.0.0';

		/**
		 * Holds wp object type: taxonomy or post
		 * @var string
		 */
		protected $object = 'post';

		/**
		 * Hold object type: post, page or any other registered post type
		 * @var string
		 */
		protected $object_type = 'page';

		/**
		 * Holds condition for current post check against trending
		 * @var bool
		 */
		public $is_trending_page = false;

		/**
		 * Holds class single instance
		 * @var null
		 */
		private static $_instance = NULL;

		/**
		 * Get instance
		 * @return Boombox_Page_Meta_Box|null
		 */
		public static function get_instance() {

			if ( NULL == static::$_instance ) {
				static::$_instance = new self();
			}

			return static::$_instance;

		}

		/**
		 * Boombox_Page_Meta_Box constructor.
		 */
		private function __construct() {
			$this->hooks();
		}

		/**
		 * A dummy magic method to prevent Boombox_Page_Meta_Box from being cloned.
		 *
		 */
		public function __clone() {
			throw new Exception( 'Cloning ' . __CLASS__ . ' is forbidden' );
		}

		/**
		 * Setup Hooks
		 */
		public function hooks() {
			add_action( 'add_meta_boxes', array( $this, 'add_metabox' ), 1 );
			add_action( 'save_post', array( $this, 'save_metabox' ), 10, 2 );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		}

		/**
		 * Setup structure
		 * @return array
		 */
		public function get_structure() {
			global $post;

			$choices_helper = Boombox_Choices_Helper::get_instance();
			$this->is_trending_page = $this->is_trending_admin_page_template( $post );

			$this->get_listing_type_meta_key( $post );
			$excluded_listing_types = array();
			if ( $this->is_trending_page ) {
				$excluded_listing_types[] = 'none';
			}
			$listing_types_choices = $choices_helper->get_listing_types( 'value=>image', $excluded_listing_types );
			$boombox_conditions_choices = $choices_helper->get_conditions();
			$boombox_time_range_choices = $choices_helper->get_time_ranges();
			$boombox_category_choices = $choices_helper->get_categories();

			$structure = array(
				'tab_global'  => array(
					'title'  => __( 'Main', 'boombox' ),
					'active' => true,
					'icon'   => false,
					'order'  => 20,
					'fields' => array(
						// Sidebar Type
						'boombox_sidebar_type'        => array(
							'id'       => 'boombox_sidebar_type',
							'name'     => 'boombox_sidebar_type',
							'type'     => 'radio-image',
							'label'    => __( 'Sidebar Type', 'boombox' ),
							'priority' => 30,
							'default'  => '1-sidebar-1_3',
							'choices'  => $choices_helper->get_sidebar_types(),
							'callback' => 'sanitize_text_field',
						),
						// Sidebar Orientation
						'boombox_sidebar_orientation' => array(
							'name'       => 'boombox_sidebar_orientation',
							'type'       => 'radio',
							'label'      => __( 'Sidebar Orientation', 'boombox' ),
							'priority'   => 40,
							'default'    => 'right',
							'config'     => array(
								'axis' => 'vertical',
							),
							'choices'    => array(
								'right' => __( 'Right', 'boombox' ),
								'left'  => __( 'Left', 'boombox' ),
							),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_sidebar_type',
									'value'    => 'no-sidebar',
									'compare'  => '!=',
								),
							),
							'callback'   => 'sanitize_text_field',
						),
						// Primary sidebar
						'boombox_primary_sidebar'     => array(
							'name'       => 'boombox_primary_sidebar',
							'type'       => 'select',
							'label'      => __( 'Primary sidebar', 'boombox' ),
							'priority'   => 50,
							'default'    => 'default-sidebar',
							'choices'    => $choices_helper->get_primary_sidebars(),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_sidebar_type',
									'value'    => 'no-sidebar',
									'compare'  => '!=',
								),
							),
							'callback'   => 'sanitize_text_field',
						),
						// Secondary sidebar
						'boombox_secondary_sidebar'   => array(
							'name'       => 'boombox_secondary_sidebar',
							'type'       => 'select',
							'label'      => __( 'Secondary sidebar', 'boombox' ),
							'priority'   => 60,
							'default'    => 'page-secondary',
							'choices'    => $choices_helper->get_secondary_sidebars(),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_sidebar_type',
									'value'    => array( '2-sidebars-1_4-1_4', '2-sidebars-small-big' ),
									'compare'  => 'IN',
								)
							),
							'callback'   => 'sanitize_text_field',
						)
						// other fields go here
					),
				),
				'tab_featured_area' => array(
					'title'  => __( 'Featured Area', 'boombox' ),
					'active' => false,
					'icon'   => false,
					'order'  => 30,
					'fields' => array(
						// Featured Area Type
						'boombox_featured_area_type'       => array(
							'id'       => 'boombox_featured_area_type',
							'name'     => 'boombox_featured_area_type',
							'type'     => 'radio-image',
							'label'    => __( 'Area Type', 'boombox' ),
							'priority' => 20,
							'default'  => 'type-1-1',
							'choices'  => $choices_helper->get_featured_area_types(),
							'callback' => 'sanitize_text_field',
						),
						// "Layout" heading
						'boombox_featured_area_layout_heading' => array(
							'name'     => 'boombox_featured_area_layout_heading',
							'type'     => 'custom',
							'priority' => 30,
							'sub_priority' => 1,
							'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Layout', 'boombox' ) ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
						),
						// Disable Gap Between Thumbnails
						'boombox_featured_disable_gap'      => array(
							'name'       => 'boombox_featured_disable_gap',
							'type'       => 'checkbox',
							'label'      => __( 'Disable Gap Between Thumbnails', 'boombox' ),
							'priority'   => 30,
							'sub_priority' => 2,
							'default'    => 0,
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
							'callback'   => array( $this, 'sanitize_checkbox' ),
						),
						// Featured Area Hide Elements
						'boombox_featured_hide_elements'   => array(
							'name'       => 'boombox_featured_hide_elements',
							'type'       => 'multicheck',
							'label'      => __( 'Hide Elements', 'boombox' ),
							'priority'   => 40,
							'default'    => array(),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
							'choices'    => $choices_helper->get_featured_area_hide_elements(),
							'callback'   => array( $this, 'sanitize_multicheck' ),
						),
						// "Posts Loop" heading
						'boombox_featured_area_posts_loop_heading' => array(
							'name'     => 'boombox_featured_area_posts_loop_heading',
							'type'     => 'custom',
							'priority' => 50,
							'sub_priority' => 1,
							'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Posts Loop', 'boombox' ) ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
						),
						// Featured Area Conditions
						'boombox_featured_area_conditions' => array(
							'name'       => 'boombox_featured_area_conditions',
							'type'       => 'select',
							'label'      => __( 'Order Criteria', 'boombox' ),
							'priority'   => 50,
							'sub_priority' => 2,
							'default'    => 0,
							'choices'    => $boombox_conditions_choices,
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
							'callback'   => 'sanitize_text_field',
						),
						// Featured Area Time Range
						'boombox_featured_area_time_range' => array(
							'name'       => 'boombox_featured_area_time_range',
							'type'       => 'select',
							'label'      => __( 'Time Range', 'boombox' ),
							'priority'   => 60,
							'default'    => 0,
							'choices'    => $boombox_time_range_choices,
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
							'callback'   => 'sanitize_text_field',
						),
						// Featured Area Categories
						'boombox_featured_area_category'   => array(
							'id'         => 'boombox_featured_area_category',
							'name'       => 'boombox_featured_area_category',
							'type'       => 'select',
							'label'      => __( 'Categories Filter', 'boombox' ),
							'attributes' => array(
								'multiple' => 'multiple',
								'size'     => 6,
							),
							'priority'   => 70,
							'default'    => array( '' ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
							'choices'    => $boombox_category_choices,
						),
						// Featured Area Tags
						'boombox_featured_area_tags'       => array(
							'name'           => 'boombox_featured_area_tags',
							'type'           => 'textarea',
							'label'          => __( 'Tags Filter', 'boombox' ),
							'description'    => __( 'Comma separated list of tags slugs', 'boombox' ),
							'class'          => 'large-text',
							'attributes'     => array(
								'rows' => 6,
							),
							'priority'       => 80,
							'default'        => 0,
							'callback'       => array( $this, 'sanitize_tags_field' ),
							'active_callback'     => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
							'value_callback' => array( $this, 'render_tags_field' ),
						),
						'boombox_featured_area_exclude_from_main_loop' => array(
							'name'       => 'boombox_featured_area_exclude_from_main_loop',
							'type'       => 'checkbox',
							'label'      => __( 'Exclude Featured Entries From Main Posts Loop', 'boombox' ),
							'priority'   => 90,
							'default'    => 1,
							'active_callback' => array(
								array(
									'field_id' => 'boombox_featured_area_type',
									'value'    => 'disable',
									'compare'  => '!=',
								)
							),
						)
						// other fields go here
					)
				),
				'tab_listing' => array(
					'title'  => __( 'Posts Listing', 'boombox' ),
					'active' => false,
					'icon'   => false,
					'order'  => 40,
					'fields' => array(
						// Listing Type
						'boombox_listing_type' => array(
							'id'       => 'boombox_listing_type',
							'label'    => __( 'Listing Type', 'boombox' ),
							'type'     => 'radio-image',
							'name'     => $this->is_trending_page ? 'boombox_trending_listing_type' : 'boombox_listing_type',
							'choices'  => $listing_types_choices,
							'priority' => 20,
							'default'  => $this->is_trending_page ? 'list' : 'none',
							'callback' => 'sanitize_text_field',
						),
						// "Layout" heading
						'boombox_listing' => array(
							'name'     => 'boombox_listing',
							'type'     => 'custom',
							'priority' => 30,
							'sub_priority' => 1,
							'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Layout', 'boombox' ) ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_listing_type',
									'value'    => 'none',
									'compare'  => '!=',
								)
							),
						),
						// Hide elements
						'boombox_listing_hide_elements'           => array(
							'label'      => __( 'Hide Elements', 'boombox' ),
							'type'       => 'multicheck',
							'name'       => 'boombox_listing_hide_elements',
							'choices'    => $choices_helper->get_grid_hide_elements(),
							'priority'   => 30,
							'sub_priority' => 10,
							'active_callback' => array(
								array(
									'field_id' => 'boombox_listing_type',
									'value'    => 'none',
									'compare'  => '!=',
								)
							),
							'default'    => array(),
							'callback'   => array( $this, 'sanitize_multicheck' ),
						),
						// "Posts Loop" heading
						'boombox_listing_loop_heading' => array(
							'name'     => 'boombox_listing_loop_heading',
							'type'     => 'custom',
							'priority' => 40,
							'sub_priority' => 1,
							'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Posts Loop', 'boombox' ) ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_listing_type',
									'value'    => 'none',
									'compare'  => '!=',
								)
							),
						),
						// Pagination type
						'boombox_pagination_type'                 => array(
							'label'      => __( 'Pagination Type', 'boombox' ),
							'type'       => 'select',
							'name'       => 'boombox_pagination_type',
							'choices'    => $choices_helper->get_pagination_types(),
							'priority'   => 40,
							'sub_priority' => 200,
							'default'    => 'load_more',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_listing_type',
									'value'    => 'none',
									'compare'  => '!=',
								)
							),
							'callback'   => 'sanitize_text_field',
						),
						// Posts Per Page
						'boombox_posts_per_page'                  => array(
							'label'      => __( 'Posts Per Page', 'boombox' ),
							'type'       => 'number',
							'name'       => 'boombox_posts_per_page',
							'priority'   => 50,
							'default'    => get_option( 'posts_per_page' ),
							'attributes' => array(
								'min' => 1,
							),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_listing_type',
									'value'    => 'none',
									'compare'  => '!=',
								)
							),
							'callback'   => 'sanitize_text_field',
						),
						// other fields go here
					),
				),
				'tab_title_area'  => array(
					'title'  => __( 'Title Area', 'boombox' ),
					'active' => false,
					'icon'   => false,
					'order'  => 50,
					'fields' => array(
						// Disable Title Area
						'boombox_hide_title_area' => array(
							'name'     => 'boombox_hide_title_area',
							'type'     => 'checkbox',
							'label'    => __( 'Disable Title Area', 'boombox' ),
							'priority' => 20,
							'default'  => 0,
							'callback' => array( $this, 'sanitize_checkbox' ),
						),
						// Style
						'boombox_title_area_style' => array(
							'name'     => 'boombox_title_area_style',
							'type'     => 'select',
							'label'    => __( 'Style', 'boombox' ),
							'priority' => 30,
							'default'  => 'style1',
							'choices'  => $choices_helper->get_template_header_style_choices(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare' => '==',
									'value'   => 0
								)
							)
						),
						// Container Type
						'boombox_title_area_background_container' => array(
							'name'     => 'boombox_title_area_background_container',
							'type'     => 'select',
							'label'    => __( 'Container Type', 'boombox' ),
							'priority' => 40,
							'default'  => 'boxed',
							'choices'  => $choices_helper->get_template_header_background_container_choices(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare' => '==',
									'value'   => 0
								)
							)
						),
						// Background Color
						'boombox_title_area_text_color' => array(
							'name'     => 'boombox_title_area_text_color',
							'type'     => 'color',
							'label'    => __( 'Text Color', 'boombox' ),
							'priority' => 50,
							'default'  => '',
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								'relation' => 'AND',
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare' => '==',
									'value'   => 0
								),
							)
						),
						// Background Color
						'boombox_title_area_background_color' => array(
							'name'     => 'boombox_title_area_background_color',
							'type'     => 'color',
							'label'    => __( 'Background Color', 'boombox' ),
							'priority' => 60,
							'default'  => '',
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								'relation' => 'AND',
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare' => '==',
									'value'   => 0
								),
							)
						),
						// Gradient Color
						'boombox_title_area_background_gradient_color' => array(
							'name'            => 'boombox_title_area_background_gradient_color',
							'type'            => 'color',
							'label'           => __( 'Gradient Color', 'boombox' ),
							'priority'        => 70,
							'default'         => '',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare'  => '==',
									'value'    => 0,
								),
							),
						),
						// Gradient Direction
						'boombox_title_area_background_gradient_direction' => array(
							'name'     => 'boombox_title_area_background_gradient_direction',
							'type'     => 'select',
							'label'    => __( 'Gradient Direction', 'boombox' ),
							'priority' => 80,
							'default'  => 'top',
							'choices'  => $choices_helper->get_template_header_background_gradient_direction_choices(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare'  => '==',
									'value'    => 0,
								),
							),
						),
						// Background Image
						'boombox_title_area_background_image' => array(
							'name'     => 'boombox_title_area_background_image',
							'type'     => 'image',
							'label'    => __( 'Background Image', 'boombox' ),
							'priority' => 90,
							'default'  => '',
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare'  => '==',
									'value'    => 0,
								),
							),

						),
						// Background Image Size
						'boombox_title_area_background_image_size' => array(
							'name'     => 'boombox_title_area_background_image_size',
							'type'     => 'select',
							'label'    => __( 'Background Image Size', 'boombox' ),
							'priority' => 100,
							'default'  => 'auto',
							'choices'  => $choices_helper->get_template_header_background_image_size_choices(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare'  => '==',
									'value'    => 0,
								),
							),
						),
						// Background Image Position
						'boombox_title_area_background_image_position' => array(
							'name'     => 'boombox_title_area_background_image_position',
							'type'     => 'select',
							'label'    => __( 'Background Image Position', 'boombox' ),
							'priority' => 110,
							'default'  => 'center',
							'choices'  => $choices_helper->get_template_header_background_image_position_choices(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare'  => '==',
									'value'    => 0,
								),
							),
						),
						// Background Image Repeat
						'boombox_title_area_background_image_repeat' => array(
							'name'     => 'boombox_title_area_background_image_repeat',
							'type'     => 'select',
							'label'    => __( 'Background Image Repeat', 'boombox' ),
							'priority' => 120,
							'default'  => 'repeat-no',
							'choices'  => $choices_helper->get_template_header_background_image_repeat_choices(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_hide_title_area',
									'compare'  => '==',
									'value'    => 0,
								),
							),
						),
					),
				),
				'tab_posts_strip' => array(
					'title'  => __( 'Posts Strip', 'boombox' ),
					'active' => false,
					'icon'   => false,
					'order'  => 60,
					'fields' => array(
						// Configuration
						'boombox_strip_configuration' => array(
							'label'    => __( 'Configuration', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_configuration',
							'priority' => 30,
							'default'  => 'inherit',
							'choices'  => array_merge(
								array( 'none' => __( 'None', 'boombox' ) ),
								$choices_helper->get_strip_configurations()
							),
							'callback' => 'sanitize_text_field',
						),
						// "Layout" heading
						'boombox_strip_layout_heading' => array(
							'name'     => 'boombox_strip_layout_heading',
							'type'     => 'custom',
							'priority' => 40,
							'sub_priority' => 1,
							'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Layout', 'boombox' ) ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Type
						'boombox_strip_type' => array(
							'label'    => __( 'Type', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_type',
							'priority' => 40,
							'sub_priority' => 2,
							'default'  => 'slider',
							'choices'  => $choices_helper->get_strip_types(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Width
						'boombox_strip_width' => array(
							'label'    => __( 'Width', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_width',
							'priority' => 50,
							'default'  => 'boxed',
							'choices'  => $choices_helper->get_strip_dimensions(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Size
						'boombox_strip_size' => array(
							'label'    => __( 'Size', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_size',
							'priority' => 60,
							'default'  => 'big',
							'choices'  => $choices_helper->get_strip_sizes(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Titles Position
						'boombox_strip_title_position' => array(
							'label'    => __( 'Titles Position', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_title_position',
							'priority' => 70,
							'default'  => 'inside',
							'choices'  => $choices_helper->get_strip_title_positions(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Gap Between Thumbnails
						'boombox_strip_disable_gap' => array(
							'label'    => __( 'Disable Gap Between Thumbnails', 'boombox' ),
							'type'     => 'checkbox',
							'name'     => 'boombox_strip_disable_gap',
							'priority' => 80,
							'default'  => 1,
							'callback' => array( $this, 'sanitize_checkox' ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// "Posts Loop" heading
						'boombox_strip_posts_loop_heading' => array(
							'name'     => 'boombox_strip_posts_loop_heading',
							'type'     => 'custom',
							'priority' => 90,
							'sub_priority' => 1,
							'html'     => sprintf( '<h2>%s</h2><hr/>', __( 'Posts Loop', 'boombox' ) ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Order Criteria
						'boombox_strip_conditions' => array(
							'label'    => __( 'Order Criteria', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_conditions',
							'priority' => 90,
							'sub_priority' => 2,
							'default'  => 'recent',
							'choices'  => $choices_helper->get_conditions(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Time Range
						'boombox_strip_time_range' => array(
							'label'    => __( 'Time Range', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_time_range',
							'priority' => 100,
							'default'  => 'all',
							'choices'  => $choices_helper->get_time_ranges(),
							'callback' => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Items Count
						'boombox_strip_items_count' => array(
							'label'       => __( 'Items Count', 'boombox' ),
							'description' => __( 'Minimum count: 6. To show all items, please enter -1.', 'boombox' ),
							'type'        => 'number',
							'name'        => 'boombox_strip_items_count',
							'priority'    => 110,
							'default'     => 18,
							'attributes'   => array(
								'min'  => -1,
								'step' => 1
							),
							'callback'    => 'sanitize_text_field',
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Categories Filter
						'boombox_strip_category' => array(
							'label'    => __( 'Categories Filter', 'boombox' ),
							'type'     => 'select',
							'name'     => 'boombox_strip_category',
							'priority' => 120,
							'default'  => array( '' ),
							'choices'  => $choices_helper->get_categories(),
							'attributes' => array(
								'multiple' => 'multiple',
								'size'     => 6,
							),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
						// Tags Filter
						'boombox_strip_tags' => array(
							'label'       => __( 'Tags Filter', 'boombox' ),
							'description' => __( 'Comma separated list of tags slugs', 'boombox' ),
							'type'        => 'textarea',
							'name'        => 'boombox_strip_tags',
							'attributes'     => array(
								'rows' => 6,
							),
							'priority'    => 130,
							'default'     => '',
							'callback'    => array( $this, 'sanitize_tags_field' ),
							'value_callback' => array( $this, 'render_tags_field' ),
							'active_callback' => array(
								array(
									'field_id' => 'boombox_strip_configuration',
									'compare' => '==',
									'value'   => 'custom'
								)
							)
						),
					),
				)
				// other tabs go here
			);

			if( ! $this->is_trending_page ) {
				// Tab "Listing"
				$structure[ 'tab_listing' ][ 'fields' ] = array_merge( $structure[ 'tab_listing' ][ 'fields' ], array(
					// Order Criteria
					'boombox_listing_condition'  => array(
						'label'           => __( 'Order Criteria', 'boombox' ),
						'type'            => 'select',
						'name'            => 'boombox_listing_condition',
						'choices'         => $boombox_conditions_choices,
						'priority'        => 40,
						'sub_priority'    => 10,
						'default'         => 'recent',
						'active_callback' => array(
							array(
								'field_id' => 'boombox_listing_type',
								'value'    => 'none',
								'compare'  => '!=',
							)
						),
						'callback'        => 'sanitize_text_field',
					),
					// Time Range
					'boombox_listing_time_range' => array(
						'label'           => __( 'Time Range', 'boombox' ),
						'type'            => 'select',
						'name'            => 'boombox_listing_time_range',
						'choices'         => $boombox_time_range_choices,
						'priority'        => 40,
						'sub_priority'    => 20,
						'default'         => 'all',
						'active_callback' => array(
							array(
								'field_id' => 'boombox_listing_type',
								'value'    => 'none',
								'compare'  => '!=',
							)
						),
						'callback'        => 'sanitize_text_field',
					),
					// Categories Filter
					'boombox_listing_categories' => array(
						'label'           => __( 'Categories Filter', 'boombox' ),
						'type'            => 'select',
						'name'            => 'boombox_listing_categories',
						'choices'         => $boombox_category_choices,
						'priority'        => 40,
						'sub_priority'    => 30,
						'default'         => array( '' ),
						'active_callback' => array(
							array(
								'field_id' => 'boombox_listing_type',
								'value'    => 'none',
								'compare'  => '!=',
							)
						),
						'attributes'      => array(
							'multiple' => 'multiple',
						),
					),
					// Tags Filter
					'boombox_listing_tags'       => array(
						'label'           => __( 'Tags Filter', 'boombox' ),
						'description'     => __( 'Comma separated list of tags slugs', 'boombox' ),
						'type'            => 'textarea',
						'name'            => 'boombox_listing_tags',
						'priority'        => 40,
						'sub_priority'    => 40,
						'default'         => '',
						'attributes'      => array(
							'rows' => '5',
						),
						'active_callback' => array(
							array(
								'field_id' => 'boombox_listing_type',
								'value'    => 'none',
								'compare'  => '!=',
							)
						),
						'callback'        => array( $this, 'sanitize_tags_field' ),
						'value_callback'  => array( $this, 'render_tags_field' ),
					),
				) );

				// Tab "Title Area"
				$structure[ 'tab_title_area' ][ 'fields' ] = array_merge( $structure[ 'tab_title_area' ][ 'fields' ], array(
					// Hide Filter
					'boombox_title_area_hide_filter' => array(
						'name'     => 'boombox_title_area_hide_filter',
						'type'     => 'checkbox',
						'label'    => __( 'Hide Filter', 'boombox' ),
						'priority' => 130,
						'default'  => true,
						'callback' => 'sanitize_text_field',
						'active_callback' => array(
							array(
								'field_id' => 'boombox_hide_title_area',
								'compare' => '==',
								'value'   => 0
							)
						)
					),
				) );
			}

			return apply_filters( 'boombox/admin/page/meta-boxes/structure', $structure, $this );
		}

		/**
		 * Add meta box.
		 */
		public function add_metabox() {
			/**
			 * Add Advanced Fields to Post screen
			 */
			if ( apply_filters( 'boombox/admin/page/meta-boxes/show-screen-main', true ) ) {
				add_meta_box(
					'boombox-page-metabox',
					__( 'Boombox Page Advanced Fields', 'boombox' ),
					array( $this, 'render_metabox' ),
					$this->object_type,
					'normal',
					'high'
				);
			}
		}

		/**
		 * Enqueue Scripts and Styles
		 */
		public function admin_enqueue_scripts() {
			$screen = get_current_screen();

			if ( $screen && ( $this->object_type === $screen->id ) ) {
				wp_enqueue_style( 'boombox-admin-meta-style', BOOMBOX_ADMIN_URL . 'metaboxes/assets/css/boombox-metabox-style.css', array(), boombox_get_assets_version() );
				wp_enqueue_script( 'boombox-admin-meta-script', BOOMBOX_ADMIN_URL . 'metaboxes/assets/js/boombox-metabox-script.js', array( 'jquery' ), boombox_get_assets_version(), true );
			}
		}

		/**
		 * Render the advanced fields meta box.
		 *
		 * @param $post
		 */
		function render_metabox( $post ) {

			$this->data = boombox_get_post_meta( $post->ID );

			$this->render();

		}

		/**
		 * Handles saving the meta box.
		 *
		 * @param int     $post_id Post ID.
		 * @param WP_Post $post    Post object.
		 *
		 * @return null
		 */
		public function save_metabox( $post_id, $post ) {

			if ( $post->post_type != $this->object_type ) {
				return;
			}

			add_filter( 'boombox/admin/' . $this->object_type . '/meta-boxes/save/values', array( $this, 'set_page_version' ), 10, 2 );
			parent::save_data( $post_id );
			remove_filter( 'boombox/admin/' . $this->object_type . '/meta-boxes/save/values', array( $this, 'set_page_version' ), 10 );

			do_action( 'boombox/admin/' . $this->object_type . '/meta-boxes/save', $post_id, $post );

		}

		/**
		 * Get trending page ID by trending type
		 *
		 * @param string $type 'trending' |'hot' |'popular'
		 *
		 * @return int
		 */
		public function get_trending_page_id( $type ) {
			$customize_setting_slug = 'extras_post_ranking_system_' . $type . '_page';
			$trending_page_id = boombox_get_theme_option( $customize_setting_slug );

			return absint( $trending_page_id );
		}

		/**
		 * Return true, if is admin trending page
		 *
		 * @param WP_Post $post Current post
		 *
		 * @return bool
		 */
		public function is_trending_admin_page_template( $post ) {

			global $current_screen;
			$is_trending_page = false;
			$trending_page_id = $this->get_trending_page_id( 'trending' );
			$hot_page_id = $this->get_trending_page_id( 'hot' );
			$popular_page_id = $this->get_trending_page_id( 'popular' );

			if (
				( $this->object_type == $current_screen->post_type )
				&& (
					( $trending_page_id == $post->ID )
					|| ( $hot_page_id == $post->ID )
					|| ( $popular_page_id == $post->ID )
				)
			) {
				$is_trending_page = true;
			}

			return $is_trending_page;
		}

		/**
		 * Get meta key for listing_type
		 *
		 * @param $post
		 *
		 * @return string
		 */
		public function get_listing_type_meta_key( $post ) {
			return $this->is_trending_admin_page_template( $post ) ? 'boombox_trending_listing_type' : 'boombox_listing_type';
		}

		/**
		 * Sanitize tags field
		 *
		 * @param string $value Current value
		 *
		 * @return array
		 */
		public function sanitize_tags_field( $value = '' ) {
			$value = sanitize_text_field( $value );
			if ( $value ) {
				$value = explode( ',', preg_replace( '/\s+/', '', $value ) );
			} else {
				$value = array();
			}

			return $value;
		}

		/**
		 * Render tags field value
		 *
		 * @param array|string $value Current value
		 *
		 * @return string
		 */
		public function render_tags_field( $value ) {
			return implode( ',', (array)$value );
		}

		/**
		 * Set page version for future usage
		 *
		 * @param array   $metadata Current meta data
		 * @param int     $post_id  Post ID
		 * @param WP_Post $post     Post object
		 *
		 * @return array
		 */
		public function set_page_version( $metadata, $post_id ) {
			if ( ! isset( $metadata[ 'boombox_page_version' ] )
				|| isset(
					$metadata[ 'boombox_page_version' ] )
				&& ( version_compare( $metadata[ 'boombox_page_version' ], self::PAGE_VERSION ) == '-1' )
			) {
				$metadata[ 'boombox_page_version' ] = self::PAGE_VERSION;
			}

			return $metadata;
		}

	}

	Boombox_Page_Meta_Box::get_instance();
}