<?php
/**
 * Register a post meta box using a class.
 *
 * @package BoomBox_Theme
 * @since   2.0.0
 * @version 2.0.0
 */

// Prevent direct script access.
if ( !defined( 'ABSPATH' ) ) {
	die( 'No direct script access allowed' );
}

/**
 * Class AIOM_Base
 */
abstract class AIOM_Base {

	abstract function get_structure();

	const KEY = 'boombox_meta';

	public $data;

	/**
	 * Holds wp object type: taxonomy or post
	 */
	protected $object = NULL;

	/**
	 * Hold object type: post, page or any other registered post type for $object = 'post'
	 *                   category, post_tag or any other registered taxonomy for $object = 'taxonomy'
	 * @var string|NULL
	 */
	protected $object_type = NULL;

	/**
	 * Render field
	 *
	 * @param array       $field  Field arguments
	 * @param string|bool $tab_id Field tab ID
	 *
	 * @throws Exception
	 */
	protected function render_field( $field, $tab_id = false ) {

		$field_type = preg_replace( '/\s+/', '', strtolower( $field[ 'type' ] ) );
		$field_type = strtr( $field_type, array( '-' => '_' ) );
		$render_action = 'render_' . $field_type;


		if ( method_exists( $this, $render_action ) ) {

			$field = $this->parse_field_args( $field );

			if ( !$field[ 'id' ] ) return;

			$this->$render_action( $field, $tab_id );
		} else {
			throw new Exception( 'Undefined field type: ' . $field_type );
		}

	}

	/**
	 * Parse field arguments
	 *
	 * @param array $field Parse field arguments
	 *
	 * @return array
	 */
	private function parse_field_args( $field ) {
		return wp_parse_args( $field, array(
			'active_callback'  => array(),
			'css'         => array(
				'id'    => '',
				'class' => '',
			),
			'default'     => '',
			'label'       => '',
			'id'          => '',
			'meta_key'    => '',
			'description' => '',
			'callback'    => '',
			'attr'        => '',
		) );
	}

	/**
	 * Render meta box
	 */
	protected function render() {

		wp_nonce_field( 'boombox_advanced_fields_nonce_action', 'boombox_nonce' ); ?>

		<div class="boombox-advanced-fields">
			<?php
			if ( count( $this->get_structure() ) > 1 ) {
				$this->renderAsTabs();
			} else {
				$this->renderAsSingle();
			} ?>
		</div>
		<?php
	}

	/**
	 * Render meta box as tabs
	 */
	protected function renderAsTabs() {
		$structure = $this->get_structure();

		uasort( $structure, function( $a, $b ){
			$a_order = isset( $a['order'] ) ? absint( $a['order'] ) : 0;
			$b_order = isset( $b['order'] ) ? absint( $b['order'] ) : 0;

			return $a_order - $b_order;
		} );

		?>
		<div class="boombox-admin-tabs boombox-admin-tabs-horizontal">
			<ul class="boombox-admin-tabs-menu">
				<?php foreach ( $structure as $t_id => $tab ) { ?>
					<li class="<?php echo $tab[ 'active' ] ? 'active' : ''; ?>">
						<a href="#<?php echo $t_id; ?>">
							<?php echo ( isset( $tab[ 'icon' ] ) && $tab[ 'icon' ] ) ? $tab[ 'icon' ] : ''; ?>
							<?php echo $tab[ 'title' ]; ?>
						</a>
					</li>
				<?php } ?>
			</ul>
			<div class="boombox-admin-tabs-content">
				<?php foreach ( $this->get_structure() as $t_id => $tab ) {
					$this->renderTabContent( $tab, $t_id );
				} ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render meta box as single tab
	 */
	protected function renderAsSingle() {
		reset( $this->get_structure() );
		$tab_id = key( $this->get_structure() );

		$tab = $this->get_structure()[ $tab_id ];
		$this->renderTabContent( $tab, $tab_id, true );
	}

	/**
	 * Render single tab content
	 *
	 * @param array      $tab       Tab data
	 * @param string     $tab_id    Tab ID
	 * @param bool|false $is_single If this is a single tab
	 */
	protected function renderTabContent( $tab, $tab_id, $is_single = false ) {
		$fields = $tab[ 'fields' ];
		uasort( $fields, function ( $a, $b ) {
			$a_priority = isset( $a[ 'priority' ] ) ? $a[ 'priority' ] : 0;
			$b_priority = isset( $b[ 'priority' ] ) ? $b[ 'priority' ] : 0;

			$return = $a_priority - $b_priority;
			if ( !$return ) {
				$a_sub_priority = isset( $a[ 'sub_priority' ] ) ? $a[ 'sub_priority' ] : 0;
				$b_sub_priority = isset( $b[ 'sub_priority' ] ) ? $b[ 'sub_priority' ] : 0;

				$return = $a_sub_priority - $b_sub_priority;
			}

			return $return;
		} );

		?>
		<div id="<?php echo $tab_id; ?>"
		     class="boombox-admin-tab-content <?php echo ( $is_single || $tab[ 'active' ] ) ? 'active' : ''; ?>">
			<?php
			foreach ( $fields as $f_id => $field ) {
				$this->render_field( array_merge( $field, array( 'id' => $f_id ) ), $tab_id );
			}
			?>
		</div>
		<?php
	}

	/**
	 * Save post meta data
	 *
	 * @param int $post_id Current post ID
	 *
	 * @return bool|int
	 */
	private function _save_post_data( $post_id ) {
		// Add nonce for security and authentication.
		$nonce_name = isset( $_POST[ 'boombox_nonce' ] ) ? $_POST[ 'boombox_nonce' ] : '';
		$nonce_action = 'boombox_advanced_fields_nonce_action';

		// Check if nonce is valid.
		if ( !wp_verify_nonce( $nonce_name, $nonce_action ) ) {
			return false;
		}

		// Check if user has permissions to save data.
		if ( !current_user_can( 'edit_post', $post_id ) ) {
			return false;
		}

		// Check if not an autosave.
		if ( wp_is_post_autosave( $post_id ) ) {
			return false;
		}

		// Check if not a revision.
		if ( wp_is_post_revision( $post_id ) ) {
			return false;
		}

		$meta_values = array();
		foreach ( $this->get_structure() as $t_key => $tab ) {
			foreach ( $tab[ 'fields' ] as $f_key => $field ) {
				$callback = isset( $field[ 'callback' ] ) ? $field[ 'callback' ] : false;
				$key = isset( $field[ 'meta_key' ] ) && $field[ 'meta_key' ] ? $field[ 'meta_key' ] : $field[ 'name' ];

				$value = isset( $_POST[ self::KEY ][ $key ] ) ? $_POST[ self::KEY ][ $key ] : '';

				if ( is_callable( $callback ) ) {
					$value = call_user_func( $callback, $value );
				}

				$value = apply_filters( 'boombox/admin/' . $this->object_type . '/meta-boxes/save-value?field=' . $key, $value, $post_id );
				$meta_values[ $key ] = $value;

			}
		}

		$meta_values = apply_filters( 'boombox/admin/' . $this->object_type . '/meta-boxes/save/values', $meta_values, $post_id );

		return update_post_meta( $post_id, self::KEY, $meta_values );
	}

	/**
	 * Save term meta data
	 *
	 * @param int $term_id    Current term ID
	 * @return bool|int
	 */
	private function _save_term_data( $term_id ) {

		return false;
	}

	/**
	 * Save meta data
	 *
	 * @param int $object_id Current object ID: The post ID for $object = 'post' and term ID for $object = 'taxonomy'
	 *
	 * @return bool|int
	 */
	protected function save_data( $object_id ) {
		switch( $this->object ) {
			case 'post':
				$result = $this->_save_post_data( $object_id );
				break;
			case 'taxonomy':
				$result = $this->_save_term_data( $object_id );
				break;
			default:
				$result = false;
		}

		return $result;

	}

	/** =========================== field types =========================== */

	/**
	 * Render "text" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_text( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Text_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'text.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Text_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "number" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_number( $field, $tab_id = false ) {

		if ( !class_exists( 'AIOM_Number_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'number.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Number_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();

	}

	/**
	 * Render "textarea" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_textarea( $field, $tab_id = false ) {

		if ( !class_exists( 'AIOM_Textarea_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'textarea.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Textarea_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "checkbox" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_checkbox( $field, $tab_id = false ) {

		if ( !class_exists( 'AIOM_Checkbox_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'checkbox.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Checkbox_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();

	}

	/**
	 * Render "radio" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_radio( $field, $tab_id = false ) {

		if ( !class_exists( 'AIOM_Radio_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'radio.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Radio_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();

	}

	/**
	 * Render "dropdown" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_select( $field, $tab_id = false ) {

		if ( !class_exists( 'AIOM_Select_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'dropdown.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Select_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "custom" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_custom( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Custom_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'custom.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Custom_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "sub tab" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_radio_image( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Radio_Image_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'radio_image.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Radio_Image_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "multicheck" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_multicheck( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Multicheck_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'multicheck.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Multicheck_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "color picker" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_color( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Color_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'color.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Color_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "color picker" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_multicolor( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Multicolor_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'multicolor.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Multicolor_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/**
	 * Render "image" field
	 *
	 * @param array       $field  Field configuration
	 * @param string|bool $tab_id Field tab ID
	 */
	private function render_image( $field, $tab_id = false ) {
		if ( !class_exists( 'AIOM_Image_Field' ) ) {
			require_once( 'fields' . DIRECTORY_SEPARATOR . 'image.php' );
		}

		$data = isset( $this->data[ self::KEY ] ) ? $this->data[ self::KEY ] : array();
		$instance = new AIOM_Image_Field( $field, $tab_id, $data, $this->get_structure() );
		$instance->render();
	}

	/** =========================== /end - field types =========================== */

	/** =========================== custom sanitation rules =========================== */

	/**
	 * Sanitize video url
	 *
	 * @param string $value Current value
	 *
	 * @return string
	 */
	protected function sanitize_video_url( $value = '' ) {
		if ( $value ) {

			if ( strpos( $value, '<iframe' ) !== false ) {
				preg_match( '/src="([^"]+)"/', $value, $src_matches );
				$value = $src_matches[ 1 ];
			}
			$video_url = trim( $value );

			while ( true ) {

				/***** "Youtube" */
				preg_match( boombox_get_regex( 'youtube' ), $video_url, $youtube_matches );
				if ( isset( $youtube_matches[ 1 ] ) && $youtube_matches[ 1 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Vimeo" */
				preg_match( boombox_get_regex( 'vimeo' ), $video_url, $vimeo_matches );
				if ( isset( $vimeo_matches[ 5 ] ) && $vimeo_matches[ 5 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Dailymotion" */
				preg_match( boombox_get_regex( 'dailymotion' ), $video_url, $dailymotion_matches );
				if ( isset( $dailymotion_matches[ 1 ] ) && $dailymotion_matches[ 1 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Vine" */
				preg_match( boombox_get_regex( 'vine' ), $video_url, $vine_matches );
				if ( isset( $vine_matches[ 1 ] ) && $vine_matches[ 1 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Ok" */
				preg_match( boombox_get_regex( 'ok' ), $video_url, $ok_matches );
				if ( isset( $ok_matches[ 2 ] ) && $ok_matches[ 2 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Facebook" */
				preg_match( boombox_get_regex( 'facebook' ), $video_url, $fb_matches );
				if ( isset( $fb_matches[ 1 ] ) && $fb_matches[ 1 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Vidme" */
				preg_match( boombox_get_regex( 'vidme' ), $video_url, $vidme_matches );
				if ( isset( $vidme_matches[ 1 ] ) && $vidme_matches[ 1 ] ) {
					$value = $video_url;
					break;
				}

				/***** "VK" */
				preg_match( boombox_get_regex( 'vk' ), $video_url, $vk_matches );
				if ( isset( $vk_matches[ 2 ] ) && $vk_matches[ 2 ] ) {
					parse_str( $vk_matches[ 2 ], $vk_matches );
					if ( isset( $vk_matches[ 'id' ], $vk_matches[ 'oid' ], $vk_matches[ 'hash' ] ) ) {
						$value = $video_url;
					}
					break;
				}

				/***** "Twitch" */
				preg_match( boombox_get_regex( 'twitch' ), $video_url, $twitch_matches );
				if ( isset( $twitch_matches[ 2 ] ) && $twitch_matches[ 2 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Coub" */
				preg_match( boombox_get_regex( 'coub' ), $video_url, $coub_matches );
				if ( isset( $coub_matches[ 3 ] ) && $coub_matches[ 3 ] ) {
					$value = $video_url;
					break;
				}

				/***** "Twitter" */
				preg_match( boombox_get_regex( 'twitter' ), $video_url, $twitter_matches );
				if ( isset( $twitter_matches[ 1 ] ) && isset( $twitter_matches[ 2 ] ) && $twitter_matches[ 1 ] && $twitter_matches[ 2 ] ) {
					$value = $video_url;
					break;
				}

				/***** "HTML video" */
				$video_type = wp_check_filetype( $video_url );
				if ( isset( $video_type[ 'type' ] ) && $video_type[ 'type' ] && preg_match( "~^(?:f|ht)tps?://~i", $video_url ) ) {
					$value = $video_url;
					break;
				}

				$value = '';
				break;
			}

		}

		return $value;
	}

	/**
	 * Sanitize checkbox value
	 *
	 * @param int $value Current value
	 *
	 * @return int|mixed
	 */
	protected function sanitize_checkbox( $value = 0 ) {
		$value = filter_var( $value, FILTER_SANITIZE_NUMBER_INT );
		if ( !$value ) {
			$value = 0;
		}

		return $value;
	}

	/**
	 * Sanitize "multicheck" value
	 * @param mixed $value Current value
	 *
	 * @return array
	 */
	protected function sanitize_multicheck( $value ) {
		if( ! $value ) {
			$value = array();
		}

		return $value;
	}
}