<?php

/**
 * Elementor Single Widget
 * @package bizex Tools
 * @since 1.0.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class Bizex_Post extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bizex-post-id';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Blog Grid', 'bizex-tools' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'appi-custom-icon';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bizex_widgets' ];
	}

	
	protected function register_controls() {

		$this->start_controls_section(
			'bizex_projects_option',
			[
				'label' => esc_html__( 'News Option', 'bizex-tools' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		
		$this->add_control(
			'post_order',
			[
				'label'     => esc_html__( 'Project Order', 'bizex-tools' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'ASC',
				'options'   => [
					'ASC'  => esc_html__( 'Ascending', 'bizex-tools' ),
					'DESC' => esc_html__( 'Descending', 'bizex-tools' ),
				],
			]
		);
		
		$this->add_control(
			'post_per_page',
			[
				'label'   => __( 'Posts Per Page', 'bizex-tools' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'default' => 5,
			]
		);
		
		$this->add_control(
			'post_categories',
			[
				'type'        => Controls_Manager::SELECT2,
				'label'       => esc_html__( 'Select Categories', 'bizex-tools' ),
				'options'     => bizex_blog_category(),
				'label_block' => true,
				'multiple'    => true,
			]
		);
		$this->add_control(
			'title_length',
			[
				'label'     => __( 'Title Length', 'bizex-tools' ),
				'type'      => Controls_Manager::NUMBER,
				'step'      => 1,
				'default'   => 20,
			]
		);
        $this->add_control(
			'more_button', [
				'label' => esc_html__( 'Blog More Button', 'bizex-tools' ),
				'default' => esc_html__( 'Read More', 'bizex-tools' ),
				'type' => Controls_Manager::TEXT,
                'label_block' => true,
			]
		);
		$this->add_control(
			'show_pagination',
			[
				'label' => esc_html__( 'Show Pagination', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'bizex-tools' ),
				'label_off' => esc_html__( 'Hide', 'bizex-tools' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);
		$this->end_controls_section();
        $this->start_controls_section(
            'section_style_1',
            [
                'label' => __( 'Section Style', 'appilo' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'Admin-color',
                'types' => [ 'gradient' ],
                'selector' => '{{WRAPPER}} .bz-blog-item .bz-blog-author span a',
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Admin Color', 'bizex-tools'),
                    ],
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'Admin-Typography',
                'label' => 'Admin Typography',
                'selector' => '{{WRAPPER}} .bz-blog-item .bz-blog-author span',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'Meta-Icon-color',
                'types' => [ 'gradient' ],
                'selector' => '{{WRAPPER}} .bz-blog-item .blog-meta a i',
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Meta Icon Color', 'bizex-tools'),
                    ],
                ],
            ]
        );
        $this->add_control(
            'Meta-Color',
            [
                'label' => esc_html__( 'Meta Color', 'bizex-tools' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bz-blog-item .blog-meta a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'Meta-Typography',
                'label' => 'Meta Typography',
                'selector' => '{{WRAPPER}} .bz-blog-item .blog-meta a',
            ]
        );
        $this->add_control(
            'Blog-Title-Color',
            [
                'label' => esc_html__( 'Blog Title Color', 'bizex-tools' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bz-blog-item .blog-meta-text h3' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'Blog-Title-Typography',
                'label' => 'Blog Title Typography',
                'selector' => '{{WRAPPER}} .bz-blog-item .blog-meta-text h3',
            ]
        );
        $this->add_control(
            'Button-Color',
            [
                'label' => esc_html__( 'Button Color', 'bizex-tools' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bz-blog-item .blog-meta-text .read_more' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .bz-blog-item .blog-meta-text .read_more:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'Button-Typography',
                'label' => 'Button Typography',
                'selector' => '{{WRAPPER}} .bz-blog-item .blog-meta-text .read_more',
            ]
        );
        $this->end_controls_section();
	}


	protected function render() {
		$settings = $this->get_settings_for_display();		
        $paged = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;
        $args = array(
			'post_type'           => 'post',
			'posts_per_page'      => !empty( $settings['post_per_page'] ) ? $settings['post_per_page'] : 1,
			'post_status'         => 'publish',
			'ignore_sticky_posts' => 1,
			'order'               => $settings['post_order'],
            'paged' => $paged
		);
		if( ! empty($settings['post_categories'] ) ){
			$args['category_name'] = implode(',', $settings['post_categories']);
		}
		
		$query = new \WP_Query( $args );
    
    ?>
    <div class="bz-blog-content">
        <div class="row justify-content-center">
            <?php
                if ( $query->have_posts() ) {
                $iten_number = 0;
                while ( $query->have_posts() ) {
                $query->the_post();
                    $bizex_thumb_size = 'bizex-img-size-1';
                    $iten_number++;
                    $title = wp_trim_words( get_the_title(), $settings['title_length'], '' );
            ?>
            <div class="col-lg-4 col-md-6 wow fadeInUp" data-wow-delay="500ms" data-wow-duration="1500ms">
                <div class="bz-blog-item">
                    <div class="blog-img position-relative">
                        <img src="<?php echo get_the_post_thumbnail_url(get_the_ID(), $bizex_thumb_size )?>" alt="">
                        <div class="bz-blog-author d-flex align-items-center position-absolute">
                            <?php bizex_main_author_avatars(28);?>
                            <span class="text-uppercase"><?php esc_html_e( 'BY:', 'bizex-tools' );?> <a href="#"> <?php the_author()?></a></span>
                        </div>
                    </div>
                    <div class="blog-meta-text headline">
                        <div class="blog-meta">
                            <a href="#"><i class="fal fa-calendar-alt"></i> <?php echo date(get_option('date_format')); ?></a>
                            <a href="#"><i class="fas fa-comments"></i> <?php esc_html_e( 'Comments', 'bizex-tools' )?> (<?php echo esc_attr(get_comments_number());?>)</a>
                        </div>
                        <h3><a href="<?php the_permalink()?>"><?php echo $title;?></a></h3>
                        <a class="read_more" href="<?php the_permalink()?>"><?php echo esc_html($settings['more_button']);?> <i class="fal fa-long-arrow-right"></i></a>
                    </div>
                </div>
            </div>
            <?php } wp_reset_query(); ?>
            <?php if($settings['show_pagination'] == 'yes'):?>
            <div class="biz-pagination d-flex text-center ul-li">
                <?php 
                    echo paginate_links( array(
                        'base'         => str_replace( 999999999, '%#%', esc_url( get_pagenum_link( 999999999 ) ) ),
                        'total'        => $query->max_num_pages,
                        'current'      => max( 1, get_query_var( 'paged' ) ),
                        'format'       => '?paged=%#%',
                        'show_all'     => false,
                        'type'         => 'plain',
                        'end_size'     => 2,
                        'mid_size'     => 1,
                        'prev_next'    => true,
                        'prev_text' => '<i class="far fa-long-arrow-left"></i>',
                        'next_text' => '<i class="far fa-long-arrow-right"></i>',
                        'add_args'     => false,
                        'add_fragment' => '',
                    ) );
                ?>
            </div>
            <?php endif;?>
        <?php } ?>
        </div>
    </div>

    <?php 
    }
		
	
}


Plugin::instance()->widgets_manager->register( new Bizex_Post() );