<?php

/**
 * Elementor Single Widget
 * @package bizex Tools
 * @since 1.0.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class Bizex_Breadcrumb extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'hos-breadcrumb-id';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Breadcrumb', 'bizex-tools' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'appi-custom-icon';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bizex_widgets' ];
	}

	
	protected function register_controls() {
		$this->start_controls_section(
			'bizex_sec_h_option',
			[
				'label' => esc_html__( 'Breadcrumb Option', 'bizex-tools' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'title_shape_bg',
				'types' => [ 'classic'],
				'exclude' => [ 'color' ],
				'selector' => '{{WRAPPER}} .bz-breadcrumb-content h2:before',
			]
		);
        $this->add_control(
			'breadcrumb_shape',
			[
				'label' => esc_html__( 'Breadcrumb Shape', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				]
			]
		);
		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter your title', 'bizex-tools' ),
				'default' => esc_html__( 'Add Your Heading Text Here', 'bizex-tools' ),
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => '',
				],
				'separator' => 'before',
			]
		);

		$this->add_control(
			'size',
			[
				'label' => esc_html__( 'Size', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => esc_html__( 'Default', 'bizex-tools' ),
					'small' => esc_html__( 'Small', 'bizex-tools' ),
					'medium' => esc_html__( 'Medium', 'bizex-tools' ),
					'large' => esc_html__( 'Large', 'bizex-tools' ),
					'xl' => esc_html__( 'XL', 'bizex-tools' ),
					'xxl' => esc_html__( 'XXL', 'bizex-tools' ),
				],
			]
		);

		$this->add_control(
			'header_size',
			[
				'label' => esc_html__( 'HTML Tag', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h2',
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => esc_html__( 'Alignment', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'bizex-tools' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'bizex-tools' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'bizex-tools' ),
						'icon' => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'bizex-tools' ),
						'icon' => 'eicon-text-align-justify',
					],
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__( 'Title', 'bizex-tools' ),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Text Color', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-heading-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'typography',
				'selector' => '{{WRAPPER}} .elementor-heading-title',
			]
		);
        

		$this->add_group_control(
			\Elementor\Group_Control_Text_Stroke::get_type(),
			[
				'name' => 'text_stroke',
				'selector' => '{{WRAPPER}} .elementor-heading-title',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'text_shadow',
				'selector' => '{{WRAPPER}} .elementor-heading-title',
			]
		);

		$this->add_control(
			'blend_mode',
			[
				'label' => esc_html__( 'Blend Mode', 'bizex-tools' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'' => esc_html__( 'Normal', 'bizex-tools' ),
					'multiply' => 'Multiply',
					'screen' => 'Screen',
					'overlay' => 'Overlay',
					'darken' => 'Darken',
					'lighten' => 'Lighten',
					'color-dodge' => 'Color Dodge',
					'saturation' => 'Saturation',
					'color' => 'Color',
					'difference' => 'Difference',
					'exclusion' => 'Exclusion',
					'hue' => 'Hue',
					'luminosity' => 'Luminosity',
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-heading-title' => 'mix-blend-mode: {{VALUE}}',
				],
				'separator' => 'none',
			]
		);

		$this->end_controls_section();
	}


	protected function render() {
		$settings = $this->get_settings_for_display();	
        extract($settings);
    ?>
    <section id="bz-breadcrumb" class="bz-breadcrumb-section">
		<div class="bz-breadcrumb-content position-relative text-capitalize text-center headline">
			<?php 

            if ( '' === $settings['title'] ) {
                return;
            }
    
            $this->add_render_attribute( 'title', 'class', 'elementor-heading-title' );
    
            if ( ! empty( $settings['size'] ) ) {
                $this->add_render_attribute( 'title', 'class', 'elementor-size-' . $settings['size'] );
            }
    
            $this->add_inline_editing_attributes( 'title' );
    
            $title = $settings['title'];
    
            if ( ! empty( $settings['link']['url'] ) ) {
                $this->add_link_attributes( 'url', $settings['link'] );
    
                $title = sprintf( '<a %1$s>%2$s</a>', $this->get_render_attribute_string( 'url' ), $title );
            }
    
            $title_html = sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['header_size'] ), $this->get_render_attribute_string( 'title' ), $title );
    
            // PHPCS - the variable $title_html holds safe data.
            echo $title_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            ?>
            <?php if(!empty($settings['breadcrumb_shape']['url'])):?>
			<span class="bz-bread-side-shape position-absolute"><img src="<?php echo esc_url($settings['breadcrumb_shape']['url']);?>" alt=""></span>
            <?php endif;?>
		</div>
	</section>
    <?php 
    }
		
	
}


Plugin::instance()->widgets_manager->register( new Bizex_Breadcrumb() );