<?php
/**
 * BigBazaar functions and definitions
 *
 * Set up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme you can override certain functions (those wrapped
 * in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before
 * the parent theme's file, so the child theme functions would be used.
 *
 * @link https://codex.wordpress.org/Theme_Development
 * @link https://codex.wordpress.org/Child_Themes
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters,
 * {@link https://codex.wordpress.org/Plugin_API}
 *
 * @package WordPress
 * @subpackage BigBazaar
 * @since BigBazaar 1.0
 */

 
 
/**
 * Define variables
 */

define('BIGBAZAAR_DIR',                   	get_template_directory());    	// template directory
define('BIGBAZAAR_URI',                   	get_template_directory_uri());  // template directory uri
define('BIGBAZAAR_LIB',                   	BIGBAZAAR_DIR . '/framework');  // library directory
define('BIGBAZAAR_INC',                   	BIGBAZAAR_DIR . '/inc');        // library directory
define('BIGBAZAAR_ADMIN',                	BIGBAZAAR_URI . '/admin');      // admin directory
define('BIGBAZAAR_CSS',                   	BIGBAZAAR_URI . '/css');        // css uri
define('BIGBAZAAR_JS',                    	BIGBAZAAR_URI . '/js');         // javascript uri
define('BIGBAZAAR_IMAGES',                	BIGBAZAAR_URI . '/images');     // image url
define('BIGBAZAAR_ADMIN_IMAGES',            BIGBAZAAR_ADMIN . '/assets/images'); // admin image url
define('BIGBAZAAR_API', 					'http://presslayouts.com/demo/api/');

$theme = wp_get_theme();
define('BIGBAZAAR_VERSION',               $theme->get('Version'));                    // get current version
define('BIGBAZAAR_PREFIX', '_pl_');
 
/**
 * Set the content width based on the theme's design and stylesheet.
 *
 * @since BigBazaar 1.0
 */
if ( ! isset( $content_width ) ) {
	$content_width = 1200;
}

/**
 * BigBazaar only works in WordPress 4.1 or later.
 */
if ( version_compare( $GLOBALS['wp_version'], '4.1-alpha', '<' ) ) {
	require get_template_directory() . '/inc/back-compat.php';
}

if ( ! function_exists( 'bigbazaar_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * @since BigBazaar 1.0
 */
function bigbazaar_setup() {

	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on bigbazaar, use a find and replace
	 * to change 'bigbazaar' to the name of your theme in all the template files
	 */
	// wp-content/themes/bigbazaar/languages/it_IT.mo
	load_theme_textdomain( 'bigbazaar', get_template_directory() . '/languages' );
	
	// wp-content/languages/themes/bigbazaar-it_IT.mo
	load_theme_textdomain( 'bigbazaar', trailingslashit( WP_LANG_DIR ) . 'themes/' );

	// wp-content/themes/bigbazaar-child/languages/it_IT.mo
	load_theme_textdomain( 'bigbazaar', get_stylesheet_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );
	add_editor_style( array( 'style.css') );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * See: https://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 825, 510, true );
	
	// Adding additional image sizes.
	//add_image_size( 'blog-large', 870, 510, true );
	//add_image_size( 'blog-medium', 463, 348, true );
	
	// Update image sizes.
	update_option( 'thumbnail_size_w', 150 );
	update_option( 'thumbnail_size_h', 150 );

	update_option( 'medium_size_w', 463 );
	update_option( 'medium_size_h', 348 );

	update_option( 'large_size_w', 870 );
	update_option( 'large_size_h', 510 );

	// This theme uses wp_nav_menu() in two locations.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu',      'bigbazaar' ),
		'vertical_menu' => esc_html__( 'Vertical Menu',      'bigbazaar' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form', 'comment-form', 'comment-list', 'gallery', 'caption'
	) );

	/*
	 * Enable support for Post Formats.
	 *
	 * See: https://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'post-formats', array(
		'aside', 'image', 'video', 'quote', 'link', 'gallery', 'status', 'audio', 'chat'
	) );

	/*
	 * This theme styles the visual editor to resemble the theme style,
	 * specifically font, colors, icons, and column width.
	 */
	add_editor_style( array( 'css/editor-style.css', 'genericons/genericons.css', bigbazaar_fonts_url() ) );
}
endif; // bigbazaar_setup
add_action( 'after_setup_theme', 'bigbazaar_setup' );

/**
 * Register widget area.
 *
 * @since BigBazaar 1.0
 *
 * @link https://codex.wordpress.org/Function_Reference/register_sidebar
 */
function bigbazaar_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Widget Area', 'bigbazaar' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Shop Page Widget Area', 'bigbazaar' ),
		'id'            => 'shop-page',
		'description'   => esc_html__( 'Add widgets here to appear in shop page sidebar.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Single Product Widget Area', 'bigbazaar' ),
		'id'            => 'single-product',
		'description'   => esc_html__( 'Add widgets here to appear in single product page sidebar.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Menu Widget Area 1', 'bigbazaar' ),
		'id'            => 'menu-widget-area-1',
		'description'   => esc_html__( 'Add widgets here to appear in your menu.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Menu Widget Area 2', 'bigbazaar' ),
		'id'            => 'menu-widget-area-2',
		'description'   => esc_html__( 'Add widgets here to appear in your menu.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Menu Widget Area 3', 'bigbazaar' ),
		'id'            => 'menu-widget-area-3',
		'description'   => esc_html__( 'Add widgets here to appear in your menu.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	
	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget Area 1', 'bigbazaar' ),
		'id'            => 'footer-widget-area-1',
		'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget Area 2', 'bigbazaar' ),
		'id'            => 'footer-widget-area-2',
		'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget Area 3', 'bigbazaar' ),
		'id'            => 'footer-widget-area-3',
		'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget Area 4', 'bigbazaar' ),
		'id'            => 'footer-widget-area-4',
		'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'bigbazaar' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'bigbazaar_widgets_init' );

if ( ! function_exists( 'bigbazaar_fonts_url' ) ) :
/**
 * Register Google fonts for BigBazaar.
 *
 * @since BigBazaar 1.0
 *
 * @return string Google fonts URL for the theme.
 */
function bigbazaar_fonts_url() {
	
	$font_families = array();
		$font_families[] = 'Open Sans:300italic,400italic,600italic,700italic,800italic,400,300,600,700,800';
		$query_args = array(
			'family' => urlencode( implode( '|', $font_families ) )
		);
		$fonts_url = add_query_arg( $query_args, '//fonts.googleapis.com/css' );
	return $fonts_url;
}
endif;

/**
 * JavaScript Detection.
 *
 * Adds a `js` class to the root `<html>` element when JavaScript is detected.
 *
 * @since BigBazaar 1.1
 */
function bigbazaar_javascript_detection() {
	echo "<script>(function(html){html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);</script>\n";
}
add_action( 'wp_head', 'bigbazaar_javascript_detection', 0 );

/**
 * Add Post navigation.
 *
 * @since BigBazaar 1.0
 *
 */
if ( ! function_exists( 'bigbazaar_pagination_nav' ) ) :
/**
 * Display navigation to next/previous set of posts when applicable.
 * @since BigBazaar 1.0
 * @return void
 */
function bigbazaar_pagination_nav() {
	global $wp_query;

	// Don't print empty markup if there's only one page.
	if ( $wp_query->max_num_pages < 2 )
		return;
	
    $big = 999999999; // need an unlikely integer
    $pages = paginate_links( array(
            'base' 			=> str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
            'format' 		=> '?paged=%#%',
            'current' 		=> max( 1, get_query_var('paged') ),
            'total' 		=> $wp_query->max_num_pages,
            'prev_next'	 	=> true,
			'prev_text' 	=> '&laquo;',
			'next_text' 	=> '&raquo;',
            'type'  		=> 'array',
			'end_size'     	=> 2,
			'mid_size'     	=> 2
        ) );
        if( is_array( $pages ) ) {
            $paged = ( get_query_var('paged') == 0 ) ? 1 : get_query_var('paged');
            echo '<nav class="posts-navigation"><ul class="pagination">';
           // echo '<li><span>Showing '. $paged . ' of ' . $wp_query->max_num_pages .' pages</span></li>';
            foreach ( $pages as $page ) {
                echo "<li>$page</li>";
            }
           echo '</ul></nav>';
        }
		?>
	<?php
}
endif;

/**
 * Add featured image as background image to post navigation elements.
 *
 * @since BigBazaar 1.0
 *
 * @see wp_add_inline_style()
 */
function bigbazaar_post_nav_background() {
	if ( ! is_single() ) {
		return;
	}

	$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '', true );
	$next     = get_adjacent_post( false, '', false );
	$css      = '';

	if ( is_attachment() && 'attachment' == $previous->post_type ) {
		return;
	}

	if ( $previous &&  has_post_thumbnail( $previous->ID ) ) {
		$prevthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $previous->ID ), 'post-thumbnail' );
		$css .= '
			.post-navigation .nav-previous { background-image: url(' . esc_url( $prevthumb[0] ) . '); }
			.post-navigation .nav-previous .post-title, .post-navigation .nav-previous a:hover .post-title, .post-navigation .nav-previous .meta-nav { color: #fff; }
			.post-navigation .nav-previous a:before { background-color: rgba(0, 0, 0, 0.4); }
		';
	}

	if ( $next && has_post_thumbnail( $next->ID ) ) {
		$nextthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $next->ID ), 'post-thumbnail' );
		$css .= '
			.post-navigation .nav-next { background-image: url(' . esc_url( $nextthumb[0] ) . '); border-top: 0; }
			.post-navigation .nav-next .post-title, .post-navigation .nav-next a:hover .post-title, .post-navigation .nav-next .meta-nav { color: #fff; }
			.post-navigation .nav-next a:before { background-color: rgba(0, 0, 0, 0.4); }
		';
	}

	wp_add_inline_style( 'bigbazaar-style', $css );
}
add_action( 'wp_enqueue_scripts', 'bigbazaar_post_nav_background' );

/**
 * Display descriptions in main navigation.
 *
 * @since BigBazaar 1.0
 *
 * @param string  $item_output The menu item output.
 * @param WP_Post $item        Menu item object.
 * @param int     $depth       Depth of the menu.
 * @param array   $args        wp_nav_menu() arguments.
 * @return string Menu item with possible description.
 */
function bigbazaar_nav_description( $item_output, $item, $depth, $args ) {
	if ( 'primary' == $args->theme_location && $item->description ) {
		$item_output = str_replace( $args->link_after . '</a>', '<div class="menu-item-description">' . $item->description . '</div>' . $args->link_after . '</a>', $item_output );
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'bigbazaar_nav_description', 10, 4 );

/**
 * Add a `screen-reader-text` class to the search form's submit button.
 *
 * @since BigBazaar 1.0
 *
 * @param string $html Search form HTML.
 * @return string Modified search form HTML.
 */
function bigbazaar_search_form_modify( $html ) {
	return str_replace( 'class="search-submit"', 'class="search-submit screen-reader-text"', $html );
}
add_filter( 'get_search_form', 'bigbazaar_search_form_modify' );

/**
 * Custom template tags for this theme.
 *
 * @since BigBazaar 1.0
 */
require BIGBAZAAR_DIR . '/inc/template-tags.php';

/**
 * Include Customizer Function.
 * @since BigBazaar 1.0
 */
include ( BIGBAZAAR_INC . '/core-function.php' );

/**
 * Implement the Customize theme style.
 *
 * @since BigBazaar 1.0
 */
require BIGBAZAAR_DIR . '/inc/customizer-style.php';