<?php
/**
 * Baza functions and definitions.
 */

/**
 * Table of Contents:
 * Theme Support
 * Required Files
 * Register Styles
 * Register Scripts
 * Register Menus
 * Register Sidebars
 * WP Body Open
 */


/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function baza_theme_support()
{
    /*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
    add_theme_support('title-tag');

    // Register logo.
    $defaults = array(
        'flex-height' => true,
        'flex-width' => true,
        'header-text' => false,
        'unlink-homepage-logo' => false,
    );
    add_theme_support('custom-logo', $defaults);

    /*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
    add_theme_support(
        'html5',
        array(
            'comment-form',
            'comment-list',
        )
    );

    // Woocommerce support
    add_theme_support('woocommerce');
    add_theme_support('wc-product-gallery-zoom');
    add_theme_support('wc-product-gallery-lightbox');
    add_theme_support('wc-product-gallery-slider');

    // RSS
    add_theme_support('automatic-feed-links');

    // Post thumbnails
    add_theme_support('post-thumbnails');
}

add_action('after_setup_theme', 'baza_theme_support');


/**
 * Add SVG support for the logo.
 * @param $html
 * @return string|string[]
 */

function baza_change_logo_class($html)
{

    $html = str_replace('class="custom-logo"', 'class="custom-logo style-svg"', $html);

    return $html;
}

add_filter('get_custom_logo', 'baza_change_logo_class');


/**
 * Required Files. Include required files.
 */
require get_template_directory() . '/inc/template-tags.php';

// Handle Customizer settings.
require get_template_directory() . '/classes/class-theme-customize.php';
require get_template_directory() . '/inc/get-customizer-css.php';
require get_template_directory() . '/inc/get-viewport-script.php';

// Handle SVG icons.
require get_template_directory() . '/classes/class-theme-svg-icons.php';
require get_template_directory() . '/inc/svg-icons.php';

// Custom comment walker.
require get_template_directory() . '/classes/class-theme-walker-comment.php';

// Cart icon.
require get_template_directory() . '/inc/cart-icon.php';

// Kses allowed HTML tags.
require get_template_directory() . '/inc/kses-allowed-html.php';


/**
 * Register and Enqueue Styles.
 */
function baza_register_styles()
{
    $theme_version = wp_get_theme()->get('Version');
    wp_enqueue_style('inter', get_template_directory_uri() . '/assets/fonts/inter.css', array(), $theme_version);
    wp_enqueue_style('bootstrap', get_template_directory_uri() . '/assets/css/bootstrap.css', array(), $theme_version);
    wp_enqueue_style('baza', get_stylesheet_uri(), array(), $theme_version);
    wp_add_inline_style('baza', baza_get_customizer_css());
}

add_action('wp_enqueue_scripts', 'baza_register_styles');


/**
 * Register and Enqueue Scripts.
 */
function baza_register_scripts()
{
    $theme_version = wp_get_theme()->get('Version');
    wp_enqueue_script('bootstrap', get_template_directory_uri() . '/assets/js/bootstrap.bundle.min.js', array('jquery'), $theme_version);
    wp_enqueue_script('baza', get_template_directory_uri() . '/assets/js/index.js', array('jquery'), $theme_version);
    if (get_theme_mod('sticky_header_switcher', false)) {
        wp_enqueue_script('sticky-header', get_template_directory_uri() . '/assets/js/sticky-header.js', array('jquery'), $theme_version);
    }

    wp_script_add_data('baza', 'async', true);
    wp_add_inline_script('baza', baza_get_viewport_script(), 'before'); // Uses inline method to escape layout shift
}

add_action('wp_enqueue_scripts', 'baza_register_scripts');


/**
 * Register navigation menus.
 */
function baza_menus()
{
    $locations = array(
        'primary' => esc_html__('Header Menu', 'baza'),
        'footer' => esc_html__('Footer Menu', 'baza'),
    );
    register_nav_menus($locations);
}

add_action('init', 'baza_menus');


/**
 * Register widget areas.
 */
function baza_sidebar_registration()
{

    // Arguments used in all register_sidebar() calls.
    $shared_args = array(
        'before_title' => '<h2 class="widget-title">',
        'after_title' => '</h2>',
        'before_widget' => '<div class="widget %2$s"><div class="widget-content">',
        'after_widget' => '</div></div>',
    );

    // Footer #1.
    register_sidebar(
        array_merge(
            $shared_args,
            array(
                'name' => esc_html__('Footer #1', 'baza'),
                'id' => 'sidebar-1',
                'description' => esc_html__('Widgets in this area will be displayed in the first column in the footer.', 'baza'),
            )
        )
    );

    // Footer #2.
    register_sidebar(
        array_merge(
            $shared_args,
            array(
                'name' => esc_html__('Footer #2', 'baza'),
                'id' => 'sidebar-2',
                'description' => esc_html__('Widgets in this area will be displayed in the second column in the footer.', 'baza'),
            )
        )
    );

    // Footer #3.
    register_sidebar(
        array_merge(
            $shared_args,
            array(
                'name' => esc_html__('Footer #3', 'baza'),
                'id' => 'sidebar-3',
                'description' => esc_html__('Widgets in this area will be displayed in the third column in the footer.', 'baza'),
            )
        )
    );

    // Blog sidebar
    register_sidebar(
        array_merge(
            $shared_args,
            array(
                'name' => esc_html__('Blog sidebar', 'baza'),
                'id' => 'blog-sidebar',
                'description' => esc_html__('Widgets in this area will be displayed in the second column on blog pages.', 'baza'),
            )
        )
    );

    // Single product sidebar
    register_sidebar(
        array_merge(
            $shared_args,
            array(
                'name' => esc_html__('Single product sidebar', 'baza'),
                'id' => 'sidebar-single-product',
                'description' => esc_html__('Widgets in this area will be displayed in the second column on single product pages.', 'baza'),
            )
        )
    );

    // Shop sidebar
    register_sidebar(
        array_merge(
            $shared_args,
            array(
                'name' => esc_html__('Shop sidebar', 'baza'),
                'id' => 'sidebar-shop',
                'description' => esc_html__('Widgets in this area will be displayed in the second column on the shop page.', 'baza'),
            )
        )
    );

}

add_action('widgets_init', 'baza_sidebar_registration');


/**
 * Shim for wp_body_open, ensuring backward compatibility with versions of WordPress older than 5.2.
 */
if (!function_exists('wp_body_open')) {
    function wp_body_open()
    {
        do_action('wp_body_open');
    }
}


/**
 * Allow to add class to menu's link.
 */
function baza_add_menu_link_class($atts, $item, $args)
{
    if (property_exists($args, 'link_class')) {
        $atts['class'] = $args->link_class;
    }
    return $atts;
}

add_filter('nav_menu_link_attributes', 'baza_add_menu_link_class', 1, 3);


/**
 * Allow to add class to menu's <li> item.
 */
function baza_add_additional_class_on_li($classes, $item, $args)
{
    if (isset($args->add_li_class)) {
        $classes[] = $args->add_li_class;
    }
    return $classes;
}

add_filter('nav_menu_css_class', 'baza_add_additional_class_on_li', 1, 3);


/**
 * Register Custom Navigation Walker.
 */
function baza_register_navwalker()
{
    require_once get_template_directory() . '/classes/class-wp-bootstrap-navwalker.php';
}

add_action('after_setup_theme', 'baza_register_navwalker');


/**
 * Change widget titles to h4.
 * @param $params
 * @return mixed
 */
function baza_filter_widget_title_tag($params)
{
    $params[0]['before_title'] = '<h4 class="widget-title">';
    $params[0]['after_title'] = '</h4>';
    return $params;
}

add_filter('dynamic_sidebar_params', 'baza_filter_widget_title_tag');


/**
 * Hide Add to cart button from non-product pages.
 */
if (get_theme_mod('add_to_cart_switcher')) {
    remove_action('woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart');
}


/**
 * Enqueue comment-reply script.
 */
function baza_enqueue_comments_reply()
{
    if (get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}

add_action('comment_form_before', 'baza_enqueue_comments_reply');


/**
 * Check Woocommerce is active for cart functions.
 * @return bool
 */
function baza_is_product(): bool
{
    if (class_exists('woocommerce')) {
        return is_product();
    } else {
        return false;
    }
}


/**
 * Check Woocommerce is active for cart functions.
 * @return bool
 */
function baza_is_shop(): bool
{
    if (class_exists('woocommerce')) {
        return is_shop();
    } else {
        return false;
    }
}


/**
 * Remove breadcrumbs from pages.
 */
add_filter('woocommerce_before_main_content', 'baza_remove_breadcrumbs');
function baza_remove_breadcrumbs()
{
    if (is_shop() || is_product_category() || is_product_tag() || is_product_taxonomy()) {
        remove_action('woocommerce_before_main_content', 'woocommerce_breadcrumb', 20);
    }
}


/**
 * Use comment's author display name for comments.
 */
add_filter('get_comment_author', 'baza_comment_author_display_name');
function baza_comment_author_display_name($author): string
{
    global $comment;
    if (!empty($comment->user_id)) {
        $user = get_userdata($comment->user_id);
        $author = $user->display_name;
    }
    return $author;
}


/**
 * Set up the content width value based on the theme's design.
 *
 * @see _action_theme_content_width()
 */
if (!isset($content_width)) {
    $content_width = 1199;
}


/**
 * Change number of products that are displayed per page (shop page).
 *
 * @see https://docs.woocommerce.com/document/change-number-of-products-displayed-per-page
 */
add_filter( 'loop_shop_per_page', 'baza_loop_shop_per_page', 20 );
function baza_loop_shop_per_page( $cols ): int
{
    // $cols contains the current number of products per page based on the value stored on Options –> Reading
    // Return the number of products you wanna show per page.
    $cols = 12;
    return $cols;
}


/**
 * Change number of products per row to 2.
 */
add_filter('loop_shop_columns', 'loop_columns');
if (!function_exists('loop_columns')) {
    function loop_columns(): int
    {
        return 3;
    }
}


/**
 * Change number of related products output.
 */
add_filter('woocommerce_output_related_products_args', 'baza_related_products_args', 20);
function baza_related_products_args($args)
{
    $args['posts_per_page'] = 3;
    $args['columns'] = 3;
    return $args;
}


/**
 * Change number of upsell products output.
 */
add_filter('woocommerce_upsell_display_args', 'baza_upsell_products_args', 20);
function baza_upsell_products_args($args)
{
    $args['posts_per_page'] = 3;
    $args['columns'] = 3;
    return $args;
}


/**
 * TGM Plugin Activation.
 */
require_once get_template_directory() . '/classes/class-tgm-plugin-activation.php';

/** @internal */
function baza_action_theme_register_required_plugins()
{

    $config = array(
        'id' => 'baza',
        'menu' => 'baza-install-plugins',
        'parent_slug' => 'themes.php',
        'capability' => 'edit_theme_options',
        'has_notices' => true,
        'dismissable' => true,
        'is_automatic' => true,
    );

    tgmpa(array(
        array(
            'name' => esc_html__('One Click Demo Import', 'baza'),
            'slug' => 'one-click-demo-import',
            'required' => false,
        ),
        array(
            'name' => esc_html__('Envato Market', 'baza'),
            'slug' => 'envato-market',
            'source' => get_template_directory() . '/inc/plugins/envato-market.zip',
            'required' => false,
        ),
        array(
            'name' => esc_html__('Elementor', 'baza'),
            'slug' => 'elementor',
            'required' => true,
        ),
    ), $config);

}
add_action('tgmpa_register', 'baza_action_theme_register_required_plugins');


/**
 * List of demo templates.
 */
function ocdi_import_files() {
    return [
        [
            'import_file_name'           => 'SMM Instagram',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/smm-instagram/N6Prucax6MSu3pFB/smm-instagram.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/smm-instagram/N6Prucax6MSu3pFB/smm-instagram.png',
            'preview_url'                => 'https://demo.bazatheme.com/smm-instagram',
        ],
        [
            'import_file_name'           => 'SEO Agency',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/seo-agency/2kPPuA4eSvy47nug/seo-agency.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/seo-agency/2kPPuA4eSvy47nug/seo-agency.png',
            'preview_url'                => 'https://demo.bazatheme.com/seo-agency',
        ],
        [
            'import_file_name'           => 'Language School',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/language-school/2J4tW5LBkUtcJwKG/language-school.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/language-school/2J4tW5LBkUtcJwKG/language-school.png',
            'preview_url'                => 'https://demo.bazatheme.com/language-school',
        ],
        [
            'import_file_name'           => 'Law 2',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/law-2/3hRdh6u86sFa9F9n/law-2.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/law-2/3hRdh6u86sFa9F9n/law-2.png',
            'preview_url'                => 'https://demo.bazatheme.com/law-2',
        ],
        [
            'import_file_name'           => 'Law',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/law/Va9XHFGch5jMwgdm/law.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/law/Va9XHFGch5jMwgdm/law.png',
            'preview_url'                => 'https://demo.bazatheme.com/law',
        ],
        [
            'import_file_name'           => 'Business Consulting 3',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-3/7FNs4kP8wA3TZ5D6/business-consulting-3.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-3/7FNs4kP8wA3TZ5D6/business-consulting-3.png',
            'preview_url'                => 'https://demo.bazatheme.com/business-consulting-3',
        ],
        [
            'import_file_name'           => 'Broadband',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/broadband/vAuMEztscVQ6AF3D/broadband.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/broadband/vAuMEztscVQ6AF3D/broadband.png',
            'preview_url'                => 'https://demo.bazatheme.com/broadband',
        ],
        [
            'import_file_name'           => 'Business Consulting 1',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-1/XqCGf4peTx85Gcrb/business-consulting-1.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-1/XqCGf4peTx85Gcrb/business-consulting-1.png',
            'preview_url'                => 'https://demo.bazatheme.com/business-consulting-1',
        ],
        [
            'import_file_name'           => 'Business Consulting 2',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-2/pHGu4CxNswLYyMDD/business-consulting-2.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-2/pHGu4CxNswLYyMDD/business-consulting-2.png',
            'preview_url'                => 'https://demo.bazatheme.com/business-consulting-2',
        ],
        [
            'import_file_name'           => 'Content Writing Agency',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/content-writing-agency/mG4t57QmwLFCvh4z/content-writing-agency.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/content-writing-agency/mG4t57QmwLFCvh4z/content-writing-agency.png',
            'preview_url'                => 'https://demo.bazatheme.com/content-writing-agency',
        ],
        [
            'import_file_name'           => 'Digital Marketing',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/digital-marketing/Rq9MgQBkyWMp49QU/digital-marketing.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/digital-marketing/Rq9MgQBkyWMp49QU/digital-marketing.png',
            'preview_url'                => 'https://demo.bazatheme.com/digital-marketing',
        ],
        [
            'import_file_name'           => 'Event 2',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/event-2/ZC9heAdBWKC5TUEw/event-2.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/event-2/ZC9heAdBWKC5TUEw/event-2.png',
            'preview_url'                => 'https://demo.bazatheme.com/event-2',
        ],
        [
            'import_file_name'           => 'Event 3',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/event-3/JgmDu2GfMcV3uwd8/event-3.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/event-3/JgmDu2GfMcV3uwd8/event-3.png',
            'preview_url'                => 'https://demo.bazatheme.com/event-3',
        ],
        [
            'import_file_name'           => 'Financial Consultant',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/financial-consultant/kYxaCnBsJ6PsNXKJ/financial-consultant.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/financial-consultant/kYxaCnBsJ6PsNXKJ/financial-consultant.png',
            'preview_url'                => 'https://demo.bazatheme.com/financial-consultant',
        ],
        [
            'import_file_name'           => 'IT HR services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/it-hr-services/ZqUkzf4QM9GQkJjz/it-hr-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/it-hr-services/ZqUkzf4QM9GQkJjz/it-hr-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/it-hr-services',
        ],
        [
            'import_file_name'           => 'Creative Agency',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/creative-agency/HDbN2pxpYFUhDgN5/creative-agency.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/creative-agency/HDbN2pxpYFUhDgN5/creative-agency.png',
            'preview_url'                => 'https://demo.bazatheme.com/creative-agency',
        ],
        [
            'import_file_name'           => 'NFT Single Item Landing',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/nft-single-item-landing/MN5GsP2Kb4RqcJyz/nft-single-item-landing.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/nft-single-item-landing/MN5GsP2Kb4RqcJyz/nft-single-item-landing.png',
            'preview_url'                => 'https://demo.bazatheme.com/nft-single-item-landing',
        ],
        [
            'import_file_name'           => 'Business Consultant',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consultant/8DKDyKqyDcA2xNxE/business-consultant.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consultant/8DKDyKqyDcA2xNxE/business-consultant.png',
            'preview_url'                => 'https://demo.bazatheme.com/business-consultant',
        ],
        [
            'import_file_name'           => 'Graphic Design Services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/graphic-design-services/6wEPTVTTCJBztKCd/graphic-design-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/graphic-design-services/6wEPTVTTCJBztKCd/graphic-design-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/graphic-design-services',
        ],
        [
            'import_file_name'           => 'Event',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/event/nqTtntuWuZz45gFB/event.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/event/nqTtntuWuZz45gFB/event.png',
            'preview_url'                => 'https://demo.bazatheme.com/event',
        ],
        [
            'import_file_name'           => 'Email Marketing Services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/email-marketing-services/aNQ39Wtsf7aJRCVA/email-marketing-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/email-marketing-services/aNQ39Wtsf7aJRCVA/email-marketing-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/email-marketing-services',
        ],
        [
            'import_file_name'           => 'SMM Services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/smm-services/rt9L9bdHjQ3W722K/smm-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/smm-services/rt9L9bdHjQ3W722K/smm-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/smm-services',
        ],
        [
            'import_file_name'           => 'Saas Services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/saas-services/2mSEseAH2QbJsCXD/saas-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/saas-services/2mSEseAH2QbJsCXD/saas-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/saas-services',
        ],
        [
            'import_file_name'           => 'Web Development Agency',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/web-development-agency/W8n62XW76TXnayFq/web-development-agency.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/web-development-agency/W8n62XW76TXnayFq/web-development-agency.png',
            'preview_url'                => 'https://demo.bazatheme.com/web-development-agency',
        ],
        [
            'import_file_name'           => 'Agency',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/agency/T6twj5qQSRfSQGHD/agency.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/agency/T6twj5qQSRfSQGHD/agency.png',
            'preview_url'                => 'https://demo.bazatheme.com/agency',
        ],
        [
            'import_file_name'           => 'SEO Services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/seo-services/apA4da4RV3Neugbg/seo-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/seo-services/apA4da4RV3Neugbg/seo-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/seo-services',
        ],
        [
            'import_file_name'           => 'Software Landing',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/software-landing/Y472eENnhaPHhNWa/software-landing.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/software-landing/Y472eENnhaPHhNWa/software-landing.png',
            'preview_url'                => 'https://demo.bazatheme.com/software-landing',
        ],
        [
            'import_file_name'           => 'Business Consulting Services',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-services/pvm8Bg5jz4DSkYcm/business-consulting-services.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/business-consulting-services/pvm8Bg5jz4DSkYcm/business-consulting-services.png',
            'preview_url'                => 'https://demo.bazatheme.com/business-consulting-services',
        ],
        [
            'import_file_name'           => 'Webinar',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/webinar/N37X5He54hBbxCDY/webinar.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/webinar/N37X5He54hBbxCDY/webinar.png',
            'preview_url'                => 'https://demo.bazatheme.com/webinar',
        ],
        [
            'import_file_name'           => 'Digital Marketing Agency',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/digital-marketing-agency/x92gWwFv7bTnRLeS/digital-marketing-agency.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/digital-marketing-agency/x92gWwFv7bTnRLeS/digital-marketing-agency.png',
            'preview_url'                => 'https://demo.bazatheme.com/digital-marketing-agency',
        ],
        [
            'import_file_name'           => 'IT Consultant',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/it-consultant/Mcj4U3KREFByVsdw/it-consultant.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/it-consultant/Mcj4U3KREFByVsdw/it-consultant.png',
            'preview_url'                => 'https://demo.bazatheme.com/it-consultant',
        ],
        [
            'import_file_name'           => 'Online Course Landing',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/online-course-landing/Hf5g2WXYC9rPs9W4/online-course-landing.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/online-course-landing/Hf5g2WXYC9rPs9W4/online-course-landing.png',
            'preview_url'                => 'https://demo.bazatheme.com/online-course-landing',
        ],
        [
            'import_file_name'           => 'Mobile App Landing',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/mobile-app-landing/4xCWL4KsByLQ5g4B/mobile-app-landing.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/mobile-app-landing/4xCWL4KsByLQ5g4B/mobile-app-landing.png',
            'preview_url'                => 'https://demo.bazatheme.com/mobile-app-landing',
        ],
        [
            'import_file_name'           => 'Ebook Landing',
            'categories'                 => [ 'Main Demos' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/ebook-landing/HrcXrHStu94K48Q9/ebook-landing.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/ebook-landing/HrcXrHStu94K48Q9/ebook-landing.png',
            'preview_url'                => 'https://demo.bazatheme.com/ebook-landing',
        ],
        [
            'import_file_name'           => 'Contact Us 1',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us/Y5n3WA67VNNqGYaP/contact-us.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us/Y5n3WA67VNNqGYaP/contact-us.png',
            'preview_url'                => 'https://demo.bazatheme.com/contact-us',
        ],
        [
            'import_file_name'           => 'Contact Us 2',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us-2/Hq3GYqQ2Tyc8kwZs/contact-us-2.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us-2/Hq3GYqQ2Tyc8kwZs/contact-us-2.png',
            'preview_url'                => 'https://demo.bazatheme.com/contact-us-2',
        ],
        [
            'import_file_name'           => 'Contact Us 3',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us-3/qVTfj2rqRrd23H4b/contact-us-3.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us-3/qVTfj2rqRrd23H4b/contact-us-3.png',
            'preview_url'                => 'https://demo.bazatheme.com/contact-us-3',
        ],
        [
            'import_file_name'           => 'Contact Us 4',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us-4/gkudZ39FTa5rE7rj/contact-us-4.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/contact-us-4/gkudZ39FTa5rE7rj/contact-us-4.png',
            'preview_url'                => 'https://demo.bazatheme.com/contact-us-4',
        ],
        [
            'import_file_name'           => 'Blog One Column',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-one-column/VmqRMgQd4wNKz5H7/blog-one-column.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-one-column/VmqRMgQd4wNKz5H7/blog-one-column.png',
            'preview_url'                => 'https://demo.bazatheme.com/blog-one-column',
        ],
        [
            'import_file_name'           => 'Blog One Column With Sidebar',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-one-column-with-sidebar/XhfJyv9hWEEgZsGw/blog-one-column-with-sidebar.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-one-column-with-sidebar/XhfJyv9hWEEgZsGw/blog-one-column-with-sidebar.png',
            'preview_url'                => 'https://demo.bazatheme.com/blog-one-column-with-sidebar',
        ],
        [
            'import_file_name'           => 'Blog Two Columns',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-two-columns/hWYxBjFSnfG2Ve9H/blog-two-columns.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-two-columns/hWYxBjFSnfG2Ve9H/blog-two-columns.png',
            'preview_url'                => 'https://demo.bazatheme.com/blog-two-columns',
        ],
        [
            'import_file_name'           => 'Blog Two Columns With Sidebar',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-two-columns-with-sidebar/3gR9TgNdtwfWkkKg/blog-two-columns-with-sidebar.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-two-columns-with-sidebar/3gR9TgNdtwfWkkKg/blog-two-columns-with-sidebar.png',
            'preview_url'                => 'https://demo.bazatheme.com/blog-two-columns-with-sidebar',
        ],
        [
            'import_file_name'           => 'Blog Three Columns',
            'categories'                 => [ 'Inner Pages' ],
            'import_file_url'            => 'http://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-three-columns/RYSJmM8z6dSKtqDC/blog-three-columns.xml',
            'import_preview_image_url'   => 'https://files.bazatheme.com/wp-content/uploads/demos/v2/haVf5DD5Gk7LRMVp/blog-three-columns/RYSJmM8z6dSKtqDC/blog-three-columns.png',
            'preview_url'                => 'https://demo.bazatheme.com/blog-three-columns',
        ],
    ];
}
add_filter( 'ocdi/import_files', 'ocdi_import_files' );