<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('GSF_Inc_Helper')) {
	class GSF_Inc_Helper {
		private static $_instance;
		public static function getInstance()
		{
			if (self::$_instance == NULL) {
				self::$_instance = new self();
			}

			return self::$_instance;
		}

		/**
		 * Create field object from field type
		 *
		 * @param $type
		 * @return GSF_Field
		 */
		public function createField($type) {
			$class_name = str_replace('_', ' ', $type);
			$class_name = ucwords($class_name);
			$class_name = str_replace(' ', '_', $class_name);
			$class_name = 'GSF_Field_' . $class_name;
			if (class_exists($class_name)) {
				return new $class_name();
			}
			return null;
		}

		public function setFieldPrefix($prefix) {
			$GLOBALS['gsf_field_prefix'] = $prefix;
		}

		public function getFieldPrefix() {
			return isset($GLOBALS['gsf_field_prefix']) ? $GLOBALS['gsf_field_prefix'] : '';
		}

		/**
		 * Set field layout
		 * @param $layout
		 */
		public function setFieldLayout($layout) {
			if (!in_array($layout, array('inline', 'full'))) {
				$layout = 'inline';
			}
			$GLOBALS['gsf_field_layout'] = $layout;
		}

		/**
		 * Get field layout
		 * @return string
		 */
		public function getFieldLayout() {
			if (!isset($GLOBALS['gsf_field_layout'])) {
				$GLOBALS['gsf_field_layout'] = 'inline';
			}
			return $GLOBALS['gsf_field_layout'];
		}

		/**
		 * Get template
		 * @param $slug
		 * @param $args
		 */
		public function getTemplate($slug, $args = array()) {
			if ($args && is_array($args)) {
				extract($args);
			}
			$located = GSF()->pluginDir($slug . '.php');
			if (!file_exists($located)) {
				_doing_it_wrong(__FUNCTION__, sprintf('<code>%s</code> does not exist.', $slug), '1.0');

				return;
			}
			include($located);
		}

		/**
		 * Get plugin assets url
		 * @param $file
		 * @return string
		 */
		public function getAssetUrl($file) {
			if (!file_exists(GSF()->pluginDir($file)) || (defined('SCRIPT_DEBUG') && SCRIPT_DEBUG)) {
				$ext = explode('.', $file);
				$ext = end($ext);
				$normal_file = preg_replace('/((\.min\.css)|(\.min\.js))$/', '', $file);
				if ($normal_file != $file) {
					$normal_file = untrailingslashit($normal_file) . ".{$ext}";
					if (file_exists(GSF()->pluginDir($normal_file))) {
						return GSF()->pluginUrl(untrailingslashit($normal_file));
					}
				}
			}
			return GSF()->pluginUrl(untrailingslashit($file));
		}

		public function renderFields(&$config, &$values) {
			$list_section = array();
			if (isset($config['section'])) {
				foreach ($config['section'] as &$section) {
					$list_section[] = array(
						'id'    => $section['id'],
						'title' => $section['title'],
						'icon'  => isset($section['icon']) ? $section['icon'] : 'dashicons-admin-generic',
					);
				}
			}
			$this->getTemplate('admin/templates/meta-start', array(
				'list_section' => $list_section
			));

			if (!empty($config)) {
				if (isset($config['section'])) {
					?>
					<?php foreach ($config['section'] as &$section): ?>
						<div id="section_<?php echo esc_attr($section['id'])?>" class="gsf-section-container">
							<h4 class="gsf-section-title">
								<i class="gsf-section-title-icon <?php echo isset($section['icon']) ? esc_attr($section['icon']) : 'dashicons dashicons-admin-generic' ; ?>"></i>
								<span><?php echo esc_html($section['title']); ?></span>
								<span class="gsf-section-title-toggle"></span>
							</h4>
							<div class="gsf-section-inner">
								<?php if (isset($section['fields'])): ?>
									<?php $this->renderSubFields($section['fields'], $values) ?>
								<?php endif;?>
							</div>
						</div><!-- /.gsf-section-container  -->
					<?php endforeach;?>
				<?php
				} else {
					$this->renderSubFields($config['fields'], $values);
				}
			}

			$this->getTemplate('admin/templates/meta-end');
		}

		public function renderSubFields(&$fields, &$values) {
			foreach ($fields as &$config) {
				$type = isset($config['type']) ? $config['type'] : '';
				if (empty($type)) {
					continue;
				}
				$id = isset($config['id']) ? $config['id'] : '';
				$field = $this->createField($config['type']);
				$field->_setting = &$config;
				if (in_array($type, array('group', 'row'))) {
					$field->_value = $values;
				}
				else {
					if (!empty($id)) {
						$field->_value = isset($values[$id]) ? $values[$id] : null;
					}
					else {
						$field->_value = null;
					}
				}

				$field->render();
			}
		}

		/**
		 * Get Config Keys
		 *
		 * @param $configs
		 * @return array
		 */
		public function getConfigKeys(&$configs) {
			$field_keys = array();
			if (isset($configs['section'])) {
				foreach ($configs['section'] as $section) {
					if (isset($section['fields'])) {
						$field_keys = array_merge($field_keys, $this->getConfigFieldKeys($section['fields']));
					}
				}
			} else {
				if (isset($configs['fields'])) {
					$field_keys = array_merge($field_keys, $this->getConfigFieldKeys($configs['fields']));
				}
			}
			return $field_keys;
		}

		private function getConfigFieldKeys(&$fields) {
			$field_keys = array();
			foreach ($fields as $field) {
				if (!isset($field['type'])) {
					continue;
				}
				$field_type = $field['type'];

				switch ($field_type) {
					case 'row':
					case 'group':
						$field_keys = array_merge($field_keys, $this->getConfigFieldKeys($field['fields']));
						break;
					default:
						if (!isset($field['id'])) {
							break;
						}
						$field_obj = $this->createField($field_type);
						$field_obj->_setting = $field;
						$field_keys[$field['id']] = array(
							'type' => $field_type,
							'empty_value' => $field_obj->getEmptyValue()
						);
						break;
				}
			}
			return $field_keys;
		}

		public function getConfigDefault(&$configs, $current_section = '') {
			$field_default = array();
			if (!empty($current_section)) {
				if (isset($configs['section'])) {
					foreach ($configs['section'] as $section) {
						if ('section_' . $section['id'] == $current_section) {
							if (isset($section['fields'])) {
								$field_default = array_merge($field_default, $this->getConfigDefaultField($section['fields']));
							}
						}
					}
				}
			}
			else {
				if (isset($configs['section'])) {
					foreach ($configs['section'] as $section) {
						if (isset($section['fields'])) {
							$field_default = array_merge($field_default, $this->getConfigDefaultField($section['fields']));
						}
					}
				} else {
					if (isset($configs['fields'])) {
						$field_default = array_merge($field_default, $this->getConfigDefaultField($configs['fields']));
					}
				}
			}
			return $field_default;
		}

		private function getConfigDefaultField(&$fields) {
			$field_default = array();
			foreach ($fields as $field) {
				if (!isset($field['type'])) {
					continue;
				}
				$field_type = $field['type'];

				switch ($field_type) {
					case 'row':
					case 'group':
						$field_default = array_merge($field_default, $this->getConfigDefaultField($field['fields']));
						break;
					default:
						if (!isset($field['id'])) {
							break;
						}
						$field_obj = $this->createField($field_type);
						$field_obj->_setting = $field;
						$field_default[$field['id']] = $field_obj->getFieldDefault();
						break;
				}
			}
			return $field_default;
		}

		/**
		 * Get list sidebars
		 *
		 * @return array
		 */
		public function getSidebars()
		{
			$sidebars = array();
			if (is_array($GLOBALS['wp_registered_sidebars'])) {
				foreach ($GLOBALS['wp_registered_sidebars'] as $sidebar) {
					$sidebars[$sidebar['id']] = ucwords($sidebar['name']);
				}
			}
			return $sidebars;
		}

		/**
		 * Get listing menu
		 *
		 * @return array
		 */
		public function getMenus()
		{
			$user_menus = get_categories(array(
				'taxonomy'   => 'nav_menu',
				'hide_empty' => false,
				'orderby'    => 'name',
				'order'      => 'ASC'
			));
			$menus = array();
			foreach ($user_menus as $menu) {
				$menus[$menu->term_id] = $menu->name;
			}

			return $menus;
		}

		/**
		 * Get listing taxonomies
		 *
		 * @param array $params
		 * @return array
		 */
		public function getTaxonomies($params = array())
		{
			$args = array(
				'orderby' => 'name',
				'order'   => 'ASC',
				'hide_empty' => false,
			);
			if (!empty($params)) {
				$args = array_merge($args, $params);
			}

			$categories = get_categories($args);
			$taxs = array();
			foreach ($categories as $cate) {
				$taxs[$cate->term_id] = $cate->name;
			}

			return $taxs;
		}

		/**
		 * Get listing post
		 *
		 * @param array $params
		 * @return array
		 */
		public function getPosts($params = array())
		{
			$args = array(
				'numberposts' => 20,
				'orderby' => 'post_title',
				'order'   => 'ASC',
			);
			if (!empty($params)) {
				$args = array_merge($args, $params);
			}
			$posts = get_posts($args);
			$ret_posts = array();
			foreach ($posts as $post) {
				$ret_posts[$post->ID] = $post->post_title;
			}

			return $ret_posts;
		}

		/**
		 * Render selected attribute
		 *
		 * @param $value
		 * @param $current
		 */
		public function theSelected($value, $current) {
			echo ((is_array($current) && in_array($value, $current)) || (!is_array($current) && ($value == $current))) ? 'selected="selected"' : '';
		}

		/**
		 * Render checked attribute
		 *
		 * @param $value
		 * @param $current
		 */
		public function theChecked($value, $current) {
			echo ((is_array($current) && in_array($value, $current)) || (!is_array($current) && ($value == $current))) ? 'checked="checked"' : '';
		}

		/**
		 * Get attachment id by url
		 *
		 * @param $url
		 * @return int
		 */
		public function getAttachmentIdByUrl($url) {
			global $wpdb;
			$attachment = $wpdb->get_col($wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE guid='%s';", $url));
			if (!empty($attachment)) {
				return $attachment[0];
			}

			return 0;
		}

		/**
		 * Get framework nonce verify key
		 * @return mixed|void
		 */
		public function getNonceVerifyKey() {
			return apply_filters('gsf_nonce_verify_key', 'GSF_SMART_FRAMEWORK_VERIFY');
		}

		/**
		 * Get framework nonce value
		 * @return string
		 */
		public function getNonceValue() {
			return wp_create_nonce($this->getNonceVerifyKey());
		}

        public function getFontIcons() {
            return apply_filters('gsf_font_icon_config', array(
                'fontawesome' => array(
                    'label' => esc_html__('Font Awesome 5Pro','auteur-framework'),
                    'iconGroup' => json_decode('
[{"id":"logistics","title":"Logistics","icons":["fal fa-barcode-alt","fal fa-barcode-read","fal fa-barcode-scan","fal fa-box","fal fa-box-check","fal fa-boxes","fal fa-clipboard-check","fal fa-clipboard-list","fal fa-conveyor-belt","fal fa-conveyor-belt-alt","fal fa-dolly","fal fa-dolly-empty","fal fa-dolly-flatbed","fal fa-dolly-flatbed-alt","fal fa-dolly-flatbed-empty","fal fa-forklift","fal fa-hand-holding-box","fal fa-hand-receiving","fal fa-inventory","fal fa-pallet","fal fa-pallet-alt","fal fa-scanner","fal fa-scanner-keyboard","fal fa-scanner-touchscreen","fal fa-shipping-fast","fal fa-shipping-timed","fal fa-tablet-rugged","fal fa-truck","fal fa-warehouse","fal fa-warehouse-alt","fas fa-barcode-alt","fas fa-barcode-read","fas fa-barcode-scan","fas fa-box","fas fa-box-check","fas fa-boxes","fas fa-clipboard-check","fas fa-clipboard-list","fas fa-conveyor-belt","fas fa-conveyor-belt-alt","fas fa-dolly","fas fa-dolly-empty","fas fa-dolly-flatbed","fas fa-dolly-flatbed-alt","fas fa-dolly-flatbed-empty","fas fa-forklift","fas fa-hand-holding-box","fas fa-hand-receiving","fas fa-inventory","fas fa-pallet","fas fa-pallet-alt","fas fa-scanner","fas fa-scanner-keyboard","fas fa-scanner-touchscreen","fas fa-shipping-fast","fas fa-shipping-timed","fas fa-tablet-rugged","fas fa-truck","fas fa-warehouse","fas fa-warehouse-alt"]},{"id":"files","title":"Files","icons":["fal fa-archive","fal fa-clone","fal fa-copy","fal fa-cut","fal fa-file","fal fa-file-alt","fal fa-file-archive","fal fa-file-audio","fal fa-file-check","fal fa-file-code","fal fa-file-edit","fal fa-file-excel","fal fa-file-exclamation","fal fa-file-image","fal fa-file-minus","fal fa-file-pdf","fal fa-file-plus","fal fa-file-powerpoint","fal fa-file-times","fal fa-file-video","fal fa-file-word","fal fa-folder","fal fa-folder-open","fal fa-paste","fal fa-save","fal fa-sticky-note","fas fa-archive","fas fa-clone","fas fa-copy","fas fa-cut","fas fa-file","fas fa-file-alt","fas fa-file-archive","fas fa-file-audio","fas fa-file-check","fas fa-file-code","fas fa-file-edit","fas fa-file-excel","fas fa-file-exclamation","fas fa-file-image","fas fa-file-minus","fas fa-file-pdf","fas fa-file-plus","fas fa-file-powerpoint","fas fa-file-times","fas fa-file-video","fas fa-file-word","fas fa-folder","fas fa-folder-open","fas fa-paste","fas fa-save","fas fa-sticky-note"]},{"id":"code","title":"Code","icons":["fal fa-archive","fal fa-barcode","fal fa-bath","fal fa-browser","fal fa-bug","fal fa-code","fal fa-code-branch","fal fa-code-commit","fal fa-code-merge","fal fa-coffee","fal fa-file","fal fa-file-alt","fal fa-file-code","fal fa-filter","fal fa-fire-extinguisher","fal fa-folder","fal fa-folder-open","fal fa-keyboard","fal fa-microchip","fal fa-project-diagram","fal fa-qrcode","fal fa-shield","fal fa-shield-alt","fal fa-shield-check","fal fa-sitemap","fal fa-stream","fal fa-terminal","fal fa-user-secret","fal fa-window","fal fa-window-alt","fal fa-window-close","fal fa-window-maximize","fal fa-window-minimize","fal fa-window-restore","fas fa-archive","fas fa-barcode","fas fa-bath","fas fa-browser","fas fa-bug","fas fa-code","fas fa-code-branch","fas fa-code-commit","fas fa-code-merge","fas fa-coffee","fas fa-file","fas fa-file-alt","fas fa-file-code","fas fa-filter","fas fa-fire-extinguisher","fas fa-folder","fas fa-folder-open","fas fa-keyboard","fas fa-microchip","fas fa-project-diagram","fas fa-qrcode","fas fa-shield","fas fa-shield-alt","fas fa-shield-check","fas fa-sitemap","fas fa-stream","fas fa-terminal","fas fa-user-secret","fas fa-window","fas fa-window-alt","fas fa-window-close","fas fa-window-maximize","fas fa-window-minimize","fas fa-window-restore"]},{"id":"communication","title":"Communication","icons":["fal fa-address-book","fal fa-address-card","fal fa-american-sign-language-interpreting","fal fa-assistive-listening-systems","fal fa-at","fal fa-bell","fal fa-bell-slash","fab fa-bluetooth","fab fa-bluetooth-b","fal fa-broadcast-tower","fal fa-bullhorn","fal fa-chalkboard","fal fa-comment","fal fa-comment-alt","fal fa-comments","fal fa-envelope","fal fa-envelope-open","fal fa-envelope-square","fal fa-fax","fal fa-inbox","fal fa-inbox-in","fal fa-inbox-out","fal fa-language","fal fa-microphone","fal fa-microphone-alt","fal fa-microphone-alt-slash","fal fa-microphone-slash","fal fa-mobile","fal fa-mobile-alt","fal fa-mobile-android","fal fa-mobile-android-alt","fal fa-paper-plane","fal fa-phone","fal fa-phone-slash","fal fa-phone-square","fal fa-phone-volume","fal fa-rss","fal fa-rss-square","fal fa-tty","fal fa-wifi","fas fa-address-book","fas fa-address-card","fas fa-american-sign-language-interpreting","fas fa-assistive-listening-systems","fas fa-at","fas fa-bell","fas fa-bell-slash","fas fa-broadcast-tower","fas fa-bullhorn","fas fa-chalkboard","fas fa-comment","fas fa-comment-alt","fas fa-comments","fas fa-envelope","fas fa-envelope-open","fas fa-envelope-square","fas fa-fax","fas fa-inbox","fas fa-inbox-in","fas fa-inbox-out","fas fa-language","fas fa-microphone","fas fa-microphone-alt","fas fa-microphone-alt-slash","fas fa-microphone-slash","fas fa-mobile","fas fa-mobile-alt","fas fa-mobile-android","fas fa-mobile-android-alt","fas fa-paper-plane","fas fa-phone","fas fa-phone-slash","fas fa-phone-square","fas fa-phone-volume","fas fa-rss","fas fa-rss-square","fas fa-tty","fas fa-wifi"]},{"id":"accessibility","title":"Accessibility","icons":["fab fa-accessible-icon","fal fa-american-sign-language-interpreting","fal fa-assistive-listening-systems","fal fa-audio-description","fal fa-blind","fal fa-braille","fal fa-closed-captioning","fal fa-deaf","fal fa-low-vision","fal fa-phone-volume","fal fa-question-circle","fal fa-sign-language","fal fa-tty","fal fa-universal-access","fal fa-wheelchair","fas fa-american-sign-language-interpreting","fas fa-assistive-listening-systems","fas fa-audio-description","fas fa-blind","fas fa-braille","fas fa-closed-captioning","fas fa-deaf","fas fa-low-vision","fas fa-phone-volume","fas fa-question-circle","fas fa-sign-language","fas fa-tty","fas fa-universal-access","fas fa-wheelchair"]},{"id":"audio-video","title":"Audio & Video","icons":["fal fa-audio-description","fal fa-backward","fal fa-broadcast-tower","fal fa-circle","fal fa-closed-captioning","fal fa-compress","fal fa-compress-alt","fal fa-compress-wide","fal fa-eject","fal fa-expand","fal fa-expand-alt","fal fa-expand-arrows","fal fa-expand-arrows-alt","fal fa-expand-wide","fal fa-fast-backward","fal fa-fast-forward","fal fa-file-audio","fal fa-file-video","fal fa-film","fal fa-film-alt","fal fa-forward","fal fa-headphones","fal fa-microphone","fal fa-microphone-alt","fal fa-microphone-alt-slash","fal fa-microphone-slash","fal fa-music","fal fa-pause","fal fa-pause-circle","fal fa-phone-volume","fal fa-play","fal fa-play-circle","fal fa-podcast","fal fa-random","fal fa-rectangle-landscape","fal fa-rectangle-portrait","fal fa-rectangle-wide","fal fa-redo","fal fa-redo-alt","fal fa-repeat","fal fa-repeat-1","fal fa-repeat-1-alt","fal fa-repeat-alt","fal fa-rss","fal fa-rss-square","fal fa-scrubber","fal fa-step-backward","fal fa-step-forward","fal fa-stop","fal fa-stop-circle","fal fa-sync","fal fa-sync-alt","fal fa-undo","fal fa-undo-alt","fal fa-video","fal fa-volume-down","fal fa-volume-mute","fal fa-volume-off","fal fa-volume-up","fab fa-youtube","fas fa-audio-description","fas fa-backward","fas fa-broadcast-tower","fas fa-circle","fas fa-closed-captioning","fas fa-compress","fas fa-compress-alt","fas fa-compress-wide","fas fa-eject","fas fa-expand","fas fa-expand-alt","fas fa-expand-arrows","fas fa-expand-arrows-alt","fas fa-expand-wide","fas fa-fast-backward","fas fa-fast-forward","fas fa-file-audio","fas fa-file-video","fas fa-film","fas fa-film-alt","fas fa-forward","fas fa-headphones","fas fa-microphone","fas fa-microphone-alt","fas fa-microphone-alt-slash","fas fa-microphone-slash","fas fa-music","fas fa-pause","fas fa-pause-circle","fas fa-phone-volume","fas fa-play","fas fa-play-circle","fas fa-podcast","fas fa-random","fas fa-rectangle-landscape","fas fa-rectangle-portrait","fas fa-rectangle-wide","fas fa-redo","fas fa-redo-alt","fas fa-repeat","fas fa-repeat-1","fas fa-repeat-1-alt","fas fa-repeat-alt","fas fa-rss","fas fa-rss-square","fas fa-scrubber","fas fa-step-backward","fas fa-step-forward","fas fa-stop","fas fa-stop-circle","fas fa-sync","fas fa-sync-alt","fas fa-undo","fas fa-undo-alt","fas fa-video","fas fa-volume-down","fas fa-volume-mute","fas fa-volume-off","fas fa-volume-up"]},{"id":"currency","title":"Currency","icons":["fab fa-bitcoin","fab fa-btc","fal fa-dollar-sign","fal fa-euro-sign","fab fa-gg","fab fa-gg-circle","fal fa-lira-sign","fal fa-money-bill","fal fa-money-bill-alt","fal fa-money-bill-wave","fal fa-money-bill-wave-alt","fal fa-money-check","fal fa-money-check-alt","fal fa-pound-sign","fal fa-ruble-sign","fal fa-rupee-sign","fal fa-shekel-sign","fal fa-usd-circle","fal fa-usd-square","fal fa-won-sign","fal fa-yen-sign","fas fa-dollar-sign","fas fa-euro-sign","fas fa-lira-sign","fas fa-money-bill","fas fa-money-bill-alt","fas fa-money-bill-wave","fas fa-money-bill-wave-alt","fas fa-money-check","fas fa-money-check-alt","fas fa-pound-sign","fas fa-ruble-sign","fas fa-rupee-sign","fas fa-shekel-sign","fas fa-usd-circle","fas fa-usd-square","fas fa-won-sign","fas fa-yen-sign"]},{"id":"design","title":"Design","icons":["fal fa-adjust","fal fa-bezier-curve","fal fa-brush","fal fa-clone","fal fa-copy","fal fa-crop","fal fa-crop-alt","fal fa-crosshairs","fal fa-cut","fal fa-drafting-compass","fal fa-edit","fal fa-eraser","fal fa-eye","fal fa-eye-dropper","fal fa-eye-slash","fal fa-fill","fal fa-fill-drip","fal fa-highlighter","fal fa-magic","fal fa-marker","fal fa-object-group","fal fa-object-ungroup","fal fa-paint-brush","fal fa-paint-brush-alt","fal fa-paint-roller","fal fa-palette","fal fa-paste","fal fa-pen","fal fa-pen-alt","fal fa-pen-fancy","fal fa-pen-nib","fal fa-pencil","fal fa-pencil-alt","fal fa-pencil-ruler","fal fa-ruler-combined","fal fa-ruler-horizontal","fal fa-ruler-vertical","fal fa-save","fal fa-splotch","fal fa-spray-can","fal fa-stamp","fal fa-swatchbook","fal fa-tint","fal fa-tint-slash","fal fa-vector-square","fas fa-adjust","fas fa-bezier-curve","fas fa-brush","fas fa-clone","fas fa-copy","fas fa-crop","fas fa-crop-alt","fas fa-crosshairs","fas fa-cut","fas fa-drafting-compass","fas fa-edit","fas fa-eraser","fas fa-eye","fas fa-eye-dropper","fas fa-eye-slash","fas fa-fill","fas fa-fill-drip","fas fa-highlighter","fas fa-magic","fas fa-marker","fas fa-object-group","fas fa-object-ungroup","fas fa-paint-brush","fas fa-paint-brush-alt","fas fa-paint-roller","fas fa-palette","fas fa-paste","fas fa-pen","fas fa-pen-alt","fas fa-pen-fancy","fas fa-pen-nib","fas fa-pencil","fas fa-pencil-alt","fas fa-pencil-ruler","fas fa-ruler-combined","fas fa-ruler-horizontal","fas fa-ruler-vertical","fas fa-save","fas fa-splotch","fas fa-spray-can","fas fa-stamp","fas fa-swatchbook","fas fa-tint","fas fa-tint-slash","fas fa-vector-square"]},{"id":"chat","title":"Chat","icons":["fal fa-comment","fal fa-comment-alt","fal fa-comment-alt-check","fal fa-comment-alt-dots","fal fa-comment-alt-edit","fal fa-comment-alt-exclamation","fal fa-comment-alt-lines","fal fa-comment-alt-minus","fal fa-comment-alt-plus","fal fa-comment-alt-slash","fal fa-comment-alt-smile","fal fa-comment-alt-times","fal fa-comment-check","fal fa-comment-dots","fal fa-comment-edit","fal fa-comment-exclamation","fal fa-comment-lines","fal fa-comment-minus","fal fa-comment-plus","fal fa-comment-slash","fal fa-comment-smile","fal fa-comment-times","fal fa-comments","fal fa-comments-alt","fal fa-frown","fal fa-meh","fal fa-phone","fal fa-phone-plus","fal fa-phone-slash","fal fa-poo","fal fa-quote-left","fal fa-quote-right","fal fa-smile","fal fa-smile-plus","fal fa-video","fal fa-video-plus","fal fa-video-slash","fas fa-comment","fas fa-comment-alt","fas fa-comment-alt-check","fas fa-comment-alt-dots","fas fa-comment-alt-edit","fas fa-comment-alt-exclamation","fas fa-comment-alt-lines","fas fa-comment-alt-minus","fas fa-comment-alt-plus","fas fa-comment-alt-slash","fas fa-comment-alt-smile","fas fa-comment-alt-times","fas fa-comment-check","fas fa-comment-dots","fas fa-comment-edit","fas fa-comment-exclamation","fas fa-comment-lines","fas fa-comment-minus","fas fa-comment-plus","fas fa-comment-slash","fas fa-comment-smile","fas fa-comment-times","fas fa-comments","fas fa-comments-alt","fas fa-frown","fas fa-meh","fas fa-phone","fas fa-phone-plus","fas fa-phone-slash","fas fa-poo","fas fa-quote-left","fas fa-quote-right","fas fa-smile","fas fa-smile-plus","fas fa-video","fas fa-video-plus","fas fa-video-slash"]},{"id":"images","title":"Images","icons":["fal fa-adjust","fal fa-bolt","fal fa-camera","fal fa-camera-alt","fal fa-camera-retro","fal fa-chalkboard","fal fa-clone","fal fa-compress","fal fa-compress-wide","fal fa-expand","fal fa-expand-wide","fal fa-eye","fal fa-eye-dropper","fal fa-eye-slash","fal fa-file-image","fal fa-film","fal fa-film-alt","fal fa-id-badge","fal fa-id-card","fal fa-image","fal fa-images","fal fa-portrait","fal fa-rectangle-landscape","fal fa-rectangle-portrait","fal fa-rectangle-wide","fal fa-sliders-h","fal fa-sliders-h-square","fal fa-sliders-v","fal fa-sliders-v-square","fal fa-tint","fas fa-adjust","fas fa-bolt","fas fa-camera","fas fa-camera-alt","fas fa-camera-retro","fas fa-chalkboard","fas fa-clone","fas fa-compress","fas fa-compress-wide","fas fa-expand","fas fa-expand-wide","fas fa-eye","fas fa-eye-dropper","fas fa-eye-slash","fas fa-file-image","fas fa-film","fas fa-film-alt","fas fa-id-badge","fas fa-id-card","fas fa-image","fas fa-images","fas fa-portrait","fas fa-rectangle-landscape","fas fa-rectangle-portrait","fas fa-rectangle-wide","fas fa-sliders-h","fas fa-sliders-h-square","fas fa-sliders-v","fas fa-sliders-v-square","fas fa-tint"]},{"id":"sports","title":"Sports","icons":["fal fa-baseball","fal fa-baseball-ball","fal fa-basketball-ball","fal fa-basketball-hoop","fal fa-bowling-ball","fal fa-bowling-pins","fal fa-boxing-glove","fal fa-cricket","fal fa-curling","fal fa-dumbbell","fal fa-field-hockey","fal fa-football-ball","fal fa-football-helmet","fal fa-futbol","fal fa-golf-ball","fal fa-golf-club","fal fa-hockey-puck","fal fa-hockey-sticks","fal fa-luchador","fal fa-pennant","fal fa-quidditch","fal fa-racquet","fal fa-shuttlecock","fal fa-table-tennis","fal fa-tennis-ball","fal fa-volleyball-ball","fal fa-whistle","fas fa-baseball","fas fa-baseball-ball","fas fa-basketball-ball","fas fa-basketball-hoop","fas fa-bowling-ball","fas fa-bowling-pins","fas fa-boxing-glove","fas fa-cricket","fas fa-curling","fas fa-dumbbell","fas fa-field-hockey","fas fa-football-ball","fas fa-football-helmet","fas fa-futbol","fas fa-golf-ball","fas fa-golf-club","fas fa-hockey-puck","fas fa-hockey-sticks","fas fa-luchador","fas fa-pennant","fas fa-quidditch","fas fa-racquet","fas fa-shuttlecock","fas fa-table-tennis","fas fa-tennis-ball","fas fa-volleyball-ball","fas fa-whistle"]},{"id":"spinners","title":"Spinners","icons":["fal fa-asterisk","fal fa-badge","fal fa-certificate","fal fa-circle-notch","fal fa-cog","fal fa-compass","fal fa-crosshairs","fal fa-life-ring","fal fa-snowflake","fal fa-spinner","fal fa-spinner-third","fal fa-sun","fal fa-sync","fas fa-asterisk","fas fa-badge","fas fa-certificate","fas fa-circle-notch","fas fa-cog","fas fa-compass","fas fa-crosshairs","fas fa-life-ring","fas fa-snowflake","fas fa-spinner","fas fa-spinner-third","fas fa-sun","fas fa-sync"]},{"id":"emoji","title":"Emoji","icons":["fal fa-angry","fal fa-dizzy","fal fa-flushed","fal fa-frown","fal fa-frown-open","fal fa-grimace","fal fa-grin","fal fa-grin-alt","fal fa-grin-beam","fal fa-grin-beam-sweat","fal fa-grin-hearts","fal fa-grin-squint","fal fa-grin-squint-tears","fal fa-grin-stars","fal fa-grin-tears","fal fa-grin-tongue","fal fa-grin-tongue-squint","fal fa-grin-tongue-wink","fal fa-grin-wink","fal fa-kiss","fal fa-kiss-beam","fal fa-kiss-wink-heart","fal fa-laugh","fal fa-laugh-beam","fal fa-laugh-squint","fal fa-laugh-wink","fal fa-meh","fal fa-meh-blank","fal fa-meh-rolling-eyes","fal fa-sad-cry","fal fa-sad-tear","fal fa-smile","fal fa-smile-beam","fal fa-smile-wink","fal fa-surprise","fal fa-tired","fas fa-angry","fas fa-dizzy","fas fa-flushed","fas fa-frown","fas fa-frown-open","fas fa-grimace","fas fa-grin","fas fa-grin-alt","fas fa-grin-beam","fas fa-grin-beam-sweat","fas fa-grin-hearts","fas fa-grin-squint","fas fa-grin-squint-tears","fas fa-grin-stars","fas fa-grin-tears","fas fa-grin-tongue","fas fa-grin-tongue-squint","fas fa-grin-tongue-wink","fas fa-grin-wink","fas fa-kiss","fas fa-kiss-beam","fas fa-kiss-wink-heart","fas fa-laugh","fas fa-laugh-beam","fas fa-laugh-squint","fas fa-laugh-wink","fas fa-meh","fas fa-meh-blank","fas fa-meh-rolling-eyes","fas fa-sad-cry","fas fa-sad-tear","fas fa-smile","fas fa-smile-beam","fas fa-smile-wink","fas fa-surprise","fas fa-tired"]},{"id":"computers","title":"Computers","icons":["fal fa-database","fal fa-desktop","fal fa-desktop-alt","fal fa-download","fal fa-hdd","fal fa-headphones","fal fa-keyboard","fal fa-laptop","fal fa-memory","fal fa-microchip","fal fa-mobile","fal fa-mobile-alt","fal fa-mobile-android","fal fa-mobile-android-alt","fal fa-plug","fal fa-power-off","fal fa-print","fal fa-save","fal fa-server","fal fa-stream","fal fa-tablet","fal fa-tablet-alt","fal fa-tablet-android","fal fa-tablet-android-alt","fal fa-tv","fal fa-tv-retro","fal fa-upload","fas fa-database","fas fa-desktop","fas fa-desktop-alt","fas fa-download","fas fa-hdd","fas fa-headphones","fas fa-keyboard","fas fa-laptop","fas fa-memory","fas fa-microchip","fas fa-mobile","fas fa-mobile-alt","fas fa-mobile-android","fas fa-mobile-android-alt","fas fa-plug","fas fa-power-off","fas fa-print","fas fa-save","fas fa-server","fas fa-stream","fas fa-tablet","fas fa-tablet-alt","fas fa-tablet-android","fas fa-tablet-android-alt","fas fa-tv","fas fa-tv-retro","fas fa-upload"]},{"id":"travel","title":"Travel","icons":["fal fa-archway","fal fa-atlas","fal fa-bed","fal fa-bus","fal fa-bus-alt","fal fa-cocktail","fal fa-concierge-bell","fal fa-dumbbell","fal fa-glass-martini","fal fa-glass-martini-alt","fal fa-globe-africa","fal fa-globe-americas","fal fa-globe-asia","fal fa-hot-tub","fal fa-hotel","fal fa-luggage-cart","fal fa-map","fal fa-map-marked","fal fa-map-marked-alt","fal fa-monument","fal fa-passport","fal fa-plane","fal fa-plane-arrival","fal fa-plane-departure","fal fa-shuttle-van","fal fa-spa","fal fa-suitcase","fal fa-suitcase-rolling","fal fa-swimmer","fal fa-swimming-pool","fal fa-taxi","fal fa-umbrella-beach","fal fa-wine-glass","fal fa-wine-glass-alt","fas fa-archway","fas fa-atlas","fas fa-bed","fas fa-bus","fas fa-bus-alt","fas fa-cocktail","fas fa-concierge-bell","fas fa-dumbbell","fas fa-glass-martini","fas fa-glass-martini-alt","fas fa-globe-africa","fas fa-globe-americas","fas fa-globe-asia","fas fa-hot-tub","fas fa-hotel","fas fa-luggage-cart","fas fa-map","fas fa-map-marked","fas fa-map-marked-alt","fas fa-monument","fas fa-passport","fas fa-plane","fas fa-plane-arrival","fas fa-plane-departure","fas fa-shuttle-van","fas fa-spa","fas fa-suitcase","fas fa-suitcase-rolling","fas fa-swimmer","fas fa-swimming-pool","fas fa-taxi","fas fa-umbrella-beach","fas fa-wine-glass","fas fa-wine-glass-alt"]},{"id":"editors","title":"Editors","icons":["fal fa-align-center","fal fa-align-justify","fal fa-align-left","fal fa-align-right","fal fa-bold","fal fa-clipboard","fal fa-clone","fal fa-columns","fal fa-copy","fal fa-cut","fal fa-edit","fal fa-eraser","fal fa-file","fal fa-file-alt","fal fa-font","fal fa-glasses","fal fa-h1","fal fa-h2","fal fa-h3","fal fa-heading","fal fa-highlighter","fal fa-i-cursor","fal fa-indent","fal fa-italic","fal fa-link","fal fa-list","fal fa-list-alt","fal fa-list-ol","fal fa-list-ul","fal fa-marker","fal fa-outdent","fal fa-paper-plane","fal fa-paperclip","fal fa-paragraph","fal fa-paste","fal fa-pen","fal fa-pen-alt","fal fa-pen-fancy","fal fa-pen-nib","fal fa-pencil","fal fa-pencil-alt","fal fa-print","fal fa-quote-left","fal fa-quote-right","fal fa-redo","fal fa-redo-alt","fal fa-reply","fal fa-reply-all","fal fa-screwdriver","fal fa-share","fal fa-share-all","fal fa-strikethrough","fal fa-subscript","fal fa-superscript","fal fa-sync","fal fa-sync-alt","fal fa-table","fal fa-tasks","fal fa-text-height","fal fa-text-width","fal fa-th","fal fa-th-large","fal fa-th-list","fal fa-trash","fal fa-trash-alt","fal fa-underline","fal fa-undo","fal fa-undo-alt","fal fa-unlink","fal fa-wrench","fas fa-align-center","fas fa-align-justify","fas fa-align-left","fas fa-align-right","fas fa-bold","fas fa-clipboard","fas fa-clone","fas fa-columns","fas fa-copy","fas fa-cut","fas fa-edit","fas fa-eraser","fas fa-file","fas fa-file-alt","fas fa-font","fas fa-glasses","fas fa-h1","fas fa-h2","fas fa-h3","fas fa-heading","fas fa-highlighter","fas fa-i-cursor","fas fa-indent","fas fa-italic","fas fa-link","fas fa-list","fas fa-list-alt","fas fa-list-ol","fas fa-list-ul","fas fa-marker","fas fa-outdent","fas fa-paper-plane","fas fa-paperclip","fas fa-paragraph","fas fa-paste","fas fa-pen","fas fa-pen-alt","fas fa-pen-fancy","fas fa-pen-nib","fas fa-pencil","fas fa-pencil-alt","fas fa-print","fas fa-quote-left","fas fa-quote-right","fas fa-redo","fas fa-redo-alt","fas fa-reply","fas fa-reply-all","fas fa-screwdriver","fas fa-share","fas fa-share-all","fas fa-strikethrough","fas fa-subscript","fas fa-superscript","fas fa-sync","fas fa-sync-alt","fas fa-table","fas fa-tasks","fas fa-text-height","fas fa-text-width","fas fa-th","fas fa-th-large","fas fa-th-list","fas fa-trash","fas fa-trash-alt","fas fa-underline","fas fa-undo","fas fa-undo-alt","fas fa-unlink","fas fa-wrench"]},{"id":"writing","title":"Writing","icons":["fal fa-archive","fal fa-book","fal fa-bookmark","fal fa-edit","fal fa-envelope","fal fa-envelope-open","fal fa-eraser","fal fa-file","fal fa-file-alt","fal fa-folder","fal fa-folder-open","fal fa-keyboard","fal fa-newspaper","fal fa-paper-plane","fal fa-paperclip","fal fa-paragraph","fal fa-pen","fal fa-pen-alt","fal fa-pen-square","fal fa-pencil","fal fa-pencil-alt","fal fa-quote-left","fal fa-quote-right","fal fa-sticky-note","fal fa-thumbtack","fas fa-archive","fas fa-book","fas fa-bookmark","fas fa-edit","fas fa-envelope","fas fa-envelope-open","fas fa-eraser","fas fa-file","fas fa-file-alt","fas fa-folder","fas fa-folder-open","fas fa-keyboard","fas fa-newspaper","fas fa-paper-plane","fas fa-paperclip","fas fa-paragraph","fas fa-pen","fas fa-pen-alt","fas fa-pen-square","fas fa-pencil","fas fa-pencil-alt","fas fa-quote-left","fas fa-quote-right","fas fa-sticky-note","fas fa-thumbtack"]},{"id":"maps","title":"Maps","icons":["fal fa-ambulance","fal fa-anchor","fal fa-balance-scale","fal fa-balance-scale-left","fal fa-balance-scale-right","fal fa-bath","fal fa-bed","fal fa-beer","fal fa-bell","fal fa-bell-slash","fal fa-bicycle","fal fa-binoculars","fal fa-birthday-cake","fal fa-blind","fal fa-bomb","fal fa-book","fal fa-bookmark","fal fa-briefcase","fal fa-building","fal fa-car","fal fa-coffee","fal fa-crosshairs","fal fa-dollar-sign","fal fa-eye","fal fa-eye-slash","fal fa-fighter-jet","fal fa-fire","fal fa-fire-extinguisher","fal fa-flag","fal fa-flag-checkered","fal fa-flask","fal fa-gamepad","fal fa-gavel","fal fa-gift","fal fa-glass-martini","fal fa-globe","fal fa-graduation-cap","fal fa-h-square","fal fa-heart","fal fa-heartbeat","fal fa-helicopter","fal fa-home","fal fa-hospital","fal fa-image","fal fa-images","fal fa-industry","fal fa-industry-alt","fal fa-info","fal fa-info-circle","fal fa-info-square","fal fa-key","fal fa-leaf","fal fa-lemon","fal fa-life-ring","fal fa-lightbulb","fal fa-location-arrow","fal fa-low-vision","fal fa-magnet","fal fa-male","fal fa-map","fal fa-map-marker","fal fa-map-marker-alt","fal fa-map-pin","fal fa-map-signs","fal fa-medkit","fal fa-money-bill","fal fa-money-bill-alt","fal fa-motorcycle","fal fa-music","fal fa-newspaper","fal fa-paw","fal fa-phone","fal fa-phone-square","fal fa-phone-volume","fal fa-plane","fal fa-plane-alt","fal fa-plug","fal fa-plus","fal fa-plus-square","fal fa-print","fal fa-recycle","fal fa-road","fal fa-rocket","fal fa-route","fal fa-search","fal fa-search-minus","fal fa-search-plus","fal fa-ship","fal fa-shoe-prints","fal fa-shopping-bag","fal fa-shopping-basket","fal fa-shopping-cart","fal fa-shower","fal fa-street-view","fal fa-subway","fal fa-suitcase","fal fa-tag","fal fa-tags","fal fa-taxi","fal fa-thumbtack","fal fa-ticket","fal fa-ticket-alt","fal fa-tint","fal fa-train","fal fa-tree","fal fa-tree-alt","fal fa-trophy","fal fa-trophy-alt","fal fa-truck","fal fa-tty","fal fa-umbrella","fal fa-university","fal fa-usd-circle","fal fa-usd-square","fal fa-utensil-fork","fal fa-utensil-knife","fal fa-utensil-spoon","fal fa-utensils","fal fa-utensils-alt","fal fa-wheelchair","fal fa-wifi","fal fa-wine-glass","fal fa-wrench","fas fa-ambulance","fas fa-anchor","fas fa-balance-scale","fas fa-balance-scale-left","fas fa-balance-scale-right","fas fa-bath","fas fa-bed","fas fa-beer","fas fa-bell","fas fa-bell-slash","fas fa-bicycle","fas fa-binoculars","fas fa-birthday-cake","fas fa-blind","fas fa-bomb","fas fa-book","fas fa-bookmark","fas fa-briefcase","fas fa-building","fas fa-car","fas fa-coffee","fas fa-crosshairs","fas fa-dollar-sign","fas fa-eye","fas fa-eye-slash","fas fa-fighter-jet","fas fa-fire","fas fa-fire-extinguisher","fas fa-flag","fas fa-flag-checkered","fas fa-flask","fas fa-gamepad","fas fa-gavel","fas fa-gift","fas fa-glass-martini","fas fa-globe","fas fa-graduation-cap","fas fa-h-square","fas fa-heart","fas fa-heartbeat","fas fa-helicopter","fas fa-home","fas fa-hospital","fas fa-image","fas fa-images","fas fa-industry","fas fa-industry-alt","fas fa-info","fas fa-info-circle","fas fa-info-square","fas fa-key","fas fa-leaf","fas fa-lemon","fas fa-life-ring","fas fa-lightbulb","fas fa-location-arrow","fas fa-low-vision","fas fa-magnet","fas fa-male","fas fa-map","fas fa-map-marker","fas fa-map-marker-alt","fas fa-map-pin","fas fa-map-signs","fas fa-medkit","fas fa-money-bill","fas fa-money-bill-alt","fas fa-motorcycle","fas fa-music","fas fa-newspaper","fas fa-paw","fas fa-phone","fas fa-phone-square","fas fa-phone-volume","fas fa-plane","fas fa-plane-alt","fas fa-plug","fas fa-plus","fas fa-plus-square","fas fa-print","fas fa-recycle","fas fa-road","fas fa-rocket","fas fa-route","fas fa-search","fas fa-search-minus","fas fa-search-plus","fas fa-ship","fas fa-shoe-prints","fas fa-shopping-bag","fas fa-shopping-basket","fas fa-shopping-cart","fas fa-shower","fas fa-street-view","fas fa-subway","fas fa-suitcase","fas fa-tag","fas fa-tags","fas fa-taxi","fas fa-thumbtack","fas fa-ticket","fas fa-ticket-alt","fas fa-tint","fas fa-train","fas fa-tree","fas fa-tree-alt","fas fa-trophy","fas fa-trophy-alt","fas fa-truck","fas fa-tty","fas fa-umbrella","fas fa-university","fas fa-usd-circle","fas fa-usd-square","fas fa-utensil-fork","fas fa-utensil-knife","fas fa-utensil-spoon","fas fa-utensils","fas fa-utensils-alt","fas fa-wheelchair","fas fa-wifi","fas fa-wine-glass","fas fa-wrench"]},{"id":"health","title":"Health","icons":["fab fa-accessible-icon","fal fa-ambulance","fal fa-h-square","fal fa-heart","fal fa-heartbeat","fal fa-hospital","fal fa-medkit","fal fa-plus-square","fal fa-prescription","fal fa-stethoscope","fal fa-user-md","fal fa-wheelchair","fas fa-ambulance","fas fa-h-square","fas fa-heart","fas fa-heartbeat","fas fa-hospital","fas fa-medkit","fas fa-plus-square","fas fa-prescription","fas fa-stethoscope","fas fa-user-md","fas fa-wheelchair"]},{"id":"chess","title":"Chess","icons":["fal fa-chess","fal fa-chess-bishop","fal fa-chess-bishop-alt","fal fa-chess-board","fal fa-chess-clock","fal fa-chess-clock-alt","fal fa-chess-king","fal fa-chess-king-alt","fal fa-chess-knight","fal fa-chess-knight-alt","fal fa-chess-pawn","fal fa-chess-pawn-alt","fal fa-chess-queen","fal fa-chess-queen-alt","fal fa-chess-rook","fal fa-chess-rook-alt","fal fa-square-full","fas fa-chess","fas fa-chess-bishop","fas fa-chess-bishop-alt","fas fa-chess-board","fas fa-chess-clock","fas fa-chess-clock-alt","fas fa-chess-king","fas fa-chess-king-alt","fas fa-chess-knight","fas fa-chess-knight-alt","fas fa-chess-pawn","fas fa-chess-pawn-alt","fas fa-chess-queen","fas fa-chess-queen-alt","fas fa-chess-rook","fas fa-chess-rook-alt","fas fa-square-full"]},{"id":"mathematics","title":"Mathematics","icons":["fal fa-divide","fal fa-equals","fal fa-greater-than","fal fa-greater-than-equal","fal fa-less-than","fal fa-less-than-equal","fal fa-minus","fal fa-not-equal","fal fa-percentage","fal fa-plus","fal fa-subscript","fal fa-superscript","fal fa-times","fas fa-divide","fas fa-equals","fas fa-greater-than","fas fa-greater-than-equal","fas fa-less-than","fas fa-less-than-equal","fas fa-minus","fas fa-not-equal","fas fa-percentage","fas fa-plus","fas fa-subscript","fas fa-superscript","fas fa-times"]},{"id":"payments-shopping","title":"Payments & Shopping","icons":["fab fa-amazon-pay","fab fa-apple-pay","fal fa-badge","fal fa-bell","fal fa-bookmark","fal fa-bullhorn","fal fa-camera","fal fa-camera-alt","fal fa-camera-retro","fal fa-cart-arrow-down","fal fa-cart-plus","fab fa-cc-amazon-pay","fab fa-cc-amex","fab fa-cc-apple-pay","fab fa-cc-diners-club","fab fa-cc-discover","fab fa-cc-jcb","fab fa-cc-mastercard","fab fa-cc-paypal","fab fa-cc-stripe","fab fa-cc-visa","fal fa-certificate","fal fa-credit-card","fal fa-credit-card-blank","fal fa-credit-card-front","fab fa-ethereum","fal fa-gem","fal fa-gift","fab fa-google-wallet","fal fa-handshake","fal fa-heart","fal fa-key","fal fa-money-check","fal fa-money-check-alt","fab fa-paypal","fal fa-shopping-bag","fal fa-shopping-basket","fal fa-shopping-cart","fal fa-star","fal fa-star-exclamation","fab fa-stripe","fab fa-stripe-s","fal fa-tag","fal fa-tags","fal fa-thumbs-down","fal fa-thumbs-up","fal fa-trophy","fal fa-trophy-alt","fas fa-badge","fas fa-bell","fas fa-bookmark","fas fa-bullhorn","fas fa-camera","fas fa-camera-alt","fas fa-camera-retro","fas fa-cart-arrow-down","fas fa-cart-plus","fas fa-certificate","fas fa-credit-card","fas fa-credit-card-blank","fas fa-credit-card-front","fas fa-gem","fas fa-gift","fas fa-handshake","fas fa-heart","fas fa-key","fas fa-money-check","fas fa-money-check-alt","fas fa-shopping-bag","fas fa-shopping-basket","fas fa-shopping-cart","fas fa-star","fas fa-star-exclamation","fas fa-tag","fas fa-tags","fas fa-thumbs-down","fas fa-thumbs-up","fas fa-trophy","fas fa-trophy-alt"]},{"id":"date-time","title":"Date & Time","icons":["fal fa-alarm-clock","fal fa-bell","fal fa-bell-slash","fal fa-calendar","fal fa-calendar-alt","fal fa-calendar-check","fal fa-calendar-edit","fal fa-calendar-exclamation","fal fa-calendar-minus","fal fa-calendar-plus","fal fa-calendar-times","fal fa-clock","fal fa-hourglass","fal fa-hourglass-end","fal fa-hourglass-half","fal fa-hourglass-start","fal fa-stopwatch","fal fa-watch","fas fa-alarm-clock","fas fa-bell","fas fa-bell-slash","fas fa-calendar","fas fa-calendar-alt","fas fa-calendar-check","fas fa-calendar-edit","fas fa-calendar-exclamation","fas fa-calendar-minus","fas fa-calendar-plus","fas fa-calendar-times","fas fa-clock","fas fa-hourglass","fas fa-hourglass-end","fas fa-hourglass-half","fas fa-hourglass-start","fas fa-stopwatch","fas fa-watch"]},{"id":"status","title":"Status","icons":["fal fa-badge","fal fa-badge-check","fal fa-ban","fal fa-battery-bolt","fal fa-battery-empty","fal fa-battery-full","fal fa-battery-half","fal fa-battery-quarter","fal fa-battery-slash","fal fa-battery-three-quarters","fal fa-bell","fal fa-bell-slash","fal fa-calendar","fal fa-calendar-alt","fal fa-calendar-check","fal fa-calendar-edit","fal fa-calendar-exclamation","fal fa-calendar-minus","fal fa-calendar-plus","fal fa-calendar-times","fal fa-cart-arrow-down","fal fa-cart-plus","fal fa-door-closed","fal fa-door-open","fal fa-exclamation","fal fa-exclamation-circle","fal fa-exclamation-square","fal fa-exclamation-triangle","fal fa-eye","fal fa-eye-slash","fal fa-file","fal fa-file-alt","fal fa-file-check","fal fa-file-edit","fal fa-file-exclamation","fal fa-file-minus","fal fa-file-plus","fal fa-file-times","fal fa-folder","fal fa-folder-open","fal fa-info","fal fa-info-circle","fal fa-info-square","fal fa-lock","fal fa-lock-alt","fal fa-lock-open","fal fa-lock-open-alt","fal fa-minus","fal fa-minus-circle","fal fa-minus-hexagon","fal fa-minus-octagon","fal fa-minus-square","fal fa-plus","fal fa-plus-circle","fal fa-plus-hexagon","fal fa-plus-octagon","fal fa-plus-square","fal fa-question","fal fa-question-circle","fal fa-question-square","fal fa-shield","fal fa-shield-alt","fal fa-shield-check","fal fa-shopping-cart","fal fa-sign-in","fal fa-sign-in-alt","fal fa-sign-out","fal fa-sign-out-alt","fal fa-stream","fal fa-thermometer-empty","fal fa-thermometer-full","fal fa-thermometer-half","fal fa-thermometer-quarter","fal fa-thermometer-three-quarters","fal fa-thumbs-down","fal fa-thumbs-up","fal fa-toggle-off","fal fa-toggle-on","fal fa-unlock","fal fa-unlock-alt","fas fa-badge","fas fa-badge-check","fas fa-ban","fas fa-battery-bolt","fas fa-battery-empty","fas fa-battery-full","fas fa-battery-half","fas fa-battery-quarter","fas fa-battery-slash","fas fa-battery-three-quarters","fas fa-bell","fas fa-bell-slash","fas fa-calendar","fas fa-calendar-alt","fas fa-calendar-check","fas fa-calendar-edit","fas fa-calendar-exclamation","fas fa-calendar-minus","fas fa-calendar-plus","fas fa-calendar-times","fas fa-cart-arrow-down","fas fa-cart-plus","fas fa-door-closed","fas fa-door-open","fas fa-exclamation","fas fa-exclamation-circle","fas fa-exclamation-square","fas fa-exclamation-triangle","fas fa-eye","fas fa-eye-slash","fas fa-file","fas fa-file-alt","fas fa-file-check","fas fa-file-edit","fas fa-file-exclamation","fas fa-file-minus","fas fa-file-plus","fas fa-file-times","fas fa-folder","fas fa-folder-open","fas fa-info","fas fa-info-circle","fas fa-info-square","fas fa-lock","fas fa-lock-alt","fas fa-lock-open","fas fa-lock-open-alt","fas fa-minus","fas fa-minus-circle","fas fa-minus-hexagon","fas fa-minus-octagon","fas fa-minus-square","fas fa-plus","fas fa-plus-circle","fas fa-plus-hexagon","fas fa-plus-octagon","fas fa-plus-square","fas fa-question","fas fa-question-circle","fas fa-question-square","fas fa-shield","fas fa-shield-alt","fas fa-shield-check","fas fa-shopping-cart","fas fa-sign-in","fas fa-sign-in-alt","fas fa-sign-out","fas fa-sign-out-alt","fas fa-stream","fas fa-thermometer-empty","fas fa-thermometer-full","fas fa-thermometer-half","fas fa-thermometer-quarter","fas fa-thermometer-three-quarters","fas fa-thumbs-down","fas fa-thumbs-up","fas fa-toggle-off","fas fa-toggle-on","fas fa-unlock","fas fa-unlock-alt"]},{"id":"buildings","title":"Buildings","icons":["fal fa-building","fal fa-church","fal fa-home","fal fa-hospital","fal fa-hospital-alt","fal fa-industry","fal fa-industry-alt","fal fa-school","fal fa-university","fas fa-building","fas fa-church","fas fa-home","fas fa-hospital","fas fa-hospital-alt","fas fa-industry","fas fa-industry-alt","fas fa-school","fas fa-university"]},{"id":"business","title":"Business","icons":["fal fa-address-book","fal fa-address-card","fal fa-archive","fal fa-badge","fal fa-badge-check","fal fa-balance-scale","fal fa-balance-scale-left","fal fa-balance-scale-right","fal fa-birthday-cake","fal fa-book","fal fa-briefcase","fal fa-building","fal fa-bullhorn","fal fa-bullseye","fal fa-calculator","fal fa-calendar","fal fa-calendar-alt","fal fa-certificate","fal fa-chart-area","fal fa-chart-bar","fal fa-chart-line","fal fa-chart-pie","fal fa-clipboard","fal fa-coffee","fal fa-columns","fal fa-compass","fal fa-copy","fal fa-copyright","fal fa-cut","fal fa-edit","fal fa-envelope","fal fa-envelope-open","fal fa-envelope-square","fal fa-eraser","fal fa-fax","fal fa-file","fal fa-file-alt","fal fa-folder","fal fa-folder-open","fal fa-glasses","fal fa-globe","fal fa-highlighter","fal fa-industry","fal fa-industry-alt","fal fa-marker","fal fa-paperclip","fal fa-paste","fal fa-pen","fal fa-pen-alt","fal fa-pen-fancy","fal fa-pen-nib","fal fa-pen-square","fal fa-pencil","fal fa-pencil-alt","fal fa-percent","fal fa-phone","fal fa-phone-slash","fal fa-phone-square","fal fa-phone-volume","fal fa-project-diagram","fal fa-registered","fal fa-save","fal fa-sitemap","fal fa-sticky-note","fal fa-stream","fal fa-suitcase","fal fa-table","fal fa-tag","fal fa-tags","fal fa-tasks","fal fa-thumbtack","fal fa-trademark","fal fa-wallet","fas fa-address-book","fas fa-address-card","fas fa-archive","fas fa-badge","fas fa-badge-check","fas fa-balance-scale","fas fa-balance-scale-left","fas fa-balance-scale-right","fas fa-birthday-cake","fas fa-book","fas fa-briefcase","fas fa-building","fas fa-bullhorn","fas fa-bullseye","fas fa-calculator","fas fa-calendar","fas fa-calendar-alt","fas fa-certificate","fas fa-chart-area","fas fa-chart-bar","fas fa-chart-line","fas fa-chart-pie","fas fa-clipboard","fas fa-coffee","fas fa-columns","fas fa-compass","fas fa-copy","fas fa-copyright","fas fa-cut","fas fa-edit","fas fa-envelope","fas fa-envelope-open","fas fa-envelope-square","fas fa-eraser","fas fa-fax","fas fa-file","fas fa-file-alt","fas fa-folder","fas fa-folder-open","fas fa-glasses","fas fa-globe","fas fa-highlighter","fas fa-industry","fas fa-industry-alt","fas fa-marker","fas fa-paperclip","fas fa-paste","fas fa-pen","fas fa-pen-alt","fas fa-pen-fancy","fas fa-pen-nib","fas fa-pen-square","fas fa-pencil","fas fa-pencil-alt","fas fa-percent","fas fa-phone","fas fa-phone-slash","fas fa-phone-square","fas fa-phone-volume","fas fa-project-diagram","fas fa-registered","fas fa-save","fas fa-sitemap","fas fa-sticky-note","fas fa-stream","fas fa-suitcase","fas fa-table","fas fa-tag","fas fa-tags","fas fa-tasks","fas fa-thumbtack","fas fa-trademark","fas fa-wallet"]},{"id":"interfaces","title":"Interfaces","icons":["fal fa-award","fal fa-badge","fal fa-badge-check","fal fa-ban","fal fa-barcode","fal fa-bars","fal fa-beer","fal fa-bell","fal fa-bell-slash","fal fa-bug","fal fa-bullhorn","fal fa-bullseye","fal fa-calculator","fal fa-calendar","fal fa-calendar-alt","fal fa-calendar-check","fal fa-calendar-edit","fal fa-calendar-exclamation","fal fa-calendar-minus","fal fa-calendar-plus","fal fa-calendar-times","fal fa-certificate","fal fa-check","fal fa-check-circle","fal fa-check-double","fal fa-check-square","fal fa-circle","fal fa-clipboard","fal fa-clone","fal fa-cloud","fal fa-cloud-download","fal fa-cloud-download-alt","fal fa-cloud-upload","fal fa-cloud-upload-alt","fal fa-coffee","fal fa-cog","fal fa-cogs","fal fa-copy","fal fa-cut","fal fa-database","fal fa-dot-circle","fal fa-download","fal fa-edit","fal fa-ellipsis-h","fal fa-ellipsis-h-alt","fal fa-ellipsis-v","fal fa-ellipsis-v-alt","fal fa-envelope","fal fa-envelope-open","fal fa-eraser","fal fa-exclamation","fal fa-exclamation-circle","fal fa-exclamation-square","fal fa-exclamation-triangle","fal fa-external-link","fal fa-external-link-alt","fal fa-external-link-square","fal fa-external-link-square-alt","fal fa-eye","fal fa-eye-slash","fal fa-file","fal fa-file-alt","fal fa-file-download","fal fa-file-export","fal fa-file-import","fal fa-file-upload","fal fa-filter","fal fa-fingerprint","fal fa-flag","fal fa-flag-checkered","fal fa-folder","fal fa-folder-open","fal fa-frown","fal fa-glasses","fal fa-grip-horizontal","fal fa-grip-vertical","fal fa-hashtag","fal fa-heart","fal fa-history","fal fa-home","fal fa-i-cursor","fal fa-info","fal fa-info-circle","fal fa-info-square","fal fa-language","fal fa-magic","fal fa-marker","fal fa-medal","fal fa-meh","fal fa-microphone","fal fa-microphone-alt","fal fa-microphone-slash","fal fa-minus","fal fa-minus-circle","fal fa-minus-hexagon","fal fa-minus-octagon","fal fa-minus-square","fal fa-paste","fal fa-pen","fal fa-pen-alt","fal fa-pen-fancy","fal fa-pencil","fal fa-pencil-alt","fal fa-plus","fal fa-plus-circle","fal fa-plus-hexagon","fal fa-plus-octagon","fal fa-plus-square","fal fa-poo","fal fa-qrcode","fal fa-question","fal fa-question-circle","fal fa-question-square","fal fa-quote-left","fal fa-quote-right","fal fa-redo","fal fa-redo-alt","fal fa-reply","fal fa-reply-all","fal fa-rss","fal fa-rss-square","fal fa-save","fal fa-screwdriver","fal fa-search","fal fa-search-minus","fal fa-search-plus","fal fa-share","fal fa-share-all","fal fa-share-alt","fal fa-share-alt-square","fal fa-share-square","fal fa-shield","fal fa-shield-alt","fal fa-sign-in","fal fa-sign-in-alt","fal fa-sign-out","fal fa-sign-out-alt","fal fa-signal","fal fa-sitemap","fal fa-sliders-h","fal fa-sliders-h-square","fal fa-sliders-v","fal fa-sliders-v-square","fal fa-smile","fal fa-sort","fal fa-sort-alpha-down","fal fa-sort-alpha-up","fal fa-sort-amount-down","fal fa-sort-amount-up","fal fa-sort-down","fal fa-sort-numeric-down","fal fa-sort-numeric-up","fal fa-sort-up","fal fa-star","fal fa-star-exclamation","fal fa-star-half","fal fa-sync","fal fa-sync-alt","fal fa-thumbs-down","fal fa-thumbs-up","fal fa-times","fal fa-times-circle","fal fa-times-hexagon","fal fa-times-octagon","fal fa-times-square","fal fa-toggle-off","fal fa-toggle-on","fal fa-trash","fal fa-trash-alt","fal fa-trophy","fal fa-trophy-alt","fal fa-undo","fal fa-undo-alt","fal fa-upload","fal fa-user","fal fa-user-alt","fal fa-user-circle","fal fa-wifi","fal fa-wrench","fas fa-award","fas fa-badge","fas fa-badge-check","fas fa-ban","fas fa-barcode","fas fa-bars","fas fa-beer","fas fa-bell","fas fa-bell-slash","fas fa-bug","fas fa-bullhorn","fas fa-bullseye","fas fa-calculator","fas fa-calendar","fas fa-calendar-alt","fas fa-calendar-check","fas fa-calendar-edit","fas fa-calendar-exclamation","fas fa-calendar-minus","fas fa-calendar-plus","fas fa-calendar-times","fas fa-certificate","fas fa-check","fas fa-check-circle","fas fa-check-double","fas fa-check-square","fas fa-circle","fas fa-clipboard","fas fa-clone","fas fa-cloud","fas fa-cloud-download","fas fa-cloud-download-alt","fas fa-cloud-upload","fas fa-cloud-upload-alt","fas fa-coffee","fas fa-cog","fas fa-cogs","fas fa-copy","fas fa-cut","fas fa-database","fas fa-dot-circle","fas fa-download","fas fa-edit","fas fa-ellipsis-h","fas fa-ellipsis-h-alt","fas fa-ellipsis-v","fas fa-ellipsis-v-alt","fas fa-envelope","fas fa-envelope-open","fas fa-eraser","fas fa-exclamation","fas fa-exclamation-circle","fas fa-exclamation-square","fas fa-exclamation-triangle","fas fa-external-link","fas fa-external-link-alt","fas fa-external-link-square","fas fa-external-link-square-alt","fas fa-eye","fas fa-eye-slash","fas fa-file","fas fa-file-alt","fas fa-file-download","fas fa-file-export","fas fa-file-import","fas fa-file-upload","fas fa-filter","fas fa-fingerprint","fas fa-flag","fas fa-flag-checkered","fas fa-folder","fas fa-folder-open","fas fa-frown","fas fa-glasses","fas fa-grip-horizontal","fas fa-grip-vertical","fas fa-hashtag","fas fa-heart","fas fa-history","fas fa-home","fas fa-i-cursor","fas fa-info","fas fa-info-circle","fas fa-info-square","fas fa-language","fas fa-magic","fas fa-marker","fas fa-medal","fas fa-meh","fas fa-microphone","fas fa-microphone-alt","fas fa-microphone-slash","fas fa-minus","fas fa-minus-circle","fas fa-minus-hexagon","fas fa-minus-octagon","fas fa-minus-square","fas fa-paste","fas fa-pen","fas fa-pen-alt","fas fa-pen-fancy","fas fa-pencil","fas fa-pencil-alt","fas fa-plus","fas fa-plus-circle","fas fa-plus-hexagon","fas fa-plus-octagon","fas fa-plus-square","fas fa-poo","fas fa-qrcode","fas fa-question","fas fa-question-circle","fas fa-question-square","fas fa-quote-left","fas fa-quote-right","fas fa-redo","fas fa-redo-alt","fas fa-reply","fas fa-reply-all","fas fa-rss","fas fa-rss-square","fas fa-save","fas fa-screwdriver","fas fa-search","fas fa-search-minus","fas fa-search-plus","fas fa-share","fas fa-share-all","fas fa-share-alt","fas fa-share-alt-square","fas fa-share-square","fas fa-shield","fas fa-shield-alt","fas fa-sign-in","fas fa-sign-in-alt","fas fa-sign-out","fas fa-sign-out-alt","fas fa-signal","fas fa-sitemap","fas fa-sliders-h","fas fa-sliders-h-square","fas fa-sliders-v","fas fa-sliders-v-square","fas fa-smile","fas fa-sort","fas fa-sort-alpha-down","fas fa-sort-alpha-up","fas fa-sort-amount-down","fas fa-sort-amount-up","fas fa-sort-down","fas fa-sort-numeric-down","fas fa-sort-numeric-up","fas fa-sort-up","fas fa-star","fas fa-star-exclamation","fas fa-star-half","fas fa-sync","fas fa-sync-alt","fas fa-thumbs-down","fas fa-thumbs-up","fas fa-times","fas fa-times-circle","fas fa-times-hexagon","fas fa-times-octagon","fas fa-times-square","fas fa-toggle-off","fas fa-toggle-on","fas fa-trash","fas fa-trash-alt","fas fa-trophy","fas fa-trophy-alt","fas fa-undo","fas fa-undo-alt","fas fa-upload","fas fa-user","fas fa-user-alt","fas fa-user-circle","fas fa-wifi","fas fa-wrench"]},{"id":"objects","title":"Objects","icons":["fal fa-alarm-clock","fal fa-ambulance","fal fa-anchor","fal fa-archive","fal fa-award","fal fa-balance-scale","fal fa-balance-scale-left","fal fa-balance-scale-right","fal fa-bath","fal fa-bed","fal fa-beer","fal fa-bell","fal fa-bicycle","fal fa-binoculars","fal fa-birthday-cake","fal fa-blender","fal fa-bomb","fal fa-book","fal fa-bookmark","fal fa-briefcase","fal fa-broadcast-tower","fal fa-bug","fal fa-building","fal fa-bullhorn","fal fa-bullseye","fal fa-bus","fal fa-calculator","fal fa-calendar","fal fa-calendar-alt","fal fa-camera","fal fa-camera-alt","fal fa-camera-retro","fal fa-car","fal fa-church","fal fa-clipboard","fal fa-cloud","fal fa-coffee","fal fa-cog","fal fa-cogs","fal fa-compass","fal fa-cookie","fal fa-cookie-bite","fal fa-copy","fal fa-cube","fal fa-cubes","fal fa-cut","fal fa-dice","fal fa-dice-five","fal fa-dice-four","fal fa-dice-one","fal fa-dice-six","fal fa-dice-three","fal fa-dice-two","fal fa-digital-tachograph","fal fa-door-closed","fal fa-door-open","fal fa-drum","fal fa-drum-steelpan","fal fa-envelope","fal fa-envelope-open","fal fa-eraser","fal fa-eye","fal fa-eye-dropper","fal fa-fax","fal fa-feather","fal fa-feather-alt","fal fa-fighter-jet","fal fa-file","fal fa-file-alt","fal fa-file-prescription","fal fa-film","fal fa-film-alt","fal fa-fire","fal fa-fire-extinguisher","fal fa-flag","fal fa-flag-checkered","fal fa-flask","fal fa-futbol","fal fa-gamepad","fal fa-gavel","fal fa-gem","fal fa-gift","fal fa-glass-martini","fal fa-glasses","fal fa-globe","fal fa-graduation-cap","fal fa-hdd","fal fa-headphones","fal fa-headphones","fal fa-headphones-alt","fal fa-headset","fal fa-heart","fal fa-helicopter","fal fa-highlighter","fal fa-home","fal fa-hospital","fal fa-hourglass","fal fa-image","fal fa-images","fal fa-industry","fal fa-industry-alt","fal fa-jack-o-lantern","fal fa-key","fal fa-keyboard","fal fa-laptop","fal fa-leaf","fal fa-lemon","fal fa-life-ring","fal fa-lightbulb","fal fa-lock","fal fa-lock-alt","fal fa-lock-open","fal fa-lock-open-alt","fal fa-magic","fal fa-magnet","fal fa-map","fal fa-map-marker","fal fa-map-marker-alt","fal fa-map-pin","fal fa-map-signs","fal fa-marker","fal fa-medal","fal fa-medkit","fal fa-memory","fal fa-microchip","fal fa-microphone","fal fa-microphone-alt","fal fa-mobile","fal fa-mobile-alt","fal fa-mobile-android","fal fa-mobile-android-alt","fal fa-money-bill","fal fa-money-bill","fal fa-money-bill-alt","fal fa-money-check","fal fa-money-check-alt","fal fa-moon","fal fa-motorcycle","fal fa-newspaper","fal fa-paint-brush","fal fa-paper-plane","fal fa-paperclip","fal fa-paste","fal fa-paw","fal fa-pen","fal fa-pen-alt","fal fa-pen-fancy","fal fa-pen-nib","fal fa-pencil","fal fa-pencil-alt","fal fa-phone","fal fa-plane","fal fa-plane-alt","fal fa-plug","fal fa-print","fal fa-puzzle-piece","fal fa-road","fal fa-rocket","fal fa-ruler-combined","fal fa-ruler-horizontal","fal fa-ruler-vertical","fal fa-save","fal fa-school","fal fa-screwdriver","fal fa-search","fal fa-shield","fal fa-shield-alt","fal fa-shopping-bag","fal fa-shopping-basket","fal fa-shopping-cart","fal fa-shower","fal fa-snowflake","fal fa-space-shuttle","fal fa-star","fal fa-sticky-note","fal fa-stopwatch","fal fa-stroopwafel","fal fa-subway","fal fa-suitcase","fal fa-sun","fal fa-tablet","fal fa-tablet-alt","fal fa-tablet-android","fal fa-tablet-android-alt","fal fa-tachometer","fal fa-tachometer-alt","fal fa-tag","fal fa-tags","fal fa-taxi","fal fa-thumbtack","fal fa-ticket","fal fa-ticket-alt","fal fa-toolbox","fal fa-train","fal fa-trash","fal fa-trash-alt","fal fa-tree","fal fa-tree-alt","fal fa-trophy","fal fa-trophy-alt","fal fa-truck","fal fa-tv","fal fa-tv-retro","fal fa-umbrella","fal fa-university","fal fa-unlock","fal fa-unlock-alt","fal fa-utensil-fork","fal fa-utensil-knife","fal fa-utensil-spoon","fal fa-utensils","fal fa-utensils-alt","fal fa-wallet","fal fa-watch","fal fa-weight","fal fa-wheelchair","fal fa-wine-glass","fal fa-wrench","fas fa-alarm-clock","fas fa-ambulance","fas fa-anchor","fas fa-archive","fas fa-award","fas fa-balance-scale","fas fa-balance-scale-left","fas fa-balance-scale-right","fas fa-bath","fas fa-bed","fas fa-beer","fas fa-bell","fas fa-bicycle","fas fa-binoculars","fas fa-birthday-cake","fas fa-blender","fas fa-bomb","fas fa-book","fas fa-bookmark","fas fa-briefcase","fas fa-broadcast-tower","fas fa-bug","fas fa-building","fas fa-bullhorn","fas fa-bullseye","fas fa-bus","fas fa-calculator","fas fa-calendar","fas fa-calendar-alt","fas fa-camera","fas fa-camera-alt","fas fa-camera-retro","fas fa-car","fas fa-church","fas fa-clipboard","fas fa-cloud","fas fa-coffee","fas fa-cog","fas fa-cogs","fas fa-compass","fas fa-cookie","fas fa-cookie-bite","fas fa-copy","fas fa-cube","fas fa-cubes","fas fa-cut","fas fa-dice","fas fa-dice-five","fas fa-dice-four","fas fa-dice-one","fas fa-dice-six","fas fa-dice-three","fas fa-dice-two","fas fa-digital-tachograph","fas fa-door-closed","fas fa-door-open","fas fa-drum","fas fa-drum-steelpan","fas fa-envelope","fas fa-envelope-open","fas fa-eraser","fas fa-eye","fas fa-eye-dropper","fas fa-fax","fas fa-feather","fas fa-feather-alt","fas fa-fighter-jet","fas fa-file","fas fa-file-alt","fas fa-file-prescription","fas fa-film","fas fa-film-alt","fas fa-fire","fas fa-fire-extinguisher","fas fa-flag","fas fa-flag-checkered","fas fa-flask","fas fa-futbol","fas fa-gamepad","fas fa-gavel","fas fa-gem","fas fa-gift","fas fa-glass-martini","fas fa-glasses","fas fa-globe","fas fa-graduation-cap","fas fa-hdd","fas fa-headphones","fas fa-headphones","fas fa-headphones-alt","fas fa-headset","fas fa-heart","fas fa-helicopter","fas fa-highlighter","fas fa-home","fas fa-hospital","fas fa-hourglass","fas fa-image","fas fa-images","fas fa-industry","fas fa-industry-alt","fas fa-jack-o-lantern","fas fa-key","fas fa-keyboard","fas fa-laptop","fas fa-leaf","fas fa-lemon","fas fa-life-ring","fas fa-lightbulb","fas fa-lock","fas fa-lock-alt","fas fa-lock-open","fas fa-lock-open-alt","fas fa-magic","fas fa-magnet","fas fa-map","fas fa-map-marker","fas fa-map-marker-alt","fas fa-map-pin","fas fa-map-signs","fas fa-marker","fas fa-medal","fas fa-medkit","fas fa-memory","fas fa-microchip","fas fa-microphone","fas fa-microphone-alt","fas fa-mobile","fas fa-mobile-alt","fas fa-mobile-android","fas fa-mobile-android-alt","fas fa-money-bill","fas fa-money-bill","fas fa-money-bill-alt","fas fa-money-check","fas fa-money-check-alt","fas fa-moon","fas fa-motorcycle","fas fa-newspaper","fas fa-paint-brush","fas fa-paper-plane","fas fa-paperclip","fas fa-paste","fas fa-paw","fas fa-pen","fas fa-pen-alt","fas fa-pen-fancy","fas fa-pen-nib","fas fa-pencil","fas fa-pencil-alt","fas fa-phone","fas fa-plane","fas fa-plane-alt","fas fa-plug","fas fa-print","fas fa-puzzle-piece","fas fa-road","fas fa-rocket","fas fa-ruler-combined","fas fa-ruler-horizontal","fas fa-ruler-vertical","fas fa-save","fas fa-school","fas fa-screwdriver","fas fa-search","fas fa-shield","fas fa-shield-alt","fas fa-shopping-bag","fas fa-shopping-basket","fas fa-shopping-cart","fas fa-shower","fas fa-snowflake","fas fa-space-shuttle","fas fa-star","fas fa-sticky-note","fas fa-stopwatch","fas fa-stroopwafel","fas fa-subway","fas fa-suitcase","fas fa-sun","fas fa-tablet","fas fa-tablet-alt","fas fa-tablet-android","fas fa-tablet-android-alt","fas fa-tachometer","fas fa-tachometer-alt","fas fa-tag","fas fa-tags","fas fa-taxi","fas fa-thumbtack","fas fa-ticket","fas fa-ticket-alt","fas fa-toolbox","fas fa-train","fas fa-trash","fas fa-trash-alt","fas fa-tree","fas fa-tree-alt","fas fa-trophy","fas fa-trophy-alt","fas fa-truck","fas fa-tv","fas fa-tv-retro","fas fa-umbrella","fas fa-university","fas fa-unlock","fas fa-unlock-alt","fas fa-utensil-fork","fas fa-utensil-knife","fas fa-utensil-spoon","fas fa-utensils","fas fa-utensils-alt","fas fa-wallet","fas fa-watch","fas fa-weight","fas fa-wheelchair","fas fa-wine-glass","fas fa-wrench"]},{"id":"moving","title":"Moving","icons":["fal fa-archive","fal fa-blanket","fal fa-box-alt","fal fa-box-fragile","fal fa-box-full","fal fa-box-open","fal fa-box-up","fal fa-boxes-alt","fal fa-container-storage","fal fa-couch","fal fa-dolly","fal fa-dolly-empty","fal fa-fragile","fal fa-lamp","fal fa-loveseat","fal fa-people-carry","fal fa-person-carry","fal fa-person-dolly","fal fa-person-dolly-empty","fal fa-ramp-loading","fal fa-route","fal fa-sign","fal fa-suitcase","fal fa-tape","fal fa-truck-container","fal fa-truck-couch","fal fa-truck-loading","fal fa-truck-moving","fal fa-truck-ramp","fal fa-wine-glass","fas fa-archive","fas fa-blanket","fas fa-box-alt","fas fa-box-fragile","fas fa-box-full","fas fa-box-open","fas fa-box-up","fas fa-boxes-alt","fas fa-container-storage","fas fa-couch","fas fa-dolly","fas fa-dolly-empty","fas fa-fragile","fas fa-lamp","fas fa-loveseat","fas fa-people-carry","fas fa-person-carry","fas fa-person-dolly","fas fa-person-dolly-empty","fas fa-ramp-loading","fas fa-route","fas fa-sign","fas fa-suitcase","fas fa-tape","fas fa-truck-container","fas fa-truck-couch","fas fa-truck-loading","fas fa-truck-moving","fas fa-truck-ramp","fas fa-wine-glass"]},{"id":"hands","title":"Hands","icons":["fal fa-allergies","fal fa-hand-holding","fal fa-hand-holding-box","fal fa-hand-lizard","fal fa-hand-paper","fal fa-hand-peace","fal fa-hand-point-down","fal fa-hand-point-left","fal fa-hand-point-right","fal fa-hand-point-up","fal fa-hand-pointer","fal fa-hand-receiving","fal fa-hand-rock","fal fa-hand-scissors","fal fa-hand-spock","fal fa-hands","fal fa-handshake","fal fa-thumbs-down","fal fa-thumbs-up","fas fa-allergies","fas fa-hand-holding","fas fa-hand-holding-box","fas fa-hand-lizard","fas fa-hand-paper","fas fa-hand-peace","fas fa-hand-point-down","fas fa-hand-point-left","fas fa-hand-point-right","fas fa-hand-point-up","fas fa-hand-pointer","fas fa-hand-receiving","fas fa-hand-rock","fas fa-hand-scissors","fas fa-hand-spock","fas fa-hands","fas fa-handshake","fas fa-thumbs-down","fas fa-thumbs-up"]},{"id":"charity","title":"Charity","icons":["fal fa-book-heart","fal fa-box-heart","fal fa-box-usd","fal fa-dollar-sign","fal fa-donate","fal fa-dove","fal fa-gift","fal fa-globe","fal fa-hand-heart","fal fa-hand-holding-heart","fal fa-hand-holding-seedling","fal fa-hand-holding-usd","fal fa-hand-holding-water","fal fa-hands-heart","fal fa-hands-helping","fal fa-hands-usd","fal fa-handshake","fal fa-handshake-alt","fal fa-heart","fal fa-heart-circle","fal fa-heart-square","fal fa-home-heart","fal fa-leaf","fal fa-leaf-heart","fal fa-parachute-box","fal fa-piggy-bank","fal fa-ribbon","fal fa-seedling","fal fa-usd-circle","fal fa-usd-square","fas fa-book-heart","fas fa-box-heart","fas fa-box-usd","fas fa-dollar-sign","fas fa-donate","fas fa-dove","fas fa-gift","fas fa-globe","fas fa-hand-heart","fas fa-hand-holding-heart","fas fa-hand-holding-seedling","fas fa-hand-holding-usd","fas fa-hand-holding-water","fas fa-hands-heart","fas fa-hands-helping","fas fa-hands-usd","fas fa-handshake","fas fa-handshake-alt","fas fa-heart","fas fa-heart-circle","fas fa-heart-square","fas fa-home-heart","fas fa-leaf","fas fa-leaf-heart","fas fa-parachute-box","fas fa-piggy-bank","fas fa-ribbon","fas fa-seedling","fas fa-usd-circle","fas fa-usd-square"]},{"id":"animals","title":"Animals","icons":["fal fa-crow","fal fa-dove","fal fa-feather","fal fa-feather-alt","fal fa-fish","fal fa-frog","fal fa-kiwi-bird","fas fa-crow","fas fa-dove","fas fa-feather","fas fa-feather-alt","fas fa-fish","fas fa-frog","fas fa-kiwi-bird"]},{"id":"gender","title":"Genders","icons":["fal fa-genderless","fal fa-mars","fal fa-mars-double","fal fa-mars-stroke","fal fa-mars-stroke-h","fal fa-mars-stroke-v","fal fa-mercury","fal fa-neuter","fal fa-transgender","fal fa-transgender-alt","fal fa-venus","fal fa-venus-double","fal fa-venus-mars","fas fa-genderless","fas fa-mars","fas fa-mars-double","fas fa-mars-stroke","fas fa-mars-stroke-h","fas fa-mars-stroke-v","fas fa-mercury","fas fa-neuter","fas fa-transgender","fas fa-transgender-alt","fas fa-venus","fas fa-venus-double","fas fa-venus-mars"]},{"id":"medical","title":"Medical","icons":["fal fa-allergies","fal fa-ambulance","fal fa-band-aid","fal fa-briefcase-medical","fal fa-burn","fal fa-cannabis","fal fa-capsules","fal fa-diagnoses","fal fa-dna","fal fa-file-medical","fal fa-file-medical-alt","fal fa-file-prescription","fal fa-first-aid","fal fa-heart","fal fa-heartbeat","fal fa-hospital","fal fa-hospital-alt","fal fa-hospital-symbol","fal fa-id-card-alt","fal fa-notes-medical","fal fa-pills","fal fa-plus","fal fa-prescription","fal fa-prescription-bottle","fal fa-prescription-bottle-alt","fal fa-procedures","fal fa-smoking","fal fa-stethoscope","fal fa-syringe","fal fa-tablets","fal fa-thermometer","fal fa-user-md","fal fa-vial","fal fa-vials","fal fa-weight","fal fa-x-ray","fas fa-allergies","fas fa-ambulance","fas fa-band-aid","fas fa-briefcase-medical","fas fa-burn","fas fa-cannabis","fas fa-capsules","fas fa-diagnoses","fas fa-dna","fas fa-file-medical","fas fa-file-medical-alt","fas fa-file-prescription","fas fa-first-aid","fas fa-heart","fas fa-heartbeat","fas fa-hospital","fas fa-hospital-alt","fas fa-hospital-symbol","fas fa-id-card-alt","fas fa-notes-medical","fas fa-pills","fas fa-plus","fas fa-prescription","fas fa-prescription-bottle","fas fa-prescription-bottle-alt","fas fa-procedures","fas fa-smoking","fas fa-stethoscope","fas fa-syringe","fas fa-tablets","fas fa-thermometer","fas fa-user-md","fas fa-vial","fas fa-vials","fas fa-weight","fas fa-x-ray"]},{"id":"vehicles","title":"Vehicles","icons":["fab fa-accessible-icon","fal fa-ambulance","fal fa-bicycle","fal fa-bus","fal fa-car","fal fa-fighter-jet","fal fa-helicopter","fal fa-motorcycle","fal fa-paper-plane","fal fa-plane","fal fa-plane-alt","fal fa-rocket","fal fa-ship","fal fa-shopping-cart","fal fa-space-shuttle","fal fa-subway","fal fa-taxi","fal fa-train","fal fa-truck","fal fa-wheelchair","fas fa-ambulance","fas fa-bicycle","fas fa-bus","fas fa-car","fas fa-fighter-jet","fas fa-helicopter","fas fa-motorcycle","fas fa-paper-plane","fas fa-plane","fas fa-plane-alt","fas fa-rocket","fas fa-ship","fas fa-shopping-cart","fas fa-space-shuttle","fas fa-subway","fas fa-taxi","fas fa-train","fas fa-truck","fas fa-wheelchair"]},{"id":"arrows","title":"Arrows","icons":["fal fa-angle-double-down","fal fa-angle-double-left","fal fa-angle-double-right","fal fa-angle-double-up","fal fa-angle-down","fal fa-angle-left","fal fa-angle-right","fal fa-angle-up","fal fa-arrow-alt-circle-down","fal fa-arrow-alt-circle-left","fal fa-arrow-alt-circle-right","fal fa-arrow-alt-circle-up","fal fa-arrow-alt-down","fal fa-arrow-alt-from-bottom","fal fa-arrow-alt-from-left","fal fa-arrow-alt-from-right","fal fa-arrow-alt-from-top","fal fa-arrow-alt-left","fal fa-arrow-alt-right","fal fa-arrow-alt-square-down","fal fa-arrow-alt-square-left","fal fa-arrow-alt-square-right","fal fa-arrow-alt-square-up","fal fa-arrow-alt-to-bottom","fal fa-arrow-alt-to-left","fal fa-arrow-alt-to-right","fal fa-arrow-alt-to-top","fal fa-arrow-alt-up","fal fa-arrow-circle-down","fal fa-arrow-circle-left","fal fa-arrow-circle-right","fal fa-arrow-circle-up","fal fa-arrow-down","fal fa-arrow-from-bottom","fal fa-arrow-from-left","fal fa-arrow-from-right","fal fa-arrow-from-top","fal fa-arrow-left","fal fa-arrow-right","fal fa-arrow-square-down","fal fa-arrow-square-left","fal fa-arrow-square-right","fal fa-arrow-square-up","fal fa-arrow-to-bottom","fal fa-arrow-to-left","fal fa-arrow-to-right","fal fa-arrow-to-top","fal fa-arrow-up","fal fa-arrows","fal fa-arrows-alt","fal fa-arrows-alt-h","fal fa-arrows-alt-v","fal fa-arrows-h","fal fa-arrows-v","fal fa-caret-circle-down","fal fa-caret-circle-left","fal fa-caret-circle-right","fal fa-caret-circle-up","fal fa-caret-down","fal fa-caret-left","fal fa-caret-right","fal fa-caret-square-down","fal fa-caret-square-left","fal fa-caret-square-right","fal fa-caret-square-up","fal fa-caret-up","fal fa-cart-arrow-down","fal fa-chart-line","fal fa-chevron-circle-down","fal fa-chevron-circle-left","fal fa-chevron-circle-right","fal fa-chevron-circle-up","fal fa-chevron-double-down","fal fa-chevron-double-left","fal fa-chevron-double-right","fal fa-chevron-double-up","fal fa-chevron-down","fal fa-chevron-left","fal fa-chevron-right","fal fa-chevron-square-down","fal fa-chevron-square-left","fal fa-chevron-square-right","fal fa-chevron-square-up","fal fa-chevron-up","fal fa-cloud-download","fal fa-cloud-download-alt","fal fa-cloud-upload","fal fa-cloud-upload-alt","fal fa-compress-alt","fal fa-download","fal fa-exchange","fal fa-exchange-alt","fal fa-expand-alt","fal fa-expand-arrows","fal fa-expand-arrows-alt","fal fa-external-link","fal fa-external-link-alt","fal fa-external-link-square","fal fa-external-link-square-alt","fal fa-hand-point-down","fal fa-hand-point-left","fal fa-hand-point-right","fal fa-hand-point-up","fal fa-hand-pointer","fal fa-history","fal fa-inbox-in","fal fa-inbox-out","fal fa-level-down","fal fa-level-down-alt","fal fa-level-up","fal fa-level-up-alt","fal fa-location-arrow","fal fa-long-arrow-alt-down","fal fa-long-arrow-alt-left","fal fa-long-arrow-alt-right","fal fa-long-arrow-alt-up","fal fa-long-arrow-down","fal fa-long-arrow-left","fal fa-long-arrow-right","fal fa-long-arrow-up","fal fa-mouse-pointer","fal fa-play","fal fa-random","fal fa-recycle","fal fa-redo","fal fa-redo-alt","fal fa-repeat","fal fa-repeat-1","fal fa-repeat-1-alt","fal fa-repeat-alt","fal fa-reply","fal fa-reply-all","fal fa-retweet","fal fa-retweet-alt","fal fa-share","fal fa-share-all","fal fa-share-square","fal fa-sign-in","fal fa-sign-in-alt","fal fa-sign-out","fal fa-sign-out-alt","fal fa-sort","fal fa-sort-alpha-down","fal fa-sort-alpha-up","fal fa-sort-amount-down","fal fa-sort-amount-up","fal fa-sort-down","fal fa-sort-numeric-down","fal fa-sort-numeric-up","fal fa-sort-up","fal fa-sync","fal fa-sync-alt","fal fa-text-height","fal fa-text-width","fal fa-triangle","fal fa-undo","fal fa-undo-alt","fal fa-upload","fas fa-angle-double-down","fas fa-angle-double-left","fas fa-angle-double-right","fas fa-angle-double-up","fas fa-angle-down","fas fa-angle-left","fas fa-angle-right","fas fa-angle-up","fas fa-arrow-alt-circle-down","fas fa-arrow-alt-circle-left","fas fa-arrow-alt-circle-right","fas fa-arrow-alt-circle-up","fas fa-arrow-alt-down","fas fa-arrow-alt-from-bottom","fas fa-arrow-alt-from-left","fas fa-arrow-alt-from-right","fas fa-arrow-alt-from-top","fas fa-arrow-alt-left","fas fa-arrow-alt-right","fas fa-arrow-alt-square-down","fas fa-arrow-alt-square-left","fas fa-arrow-alt-square-right","fas fa-arrow-alt-square-up","fas fa-arrow-alt-to-bottom","fas fa-arrow-alt-to-left","fas fa-arrow-alt-to-right","fas fa-arrow-alt-to-top","fas fa-arrow-alt-up","fas fa-arrow-circle-down","fas fa-arrow-circle-left","fas fa-arrow-circle-right","fas fa-arrow-circle-up","fas fa-arrow-down","fas fa-arrow-from-bottom","fas fa-arrow-from-left","fas fa-arrow-from-right","fas fa-arrow-from-top","fas fa-arrow-left","fas fa-arrow-right","fas fa-arrow-square-down","fas fa-arrow-square-left","fas fa-arrow-square-right","fas fa-arrow-square-up","fas fa-arrow-to-bottom","fas fa-arrow-to-left","fas fa-arrow-to-right","fas fa-arrow-to-top","fas fa-arrow-up","fas fa-arrows","fas fa-arrows-alt","fas fa-arrows-alt-h","fas fa-arrows-alt-v","fas fa-arrows-h","fas fa-arrows-v","fas fa-caret-circle-down","fas fa-caret-circle-left","fas fa-caret-circle-right","fas fa-caret-circle-up","fas fa-caret-down","fas fa-caret-left","fas fa-caret-right","fas fa-caret-square-down","fas fa-caret-square-left","fas fa-caret-square-right","fas fa-caret-square-up","fas fa-caret-up","fas fa-cart-arrow-down","fas fa-chart-line","fas fa-chevron-circle-down","fas fa-chevron-circle-left","fas fa-chevron-circle-right","fas fa-chevron-circle-up","fas fa-chevron-double-down","fas fa-chevron-double-left","fas fa-chevron-double-right","fas fa-chevron-double-up","fas fa-chevron-down","fas fa-chevron-left","fas fa-chevron-right","fas fa-chevron-square-down","fas fa-chevron-square-left","fas fa-chevron-square-right","fas fa-chevron-square-up","fas fa-chevron-up","fas fa-cloud-download","fas fa-cloud-download-alt","fas fa-cloud-upload","fas fa-cloud-upload-alt","fas fa-compress-alt","fas fa-download","fas fa-exchange","fas fa-exchange-alt","fas fa-expand-alt","fas fa-expand-arrows","fas fa-expand-arrows-alt","fas fa-external-link","fas fa-external-link-alt","fas fa-external-link-square","fas fa-external-link-square-alt","fas fa-hand-point-down","fas fa-hand-point-left","fas fa-hand-point-right","fas fa-hand-point-up","fas fa-hand-pointer","fas fa-history","fas fa-inbox-in","fas fa-inbox-out","fas fa-level-down","fas fa-level-down-alt","fas fa-level-up","fas fa-level-up-alt","fas fa-location-arrow","fas fa-long-arrow-alt-down","fas fa-long-arrow-alt-left","fas fa-long-arrow-alt-right","fas fa-long-arrow-alt-up","fas fa-long-arrow-down","fas fa-long-arrow-left","fas fa-long-arrow-right","fas fa-long-arrow-up","fas fa-mouse-pointer","fas fa-play","fas fa-random","fas fa-recycle","fas fa-redo","fas fa-redo-alt","fas fa-repeat","fas fa-repeat-1","fas fa-repeat-1-alt","fas fa-repeat-alt","fas fa-reply","fas fa-reply-all","fas fa-retweet","fas fa-retweet-alt","fas fa-share","fas fa-share-all","fas fa-share-square","fas fa-sign-in","fas fa-sign-in-alt","fas fa-sign-out","fas fa-sign-out-alt","fas fa-sort","fas fa-sort-alpha-down","fas fa-sort-alpha-up","fas fa-sort-amount-down","fas fa-sort-amount-up","fas fa-sort-down","fas fa-sort-numeric-down","fas fa-sort-numeric-up","fas fa-sort-up","fas fa-sync","fas fa-sync-alt","fas fa-text-height","fas fa-text-width","fas fa-triangle","fas fa-undo","fas fa-undo-alt","fas fa-upload"]},{"id":"shapes","title":"Shapes","icons":["fal fa-badge","fal fa-bookmark","fal fa-calendar","fal fa-certificate","fal fa-circle","fal fa-cloud","fal fa-club","fal fa-comment","fal fa-diamond","fal fa-file","fal fa-folder","fal fa-heart","fal fa-hexagon","fal fa-map-marker","fal fa-octagon","fal fa-play","fal fa-rectangle-landscape","fal fa-rectangle-portrait","fal fa-rectangle-wide","fal fa-shield","fal fa-spade","fal fa-square","fal fa-star","fal fa-ticket","fal fa-triangle","fas fa-badge","fas fa-bookmark","fas fa-calendar","fas fa-certificate","fas fa-circle","fas fa-cloud","fas fa-club","fas fa-comment","fas fa-diamond","fas fa-file","fas fa-folder","fas fa-heart","fas fa-hexagon","fas fa-map-marker","fas fa-octagon","fas fa-play","fas fa-rectangle-landscape","fas fa-rectangle-portrait","fas fa-rectangle-wide","fas fa-shield","fas fa-spade","fas fa-square","fas fa-star","fas fa-ticket","fas fa-triangle"]},{"id":"users-people","title":"Users & People","icons":["fab fa-accessible-icon","fal fa-address-book","fal fa-address-card","fal fa-bed","fal fa-blind","fal fa-chalkboard-teacher","fal fa-child","fal fa-female","fal fa-frown","fal fa-id-badge","fal fa-id-card","fal fa-id-card-alt","fal fa-male","fal fa-meh","fal fa-poo","fal fa-portrait","fal fa-power-off","fal fa-smile","fal fa-street-view","fal fa-user","fal fa-user-alt","fal fa-user-alt-slash","fal fa-user-astronaut","fal fa-user-check","fal fa-user-circle","fal fa-user-clock","fal fa-user-cog","fal fa-user-edit","fal fa-user-friends","fal fa-user-graduate","fal fa-user-lock","fal fa-user-md","fal fa-user-minus","fal fa-user-ninja","fal fa-user-plus","fal fa-user-secret","fal fa-user-shield","fal fa-user-slash","fal fa-user-tag","fal fa-user-tie","fal fa-user-times","fal fa-users","fal fa-wheelchair","fas fa-address-book","fas fa-address-card","fas fa-bed","fas fa-blind","fas fa-chalkboard-teacher","fas fa-child","fas fa-female","fas fa-frown","fas fa-id-badge","fas fa-id-card","fas fa-id-card-alt","fas fa-male","fas fa-meh","fas fa-poo","fas fa-portrait","fas fa-power-off","fas fa-smile","fas fa-street-view","fas fa-user","fas fa-user-alt","fas fa-user-alt-slash","fas fa-user-astronaut","fas fa-user-check","fas fa-user-circle","fas fa-user-clock","fas fa-user-cog","fas fa-user-edit","fas fa-user-friends","fas fa-user-graduate","fas fa-user-lock","fas fa-user-md","fas fa-user-minus","fas fa-user-ninja","fas fa-user-plus","fas fa-user-secret","fas fa-user-shield","fas fa-user-slash","fas fa-user-tag","fas fa-user-tie","fas fa-user-times","fas fa-users","fas fa-wheelchair"]},{"id":"brands","title":"Brands","icons":["fab fa-500px","fab fa-accessible-icon","fab fa-accusoft","fab fa-adn","fab fa-adversal","fab fa-affiliatetheme","fab fa-algolia","fab fa-amazon","fab fa-amazon-pay","fab fa-amilia","fab fa-android","fab fa-angellist","fab fa-angrycreative","fab fa-angular","fab fa-app-store","fab fa-app-store-ios","fab fa-apper","fab fa-apple","fab fa-apple-pay","fab fa-asymmetrik","fab fa-audible","fab fa-autoprefixer","fab fa-avianex","fab fa-aviato","fab fa-aws","fab fa-bandcamp","fab fa-behance","fab fa-behance-square","fab fa-bimobject","fab fa-bitbucket","fab fa-bitcoin","fab fa-bity","fab fa-black-tie","fab fa-blackberry","fab fa-blogger","fab fa-blogger-b","fab fa-bluetooth","fab fa-bluetooth-b","fab fa-btc","fab fa-buromobelexperte","fab fa-buysellads","fab fa-cc-amazon-pay","fab fa-cc-amex","fab fa-cc-apple-pay","fab fa-cc-diners-club","fab fa-cc-discover","fab fa-cc-jcb","fab fa-cc-mastercard","fab fa-cc-paypal","fab fa-cc-stripe","fab fa-cc-visa","fab fa-centercode","fab fa-chrome","fab fa-cloudscale","fab fa-cloudsmith","fab fa-cloudversify","fab fa-codepen","fab fa-codiepie","fab fa-connectdevelop","fab fa-contao","fab fa-cpanel","fab fa-creative-commons","fab fa-creative-commons-by","fab fa-creative-commons-nc","fab fa-creative-commons-nc-eu","fab fa-creative-commons-nc-jp","fab fa-creative-commons-nd","fab fa-creative-commons-pd","fab fa-creative-commons-pd-alt","fab fa-creative-commons-remix","fab fa-creative-commons-sa","fab fa-creative-commons-sampling","fab fa-creative-commons-sampling-plus","fab fa-creative-commons-share","fab fa-creative-commons-zero","fab fa-css3","fab fa-css3-alt","fab fa-cuttlefish","fab fa-d-and-d","fab fa-dashcube","fab fa-delicious","fab fa-deploydog","fab fa-deskpro","fab fa-deviantart","fab fa-digg","fab fa-digital-ocean","fab fa-discord","fab fa-discourse","fab fa-dochub","fab fa-docker","fab fa-draft2digital","fab fa-dribbble","fab fa-dribbble-square","fab fa-dropbox","fab fa-drupal","fab fa-dyalog","fab fa-earlybirds","fab fa-ebay","fab fa-edge","fab fa-elementor","fab fa-ember","fab fa-empire","fab fa-envira","fab fa-erlang","fab fa-ethereum","fab fa-etsy","fab fa-expeditedssl","fab fa-facebook","fab fa-facebook-f","fab fa-facebook-messenger","fab fa-facebook-square","fab fa-firefox","fab fa-first-order","fab fa-first-order-alt","fab fa-firstdraft","fab fa-flickr","fab fa-flipboard","fab fa-fly","fab fa-font-awesome","fab fa-font-awesome-alt","fab fa-font-awesome-flag","fab fa-fonticons","fab fa-fonticons-fi","fab fa-fort-awesome","fab fa-fort-awesome-alt","fab fa-forumbee","fab fa-foursquare","fab fa-free-code-camp","fab fa-freebsd","fab fa-fulcrum","fab fa-galactic-republic","fab fa-galactic-senate","fab fa-get-pocket","fab fa-gg","fab fa-gg-circle","fab fa-git","fab fa-git-square","fab fa-github","fab fa-github-alt","fab fa-github-square","fab fa-gitkraken","fab fa-gitlab","fab fa-gitter","fab fa-glide","fab fa-glide-g","fab fa-gofore","fab fa-goodreads","fab fa-goodreads-g","fab fa-google","fab fa-google-drive","fab fa-google-play","fab fa-google-plus","fab fa-google-plus-g","fab fa-google-plus-square","fab fa-google-wallet","fab fa-gratipay","fab fa-grav","fab fa-gripfire","fab fa-grunt","fab fa-gulp","fab fa-hacker-news","fab fa-hacker-news-square","fab fa-hips","fab fa-hire-a-helper","fab fa-hooli","fab fa-hornbill","fab fa-hotjar","fab fa-houzz","fab fa-html5","fab fa-hubspot","fab fa-imdb","fab fa-instagram","fab fa-internet-explorer","fab fa-ioxhost","fab fa-itunes","fab fa-itunes-note","fab fa-java","fab fa-jedi-order","fab fa-jenkins","fab fa-joget","fab fa-joomla","fab fa-js","fab fa-js-square","fab fa-jsfiddle","fab fa-keybase","fab fa-keycdn","fab fa-kickstarter","fab fa-kickstarter-k","fab fa-korvue","fab fa-laravel","fab fa-lastfm","fab fa-lastfm-square","fab fa-leanpub","fab fa-less","fab fa-line","fab fa-linkedin","fab fa-linkedin-in","fab fa-linode","fab fa-linux","fab fa-lyft","fab fa-magento","fab fa-mailchimp","fab fa-mandalorian","fab fa-mastodon","fab fa-maxcdn","fab fa-medapps","fab fa-medium","fab fa-medium-m","fab fa-medrt","fab fa-meetup","fab fa-megaport","fab fa-microsoft","fab fa-mix","fab fa-mixcloud","fab fa-mizuni","fab fa-modx","fab fa-monero","fab fa-napster","fab fa-nimblr","fab fa-nintendo-switch","fab fa-node","fab fa-node-js","fab fa-npm","fab fa-ns8","fab fa-nutritionix","fab fa-odnoklassniki","fab fa-odnoklassniki-square","fab fa-old-republic","fab fa-opencart","fab fa-openid","fab fa-opera","fab fa-optin-monster","fab fa-osi","fab fa-page4","fab fa-pagelines","fab fa-palfed","fab fa-patreon","fab fa-paypal","fab fa-periscope","fab fa-phabricator","fab fa-phoenix-framework","fab fa-phoenix-squadron","fab fa-php","fab fa-pied-piper","fab fa-pied-piper-alt","fab fa-pied-piper-hat","fab fa-pied-piper-pp","fab fa-pinterest","fab fa-pinterest-p","fab fa-pinterest-square","fab fa-playstation","fab fa-product-hunt","fab fa-pushed","fab fa-python","fab fa-qq","fab fa-quinscape","fab fa-quora","fab fa-r-project","fab fa-ravelry","fab fa-react","fab fa-readme","fab fa-rebel","fab fa-red-river","fab fa-reddit","fab fa-reddit-alien","fab fa-reddit-square","fab fa-rendact","fab fa-renren","fab fa-replyd","fab fa-researchgate","fab fa-resolving","fab fa-rev","fab fa-rocketchat","fab fa-rockrms","fab fa-safari","fab fa-sass","fab fa-schlix","fab fa-scribd","fab fa-searchengin","fab fa-sellcast","fab fa-sellsy","fab fa-servicestack","fab fa-shirtsinbulk","fab fa-shopware","fab fa-simplybuilt","fab fa-sistrix","fab fa-sith","fab fa-skyatlas","fab fa-skype","fab fa-slack","fab fa-slack-hash","fab fa-slideshare","fab fa-snapchat","fab fa-snapchat-ghost","fab fa-snapchat-square","fab fa-soundcloud","fab fa-speakap","fab fa-spotify","fab fa-squarespace","fab fa-stack-exchange","fab fa-stack-overflow","fab fa-staylinked","fab fa-steam","fab fa-steam-square","fab fa-steam-symbol","fab fa-sticker-mule","fab fa-strava","fab fa-stripe","fab fa-stripe-s","fab fa-studiovinari","fab fa-stumbleupon","fab fa-stumbleupon-circle","fab fa-superpowers","fab fa-supple","fab fa-teamspeak","fab fa-telegram","fab fa-telegram-plane","fab fa-tencent-weibo","fab fa-themeco","fab fa-themeisle","fab fa-trade-federation","fab fa-trello","fab fa-tripadvisor","fab fa-tumblr","fab fa-tumblr-square","fab fa-twitch","fab fa-twitter","fab fa-twitter-square","fab fa-typo3","fab fa-uber","fab fa-uikit","fab fa-uniregistry","fab fa-untappd","fab fa-usb","fab fa-ussunnah","fab fa-vaadin","fab fa-viacoin","fab fa-viadeo","fab fa-viadeo-square","fab fa-viber","fab fa-vimeo","fab fa-vimeo-square","fab fa-vimeo-v","fab fa-vine","fab fa-vk","fab fa-vnv","fab fa-vuejs","fab fa-weebly","fab fa-weibo","fab fa-weixin","fab fa-whatsapp","fab fa-whatsapp-square","fab fa-whmcs","fab fa-wikipedia-w","fab fa-windows","fab fa-wix","fab fa-wolf-pack-battalion","fab fa-wordpress","fab fa-wordpress-simple","fab fa-wpbeginner","fab fa-wpexplorer","fab fa-wpforms","fab fa-xbox","fab fa-xing","fab fa-xing-square","fab fa-y-combinator","fab fa-yahoo","fab fa-yandex","fab fa-yandex-international","fab fa-yelp","fab fa-yoast","fab fa-youtube","fab fa-youtube-square"]}]', true)
                ),
            ));
        }
	}
}